/* -*- Mode: C++ -*- */

/* This set of classes flexibly represents an abstract state space */

#ifndef _ABSTRACT_STATE_H_
#define _ABSTRACT_STATE_H_

#include <iostream>
#include <fstream>
#include <vector>
#include "WorldState.h"
#include "FieldImage.h"
#include "CoachMessageQueue.h"
class AdviceTree;

/***********************************************************************/
class AbstractStateFilter;
class AbstractStateFactor;
class AbstractStateDescription;
class AbstractState;

/***********************************************************************/

// This virtual base class provides a filter for saying some states simply
// do not have equivalents in the abstract state space
class AbstractStateFilter
{
public:
  AbstractStateFilter() {}
  virtual ~AbstractStateFilter() {}

  // returns false to NOT consider this state
  virtual bool isValid(const WorldState& ws, TeamSide my_side) = 0;

  virtual void print(std::ostream& os) const = 0;
  friend std::ostream& operator << (std::ostream &os, const AbstractStateFilter& f)
  { f.print(os); return os; }

  virtual void draw(FieldImage* pimg, RegionWorldModelInterface* wmi) {}

  virtual void createInitialAdvice(CoachMessageQueue& mqueue) {}

  virtual rcss::clang::Cond* createCondition() = 0;
};


/***********************************************************************/

enum ASFactorType {
  ASF_None,
  ASF_And,
  ASF_Or,
  ASF_POElement,
  ASF_POSet,
  ASF_BallGrid,
  ASF_BallOwner,
  ASF_Goal,
  ASF_DeadBall,
  ASF_Constant,
  ASF_MAX
};

std::ostream& operator<<(std::ostream& os, const ASFactorType& asft);
std::istream& operator<<(std::istream& is, ASFactorType& asft);

class AbstractStateFactor
{
public:
  // init to 1 for base level factors
  AbstractStateFactor(ASFactorType type) : type(type), leaf_count(1) { }
  virtual ~AbstractStateFactor() {}

  ASFactorType getType() const { return type; }
  
  // Returns the number of states in this factor
  virtual int getNumStates() = 0;

  int getLeafCount() const { return leaf_count; }
  
  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  virtual int setStateFor(AbstractState* pstate, int first_idx,
			  const WorldState& ws, TeamSide my_side) = 0;

  // calculates the global state index for this factor and this abstract state
  // pfirst_idx is an IO param for where in the state we are
  virtual int calcStateIdx(const AbstractState& state, int* pfirst_idx) const = 0;
  
  // sets the abstract state vals given a global state index
  virtual void setStateForIdx(int global_idx, AbstractState* state, int* pfirst_idx) = 0;
  
  virtual void print(std::ostream& os) const = 0;
  friend std::ostream& operator << (std::ostream &os, const AbstractStateFactor& f)
  { f.print(os); return os; }

  // defaults to one so that base elements can leave this
  virtual int calcLeafCount() const { return 1; }

  int updateLeafCount() { leaf_count = calcLeafCount(); return leaf_count; }

  virtual int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
		   const AbstractState* pstate, int first_idx) = 0;

  //returns true if the ball pos was set
  virtual bool extractBallPos(VecPosition* pv, const AbstractState* pstate, int *pfirst_idx) = 0;

  // Mostly for any defines that this factor wants
  virtual void createInitialAdvice(CoachMessageQueue& mqueue) {}

  virtual rcss::clang::Cond* createCondition(const AbstractState* pstate,
					     int* pfirst_idx) = 0;

  // First_idx is the index into the AbstractState that we should use
  AdviceTree* createAdviceTree(int first_idx);
  
  virtual void addAdviceTreeLevels(AdviceTree* ptree, int* pfirst_idx) = 0;

private:
  ASFactorType type;
  int leaf_count;
};

/***********************************************************************/

class AbstractStateFactorLeaf
  : public AbstractStateFactor
{
public:
  // init to 1 for base level factors
  AbstractStateFactorLeaf(ASFactorType type) : AbstractStateFactor(type) { }
  virtual ~AbstractStateFactorLeaf() {}

  // Returns the number of states in this factor
  virtual int getNumStates() = 0;

  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  virtual int setStateFor(AbstractState* pstate, int first_idx,
			  const WorldState& ws, TeamSide my_side) = 0;

  // calculates the global state index for this factor and this abstract state
  // pfirst_idx is an IO param for where in the state we are
  // just returns the single element
  int calcStateIdx(const AbstractState& state, int* pfirst_idx) const;
  
  // sets the abstract state vals given a global state index
  // just sets the element in the state
  // and advances pfirst_idx by one
  void setStateForIdx(int global_idx, AbstractState* state, int* pfirst_idx);
  
  virtual void print(std::ostream& os) const = 0;

  int calcLeafCount() const { return 1; }

  virtual int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
		   const AbstractState* pstate, int first_idx) = 0;

  //returns true if the ball pos was set
  // default is to do nothing
  virtual bool extractBallPos(VecPosition* pv, const AbstractState* pstate, int *pfirst_idx)
  { (*pfirst_idx)++; return false; }

  // Mostly for any defines that this factor wants
  virtual void createInitialAdvice(CoachMessageQueue& mqueue) {}

  rcss::clang::Cond* createCondition(const AbstractState* pstate,
				     int* pfirst_idx);
  // This is a leaf specific variant; called after the idx/value for this factor has
  // been extracted.
  virtual rcss::clang::Cond* createCondition(int my_val) = 0;

  // First_idx is the index into the AbstractState that we should use
  AdviceTree* createAdviceTree(int first_idx);
  
  // default for a leaf factor
  // We leave it virtual so that elements can "opt out" of being in an advice tree
  virtual void addAdviceTreeLevels(AdviceTree* ptree, int* pfirst_idx);

};


/***********************************************************************/

class AbstractStateFactorWFilters
  : public AbstractStateFactor
{
public:
  typedef std::vector<AbstractStateFilter*> FilterStorage;
public:
  AbstractStateFactorWFilters(ASFactorType type) : AbstractStateFactor(type), filters() {}
  virtual ~AbstractStateFactorWFilters();

  virtual void print(std::ostream& os) const;

  // takes over the memory
  void addFilter(AbstractStateFilter* p) { filters.push_back(p); }

protected:
  // return true if this state is okay
  bool checkFilters(const WorldState& ws, TeamSide my_side);

  // This is a separate method from the standard draw hierarchy since
  // it son't actually consume indices from the AbstractState
  void drawFilters(FieldImage* pimg, RegionWorldModelInterface* wmi);

  // Mostly for any defines that these filters want
  void createFilterInitialAdvice(CoachMessageQueue& mqueue);
  rcss::clang::Cond* createFilterCondition();

private:
  FilterStorage filters;
};

/***********************************************************************/

class AbstractStateFactorWChildren
  : public AbstractStateFactorWFilters
{
public:
  typedef std::vector<AbstractStateFactor*> FactorStorage;

public:
  AbstractStateFactorWChildren(ASFactorType type)
    : AbstractStateFactorWFilters(type), factors(), num_states(0) {}
  virtual ~AbstractStateFactorWChildren();

  // Returns the number of states in this factor
  int getNumStates() { return num_states; }

  //returns true if the ball pos was set
  bool extractBallPos(VecPosition* pv, const AbstractState* pstate, int *pfirst_idx);

  // takes over the memory
  void addFactor(AbstractStateFactor* p)
  { factors.push_back(p); notifyAddFactor(); updateNumStates(); updateLeafCount(); }

  int getNumChildren() const { return factors.size(); }

  AbstractStateFactor* getChild(int idx);
  
  void updateNumStates() { num_states = calcNumStates(); }
  
  void createInitialAdvice(CoachMessageQueue& mqueue);
  
protected:
  virtual void notifyAddFactor() = 0;

  virtual int calcNumStates() = 0;
  
  FactorStorage factors;

private:

  int num_states;

};

/***********************************************************************/
/* represents a factor where all of the subfactors combine every time (cross product) */
class AbstractStateFactorAnd
  : public AbstractStateFactorWChildren
{
public:
  // The int is the multiplication factor for converting to state index
  typedef std::vector<int> MultValsStorage;

public:
  AbstractStateFactorAnd();
  ~AbstractStateFactorAnd() {}

  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  int setStateFor(AbstractState* pstate, int first_idx,
		  const WorldState& ws, TeamSide my_side);
  
  // calculates the global state index for this factor and this abstract state
  // pfirst_idx is an IO param for where in the state we are
  int calcStateIdx(const AbstractState& state, int* pfirst_idx) const;

  // sets the abstract state vals given a global state index
  void setStateForIdx(int global_idx, AbstractState* state, int* pfirst_idx);

  void print(std::ostream& os) const;

  int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
	   const AbstractState* pstate, int first_idx);

  rcss::clang::Cond* createCondition(const AbstractState* pstate,
				     int* pfirst_idx);
  
  void addAdviceTreeLevels(AdviceTree* ptree, int* pfirst_idx);
  
protected:
  void notifyAddFactor()
  { updateMultFactor(); }

  int calcLeafCount() const;
  
  int calcNumStates()
  { return mult_vals.empty() ? 0 : *(mult_vals.end()-1) * (*(factors.end()-1))->getNumStates(); }

private:
  void updateMultFactor();

  /* The low-order element is stored first */
  MultValsStorage mult_vals; // the mult factor used to convert to a single index

};

/***********************************************************************/
/* represents a factor which is an additive sum, if the first factor is valid,
   that's it, oitherwise, check the next, etc. */
class AbstractStateFactorOr
  : public AbstractStateFactorWChildren
{
public:
  // The int is the additive
  typedef std::vector<int> AddValsStorage;

public:
  AbstractStateFactorOr();
  ~AbstractStateFactorOr() {}

  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  int setStateFor(AbstractState* pstate, int first_idx,
		  const WorldState& ws, TeamSide my_side);
  
  // calculates the global state index for this factor and this abstract state
  // pfirst_idx is an IO param for where in the state we are
  int calcStateIdx(const AbstractState& state, int* pfirst_idx) const;

  // sets the abstract state vals given a global state index
  void setStateForIdx(int global_idx, AbstractState* state, int* pfirst_idx);

  void print(std::ostream& os) const;

  int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
	   const AbstractState* pstate, int first_idx);

  rcss::clang::Cond* createCondition(const AbstractState* pstate,
				     int* pfirst_idx);

  void addAdviceTreeLevels(AdviceTree* ptree, int* pfirst_idx);

  void update() { updateAddVals(); updateNumStates(); updateLeafCount(); }

protected:
  int calcLeafCount() const;

  // takes over the memory
  void notifyAddFactor() { updateAddVals(); }

  int calcNumStates()
  { return add_vals.empty() ? 0 : *(add_vals.end()-1) + (*(factors.end()-1))->getNumStates(); }

private:
  void updateAddVals();

  /* The low-order element is stored first */
  AddValsStorage add_vals; // the mult factor used to convert to a single index

};


/***********************************************************************/

/* This class decribes an abstract state space. It does not represent a particular
   abstract state. See AbstractState for that */
class AbstractStateDescription
{
public:
  AbstractStateDescription(AbstractStateFactor* p = NULL) : pfactor(p) {}
  ~AbstractStateDescription() { if (pfactor) delete pfactor; }

  void setFactor(AbstractStateFactor* p);
  const AbstractStateFactor* getFactor() const { return pfactor; }
  AbstractStateFactor* getFactor() { return pfactor; }
  
  // returns the number of possible states
  int getNumStates() const { return (pfactor == NULL) ? -1 : pfactor->getNumStates(); }
  
  // If pstate is NULL, a new one is allocated
  // Returns NULL if this is not a valid WorldState for this abstract state space
  AbstractState* 
  getStateForWorldState(const WorldState& ws,
			TeamSide my_side,
			AbstractState* pstate = NULL);

  AdviceTree* createAdviceTree();
  
  friend std::ostream& operator << (std::ostream &os, const AbstractStateDescription& d);

private:
  AbstractStateFactor* pfactor; // the root of the tree of factors
};

/***********************************************************************/

/* The factors actually form a tree. What we represent here are the leaves of the
   the tree. -1 is always used as an invalid */
class AbstractState
{
public:
  AbstractState(AbstractStateDescription* pdesc)
    : pdesc(pdesc), idx_storage(), valid(false)
  { updateStorageSize(); }
  ~AbstractState() {}
  
  int getFactorIdx(int fac) const;
  void setFactorIdx(int fac, int idx);

  int getStateIdx() const;
  void setStateIdx(int idx);

  void updateStorageSize();
  int getStorageSize() const { return idx_storage.size(); }
  
  AbstractStateDescription* getStateDescription() const { return pdesc; }
  void setStateDescription(AbstractStateDescription* p) { pdesc = p; }

  bool isValid() const { return valid; }
  void setValid(bool v) { valid = v; }

  void draw(FieldImage* pimg, RegionWorldModelInterface* wmi);

  bool extractBallPos(VecPosition* pv);
  
  rcss::clang::Cond* createCondition();
  
  friend std::ostream& operator << (std::ostream &os, const AbstractState& s);

private:
  AbstractStateDescription* pdesc;
  
  // Stores the indexes of the factors for a state
  typedef std::vector<int> IdxStorage;
  IdxStorage idx_storage;
  bool valid;
};

/***********************************************************************/

class AbstractStateTracker
{
public:
  // my_side must be TS_Left or TS_Right
  // per_cycle_fn is NULL or empty to turn per cycle output off
  // Does NOT take over the AbstractStateDescription memory
  AbstractStateTracker(TeamSide my_side,
		       AbstractStateDescription* pdesc,
		       const char* per_cycle_fn);
  ~AbstractStateTracker() {}

  TeamSide getMySide() const { return my_side; }
  void setMySide(TeamSide s) { my_side = s; }

  const AbstractStateDescription* getStateDescription() const { return pdesc; }
  
  void writeSummaryInfo(std::ostream& os);

  void processWorldState(const WorldState& ws);

private:
  typedef std::vector<unsigned short> StateSummary;

  TeamSide my_side;
  AbstractStateDescription* pdesc;
  bool write_per_cycle;
  std::ofstream os_per_cycle;
  StateSummary state_summary;

  // even though this is transient date, we have it here to avoid reallocation
  AbstractState state;

};

/***********************************************************************/

#endif
