/* -*- Mode: C++ -*- */

/* This set of classes flexibly represents an abstract state space */

#include "AbstractState.h"
#include "AdviceTree.h"
#include "Logger.h"
#include "utility.h"

using namespace spades;

#define USE_SANITY_CHECKS

/***********************************************************************/
class FilterDrawCaller
{
public:
  FilterDrawCaller(FieldImage* pimg, RegionWorldModelInterface* wmi)
    : pimg(pimg), wmi(wmi) {}
  
  void operator() (AbstractStateFilter* p) 
  { p->draw(pimg, wmi); }
    
private:
  FieldImage* pimg;
  RegionWorldModelInterface* wmi;
};

/***********************************************************************/
class FactorDrawCaller
{
public:
  // we do this crazy *pidx thing because local state can't be accumulated
  // by the argument of for_each
  FactorDrawCaller(FieldImage* pimg, RegionWorldModelInterface* wmi,
		   const AbstractState* pstate, int *pidx)
    : pimg(pimg), wmi(wmi), pstate(pstate), pidx(pidx) {}
  
  void operator() (AbstractStateFactor* p) 
  { *pidx = p->draw(pimg, wmi, pstate, *pidx); }
  
private:
  FieldImage* pimg;
  RegionWorldModelInterface* wmi;
  const AbstractState* pstate;
  int* pidx;
};


/***********************************************************************/

static const char* AS_FACTOR_TYPE_STRINGS[] =
  { "None", "And", "Or", "POSet", "BallGrid", "BallOwner", "Goal", "DeadBall", "Constant" 
  };

std::ostream&
operator<<(std::ostream& os, const ASFactorType& asft)
{
  if (asft < 0 || asft >= ASF_MAX)
    os << "Invalid ASFactorType(" << (int)asft << ")";
  else
    os << AS_FACTOR_TYPE_STRINGS[(int)asft];
  return os;
}

std::istream&
operator<<(std::istream& is, ASFactorType& asft)
{
  std::string s;
  is >> s;
  if (is.fail())
    return is;
  for (int i=0; i < ASF_MAX; i++)
    {
      if (strcasecmp(AS_FACTOR_TYPE_STRINGS[i], s.c_str()) == 0)
	{
	  asft = (ASFactorType)i;
	  return is;
	}
    }
  is.setstate(std::ios::failbit);
  return is;
}


/***********************************************************************/

// First_idx is the index into the AbstractState that we should use
AdviceTree*
AbstractStateFactor::createAdviceTree(int first_idx)
{
  AdviceTree* ptree = new AdviceTree;
  addAdviceTreeLevels(ptree, &first_idx);
  ptree->createTreeFromLevels();
  return ptree;
}

/***********************************************************************/
// calculates the global state index for this factor and this abstract state
// pfirst_idx is an IO param for where in the state we are
// just returns the single element
int
AbstractStateFactorLeaf::calcStateIdx(const AbstractState& state, int* pfirst_idx) const
{
  return state.getFactorIdx((*pfirst_idx)++);
}

// sets the abstract state vals given a global state index
// just sets the element in the state
// and advances pfirst_idx by one
void
AbstractStateFactorLeaf::setStateForIdx(int global_idx, AbstractState* pstate, int* pfirst_idx)
{
  pstate->setFactorIdx(*pfirst_idx, global_idx);
  (*pfirst_idx)++;
}

rcss::clang::Cond*
AbstractStateFactorLeaf::createCondition(const AbstractState* pstate,
					 int* pfirst_idx)
{
  int myval = pstate->getFactorIdx(*pfirst_idx);
  (*pfirst_idx)++;

  return createCondition(myval);
}


void
AbstractStateFactorLeaf::addAdviceTreeLevels(AdviceTree* ptree, int* pfirst_idx)
{
  ptree->addLevel( AdviceTree::LevelInfo(this, *pfirst_idx) );
  (*pfirst_idx)++;
}


/***********************************************************************/
AbstractStateFactorWFilters::~AbstractStateFactorWFilters()
{
  std::for_each(filters.begin(), filters.end(), deleteptr<AbstractStateFilter>());
}

void
AbstractStateFactorWFilters::print(std::ostream& os) const
{
  os << "StateFilters: " << std::endl;
  for (FilterStorage::const_iterator iter = filters.begin();
       iter != filters.end();
       iter++)
    os << "\t" << **iter << std::endl;
}

bool
AbstractStateFactorWFilters::checkFilters(const WorldState& ws, TeamSide my_side)
{
  for (FilterStorage::iterator iter = filters.begin();
       iter != filters.end();
       iter++)
    {
      if (!(*iter)->isValid(ws, my_side))
	{
	  actionlog(120) << "AbstractStateFactorWFilters::checkFilters: "
			 << (**iter) << " rejected this state" << ende;
	  return false;
	}
    }
  return true;
}

// This is a separate method from the standard draw hierarchy since
// it son't actually consume indices from the AbstractState
void
AbstractStateFactorWFilters::drawFilters(FieldImage* pimg,
					 RegionWorldModelInterface* wmi)
{
  FilterDrawCaller caller(pimg, wmi);
  std::for_each(filters.begin(), filters.end(), caller);
}

void
AbstractStateFactorWFilters::createFilterInitialAdvice(CoachMessageQueue& mqueue)
{
  for (FilterStorage::iterator iter = filters.begin();
       iter != filters.end();
       iter++)
    {
      (*iter)->createInitialAdvice(mqueue);
    }
}


rcss::clang::Cond*
AbstractStateFactorWFilters::createFilterCondition()
{
  rcss::clang::CondAnd* pAnd = new rcss::clang::CondAnd();

  for (FilterStorage::iterator iter = filters.begin();
       iter != filters.end();
       iter++)
    {
      rcss::clang::Cond* p = (*iter)->createCondition();
      if (p)
	pAnd->push_back(std::auto_ptr<rcss::clang::Cond>(p));
    }
  if (pAnd->getConds().empty())
    {
      delete pAnd;
      pAnd = NULL;
    }
  return pAnd;
}



/***********************************************************************/

//returns true if the ball pos was set
bool
AbstractStateFactorWChildren::extractBallPos(VecPosition* pv,
					     const AbstractState* pstate, int *pfirst_idx)
{
  bool changed = false;
  for (FactorStorage::iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    {
      bool thisres = (*iter)->extractBallPos(pv, pstate, pfirst_idx);
      changed = changed || thisres;
    }
  return changed;
}

AbstractStateFactorWChildren::~AbstractStateFactorWChildren()
{
  std::for_each(factors.begin(), factors.end(), deleteptr<AbstractStateFactor>());
}

AbstractStateFactor*
AbstractStateFactorWChildren::getChild(int idx)
{
  if (idx < 0 || idx >= (signed)factors.size())
    {
      errorlog << "getChild: idx out of range: " << idx << ", max=" << factors.size() << ende;
      return NULL;
    }
  return factors[idx];
}

void
AbstractStateFactorWChildren::createInitialAdvice(CoachMessageQueue& mqueue)
{
  createFilterInitialAdvice(mqueue);
  for (FactorStorage::iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    {
      (*iter)->createInitialAdvice(mqueue);
    }
}


/***********************************************************************/

AbstractStateFactorAnd::AbstractStateFactorAnd()
  : AbstractStateFactorWChildren(ASF_And), mult_vals()
{
}

int
AbstractStateFactorAnd::calcLeafCount() const
{
  int sum = 0;
  for (FactorStorage::const_iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    sum += (*iter)->updateLeafCount();
  return sum;
}

// Sets elements in the AbstractState for this factor
// returns the next factor idx to use
int
AbstractStateFactorAnd::setStateFor(AbstractState* pstate, int idx,
				    const WorldState& ws, TeamSide my_side)
{
  bool overall_valid = true;
  //First check if this is valid by our filters
  if (!checkFilters(ws, my_side))
    {
      for (int i = idx; i < idx + getLeafCount(); i++)
	pstate->setFactorIdx(i, -1);
      pstate->setValid(false);
      return idx + getLeafCount();
    }

  for (FactorStorage::const_iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    {
      idx = (*iter)->setStateFor(pstate, idx, ws, my_side);
      if (!pstate->isValid())
	overall_valid = false;
    }
  pstate->setValid(overall_valid);
  return idx;
}

// calculates the global state index for this factor and this abstract state
// pfirst_idx is an IO param for where in the state we are
int
AbstractStateFactorAnd::calcStateIdx(const AbstractState& state, int* pfirst_idx) const
{
  int result = 0;
  bool result_valid = true;
  for (int fac = 0; fac < (signed)factors.size(); fac++)
    {
      int idx = factors[fac]->calcStateIdx(state, pfirst_idx);
      if (idx == -1)
	result_valid = false;
      result += idx * mult_vals[fac];
    }
  return result_valid ? result : -1;
}

// sets the abstract state vals given a global state index
void
AbstractStateFactorAnd::setStateForIdx(int global_idx,
				       AbstractState* pstate, int* pfirst_idx)
{
  std::vector<int> indices(factors.size());
  for (int fac=factors.size()-1; fac >= 0; fac--)
    {
      indices[fac] = global_idx / mult_vals[fac];
      global_idx %= mult_vals[fac];
    }
  for (int fac=0; fac < (signed)factors.size(); fac++)
    {
      factors[fac]->setStateForIdx(indices[fac], pstate, pfirst_idx);
    }
}

void
AbstractStateFactorAnd::print(std::ostream& os) const
{
  os << "AndFactor: " << std::endl;
  AbstractStateFactorWFilters::print(os);
  os << "Factors: " << std::endl;
  for (FactorStorage::const_iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    os << "\t" << **iter << std::endl;
}

int
AbstractStateFactorAnd::draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
			     const AbstractState* pstate, int first_idx)
{
  //pimg->addLegendLine("AND factor");
  drawFilters(pimg, wmi);
  FactorDrawCaller caller(pimg, wmi, pstate, &first_idx);
  std::for_each(factors.begin(), factors.end(), caller);
  return first_idx;
}

rcss::clang::Cond*
AbstractStateFactorAnd::createCondition(const AbstractState* pstate,
					int* pfirst_idx)
{
  // we simply make an AND condition of everything
  // However, if any factor returns a NULL, the whole thing is
  // invalid and we'll throw it away
  // We still have to go through all the factors so that
  // pfirst_idx ends up correct
  bool factor_valid = true;
  rcss::clang::CondAnd* pAnd = new rcss::clang::CondAnd;

  rcss::clang::Cond* pFilterCond = createFilterCondition();
  if (pFilterCond)
    pAnd->push_back(std::auto_ptr<rcss::clang::Cond>(pFilterCond));

  for (FactorStorage::iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    {
      rcss::clang::Cond* p = (*iter)->createCondition(pstate, pfirst_idx);
      if (p)
	{
	  // I"m going to put on the front solely because of the choices
	  // I made in constructing the state space. One of the most expensive evals
	  // is first in the most important AND factor. 
	  pAnd->push_front(std::auto_ptr<rcss::clang::Cond>(p));
	}
      else
	{
	  factor_valid = false;
	}
    }

  if (!factor_valid)
    {
      delete pAnd;
      pAnd = NULL;
    }

  return pAnd;
}

void
AbstractStateFactorAnd::addAdviceTreeLevels(AdviceTree* ptree, int* pfirst_idx)
{
  for (FactorStorage::iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    {
      (*iter)->addAdviceTreeLevels(ptree, pfirst_idx);
    }
}


void
AbstractStateFactorAnd::updateMultFactor()
{
  int fac = 1;
  mult_vals.resize(factors.size());

  for (FactorStorage::iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    {
      mult_vals[iter-factors.begin()] = fac;
      fac *= (*iter)->getNumStates();
    }
  actionlog(200) << "AbstractStateFactorAnd: Updated mult_vals: " << mult_vals << ende;
}


/***********************************************************************/

AbstractStateFactorOr::AbstractStateFactorOr()
  : AbstractStateFactorWChildren(ASF_Or), add_vals()
{
}

int
AbstractStateFactorOr::calcLeafCount() const
{
  int sum = 0;
  for (FactorStorage::const_iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    sum += (*iter)->updateLeafCount();
  return sum;
}

// Sets elements in the AbstractState for this factor
// returns the next factor idx to use
int
AbstractStateFactorOr::setStateFor(AbstractState* pstate, int idx,
				    const WorldState& ws, TeamSide my_side)
{
  //First check if this is valid by our filters
  if (!checkFilters(ws, my_side))
    {
      for (int i = idx; i < idx + getLeafCount(); i++)
	pstate->setFactorIdx(i, -1);
      pstate->setValid(false);
      return idx + getLeafCount();
    }

  bool overall_valid = false;
  for (FactorStorage::const_iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    {
      idx = (*iter)->setStateFor(pstate, idx, ws, my_side);
      //all we need is one state to be valid
      if (pstate->isValid())
	overall_valid = true;
    }
  pstate->setValid(overall_valid);
  return idx;
}


// calculates the global state index for this factor and this abstract state
// pfirst_idx is an IO param for where in the state we are
int
AbstractStateFactorOr::calcStateIdx(const AbstractState& state, int* pfirst_idx) const
{
  int result = -1;
  for (int fac = 0; fac < (signed)factors.size(); fac++)
    {
      int idx = factors[fac]->calcStateIdx(state, pfirst_idx);
      if (idx != -1 && result < 0)
	result = idx + add_vals[fac];
    }
  return result;
}

// sets the abstract state vals given a global state index
void
AbstractStateFactorOr::setStateForIdx(int global_idx,
				       AbstractState* pstate, int* pfirst_idx)
{
  bool found_target = false;
  for (int fac=0; fac < (signed)factors.size(); fac++)
    {
      if (found_target ||
	  global_idx >= add_vals[fac] + factors[fac]->getNumStates())
	{
	  for (int i=factors[fac]->getLeafCount(); i > 0; i--)
	    {
	      pstate->setFactorIdx(*pfirst_idx, -1);
	      (*pfirst_idx)++;
	    }
	  continue;
	}
      found_target = true;
      factors[fac]->setStateForIdx(global_idx - add_vals[fac], pstate, pfirst_idx);
    }
  if (!found_target)
    errorlog << "Or: setStateForIdx: I never found my target! " << global_idx << ende;
}

void
AbstractStateFactorOr::print(std::ostream& os) const
{
  //os << "OrFactor: " << std::endl;
  AbstractStateFactorWFilters::print(os);
  os << "Factors: " << std::endl;
  for (FactorStorage::const_iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    os << "\t" << **iter << std::endl;
}


int
AbstractStateFactorOr::draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
			    const AbstractState* pstate, int first_idx)
{
  pimg->addLegendLine("OR factor");
  drawFilters(pimg, wmi);
  FactorDrawCaller caller(pimg, wmi, pstate, &first_idx);
  std::for_each(factors.begin(), factors.end(), caller);
  return first_idx;
}

rcss::clang::Cond*
AbstractStateFactorOr::createCondition(const AbstractState* pstate,
					int* pfirst_idx)
{
  // Here, we just need to take the first condition that is not NULL
  // This is not exactly right since we really want something about the
  // not of all the previous conditions
  // However, for what I am doing here, this is right
  // We'll also and it with the filter conditions if that
  // returns anything
  // We still have to go through all the factors so that
  // pfirst_idx ends up correct

  // this stores the real condition
  rcss::clang::Cond* pFactorCond = NULL;

  for (FactorStorage::iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    {
      rcss::clang::Cond* p = (*iter)->createCondition(pstate, pfirst_idx);
      if (p && pFactorCond == NULL)
	{
	  // found our factor!
	  pFactorCond = p;
	}
      else
	{
	  delete p;
	}
    }

  // check if we found anything
  if (pFactorCond == NULL)
    return NULL;

  // handle filters
  rcss::clang::Cond* pFilterCond = createFilterCondition();
  if (pFilterCond)
    {
      rcss::clang::CondAnd* pAnd = new rcss::clang::CondAnd;
      pAnd->push_back(std::auto_ptr<rcss::clang::Cond>(pFilterCond));
      pAnd->push_back(std::auto_ptr<rcss::clang::Cond>(pFactorCond));
      return pAnd;
    }

  return pFactorCond;
}

void
AbstractStateFactorOr::addAdviceTreeLevels(AdviceTree* ptree, int* pfirst_idx)
{
  errorlog << "Cannot create advice tree with an OR factor" << ende;
}


void
AbstractStateFactorOr::updateAddVals()
{
  int val = 0;
  add_vals.resize(factors.size());

  for (FactorStorage::iterator iter = factors.begin();
       iter != factors.end();
       iter++)
    {
      add_vals[iter-factors.begin()] = val;
      val += (*iter)->getNumStates();
    }
  actionlog(200) << "AbstractStateFactorOr: Updated add_vals: " << add_vals << ende;
}

/***********************************************************************/

void
AbstractStateDescription::setFactor(AbstractStateFactor* p)
{
  if (pfactor)
    delete pfactor;
  pfactor = p;
}

// If pstate is NULL, a new one is allocated
AbstractState* 
AbstractStateDescription::getStateForWorldState(const WorldState& ws,
						TeamSide my_side,
						AbstractState* pstate)
{
  if (pstate == NULL)
    {
      pstate = new AbstractState(this);
    }

#ifdef USE_SANITY_CHECKS
  if (pstate->getStateDescription() != this)
    {
      errorlog << "getStateForWorldState: different abstract state descriptionss "
	       << pstate->getStateDescription() << " " << this
	       << ende;
      return pstate;
    }
#endif

  int last_idx = pfactor->setStateFor(pstate, 0, ws, my_side);
#ifdef USE_SANITY_CHECKS
  if (last_idx != pstate->getStorageSize())
    errorlog << "getStateForWorldState: didn't set all the elements! "
	     << last_idx << " " << pstate->getStorageSize() << ende;
#endif  
  return pstate;

}

AdviceTree*
AbstractStateDescription::createAdviceTree()
{
#ifdef USE_SANITY_CHECKS
  if (pfactor == NULL)
    {
      errorlog << "AbstractStateDescription::createAdviceTree: NULL pfactor" << ende;
      return NULL;
    }
#endif
  return pfactor->createAdviceTree(0);
}

std::ostream&
operator<< (std::ostream &os, const AbstractStateDescription& d)
{
  os << "AbstractStateDescription: " << std::endl;
  os << *d.pfactor;
  return os;
}


/***********************************************************************/

int
AbstractState::getFactorIdx(int fac) const
{
#ifdef USE_SANITY_CHECKS
  if (pdesc == NULL)
    {
      errorlog << "AbstractState::getFactorIdx: pdesc is NULL" << ende;
      return -1;
    }
  if (fac < 0 || fac >= (signed)idx_storage.size())
    {
      errorlog << "AbstractState::getFactorIdx: fac out of range: "
	       << fac << " " << idx_storage.size() << ende;
      return -1;
    }
#endif
  return idx_storage[fac];
}

void
AbstractState::setFactorIdx(int fac, int idx)
{
#ifdef USE_SANITY_CHECKS
  if (pdesc == NULL)
    {
      errorlog << "AbstractState::setFactorIdx: pdesc is NULL" << ende;
      return;
    }
  if (fac < 0 || fac >= (signed)idx_storage.size())
    {
      errorlog << "AbstractState::setFactorIdx: fac out of range: "
	       << fac << " " << idx_storage.size() << ende;
      return;
    }
#endif	
  idx_storage[fac] = idx;
}

int
AbstractState::getStateIdx() const
{
#ifdef USE_SANITY_CHECKS
  if (pdesc == NULL)
    {
      errorlog << "AbstractState::getStateIdx: pdesc is NULL" << ende;
      return -1;
    }
  if (pdesc->getFactor() == NULL)
    {
      errorlog << "AbstractState::getStateIdx: factor is NULL" << ende;
      return -1;
    }
#endif
  int fac_idx = 0;
  int result = pdesc->getFactor()->calcStateIdx(*this, &fac_idx);
#ifdef USE_SANITY_CHECKS
  if (fac_idx != (signed)idx_storage.size())
    errorlog << "getStateIdx: calcStateIdx didn't use all my indices "
	     << fac_idx << " " << idx_storage.size()
	     << ende;
#endif
  return result;
}

void
AbstractState::setStateIdx(int idx)
{
#ifdef USE_SANITY_CHECKS
  if (pdesc == NULL)
    {
      errorlog << "AbstractState::setStateIdx: pdesc is NULL" << ende;
      return;
    }
  if (pdesc->getFactor() == NULL)
    {
      errorlog << "AbstractState::setStateIdx: factor is NULL" << ende;
      return;
    }
#endif	
  int fac_idx = 0;
  pdesc->getFactor()->setStateForIdx(idx, this, &fac_idx);
#ifdef USE_SANITY_CHECKS
  if (fac_idx != (signed)idx_storage.size())
    errorlog << "setStateIdx: setStateForIdx didn't use all my indices "
	     << fac_idx << " " << idx_storage.size()
	     << ende;
#endif
}

void
AbstractState::updateStorageSize()
{
#ifdef USE_SANITY_CHECKS
  if (pdesc == NULL)
    {
      errorlog << "AbstractState::updateStorageSize: pdesc is NULL" << ende;
      return;
    }
#endif	
  int leaf_count;
  if (pdesc->getFactor() == NULL)
    {
      actionlog(150) << "AbstractState::updateStorageSize: factor is NULL, setting to 0" << ende;
      leaf_count = 0;
    }
  else
    {
      leaf_count = pdesc->getFactor()->getLeafCount();
    }
  
  idx_storage.resize(leaf_count, 0);
}

void
AbstractState::draw(FieldImage* pimg, RegionWorldModelInterface* wmi)
{
#ifdef USE_SANITY_CHECKS
  if (pdesc == NULL)
    {
      errorlog << "AbstractState::draw: pdesc is NULL" << ende;
      return;
    }
  if (pdesc->getFactor() == NULL)
    {
      errorlog << "AbstractState::draw: factor is NULL" << ende;
      return;
    }
#endif
  std::ostringstream label;
  label << "AbstractState idx " << getStateIdx() << ": ";
  std::copy(idx_storage.begin(), idx_storage.end(), std::ostream_iterator<int>(label, " "));
  pimg->addLegendLine(label.str().c_str());
  pdesc->getFactor()->draw(pimg, wmi, this, 0);
}


bool
AbstractState::extractBallPos(VecPosition* pv)
{
#ifdef USE_SANITY_CHECKS
  if (pdesc == NULL)
    {
      errorlog << "AbstractState::extractBallPos: pdesc is NULL" << ende;
      return false;
    }
  if (pdesc->getFactor() == NULL)
    {
      errorlog << "AbstractState::extractBallPos: factor is NULL" << ende;
      return false;
    }
#endif
  int fac_idx = 0;
  bool res = pdesc->getFactor()->extractBallPos(pv, this, &fac_idx);
#ifdef USE_SANITY_CHECKS
  if (fac_idx != (signed)idx_storage.size())
    errorlog << "extractBallPos: didn't use all my indices "
	     << fac_idx << " " << idx_storage.size()
	     << ende;
#endif
  return res;
}

rcss::clang::Cond*
AbstractState::createCondition()
{
#ifdef USE_SANITY_CHECKS
  if (pdesc == NULL)
    {
      errorlog << "AbstractState::createCondition: pdesc is NULL" << ende;
      return false;
    }
  if (pdesc->getFactor() == NULL)
    {
      errorlog << "AbstractState::createCondition: factor is NULL" << ende;
      return false;
    }
#endif
  int fac_idx = 0;
  rcss::clang::Cond* pCond = pdesc->getFactor()->createCondition(this, &fac_idx);
#ifdef USE_SANITY_CHECKS
  if (fac_idx != (signed)idx_storage.size())
    errorlog << "createCondition: didn't use all my indices "
	     << fac_idx << " " << idx_storage.size()
	     << ende;
#endif
  return pCond;
}


std::ostream&
operator<< (std::ostream &os, const AbstractState& s)
{
  copy(s.idx_storage.begin(), s.idx_storage.end(), std::ostream_iterator<int>(os, " "));
  return os;
}


/***********************************************************************/

AbstractStateTracker::AbstractStateTracker(TeamSide my_side,
					   AbstractStateDescription* pdesc,
					   const char* per_cycle_fn)
  : my_side(my_side),
    pdesc(pdesc),
    write_per_cycle(per_cycle_fn != NULL && per_cycle_fn[0] != 0),
    os_per_cycle(),
    state_summary(pdesc->getNumStates(), 0),
    state(pdesc)
{
  if (write_per_cycle)
    {
      os_per_cycle.open(per_cycle_fn);
      if (!os_per_cycle)
	{
	  errorlog << "AbstractStateTracker: could not open per cycle file '"
		   << per_cycle_fn << "'" << ende;
	  write_per_cycle = false;
	}
      else
	{
	  os_per_cycle << "# This is a per cycle file generated by AbstractStateTracker " << std::endl;
	  os_per_cycle << "# Format: <time>: <overall index> " << std::endl;
	  os_per_cycle << "# The PlayerOccupancySet used was" << std::endl;
	  writeCommented(os_per_cycle, *pdesc);
	}      
    }
}

void
AbstractStateTracker::writeSummaryInfo(std::ostream& os)
{
  os << "# This file is the summary info of AbstractStateTracker" << std::endl;
  os << "# Format: <State idx>: <num of times seen>" << std::endl;
  os << "# The AbstractStateTracker used was: " << std::endl;
  writeCommented(os, *pdesc);
  for (unsigned i=0; i < state_summary.size(); i++)
    os << i << ": " << state_summary[i] << std::endl;
}


void
AbstractStateTracker::processWorldState(const WorldState& ws)
{
  if (!pdesc->getStateForWorldState(ws, my_side, &state))
    {
      actionlog(100) << "AbstractStateTracker: the state description rejected this world state, doing nothing" << ende;
      return;
    }

  int idx = state.getStateIdx();

  if (!state.isValid())
    {
      actionlog(100) << "AbstractStateTracker: the state came out as invalid, doing nothing" << ende;
      return;
    }
  
  actionlog(160) << "AbstractStateTracker: state " << state << " has idx " << idx << ende;
  if (write_per_cycle)
    os_per_cycle << ws.getTime() << ": " << idx << std::endl;
  state_summary[idx]++;
}

