/* -*- Mode: C++ -*- */

#include "ABMDP.h"
#include "AbstractStateDifference.h"
#include "AdviceTree.h"
#include "Logger.h"
#include "CoachParam.h"

using namespace spades;

ABMDP::ABMDP()
  : pdesc(NULL), soccer_model(),
    lfilters_state(), lfilters_act(),
    rules_leading_char(0), top_rule_name(""),
    tree_assoc_fn(),
    p_advice_tree(NULL)
{
}

ABMDP::~ABMDP()
{
  soccer_model.forgetStateDescription();

  delete pdesc;
  
  std::for_each(lfilters_state.begin(), lfilters_state.end(),
		deleteptr<SoccerStateFilter>());
  std::for_each(lfilters_act.begin(), lfilters_act.end(),
		deleteptr<SoccerActionFilter>());
  lfilters_state.clear();
  lfilters_act.clear();

  delete p_advice_tree;
}


void
ABMDP::setStateDescription(AbstractStateDescription* p)
{
  pdesc = p;
  soccer_model.setStateDescription(p);
}

bool
ABMDP::readMDPAndQTable(const char* mdpfn, const char* qtablefn)
{
  if (!soccer_model.readMDPFrom(mdpfn))
    {
      errorlog << "ABMDP: could not read mdp from '" << mdpfn << "'" << ende;
      return false;
    }
  if (!soccer_model.readQTableFrom(qtablefn))
    {
      errorlog << "ModAbstract: could not read qtable from '" << qtablefn << "'" << ende;
      return false;
    }
  return true;
}


void
ABMDP::generateAdvice(CoachMessageQueue& mqueue)
{
  std::string leading_str(1, rules_leading_char);

  pdesc->getFactor()->createInitialAdvice(mqueue);

  p_advice_tree = createAdviceTree();
  
  SoccerModelAdviser* padviser = NULL;
  if (CoachParam::instance()->getAbstractABMDPSendAdvice())
    padviser = new SoccerModelAdviserTree(p_advice_tree);
  else
    padviser = new SoccerModelAdviserNull();

  int no_act_count;
  int cnt = soccer_model.adviseFor(lfilters_state, lfilters_act, padviser, &no_act_count);
				   
  actionlog(50) << "ABMDP: advised about " << cnt << " states " << ende;
  std::cout << "ABMDP: Queued up advice for " << cnt << " states " << std::endl;

  warninglog(10) << "ABMDP: no actions in advice for " << no_act_count << " states" << ende;
  
  std::ostream* p_os_tree_assoc = NULL;
  if (tree_assoc_fn.length() > 0)
    {
      std::string thisfn = findReplaceInStr(tree_assoc_fn, "%C", leading_str.c_str());
      p_os_tree_assoc = new std::ofstream(thisfn.c_str());
      if (!p_os_tree_assoc)
	{
	  errorlog << "Could not open file '"
		   << thisfn
		   << "' for tree association file"
		   << ende;
	  delete p_os_tree_assoc;
	  p_os_tree_assoc = NULL;
	}
      
    }
  
  // Now we actually have to send the advice
  std::string root_rule_name = p_advice_tree->createAdvice(&mqueue,
                                                           leading_str.c_str(),
                                                           p_os_tree_assoc);
  if (p_os_tree_assoc)
    {
      delete p_os_tree_assoc;
      p_os_tree_assoc = NULL;
    }

  //Now we'll create a nested rule for the advice tree top rul
  if (CoachParam::instance()->getAbstractABMDPSendAdvice() &&
      !CoachParam::instance()->getAbstractABMDPUseBackChannel() &&
      !top_rule_name.empty())
    {
      rcss::clang::CondAnd* pAnd = new rcss::clang::CondAnd;
      pAnd->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::PlayOn)));
      pAnd->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondBallOwner(true, rcss::clang::UNumSet(rcss::clang::UNum(rcss::clang::UNum::uAll)))));
      rcss::clang::NestedRule* pRule =
	new rcss::clang::NestedRule( std::auto_ptr<rcss::clang::Cond>(pAnd) );
      rcss::clang::RuleIDList idlist;
      idlist.push_back(root_rule_name);
      pRule->getRules().push_back( new rcss::clang::IDListRule(idlist) );

      rcss::clang::DefRule* pDef =
	new rcss::clang::DefRule(top_rule_name,
				 std::auto_ptr<rcss::clang::Rule>(pRule),
				 false);
      mqueue.getDefineContainer().push(pDef);
    }

  delete padviser;
}

void
ABMDP::setState(CoachMessageQueue& mqueue, bool state)
{
  rcss::clang::RuleIDList idlist;
  idlist.push_back(top_rule_name);
  mqueue.getRuleContainer().push(new rcss::clang::ActivateRules(state, idlist));
}


AdviceTree*
ABMDP::createAdviceTree()
{
  // We'll make a pattern to find the factor we want, then create the tree
  // This is a bit of a hack! we saw that the factor we wnat is the one
  // that has a ball owner element under it
  ASDiffPatternChildren* pOr = new ASDiffPatternChildren(ASF_Or);
  ASDiffPatternChildren* pAnd = new ASDiffPatternChildren(ASF_And);
  pAnd->addChild(new ASDiffPatternSimple(ASF_BallOwner));
  pOr->addChild(pAnd);
  if (!pOr->matchTo(pdesc))
    {
      errorlog << "ABMDP::createAdviceTree: How did the pattern not match?"
	       << ende;
      return NULL;
    }

  AbstractStateFactor* p_root_factor = pAnd->getMatchingFactor();

  AdviceTree* p = p_root_factor->createAdviceTree(pAnd->getMatchingFactorIdx());
	
  delete pOr;

  return p;
}

