/* -*-c++-*- */
/*
 *Header:
 *File: utility.h (for C++ & cc)
 *Author: Noda Itsuki
 *Date: 1995/02/24
 *EndHeader:
 */

/*
 *Copyright:

    Copyright (C) 1996-2000 Electrotechnical Laboratory. 
    	Itsuki Noda, Yasuo Kuniyoshi and Hitoshi Matsubara.
    Copyright (C) 2000, 2001 RoboCup Soccer Server Maintainance Group.
    	Patrick Riley, Tom Howard, Daniel Polani, Itsuki Noda,
	Mikhail Prokopenko, Jan Wendler 

    This file is a part of SoccerServer.

    This code is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

#ifndef _UTILITY_H_
#define _UTILITY_H_

#include <cmath>
#include <cstdlib>
#include <string>
#include <sys/time.h>
#include <unistd.h>
#include <string>
#include <vector>
#include <functional>
#include <iterator>
#include <iostream>
#include <sstream>
#include <map>


/*
 *ModifyHistory
 *
 *EndModifyHistory
 */

#define EPS 1.0e-10
#define INFINITE 1.0e10	
//PI is a name conflict with some player stuff
//#define PI M_PI

#include <algorithm> //min and max

#define Abs(x)	(((x) > 0.0) ? (x) : (-(x)) ) 
#define Atan(y,x) (((x==0.0) && (y==0.0)) ? (0.0) : atan2((y),(x)))
#define RAD2DEG	(180.0/M_PI)
#define DEG2RAD (M_PI/180.0)

#define Rad2Deg(a) 	((double)(a) * RAD2DEG)
#define Rad2IDeg(a)	((int)(Rad2Deg(a)))
#define Rad2IDegRound(a)	((int)rint(Rad2Deg(a)))
#define Deg2Rad(a)	((double)(a) * DEG2RAD)

#define Quantize(v,q)	((rint((v)/(q)))*(q))

#define Pow(x)	((x)*(x))

//#include "random.h"

template <class T> int cmp3way(const T& a, const T& b) 
{
  if (a < b) return -1;
  if (b < a) return 1;
  return 0;
}

enum CmpType {
  CT_Less,
  CT_LessEq,
  CT_Eq,
  CT_Greater,
  CT_GreaterEq
};

template <class T> bool cmpByCmpType(const T& a, const T& b, const CmpType& ct)
{
  switch (ct)
    {
    case CT_Less: return a < b;
    case CT_LessEq: return a <= b;
    case CT_Eq: return a == b;
    case CT_Greater: return a > b;
    case CT_GreaterEq: return a >= b;
    }
  return false;
}

// This function provides a thing that acts like [] on a map, but it const safe
// and returns a new object if the key does not exist
// Note that this means that it returns a value, not a reference
template <class Key, class Value>
Value safeConstMapAccess(const std::map<Key, Value>& m, const Key& k)
{
  typename std::map<Key, Value>::const_iterator iter = m.find(k);
  if (iter == m.end())
    return Value();
  return iter->second;
}


#ifdef __cplusplus
extern "C" {
#endif

  
extern unsigned int randomize(void) ;
/*
extern double drand(double l, double h) ;
*/
//pfr: I removed this because it operates in radians, which we never do
//extern double normalize_angle(double ang) ;

extern int lcm(int a, int b);

#ifdef __cplusplus
} 
#endif

#ifdef X11R5
# define mySetArg_1st(slot,val)	XtSetArg(Args[0], slot, (val)); Argn = 1
# define mySetArg(slot,val)	XtSetArg(Args[Argn], slot, (val)); Argn++
#endif


#define nstonl(x) htonl((long)(((double)(short)ntohs(x) / SHOWINFO_SCALE) * SHOWINFO_SCALE2))
#define nltons(x) htons((short)(((double)(long)ntohl(x) / SHOWINFO_SCALE2) * SHOWINFO_SCALE))


std::string
tildeExpand( const std::string& path_name );


void lowerString(std::string& str);

/* This part came from the coach utility */

//useful for for_each to delete the contents of a container
template<class T> struct deleteptr : public std::unary_function<T*, void>
{
  void operator() (T* x) { delete x; }
};

//useful for for_each to delete the contents of a map container
template<class Map> struct deletemapptr : public std::unary_function<typename Map::value_type, void>
{
  void operator() (typename Map::value_type x) { delete x.second; }
};

template<class T> class derefprinter : public std::unary_function<const T*, void>
{
public:
  derefprinter(std::ostream& os, std::string sep = "") : os(os), sep(sep) {}
  void operator() (const T* x) { os << *x << sep; }
    
private:
  std::ostream& os;
  std::string sep;
};
  


namespace spades
{
  

  std::string tildeExpand (const std::string & path_name);

  /* the starting_file_path is assumed to be the path to a file, so the dirname
     of it is taken unless it ends with /
     the result is an absolute path name, assuming the cwd is the dir of
     starting_file_path (if that is relative, it's resolved relative to the real
     cwd). Note this does NOT require the path exists (though starting_file_path
     must). See also resolvePathGivenStartingFile */
  std::string createAbsPathGivenStartingFile( const char* starting_file_path,
					      const char* path);

  /* the starting_file_path is assumed to be the path to a file, so the dirname
     of it is taken unless it ends with /
     the return is the canonical file path of path, assuming you start from
     the directory of starting_file_path
  Note that this requires that the files exists! */  
  std::string resolvePathGivenStartingFile( const char* starting_file_path,
					    const char* path);

  /* relative paths are resolves relative to the current working directory */
  bool doesFileExist(const char* path);

  //return true iff the first strlen(prefix) characters of str match prefix
  inline bool prefixMatch(const char* prefix, const char* str)
  { return strncmp(prefix, str, strlen(prefix)) == 0; }

  // 0 fills if min_wid > 0
  std::string toString(int i, int min_wid = -1);

  // 0 fills so that everything (in the positive range) has the same width
  std::string toStringGivenMax(int i, int max);

  std::string toString(double d);
  
  // finds and replaces all occurences of find with repl in str
  // operates recursively, so this could make an infinite loop if you
  // are not careful
  std::string findReplaceInStr(const std::string& str, const char* find, const char* repl);
  
  void seedRandom(int seed = -1, bool print = false);

  // returns false if eof
  bool skip_white_space(std::istream& infile) ;
  // returns false if eof or eol
  bool skip_white_space_on_line(std::istream& infile) ;
  //returns false if eof
  bool skip_line(std::istream& infile);
  //returns false if eof or other error
  bool skip_to_character(std::istream& infile, char c);
  // returns FALSE if eof
  bool skip_to_non_comment(std::istream& infile, char comm_char = '#');
  /* advances to the specifed character, but only on the same line */
  bool advance_to_on_line(std::istream& infile, char c);

  /* Set the FD_CLOEXEC flag if value is true or clear the flag if value is false */
  bool set_cloexec_flag (int desc, bool value);

  /* return 0 on error, the character representing the state otherwise */
  char get_process_status(int pid);
  
  //returns a - b
  inline double tv2seconds(struct timeval tv) { return tv.tv_sec + tv.tv_usec * .000001; }
  struct timeval seconds2tv(double s);

  struct timeval operator+(const struct timeval & t1, const struct timeval & t2);
  struct timeval operator-(const struct timeval & t1, const struct timeval & t2);

  inline struct timeval operator+=(struct timeval & t1, const struct timeval & t2)
  { t1 = t1 + t2; return t1; }
  inline struct timeval operator-=(struct timeval & t1, const struct timeval & t2)
  { t1 = t1 - t2; return t1; }
  
  inline bool operator <(const struct timeval & t1, const struct timeval & t2)
  { return timercmp(&t1, &t2, <); }
  inline bool operator <=(const struct timeval & t1, const struct timeval & t2)
  { return timercmp(&t1, &t2, <=); }
  inline bool operator >(const struct timeval & t1, const struct timeval & t2)
  { return timercmp(&t1, &t2, >); }
  inline bool operator >=(const struct timeval & t1, const struct timeval & t2)
  { return timercmp(&t1, &t2, >=); }
  inline bool operator ==(const struct timeval & t1, const struct timeval & t2)
  { return (t1.tv_sec == t2.tv_sec && t1.tv_usec == t2.tv_usec); }
  inline bool operator !=(const struct timeval & t1, const struct timeval & t2)
  { return (t1.tv_sec != t2.tv_sec || t1.tv_usec != t2.tv_usec); }

  std::ostream& operator << (std::ostream& o, const struct timeval & t) ;

  //breaks the strings into arguments which are space delimited
  /* The argument parsing is pretty simplistic. The rules are quite simple
     (space) spearates arguments
     " can be used to bound an argument, i.e. get a space in an argument name
     The " will not be passed to the executable
     If you want to include a " in a string, you must include that argument in " "
     and use "" to get a single "
     No other characters are special
     Note in particular that this means that \ does not mean anything special, nor
     can any shell style redirection or anything like that be done
     Example:
     foo -a b --code \\
     gives 'foo', '-a', 'b', '--code', '\\'
     bar -a "Some Name"
     gives 'bar', '-a', 'Some Name'
     baz -a "Some""Name"
     gives 'bar', '-a', 'Some"Name'
     bif -a """SomeName"""
     gives 'bif', '-a', '"Some Name"'

     end is the character one after the last one we are to process
  */
  bool breakIntoArgs(std::string s,
		     std::string::size_type begin, std::string::size_type end,
		     std::vector<char*>& vArgs);
  inline bool breakIntoArgs(std::string s,
		     std::vector<char*>& vArgs)
  { return breakIntoArgs(s, 0, s.size(), vArgs); }

  /* returns an idx such that begin - return (not including the character at return
     is a brace balanced string which includes at least
     1 character of string s
     end is one character after the last one to consider
     Returns string::npas if not such substring can be found */
  std::string::size_type findBraceBalanced(std::string s,
					   std::string::size_type begin,
					   std::string::size_type end);
  inline std::string::size_type findBraceBalanced(std::string s,
						  std::string::size_type begin)
  { return findBraceBalanced(s, begin, s.size()); }
  inline std::string::size_type findBraceBalanced(std::string s)
  { return findBraceBalanced(s, 0, s.size()); }
  
  struct cstr_lt : public std::binary_function <const char *, const char *, bool>
  {
    bool operator () (const char * s1, const char * s2)
    { return strcmp(s1, s2) < 0; }
  };

  template<class T> void readVector(std::istream& is, std::vector<T>& v)
  {
    v.clear();
    while (is.good())
      {
	T val;
	is >> val;
	if (is.fail())
	  break;
	v.push_back(val);
      }
  }

  template<class T> std::ostream & operator << (std::ostream & o, const std::vector<T>& v)
  { std::copy(v.begin(), v.end(), std::ostream_iterator<T>(o, " ")); return o; }
  
  template<class T> bool readVectorWBounds(std::istream& is, std::vector<T>& v, int min, int max)
  {
    readVector(is, v);
    return ((signed)v.size() >= min && (max < 0 || (signed)v.size() <= max));
  }

  //Only maintains info for mean
  class MeanSummary 
  {
  public:
    MeanSummary() { reset(); }
    MeanSummary(const MeanSummary& ms) : n(ms.n), sum(ms.sum) { }
    MeanSummary(int new_n, double new_sum)
    { n = new_n; sum = new_sum; }

    void reset();
  
    void addPoint(double x, int count = 1);
    double getMean() const;
    int getNumPoints() const { return n; }

    void addIn(const MeanSummary& ms) { n+=ms.n; sum+=ms.sum; }

    friend std::ostream& operator<<(std::ostream& os, const MeanSummary& ms)
    { os << "MeanSummary(n:" << ms.n << ", sum:" << ms.sum << ")"; return os; }
      
  private:
    int n;
    double sum;
  } ;
 
} //spades namespace

template<class X>
void writeCommented(std::ostream& os, const X& x, char comment_ch = '#')
{
  os << comment_ch << ' ';
  std::ostringstream text;
  text << x;
  std::string repl("\n");
  repl += comment_ch;
  repl += ' ';
  os << spades::findReplaceInStr(text.str(), "\n", repl.c_str());
  os << std::endl;
}

#endif
