/* -*- Mode: C -*-
 *Header:
 *File: utility.C 
 *Author: Noda Itsuki
 *Date: 1995/02/24
 *EndHeader:
 */

/*
 *Copyright:

    Copyright (C) 1996-2000 Electrotechnical Laboratory. 
      Itsuki Noda, Yasuo Kuniyoshi and Hitoshi Matsubara.
    Copyright (C) 2000, 2001 RoboCup Soccer Server Maintainance Group.
      Patrick Riley, Tom Howard, Daniel Polani, Itsuki Noda,
  Mikhail Prokopenko, Jan Wendler 

    This file is a part of SoccerServer.

    This code is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

#include <cstring>
#include <math.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <pwd.h>
#include <strstream>
#include <math.h>
#include <iomanip>
#include <string>
#include <libgen.h>
#include <unistd.h>
#include <fcntl.h>
#include <algorithm>
#include "param.h"
#include "types.h"
#include "utility.h"
#include "utility.h"
#include "Logger.h"

/*
 *===================================================================
 *Part:      Random number utility
 *===================================================================
 */
unsigned int randomize(void)
{
  static unsigned int l ;

  static struct timeval Tv ;
  static struct timezone Tz ;
  gettimeofday( &Tv , &Tz ) ;
  l = (unsigned long)(Tv.tv_usec) ;
  srandom( l ) ;
  return l ;
}


/*
 *===================================================================
 *Part:      Angle
 *===================================================================
 */
/*
pfr: I removed this because it operates in radians, which nothing else does
double normalize_angle(double ang)
{
  while(1) {
    if (ang < -M_PI)
      ang += 2 * M_PI ;
    else if (ang > M_PI)
      ang -= 2 * M_PI ;
    else
      break ;
  }
  return ang ;
}
*/

/*
 *==================================================================
 *Part:     Lowest Common Multiple
 *==================================================================
 */
int lcm (int a, int b)
{
  int tmp = 0, idx = 0, larger = std::max(a, b);
  do {
    idx++;
    tmp = larger * idx;
  } while (tmp % a != 0 || tmp % b != 0);
  return tmp;
}


/*
 *==================================================================
 *Part:     Tilde Expand
 *==================================================================
 */
// This will expand ~ and ~user at the beginning of the path specified
// in path_name.  No attempt is made to check or or canonize the path.
//
// Throws when an OS API gives us something unusable like a NULL pointer.

std::string tildeExpand( const std::string& path_name )
{
  // There must be a ~ at the start of the path for a valid
  // expansioin.
  if (path_name.length() == 0 || path_name[0] != '~')
  {
    return path_name;
  }

  std::string newPath;    // Used to store the new ~ expanded path.
  std::string username;  // Used to store user name of interest.

  if (path_name.length() == 1 || path_name[1] == '/')
  {
      // Get the current user.
      char* err = getenv("USER");
      if( err == NULL )
      {
          // On Windows USERNAME is used instead
          err = getenv( "USERNAME" );
          if( err == 0 )
              return path_name;
      }

      username = err;

      // if succeeded, remove the tilde
      newPath = path_name.substr( 1, path_name.length() );	
  }
  else
  {
    // Fish out the user name from path_name and remove it
    // from newPath.
    std::string::size_type userEnd = path_name.find('/');
    if (userEnd == std::string::npos)
    {
      // No / so whole path must be the username.
      userEnd = path_name.length();
    }
    username = path_name.substr(1, userEnd - 1);
    newPath = path_name.substr(userEnd, path_name.length());
  }

  // Get the passwd file entry for the user and place their home
  // directory path at the start of newPath.
  struct passwd *pwdEntry = getpwnam(username.c_str());
  if (pwdEntry == NULL)
    return path_name;

  newPath.insert(0, pwdEntry->pw_dir);

  return newPath;
}

void
lowerString(std::string& str)
{
  std::transform(str.begin(), str.end(), str.begin(), ::tolower);
}


/*
 *==================================================================
 *Part:     From coach utilty
 *==================================================================
 */


using namespace std;
using namespace spades;

// This function will return a copy of path_name with `~' replaced
// with the value of the `HOME' environment variable
// This means `~someone/foo' wont work.  Feel free to write
// a patch :) 
string
spades::tildeExpand (const string & path_name)
{
  int tilde_pos = 0;
  bool found = false;
  for (int i = 0; i < (signed) path_name.length (); i++)
    if (path_name[i] == '~')
      {
	tilde_pos = i;
	found = true;
	break;
      }

  string path = path_name;

  if (found)
    {
      path.erase (tilde_pos, tilde_pos + 1);
      path.insert (tilde_pos, string (getenv ("HOME")));
    }
  return path;
}

/***********************************************************************/

/* the starting_file_path is assumed to be the path to a file, so the dirname
   of it is taken unless it ends with /
   the result is an absolute path name, assuming the cwd is the dir of
   starting_file_path (if that is relative, it's resolved relative to the real
   cwd). Note this does NOT require the path exists (though starting_file_path
   must). See also resolvePathGivenStartingFile */
std::string
spades::createAbsPathGivenStartingFile( const char* starting_file_path,
					const char* path)
{
  if (path[0] == '/')
    {
      // the path is absolute, we ignore starting_file_path
      return std::string(path);
    }

  //Now we need to get a starting file path that is absolute
  bool start_is_file = (starting_file_path[strlen(starting_file_path)-1] != '/');
  char starting_dir[PATH_MAX];
  char starting_dir_abs[PATH_MAX];
  if (start_is_file)
    {
      // dirname could modify, so we do a silly copy first
      strcpy(starting_dir_abs, starting_file_path);
      strcpy(starting_dir, dirname(starting_dir_abs));
    }
  else
    {
      strcpy(starting_dir, starting_file_path);
    }
  if (realpath(starting_dir, starting_dir_abs) == NULL)
    {
      warninglog(10) << "createAbsPathGivenStartingFile: could not resolve starting_file_path "
		     << "'" << starting_file_path << "'"
		     << errno << ": " << strerror(errno) << ende;
      return std::string("");
    }

  //Now starting_dir should be the right directory. Let's create the final thing now
  std::string ret(starting_dir_abs);
  ret += '/';
  ret += path;

  return ret;
}

/***********************************************************************/
/* the starting_file_path is assumed to be the path to a file, so the dirname
   of it is taken unless it ends with /
   the return is the canonical file path of path, assuming you start from
   the directory of starting_file_path 
   Note that this requires that the files exists! */
std::string
spades::resolvePathGivenStartingFile( const char* starting_file_path,
				      const char* path)
{
  std::string preresolve = createAbsPathGivenStartingFile(starting_file_path, path);
  if (preresolve.empty())
    return preresolve;

  char postresolve[PATH_MAX];
  char* res = realpath(preresolve.c_str(), postresolve);
  if (res == NULL)
    {
      /*
      warninglog(10) << "Could not resolve path name '" << preresolve << "': "
		     << errno << ' ' << strerror(errno)
		     << ende;
      */
      return string("");
    }

  return string(postresolve);
}

/***********************************************************************/
/* relative paths are resolves relative to the current working directory */
bool
spades::doesFileExist(const char* path)
{
  int fd = open( path, O_RDONLY | O_NONBLOCK );
  if (fd != -1)
    {
      close(fd);
      return true;
    }
  actionlog(200) << "doesFileExist failed for '" << path << "' with error "
		 << errno << ": " << strerror(errno) << ende;
  return false;
}

/***********************************************************************/

string
spades::toString(int i, int min_wid)
{
  ostringstream out;
  out.fill('0');
  if (min_wid > 0)
    out << setw(min_wid);
  out << i;
  return out.str();
}

// 0 fills so that everything (in the positive range) has the same width
std::string
spades::toStringGivenMax(int i, int max)
{
  // we add one to account for the fact that e.g. log(100) = 2, but we want 3
  int wid = (int)ceil(log10((double)(max+1)));
  return toString(i, wid);
}

std::string
spades::toString(double d)
{
  ostringstream out;
  out << d;
  return out.str();
}

  
/***********************************************************************/

std::string
spades::findReplaceInStr(const std::string& str, const char* find, const char* repl)
{
  string ret(str);
  int find_len = strlen(find);
  string::size_type last_find_pos = 0;
  for(;;)
    {
      string::size_type start = ret.find(find, last_find_pos + 1);
      if (start == string::npos)
	break;
      ret.replace(ret.begin() + start, ret.begin() + start + find_len, repl);
      last_find_pos = start;
    }
  return ret;
}


/***********************************************************************/
/* reads from /dev/random */
void
spades::seedRandom(int seed, bool print)
{
  if (seed > 0)
    {
      actionlog (50) << "Using given random seed: " << seed << ende;
      if (print)
	cout << "Using given random seed: " << seed << endl;
      srandom((unsigned)seed);
    }
  else
    {
      /* check out 'man 4 random' to see the differences between random and urandom
	 We use urandom to avoid blocking reads */
      ifstream in("/dev/urandom");
      signed seed = 1;
      if (!in)
	{
	  errorlog << "Could not open /dev/urandom" << ende;
	}
      else
	{
	  in.read((char*)&seed, sizeof(signed));
	  if (!in)
	    errorlog << "Could not reed random seed from /dev/urandom" << ende;
	}
      //we do this so that we can fit this into an int
      seed = abs(seed);
      actionlog (50) << "Read new random seed: " << seed << ende;
      if (print)
	cout << "Read new random seed: " << seed << endl;
      srandom(seed);
    }
}

/***********************************************************************/

bool
spades::skip_white_space(istream& infile)
{
  char c;

  while(1) {
    infile.get(c);
    if (!infile) return false;
    if (!isspace(c)) {
      infile.putback(c);
      return true; 
    } 
  } 
} 

bool
spades::skip_white_space_on_line(istream& infile)
{
  char c;

  while(1)
    {
      infile.get(c);
      if (!infile) return false;
      if (c == '\n')
        {
          infile.putback(c);
          return false;
        }
      if (!isspace(c))
        {
          infile.putback(c);
          return true; 
        } 
    } 
} 

//returns false if eof
bool
spades::skip_line(istream& infile)
{
  char c;

  while(1)
    {
      infile.get(c);
      if (!infile) return false;
      if (c == '\n')
        {
          return true;
        }
    } 
}

//returns false if eof or other error
bool
spades::skip_to_character(std::istream& infile, char c)
{
  while (infile.good() && infile.get() != c)
    ;
  return infile.good();
}

bool
spades::skip_to_non_comment(std::istream& infile, char comm_char)
{
  while (1) {
    if (!skip_white_space(infile)) return false;
    if (infile.peek() == comm_char) {
      if (!skip_line(infile)) return false;    
    }
    else 
      break;
  }
  return true;
}

/* advances to the specifed character, but only on the same line */
bool
spades::advance_to_on_line(std::istream& infile, char match_c)
{
  char c;

  while(1) {
    infile.get(c);
    if (!infile) return false;
    if (c == match_c) {
      infile.putback(c);
      return true; 
    }
    if (c == '\n') {
      return false; 
    } 
  } 
}



/***********************************************************************/
/* Set the FD_CLOEXEC flag if value is true or clear the flag if value is false */
bool
spades::set_cloexec_flag (int desc, bool value)
{
  int oldflags = fcntl (desc, F_GETFD, 0);
  /* If reading the flags failed, return error indication now. */
  if (oldflags < 0)
    return false;
  /* Set just the flag we want to set. */
  if (value)
    oldflags |= FD_CLOEXEC;
  else
    oldflags &= ~FD_CLOEXEC;
  /* Store modified flag word in the descriptor. */
  return fcntl (desc, F_SETFD, oldflags) != -1;
}

/***********************************************************************/

char
spades::get_process_status(int pid)
{
  std::string procfn = "/proc/" + toString(pid) + "/stat";
  std::ifstream in(procfn.c_str());

  if (!in)
    {
      actionlog(50) << "Process " << pid << " found gone in get_process_status" << ende;
      return 0;
    }

  //This may not DTRT if there are () in the command line
  int c;
  do
    {
      c = in.get();
      if (c == EOF)
	{
	  errorlog << "Malformatted stat 1" << ende;
	  return 0;
	}
    }
  while (c != ')');

  if (!skip_white_space(in))
    {
      errorlog << "Malformatted stat 2" << ende;
      return 0;
    }
    
  c = in.get();

  if (c == EOF)
    {
      errorlog << "Malformatted stat 3" << ende;
      return 0;
    }

  return c;
}

/***********************************************************************/

struct timeval
spades::operator+(const struct timeval & t1, const struct timeval & t2)
{
  struct timeval res;
  res.tv_sec = t1.tv_sec + t2.tv_sec;
  res.tv_usec = t1.tv_usec + t2.tv_usec;
  if (res.tv_usec > 0)
    {
      res.tv_sec += res.tv_usec / 1000000;
      res.tv_usec %= 1000000;
    }
  return res;
}

struct timeval
spades::operator-(const struct timeval & t1, const struct timeval & t2)
{
  struct timeval res;
  res.tv_sec = t1.tv_sec - t2.tv_sec;
  res.tv_usec = t1.tv_usec - t2.tv_usec;
  if (res.tv_usec < 0)
    {
      res.tv_sec -= 1;
      res.tv_usec += 1000000;
    }
  return res;
}

struct timeval
spades::seconds2tv(double s)
{
  struct timeval tv;
  if (s < 0)
    errorlog << "Can not convert time < 0 to timeval " << s << ende;
  tv.tv_sec = (long)floor(s);
  tv.tv_usec = (long)(rint(fmod(s, 1.0) / .000001));
  return tv;
}


std::ostream&
spades::operator<< (std::ostream& o, const struct timeval & t) 
{
  o << t.tv_sec << "." << std::setfill('0') << std::setw(6) << t.tv_usec;
  return o;
}


/***********************************************************************/
bool
spades::breakIntoArgs(string s,
		      std::string::size_type begin, std::string::size_type end,
		      vector<char*>& vArgs)
{
  const char* space_chars = " \f\n\r\t\v";

  string::size_type idx;

  idx = begin;      

  while (idx < s.size())
    {
      string::size_type arg_start = idx;
      //end will be the character one past what we want to copy
      string::size_type arg_end = 0;
      //we need this because the presence of quotes affects where the next argument starts
      string::size_type next_arg_start = 0;
	  
      /* first, let's find the extent of this argument */
      while (arg_start < end && isspace(s[arg_start]))
	arg_start++;

      if (arg_start >= end)
	break;

      if (s[arg_start] == '"')
	{
	  /* this is a quote separated argument */
	  arg_start++;
	  arg_end = arg_start;
	  for (;;)
	    {
	      arg_end = s.find('"', arg_end);
	      if (arg_end == string::npos)
		{
		  warninglog(10) << "No trailing quote in: " << s << ende;
		  return false;
		}

	      /* check whether this is "" duple */
	      if (arg_end == end-1)
		break;

	      if (s[arg_end+1] != '"')
		break;

	      /* it is, so we skip over it */
	      arg_end += 2;
	    }

	  // start after the last quote
	  next_arg_start = arg_end + 1;
	}
      else
	{
	  /* this is a normal space separated argument */
	  arg_end = s.find_first_of(space_chars, arg_start);
	  if (arg_end == string::npos) //reached end of string
	    arg_end = end;
	  next_arg_start = arg_end;
	}

      string arg(s, arg_start, arg_end-arg_start);
      //cout << "I found arg: '" << arg << "'" << endl;
	  
      /* Now, replace all the "" duples with " */
      string::size_type quote_idx = 0;
      while ( (quote_idx = arg.find("\"\"", quote_idx)) != string::npos)
	{
	  //this removes one of the quotes, leaving the other
	  arg.erase(quote_idx, 1);
	  quote_idx += 1;
	}

      /* Now create a c style string for this argument */
      char* carg = new char[arg.size() + 1];
      strcpy(carg, arg.c_str());
      vArgs.push_back(carg);

      /* increment for next arg */
      idx = next_arg_start;
    }

  return true;
}

/***********************************************************************/

/* returns an idx such that begin - return (not including the character at return
     is a brace balanced string which includes at least
     1 character of string s
     end is one character after the last one to consider
     Returns string::npas if not such substring can be found */
std::string::size_type
spades::findBraceBalanced(string s,
			  std::string::size_type begin,
			  std::string::size_type end)
{
  //we just want to return 1 character
  if (s[begin] != '{')
    {
      if (end > begin)
	return begin+1;
      return string::npos;
    }

  int depth = 1;
  string::size_type idx;
  begin++;
  
  for (;;)
    {
      if (depth == 0)
	return begin;

      if (begin >= end)
	return string::npos;
      
      idx = s.find_first_of("{}", begin);
      //no brace balanced expr
      if (idx == string::npos || idx >= end)
	return string::npos;

      if (s[idx] == '{')
	depth++;
      else if (s[idx] == '}')
	depth--;
      else
	errorlog << "What is s[idx]? '" << s[idx] << "'" << ende;

      begin = idx+1;
    }

  errorlog << "How did I get here?";
  return string::npos;
}



/***********************************************************************/
//Only maintains info for mean
void
spades::MeanSummary::reset()
{
  sum = 0.0;
  n = 0;
}
  
void
spades::MeanSummary::addPoint(double x, int count) 
{
  sum += x * count;
  n += count;
}

double
spades::MeanSummary::getMean() const
{
  if (n == 0) {
    errorlog << "Tried to get mean with 0 points!" << ende;
    return 0.0;
  } 
  return sum / n;
}
