/* -*- Mode: c++ -*- */

/*
 *Copyright:

    Copyright (C) 1996-2000 Electrotechnical Laboratory. 
    	Itsuki Noda, Yasuo Kuniyoshi and Hitoshi Matsubara.
    Copyright (C) 2000, 2001 RoboCup Soccer Server Maintainance Group.
    	Patrick Riley, Tom Howard, Daniel Polani, Itsuki Noda,
	Mikhail Prokopenko, Jan Wendler 

    This file is a part of SoccerServer.

    This code is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

#include <cstring>

#include "serverparam.h"
#include <string>
#include "config.h"

#define name_of(var)	(#var)
#define str_of(var)	(string(#var))


const std::string ServerParam::LANDMARK_FILE = "~/.rcssserver-landmark.xml";
const std::string ServerParam::SERVER_CONF = "~/.rcssserver-server.conf";

const int ServerParam::SEND_COMMS = false;

const int ServerParam::TEXT_LOGGING = true;
const int ServerParam::GAME_LOGGING = true;
const int ServerParam::GAME_LOG_VERSION = 3;
const std::string ServerParam::TEXT_LOG_DIR = "./";
const std::string ServerParam::GAME_LOG_DIR = "./";
const std::string ServerParam::TEXT_LOG_FIXED_NAME = "rcssserver";
const std::string ServerParam::GAME_LOG_FIXED_NAME = "rcssserver";
const int ServerParam::TEXT_LOG_FIXED = false;
const int ServerParam::GAME_LOG_FIXED = false;
const int ServerParam::TEXT_LOG_DATED = true;
const int ServerParam::GAME_LOG_DATED = true;
const std::string ServerParam::LOG_DATE_FORMAT = "%Y%m%d%H%M-";
const int ServerParam::LOG_TIMES = false;
const int ServerParam::RECORD_MESSAGES = false;
const int ServerParam::TEXT_LOG_COMPRESSION = 0;
const int ServerParam::GAME_LOG_COMPRESSION = 0;
const bool ServerParam::PROFILE = false;

const int ServerParam::KAWAY_LOGGING = true;
const std::string ServerParam::KAWAY_LOG_DIR = "./";
const std::string ServerParam::KAWAY_LOG_FIXED_NAME = "rcssserver";
const int ServerParam::KAWAY_LOG_FIXED = false;
const int ServerParam::KAWAY_LOG_DATED = true;

const int ServerParam::KAWAY_START = -1;

const int ServerParam::POINT_TO_BAN = 5;
const int ServerParam::POINT_TO_DURATION= 20;
const unsigned int ServerParam::SAY_MSG_SIZE = 10;
const unsigned int ServerParam::HEAR_MAX = 1;
const unsigned int ServerParam::HEAR_INC = 1;
const unsigned int ServerParam::HEAR_DECAY = 1;

const double ServerParam::TACKLE_DIST = 2.5;
const double ServerParam::TACKLE_BACK_DIST = 0.5;
const double ServerParam::TACKLE_WIDTH = 1.25;
const double ServerParam::TACKLE_EXPONENT = 6.0;
const unsigned int ServerParam::TACKLE_CYCLES = 10;
const double ServerParam::TACKLE_POWER_RATE = 0.027;

const unsigned int ServerParam::FREEFORM_WAIT_PERIOD = 600;
const unsigned int ServerParam::FREEFORM_SEND_PERIOD = 20;

const bool ServerParam::FREE_KICK_FAULTS = true;
const bool ServerParam::BACK_PASSES = true;

const bool ServerParam::PROPER_GOAL_KICKS = false;
const double ServerParam::STOPPED_BALL_VEL = 0.01;
const int ServerParam::MAX_GOAL_KICKS = 3;

const int ServerParam::CLANG_DEL_WIN = 1;
const int ServerParam::CLANG_RULE_WIN = 1;

const bool ServerParam::S_AUTO_MODE = false;
const int ServerParam::S_KICK_OFF_WAIT = 100;
const int ServerParam::S_CONNECT_WAIT = 300;
const int ServerParam::S_GAME_OVER_WAIT = 100;

const std::string ServerParam::S_TEAM_L_START = "";
const std::string ServerParam::S_TEAM_R_START = "";

ServerParam::ServerParam() 
{ 
  setDefaults();
  createMaps();
}

ServerParam::~ServerParam()
{
}

void 
ServerParam::createMaps() 
{ 
  add2Maps ( "goal_width", group( &gwidth, 7 ) ); 
  add2Maps ( "player_size", group( &psize, 7 ) ); 
  add2Maps ( "player_decay", group( &pdecay, 7 ) ); 
  add2Maps ( "player_rand", group( &prand, 7 ) ); 
  add2Maps ( "player_weight", group( &pweight, 7 ) ); 
  add2Maps ( "player_speed_max", group( &pspeed_max, 7 ) ); 
  // th 6.3.00
  add2Maps ( "player_accel_max", group( &paccel_max, 7 ) ); 
  //
  add2Maps ( "stamina_max", group( &stamina_max, 7 ) ); 
  add2Maps ( "stamina_inc_max", group( &stamina_inc, 7 ) ); 
  add2Maps ( "recover_dec_thr", group( &recover_dthr, 7 ) ); 
  add2Maps ( "recover_min", group( &recover_min, 7 ) ); 
  add2Maps ( "recover_dec", group( &recover_dec, 7 ) ); 
  add2Maps ( "effort_init", group( &effort_init, 7 ) ); 
  add2Maps ( "effort_dec_thr", group( &effort_dthr, 7 ) ); 
  add2Maps ( "effort_min", group( &effort_min, 7 ) ); 
  add2Maps ( "effort_dec", group( &effort_dec, 7 ) ); 
  add2Maps ( "effort_inc_thr", group( &effort_ithr, 7 ) ); 
  add2Maps ( "effort_inc", group( &effort_inc, 7 ) ); 
  // pfr 8/14/00: for RC2000 evaluation
  add2Maps ( "kick_rand",           group( &kick_rand, 7 ) ); 
  add2Maps ( "team_actuator_noise", group( &team_actuator_noise, 7 ), true ); 
  add2Maps ( "prand_factor_l",      group( &prand_factor_l, 7 ) ); 
  add2Maps ( "prand_factor_r",      group( &prand_factor_r, 7 ) ); 
  add2Maps ( "kick_rand_factor_l",  group( &kick_rand_factor_l, 7 ) ); 
  add2Maps ( "kick_rand_factor_r",  group( &kick_rand_factor_r, 7 ) ); 

  add2Maps ( "ball_size", group( &bsize, 7 ) ); 
  add2Maps ( "ball_decay", group( &bdecay, 7 ) ); 
  add2Maps ( "ball_rand", group( &brand, 7 ) ); 
  add2Maps ( "ball_weight", group( &bweight, 7 ) ); 
  add2Maps ( "ball_speed_max", group( &bspeed_max, 7 ) ); 
  // th 6.3.00
  add2Maps ( "ball_accel_max", group( &baccel_max, 7 ) ); 
  //
  add2Maps ( "dash_power_rate", group( &dprate, 7 ) ); 
  add2Maps ( "kick_power_rate", group( &kprate, 7 )); 
  add2Maps ( "kickable_margin", group( &kmargin, 7 ) ); 
  add2Maps ( "control_radius", group( &ctlradius, 7 ) ); 

  add2Maps ( "catch_probability", group( &catch_prob, 7 ) ); 
  add2Maps ( "catchable_area_l",    group( &catch_area_l, 7 ) ); 
  add2Maps ( "catchable_area_w",    group( &catch_area_w, 7 ) ); 
  add2Maps ( "goalie_max_moves",    group( &goalie_max_moves, 7 ) ); 

  add2Maps ( "maxpower", group( &maxp, 7 ) ); 
  add2Maps ( "minpower", group( &minp, 7 ) ); 
  add2Maps ( "maxmoment", group( &maxm, 7 ) ); 
  add2Maps ( "minmoment", group( &minm, 7 ) ); 
  add2Maps ( "maxneckmoment", group( &maxnm, 7 ) ); 
  add2Maps ( "minneckmoment", group( &minnm, 7 ) ); 
  add2Maps ( "maxneckang", group( &maxn, 7 ) ); 
  add2Maps ( "minneckang", group( &minn, 7 ) ); 
  add2Maps ( "visible_angle", group( &visangle, 7 ) ); 
  add2Maps ( "visible_distance", group( &visdist, 7 ) ); 
  add2Maps ( "audio_cut_dist", group( &audio_dist, 7 ) ); 
  add2Maps ( "quantize_step", group( &dist_qstep, 7 ) ); 
  add2Maps ( "quantize_step_l", group( &land_qstep, 7 ) ); 
#ifdef NEW_QSTEP
  add2Maps ( "quantize_step_dir", group( &dir_qstep, 7 ) ); 
  add2Maps ( "quantize_step_dist_team_l", group( &dist_qstep_l, 7 ) ); 
  add2Maps ( "quantize_step_dist_team_r", group( &dist_qstep_r, 7 ) ); 
  add2Maps ( "quantize_step_dist_l_team_l", group( &land_qstep_l, 7 ) ); 
  add2Maps ( "quantize_step_dist_l_team_r", group( &land_qstep_r, 7 ) ); 
  add2Maps ( "quantize_step_dir_team_l", group( &dir_qstep_l, 7 ) ); 
  add2Maps ( "quantize_step_dir_team_r", group( &dir_qstep_r, 7 ) ); 
#endif
  add2Maps ( "ckick_margin", group( &ckmargin, 7 ) ); 
  add2Maps ( "wind_dir", group( &windir, 7 ) ); 
  add2Maps ( "wind_force", group( &winforce, 7 ) ); 
  add2Maps ( "wind_ang", group( &winang, 7 ) ); 
  add2Maps ( "wind_rand", group( &winrand, 7 ) ); 
  add2Maps ( "inertia_moment", group( &inertia_moment, 7 ) ); 
  add2Maps ( "wind_none", group( &win_no, 7 ) ); 
  add2Maps ( "wind_random", group( &win_random, 7 ) ); 
  add2Maps ( "half_time", group( &half_time, 7 ) ); 
  add2Maps ( "drop_ball_time", group( &drop_time, 7 ) ); 
  add2Maps ( "port", group( &portnum, 8 ) ); 
  add2Maps ( "coach_port", group( &coach_pnum, 8 ) ); 
  add2Maps ( "olcoach_port", group( &olcoach_pnum, 8 ) ); 
  add2Maps ( "say_coach_cnt_max", group( &say_cnt_max, 7 ) ); 
  add2Maps ( "say_coach_msg_size", group( &SayCoachMsgSize, 7 ) ); 
  add2Maps ( "simulator_step", group( &sim_st, 7 ) ); 
  add2Maps ( "send_step", group( &send_st, 7 ) ); 
  add2Maps ( "recv_step", group( &recv_st, 7 ) ); 
  add2Maps ( "sense_body_step", group( &sb_step, 7 ) ); 
  add2Maps ( "say_msg_size", group( &M_say_msg_size, 7 ) ); 
  add2Maps ( "clang_win_size",      group( &clang_win_size, 7 ) ); 
  add2Maps ( "clang_define_win",    group( &clang_define_win, 7 ) ); 
  add2Maps ( "clang_meta_win",      group( &clang_meta_win, 7 ) ); 
  add2Maps ( "clang_advice_win",    group( &clang_advice_win, 7 ) ); 
  add2Maps ( "clang_info_win",      group( &clang_info_win, 7 ) ); 
  add2Maps ( "clang_del_win",      group( &M_clang_del_win, 8 ) ); 
  add2Maps ( "clang_rule_win",      group( &M_clang_rule_win, 8 ) ); 
  add2Maps ( "clang_mess_delay",    group( &clang_mess_delay, 7 ) ); 
  add2Maps ( "clang_mess_per_cycle",      group( &clang_mess_per_cycle, 7 ) ); 
  add2Maps ( "hear_max", group( &M_hear_max, 7 ) ); 
  add2Maps ( "hear_inc", group( &M_hear_inc, 7 ) ); 
  add2Maps ( "hear_decay", group( &M_hear_decay, 7 ) ); 
  add2Maps ( "catch_ban_cycle", group( &cban_cycle, 7 ) ); 
  add2Maps ( "coach", group( &CoachMode, 7 ) ); 
  add2Maps ( "coach_w_referee", group( &CwRMode, 7 ) ); 
  add2Maps ( "old_coach_hear", group( &old_hear, 7 ) ); 
  add2Maps ( "send_vi_step",        group( &sv_st, 7 ) ); 
  add2Maps ( "use_offside", group( &useoffside, 7 ), true ); 
  add2Maps ( "offside_active_area_size", group( &offside_area, 7 ) ); 
  add2Maps ( "forbid_kick_off_offside", group( &kickoffoffside, 7 ), true ); 
  add2Maps ( "verbose", group( &verbose, 8 ), true ); 
//    add2Maps ( "replay", group( &replay, 8 ) ); 
  add2Maps ( "offside_kick_margin", group( &offside_kick_margin, 7 ) ); 
  add2Maps ( "slow_down_factor",        group( &slow_down_factor, 7 ) ); 
  add2Maps ( "synch_mode",              group( &synch_mode, 8 ), true ); //pfr:SYNCH
  add2Maps ( "synch_offset",            group( &synch_offset, 8 ) ); //pfr:SYNCH
  add2Maps ( "synch_micro_sleep",       group( &synch_micro_sleep, 8 ) ); //pfr:SYNCH  

  add2Maps ( "start_goal_l",            group( &start_goal_l, 7 ) ); 
  add2Maps ( "start_goal_r",            group( &start_goal_r, 7 ) ); 

  add2Maps ( "fullstate_l",             group( &fullstate_l, 7 ), true ); 
  add2Maps ( "fullstate_r",             group( &fullstate_r, 7 ), true ); 
  add2Maps ( "slowness_on_top_for_left_team", group( &slowness_on_top_for_left_team, 8 ) ); 
  add2Maps ( "slowness_on_top_for_right_team", group( &slowness_on_top_for_right_team, 8 ) ); 
  add2Maps ( "landmark_file",           group( &landmark_file, 8 ) ); 
  add2Maps ( "send_comms", group( &send_comms, 8 ), true ); 
  add2Maps ( "text_logging", group( &text_logging, 8 ), true ); 
  add2Maps ( "game_logging", group( &game_logging, 8 ), true ); 
  add2Maps ( "game_log_version", group( &game_log_version, 8 ) ); 
  add2Maps ( "text_log_dir", group( &text_log_dir, 8 ) ); 
  add2Maps ( "game_log_dir", group( &game_log_dir, 8 ) ); 
  add2Maps ( "text_log_fixed_name", group( &text_log_fixed_name, 8 ) ); 
  add2Maps ( "game_log_fixed_name", group( &game_log_fixed_name, 8 ) ); 
  add2Maps ( "text_log_fixed", group( &text_log_fixed, 8 ), true ); 
  add2Maps ( "game_log_fixed", group( &game_log_fixed, 8 ), true ); 
  add2Maps ( "text_log_dated", group( &text_log_dated, 8 ), true ); 
  add2Maps ( "game_log_dated", group( &game_log_dated, 8 ), true ); 
  add2Maps ( "log_date_format", group( &log_date_format, 8 ) ); 
  add2Maps ( "log_times", group( &log_times, 8 ), true ); 
  add2Maps ( "record_messages", group( &record_messages, 8 ), true ); 
  add2Maps ( "text_log_compression", group( &text_log_compression, 8 ) ); 
  add2Maps ( "game_log_compression", group( &game_log_compression, 8 ) ); 
  add2Maps ( "profile", group( &M_profile, 8 ), true ); 
  add2Maps ( "point_to_ban", group( &M_point_to_ban, 8 ) ); 
  add2Maps ( "point_to_duration", group( &M_point_to_duration, 8 ) ); 

  add2Maps ( "tackle_dist", group( &M_tackle_dist, 8 ) );
  add2Maps ( "tackle_back_dist", group( &M_tackle_back_dist, 8 ) );
  add2Maps ( "tackle_width", group( &M_tackle_width, 8 ) );
  add2Maps ( "tackle_exponent", group( &M_tackle_exponent, 8 ) );
  add2Maps ( "tackle_cycles", group( &M_tackle_cycles, 8 ) );
  add2Maps ( "tackle_power_rate", group( &M_tackle_power_rate, 8 ) );

  add2Maps ( "freeform_wait_period", group( &M_freeform_wait_period, 8 ) );
  add2Maps ( "freeform_send_period", group( &M_freeform_send_period, 8 ) );

  add2Maps ( "free_kick_faults", group( &M_free_kick_faults, 8 ), true );
  add2Maps ( "back_passes", group( &M_back_passes, 8 ), true );

  add2Maps ( "proper_goal_kicks", group( &M_proper_goal_kicks, 8 ), true );
  add2Maps ( "stopped_ball_vel", group( &M_stopped_ball_vel, 8 ) );
  add2Maps ( "max_goal_kicks", group( &M_max_goal_kicks, 8 ) );

  add2Maps( "auto_mode", group( &M_auto_mode, 9 ), true );
  add2Maps( "kick_off_wait", group( &M_kick_off_wait, 9 ) );
  add2Maps( "connect_wait", group( &M_connect_wait, 9 ) );
  add2Maps( "game_over_wait", group( &M_game_over_wait, 9 ) );

  add2Maps( "team_l_start", group( &M_team_l_start, 9 ) );
  add2Maps( "team_r_start", group( &M_team_r_start, 9 ) );

  add2Maps ( "keepaway", group( &kaway, 9 ), true ); 
  add2Maps ( "keepaway_length", group( &ka_length, 9 ) ); 
  add2Maps ( "keepaway_width", group( &ka_width, 9 ) ); 

  add2Maps ( "keepaway_logging", group( &kaway_logging, 9 ), true ); 
  add2Maps ( "keepaway_log_dir", group( &kaway_log_dir, 9 ) ); 
  add2Maps ( "keepaway_log_fixed_name", group( &kaway_log_fixed_name, 9 ) ); 
  add2Maps ( "keepaway_log_fixed", group( &kaway_log_fixed, 9 ), true );
  add2Maps ( "keepaway_log_dated", group( &kaway_log_dated, 9 ), true ); 

  add2Maps ( "keepaway_start", group( &kaway_start, 9 ) );

};

void
ServerParam::setDefaults()
{
  /* set default parameter */
  gwidth = GOAL_WIDTH ;
  psize = PLAYER_SIZE ;
  pdecay = PLAYER_DECAY ;
  prand = PLAYER_RAND ;
  pweight = PLAYER_WEIGHT ;
  stamina_max = STAMINA_MAX ;
  stamina_inc = STAMINA_INC_MAX ;
  recover_dthr = RECOVERY_DEC_THR ;
  recover_min = RECOVERY_MIN ;
  recover_dec = RECOVERY_DEC ;
  effort_dthr = EFFORT_DEC_THR ;
  effort_min = EFFORT_MIN ;
  effort_dec = EFFORT_DEC ;
  effort_ithr = EFFORT_INC_THR ;
  effort_inc= EFFORT_INC ;
  inertia_moment = IMPARAM ;
  // pfr 8/14/00: for RC2000 evaluation
  kick_rand = KICK_RAND;
  team_actuator_noise = FALSE;
  prand_factor_l = PRAND_FACTOR_L;
  prand_factor_r = PRAND_FACTOR_R;
  kick_rand_factor_l = KICK_RAND_FACTOR_L;
  kick_rand_factor_r = KICK_RAND_FACTOR_R;

  bsize = BALL_SIZE ;
  bdecay = BALL_DECAY ;
  brand = BALL_RAND ;
  bweight = BALL_WEIGHT ;
  bspeed_max = BALL_SPEED_MAX ;
  pspeed_max = PLAYER_SPEED_MAX ;
  // th 6.3.00
  baccel_max = BALL_ACCEL_MAX ;
  paccel_max = PLAYER_ACCEL_MAX ;
  //
  dprate = DASHPOWERRATE ;
  kprate = KICKPOWERRATE ;
  kmargin = KICKABLE_MARGIN ;
  ctlradius = CONTROL_RADIUS ;
  ckmargin = CORNER_KICK_MARGIN ;
  catch_prob = GOALIE_CATCHABLE_POSSIBILITY ;
  catch_area_l = GOALIE_CATCHABLE_AREA_LENGTH ; 
  catch_area_w = GOALIE_CATCHABLE_AREA_WIDTH ;
  goalie_max_moves = GOALIE_MAX_MOVES;
  maxp = MAXPOWER ;
  minp = MINPOWER ;
  maxm = MAXMOMENT ;
  minm = MINMOMENT ;
  maxnm = MAX_NECK_MOMENT ;
  minnm = MIN_NECK_MOMENT ;
  maxn = MAX_NECK_ANGLE ;
  minn = MIN_NECK_ANGLE ;
  visangle = VisibleAngle ;
  visdist = VisibleDistance ;
  half_time = HALF_TIME ;
  drop_time = DROP_TIME ;
  portnum = DEFAULT_PORT_NUMBER ;
  coach_pnum = COACH_PORT_NUMBER ;
  olcoach_pnum = OLCOACH_PORT_NUMBER ;
  say_cnt_max = DEF_SAY_COACH_CNT_MAX ;
  SayCoachMsgSize = DEF_SAY_COACH_MSG_SIZE ;
  sim_st = SIMULATOR_STEP_INTERVAL_MSEC ;
  send_st = UDP_SEND_STEP_INTERVAL_MSEC ;
  recv_st = UDP_RECV_STEP_INTERVAL_MSEC ;
  sb_step = SENSE_BODY_INTERVAL_MSEC ;
  clang_win_size = 300;
  clang_define_win = 1;
  clang_meta_win = 1;
  clang_advice_win = 1;
  clang_info_win = 1;
  M_clang_del_win = CLANG_DEL_WIN;
  M_clang_rule_win = CLANG_RULE_WIN;
  clang_mess_delay = 50;
  clang_mess_per_cycle = 1;
  CoachMode = FALSE ;
  CwRMode = FALSE ;
  old_hear = FALSE ;
  sv_st = SEND_VISUALINFO_INTERVAL_MSEC ;
//    replay[0] = NULLCHAR ;
  audio_dist = AUDIO_CUT_OFF_DIST ;
  cban_cycle = GOALIE_CATCH_BAN_CYCLE ;
  slow_down_factor = 1;
  verbose = FALSE ;

  dist_qstep = DIST_QSTEP ;
  land_qstep = LAND_QSTEP ;
#ifdef NEW_QSTEP
  dir_qstep  = DIR_QSTEP ;
#endif

  windir = WIND_DIR ;
  winforce = WIND_FORCE ;
  winrand = WIND_RAND ;
  win_no = FALSE ;
  win_random = FALSE ;
  useoffside = TRUE ;
  offside_area = OFFSIDE_ACTIVE_AREA_SIZE ;
  kickoffoffside = TRUE; 
  offside_kick_margin = OFFSIDE_KICK_MARGIN ;

  kaway = FALSE ;
  ka_length = KEEPAWAY_LENGTH ;
  ka_width = KEEPAWAY_WIDTH ;

//pfr:SYNCH
  synch_mode = false;
  synch_offset = 60;
  synch_micro_sleep = 1;

  start_goal_l = 0;
  start_goal_r = 0;

  fullstate_l = FALSE;
  fullstate_r = FALSE;

  /*  // skip command name */
  /*  int tmp_argc = argc-1; */
  /*  argv++ ; argc-- ; */

#ifdef NEW_QSTEP
  bool defined_qstep_l = false;
  bool defined_qstep_r = false;
  bool defined_qstep_l_l = false;
  bool defined_qstep_l_r = false;
  bool defined_qstep_dir_l = false;
  bool defined_qstep_dir_r = false;
#endif

  slowness_on_top_for_left_team = 1.0;
  slowness_on_top_for_right_team = 1.0;

  landmark_file = ServerParam::LANDMARK_FILE;


  send_comms = ServerParam::SEND_COMMS;
  text_logging = ServerParam::TEXT_LOGGING;
  game_logging = ServerParam::GAME_LOGGING;
  game_log_version = ServerParam::GAME_LOG_VERSION;
  text_log_dir = ServerParam::TEXT_LOG_DIR;
  game_log_dir = ServerParam::GAME_LOG_DIR;
  text_log_fixed_name = ServerParam::TEXT_LOG_FIXED_NAME;
  game_log_fixed_name = ServerParam::GAME_LOG_FIXED_NAME;
  text_log_fixed = ServerParam::TEXT_LOG_FIXED;
  game_log_fixed = ServerParam::GAME_LOG_FIXED;
  text_log_dated = ServerParam::TEXT_LOG_DATED;
  game_log_dated = ServerParam::GAME_LOG_DATED;
  log_date_format = ServerParam::LOG_DATE_FORMAT;
  log_times = ServerParam::LOG_TIMES;
  record_messages = ServerParam::RECORD_MESSAGES;
  text_log_compression = ServerParam::TEXT_LOG_COMPRESSION;
  game_log_compression = ServerParam::GAME_LOG_COMPRESSION;
  M_profile = ServerParam::PROFILE;

  kaway_logging = ServerParam::KAWAY_LOGGING;
  kaway_log_dir = ServerParam::KAWAY_LOG_DIR;
  kaway_log_fixed_name = ServerParam::KAWAY_LOG_FIXED_NAME;
  kaway_log_fixed = ServerParam::KAWAY_LOG_FIXED;
  kaway_log_dated = ServerParam::KAWAY_LOG_DATED;

  kaway_start = ServerParam::KAWAY_START;
  
  M_point_to_ban = ServerParam::POINT_TO_BAN;
  M_point_to_duration = ServerParam::POINT_TO_DURATION;
  M_say_msg_size = ServerParam::SAY_MSG_SIZE ;
  M_hear_max = ServerParam::HEAR_MAX ;
  M_hear_inc = ServerParam::HEAR_INC ;
  M_hear_decay = ServerParam::HEAR_DECAY ;

  M_tackle_dist = TACKLE_DIST;
  M_tackle_back_dist = TACKLE_BACK_DIST;
  M_tackle_width = TACKLE_WIDTH;
  M_tackle_exponent = TACKLE_EXPONENT;
  M_tackle_cycles = TACKLE_CYCLES;
  M_tackle_power_rate = TACKLE_POWER_RATE;

  M_freeform_wait_period = FREEFORM_WAIT_PERIOD;
  M_freeform_send_period = FREEFORM_SEND_PERIOD;

  M_free_kick_faults = FREE_KICK_FAULTS;
  M_back_passes = BACK_PASSES;

  M_proper_goal_kicks = PROPER_GOAL_KICKS;
  M_stopped_ball_vel = STOPPED_BALL_VEL;
  M_max_goal_kicks = MAX_GOAL_KICKS;

  M_auto_mode = S_AUTO_MODE;
  M_kick_off_wait = S_KICK_OFF_WAIT;
  M_connect_wait = S_CONNECT_WAIT;
  M_game_over_wait = S_GAME_OVER_WAIT;

  M_team_l_start = S_TEAM_L_START;
  M_team_r_start = S_TEAM_R_START;
}

void ServerParam::getOptions(const int& argc, const char * const *argv )
{
  /* first, search option '-sfile' */
  int i ;
  bool file_specified = false;

  for(int i = 1 ; i < argc ; i++)
    {
      if ( strcmp( argv [ i ], "-sfile" ) == 0)
        {
          file_specified = true;
          i++;
          readParams ( argv [ i ] );
        }
    }

  if (!file_specified)
    readParams ( ServerParam::SERVER_CONF );
  

  readOptions ( argc, argv );

#ifdef NEW_QSTEP
  if ( !defined_qstep_l )
    dist_qstep_l = dist_qstep ;
  if ( !defined_qstep_r )
    dist_qstep_r = dist_qstep ;
  if ( !defined_qstep_l_l )
    land_qstep_l = land_qstep ;
  if ( !defined_qstep_l_r )
    land_qstep_r = land_qstep ;
  if ( !defined_qstep_dir_l )
    dir_qstep_l = dir_qstep ;
  if ( !defined_qstep_dir_r )
    dir_qstep_r = dir_qstep ;
#endif

  half_time = half_time * 1000 / sim_st ;
  /*send_st /= 4 ; */
  kickable_area = kmargin + bsize + psize ;
  ctlradius_width = ctlradius - psize ;
  /* apply the slow down factor */
  sim_st  *= slow_down_factor;
  sb_step *= slow_down_factor;
  sv_st   *= slow_down_factor;
  send_st *= slow_down_factor;
  synch_offset *= slow_down_factor; //pfr:SYNCH
  lcm_st = lcm (sim_st, lcm (send_st, lcm (recv_st, lcm (sb_step, sv_st) ) ) );
  lcm_st = lcm (lcm_st, (synch_mode ? synch_offset : 1)); //pfr:SYNCH

  text_log_dir = tildeExpand ( text_log_dir );
  game_log_dir = tildeExpand ( game_log_dir );
  kaway_log_dir = tildeExpand ( kaway_log_dir );
}

void ServerParam::writeConfig ( std::ostream& o )
{
  o << "# Configurations for soccerserver\n";
  o << "# Lines that start '#' are comment lines.\n";
  o << "\n";
  o << "# width of goals\n";
  o << "goal_width: " << gwidth << std::endl;
  o << "\n";
  o << "# size, decay, random parameter, weight, maximum speed of players\n";
  o << "player_size: " << psize << std::endl;
  o << "player_decay: " << pdecay << std::endl;
  o << "player_rand: " << prand << std::endl;
  o << "player_weight: " << pweight << std::endl;
  o << "player_speed_max: " << pspeed_max << std::endl;
  o << "player_accel_max: " << paccel_max << std::endl;
  o << "\n";
  o << "# maximum and recovery step of player's stamina\n";
  o << "stamina_max: " << stamina_max << std::endl;
  o << "stamina_inc_max        : " << stamina_inc << std::endl;
  o << "\n";
  o << "# decriment threshold ,decriment step and minimum of player's recovery\n";
  o << "recover_dec_thr        : " << recover_dthr << std::endl;
  o << "recover_dec: " << recover_dec << std::endl;
  o << "recover_min: " << recover_min << std::endl;
  o << "\n";
  o << "# decriment threshold, decriment step, incriment threshold, incriment step\n";
  o << "# and minimum of player's effort\n";
  o << "effort_dec_thr: " << effort_dthr << std::endl;
  o << "effort_dec: " << effort_dec << std::endl;
  o << "effort_inc_thr: " << effort_ithr << std::endl;
  o << "effort_inc: " << effort_inc << std::endl;
  o << "effort_min: " << effort_min << std::endl;
  o << "\n";
  o << "# maximum, recovery step and decay of player's hear capacity\n";
  o << "hear_max: " << M_hear_max << std::endl;
  o << "hear_inc: " << M_hear_inc << std::endl;
  o << "hear_decay: " << M_hear_decay << std::endl;
  o << "\n";
  o << "# inertia moment of player\n";
  o << "inertia_moment: " << inertia_moment << std::endl;
  o << "\n";
  o << "# interval of sense_body\n";
  o << "sense_body_step: " << sb_step << std::endl;
  o << "\n";
  o << "# goalie catchable area length, width\n";
  o << "catchable_area_l        : " << catch_area_l << std::endl;
  o << "catchable_area_w        : " << catch_area_w << std::endl;
  o << "\n";
  o << "# goalie catchbale probability\n";
  o << "catch_probability       : " << catch_prob << std::endl;
  o << "\n";
  o << "# goalie catch ban cycle\n";
  o << "catch_ban_cycle         : " << cban_cycle << std::endl;
  o << "\n";
  o << "# goalie max moves after catch\n";
  o << "goalie_max_moves        : " << goalie_max_moves << std::endl;
  o << "\n";
  o << "# size, decay, random parameter, weight and maximum speed of a ball\n";
  o << "ball_size: " << bsize << std::endl;
  o << "ball_decay: " << bdecay << std::endl;
  o << "ball_rand: " << brand << std::endl;
  o << "ball_weight: " << bweight << std::endl;
  o << "ball_speed_max: " << bspeed_max << std::endl;
  o << "ball_accel_max: " << baccel_max << std::endl;
  o << "\n";
  o << "# force, direction and random parameter of wind\n";
  o << "wind_force: " << winforce << std::endl;
  o << "wind_dir: " << windir << std::endl;
  o << "wind_rand: " << winrand << std::endl;
  o << "\n";
  o << "# kickable margin. kickable_area = kickable_margin + bsize + psize\n";
  o << "kickable_margin        : " << kmargin << std::endl;
  o << "\n";
  o << "# corner kick margin\n";
  o << "ckick_margin: " << ckmargin << std::endl;
  o << "\n";
  o << "# magnification of power in dash, kick \n";
  o << "dash_power_rate        : " << dprate << std::endl;
  o << "kick_power_rate        : " << kprate << std::endl;
  o << "\n";
  o << "# kick randomeness\n";
  o << "kick_rand: " << kick_rand << std::endl;
  o << "\n";
  o << "# angle of view corn [unit: degree]\n";
  o << "visible_angle        : " << visangle << std::endl;
  o << "\n";
  o << "# audio cut off distance\n";
  o << "audio_cut_dist        : " << audio_dist << std::endl;
  o << "\n";
  o << "# quantize step of distance for move_object, landmark\n";
  o << "quantize_step: " << dist_qstep << std::endl;
  o << "quantize_step_l        : " << land_qstep << std::endl;
  o << "\n";
  o << "# max and min of power in dash and kick\n";
  o << "maxpower: " << maxp << std::endl;
  o << "minpower: " << minp << std::endl;
  o << "\n";
  o << "# max and min of power in turn\n";
  o << "maxmoment: " << maxm << std::endl;
  o << "minmoment: " << minm << std::endl;
  o << "\n";
  o << "# max and min of power in turn_neck\n";
  o << "maxneckmoment: " << maxnm << std::endl;
  o << "minneckmoment: " << minnm << std::endl;
  o << "\n";
  o << "# max and min of neck angle\n";
  o << "maxneckang: " << maxn << std::endl;
  o << "minneckang: " << minn << std::endl;
  o << "\n";
  o << "# Default port number\n";
  o << "port: " << portnum << std::endl;
  o << "\n";
  o << "# Default coach port number\n";
  o << "coach_port: " << coach_pnum << std::endl;
  o << "\n";
  o << "# Default oline coach port number\n";
  o << "olcoach_port: " << olcoach_pnum << std::endl;
  o << "\n";
  o << "# Default upper limit of the number of online coach's message\n";
  o << "say_coach_cnt_max: " << say_cnt_max << std::endl;
  o << "\n";
  o << "# Default upper limit of length of online coach's message\n";
  o << "say_coach_msg_size: " << SayCoachMsgSize << std::endl;
  o << "\n";
  o << "# Parameters for the new coach language\n";
  o << "# time window which controls how many messages can be sent\n";
  o << "clang_win_size          : " << clang_win_size << std::endl;
  o << "# number of messages per window\n";
  o << "clang_define_win        : " << clang_define_win << std::endl;
  o << "clang_meta_win          : " << clang_meta_win << std::endl;
  o << "clang_advice_win        : " << clang_advice_win << std::endl;
  o << "clang_info_win          : " << clang_info_win << std::endl;
  o << "clang_del_win           : " << M_clang_del_win << std::endl;
  o << "clang_rule_win          : " << M_clang_rule_win << std::endl;
  o << "# delay between receipt of message and send to players\n";
  o << "clang_mess_delay        : " << clang_mess_delay << std::endl;
  o << "# maximum number of coach messages sent per cycle\n";
  o << "clang_mess_per_cycle    : " << clang_mess_per_cycle << std::endl;
  o << "# How long after playon before coaches can send freeform\n";
  o << "freeform_wait_period    : " << M_freeform_wait_period << std::endl;
  o << "# How long coach can send freeform for after freeform_wait_period\n";
  o << "freeform_send_period    : " << M_freeform_send_period << std::endl;
  o << "\n";
  o << "# Default interval of online coach's look\n";
  o << "send_vi_step        : " << sv_st << std::endl;
  o << "\n";
  o << "# time step of simulation [unit:msec]\n";
  o << "simulator_step        : " << sim_st << std::endl;
  o << "\n";
  o << "# time step of visual information [unit:msec]\n";
  o << "send_step: " << send_st << std::endl;
  o << "\n";
  o << "# time step of acception of command [unit: msec]\n";
  o << "recv_step: " << recv_st << std::endl;
  o << "\n";
  o << "# length of half of game [unit:sec]\n";
  o << "# (if negative, a game does not stop automatically)\n";
  o << "half_time: " << half_time << std::endl;
  o << "\n";
  o << "# number of cycles to wait until dropping the ball automatically\n";
  o << "# for free kicks, corner kicks and so on\n";
  o << "# (0 means do not drop automatically)\n";
  o << "drop_ball_time        : " << drop_time << std::endl;
  o << "\n";
  o << "# string size of say message [unit:byte]\n";
  o << "say_msg_size        : " << M_say_msg_size << std::endl;
  o << "\n";
  o << "# flag for using off side rule. [on/off]\n";
  o << "use_offside: " << ( useoffside ? "on" : "off" ) << std::endl;
  o << "\n";
  o << "# offside active area size\n";
  o << "offside_active_area_size: " << offside_area << std::endl;
  o << "\n";
  o << "# forbid kick off offside. [on/off] \n";
  o << "forbid_kick_off_offside: " << ( kickoffoffside ? "on" : "off" ) << std::endl;
  o << "\n";
  o << "# flag for verbose mode. [on/off]\n";
  o << "verbose: " << ( verbose ? "on" : "off" ) << std::endl;
  o << "\n";
  o << "# offside kick margin. \n";
  o << "offside_kick_margin: " << offside_kick_margin << std::endl;
  o << "\n";
  o << "\n";
  o << "#coach_w_referee\n";
  o << "\n";
  o << "# how much to slow a player down when it's on the top half of the field\n";
  o << "#\n";
  o << "# 1.0 means no change\n";
  o << "# 2.0 means twice as slow, 10.0 means ten times as slow, etc\n";
  o << "# 0.5 means twice as fast\n";
  o << "# 0.0 and less is invalid.  This is not error checked\n";
  o << "slowness_on_top_for_left_team  : " 
    << slowness_on_top_for_left_team << std::endl;
  o << "slowness_on_top_for_right_team : " 
    << slowness_on_top_for_right_team << std::endl;
  o << "\n";
  o << "synch_mode              : " << ( synch_mode ? "on" : "off" ) << std::endl;
  o << "#synch_micro_sleep      : " << synch_micro_sleep << std::endl;
  o << "\n";
  o << "# Send client commands to monitors. [on/off]\n";
  o << "send_comms: " << ( send_comms ? "on" : "off" ) << std::endl;
  o << "\n";
  o << "# LOGGING\n";
  o << "# game logging. [on/off]\n";
  o << "text_logging: " << ( text_logging ? "on" : "off" ) << std::endl;
  o << "game_logging: " << ( game_logging ? "on" : "off" ) << std::endl;
  o << "# game_log version.\n";
  o << "game_log_version        : " << game_log_version << std::endl;
  o << "# Output directories\n";
  o << "text_log_dir            : " << text_log_dir << std::endl;
  o << "game_log_dir            : " << game_log_dir << std::endl;
  o << "# Fixed output names\n";
  o << "text_log_fixed_name     : " << text_log_fixed_name << std::endl;
  o << "game_log_fixed_name     : " << game_log_fixed_name << std::endl;
  o << "# Use fixed output names\n";
  o << "text_log_fixed          : " << ( text_log_fixed ? "on\n" : "off\n" );
  o << "game_log_fixed          : " << ( game_log_fixed ? "on\n" : "off\n" );
  o << "# Add date stamp to output names\n";
  o << "text_log_dated          : " << ( text_log_dated ? "on\n" : "off\n" );
  o << "game_log_dated          : " << ( game_log_dated ? "on\n" : "off\n" );
  o << "log_date_format         : " << log_date_format << std::endl;
  o << "# Write ms between cycles in text log. [on/off]\n";
  o << "log_times: " << ( log_times ? "on\n" : "off\n" );
  o << "# Record messages in game log\n";
  o << "record_messages         : " << ( record_messages ? "on\n" : "off\n" );
  o << "# Log file compression\n";
  o << "text_log_compression    : " << text_log_compression << std::endl;
  o << "game_log_compression    : " << game_log_compression << std::endl;
  o << "# Write ms taken for each part of a cycle in text log. [on/off]\n";
  o << "profile	                : " << ( M_profile ? "on\n" : "off\n" );
  o << "\n";
  o << "# Point to parameters\n";
  o << "point_to_ban            : " << M_point_to_ban << std::endl;
  o << "point_to_duration       : " << M_point_to_duration << std::endl;
  o << "\n";
  o << "# Tackle parameters\n";
  o << "tackle_dist             : " << M_tackle_dist << std::endl;
  o << "tackle_back_dist        : " << M_tackle_back_dist << std::endl;
  o << "tackle_width            : " << M_tackle_width << std::endl;
  o << "tackle_exponent         : " << M_tackle_exponent << std::endl;
  o << "tackle_cycles           : " << M_tackle_cycles << std::endl;
  o << "tackle_power_rate       : " << M_tackle_power_rate << std::endl;
  o << "\n";
  o << "# Enable free kick fault and back pass detection\n"; 
  o << "free_kick_faults        : " 
    << ( M_free_kick_faults ? "on\n" : "off\n" );
  o << "back_passes             : " 
    << ( M_back_passes ? "on\n" : "off\n" );
  o << "\n";
  o << "# Force proper goal kicks\n";
  o << "proper_goal_kicks       : " 
    << ( M_proper_goal_kicks ? "on\n" : "off\n" );
  o << "stopped_ball_vel        : " << M_stopped_ball_vel << std::endl;
  o << "max_goal_kicks          : " << M_max_goal_kicks << std::endl;
  o << "\n";
  o << "# Enable automatic kick off of games and exit when game is over\n";
  o << "auto_mode               : " << ( M_auto_mode ? "on\n" : "off\n" );
  o << "# Kick off wait is the number of cycles to wait after all the\n";
  o << "# players are connected before the game or half time is kicked off\n";
  o << "# -1 indicates wait forever\n";
  o << "kick_off_wait           : " << M_kick_off_wait << std::endl;
  o << "# Connect wait is the number of cycles to wait for all the players\n";
  o << "# to connect before kicking off\n";
  o << "# -1 indicates wait forever\n";
  o << "connect_wait            : " << M_connect_wait << std::endl;
  o << "# Game over wait is the number of cycles to wait after the end of\n";
  o << "# game before exiting\n";
  o << "# -1 indicates wait forever\n";
  o << "game_over_wait          : " << M_game_over_wait << std::endl;
  o << "\n";
  o << "# Use the following commands to automatically start the teams\n";
  o << "team_l_start            : \"" << M_team_l_start << "\"" << std::endl;
  o << "team_r_start            : \"" << M_team_r_start << "\"" << std::endl;
  o << "\n";
  o << "# Enable fullstate information\n";
  o << "fullstate_l             : " << ( fullstate_l ? "on\n" : "off\n" );
  o << "fullstate_r             : " << ( fullstate_r ? "on\n" : "off\n" );
  o << "\n";
  o << "# Enable keepaway mode\n";
  o << "keepaway             : " << ( kaway ? "on\n" : "off\n" );
  o << "# Keepaway region size\n";
  o << "keepaway_length      : " << ka_length << std::endl;
  o << "keepaway_width       : " << ka_width << std::endl;
  o << "# Automatically start Keepaway match after delay (seconds)\n";
  o << "keepaway_start       : " << kaway_start << std::endl;
  o << "# Keepaway logging. [on/off]\n";
  o << "keepaway_logging         : " << ( kaway_logging ? "on" : "off" ) << std::endl;
  o << "# Output directory\n";
  o << "keepaway_log_dir         : " << kaway_log_dir << std::endl;
  o << "# Fixed output name\n";
  o << "keepaway_log_fixed_name  : " << kaway_log_fixed_name << std::endl;
  o << "# Use fixed output name\n";
  o << "keepaway_log_fixed       : " << ( kaway_log_fixed ? "on\n" : "off\n" );
  o << "# Add date stamp to output name\n";
  o << "keepaway_log_dated       : " << ( kaway_log_dated ? "on\n" : "off\n" );
  o << "\n";
}



server_params_t ServerParam::convertToStruct ()
{
  server_params_t tmp;

  tmp.gwidth = htonl( (long)(SHOWINFO_SCALE2 * gwidth) );
  tmp.inertia_moment = htonl( (long)(SHOWINFO_SCALE2 * inertia_moment) );
  tmp.psize = htonl( (long)(SHOWINFO_SCALE2 * psize) );
  tmp.pdecay = htonl( (long)(SHOWINFO_SCALE2 * pdecay) );
  tmp.prand = htonl( (long)(SHOWINFO_SCALE2 * prand) );
  tmp.pweight = htonl( (long)(SHOWINFO_SCALE2 * pweight) );
  tmp.pspeed_max = htonl( (long)(SHOWINFO_SCALE2 * pspeed_max) );
  tmp.paccel_max = htonl( (long)(SHOWINFO_SCALE2 * paccel_max) );
  tmp.stamina_max = htonl( (long)(SHOWINFO_SCALE2 * stamina_max) );
  tmp.stamina_inc = htonl( (long)(SHOWINFO_SCALE2 * stamina_inc) );
  tmp.recover_init = htonl( (long)(SHOWINFO_SCALE2 * recover_init) );
  tmp.recover_dthr = htonl( (long)(SHOWINFO_SCALE2 * recover_dthr) );
  tmp.recover_min = htonl( (long)(SHOWINFO_SCALE2 * recover_min) );
  tmp.recover_dec = htonl( (long)(SHOWINFO_SCALE2 * recover_dec) );
  tmp.effort_init = htonl( (long)(SHOWINFO_SCALE2 * effort_init) );
  tmp.effort_dthr = htonl( (long)(SHOWINFO_SCALE2 * effort_dthr) );
  tmp.effort_min = htonl( (long)(SHOWINFO_SCALE2 * effort_min) );
  tmp.effort_dec = htonl( (long)(SHOWINFO_SCALE2 * effort_dec) );
  tmp.effort_ithr = htonl( (long)(SHOWINFO_SCALE2 * effort_ithr) );
  tmp.effort_inc = htonl( (long)(SHOWINFO_SCALE2 * effort_inc) );
  tmp.kick_rand = htonl( (long)(SHOWINFO_SCALE2 * kick_rand) );                                
  tmp.team_actuator_noise = htons( (short) team_actuator_noise );                        
  tmp.prand_factor_l = htonl( (long)(SHOWINFO_SCALE2 * prand_factor_l ) );                           
  tmp.prand_factor_r = htonl( (long)(SHOWINFO_SCALE2 * prand_factor_r ) );                           
  tmp.kick_rand_factor_l = htonl( (long)(SHOWINFO_SCALE2 * kick_rand_factor_l ) );                       
  tmp.kick_rand_factor_r = htonl( (long)(SHOWINFO_SCALE2 * kick_rand_factor_r ) );                       
  tmp.bsize = htonl( (long)(SHOWINFO_SCALE2 * bsize) );
  tmp.bdecay = htonl( (long)(SHOWINFO_SCALE2 * bdecay) );
  tmp.brand = htonl( (long)(SHOWINFO_SCALE2 * brand) );
  tmp.bweight = htonl( (long)(SHOWINFO_SCALE2 * bweight) );
  tmp.bspeed_max = htonl( (long)(SHOWINFO_SCALE2 * bspeed_max) );
  tmp.baccel_max = htonl( (long)(SHOWINFO_SCALE2 * baccel_max ) );
  tmp.dprate = htonl( (long)(SHOWINFO_SCALE2 * dprate) );
  tmp.kprate = htonl( (long)(SHOWINFO_SCALE2 * kprate) );
  tmp.kmargin = htonl( (long)(SHOWINFO_SCALE2 * kmargin) );
  tmp.ctlradius = htonl( (long)(SHOWINFO_SCALE2 * ctlradius) );
  tmp.ctlradius_width = htonl( (long)(SHOWINFO_SCALE2 * ctlradius_width) );
  tmp.maxp = htonl( (long)(SHOWINFO_SCALE2 * maxp) );
  tmp.minp = htonl( (long)(SHOWINFO_SCALE2 * minp) );
  tmp.maxm = htonl( (long)(SHOWINFO_SCALE2 * maxm) );
  tmp.minm = htonl( (long)(SHOWINFO_SCALE2 * minm) );
  tmp.maxnm = htonl( (long)(SHOWINFO_SCALE2 * maxnm) );
  tmp.minnm = htonl( (long)(SHOWINFO_SCALE2 * minnm) );
  tmp.maxn = htonl( (long)(SHOWINFO_SCALE2 * maxn) );
  tmp.minn = htonl( (long)(SHOWINFO_SCALE2 * minn) );
  tmp.visangle = htonl( (long)(SHOWINFO_SCALE2 * visangle) );
  tmp.visdist = htonl( (long)(SHOWINFO_SCALE2 * visdist) );
  tmp.windir = htonl( (long)(SHOWINFO_SCALE2 * windir) );
  tmp.winforce = htonl( (long)(SHOWINFO_SCALE2 * winforce) );
  tmp.winang = htonl( (long)(SHOWINFO_SCALE2 * winang) );
  tmp.winrand = htonl( (long)(SHOWINFO_SCALE2 * winrand) );
  tmp.kickable_area = htonl( (long)(SHOWINFO_SCALE2 * kickable_area) );
  tmp.catch_area_l = htonl( (long)(SHOWINFO_SCALE2 * catch_area_l) );
  tmp.catch_area_w = htonl( (long)(SHOWINFO_SCALE2 * catch_area_w) );
  tmp.catch_prob = htonl( (long)(SHOWINFO_SCALE2 * catch_prob) );
  tmp.goalie_max_moves = htons( (short) goalie_max_moves );                 
  tmp.ckmargin = htonl( (long)(SHOWINFO_SCALE2 * ckmargin) );
  tmp.offside_area = htonl( (long)(SHOWINFO_SCALE2 * offside_area) );
  tmp.win_no = htons( (short) win_no);
  tmp.win_random = htons( (short) win_random) ;
  tmp.say_cnt_max = htons( (short) say_cnt_max) ;
  tmp.SayCoachMsgSize = htons( (short) SayCoachMsgSize) ;
  tmp.clang_win_size = htons( (short) clang_win_size );
  tmp.clang_define_win = htons( (short) clang_define_win );
  tmp.clang_meta_win = htons( (short) clang_meta_win );
  tmp.clang_advice_win = htons( (short) clang_advice_win );
  tmp.clang_info_win = htons( (short) clang_info_win );
  tmp.clang_mess_delay = htons( (short) clang_mess_delay );
  tmp.clang_mess_per_cycle = htons( (short) clang_mess_per_cycle );
  tmp.half_time = htons( (short) half_time) ;
  tmp.sim_st = htons( (short) sim_st) ;
  tmp.send_st = htons( (short) send_st) ;
  tmp.recv_st = htons( (short) recv_st) ;
  tmp.sb_step = htons( (short) sb_step) ;
  tmp.lcm_st = htons( (short) lcm_st) ;                        
  tmp.M_say_msg_size = htons( (short) M_say_msg_size) ;
  tmp.M_hear_max = htons( (short) M_hear_max) ;
  tmp.M_hear_inc = htons( (short) M_hear_inc) ;
  tmp.M_hear_decay = htons( (short) M_hear_decay) ;
  tmp.cban_cycle = htons( (short) cban_cycle) ;
  tmp.slow_down_factor = htons( (short) slow_down_factor) ;                          
  tmp.useoffside = htons( (short) useoffside) ;
  tmp.kickoffoffside = htons( (short) kickoffoffside) ;
  tmp.offside_kick_margin = htonl( (long)(SHOWINFO_SCALE2 * offside_kick_margin) );
  tmp.audio_dist = htonl( (long)(SHOWINFO_SCALE2 * audio_dist) );
  tmp.dist_qstep = htonl( (long)(SHOWINFO_SCALE2 * dist_qstep) );
  tmp.land_qstep = htonl( (long)(SHOWINFO_SCALE2 * land_qstep) );
#ifdef NEW_QSTEP
  tmp.dir_qstep = htonl( (long)(SHOWINFO_SCALE2 * dir_qstep) );
  tmp.dist_qstep_l = htonl( (long)(SHOWINFO_SCALE2 * dist_qstep_l) );
  tmp.dist_qstep_r = htonl( (long)(SHOWINFO_SCALE2 * dist_qstep_r) );
  tmp.land_qstep_l = htonl( (long)(SHOWINFO_SCALE2 * land_qstep_l) );
  tmp.land_qstep_r = htonl( (long)(SHOWINFO_SCALE2 * land_qstep_r) );
  tmp.dir_qstep_l = htonl( (long)(SHOWINFO_SCALE2 * dir_qstep_l) );
  tmp.dir_qstep_r = htonl( (long)(SHOWINFO_SCALE2 * dir_qstep_r) );
#else
  tmp.dir_qstep = htonl( (long)(SHOWINFO_SCALE2 * -1) );
  tmp.dist_qstep_l = htonl( (long)(SHOWINFO_SCALE2 * -1) );
  tmp.dist_qstep_r = htonl( (long)(SHOWINFO_SCALE2 * -1) );
  tmp.land_qstep_l = htonl( (long)(SHOWINFO_SCALE2 * -1) );
  tmp.land_qstep_r = htonl( (long)(SHOWINFO_SCALE2 * -1) );
  tmp.dir_qstep_l = htonl( (long)(SHOWINFO_SCALE2 * -1) );
  tmp.dir_qstep_r = htonl( (long)(SHOWINFO_SCALE2 * -1) );
#endif    
  tmp.CoachMode = htons( (short) CoachMode) ;
  tmp.CwRMode = htons( (short) CwRMode) ;
  tmp.old_hear = htons( (short) old_hear) ;
  tmp.sv_st = htons( (short) sv_st) ;

  tmp.synch_mode   = htons( (short)synch_mode);//pfr:SYNCH
  tmp.synch_offset = htons( (short)synch_offset);//pfr:SYNCH
  tmp.synch_micro_sleep = htons( (short)synch_micro_sleep);//pfr:SYNCH

  tmp.start_goal_l = htons( (short)start_goal_l );
  tmp.start_goal_r = htons( (short)start_goal_r );

  tmp.fullstate_l = htons( (short)fullstate_l );
  tmp.fullstate_r = htons( (short)fullstate_r );

  tmp.drop_time = htons( (short) drop_time) ;
  
  tmp.slowness_on_top_for_left_team = htonl( (long)(SHOWINFO_SCALE2 * slowness_on_top_for_left_team) );
  tmp.slowness_on_top_for_right_team = htonl( (long)(SHOWINFO_SCALE2 * slowness_on_top_for_left_team) );

  tmp.ka_length = htonl( (long)(SHOWINFO_SCALE2 * ka_length) );
  tmp.ka_width = htonl( (long)(SHOWINFO_SCALE2 * ka_width) );
  //tmp.kaway = htons( (short) kaway);

  tmp.point_to_ban =  htons( (short)M_point_to_ban );
  tmp.point_to_duration =  htons( (short)M_point_to_duration );
  return tmp;
}


std::ostream& toStr ( std::ostream& o, const ServerParamSensor_v7::data_t& data )
{
  return o << "(server_param "
           << data.M_gwidth << " " 
           << data.M_inertia_moment << " " 
           << data.M_psize << " " 
           << data.M_pdecay << " " 
           << data.M_prand << " " 
           << data.M_pweight << " " 
           << data.M_pspeed_max << " " 
           << data.M_paccel_max << " " 
           << data.M_stamina_max << " " 
           << data.M_stamina_inc << " " 
           << data.M_recover_init << " " 
           << data.M_recover_dthr << " " 
           << data.M_recover_min << " " 
           << data.M_recover_dec << " " 
           << data.M_effort_init << " " 
           << data.M_effort_dthr << " " 
           << data.M_effort_min << " " 
           << data.M_effort_dec << " " 
           << data.M_effort_ithr << " " 
           << data.M_effort_inc << " " 
           << data.M_kick_rand << " "                             
           << data.M_team_actuator_noise << " "                     
           << data.M_prand_factor_l << " "                          
           << data.M_prand_factor_r << " "                            
           << data.M_kick_rand_factor_l << " "                       
           << data.M_kick_rand_factor_r << " "                     
           << data.M_bsize << " " 
           << data.M_bdecay << " " 
           << data.M_brand << " " 
           << data.M_bweight << " " 
           << data.M_bspeed_max << " " 
           << data.M_baccel_max << " " 
           << data.M_dprate << " " 
           << data.M_kprate << " " 
           << data.M_kmargin << " " 
           << data.M_ctlradius << " " 
           << data.M_ctlradius_width << " " 
           << data.M_maxp << " "
           << data.M_minp << " " 
           << data.M_maxm << " " 
           << data.M_minm << " " 
           << data.M_maxnm << " " 
           << data.M_minnm << " " 
           << data.M_maxn << " " 
           << data.M_minn << " " 
           << data.M_visangle << " " 
           << data.M_visdist << " " 
           << data.M_windir << " " 
           << data.M_winforce << " " 
           << data.M_winang << " " 
           << data.M_winrand << " " 
           << data.M_kickable_area << " " 
           << data.M_catch_area_l << " " 
           << data.M_catch_area_w << " " 
           << data.M_catch_prob << " " 
           << data.M_goalie_max_moves << " "                  
           << data.M_ckmargin << " " 
           << data.M_offside_area << " " 
           << data.M_win_no << " " 
           << data.M_win_random << " " 
           << data.M_say_cnt_max << " " 
           << data.M_SayCoachMsgSize << " " 
           << data.M_clang_win_size << " "
           << data.M_clang_define_win << " "
           << data.M_clang_meta_win << " "
           << data.M_clang_advice_win << " "
           << data.M_clang_info_win << " "
           << data.M_clang_mess_delay << " "
           << data.M_clang_mess_per_cycle << " "
           << data.M_half_time << " " 
           << data.M_sim_st << " " 
           << data.M_send_st << " " 
           << data.M_recv_st << " " 
           << data.M_sb_step << " " 
           << data.M_lcm_st << " "                         
           << data.M_say_msg_size << " " 
           << data.M_hear_max << " " 
           << data.M_hear_inc << " " 
           << data.M_hear_decay << " " 
           << data.M_cban_cycle << " " 
           << data.M_slow_down_factor << " "                           
           << data.M_useoffside << " " 
           << data.M_kickoffoffside << " " 
           << data.M_offside_kick_margin << " " 
           << data.M_audio_dist << " " 
           << data.M_dist_qstep << " " 
           << data.M_land_qstep << " " 
           << data.M_dir_qstep << " " 
           << data.M_dist_qstep_l << " " 
           << data.M_dist_qstep_r << " " 
           << data.M_land_qstep_l << " " 
           << data.M_land_qstep_r << " " 
           << data.M_dir_qstep_l << " " 
           << data.M_dir_qstep_r << " "
           << data.M_CoachMode << " " 
           << data.M_CwRMode << " " 
           << data.M_old_hear << " " 
           << data.M_sv_st << " "
           << data.M_start_goal_l << " "
           << data.M_start_goal_r << " "
           << data.M_fullstate_l << " "
           << data.M_fullstate_r << " "
           << data.M_drop_time << ")";
}



std::ostream& toStr ( std::ostream& o, const ServerParamSensor_v8::data_t& data )
{
  o << "(server_param ";
  std::for_each( data.int_map.begin(), data.int_map.end(), ServerParam::Printer( o, 8 ) );
  std::for_each( data.str_map.begin(), data.str_map.end(), ServerParam::QuotedPrinter( o, 8 ) );
  std::for_each( data.bool_map.begin(), data.bool_map.end(), ServerParam::Printer( o, 8 ) );
  std::for_each( data.onoff_map.begin(), data.onoff_map.end(), ServerParam::Printer( o, 8 ) );
  std::for_each( data.double_map.begin(), data.double_map.end(), ServerParam::Printer( o, 8 ) );
  o << ")";

  return o;
//    return o << "(server_param "
//             << "(goal_width " << data.M_gwidth 
//             << ") (player_size "  << data.M_psize
//             << ") (player_decay " << data.M_pdecay
//             << ") (player_rand " << data.M_prand
//             << ") (player_weight " << data.M_pweight
//             << ") (player_speed_max " << data.M_pspeed_max
//             << ") (player_accel_max " << data.M_paccel_max
//             << ") (stamina_max " << data.M_stamina_max
//             << ") (stamina_inc_max " << data.M_stamina_inc
//             << ") (recover_init " << data.M_recover_init
//             << ") (recover_dec_thr " << data.M_recover_dthr
//             << ") (recover_min " << data.M_recover_min 
//             << ") (recover_dec " << data.M_recover_dec 
//             << ") (effort_init " << data.M_effort_init 
//             << ") (effort_dec_thr " << data.M_effort_dthr
//             << ") (effort_min " << data.M_effort_min
//             << ") (effort_dec " << data.M_effort_dec
//             << ") (effort_inc_thr " << data.M_effort_ithr
//             << ") (effort_inc " << data.M_effort_inc
//             << ") (kick_rand " << data.M_kick_rand           
//             << ") (team_actuator_noise " << data.M_team_actuator_noise 
//             << ") (prand_factor_l " << data.M_prand_factor_l      
//             << ") (prand_factor_r " << data.M_prand_factor_r      
//             << ") (kick_rand_factor_l " << data.M_kick_rand_factor_l  
//             << ") (kick_rand_factor_r " << data.M_kick_rand_factor_r  
//             << ") (ball_size " << data.M_bsize
//             << ") (ball_decay " << data.M_bdecay
//             << ") (ball_rand " << data.M_brand
//             << ") (ball_weight " << data.M_bweight
//             << ") (ball_speed_max " << data.M_bspeed_max
//             << ") (ball_accel_max " << data.M_baccel_max
//             << ") (dash_power_rate " << data.M_dprate
//             << ") (kick_power_rate " << data.M_kprate
//             << ") (kickable_margin " << data.M_kmargin
//             << ") (control_radius " << data.M_ctlradius
//             << ") (control_radius_width " << data.M_ctlradius_width 
//             << ") (catch_probability " << data.M_catch_prob
//             << ") (catchable_area_l " << data.M_catch_area_l
//             << ") (catchable_area_w " << data.M_catch_area_w
//             << ") (goalie_max_moves " << data.M_goalie_max_moves 
//             << ") (maxpower " << data.M_maxp
//             << ") (minpower " << data.M_minp
//             << ") (maxmoment " << data.M_maxm
//             << ") (minmoment " << data.M_minm
//             << ") (maxneckmoment " << data.M_maxnm
//             << ") (minneckmoment " << data.M_minnm
//             << ") (maxneckang " << data.M_maxn
//             << ") (minneckang " << data.M_minn
//             << ") (visible_angle " << data.M_visangle
//             << ") (visible_distance " << data.M_visdist
//             << ") (audio_cut_dist " << data.M_audio_dist
//             << ") (quantize_step " << data.M_dist_qstep
//             << ") (quantize_step_l " << data.M_land_qstep
//             << ") (quantize_step_dir " << data.M_dir_qstep
//             << ") (quantize_step_dist_team_l " << data.M_dist_qstep_l
//             << ") (quantize_step_dist_team_r " << data.M_dist_qstep_r
//             << ") (quantize_step_dist_l_team_l " << data.M_land_qstep_l
//             << ") (quantize_step_dist_l_team_r " << data.M_land_qstep_r
//             << ") (quantize_step_dir_team_l " << data.M_dir_qstep_l
//             << ") (quantize_step_dir_team_r " << data.M_dir_qstep_r
//             << ") (ckick_margin " << data.M_ckmargin
//             << ") (wind_dir " << data.M_windir
//             << ") (wind_force " << data.M_winforce
//             << ") (wind_ang " << data.M_winang 
//             << ") (wind_rand " << data.M_winrand
//             << ") (kickable_area " << data.M_kickable_area 
//             << ") (inertia_moment " << data.M_inertia_moment
//             << ") (wind_none " << data.M_win_no
//             << ") (wind_random " << data.M_win_random
//             << ") (half_time " << data.M_half_time
//             << ") (drop_ball_time " << data.M_drop_time
//             << ") (port " << data.M_portnum
//             << ") (coach_port " << data.M_coach_pnum
//             << ") (olcoach_port " << data.M_olcoach_pnum
//             << ") (say_coach_cnt_max " << data.M_say_cnt_max
//             << ") (say_coach_msg_size " << data.M_SayCoachMsgSize
//             << ") (simulator_step " << data.M_sim_st
//             << ") (send_step " << data.M_send_st
//             << ") (recv_step " << data.M_recv_st
//             << ") (sense_body_step " << data.M_sb_step
//             << ") (lcm_step " << data.M_lcm_st                         
//             << ") (say_msg_size " << data.M_say_msg_size
//             << ") (clang_win_size " << data.M_clang_win_size
//             << ") (clang_define_win " << data.M_clang_define_win
//             << ") (clang_meta_win " << data.M_clang_meta_win
//             << ") (clang_advice_win " << data.M_clang_advice_win
//             << ") (clang_info_win " << data.M_clang_info_win
//             << ") (clang_mess_delay " << data.M_clang_mess_delay
//             << ") (clang_mess_per_cycle " << data.M_clang_mess_per_cycle
//             << ") (hear_max " << data.M_hear_max
//             << ") (hear_inc " << data.M_hear_inc
//             << ") (hear_decay " << data.M_hear_decay
//             << ") (catch_ban_cycle " << data.M_cban_cycle
//             << ") (coach " << data.M_CoachMode
//             << ") (coach_w_referee " << data.M_CwRMode
//             << ") (old_coach_hear " << data.M_old_hear
//             << ") (send_vi_step " << data.M_sv_st
//             << ") (use_offside " << data.M_useoffside
//             << ") (offside_active_area_size " << data.M_offside_area
//             << ") (forbid_kick_off_offside " << data.M_kickoffoffside
//             << ") (verbose " << data.M_verbose
//             << ") (replay " << ( data.M_replay.size() == 0 ? "0" : data.M_replay )
//             << ") (offside_kick_margin " << data.M_offside_kick_margin
//             << ") (slow_down_factor " << data.M_slow_down_factor   
//             << ") (synch_mode " << data.M_synch_mode
//             << ") (synch_offset " << data.M_synch_offset
//             << ") (synch_micro_sleep " << data.M_synch_micro_sleep
//             << ") (start_goal_l " << data.M_start_goal_l
//             << ") (start_goal_r " << data.M_start_goal_r
//             << ") (fullstate_l " << data.M_fullstate_l
//             << ") (fullstate_r " << data.M_fullstate_r
//             << ") (slowness_on_top_for_left_team " 
//             << data.M_slowness_on_top_for_left_team 
//             << ") (slowness_on_top_for_right_team " 
//             << data.M_slowness_on_top_for_right_team  
//             << ") (send_comms " << data.M_send_comms      
//             << ") (text_logging " << data.M_text_logging
//             << ") (game_logging " << data.M_game_logging
//             << ") (game_log_version " << data.M_game_log_version
//             << ") (text_log_dir " << data.M_text_log_dir
//             << ") (game_log_dir " << data.M_game_log_dir
//             << ") (text_log_fixed_name " << data.M_text_log_fixed_name
//             << ") (game_log_fixed_name " << data.M_game_log_fixed_name
//             << ") (text_log_fixed " << data.M_text_log_fixed
//             << ") (game_log_fixed " << data.M_game_log_fixed
//             << ") (text_log_dated " << data.M_text_log_dated
//             << ") (game_log_dated " << data.M_game_log_dated
//             << ") (log_date_format " << data.M_log_date_format
//             << ") (log_times " << data.M_log_times
//             << ") (record_messages " << data.M_record_messages
//             << ") (text_log_compression " << data.M_text_log_compression
//             << ") (game_log_compression " << data.M_game_log_compression
//             << ") (profile " << data.M_profile
//             << ") (point_to_ban " << data.M_point_to_ban
//             << ") (point_to_duration " << data.M_point_to_duration
//             << "))";		
}

