// -*-c++-*-

/***************************************************************************
                                    rule.h  
                           Classes for clang rules
                             -------------------
    begin                : 25-APR-2002
    copyright            : (C) 2002 by The RoboCup Soccer Server 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/

#ifndef _RULE_H_
#define _RULE_H_

#include <vector>
#include <string>
#include <list>
#include "hasa.h" 
#include "visitor.h"


namespace rcss
{
  namespace clang
  {
    typedef std::string RuleID;

    // If the rule list is empty then it counts a containing all
    // rules.  This also means that there is no support for an empty
    // rule ID list, but that is just the way we want it.
    class RuleIDList
      : public std::list< RuleID >
    {
    private:
      bool
      empty() const;
    public:
      bool
      all() const;

      bool
      unary() const;

      std::ostream&
      print( std::ostream& o ) const;

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const;
    };

    class ActivateRules
      : public util::HasA< bool >,
        public util::HasA< RuleIDList >
    {
    public:
      typedef util::HasA< bool > HasABool;
      typedef util::HasA< RuleIDList > HasARuleIDList;

      ActivateRules();

      ActivateRules( const bool& on, const RuleIDList& list );

      ~ActivateRules();

      bool
      on() const;

      bool
      all() const;

      bool
      unary() const;

      const RuleIDList&
      getList() const;

      void
      set( const bool& on );

      void
      set( const RuleIDList& rids );

      std::ostream&
      print( std::ostream& o ) const;

      std::ostream&
      printPretty( std::ostream& o, const std::string& line_header ) const;
    };

    class Cond;
    class Dir;

    class CondRule;
    class SimpleRule;
    class NestedRule;
    class IDListRule;

    class Rule
    {
    public:
      typedef util::Visitor10< CondRule*,
			       SimpleRule*,
			       NestedRule*,
			       IDListRule* > Visitor;

      typedef util::Visitor10< const CondRule*,
			       const SimpleRule*,
			       const NestedRule*,
			       const IDListRule* > ConstVisitor;

      virtual
      ~Rule();

      virtual
      std::auto_ptr< Rule >
      deepCopy() const = 0;

      virtual
      void
      accept( Visitor& v ) = 0;

      virtual
      void
      accept( ConstVisitor& v ) const = 0;

      virtual
      std::ostream& 
      print( std::ostream& out ) const = 0;
      
      virtual
      std::ostream&
      printPretty( std::ostream& out,
                   const std::string& line_header ) const = 0;


    };

    class CondRule
      : public Rule,
        public util::HasA< std::auto_ptr< Cond > >
    {
    public:
      typedef util::HasA< std::auto_ptr< Cond > > HasACond;

      CondRule( std::auto_ptr< Cond > cond );

      virtual
      ~CondRule();

      void
      accept( Visitor& v ) { v.startVisit( this );}

      void
      accept( ConstVisitor& v ) const { v.startVisit( this );}

      const Cond*
      getCond() const;

      std::auto_ptr< Cond >
      detachCond();
   };

    
    class SimpleRule
      : public CondRule,
        public util::HasMany< std::auto_ptr< Dir > >
    {
    public:
      typedef util::HasMany< std::auto_ptr< Dir > > HasManyDirs;

      SimpleRule( std::auto_ptr< Cond > cond );
      
      SimpleRule( std::auto_ptr< Cond > cond,
                  const Storage& dirs );

      virtual
      ~SimpleRule();

      void
      accept( Visitor& v ){ v.startVisit( this );}

      void
      accept( ConstVisitor& v ) const { v.startVisit( this );}

      virtual
      std::ostream& 
      print( std::ostream& out ) const;
      
      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const;

      virtual
      std::auto_ptr< Rule >
      deepCopy() const;

      const Storage&
      getDirs() const;

      Storage&
      getDirs();
    };

    class NestedRule
      : public CondRule,
        public util::HasMany< std::auto_ptr< Rule > >
    {
    public:
      typedef util::HasMany< std::auto_ptr< Rule > > HasManyRules;

      NestedRule( std::auto_ptr< Cond > cond );
      
      NestedRule( std::auto_ptr< Cond > cond,
                  const Storage& rules );

      virtual
      ~NestedRule();

      void
      accept( Visitor& v ){ v.startVisit( this );}

      void
      accept( ConstVisitor& v ) const { v.startVisit( this );}

      virtual
      std::ostream& 
      print( std::ostream& out ) const;
      
      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const;

      virtual
      std::auto_ptr< Rule >
      deepCopy() const;
      
      const Storage&
      getRules() const;

      Storage&
      getRules();
    };

    class IDListRule
      : public Rule,
        public util::HasA< RuleIDList >
    {
    public:
      typedef util::HasA< RuleIDList > HasARuleIDList;

      IDListRule();
      
      IDListRule( const RuleIDList& rules );

      virtual
      ~IDListRule();

      void
      accept( Visitor& v ){ v.startVisit( this );}

      void
      accept( ConstVisitor& v ) const { v.startVisit( this );}

      virtual
      std::ostream& 
      print( std::ostream& out ) const;
      
      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const;

      virtual
      std::auto_ptr< Rule >
      deepCopy() const;
      
      const RuleIDList&
      getIDList() const;

      /* Get the rule pointer list */
      const std::vector <Rule *>
      getRuleList() const
      { return ruleList; }
      
      /* dongryel: add Rule pointer to the list */
      void
      addRule(Rule *rule)
      { ruleList.push_back( rule ); }

      /* Change the rule pointer in the list */
      void
      changeRule(int index, Rule *rule)
      { ruleList[index] = rule; }

    private:
      
      /* Pointer to the rules that correspond to the Rule ID list */
      std::vector < Rule * > ruleList;
    };
  }
}


std::ostream&
operator<<( std::ostream& o, const rcss::clang::RuleIDList& id_list );

std::ostream&
operator<<( std::ostream& o, const rcss::clang::ActivateRules& activ_rules );

std::ostream&
operator<<( std::ostream & os, const rcss::clang::Rule& r );

 
#endif
