/* -*- Mode: C++ -*- */
/*
 *Copyright:

    Copyright (C) 2001 RoboCup Soccer Server Maintainance Group.
    	Patrick Riley, Tom Howard, Itsuki Noda,	Mikhail Prokopenko, Jan Wendler 

    This file is a part of SoccerServer.

    This code is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/* This files defines a class region to specify regions for the coach messages */

#ifndef _H_REGION
#define _H_REGION

#include <math.h>
#include <memory>
#include <iostream>
#include <list>
#include <vector>
#include <string>
#include <stdio.h>
#include "Geometry.h"
#include "RegionIterator.h"
#include "visitor.h"
#include "clangutil.h"
#include "arithop.h"
#include "hasa.h"
#include "WorldModelInterface.h"
#include "FieldImage.h"


namespace rcss
{
  namespace clang
  {

    class PointSimple;
    class PointRel;
    class PointBall;
    class PointPlayer;
    class PointArith;

    class Point
    {
    public:
      typedef util::Visitor10< PointSimple*,
        PointRel*,
        PointBall*,
        PointPlayer*,
        PointArith* > Visitor;

      typedef util::Visitor10< const PointSimple*,
        const PointRel*,
        const PointBall*,
        const PointPlayer*,
        const PointArith* > ConstVisitor;

      class TypeExtractor
        : public util::TypeExtractor10< PointSimple*,
        PointRel*,
        PointBall*,
        PointPlayer*,
        PointArith* >
      {
      protected:
        void
        visit( PointSimple* pt )
        { setValue( pt ); }

        void
        visit( PointRel* pt )
        { setValue( pt ); }

        void
        visit( PointBall* pt )
        { setValue( pt ); }

        void
        visit( PointPlayer* pt )
        { setValue( pt ); }

        void
        visit( PointArith* pt )
        { setValue( pt ); }
      }; 

      class ConstTypeExtractor
        : public util::TypeExtractor10< const PointSimple*,
        const PointRel*,
        const PointBall*,
        const PointPlayer*,
        const PointArith* >
      {
      protected:
        void
        visit( const PointSimple* pt )
        { setValue( pt ); }

        void
        visit( const PointRel* pt )
        { setValue( pt ); }

        void
        visit( const PointBall* pt )
        { setValue( pt ); }

        void
        visit( const PointPlayer* pt )
        { setValue( pt ); }

        void
        visit( const PointArith* pt )
        { setValue( pt ); }
      }; 

      Point()
      {}

      virtual 
      ~Point() 
      {}
      
      virtual
      std::ostream& 
      print( std::ostream& out ) const = 0;

      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const = 0;
  
      virtual
      void
      accept( Visitor& v ) = 0;

      virtual
      void
      accept( ConstVisitor& v ) const = 0;

      virtual
      std::auto_ptr< Point >
      deepCopy() const = 0;

      /* Added in by Dongryeol Lee: returns the global position of the
	 point object, or position that would be specified by a PointSimple
	 object */
      virtual
      VecPosition getGlobalPosition(RegionWorldModelInterface *wmi, 
				    const VarBindings& bindings) const = 0;   
  
      /* Draws the point on the image */
      /* the base definition will just call getGlobalPosition and draw there */
      virtual
      void
      draw(FieldImage* pfi, 
	   bool verbose, 
	   const FieldImage::Color& c,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;

    protected:
      //a slight performance enhancer for the children so we don't have to getGlobalPosition many times
      void
      draw(FieldImage* pfi, 
	   bool verbose, 
	   const FieldImage::Color& c,
	   VecPosition global_pos) const;
    };
  }
}

inline
std::ostream&
operator <<(std::ostream & os, const rcss::clang::Point& r )
{ return r.print(os); } 

namespace rcss
{
  namespace clang
  {


    class PointSimple
      : public Point
    {
    private:
      geom::Vector2D M_vec;
    public:
      PointSimple() 
        : Point(),
          M_vec()
      {}

      PointSimple( const double& x,
                   const double& y ) 
        : Point(),
          M_vec( x, y )
      {}

      PointSimple( const geom::Vector2D& vec ) 
        : Point(),
          M_vec( vec )
      {}

      ~PointSimple() {}

      std::ostream&
      print( std::ostream& out ) const
      { return out << "(pt " << M_vec.getX() << " " << M_vec.getY() << ")"; }

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { 
        return out << line_header << "Point(" 
                   << M_vec.getX() << ", " << M_vec.getY() << ")" << std::endl; 
      }
      
      geom::Vector2D&
      getVec()
      { return M_vec; }

      const geom::Vector2D&
      getVec() const
      { return M_vec; }

      geom::Vector2D&
      setVec( const geom::Vector2D& vec )
      { M_vec = vec; return M_vec; }

      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }
  
      std::auto_ptr< Point >
      deepCopy() const
      { return std::auto_ptr< Point >( new PointSimple( *this ) ); }


      /* Added in by Dongryeol Lee */
      VecPosition getGlobalPosition(RegionWorldModelInterface *wmi, 
				    const VarBindings& bindings) const;

      /* Draws the point on the image */
      void
      draw(FieldImage* pfi, bool verbose, const FieldImage::Color& c,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;
    };

    class PointRel
      : public Point,
        public util::HasA< std::auto_ptr< Point > >
    {
    public:
      typedef util::HasA< std::auto_ptr< Point > > HasAPoint;

      PointRel() 
        : Point(),
          HasAPoint(),
          M_offset()
      {}
      
      PointRel( const double& x,
                const double& y,
                std::auto_ptr< Point > origin ) 
        : Point(),
          HasAPoint( origin ),
          M_offset( x, y )
      {}
      
      PointRel( const PointSimple offset,
                std::auto_ptr< Point > origin ) 
        : Point(),
          HasAPoint( origin ),
          M_offset( offset )
      {}

      virtual
      ~PointRel()
      {}
      
      std::ostream& 
      print( std::ostream& out ) const
      {
        out << "(pt " << M_offset.getVec().getX() << " " 
            << M_offset.getVec().getY() << " ";
        if( get() == NULL )
          out << "(null)";
        else
          out << getRef();
        return out << ")";
      }
      
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      {
        out << line_header << "Point-Relative(" << M_offset.getVec().getX() 
            << ", " << M_offset.getVec().getY() << " ";
        if( get() == NULL )
          out << "(null)";
        else
          getRef().printPretty( out, line_header + " " );
        return out << ")" << std::endl;
      } 
     
      void
      accept( Visitor& v )
      { v.startVisit( this ); }
      
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }
      
      virtual
      std::auto_ptr< Point >
      deepCopy() const
      { return std::auto_ptr< Point >( new PointRel( *this ) ); }

      PointSimple
      getOffset() const
      { return M_offset; }

      void
      set( const PointSimple& offset )
      { M_offset = offset; }

      const Point*
      getOrigin() const
      { return get(); }

      /* Added in by Dongryeol Lee */
      VecPosition getGlobalPosition(RegionWorldModelInterface *wmi, 
				    const VarBindings& bindings) const;

      /* Draws the point on the image */
      void
      draw(FieldImage* pfi, bool verbose, const FieldImage::Color& c,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;
      
    private:
      PointSimple M_offset;
    };


    class PointBall
      : public Point
    {
    public:
      PointBall() 
        : Point()
      {}
      
      ~PointBall()
      {}

      std::ostream&
      print( std::ostream& out ) const
      { return out << "(pt ball)"; }
      
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return out << line_header << "Point-Ball" << std::endl; }
 
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Point >
      deepCopy() const
      { return std::auto_ptr< Point >( new PointBall( *this ) ); }

      /* Added in by Dongryeol Lee */
      VecPosition getGlobalPosition(RegionWorldModelInterface *wmi, 
				    const VarBindings& bindings) const;

      /* Draws the point on the image */
      void
      draw(FieldImage* pfi, bool verbose, const FieldImage::Color& c,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;
    };

    class PointPlayer
      : public Point
    {
    public:
      PointPlayer() 
        : Point(),
          M_our_side( false ),
          M_unum()
      {}

      PointPlayer( const bool& our_side, const UNum& unum ) 
        : Point(),
          M_our_side( our_side ),
          M_unum( unum )
      {}

      ~PointPlayer()
      {}

      std::ostream&
      print( std::ostream& out ) const
      { 
        return out << "(pt " << (M_our_side ? "our" : "opp") 
                   << " " << M_unum << ")";
      }
      
      std::ostream& 
      printPretty( std::ostream& out, const std::string& line_header ) const
      { 
        return out << line_header << "Point-Player(" 
                   << (M_our_side ? "our team" : "opponent")
                   << " " << M_unum << ")" << std::endl;
      }

      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Point >
      deepCopy() const
      { return std::auto_ptr< Point >( new PointPlayer( *this ) ); }
 
      /* class specific */
      bool 
      isOurSide() const
      { return M_our_side; }
      
      bool
      isTheirSide() const
      { return !M_our_side; }
      
      UNum
      getUNum() const
      { return M_unum; }
      
      void 
      setOurSide( const bool& our_side )
      { M_our_side = our_side; }
      
      void
      setUNum( const UNum& num )
      { M_unum = num; }
      
      /* Added in by Dongryeol Lee */
      VecPosition getGlobalPosition(RegionWorldModelInterface *wmi, 
				    const VarBindings& myBindings) const;

      /* Draws the point on the image */
      void
      draw(FieldImage* pfi, bool verbose, const FieldImage::Color& c,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;

    private:
      bool M_our_side;
      UNum M_unum;
    };



    class PointArith
      : public Point
    {
    public:
      PointArith() 
        : Point(),
          M_arith_op( NULL ),
          M_idx( 0 )
      {}
      
      PointArith( std::auto_ptr< Point > pt1, std::auto_ptr< Point > pt2,
                  const util::ArithOp& arith_op ) 
        : Point(),
          M_arith_op( &arith_op ),
          M_idx( 0 )
      {
        M_points[ 0 ] = pt1;
        M_points[ 1 ] = pt2;
      }
      
      PointArith( const PointArith& pt ) 
        : Point(),
          M_arith_op( pt.M_arith_op ),
          M_idx( 0 )
      {
        std::auto_ptr< Point > pt_copy1 = pt.M_points[ 0 ]->deepCopy();
        M_points[ 0 ] = pt_copy1;
        std::auto_ptr< Point > pt_copy2 = pt.M_points[ 1 ]->deepCopy();
        M_points[ 1 ] = pt_copy2;
      }
      
      ~PointArith()
      {}

      PointArith&
      operator=( const PointArith& pt )
      {
        std::auto_ptr< Point > pt_copy1 = pt.M_points[ 0 ]->deepCopy();
        M_points[ 0 ] = pt_copy1;
        std::auto_ptr< Point > pt_copy2 = pt.M_points[ 1 ]->deepCopy();
        M_points[ 1 ] = pt_copy2;        
        M_idx = pt.M_idx;
        return *this;
      }

      std::ostream&
      print( std::ostream& out ) const
      {
        out << "(";

        if( M_points[ 0 ].get() == NULL )
          out << "(null)";
        else
          out << *(M_points[ 0 ]);

        out << " ";

        if( M_arith_op == NULL )
          out << "(null)";
        else
          out << *M_arith_op;

        out << " ";

        if( M_points[ 1 ].get() == NULL )
          out << "(null)";
        else
          out << *(M_points[ 1 ]);

        return out << ")";
      }
      
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { 
        out << line_header << "Point-Arith" << std::endl;

        if( M_points[ 0 ].get() == NULL )
          out << line_header << " (null)\n";
        else
          M_points[ 0 ]->printPretty( out, line_header + " " );

        if( M_points[ 1 ].get() == NULL )
          out << line_header << " (null)\n";
        else
          M_points[ 1 ]->printPretty( out, line_header + " " );

        if( M_arith_op == NULL )
          out << line_header << " (null)\n";
        else
          out << line_header << " operation: " << *M_arith_op << std::endl;
        
        return out;
      }
 
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Point >
      deepCopy() const
      { return std::auto_ptr< Point >( new PointArith( *this ) ); }

      bool 
      setPoint( const int& i, std::auto_ptr< Point >& pt )
      {
        if (i < 0 || i > 1)
          return false;
        M_points[i] = pt;
        return true;
      }

      void 
      setAllPoints( std::auto_ptr< Point >& pt0,
                 std::auto_ptr< Point >& pt1 )
      {
        M_points[0] = pt0;
        M_points[1] = pt1;
      }

      bool      
      setPoint( std::auto_ptr< Point >& pt )
      {
        M_points[ M_idx ] = pt;
        M_idx = (M_idx + 1) % 2;

	return true;
      }

      Point* 
      getPoint( const int& i )
      { return (i>=0 && i<=1) ? M_points[i].get() : NULL; }

      const Point* 
      getPoint( const int& i ) const
      { return (i>=0 && i<=1) ? M_points[i].get() : NULL; }

      const util::ArithOp*
      getOp() const
      { return M_arith_op; }

      const util::ArithOp&
      setOp( const util::ArithOp& arith_op )
      { return *(M_arith_op = &arith_op); }
      
      /* Added in by Dongryeol Lee */
      VecPosition getGlobalPosition(RegionWorldModelInterface *wmi, 
				    const VarBindings& bindings) const;
      
      /* Draws the point on the image */
      void
      draw(FieldImage* pfi, bool verbose, const FieldImage::Color& c,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;

    private:
      std::auto_ptr< Point > M_points[2];
      const util::ArithOp* M_arith_op;
      unsigned int M_idx;
    };



    class RegNull;
    class RegQuad;
    class RegArc;
    class RegUnion;
    class RegNamed;
    class RegPoint;
    class RegTri;
    class RegRec;

    class Region 
    {
    public:
      typedef util::Visitor10< RegNull*,
        RegQuad*,
        RegArc*,
        RegUnion*,
        RegNamed*,
        RegPoint*,
        RegTri*,
        RegRec* > Visitor;

      typedef util::Visitor10< const RegNull*,
        const RegQuad*,
        const RegArc*,
        const RegUnion*,
        const RegNamed*,
        const RegPoint*,
        const RegTri*,
        const RegRec*  > ConstVisitor;

      class TypeExtractor
        : public util::TypeExtractor10< RegNull*,
        RegQuad*,
        RegArc*,
        RegUnion*,
        RegNamed*,
        RegPoint*,
        RegTri*,
        RegRec* >
      {
      protected:
        void
        visit( RegNull* reg )
        { setValue( reg ); }

        void
        visit( RegQuad* reg )
        { setValue( reg ); }

        void
        visit( RegArc* reg )
        { setValue( reg ); }

        void
        visit( RegUnion* reg )
        { setValue( reg ); }

        void
        visit( RegNamed* reg )
        { setValue( reg ); }

        void
        visit( RegPoint* reg )
        { setValue( reg ); }

        void
        visit( RegTri* reg )
        { setValue( reg ); }

        void
        visit( RegRec* reg )
        { setValue( reg ); }

      };

      class ConstTypeExtractor
        : public util::TypeExtractor10< const RegNull*,
        const RegQuad*,
        const RegArc*,
        const RegUnion*,
        const RegNamed*,
        const RegPoint*,
        const RegTri*,
        const RegRec* >
      {
      protected:
        void
        visit( const RegNull* reg )
        { setValue( reg ); }

        void
        visit( const RegQuad* reg )
        { setValue( reg ); }

        void
        visit( const RegArc* reg )
        { setValue( reg ); }

        void
        visit( const RegUnion* reg )
        { setValue( reg ); }

        void
        visit( const RegNamed* reg )
        { setValue( reg ); }

        void
        visit( const RegPoint* reg )
        { setValue( reg ); }

        void
        visit( const RegTri* reg )
        { setValue( reg ); }

        void
        visit( const RegRec* reg )
        { setValue( reg ); }
      };

      Region()
      {}

      virtual 
      ~Region() 
      {}
      
      virtual
      std::ostream& 
      print( std::ostream& out ) const = 0;

      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const = 0;
  
      virtual
      void
      accept( Visitor& v ) = 0;

      virtual
      void
      accept( ConstVisitor& v ) const = 0;

      virtual
      std::auto_ptr< Region >
      deepCopy() const = 0;

      /* Added in by Dongryeol Lee */
      virtual
      bool isPtIn(const VecPosition& p, RegionWorldModelInterface *wmi,
		  const VarBindings& bindings) const = 0;

      virtual
      std::vector <VecPosition> 
      getIntersection(const Line& l, RegionWorldModelInterface *wmi,
		      const VarBindings& bindings) const = 0;
      
      virtual
      VecPosition getRandomPtIn(RegionWorldModelInterface *wmi, 
				const VarBindings& bindings) const = 0;
      
      virtual
      VecPosition getRandomPtOut(RegionWorldModelInterface *wmi,
				 const VarBindings& bindings) const = 0;
      
      virtual
      VecPosition getClosestPtTo(const VecPosition& pt, 
				 RegionWorldModelInterface *wmi,
				 const VarBindings& bindings) const = 0;
      

      /* Useful method: to calculate area of a region */
      virtual
      double getArea(RegionWorldModelInterface *wmi,
		     const VarBindings& bindings) const = 0;



      /* RegionIterator generator: virtual function  */
      virtual RegionIterator* newIterator(double d, 
					  RegionWorldModelInterface *wmi,
					  const VarBindings& bindings) = 0;
      
      virtual RegionIterator* newIterator(RegionWorldModelInterface *wmi,
					  const VarBindings& bindings) = 0;

      /* draw the region on the field */
      virtual
      void
      draw(FieldImage* pfi, bool verbose,
	   const FieldImage::Color& c_border, 
	   const FieldImage::Color& c_inside,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const = 0;
    };

    // A random helper function; nice for some STL stuff
    struct RegionDeepCopy
    {
      rcss::clang::Region* operator() (rcss::clang::Region* pr)
      { return (pr == NULL) ? NULL : ((pr->deepCopy()).release()); }
    };

  }
}

std::ostream&
operator <<(std::ostream & os, const rcss::clang::Region& r );

namespace rcss
{
  namespace clang
  {

    class RegNull
      : public Region
    {
    public:
      RegNull() 
        : Region()
      {}
      
      ~RegNull()
      {}

      std::ostream&
      print( std::ostream& out ) const
      { return out << "(null)"; }
      
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return out << line_header << "null region" << std::endl; }

      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Region >
      deepCopy() const
      { return std::auto_ptr< Region >( new RegNull( *this ) ); }


      /* Added in by Dongryeol Lee */
      bool isPtIn(const VecPosition& p, RegionWorldModelInterface *wmi, 
		  const VarBindings& bindings) const;

      std::vector <VecPosition> getIntersection(const Line& l, 
						RegionWorldModelInterface *wmi,
						const VarBindings& bindings)
	const;

      VecPosition getRandomPtIn(RegionWorldModelInterface *wmi, 
				const VarBindings& bindings) const;
      
      VecPosition getRandomPtOut(RegionWorldModelInterface *wmi, 
				 const VarBindings& bindings) const;

      VecPosition getClosestPtTo(const VecPosition& pt, 
				 RegionWorldModelInterface *wmi,
				 const VarBindings& bindings) const;


      double getArea(RegionWorldModelInterface *,
		     const VarBindings & ) const
      { return 0.0; }


      /* RegionIterator stuff for RegNull */
      class RegNullIterator: public RegionIterator
      {
      public:
	RegNullIterator(double d, const VarBindings& bindings)
	  : RegionIterator(d), myBindings(bindings)
	{ error_flag = true; }
	
	RegNullIterator(const VarBindings& bindings)
	  : RegionIterator(), myBindings(bindings)
	{ error_flag = true; }
	
	VecPosition getPt(RegionWorldModelInterface* , 
			  const VarBindings& )
	{
	  printf("Impossible to iterate a RegNull\n");
	  
	  return VecPosition(0.0, 0.0);
	}

	void moveToNextPt()
	{
	  printf("Impossible to iterate a RegNull\n");
	}

	void resetToBegin()
	{ error_flag = true; }

      private: 
	VarBindings myBindings;

      };

      RegionIterator* newIterator(double d, 
				  RegionWorldModelInterface *,
				  const VarBindings& bindings)
      {
	return new RegNullIterator(d, bindings);
      }

      RegionIterator* newIterator(RegionWorldModelInterface *,
				  const VarBindings& bindings)
      {
	return new RegNullIterator(bindings);
      }

      /* draw the region on the field */
      void
      draw(FieldImage* pfi, bool verbose,
	   const FieldImage::Color& c_border, 
	   const FieldImage::Color& c_inside,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;
      
    };

    class RegQuad
      : public Region,
        public util::HasN< std::auto_ptr< Point >, 4 >
    {
    public:
      typedef util::HasN< std::auto_ptr< Point >, 4 > Has4Points;

      RegQuad();
      
      RegQuad( std::auto_ptr< Point > pt0,
               std::auto_ptr< Point > pt1, 
               std::auto_ptr< Point > pt2,
               std::auto_ptr< Point > pt3 );

      virtual
      ~RegQuad();

      std::ostream&
      print( std::ostream& out ) const;
      
      std::ostream& 
      printPretty( std::ostream& out, const std::string& line_header ) const;

      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      virtual
      std::auto_ptr< Region >
      deepCopy() const
      { return std::auto_ptr< Region >( new RegQuad( *this ) ); }

  /* class specific */
      const Point* 
      getPt( unsigned int i ) const
      { return get( i ); }
      
      void
      setAllPts( std::auto_ptr< Point > pt0,
                 std::auto_ptr< Point > pt1,
                 std::auto_ptr< Point > pt2,
                 std::auto_ptr< Point > pt3 );

      /* Added in by Dongryeol Lee: Unsupported region, 
	 just return false */
      bool isPtIn(const VecPosition& p, RegionWorldModelInterface *wmi, 
		  const VarBindings& bindings) const;

      std::vector <VecPosition> getIntersection(const Line &l, 
						RegionWorldModelInterface *wmi,
						const VarBindings& bindings)
	const;

      VecPosition getRandomPtIn(RegionWorldModelInterface *wmi, 
				const VarBindings& bindings) const;
      
      VecPosition getRandomPtOut(RegionWorldModelInterface *wmi, 
				 const VarBindings& bindings) const;
      
      VecPosition getClosestPtTo(const VecPosition& pt, 
				 RegionWorldModelInterface *wmi,
				 const VarBindings& bindings) const;


      double getArea(RegionWorldModelInterface *,
		     const VarBindings & ) const
      { return 0.0; }


      /* RegionIterator stuff for RegQuad */
      class RegQuadIterator: public RegionIterator
      {
      public:
	RegQuadIterator(double d, const VarBindings& bindings)
	  : RegionIterator(d), myBindings(bindings)
	{}

	RegQuadIterator(const VarBindings& bindings)
	  : RegionIterator(), myBindings(bindings)
	{}

	VecPosition getPt(RegionWorldModelInterface *, const VarBindings& )
	{
	  printf("Illegal to iterate a RegQuad\n");
	  
	  return VecPosition(0.0, 0.0);
	}

	void moveToNextPt()
	{
	  printf("Illegal to iterate a RegQuad\n");
	}

	void resetToBegin()
	{}

      private:
	VarBindings myBindings;
      };

      RegionIterator* newIterator(double d, RegionWorldModelInterface *,
				  const VarBindings& bindings)
      {
	return new RegQuadIterator(d, bindings);
      }

      RegionIterator* newIterator(RegionWorldModelInterface *,
				  const VarBindings& bindings)
      {
	return new RegQuadIterator(bindings);
      }

      /* draw the region on the field */
      void
      draw(FieldImage* , bool ,
	   const FieldImage::Color& , const FieldImage::Color& ,
	   RegionWorldModelInterface *, 
	   const VarBindings& ) const
      {
	printf("Illegal to draw a RegQuad\n");
      }
      
      
    };

    class RegArc
      : public Region,
        public util::HasA< std::auto_ptr< Point > >
    {
    public:
      typedef util::HasA< std::auto_ptr< Point > > HasAPoint;

      RegArc();
      
      RegArc( std::auto_ptr< Point > center, 
              const double& start_rad, 
              const double& end_rad, 
              const double& start_ang,
              const double& span_ang );

      RegArc( Circle c );
      
      virtual
      ~RegArc();
       
      std::ostream& 
      print( std::ostream& out ) const;
      
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const;

      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      virtual
      std::auto_ptr< Region >
      deepCopy() const
      { return std::auto_ptr< Region >( new RegArc( *this ) ); }

      const Point*
      getCenter() const 
      { return get(); }
  
      double    
      getStartRad() const
      { return M_start_rad; }

      double
      getEndRad() const
      { return M_end_rad; }
      
      double
      getStartAng() const
      { return M_start_ang; }

      double 
      getSpanAng() const
      { return M_span_ang; }
      
      bool
      setRad( const double& start_rad, const double& end_rad );
      
      bool
      setAng( const double& start_ang, const double& span_ang);

      /* Added in by Dongryeol Lee */
      bool isPtIn(const VecPosition& p, RegionWorldModelInterface *wmi, 
		  const VarBindings& bindings) const;

      std::vector <VecPosition> getIntersection(const Line &l, 
						RegionWorldModelInterface *wmi,
						const VarBindings& bindings)
	const;

      VecPosition getRandomPtIn(RegionWorldModelInterface *wmi, 
				const VarBindings& bindings) const;

      VecPosition getRandomPtOut(RegionWorldModelInterface *wmi, 
				 const VarBindings& bindings) const;

      VecPosition getClosestPtTo(const VecPosition& pt, 
				 RegionWorldModelInterface *wmi,
				 const VarBindings& bindings) const;

      double getArea(RegionWorldModelInterface *wmi,
		     const VarBindings &bindings) const
      {
	/* Calculate the span angle */
	double spanAngle = fabs(fmod(M_span_ang, 360.0));

	/* Calculate the outer area */
	double outerArea = M_PI * M_end_rad * M_end_rad * spanAngle / 360.0;

	/* Calculate the inner area */
	double innerArea = M_PI * M_start_rad * M_start_rad * spanAngle/360.0;

	/* Return the difference */
	return outerArea - innerArea;
      }


    private:
      /* start rad <= end_rad */
      double M_start_rad, M_end_rad;
      double M_start_ang, M_span_ang;


    public:

      /* RegionIterator stuff for RegArc */
      class RegArcIterator: public RegionIterator
      {
      public:
	RegArcIterator(double d, RegArc* p, const VarBindings& bindings)
	  : RegionIterator(d), pReg(p), myBindings(bindings)
	{
	  initializeInfo();
	  resetToBegin();
	}

	RegArcIterator(RegArc *p, const VarBindings& bindings)
	  : RegionIterator(), pReg(p), myBindings(bindings)
	{
	  initializeInfo();
	  resetToBegin();
	}

	VecPosition getPt(RegionWorldModelInterface *wmi, const VarBindings& s)
	{
	  
	  return ((Point *) pReg->getCenter())->getGlobalPosition(wmi, s) + 
	    VecPosition::getVecPositionFromPolar(currDist, 
						 startAng + 
						 currAng);
	}

	void moveToNextPt()
	{
	  /* Increment the angle */
	  currAng += currAngIncr;
	  
	  /* If the current row is done, advance to next row */
	  if(currAng > spanAng) {
	    currAng = 0;
	    currDist += dist_pt;
	    
	    if(currDist > endRad)
	      error_flag = true;

	    currAngIncr = getAngIncrForDist(currDist);
	  }
	}

	void resetToBegin()
	{
	  currAng = 0;
	  currDist = startRad;
	  currAngIncr = getAngIncrForDist(currDist);
	  error_flag = false;
	}
	
      private:
	RegArc *pReg;

	double currAng;   /* displacement from startAng */
	double currDist;  /* current distance from the center */
	double currAngIncr;   /* angle increment */

	double startRad;
	double endRad;
	double startAng;
	double spanAng;

	VarBindings myBindings;

	double getAngIncrForDist(double d)
	  {
	    double tmp = 2.0 * d * d;
	    double tmp2 = (tmp - dist_pt * dist_pt) / tmp;
	    
	    if(d == 0.0 || tmp2 < -1.0 || tmp2 > 1.0)
	      return 500.0;
	    
	    return acosDeg(tmp2);
	  }

	void initializeInfo()
	  {
	    startRad = pReg->getStartRad();
	    endRad = pReg->getEndRad();
	    startAng = pReg->getStartAng();
	    spanAng = pReg->getSpanAng();

	    currAngIncr = getAngIncrForDist(currDist);
	  }
      };

      RegionIterator* newIterator(double d, RegionWorldModelInterface *,
				  const VarBindings& bindings)
      {
	return new RegArcIterator(d, this, bindings);
      }

      RegionIterator* newIterator(RegionWorldModelInterface *,
				  const VarBindings& bindings)
      {
	return new RegArcIterator(this, bindings);
      }

      /* draw the region on the field */
      void
      draw(FieldImage* pfi, bool verbose,
	   const FieldImage::Color& c_border, 
	   const FieldImage::Color& c_inside,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;

    };


    class RegUnion
      : public Region,
        public util::HasMany< std::auto_ptr< Region > >
    {
    public:
      typedef util::HasMany< std::auto_ptr< Region > > HasManyRegions;

      RegUnion() 
        : Region(),
          HasManyRegions()
      {}
      
      RegUnion( const Storage& regs) 
        : Region(),
          HasManyRegions( regs )
      {}

      virtual
      ~RegUnion()
      {}

      std::ostream& 
      print( std::ostream& out ) const;
      
      std::ostream& 
      printPretty( std::ostream& out, const std::string& line_header ) const;

      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      virtual
      std::auto_ptr< Region >
      deepCopy() const
      { return std::auto_ptr< Region >( new RegUnion( *this ) ); }

      Storage&
      getRegions()
      { return get(); }

      const Storage&
      getRegions() const
      { return get(); }

      /* Added in by Dongryeol Lee */
      bool isPtIn(const VecPosition& p, 
		  RegionWorldModelInterface *wmi,
		  const VarBindings& bindings) const;

      std::vector <VecPosition> getIntersection(const Line &l, 
						RegionWorldModelInterface *wmi,
						const VarBindings &bindings)
	const;
      
      VecPosition getRandomPtIn(RegionWorldModelInterface *wmi, 
				const VarBindings &bindings) const;
      
      VecPosition getRandomPtOut(RegionWorldModelInterface *wmi, 
				 const VarBindings &bindings) const;
      
      VecPosition getClosestPtTo(const VecPosition &pt, 
				 RegionWorldModelInterface *wmi,
				 const VarBindings &bindings) const;
      

      double getArea(RegionWorldModelInterface *wmi,
		     const VarBindings &bindings) const
      {
	double totalArea = 0.0;


	/* Get the list of all regions */
	RegUnion::HasManyRegions::Storage allRegions = getRegions();

	
	/* Iterate through the children and visit them in order */
	for(RegUnion::HasManyRegions::Storage::iterator it=allRegions.begin(); 
	    it != allRegions.end(); it++) {
	  
	  Region *child = *it;
	  
	  totalArea += child->getArea(wmi, bindings);
	}
	
	return totalArea;
      }




      /* RegionIterator stuff for RegUnion */
      class RegUnionIterator: public RegionIterator
      {
      public:
	RegUnionIterator(double d, RegUnion *p, RegionWorldModelInterface *wmi,
			 const VarBindings &bindings)
	  : RegionIterator(d), lIter(), pReg(p), myBindings(bindings)
	{
	  /* Get the list of all regions */
	  RegUnion::HasManyRegions::Storage allRegions = p->getRegions();

	  /* Iterate through the children and visit them in order */
	  for(RegUnion::HasManyRegions::Storage::iterator it=allRegions.begin(); 
	      it != allRegions.end(); it++) {
	    
	    Region *child = *it;

	    lIter.push_back(child->newIterator(d, wmi, bindings));
	  }

	  resetToBegin();
	}
	
	/* Destructor to clean up memory */
	virtual ~RegUnionIterator()
	{
	  std::list <RegionIterator *>::iterator iter;

	  for(iter = lIter.begin(); iter != lIter.end(); ++iter)
	    delete(*iter);
	}
	
	RegUnionIterator(RegUnion *p, RegionWorldModelInterface *wmi,
			 const VarBindings &bindings)
	  : RegionIterator(), lIter(), pReg(p), myBindings(bindings)
	{
	  /* Get the list of all regions */
	  RegUnion::HasManyRegions::Storage allRegions = p->getRegions();
	  
	  /* Iterate through the children and visit them in order */
	  for(RegUnion::HasManyRegions::Storage::iterator it=allRegions.begin(); 
	      it != allRegions.end(); it++) {
	    
	    Region *child = *it;
	    
	    lIter.push_back(child->newIterator(wmi, bindings));
	  }
	  
	  resetToBegin();	  
	}

	VecPosition getPt(RegionWorldModelInterface *wmi, const VarBindings& s)
	{
	  return (*iter_curr)->getPt(wmi, s);
	}

	void moveToNextPt()
	{
	  (*iter_curr)->moveToNextPt();
	  forward_to_valid();
	}

	void forward_to_valid()
	{
	  while(!(*iter_curr)->valid()) {
	    iter_curr++;

	    if(iter_curr == lIter.end()) {
	      error_flag = true;
	      break;
	    }
	    (*iter_curr)->resetToBegin();
	  }
	}

	void resetToBegin()
	{
	  iter_curr = lIter.begin();
	  error_flag = false;
	  (*iter_curr)->resetToBegin();
	  forward_to_valid();
	}

      private:
	std::list <RegionIterator *> lIter;
	std::list <RegionIterator *>::iterator iter_curr;
	RegUnion *pReg;
	VarBindings myBindings;
      };

      RegionIterator* newIterator(double d, RegionWorldModelInterface *wmi,
				  const VarBindings& bindings)
      {
	return new RegUnionIterator(d, this, wmi, bindings);
      }

      RegionIterator* newIterator(RegionWorldModelInterface *wmi,
				  const VarBindings& bindings)
      {
	return new RegUnionIterator(this, wmi, bindings);
      }

      friend class RegUnionIterator;

      /* draw the region on the field */
      void
      draw(FieldImage* pfi, bool verbose,
	   const FieldImage::Color& c_border, 
	   const FieldImage::Color& c_inside,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;

    };


    class RegNamed
      : public Region
    {
    public:
      RegNamed( const std::string& name = "" ) 
        : Region(),
          M_name(name),
	  pReg(NULL)
      {}

      ~RegNamed()
      {}
      
      std::ostream&
      print( std::ostream& out ) const
      { return out << "\"" << M_name << "\""; }
  
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { 
        return out << line_header << "region named \"" 
                   << M_name << "\"" << std::endl;
      }

      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Region >
      deepCopy() const
      { return std::auto_ptr< Region >( new RegNamed( *this ) ); }

      /* class specific */
      std::string&
      getName()
      { return M_name; }
  
      const std::string&
      getName() const
      { return M_name; }
      
      /* dongryel: retrieve the pointer to the Region object */
      Region*
      getRegion() const
      { return pReg; }

      void
      setName( const std::string& name )
      { M_name = name; }

      /* dongryel: method to set the Region pointer */
      void
      setLink( Region *reg )
      { pReg = reg; }

      /* Added in by Dongryeol Lee */
      bool isPtIn(const VecPosition& p, 
		  RegionWorldModelInterface *wmi, 
		  const VarBindings& bindings) const;

      std::vector <VecPosition> getIntersection(const Line &l, 
						RegionWorldModelInterface *wmi,
						const VarBindings& bindings)
	const;
      
      VecPosition getRandomPtIn(RegionWorldModelInterface *wmi, 
				const VarBindings& bindings) const;

      VecPosition getRandomPtOut(RegionWorldModelInterface *wmi, 
				 const VarBindings& bindings) const;

      VecPosition getClosestPtTo(const VecPosition& pt, 
				 RegionWorldModelInterface *wmi,
				 const VarBindings& bindings) const;


      double getArea(RegionWorldModelInterface *wmi,
		     const VarBindings &bindings) const
      {
	return pReg->getArea(wmi, bindings);
      }



      RegionIterator* newIterator(double d, RegionWorldModelInterface *wmi,
				  const VarBindings& bindings)
      {
	if(pReg == NULL) {
	  printf("Warning: Region pointer is NULL\n");
	  return NULL;
	}
	else
	  return pReg->newIterator(d, wmi, bindings);
      }

      RegionIterator* newIterator(RegionWorldModelInterface *wmi,
				  const VarBindings &bindings)
      {
	if(pReg == NULL) {
	  printf("Warning: the Region pointer is NULL\n");
	  return NULL;
	}
	else
	  return pReg->newIterator(wmi, bindings);
      }

      /* draw the region on the field */
      void
      draw(FieldImage* pfi, bool verbose,
	   const FieldImage::Color& c_border, 
	   const FieldImage::Color& c_inside,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;

    private:
      std::string M_name;

      /* dongryel: the pointer to the Region object */
      Region *pReg;

    };

    /**** RegPoint ****/
    class RegPoint
      : public Region
    {
    public:
      RegPoint();
      RegPoint( std::auto_ptr< Point > point );
      RegPoint( const RegPoint& point ) ;
      ~RegPoint();

      RegPoint&
      operator=( const RegPoint& point );

      std::ostream&
      print( std::ostream& out ) const;
  
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const;
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Region >
      deepCopy() const
      { return std::auto_ptr< Region >( new RegPoint( *this ) ); }

      Point*
      getPoint();

      const Point*
      getPoint() const;

      std::auto_ptr< Point >
      detatchPoint();

      void
      setPoint( std::auto_ptr< Point > point );

      /* Added in by Dongryeol lee */
      bool isPtIn(const VecPosition& p, RegionWorldModelInterface *wmi, 
		  const VarBindings& bindings) const;

      std::vector <VecPosition> getIntersection(const Line &l, 
						RegionWorldModelInterface *wmi,
						const VarBindings& bindings)
	const;
      
      VecPosition getRandomPtIn(RegionWorldModelInterface *wmi, 
				 
				const VarBindings& bindings) const;

      VecPosition getRandomPtOut(RegionWorldModelInterface *wmi, 
				  
				 const VarBindings& bindings) const;

      VecPosition getClosestPtTo(const VecPosition& pt, 
				 RegionWorldModelInterface *wmi,
				  
				 const VarBindings& bindings) const;


      double getArea(RegionWorldModelInterface *wmi,
		     const VarBindings &bindings) const
      {
	return 0.0;
      }


      /* Iterator for RegPoint */
      class RegPointIterator: public RegionIterator
      {
      public:
	RegPointIterator(double d, RegPoint *p, const VarBindings& bindings)
	  : RegionIterator(d), pReg(p), myBindings(bindings)
	{}

	RegPointIterator(RegPoint *p, const VarBindings& bindings)
	  : RegionIterator(), pReg(p), myBindings(bindings) 
	{}
	
	VecPosition getPt(RegionWorldModelInterface *wmi, 
			  const VarBindings& s)
	{ return (pReg->getPoint())->getGlobalPosition(wmi, s); }
	
	void moveToNextPt() {error_flag = true; }
	
	void resetToBegin() { error_flag = false; }
	
      private:
	RegPoint *pReg;
	VarBindings myBindings;
      };

      RegionIterator* newIterator(double d, RegionWorldModelInterface *,
				  const VarBindings& bindings)
      {
	return new RegPointIterator(d, this, bindings);
      }

      RegionIterator* newIterator(RegionWorldModelInterface *,
				  const VarBindings& bindings)
      {
	return new RegPointIterator(this, bindings);
      }

      friend class RegPointIterator;

      /* draw the region on the field */
      void
      draw(FieldImage* pfi, bool verbose,
	   const FieldImage::Color& c_border, 
	   const FieldImage::Color& c_inside,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;

    private:
      std::auto_ptr< Point > M_point;
    };


    class RegTri
      : public Region,
        public util::HasN< std::auto_ptr< Point >, 3 >
    {
    public:
      typedef util::HasN< std::auto_ptr< Point >, 3 > Has3Points;

      RegTri();
      
      RegTri( std::auto_ptr< Point > pt0,
              std::auto_ptr< Point > pt1,
              std::auto_ptr< Point > pt2 );
      
      virtual
      ~RegTri();

      std::ostream&
      print( std::ostream& out ) const;
      
      std::ostream& 
      printPretty( std::ostream& out, const std::string& line_header ) const;

      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      virtual
      std::auto_ptr< Region >
      deepCopy() const
      { return std::auto_ptr< Region >( new RegTri( *this ) ); }

      const Point* 
      getPt( unsigned int i ) const
      { return get( i ); }
            
      void
      setAllPts( std::auto_ptr< Point > pt0,
                 std::auto_ptr< Point > pt1,
                 std::auto_ptr< Point > pt2 );


      /* Added in by Dongryeol Lee */
      bool isPtIn(const VecPosition& p, RegionWorldModelInterface *wmi, 
		  const VarBindings& bindings) const;
      
      std::vector <VecPosition> getIntersection(const Line &l, 
						RegionWorldModelInterface *wmi,
						const VarBindings& bindings)
	const;
      
      VecPosition getRandomPtIn(RegionWorldModelInterface *wmi, 
				const VarBindings& bindings) const;

      VecPosition getRandomPtOut(RegionWorldModelInterface *wmi, 
				 const VarBindings& bindings) const;

      VecPosition getClosestPtTo(const VecPosition& pt, 
				 RegionWorldModelInterface *wmi,
				 const VarBindings& bindings) const;

      double getArea(RegionWorldModelInterface *wmi,
		     const VarBindings &bindings) const
      {
	Point *pt0 = (Point *) getPt(0);
	Point *pt1 = (Point *) getPt(1);
	Point *pt2 = (Point *) getPt(2);
	VecPosition p0 = pt0->getGlobalPosition(wmi, bindings);
	VecPosition p1 = pt1->getGlobalPosition(wmi, bindings);
	VecPosition p2 = pt2->getGlobalPosition(wmi, bindings);

	return Triangle::calculateSignedTriangleArea(p0, p1, p2);
      }


      /* RegionIterator stuff for RegTri */
      /* Iterator based on barycentric coordinate */
      class RegTriIterator: public RegionIterator
      {
      public:
	RegTriIterator(double d, RegTri *p, RegionWorldModelInterface *wmi, 
		       const VarBindings& bindings)
	  : RegionIterator(d), pReg(p), myBindings(bindings)
	{
	  Point *pt0 = (Point *) pReg->getPt(0);
	  Point *pt1 = (Point *) pReg->getPt(1);
	  Point *pt2 = (Point *) pReg->getPt(2);
	  p0 = pt0->getGlobalPosition(wmi, myBindings);
	  p1 = pt1->getGlobalPosition(wmi, myBindings);
	  p2 = pt2->getGlobalPosition(wmi, myBindings);
	  
	  calcInterval();
	  resetToBegin();
	}

	RegTriIterator(RegTri *p, RegionWorldModelInterface *wmi, 
		       const VarBindings& bindings)
	  : RegionIterator(), pReg(p), myBindings(bindings)
	{
	  Point *pt0 = (Point *) pReg->getPt(0);
	  Point *pt1 = (Point *) pReg->getPt(1);
	  Point *pt2 = (Point *) pReg->getPt(2);
	  p0 = pt0->getGlobalPosition(wmi, myBindings);
	  p1 = pt1->getGlobalPosition(wmi, myBindings);
	  p2 = pt2->getGlobalPosition(wmi, myBindings);

	  calcInterval();
	  resetToBegin();
	}

	void calcInterval()
	{
	  /* Calculate the length of the side connecting v0 to v1 */
	  double side1 = p0.getDistanceTo(p1);
	  
	  interval = 1.0 / (side1 / dist_pt);
	}

	VecPosition getPt(RegionWorldModelInterface *wmi, 
			  const VarBindings& s);

	void moveToNextPt();
	
	void resetToBegin();

      private:
	RegTri *pReg;
	VecPosition p0;
	VecPosition p1;
	VecPosition p2;
	
	double alpha;
	double beta;
	double gamma;
	double interval;
	VarBindings myBindings;
      };

      RegionIterator* newIterator(double d, RegionWorldModelInterface *wmi,
				   
				  const VarBindings& bindings)
      {
	return new RegTriIterator(d, this, wmi, bindings);
      }

      RegionIterator* newIterator(RegionWorldModelInterface *wmi,
				   
				  const VarBindings& bindings)
      {
	return new RegTriIterator(this, wmi, bindings);
      }

      /* draw the region on the field */
      void
      draw(FieldImage* pfi, bool verbose,
	   const FieldImage::Color& c_border, 
	   const FieldImage::Color& c_inside,
	   RegionWorldModelInterface *wmi, 
	   const VarBindings& bindings) const;
    };



   class RegRec
      : public Region,
        public util::HasN< std::auto_ptr< Point >, 2 >
    {
    public:
      typedef util::HasN< std::auto_ptr< Point >, 2 > Has2Points;

      RegRec();
      
      RegRec( std::auto_ptr< Point > pt0,
              std::auto_ptr< Point > pt1 );
      
      RegRec( const Rectangle& r);
      
      virtual
      ~RegRec();

      std::ostream&
      print( std::ostream& out ) const;
      
      std::ostream& 
      printPretty( std::ostream& out, const std::string& line_header ) const;

      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }
      
      virtual
      std::auto_ptr< Region >
      deepCopy() const
      { return std::auto_ptr< Region >( new RegRec( *this ) ); }

      const Point* 
      getPt( unsigned int i ) const
      { return get( i ); }
      
      void
      setAllPts( std::auto_ptr< Point > pt0,
                 std::auto_ptr< Point > pt1 );


      /* Added in by Dongryeol Lee */
      bool isPtIn(const VecPosition &p, RegionWorldModelInterface *wmi, 
		   
		  const VarBindings &bindings) const;      

      std::vector <VecPosition> getIntersection(const Line &l, 
						RegionWorldModelInterface *wmi,
						const VarBindings& bindings)
	const;
            
      VecPosition getRandomPtIn(RegionWorldModelInterface *wmi, 
				const VarBindings& bindings) const;
      
      VecPosition getRandomPtOut(RegionWorldModelInterface *wmi, 
				 const VarBindings& bindings) const;

      VecPosition getClosestPtTo(const VecPosition& pt, 
				 RegionWorldModelInterface *wmi,
				 const VarBindings& bindings) const;


      double getArea(RegionWorldModelInterface *wmi,
		     const VarBindings &bindings) const
      {
	Point *pt0 = (Point *) getPt(0);
	Point *pt1 = (Point *) getPt(1);
	VecPosition p0 = pt0->getGlobalPosition(wmi, bindings);
	VecPosition p1 = pt1->getGlobalPosition(wmi, bindings);

	double xdiff = fabs(p0.getX() - p1.getX());
	double ydiff = fabs(p0.getY() - p1.getY());

	return xdiff * ydiff;
      }



      Rectangle convertToRectangle(RegionWorldModelInterface *wmi,
				   const VarBindings &bindings) const;
      
      /* RegionIterator stuff for RegRec */
      class RegRecIterator: public RegionIterator
      {
      public:
	RegRecIterator(double d, RegRec *p, RegionWorldModelInterface *wmi, 
		       const VarBindings& bindings)
	  : RegionIterator(d), pReg(p), myBindings(bindings)
	{
	  Point *pt0 = (Point *) pReg->getPt(0);
	  Point *pt1 = (Point *) pReg->getPt(1);

	  p0 = pt0->getGlobalPosition(wmi, myBindings);
	  p1 = pt1->getGlobalPosition(wmi, myBindings);
	  
	  resetToBegin();
	}
	
	RegRecIterator(RegRec *p, RegionWorldModelInterface *wmi, 
		       const VarBindings& bindings)
	  : RegionIterator(), pReg(p), myBindings(bindings)
	{
	  Point *pt0 = (Point *) pReg->getPt(0);
	  Point *pt1 = (Point *) pReg->getPt(1);
	  p0 = pt0->getGlobalPosition(wmi, myBindings);
	  p1 = pt1->getGlobalPosition(wmi, myBindings);
	  
	  resetToBegin();
	}
	
	VecPosition getPt(RegionWorldModelInterface *wmi, 
			  const VarBindings& s);

	void moveToNextPt();
	
	void resetToBegin();
	
      private:
	RegRec *pReg;
	
	VecPosition p0;
	VecPosition p1;

	double minx;
	double maxx;
	double miny;
	double maxy;

	double currX;
	double currY;

	VarBindings myBindings;
      };

      RegionIterator* newIterator(double d, RegionWorldModelInterface *wmi,
				  const VarBindings& bindings)
      {
	return new RegRecIterator(d, this, wmi, bindings);
      }

      RegionIterator* newIterator(RegionWorldModelInterface *wmi,
				  const VarBindings& bindings)
      {
	return new RegRecIterator(this, wmi, bindings);
      }

      /* draw the region on the field */
      void
      draw(FieldImage* pfi, bool verbose,
	   const FieldImage::Color& c_border, 
	   const FieldImage::Color& c_inside,
	   RegionWorldModelInterface *wmi,	   
	   const VarBindings& bindings) const;
    };
  }
}

#endif
