/* -*- Mode: C++ -*- */
/*
 *Copyright:

    Copyright (C) 2000, 2001 RoboCup Soccer Server Maintenance Group.
    	Patrick Riley, Tom Howard, Itsuki Noda,
	Mikhail Prokopenko, Jan Wendler 

    This file is a part of SoccerServer.

    This code is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/* This files defines a class region to specify regions for the coach messages */
#include "region.h"
#include "Logger.h"

using namespace std;
using namespace spades;


#ifndef EPSILON
#define EPSILON 0.01
#endif

/****************************************** Points **********************************/

namespace rcss
{
  namespace clang
  {

    /* Added in by Dongryeol Lee */
    /* Helper function for determining which number is the min stored in the
       vector. Returns the index number of the minimum value stored in it */
    int getMin(vector <double> v)
    {
      double min_value = 99999;
      int min_index = 0;

      for(int i = 0; i < v.size(); i++)
	
	/* Update minimum values and its index */
	if(v[i] < min_value) {
	  min_value = v[i];
	  min_index = i;
	}
      
      /* Return the index of the minimum value */
      return min_index; 
    }

    int getMax(vector <double> v)
    {
      double max_value = -99999;
      int max_index = 0;

      for(int i = 0; i < v.size(); i++)
	
	/* Update minimum values and its index */
	if(v[i] > max_value) {
	  max_value = v[i];
	  max_index = i;
	}
      
      /* Return the index of the minimum value */
      return max_index; 
    }

    /*** Newly added Point methods (imported from PointVisitor) class ****/
    VecPosition PointSimple::getGlobalPosition(RegionWorldModelInterface *wmi, 
					       const VarBindings& bindings) 
      const
    {
      VecPosition pos(getVec());
      if (wmi->useRightSideCoordinates())
	pos = pos.flipCoords();
      return pos;
    }

    VecPosition PointRel::getGlobalPosition(RegionWorldModelInterface *wmi, 
					    const VarBindings& bindings) const
    {
      /* Retrieve the "origin" of the point p */
      Point *origin = (Point *) getOrigin();
      
      /* Retrieve the offset from the "origin" */
      PointSimple offset = getOffset();

      // we don't have to mess with the right side coordinate here because the
      // getGlobalPosition will have already taken care of it
      return offset.getGlobalPosition(wmi, bindings) + 
	origin->getGlobalPosition(wmi, bindings);
    }

    VecPosition PointBall::getGlobalPosition(RegionWorldModelInterface *wmi, 
					     const VarBindings& bindings) const
    {
      /* Retrieve the global position of the ball from the world model */
      // we don't have to mess with the right side coordinate here because the
      // WorldModelInterface has already taken care of it
      return VecPosition(wmi->getBallPos());
    }
    
    VecPosition PointPlayer::getGlobalPosition(RegionWorldModelInterface *wmi,
					       const VarBindings& myBindings) 
      const
    {

      UNum::unum_t player = this->getUNum().getUNum();
      if(this->getUNum().getVar() != ""){ // look up player variable

	UNumSet set = myBindings.lookup(this->getUNum().getVar());

	/* LOGREMOVED 
	if(set.size() != 1)
	  Log.log(9, "[PointPlayer:getGlobalPosition]: WARNING: player point variable is not bound to a single player\n");
	*/

	player = ( *(set.begin()) ).getUNum();

	/* LOGREMOVED 
	Log.log(11, "[PointPlayer:getGlobalPosition]: expanded point player var %s to %d\n", 
		this->getUNum().getVar().c_str(), player);
	*/
      }


      // we don't have to mess with the right side coordinate here because the
      // WorldModelInterface has already taken care of it
      return VecPosition(wmi->getGlobalPosition(player, isOurSide()));
    }   

    VecPosition PointArith::getGlobalPosition(RegionWorldModelInterface *wmi, 
					      const VarBindings& bindings) 
      const
    {
      const Point *left = getPoint(0);
      const Point *right = getPoint(1);
      const util::ArithOp *operation = getOp();
      
      /* Retrieve the result of the visiting for left and right */
      // we don't have to mess with the right side coordinate here because the
      // getGlobalPosition has already done it
      VecPosition leftPos = left->getGlobalPosition(wmi, bindings);
      VecPosition rightPos = right->getGlobalPosition(wmi, bindings);
      
      /* perform the operation on two points */
      return operation->operate(leftPos, rightPos);
    }


    void
    Point::draw(FieldImage* pfi, bool verbose, const FieldImage::Color& c,
		RegionWorldModelInterface *wmi, 
		const VarBindings& bindings) const
    {
      VecPosition v = getGlobalPosition(wmi, bindings);
      draw(pfi, verbose, c, v);
    }

    void
    Point::draw(FieldImage* pfi, bool verbose, const FieldImage::Color& c,
		VecPosition global_pos) const
    {
      pfi->addPoint(global_pos, c);
    }

    void
    PointSimple::draw(FieldImage* pfi, 
		      bool verbose, 
		      const FieldImage::Color& c,
		      RegionWorldModelInterface *wmi, 
		      const VarBindings& bindings) const
    {
      Point::draw(pfi, verbose, c, wmi, bindings);
    }

    void
    PointRel::draw(FieldImage* pfi, 
		   bool verbose, 
		   const FieldImage::Color& c,
		   RegionWorldModelInterface *wmi, 
		   const VarBindings& bindings) const
    {
      VecPosition global_pos = getGlobalPosition(wmi, bindings);
      Point::draw(pfi, verbose, c, global_pos);
      if (verbose)
	{
	  VecPosition origin_pos = getOrigin()->getGlobalPosition(wmi, bindings);
	  pfi->addPoint(origin_pos, c);
	  pfi->addArrow(origin_pos, global_pos, c);
	}
    }

    void
    PointBall::draw(FieldImage* pfi, 
		    bool verbose, 
		    const FieldImage::Color& c,
		    RegionWorldModelInterface *wmi, 
		    const VarBindings& bindings) const
    {
      VecPosition global_pos = getGlobalPosition(wmi, bindings);
      Point::draw(pfi, verbose, c, global_pos);
      if (verbose)
	{
	  pfi->addText("B", global_pos, c);
	}
    }

    void
    PointPlayer::draw(FieldImage* pfi, 
		      bool verbose, 
		      const FieldImage::Color& c,
		      RegionWorldModelInterface *wmi, 
		      const VarBindings& bindings) const
    {
      VecPosition global_pos = getGlobalPosition(wmi, bindings);
      Point::draw(pfi, verbose, c, global_pos);
      if (verbose)
	{
	  ostrstream text;
	  text << (M_our_side ? 'T' : 'O') << '-' << M_unum << ends;
	  pfi->addText(text.str(), global_pos, c);
	  text.freeze(0);
	}
    }

    void
    PointArith::draw(FieldImage* pfi, 
		     bool verbose, 
		     const FieldImage::Color& c,
		     RegionWorldModelInterface *wmi, 
		     const VarBindings& bindings) const
    {
      VecPosition global_pos = getGlobalPosition(wmi, bindings);
      Point::draw(pfi, verbose, c, global_pos);
      if (verbose)
	{
	  //SMURF: It would be nice to show more about this arithmetic, but I don't
	  // know of a good way to show it
	  pfi->addText("A", global_pos, c);
	}
    }

  }
}


/****************************************** Regions **********************************/

std::ostream&
operator <<(std::ostream & os, const rcss::clang::Region& r )
{
  return r.print(os);
} 

namespace rcss
{
  namespace clang
  {

    /* Newly added Region Methods (Imported from RegionVisitor class) */
    bool RegNull::isPtIn(const VecPosition &p, RegionWorldModelInterface *wmi, 
			 const VarBindings& bindings) const
    { return false; }

    vector <VecPosition> RegNull::getIntersection(const Line &l,
						  RegionWorldModelInterface *wmi, 
						  const VarBindings& bindings) const
    {
      vector <VecPosition> v;

      /* Return an empty vector */
      return v;
    }

    /* Return a NULL point */
    VecPosition RegNull::getRandomPtIn(RegionWorldModelInterface *wmi, 
				       const VarBindings& bindings) const
    {
      return VecPosition(0.0, 0.0);
    }

    /* Return a NULL point */
    VecPosition RegNull::getRandomPtOut(RegionWorldModelInterface *wmi,
					const VarBindings& bindings) const
    {
      return VecPosition(0.0, 0.0);
    }

    /* Return the pt itself */
    VecPosition RegNull::getClosestPtTo(const VecPosition &pt, 
					RegionWorldModelInterface *wmi,
					const VarBindings& bindings) const
    {
      return pt;
    }

    /* draw the region on the field */
    void
    RegNull::draw(FieldImage* pfi, 
		  bool verbose,
		  const FieldImage::Color& c_border, 
		  const FieldImage::Color& c_inside,
		  RegionWorldModelInterface *wmi, 
		  const VarBindings& bindings) const
    {
      if (verbose)
	pfi->addLegendLine("Null region");
    }

    
    /* Quad regions are unsupported, so return whatever */
    bool RegQuad::isPtIn(const VecPosition& p, 
			 RegionWorldModelInterface *wmi, 
			 const VarBindings& bindings) const
    { return false; }
    
    vector <VecPosition> RegQuad::getIntersection(const Line &l,	  
						  RegionWorldModelInterface *wmi, 
						  const VarBindings& bindings) const
    {
      vector <VecPosition> v;
      
      /* Return an empty vector */
      return v;
    }

    /* Return a NULL point */
    VecPosition RegQuad::getRandomPtIn(RegionWorldModelInterface *wmi, 
				       const VarBindings &bindings) const
    {
      return VecPosition(0.0, 0.0);
    }

    /* Return a NULL point */
    VecPosition RegQuad::getRandomPtOut(RegionWorldModelInterface *wmi, 
					const VarBindings& bindings) const
    {
      return VecPosition(0.0, 0.0);
    }

    /* Return the point itself */
    VecPosition RegQuad::getClosestPtTo(const VecPosition& pt, 
					RegionWorldModelInterface *wmi,
					const VarBindings& bindings) const
    {
      return pt;
    }


    
    bool RegArc::isPtIn(const VecPosition &p, 
			RegionWorldModelInterface *wmi, 
			const VarBindings &bindings) const
    {
      /* Calculate the directional vector connecting the center and p, 
	 and its direction and its magnitude  */
      Point *center = (Point *) getCenter();
      
      VecPosition centerTop = p - center->getGlobalPosition(wmi, bindings);
      double dist = centerTop.getMagnitude();
      double startAng = (getSpanAng() >= 0 ? getStartAng() : getStartAng() + getSpanAng());
      double spanAng = fabs(getSpanAng());
      /* pfr: I thought I needed to rotate around, but you actually don't
	 By flipping all cordinates, the the arc starting point is handled
	 automatically; Really! Try it out!
      //Rotate around for the right side
      if (wmi->useRightSideCoordinates())
	startAng += 180;
      */
      double ang_from_start =
	VecPosition::normalizeAngle(centerTop.getDirection() - startAng);
      /* If the angle is negative,add 360 degrees so that it's comparable to the
	 span angle. */
      if(ang_from_start < 0)
	ang_from_start += 360.0;
      
      return (ang_from_start >= -EPSILON &&
	      ang_from_start <= spanAng + EPSILON &&
	      dist >= getStartRad() - EPSILON && 
	      dist <= getEndRad() + EPSILON);
    }
    
    vector <VecPosition> RegArc::getIntersection(const Line &l, 
						 RegionWorldModelInterface *wmi,
						 const VarBindings& bindings) const
    {
      Point *ct = (Point *) getCenter();
      VecPosition center = ct->getGlobalPosition(wmi, bindings);
      
      /* Retrieve start/end radius, start/span angles */
      double startRad = getStartRad();
      double endRad = getEndRad();

      /* Solution vector for the intersection */
      VecPosition i1, i2;
      
      /* List of solutions to return */
      vector <VecPosition> solution;
      
      /* Number of solutions returned from the intersecting */
      int numSolutions;
      
      /* 1: Solve the intersection with the first inner circle */
      numSolutions = l.getCircleIntersectionPoints
	(Circle(center, startRad), &i1, &i2);
      
      /* If there is at least one solution, take the first solution 
	 and generate random points */
      if(numSolutions > 0 && isPtIn(i1, wmi, bindings))
	solution.push_back(i1);
      
      /* If there are two solutions, take the next one also. */
      if(numSolutions > 1 && isPtIn(i2, wmi, bindings))
	solution.push_back(i2);
      
      /* 2: Solve the intersection with the outer circle */
      numSolutions = l.getCircleIntersectionPoints(Circle(center, endRad), &i1, &i2);
      
      /* If there is at least one solution, take the first solution and 
	 generate random points */
      if(numSolutions > 0 && isPtIn(i1, wmi, bindings))
	solution.push_back(i1);
      
      /* If there are two solutions, take the next one also. */
      if(numSolutions > 1 && isPtIn(i2, wmi, bindings))
	solution.push_back(i2);
      
      return solution;
    }

    /* Basically randomly generate a random radius between starting radius 
       and the ending radius, and the angle between starting angle and 
       spanning one. Then, generate a VecPosition that is offset from the 
       center. The result is simply adding this vector with the center 
       vector position */
    VecPosition RegArc::getRandomPtIn(RegionWorldModelInterface *wmi, 
				      const VarBindings &bindings) const
    {
      /* Calculate the directional vector connecting the center and p, 
	 and its direction and its magnitude  */
      Point *ct = (Point *) getCenter();
      
      VecPosition center = ct->getGlobalPosition(wmi, bindings);
      double startRad = getStartRad();
      double endRad = getEndRad();
      double startAng = getStartAng();
      double spanAng = getSpanAng();

      /* pfr: I thought I needed to rotate around, but you actually don't
	 By flipping all cordinates, the the arc starting point is handled
	 automatically; Really! Try it out!
      if (wmi->useRightSideCoordinates())
	startAng += 180;
      */
      
      /* Generate a random radius r, such that startRad <= r <= endRad. */
      double randomRad = (double) rand() / (double) RAND_MAX * 
	(endRad - startRad) + startRad;
      
      /* Generate a random angle between startAng and spanAng */
      double randomAng = (double) rand() / (double) RAND_MAX *
	spanAng + startAng;
      
      return (center + VecPosition::getVecPositionFromPolar(randomRad, randomAng));
    }

    VecPosition RegArc::getRandomPtOut(RegionWorldModelInterface *wmi, 
				       const VarBindings& bindings) const
    {
      /* Select two random points inside the region */
      VecPosition rp0 = getRandomPtIn(wmi, bindings);
      VecPosition rp1 = getRandomPtIn(wmi, bindings);
      
      /* Construct the line connecting two random points */
      Line l = Line::makeLineFromTwoPoints(rp0, rp1);
      
      /* Intersections vector */
      vector <VecPosition> result = getIntersection(l, wmi, bindings);

      /* If there is at least one intersection, return the first one */
      if(result.size() > 0)
	return result[0];

      /* Otherwise, return dummy position */
      else
	return VecPosition(0.0, -35.0);
    }

    VecPosition RegArc::getClosestPtTo(const VecPosition &pt, 
				       RegionWorldModelInterface *wmi,
				       const VarBindings& bindings) const
    {
      /* Return this final point */
      VecPosition closePt;


      /* Retrieve the center point */
      Point *ct = (Point *) getCenter();
      
      /* And get its global position */
      VecPosition center = ct->getGlobalPosition(wmi, bindings);

      /* Retrieve all the information about the arc region */
      double startRad = getStartRad();
      double endRad = getEndRad();
      double startAng = getStartAng();
      double spanAng = getSpanAng();

      /* pfr: I thought I needed to rotate around, but you actually don't
	 By flipping all cordinates, the the arc starting point is handled
	 automatically; Really! Try it out!
      if (wmi->useRightSideCoordinates())
	startAng += 180;
      */
      
      /* Some boolean flags to determine the relative position of the point inside
	 the arc region */
      bool insideDistanceWise = true;  /* true if the distance between the center
					  and the point is between start rad and
					  end rad */
      bool insideAngleWise = true;  /* true if the angle of the vector from the
				       center to the point is between start ang and
				       end ang */

      /* Used to determine which boundary the point is closest to */
      double nearDist = 0;
      double nearAng = 0;


      /* Calculate the distance from the center to the point you want to get
	 closest to. Get the angle, too, and subtract startAng from it */
      double dist = center.getDistanceTo(pt);
      double ang = (pt - center).getDirection() - startAng;
      
      ang = VecPosition::normalizeAngle(ang);
      if(ang < 0)
	ang += 360;

      /* Determine whether the point is inside the arc region distance wise */
      /* The following two cases deal with the case when the point is outside the arc 
         distance wise */
      if(dist < startRad) {
	insideDistanceWise = false;
	nearDist = startRad;
      }
      else if(dist > endRad) {
	insideDistanceWise = false;
	nearDist = endRad;
      }

      /* Determine whether the point is between start angle and end angle */
      /* The folliwng two cases deal with the case when the point is outisde the arc
	 angle wise */
      if(ang > spanAng) {
	insideAngleWise = false;

	if(ang - spanAng < 360 - ang)
	  nearAng = startAng + spanAng;
	else
	  nearAng = startAng;
      }


      /* Now, we have 4 more cases to deal with */
      if(insideDistanceWise) {
	if(insideAngleWise)

	  /* If the point we want to get close to inside the region already, return
	     that point */
	  closePt = pt;

	/* If the point we want is inside distance wise, but out of the angle
	   boundary, */
	else
	  closePt = Line::getClosestPtInBetween(pt, center + 
						VecPosition::getVecPositionFromPolar
						(startRad, nearAng),
						center +
						VecPosition::getVecPositionFromPolar
						(endRad, nearAng));
      }
      else {
	if(insideAngleWise)
	  closePt = center + VecPosition::getVecPositionFromPolar(nearDist,
								  ang + startAng);
	else
	  closePt = center + VecPosition::getVecPositionFromPolar(nearDist,
								  nearAng);
      }

      /* Return the calculated closest point */
      return closePt;
    }


    /* draw the region on the field */
    void
    RegArc::draw(FieldImage* pfi, 
		 bool verbose,
		 const FieldImage::Color& c_border, 
		 const FieldImage::Color& c_inside,
		 RegionWorldModelInterface *wmi, 
		 const VarBindings& bindings) const
    {
      VecPosition center = getCenter()->getGlobalPosition(wmi, bindings);
      pfi->addArc(center, M_start_rad, M_end_rad, M_start_ang, M_span_ang, c_border, c_inside);
      getCenter()->draw(pfi, verbose, c_border, wmi, bindings);
    }
    

    bool RegUnion::isPtIn(const VecPosition &p,
			  RegionWorldModelInterface *wmi, 
			  const VarBindings &bindings) const
    {
      /* Get the list of all regions */
      const RegUnion::HasManyRegions::Storage& allRegions = getRegions();
      
      /* Iterate through the children and visit them in order */
      for(RegUnion::HasManyRegions::Storage::const_iterator it=allRegions.begin(); 
	  it != allRegions.end(); it++) {

	Region *child = *it;
	  
	if(child->isPtIn(p, wmi, bindings))
	  {
	    actionlog(210) << "RegUnion::IsPtIn: yes for: " << *child << ende;
	    return true;
	  }
      }
      actionlog(210) << "RegUnion::IsPtIn: not in: " << *this << ende;
      
      return false;
    }

    vector <VecPosition> RegUnion::getIntersection(const Line &l, 
						   RegionWorldModelInterface *wmi,
						   const VarBindings &bindings) const
    {
      /* Get the list of all regions */
      const RegUnion::HasManyRegions::Storage& allRegions = getRegions();
      
      /* Store all the intersections into this vector */
      vector <VecPosition> result;
      
      /* Iterate through the children and visit them in order */
      for(RegUnion::HasManyRegions::Storage::const_iterator it=allRegions.begin(); 
	  it != allRegions.end(); it++) {
	
	Region *child = *it;
	vector <VecPosition> semiresult = child->getIntersection(l, wmi, bindings);
	
	/* Prepend the intersections of each sub-region to the entire list */
	result.insert(result.begin(), semiresult.begin(), semiresult.end());
      }
      
      return result;     
    }

    VecPosition RegUnion::getRandomPtIn(RegionWorldModelInterface *wmi, 
					const VarBindings& bindings) const
    {
      /* Get the list of all regions */
      const RegUnion::HasManyRegions::Storage& allRegions = getRegions();
      
      if(allRegions.size() == 0)
	return VecPosition(0.0, 0.0);
      else {
	RegUnion::HasManyRegions::Storage::const_iterator it = allRegions.begin();
	
	return (*it)->getRandomPtIn(wmi, bindings);
      }
    }	

    VecPosition RegUnion::getRandomPtOut(RegionWorldModelInterface *wmi,
					 const VarBindings& bindings) const
    {
      /* Select two random points inside the region */
      VecPosition rp0 = getRandomPtIn(wmi, bindings);
      VecPosition rp1 = getRandomPtIn(wmi, bindings);
      
      /* Construct the line connecting two random points */
      Line l = Line::makeLineFromTwoPoints(rp0, rp1);
      
      /* Intersections vector */
      vector <VecPosition> result = getIntersection(l, wmi, bindings);

      /* If there is at least one intersection, return the first one */
      if(result.size() > 0)
	return result[0];

      /* Otherwise, return dummy position */
      else
	return VecPosition(0.0, -35.0);
    }
      
    VecPosition RegUnion::getClosestPtTo(const VecPosition &pt, 
					 RegionWorldModelInterface *wmi,
					 const VarBindings &bindings) const
    {
      /* First check if the point is in the region itself; then return it */
      if(isPtIn(pt, wmi, bindings))
	return pt;
      
      /* Else, get the closest point in one of the regions */
      else {
	/* Get the list of all regions */
	const RegUnion::HasManyRegions::Storage& allRegions = getRegions();
	
	if(allRegions.size() == 0)
	  return VecPosition(0.0, 0.0);
	else {
	  RegUnion::HasManyRegions::Storage::const_iterator it = allRegions.begin();
	  
	  return (*it)->getClosestPtTo(pt, wmi, bindings);
	}
      }
    }

    /* draw the region on the field */
    void
    RegUnion::draw(FieldImage* pfi, 
		   bool verbose,
		   const FieldImage::Color& c_border, 
		   const FieldImage::Color& c_inside,
		   RegionWorldModelInterface *wmi, 
		   const VarBindings& bindings) const
    {
      for(RegUnion::HasManyRegions::Storage::const_iterator it=
	    getRegions().begin(); it != getRegions().end(); it++)
	{
	  (*it)->draw(pfi, verbose, c_border, c_inside, wmi, bindings);
	}
    }


    bool RegNamed::isPtIn(const VecPosition &p,RegionWorldModelInterface *wmi, 
			  const VarBindings& bindings) const
    {
      //std::string regName = (std::string) getName();
      //Region *theregion;
      //bool found = msi->lookup(regName, &theregion);
      
      if(!pReg){
	/* LOGREMOVED 
	Log.log(9, "[RegNamed:isPtIn]: WARNING: Could not find named region %s\n", regName.c_str());
	*/
	return false;
      }
      
      return pReg->isPtIn(p, wmi, bindings);
    }

    vector <VecPosition> RegNamed::getIntersection(const Line &l, 
						   RegionWorldModelInterface *wmi,
						   const VarBindings &bindings) const
    {
      vector <VecPosition> v;

      //std::string regName = (std::string) getName();
      
      //Region *theregion;
      //bool found = msi->lookup(regName, &theregion);
      
      if(!pReg){
	/* LOGREMOVED 
	Log.log(9, "[RegNamed:getIntersection]: WARNING: Could not find named region %s\n", regName.c_str());
	*/
	return v;
      }

      return pReg->getIntersection(l, wmi, bindings);
    }

    VecPosition RegNamed::getRandomPtIn(RegionWorldModelInterface *wmi, 
					const VarBindings& bindings) const
    {
      //std::string regName = (std::string) getName();

      //Region *theregion;
      //bool found = msi->lookup(regName, &theregion);
      
      if(!pReg){
	/* LOGREMOVED 
	Log.log(9, "[RegNamed:getRandomPtIn]: WARNING: Could not find named region %s\n", regName.c_str());
	*/
	return VecPosition(0.0, 0.0);
      }

      return pReg->getRandomPtIn(wmi, bindings);
    }

    VecPosition RegNamed::getRandomPtOut(RegionWorldModelInterface *wmi,
					 const VarBindings &bindings) const
    {
      /* Select two random points inside the region */
      VecPosition rp0 = getRandomPtIn(wmi, bindings);
      VecPosition rp1 = getRandomPtIn(wmi, bindings);
      
      /* Construct the line connecting two random points */
      Line l = Line::makeLineFromTwoPoints(rp0, rp1);
      
      /* Intersections vector */
      vector <VecPosition> result = getIntersection(l, wmi, bindings);

      /* If there is at least one intersection, return the first one */
      if(result.size() > 0)
	return result[0];

      /* Otherwise, return dummy position */
      else
	return VecPosition(0.0, -35.0);
    }
    
    VecPosition RegNamed::getClosestPtTo(const VecPosition &pt, 
					 RegionWorldModelInterface *wmi,
					 const VarBindings &bindings) const
    {
      //std::string regName = (std::string) getName();

      //Region *theregion;
      //bool found = msi->lookup(regName, &theregion);
      
      if(!pReg){
	/* LOGREMOVED 
	Log.log(9, "[RegNamed:getRandomPtIn]: WARNING: Could not find named region %s\n", regName.c_str());
	*/
	return VecPosition(0.0, 0.0);
      }

      return pReg->getClosestPtTo(pt, wmi, bindings);      
    }

    /* draw the region on the field */
    void
    RegNamed::draw(FieldImage* pfi, 
		   bool verbose,
		   const FieldImage::Color& c_border, 
		   const FieldImage::Color& c_inside,
		   RegionWorldModelInterface *wmi, 
		   const VarBindings& bindings) const
    {
      //Region *theregion;

      //if (msi->lookup(getName(), &theregion))
      if(pReg)
	{
	  pReg->draw(pfi, verbose, c_border, c_inside, wmi, bindings);

	  if (verbose)
	    {
	      VecPosition pt = pReg->getRandomPtIn(wmi, bindings);
	      pfi->addText(getName().c_str(), pt, c_border);
	    }
	}
      else
	{
	  errorlog << "RegNamed(" << getName() << "): lookup failed in draw" << ende;
	}
    }


    bool RegPoint::isPtIn(const VecPosition &p, 
			  RegionWorldModelInterface *wmi, 
			  const VarBindings &bindings) const
    {
      VecPosition pos = (getPoint())->getGlobalPosition(wmi, bindings);

      return (fabs(p.getX() - pos.getX()) < EPSILON && fabs(p.getY() - pos.getY()) < EPSILON);
    }
    
    vector <VecPosition> RegPoint::getIntersection(const Line &l, 
						   RegionWorldModelInterface *wmi,
						   const VarBindings& bindings) const
    {
      VecPosition pt = (getPoint())->getGlobalPosition(wmi, bindings);

      /* Intersection vector */
      vector <VecPosition> v;

      /* If the point is on the line, then return the point */
      if(l.getDistanceWithPoint(pt) == 0)
	v.push_back(pt);

      return v;      
    }

    VecPosition RegPoint::getRandomPtIn(RegionWorldModelInterface *wmi, 
					const VarBindings& bindings) const
    {
      Point *pt = (Point *) getPoint();
      
      /* Only one random point in a point region */
      return pt->getGlobalPosition(wmi, bindings);
    }

    VecPosition RegPoint::getRandomPtOut(RegionWorldModelInterface *wmi,
					 const VarBindings& bindings) const
    {
      /* Select two random points inside the region */
      VecPosition rp0 = getRandomPtIn(wmi, bindings);
      VecPosition rp1 = getRandomPtIn(wmi, bindings);
      
      /* Construct the line connecting two random points */
      Line l = Line::makeLineFromTwoPoints(rp0, rp1);
      
      /* Intersections vector */
      vector <VecPosition> result = getIntersection(l, wmi, bindings);
      
      /* If there is at least one intersection, return the first one */
      if(result.size() > 0)
	return result[0];
      
      /* Otherwise, return dummy position */
      else
	return VecPosition(0.0, -35.0);
    }

    VecPosition RegPoint::getClosestPtTo(const VecPosition& pt, 
					 RegionWorldModelInterface *wmi,
					 const VarBindings& bindings) const
    {
      Point *pt2 = (Point *) getPoint();
      
      /* Only one random point in a point region */
      return pt2->getGlobalPosition(wmi, bindings);
    }

    /* draw the region on the field */
    void
    RegPoint::draw(FieldImage* pfi, bool verbose,
		   const FieldImage::Color& c_border, const FieldImage::Color& c_inside,
		   RegionWorldModelInterface *wmi, 
		   const VarBindings& bindings) const
    {
      getPoint()->draw(pfi, verbose, c_border, wmi, bindings);
    }


    bool RegTri::isPtIn(const VecPosition& p, RegionWorldModelInterface *wmi, 
			const VarBindings& bindings) const
    {
      /* calculate the area formed by triangle p0, p1, p, triangle p1, p2, p,
	 and triangle p2, p0, p */
      Point *pt0 = (Point *) getPt(0);
      Point *pt1 = (Point *) getPt(1);
      Point *pt2 = (Point *) getPt(2);
      VecPosition p0 = pt0->getGlobalPosition(wmi, bindings);
      VecPosition p1 = pt1->getGlobalPosition(wmi, bindings);
      VecPosition p2 = pt2->getGlobalPosition(wmi, bindings);
      
      /* Calculate the area of the three triangles */
      double area1 = Triangle::calculateSignedTriangleArea(p, p0, p1);
      double area2 = Triangle::calculateSignedTriangleArea(p, p1, p2);
      double area3 = Triangle::calculateSignedTriangleArea(p, p2, p0);
      
      /* And check if they all have the same signs */
      return((area1 >= 0 && area2 >= 0 && area3 >= 0) ||
	     (area1 <= 0 && area2 <= 0 && area3 <= 0));
    }
    
    vector <VecPosition> RegTri::getIntersection(const Line& l, 
						 RegionWorldModelInterface *wmi,
						 const VarBindings& bindings) const
    {
      /* Retrieve three points of the triangles as VecPosition objects */
      Point *pt0 = (Point *) getPt(0);
      Point *pt1 = (Point *) getPt(1);
      Point *pt2 = (Point *) getPt(2);
      VecPosition v0 = pt0->getGlobalPosition(wmi, bindings);
      VecPosition v1 = pt1->getGlobalPosition(wmi, bindings);
      VecPosition v2 = pt2->getGlobalPosition(wmi, bindings);

      /* Generate the line for three sides, and calculate intersections */
      Line l1 = Line::makeLineFromTwoPoints(v0, v1);
      Line l2 = Line::makeLineFromTwoPoints(v1, v2);
      Line l3 = Line::makeLineFromTwoPoints(v2, v0);
      
      /* Calculate the intersections with the three sides of the triangle with the given line */
      VecPosition s1 = l.getIntersection(l1);
      VecPosition s2 = l.getIntersection(l2);
      VecPosition s3 = l.getIntersection(l3);
      
      vector <VecPosition> solution;   /* solution list */
      
      /* Push back three intersections, if they are at the boundary of the region */
      if(isPtIn(s1, wmi, bindings))
	solution.push_back(s1);
      if(isPtIn(s2, wmi, bindings))
	solution.push_back(s2);
      if(isPtIn(s3, wmi, bindings))
	solution.push_back(s3);
      
      /* Return the solution */
      return solution;
    }

    VecPosition RegTri::getRandomPtIn(RegionWorldModelInterface *wmi, 
				      const VarBindings& bindings) const
    {
      Point *pt0 = (Point *) getPt(0);
      Point *pt1 = (Point *) getPt(1);
      Point *pt2 = (Point *) getPt(2);
      VecPosition v0 = pt0->getGlobalPosition(wmi, bindings);
      VecPosition v1 = pt1->getGlobalPosition(wmi, bindings);
      VecPosition v2 = pt2->getGlobalPosition(wmi, bindings);
      
      /* This method will use the fact that all point inside a region of 
	 points, P0, P1, ... Pn can be represented by: P = a0 * P0 + 
	 a1 * P1 + .... an * Pn, where a0 + a1 + ... + an = 1. */
      double a0 = (double) rand() / (double) RAND_MAX;
      double remainder = 1.0 - a0;
      
      double a1 = (double) rand() / (double) RAND_MAX * remainder;
      double a2 = 1.0 - a0 - a1;
      
      /* Return the point inside the triangle */
      return (v0 * a0 + v1 * a1 + v2 * a2);
    }

    VecPosition RegTri::getRandomPtOut(RegionWorldModelInterface *wmi,
				       const VarBindings &bindings) const
    {
      /* Select two random points inside the region */
      VecPosition rp0 = getRandomPtIn(wmi, bindings);
      VecPosition rp1 = getRandomPtIn(wmi, bindings);
      
      /* Construct the line connecting two random points */
      Line l = Line::makeLineFromTwoPoints(rp0, rp1);
      
      /* Intersections vector */
      vector <VecPosition> result = getIntersection(l, wmi, bindings);
      
      /* If there is at least one intersection, return the first one */
      if(result.size() > 0)
	return result[0];
      
      /* Otherwise, return dummy position */
      else
	return VecPosition(0.0, -35.0);
    }

    VecPosition RegTri::getClosestPtTo(const VecPosition& pt, 
				       RegionWorldModelInterface *wmi,
				       const VarBindings &bindings) const
    {
      /* First test if the point lies inside the triangle region */
      if(isPtIn(pt, wmi, bindings))
	return pt;
      
      /* Else, we have to do the actual calculation */
      
      /* Retrieve all three points of the triangle */
      Point *pt0 = (Point *) getPt(0);
      Point *pt1 = (Point *) getPt(1);
      Point *pt2 = (Point *) getPt(2);
      VecPosition v0 = pt0->getGlobalPosition(wmi, bindings);
      VecPosition v1 = pt1->getGlobalPosition(wmi, bindings);
      VecPosition v2 = pt2->getGlobalPosition(wmi, bindings);
      
      /* Construct three lines representing each side */
      Line side1 = Line::makeLineFromTwoPoints(v0, v1);
      Line side2 = Line::makeLineFromTwoPoints(v1, v2);
      Line side3 = Line::makeLineFromTwoPoints(v2, v0);
      
      /* Get the distance from the point pt to all three lines */
      double distToSide1 = side1.getDistanceWithPoint(pt);
      double distToSide2 = side2.getDistanceWithPoint(pt);
      double distToSide3 = side3.getDistanceWithPoint(pt);
      
      /* Vector to store distances to each side */
      vector <double> dist;

      /* Push back all three distances */
      dist.push_back(distToSide1);
      dist.push_back(distToSide2);
      dist.push_back(distToSide3);


      /* Get the closest Point on the line */
      switch(getMin(dist)) {
      case 1:
	return Line::getClosestPtInBetween(pt, v0, v1);
	break;
      case 2:
	return Line::getClosestPtInBetween(pt, v1, v2);
	break;
      case 3:
	return Line::getClosestPtInBetween(pt, v2, v0);
	break;
      default:
	break;
      }
      
      /* Default return value */
      return VecPosition(0.0, 0.0);
    }

    VecPosition RegTri::RegTriIterator::getPt(RegionWorldModelInterface *wmi, 
					      const VarBindings& bindings)
    {
      VecPosition newv0 = ((Point *) pReg->getPt(0))->getGlobalPosition(wmi, bindings);
      VecPosition newv1 = ((Point *) pReg->getPt(1))->getGlobalPosition(wmi, bindings);
      VecPosition newv2 = ((Point *) pReg->getPt(2))->getGlobalPosition(wmi, bindings);

      return newv0 * alpha + newv1 * beta + newv2 * gamma;
    }
    
    /* Finish this later */
    void RegTri::RegTriIterator::moveToNextPt()
    {
      /* Given our barycentric coordinate (alpha, beta, gamma), */
     
      /* Gamma coordinate is fixed */

      /* First increase the s coordinate */
      alpha += interval;

      /* Calculate our new t coordinate */
      beta = 1.0 - alpha - gamma;

      /* If we are done with this scan line, go to the next line */
      if(beta < 0.0) {
	
	alpha = 0;
	gamma += interval;
	beta = 1.0 - gamma;
      }

      /* Test if, we are at the third vertex. If we are, we are done */
      if(gamma > 1.0)
	error_flag = true;
    }

    void RegTri::RegTriIterator::resetToBegin()
    { 
      /* First point that will be retrieved is v0 */
      
      /* Set our barycentric coordinate to (1, 0, 0) */
      alpha = 1.0;
      beta = 0.0;
      gamma = 0.0;
    }

    /* draw the region on the field */
    void
    RegTri::draw(FieldImage* pfi, bool verbose,
		 const FieldImage::Color& c_border, 
		 const FieldImage::Color& c_inside,
		 RegionWorldModelInterface *wmi, 
		 const VarBindings& bindings) const
    {
      VecPosition aPts[3];
      for (int i=0; i < 3; i++)
	aPts[i] = getPt(i)->getGlobalPosition(wmi, bindings);
      pfi->addPolygon(aPts, 3, c_border, c_inside);
      if (verbose)
	{
	  for (int i=0; i < 3; i++)
	    getPt(i)->draw(pfi, verbose, c_border, wmi, bindings);
	}
    }


    bool RegRec::isPtIn(const VecPosition &p, RegionWorldModelInterface *wmi, 
			const VarBindings &bindings) const      
    {
      Point *pt0 = (Point *) getPt(0);
      Point *pt1 = (Point *) getPt(1);
      VecPosition p0 = pt0->getGlobalPosition(wmi, bindings);
      VecPosition p1 = pt1->getGlobalPosition(wmi, bindings);
      
      double minx, miny, maxx, maxy;
      double px = p.getX();
      double py = p.getY();
      
      if(p0.getX() > p1.getX() - EPSILON) {
	minx = p1.getX() - EPSILON;
	maxx = p0.getX() + EPSILON;
      }
      else {
	minx = p0.getX() - EPSILON;
	maxx = p1.getX() + EPSILON;
      }
      
      if(p0.getY() > p1.getY()) {
	miny = p1.getY() - EPSILON;
	maxy = p0.getY() + EPSILON;
	}
      else {
	miny = p0.getY() - EPSILON;
	maxy = p1.getY() + EPSILON;
      }
      
      return (px >= minx && px <= maxx && py >= miny && py <= maxy);
    }

    vector <VecPosition> RegRec::getIntersection(const Line &l, 
						 RegionWorldModelInterface *wmi,
						 const VarBindings& bindings) const
    {
      /* Retrieve two corner points of the rectangular region */
      Point *pt0 = (Point *) getPt(0);
      Point *pt1 = (Point *) getPt(1);
      VecPosition v0 = pt0->getGlobalPosition(wmi, bindings);
      VecPosition v1 = pt1->getGlobalPosition(wmi, bindings);

      vector <VecPosition> solution;   /* solution list */
      
      VecPosition s1(v0.getX(), l.getYGivenX(v0.getX()));
      VecPosition s2(v1.getX(), l.getYGivenX(v1.getX()));
      VecPosition s3(l.getXGivenY(v0.getY()), v0.getY());
      VecPosition s4(l.getXGivenY(v1.getY()), v1.getY());

      /* Push back the intersections, if they are valid */
      if(isPtIn(s1, wmi, bindings))
	solution.push_back(s1);
      if(isPtIn(s2, wmi, bindings))
	solution.push_back(s2);
      if(isPtIn(s3, wmi, bindings))
	solution.push_back(s3);
      if(isPtIn(s4, wmi, bindings))
	solution.push_back(s4);
      
      return solution;  /* return solution vector */
    }

    VecPosition RegRec::getRandomPtIn(RegionWorldModelInterface *wmi, 
				      const VarBindings& bindings) const
    {
      //double minx, miny, maxx, maxy;
      //double randx, randy;
      Point *pt0 = (Point *) getPt(0);
      Point *pt1 = (Point *) getPt(1);
      VecPosition v0 = pt0->getGlobalPosition(wmi, bindings);
      VecPosition v1 = pt1->getGlobalPosition(wmi, bindings);
      
      /* Determine the minimum of the X coordinates */

      /*
      if(v0.getX() < v1.getX()) {
	minx = v0.getX();
	maxx = v1.getX();
      }
      else {
	minx = v1.getX();
	maxx = v0.getX();
      }
      */
      
      /* Determine the minimum of the Y coordinates */

      /*
      if(v0.getY() < v1.getY()) {
	miny = v0.getY();
	maxy = v1.getY();
      }
      else {
	miny = v1.getY();
	maxy = v0.getY();
      }
      */
      
      /* Now generate a random coordinate inside the region */
      //randx = (double) rand() / (double) RAND_MAX * (maxx - minx) + minx;
      //randy = (double) rand() / (double) RAND_MAX * (maxy - miny) + miny;
      
      /* Return the generated coordinate */
      //return VecPosition(randx, randy);

      return (v0 + v1) * 0.5;
    }

    VecPosition RegRec::getRandomPtOut(RegionWorldModelInterface *wmi,
				       const VarBindings& bindings) const
    {
      /* Select two random points inside the region */
      VecPosition rp0 = getRandomPtIn(wmi, bindings);
      VecPosition rp1 = getRandomPtIn(wmi, bindings);
      
      /* Construct the line connecting two random points */
      Line l = Line::makeLineFromTwoPoints(rp0, rp1);
      
      /* Intersections vector */
      vector <VecPosition> result = getIntersection(l, wmi, bindings);
      
      /* If there is at least one intersection, return the first one */
      if(result.size() > 0)
	return result[0];
      
      /* Otherwise, return dummy position */
      else
	return VecPosition(0.0, -35.0);
    }

    Rectangle
    RegRec::convertToRectangle(RegionWorldModelInterface *wmi,
			       const VarBindings& bindings) const
    {
      return Rectangle(getPt(0)->getGlobalPosition(wmi, bindings),
		       getPt(1)->getGlobalPosition(wmi, bindings));
    }
    
    VecPosition RegRec::getClosestPtTo(const VecPosition& pt, 
				       RegionWorldModelInterface *wmi,
				       const VarBindings& bindings) const
    {      
      /* First, check if the point is in the region, if so, return it */
      if(isPtIn(pt, wmi, bindings))
	return pt;

      
      /* Else, we have to determine all 4 corner points to construct all lines
	 representing each of the 4 sides */
      double minx, miny, maxx, maxy;
      Point *pt0 = (Point *) getPt(0);
      Point *pt1 = (Point *) getPt(1);
      VecPosition v0 = pt0->getGlobalPosition(wmi, bindings);
      VecPosition v1 = pt1->getGlobalPosition(wmi, bindings);
      VecPosition v2, v3;

      /* Determine the minimum of the X coordinates */
      if(v0.getX() < v1.getX()) {
	minx = v0.getX();
	maxx = v1.getX();
      }
      else {
	minx = v1.getX();
	maxx = v0.getX();
      }
      
      /* Determine the minimum of the Y coordinates */
      if(v0.getY() < v1.getY()) {
	miny = v0.getY();
	maxy = v1.getY();
      }
      else {
	miny = v1.getY();
	maxy = v0.getY();
      }

      /* Construct all 4 corner points */
      v0 = VecPosition(minx, miny);
      v1 = VecPosition(maxx, miny);
      v2 = VecPosition(maxx, maxy);
      v3 = VecPosition(minx, maxy);

      /* Construct all 4 sides as a line */
      Line side1 = Line::makeLineFromTwoPoints(v0, v1);
      Line side2 = Line::makeLineFromTwoPoints(v1, v2);
      Line side3 = Line::makeLineFromTwoPoints(v2, v3);
      Line side4 = Line::makeLineFromTwoPoints(v3, v0);

      /* Get the distance from the point pt to all 4 lines */
      double distToSide1 = side1.getDistanceWithPoint(pt);
      double distToSide2 = side2.getDistanceWithPoint(pt);
      double distToSide3 = side3.getDistanceWithPoint(pt);
      double distToSide4 = side4.getDistanceWithPoint(pt);

      
      /* Vector to store distances to each side */
      vector <double> dist;

      /* Push back all three distances */
      dist.push_back(distToSide1);
      dist.push_back(distToSide2);
      dist.push_back(distToSide3);
      dist.push_back(distToSide4);


      /* Get the closest Point on the line */
      switch(getMin(dist)) {
      case 1:
	return Line::getClosestPtInBetween(pt, v0, v1);
	break;
      case 2:
	return Line::getClosestPtInBetween(pt, v1, v2);
	break;
      case 3:
	return Line::getClosestPtInBetween(pt, v2, v3);
	break;
      case 4:
	return Line::getClosestPtInBetween(pt, v3, v0);
	break;
      default:
	break;
      }
      
      /* Default return value */
      return VecPosition(0.0, 0.0);
    }

    VecPosition RegRec::RegRecIterator::getPt(RegionWorldModelInterface *wmi, 
					      const VarBindings& s)
    {
      return VecPosition(currX, currY);
    }
    
    /* Finish this later */
    void RegRec::RegRecIterator::moveToNextPt()
    {
      currY += dist_pt;
      
      if(currY > maxy) {
	currX += dist_pt;
	currY = miny;
	
	if(currX > maxx)
	  error_flag = true;
      }
    }

    void RegRec::RegRecIterator::resetToBegin()
    {
      error_flag = false;
      
      
      if(p0.getX() > p1.getX()) {
	minx = p1.getX();
	maxx = p0.getX();
      }
      else {
	minx = p0.getX();
	maxx = p1.getX();
      }

      if(p0.getY() > p1.getY()) {
	miny = p1.getY();
	maxy = p0.getY();
      }
      else {
	miny = p0.getY();
	maxy = p1.getY();
      }

      currX = minx;
      currY = miny;
    }

    /* draw the region on the field */
    void
    RegRec::draw(FieldImage* pfi, 
		 bool verbose,
		 const FieldImage::Color& c_border, 
		 const FieldImage::Color& c_inside,
		 RegionWorldModelInterface *wmi, 
		 const VarBindings& bindings) const
    {
      pfi->addRectangle(convertToRectangle(wmi, bindings), c_border, c_inside);
      if (verbose)
	{
	  getPt(0)->draw(pfi, verbose, c_border, wmi, bindings);
	  getPt(1)->draw(pfi, verbose, c_border, wmi, bindings);
	}
    }

    /****************** End of newly added methods *******************/

    /****************** Start of the original code ******************/
    RegQuad::RegQuad() 
      : Region(),
        Has4Points() 
    {}
 
    RegQuad::RegQuad( std::auto_ptr< Point > pt0,
                      std::auto_ptr< Point > pt1, 
                      std::auto_ptr< Point > pt2,
                      std::auto_ptr< Point > pt3 ) 
        : Region(),
          Has4Points()
    {
      set( pt0, 0 );
      set( pt1, 1 );
      set( pt2, 2 );
      set( pt3, 3 );
    }

    RegQuad::~RegQuad()
    {}

    void 
    RegQuad::setAllPts( std::auto_ptr< Point > pt0,
                        std::auto_ptr< Point > pt1,
                        std::auto_ptr< Point > pt2,
                        std::auto_ptr< Point > pt3 )
    {
      set( pt0, 0 );
      set( pt1, 1 );
      set( pt2, 2 );
      set( pt3, 3 );
    }

    std::ostream& 
    RegQuad::print( std::ostream& out ) const
    {
      out << "(quad";
      for( unsigned int i = 0; i < Size; ++i )
        {
          if( get( i ) == NULL )
            out << " (null)";
          else
            out << " " << getRef( i );
        }
      return out << ")";
    }

    std::ostream&
    RegQuad::printPretty( std::ostream& out, const std::string& line_header ) const
    {
      out << line_header
          << "Quadrangle: ";
      for( unsigned int i = 0; i < Size; ++i )
        {
          if( get( i ) == NULL )
            out << "(null) ";
          else
            {
              getRef( i ).printPretty( out, line_header + " " );
              out << " ";
            }
        }
      return out << std::endl;
    }

    /*** RegArc ***/

    RegArc::RegArc()
      : HasAPoint(),
        M_start_rad( 0.0 ),
        M_end_rad( 0.0 ),
        M_start_ang( 0.0 ),
        M_span_ang( 0.0 )
    {}

    RegArc::RegArc( std::auto_ptr< Point > center, 
                    const double& start_rad, const double& end_rad, 
                    const double& start_ang, const double& span_ang )
      : HasAPoint( center ),
        M_start_rad( start_rad ),
        M_end_rad( end_rad ),
        M_start_ang( start_ang ),
        M_span_ang( span_ang )
    {}

    RegArc::RegArc( Circle c )
      : HasAPoint( std::auto_ptr< Point > (new PointSimple(c.getCenter().getX(), c.getCenter().getY()))),
        M_start_rad( 0 ),
        M_end_rad( c.getRadius() ),
        M_start_ang( 0 ),
        M_span_ang( 360 )
    {
    }
    
    RegArc::~RegArc()
    {}

    std::ostream&
    RegArc::print( std::ostream& out ) const
    {
      out << "(arc";
      if( get() == NULL )
        out << " (null)";
      else
        out << " " << getRef();
      return out << " " << M_start_rad 
                 << " " << M_end_rad 
                 << " " << M_start_ang 
                 << " " << M_span_ang
                 << ")";
    }

    std::ostream&
    RegArc::printPretty( std::ostream& out, const std::string& line_header ) const
    {
      out << line_header
          << "Arc: "
          << "center=";
      if( get() == NULL )
        out << "(null)";
      else
        getRef().printPretty( out, line_header + " " );
      return out << "\tradius="
                 << M_start_rad
                 << " to "
                 << M_end_rad
                 << "\tangle="
                 << M_start_ang
                 << " for "
                 << M_span_ang
                 << std::endl;
    }

    bool 
    RegArc::setRad( const double& start_rad, const double& end_rad)
    {
      if (start_rad > end_rad)
        return false;
      M_start_rad = start_rad;
      M_end_rad = end_rad;
      return true;
    }

    bool
    RegArc::setAng( const double& start_ang, const double& span_ang )
    {
      if ( start_ang > 180 || start_ang < -180 )
        return false;
      M_start_ang = start_ang;
      M_span_ang = span_ang;
      return true;
    }

/*** RegUnion ***/

    std::ostream& 
    RegUnion::print( std::ostream& out ) const
    {
      out << "(reg";
      for( Storage::const_iterator iter = get().begin();
           iter != get().end(); ++iter )
        {
          if( *iter == NULL )
            out << " (null)";
          else
            out << " " << **iter;
        }
      return out << ")";
    }

    std::ostream& 
    RegUnion::printPretty( std::ostream& out, const std::string& line_header ) const
    {
      out << line_header << "Region Union:" << std::endl;
      for( Storage::const_iterator iter = get().begin(); 
           iter != get().end(); ++iter)
        {
          if( *iter == NULL )
            out << line_header << "o (null)";
          else
            (*iter)->printPretty( out, line_header + "o " );
        }
      return out;
    }

    RegPoint::RegPoint() 
      : Region(),
        M_point( NULL )
    {}
      
    RegPoint::RegPoint( std::auto_ptr< Point > point ) 
      : Region(),
        M_point( point )
    {}
      
    RegPoint::RegPoint( const RegPoint& point ) 
      : Region(),
        M_point( point.M_point->deepCopy() )
    {}
      
    RegPoint::~RegPoint()
    {}

    RegPoint&
    RegPoint::operator=( const RegPoint& point )
    {
      std::auto_ptr< Point > pt_copy( point.M_point->deepCopy() );
      M_point = pt_copy;
      return *this;
    }
    
    std::ostream&
    RegPoint::print( std::ostream& out ) const
    { 
      if( M_point.get() == NULL )
        return out << "(null)";
      else
        return M_point->print( out );
    }
  
    std::ostream&
    RegPoint::printPretty( std::ostream& out, const std::string& line_header ) const
    {
      out << line_header << "region point\n";
      if( M_point.get() == NULL )
        return out << line_header << " (null)\n";
      else
        return M_point->printPretty( out, line_header ); 
    }
    
    Point*
    RegPoint::getPoint()
    { return M_point.get(); }

    const Point*
    RegPoint::getPoint() const
    { return M_point.get(); }

    std::auto_ptr< Point >
    RegPoint::detatchPoint()
    { return M_point; }

    void
    RegPoint::setPoint( std::auto_ptr< Point > point )
    { M_point = point; }




    RegTri::RegTri() 
      : Region(),
        Has3Points() 
    {}
 
    RegTri::RegTri( std::auto_ptr< Point > pt0,
                    std::auto_ptr< Point > pt1, 
                    std::auto_ptr< Point > pt2 ) 
      : Region(),
        Has3Points() 
    {
      set( pt0, 0 );
      set( pt1, 1 );
      set( pt2, 2 );
    }

    RegTri::~RegTri()
    {}

    void 
    RegTri::setAllPts( std::auto_ptr< Point > pt0,
                       std::auto_ptr< Point > pt1,
                       std::auto_ptr< Point > pt2 )
    {
      set( pt0, 0 );
      set( pt1, 1 );
      set( pt2, 2 );
    }

    std::ostream& 
    RegTri::print( std::ostream& out ) const
    {
      out << "(tri";
      for( unsigned int i = 0; i < Size; ++i )
        {
          if( get( i ) == NULL )
            out << " (null)";
          else
            out << " " << getRef( i );
        }
      return out << ")";
    }

    std::ostream&
    RegTri::printPretty( std::ostream& out, const std::string& line_header ) const
    {
      out << line_header
          << "Triangle: ";
      for( unsigned int i = 0; i < Size; ++i )
        {
          if( get( i ) == NULL )
            out << "(null) ";
          else
            {
              getRef( i ).printPretty( out, line_header + " " );
              out << " ";
            }
        }
      return out << std::endl;
    }

    RegRec::RegRec() 
      : Region(),
        Has2Points() 
    {}
 
    RegRec::RegRec( std::auto_ptr< Point > pt0,
                    std::auto_ptr< Point > pt1 ) 
        : Region(),
          Has2Points()
    {
      set( pt0, 0 );
      set( pt1, 1 );
    }

    RegRec::RegRec( const Rectangle& r)
        : Region(),
          Has2Points()
    {
      set( std::auto_ptr<Point>(new PointSimple(r.getPosLeftTop().getX(), r.getPosLeftTop().getY())), 0 );
      set( std::auto_ptr<Point>(new PointSimple(r.getPosRightBottom().getX(), r.getPosRightBottom().getY())), 1 );
    }
    
    RegRec::~RegRec()
    {}
    
    void 
    RegRec::setAllPts( std::auto_ptr< Point > pt0,
                       std::auto_ptr< Point > pt1 )
    {
      set( pt0, 0 );
      set( pt1, 1 );
    }

    std::ostream& 
    RegRec::print( std::ostream& out ) const
    {
      out << "(rec";
      for( unsigned int i = 0; i < Size; ++i )
        {
          if( get( i ) == NULL )
            out << " (null)";
          else
            out << " " << getRef( i );
        }
      return out << ")";
    }

    std::ostream&
    RegRec::printPretty( std::ostream& out, const std::string& line_header ) const
    {
      out << line_header
          << "Rectangle: ";
      for( unsigned int i = 0; i < Size; ++i )
        {
          if( get( i ) == NULL )
            out << "(null) ";
          else
            {
              getRef( i ).printPretty( out, line_header + " " );
              out << " ";
            }
        }
      return out << std::endl;
    }
  }
}
