// -*-c++-*-

/***************************************************************************
                                   cond.h  
                               Clang conditions
                             -------------------
    begin                : 18-MAR-2002
    copyright            : (C) 2002 by The RoboCup Soccer Server 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/

#ifndef _COND_H_
#define _COND_H_

#include "clangutil.h"
#include "visitor.h"
#include <memory>
#include <iostream>
#include "region.h"
#include "compop.h"
#include "hasa.h"

namespace rcss
{
  namespace clang
  {
    /******** Conditions ********************/
    class CondBool;
    class CondPlayerPos;
    class CondBallPos;
    class CondBallOwner;
    class CondPlayMode;
    class CondAnd;
    class CondOr;
    class CondNot;
    class CondNamed;
    class CondTime;
    class CondOppGoal;
    class CondOurGoal;
    class CondGoalDiff;
    class CondUNum;

    class Cond
    {
    public:
      class Context
      {
      public:
        Context()
        {}

        virtual 
        ~Context()
        {}

        virtual 
        bool
        lookup( const CondPlayerPos& cond ) const = 0;

        virtual 
        bool
        lookup( const CondBallPos& cond ) const = 0;

        virtual 
        bool
        lookup( const CondBallOwner& cond ) const = 0;

        virtual 
        bool
        lookup( const CondPlayMode& cond ) const = 0;

        virtual 
        bool
        lookup( const CondNamed& cond ) const = 0;

        virtual 
        bool
        lookup( const CondTime& cond ) const = 0;

        virtual 
        bool
        lookup( const CondOppGoal& cond ) const = 0;

        virtual 
        bool
        lookup( const CondOurGoal& cond ) const = 0;

        virtual 
        bool
        lookup( const CondGoalDiff& cond ) const = 0;

        virtual 
        bool
        lookup( const CondUNum& cond ) const = 0;
      }; // end Context

      typedef util::Visitor20< CondBool*,
        CondPlayerPos*,
        CondBallPos*,
        CondBallOwner*,
        CondPlayMode*,
        CondAnd*,
        CondOr*,
        CondNot*,
        CondNamed*,
        CondTime*,
        CondOppGoal*,
        CondOurGoal*,
        CondGoalDiff*,
        CondUNum* > Visitor;

      typedef util::Visitor20< const CondBool*,
        const CondPlayerPos*,
        const CondBallPos*,
        const CondBallOwner*,
        const CondPlayMode*,
        const CondAnd*,
        const CondOr*,
        const CondNot*,
        const CondNamed*,
        const CondTime*,
        const CondOppGoal*,
        const CondOurGoal*,
        const CondGoalDiff*,
        const CondUNum* > ConstVisitor;

      class TypeExtractor
        : public util::TypeExtractor20< CondBool*,
        CondPlayerPos*,
        CondBallPos*,
        CondBallOwner*,
        CondPlayMode*,
        CondAnd*,
        CondOr*,
        CondNot*,
        CondNamed*,
        CondTime*,
        CondOppGoal*,
        CondOurGoal*,
        CondGoalDiff*,
        CondUNum* >
      {
      protected:
        void
        visit( CondBool* cond )
        { setValue( cond ); }

        void
        visit( CondPlayerPos* cond )
        { setValue( cond ); }

        void
        visit( CondBallPos* cond )
        { setValue( cond ); }

        void
        visit( CondBallOwner* cond )
        { setValue( cond ); }

        void
        visit( CondPlayMode* cond )
        { setValue( cond ); }

        void
        visit( CondAnd* cond )
        { setValue( cond ); }

        void
        visit( CondOr* cond )
        { setValue( cond ); }

        void
        visit( CondNot* cond )
        { setValue( cond ); }

        void
        visit( CondNamed* cond )
        { setValue( cond ); }

        void
        visit( CondTime* cond )
        { setValue( cond ); }

        void
        visit( CondOppGoal* cond )
        { setValue( cond ); }

        void
        visit( CondOurGoal* cond )
        { setValue( cond ); }

        void
        visit( CondGoalDiff* cond )
        { setValue( cond ); }

        void
        visit( CondUNum* cond )
        { setValue( cond ); }

      };

      class ConstTypeExtractor
        : public util::TypeExtractor20< const CondBool*,
        const CondPlayerPos*,
        const CondBallPos*,
        const CondBallOwner*,
        const CondPlayMode*,
        const CondAnd*,
        const CondOr*,
        const CondNot*,
        const CondNamed*,
        const CondTime*,
        const CondOppGoal*,
        const CondOurGoal*,
        const CondGoalDiff*,
        const CondUNum* >
      {
      protected:
        void
        visit( const CondBool* cond )
        { setValue( cond ); }

        void
        visit( const CondPlayerPos* cond )
        { setValue( cond ); }

        void
        visit( const CondBallPos* cond )
        { setValue( cond ); }

        void
        visit( const CondBallOwner* cond )
        { setValue( cond ); }

        void
        visit( const CondPlayMode* cond )
        { setValue( cond ); }

        void
        visit( const CondAnd* cond )
        { setValue( cond ); }

        void
        visit( const CondOr* cond )
        { setValue( cond ); }

        void
        visit( const CondNot* cond )
        { setValue( cond ); }

        void
        visit( const CondNamed* cond )
        { setValue( cond ); }

        void
        visit( const CondTime* cond )
        { setValue( cond ); }

        void
        visit( const CondOppGoal* cond )
        { setValue( cond ); }

        void
        visit( const CondOurGoal* cond )
        { setValue( cond ); }

        void
        visit( const CondGoalDiff* cond )
        { setValue( cond ); }

        void
        visit( const CondUNum* cond )
        { setValue( cond ); }

       };


      Cond()
      {}
  
      virtual
      ~Cond()
      {}

      virtual
      std::ostream&
      print( std::ostream& out ) const = 0;

      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const = 0;

      virtual
      bool 
      eval( const Context& context ) const = 0;

      virtual
      void
      accept( Visitor& v ) = 0;

      virtual
      void
      accept( ConstVisitor& v ) const = 0;

      virtual
      std::auto_ptr< Cond >
      deepCopy() const = 0;
    };
  }
}


inline 
std::ostream& 
operator<<( std::ostream & os, const rcss::clang::Cond& c )
{ return c.print(os); }

namespace rcss
{
  namespace clang
  {
    class CondBool
      : public Cond,
        public util::HasA< bool >
    {
    public:
      typedef util::HasA< bool > HasABool;

      CondBool() 
        : Cond(),
          HasABool()
      {}

      CondBool( bool state ) 
        : Cond(),
          HasABool( state )
      {}

      ~CondBool()
      {}

      std::ostream& 
      print( std::ostream& out ) const 
      { return out << "(" << ( get() ? "true" : "false" ) << ")"; }

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return out << line_header << ( get() ? "true" : "false" ) << std::endl; }

      bool
      eval( const Context& ) const
      { return get(); }

      bool
      getState()
      { return get(); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      virtual
      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondBool( *this ) ); }
    };

    class CondPlayerPos
      : public Cond,
        public util::HasA< UNumSet >,
        public util::HasA< std::auto_ptr< Region > >
    {
    public:
      typedef util::HasA< UNumSet > HasAUNumSet;
      typedef util::HasA< std::auto_ptr< Region > > HasARegion;

      CondPlayerPos()
        : Cond(),
          HasAUNumSet(),
          HasARegion(),
          M_our_side( false ),
          M_min_match( 1 ),
          M_max_match( 11 )
      {}

      CondPlayerPos( const bool& our_side,
                     const UNumSet& players,
                     const int& min_match,
                     const int& max_match,
                     std::auto_ptr< Region > reg )
        : Cond(),
          HasAUNumSet( players ),
          HasARegion( reg ),
          M_our_side( our_side ),
          M_min_match( min_match ),
          M_max_match( max_match )
      {}
      
      virtual
      ~CondPlayerPos()
      {}

      std::ostream&
      print( std::ostream& out ) const;

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const;

      bool
      eval( const Context& context ) const
      { return context.lookup( *this ); }

      /* class specific stuff */
      bool
      isOurSide() const 
      { return M_our_side; }

      bool
      isTheirSide() const
      { return !M_our_side; }
  
      int
      getMinMatch() const
      { return M_min_match; }

      int
      getMaxMatch() const
      { return M_max_match; }

      const Region*
      getRegion() const
      { return HasARegion::get(); }

      UNumSet
      getPlayerSet() const
      { return HasAUNumSet::get(); }

      void
      setMinMatch( int x )
      { M_min_match = x; }
  
      void
      setMaxMatch( int x )
      { M_max_match = x; }
  
      void
      set( std::auto_ptr< Region > reg )
      { HasARegion::set( reg ); }
      
      std::auto_ptr< Region >
      detachRegion()
      { return HasARegion::detach(); }

      void
      setOurSide( const bool& our_side )
      { M_our_side = our_side; }
  
      void
      set( const UNumSet& players )
      { HasAUNumSet::set( players ); }
  
      void
      addPlayer( const UNum& i )
      { HasAUNumSet::getRef().add( i ); }

      void
      clearPlayers()
      { HasAUNumSet::getRef().clear(); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }
 
      virtual
      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondPlayerPos( *this ) ); }

    private:
      bool M_our_side;
      int M_min_match;
      int M_max_match;
    };

    class CondBallPos
      : public Cond,
        public util::HasA< std::auto_ptr< Region > >
    {
    public:
      typedef util::HasA< std::auto_ptr< Region > > HasARegion;

      CondBallPos()
        : Cond(),
          HasARegion()
      {}

      CondBallPos( std::auto_ptr< Region > reg )
        : Cond(),
          HasARegion( reg ) 
      {}

      ~CondBallPos()
      {}

      std::ostream&
      print( std::ostream& out ) const
      { 
        out << "(bpos ";
        if( get() == NULL )
          out << "(null)";
        else
          out << getRef();
        return out << ")";
      }

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { 
        out << line_header << "is ball position at: " << std::endl;
        if( get() == NULL )
          out << line_header << " (null)\n";
        else
          getRef().printPretty( out, line_header + " " );
        return out;
      }

      bool
      eval( const Context& context ) const 
      { return context.lookup( *this ); }
  
      const Region*
      getRegion() const
      { return get(); }

      std::auto_ptr< Region >
      detachRegion()
      { return detach(); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      virtual
      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondBallPos( *this ) ); }
    };

    class CondBallOwner
      : public Cond
    {
    public:
      CondBallOwner()
        : Cond(),
          M_our_side( false )
      {}

      CondBallOwner( const bool& our_side, const UNumSet& players )
        : Cond(),
          M_our_side( our_side ),
          M_players( players )
      {}

      ~CondBallOwner()
      {}

      std::ostream&
      print( std::ostream& out ) const;

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const;

      bool
      eval( const Context& context ) const
      { return context.lookup( *this ); }

      /* class specific stuff */
      bool
      isOurSide() const
      { return M_our_side; }
  
      bool
      isTheirSide() const
      { return !M_our_side; }

      UNumSet& 
      getPlayerSet() 
      { return M_players; }

      const UNumSet& 
      getPlayerSet() const 
      { return M_players; }

      void
      setOurSide( const bool& our_side )
      { M_our_side = our_side; }

      void
      setPlayers( const UNumSet& players )
      { M_players = players; }

      void
      addPlayer( const UNum& i )
      { M_players.add( i ); }

      void
      clearPlayers()
      { M_players.clear(); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondBallOwner( *this ) ); }

    private:
      bool M_our_side;
      UNumSet M_players;
    };



    class CondPlayMode
      : public Cond
    {
    public:
      static const char* MODE_STRINGS[];

      CondPlayMode()
        : Cond(),
          M_pm( PM_None )
      {}

      CondPlayMode( const PlayMode& pm )
        : Cond(),
          M_pm( pm )
      {}

      ~CondPlayMode()
      {}

      std::ostream&
      print( std::ostream& out ) const
      { return out << "(playm " << MODE_STRINGS[ M_pm ] << ")"; }

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return out << line_header << "is play mode '"
                   << MODE_STRINGS[ M_pm ] << "'" << std::endl; }

      bool
      eval( const Context& context ) const
      { return context.lookup( *this ); }
  
      /* class specific stuff */
      PlayMode 
      getPlayMode() const
      { return M_pm; }

      void
      setPlayMode( const PlayMode& pm)
      { M_pm = pm; }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondPlayMode( *this ) ); }

    private:
      PlayMode M_pm;
    };

    class CondAnd
      : public Cond,
        public util::HasMany< std::auto_ptr< Cond > >
    {
    public:
      typedef util::HasMany< std::auto_ptr< Cond > > HasManyConds;

      CondAnd()
        : Cond(),
          HasManyConds()
      {}

      CondAnd( const Storage& conds ) 
        : Cond(),
          HasManyConds( conds )
      {}
      virtual
      ~CondAnd()
      {}

      std::ostream&
      print( std::ostream& out ) const;

      std::ostream& 
      printPretty( std::ostream& out, const std::string& line_header ) const;

      bool
      eval( const Context& context ) const;

      const Storage& 
      getConds() const 
      { return get(); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      virtual
      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondAnd( *this ) ); }
    };


    class CondOr
      : public Cond,
        public util::HasMany< std::auto_ptr< Cond > >
    {
    public:
      typedef util::HasMany< std::auto_ptr< Cond > > HasManyConds;

      CondOr() 
        : Cond(),
          HasManyConds()
      {}

      CondOr( const Storage& conds ) 
        : Cond(),
          HasManyConds( conds )
      {}

      virtual
      ~CondOr()
      {}

      std::ostream&
      print( std::ostream& out ) const;

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const;

      bool
      eval( const Context& context ) const;

      const Storage&
      getConds() const 
      { return get(); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      virtual
      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondOr( *this ) ); }
    };


    class CondNot
      : public Cond,
        public util::HasA< std::auto_ptr< Cond > >
    {
    public:
      typedef util::HasA< std::auto_ptr< Cond > > HasACond;

      CondNot()
        : Cond(),
          HasACond()
      {}

      CondNot( std::auto_ptr< Cond > cond ) 
        : Cond(),
          HasACond( cond )
      {}

      virtual
      ~CondNot()
      {}

      std::ostream& 
      print( std::ostream& out ) const 
      { 
        out << "(not ";
        if( get() == NULL )
          out << "(null)\n";
        else
          out << getRef();
        return out << ")";
      }

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { 
        out << line_header << "not " << std::endl;
        if( get() == NULL )
          out << line_header << " (null)\n";
        else
          getRef().printPretty( out, line_header + " +" );
        return out;
      }

      bool
      eval( const Context& context ) const
      {
        if( get() == NULL )
          throw util::NullErr( __FILE__, __LINE__, 
                               "Null condition in CondNot\n" );
        else
          return !( getRef().eval( context ) );
      }
  
      const Cond*
      getCond() const
      { return get(); }

      std::auto_ptr< Cond >
      detachCond()
      { return detach(); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      virtual
      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondNot( *this ) ); }
    };


    class CondNamed
      : public Cond
    {
    public:
      CondNamed() 
        : Cond(), pCond(NULL)
      {}
  
      CondNamed( const std::string& name ) 
        : Cond(),
          M_name( name ),
	  pCond(NULL)
      {}
  

      ~CondNamed()
      {}

      std::ostream&
      print( std::ostream& out ) const
      { return out << "\"" << M_name << "\""; }

      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return out << line_header << "cond named \"" 
                   << M_name << "\"" << std::endl; }
  
      bool
      eval( const Context& context ) const 
      { return context.lookup ( *this ); }

      /* class specific */
      std::string& 
      getName()
      { return M_name; }

      const std::string& 
      getName() const
      { return M_name; }

      /* dongryel: get pointer to the Cond object */
      const Cond*
      getCond() const
      { return pCond; }

      void
      setName( const std::string& name )
      { M_name = name; }
 
      /* dongryel: set the link to the Cond object */
      void
      setLink( Cond *cond)
      { 
	pCond = cond; 
      }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondNamed( *this ) ); }

    private:
      std::string M_name;
      
      /* dongryel: pointer to the Cond object */
      Cond* pCond;
    };

    template< typename VALUE >
    class CondComp
      : public Cond
    {
    private:
      const util::CompOp* M_comp;
      VALUE M_value;

    public:
      CondComp( const VALUE& value, const util::CompOp& comp )
        : M_comp( &comp ),
          M_value( value )
      {}

      ~CondComp()
      {}

      VALUE
      getValue() const
      { return M_value; }

      VALUE
      setValue( const VALUE& value )
      { return M_value = value; }

      const util::CompOp&
      getComp() const
      { return *M_comp; }

      const util::CompOp&
      setComp( const util::CompOp& comp )
      { return *(M_comp = &comp); }      
    };

    class CondTime
      : public CondComp< int >
    {
    public:
      CondTime( const int& value, const util::CompOp& comp )
        : CondComp< int >( value, comp )
      {}

      ~CondTime()
      {}

      virtual
      std::ostream&
      print( std::ostream& out ) const
      { return out << "(time " << getComp() << " " << getValue() << ")"; }

      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return print( out << line_header ); }

      virtual
      bool 
      eval( const Context& context ) const
      { return context.lookup( *this ); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondTime( *this ) ); }
    };


    class CondOppGoal
      : public CondComp< int >
    {
    public:
      CondOppGoal( const int& value, const util::CompOp& comp )
        : CondComp< int >( value, comp )
      {}

      ~CondOppGoal()
      {}

      virtual
      std::ostream&
      print( std::ostream& out ) const
      { return out << "(opp_goals " << getComp() << " " << getValue() << ")"; }

      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return print( out << line_header ); }

      virtual
      bool 
      eval( const Context& context ) const
      { return context.lookup( *this ); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondOppGoal( *this ) ); }
    };

    class CondOurGoal
      : public CondComp< int >
    {
    public:
      CondOurGoal( const int& value, const util::CompOp& comp )
        : CondComp< int >( value, comp )
      {}

      ~CondOurGoal()
      {}

      virtual
      std::ostream&
      print( std::ostream& out ) const
      { return out << "(our_goals " << getComp() << " " << getValue() << ")"; }

      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return print( out << line_header ); }

      virtual
      bool 
      eval( const Context& context ) const
      { return context.lookup( *this ); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondOurGoal( *this ) ); }
    };

    class CondGoalDiff
      : public CondComp< int >
    {
    public:
      CondGoalDiff( const int& value, const util::CompOp& comp )
        : CondComp< int >( value, comp )
      {}

      ~CondGoalDiff()
      {}

      virtual
      std::ostream&
      print( std::ostream& out ) const
      { return out << "(goal_diff " << getComp() << " " << getValue() << ")"; }

      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return print( out << line_header ); }

      virtual
      bool 
      eval( const Context& context ) const
      { return context.lookup( *this ); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondGoalDiff( *this ) ); }
    };

    
    class CondUNum
      : public Cond
    {
    public:
     CondUNum()
        : Cond(),
          M_unum(),
          M_set()
      {}

     CondUNum( const UNum& unum )
        : Cond(),
          M_unum( unum ),
          M_set()
      {}

      CondUNum( const UNum& unum, const UNumSet& players )
        : Cond(),
          M_unum( unum ),
          M_set( players )
      {}

      ~CondUNum()
      {}

      virtual
      std::ostream&
      print( std::ostream& out ) const
      { return out << "(unum " << M_unum << " " << M_set << ")"; }

      virtual
      std::ostream&
      printPretty( std::ostream& out, const std::string& line_header ) const
      { return print( out << line_header ); }

      virtual
      bool 
      eval( const Context& context ) const
      { return context.lookup( *this ); }

      virtual
      void
      accept( Visitor& v )
      { v.startVisit( this ); }

      virtual
      void
      accept( ConstVisitor& v ) const
      { v.startVisit( this ); }

      std::auto_ptr< Cond >
      deepCopy() const
      { return std::auto_ptr< Cond >( new CondUNum( *this ) ); }

      UNum&
      setUNum( const UNum& unum )
      { return M_unum = unum; }

      UNum
      getUNum() const
      { return M_unum; }
      
      UNumSet&
      setUNum( const UNumSet& uset )
      { return M_set = uset; }

      UNumSet&
      getUNumSet()
      { return M_set; }
      
      const UNumSet&
      getUNumSet() const
      { return M_set; }
      
      UNum
      addUNum( const UNum& unum )
      { M_set.add( unum ); return unum; }

    private:
      UNum M_unum;
      UNumSet M_set;
    };
  }
}

#endif
