// -*-c++-*-

/***************************************************************************
                                  clangutil.h  
                       Utility classes for clang messages
                             -------------------
    begin                : 25-FEB-2002
    copyright            : (C) 2002 by The RoboCup Soccer Server 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/

#ifndef _CLANGUTIL_H_
#define _CLANGUTIL_H_

#include <iostream>
#include <string>
#include <vector>
#include <memory>

namespace rcss
{
  namespace clang
  {

    enum BallMoveToken
    { 
      BMT_None,
      Pass,
      Dribble,
      Clear,
      Score,
      BMT_All 
    };

    class BallMove
    {
    public:
      //It is important that None is first and All is last
      
      static const char* TOKEN_STRINGS[];
      static const int TOKEN_MAX;

      BallMove()
        : M_entries ( 0 )
      {}

      BallMove( const BallMoveToken& init_entry )
        : M_entries ( 0 )
      { addToken( init_entry ); }

      ~BallMove()
      {}

      void
      clear()
      { M_entries = 0; }

      void
      addToken( const BallMoveToken& t );

      void
      removeToken( const BallMoveToken& t );

      bool
      isMember( const BallMoveToken& t ) const;

      std::ostream&
      print( std::ostream& out ) const;


      std::auto_ptr< BallMove >
      deepCopy() const
      { return std::auto_ptr< BallMove >( new BallMove( *this ) ); }
    private:
      unsigned M_entries;
    };

    enum PlayMode
    {
      PM_None,
      BeforeKickOff,
      TimeOver,
      PlayOn,
      KickOff_Our,
      KickOff_Opp,
      KickIn_Our,
      KickIn_Opp,
      FreeKick_Our,
      FreeKick_Opp,
      CornerKick_Our,
      CornerKick_Opp,
      GoalKick_Our,
      GoalKick_Opp,
      GoalieCatch_Our,
      GoalieCatch_Opp,
      AfterGoal_Our,
      AfterGoal_Opp
    };


    class UNum
    {
    public:
      enum unum_t
      { uAll, u1, u2, u3, u4, u5, u6, u7, u8, u9, u10, u11, uMax };

    private:
      unum_t M_unum;
      
      std::string M_var;

    public:
      UNum()
        : M_unum( uMax ),
          M_var( "" )
      {}

      UNum( const unum_t& unum )
        : M_unum( unum ),
          M_var( "" )
      {}
      
      UNum( const std::string& var )
        : M_unum( uMax ),
          M_var( var )
      {}
      
      UNum( const std::string& var, const unum_t& unum )
        : M_unum( unum ),
          M_var( var )
      {}

      // we take advantage of the numbering of unum above -- be careful!
      explicit UNum( int n )
	: M_unum((unum_t) n),
	  M_var ( "" )
      {
      }
      
      ~UNum()
      {}

      unum_t
      getUNum() const
      { return M_unum; }
    
      bool
      isValid() const
      { return M_unum != uMax; }

      bool
      isWildCard() const
      { return M_unum == uAll; }

      std::string
      getVar() const
      { return M_var; }

      bool
      isBindable() const
      { return M_var.length() > 0; }

      bool
      isBound() const
      { return isValid(); }

      bool
      bind( const unum_t& unum )
      { 
        if( isBindable() && unum != uMax && unum != uAll )
          {
            M_unum = unum;
            return true;
          }
        else
          return false;
      }


      bool
      unBind()
      { 
        if( isBindable() )
          {
            M_unum = uMax;
            return true;
          }
        else
          return false;
      }

      unum_t
      setUNum( const unum_t& unum )
      { return M_unum = unum; }

      std::string
      setVar( const std::string& var )
      { return M_var = var; }

      UNum
      operator++()
      {
        if( M_unum != uAll && M_unum != uMax )
          M_unum = ( unum_t )( ( ( unsigned int )M_unum ) + 1 );
        return *this;
      }

      UNum
      operator++( int )
      {
        UNum rval = *this;
        ++( *this );
        return rval;
      }

      UNum
      operator--()
      {
        if( M_unum == u1 )
          M_unum = uMax;
        else if( M_unum != uAll && M_unum != uMax )
          M_unum = ( unum_t )( ( ( unsigned int )M_unum ) - 1 );
        return *this;
      }

      UNum
      operator--( int )
      {
        UNum rval = *this;
        --( *this );
        return rval;
      }
    };
  }
}

inline
std::ostream& 
operator<<( std::ostream & os, const rcss::clang::BallMove& s )
{ return s.print(os); } 

inline
std::ostream& 
operator<<( std::ostream & os, const rcss::clang::BallMoveToken& s )
{ return os << rcss::clang::BallMove::TOKEN_STRINGS[ s ]; } 

inline
bool
operator==( const rcss::clang::UNum& a, const rcss::clang::UNum& b )
{
  return ( a.isValid() && b.isValid()
           && ( a.getUNum() == b.getUNum() 
                || a.isWildCard() 
                || b.isWildCard() ) ); 
}

inline
std::ostream& 
operator<<( std::ostream & os, const rcss::clang::UNum& u )
{ 
  if( u.isBindable() )
    return os << u.getVar();
  else
    {
      if( u.isValid() )
        return os << (unsigned int)u.getUNum();
      else
        return os << "(null)";
    }
} 

namespace rcss
{
  namespace clang
  {
    class UNumSet
    {
    private:
      typedef std::vector< UNum > container;

      container M_entries;

    public:
      UNumSet()
      {}

      UNumSet(const UNumSet& uset)
	: M_entries(uset.M_entries)
      {}

      // makes a singleton
      explicit UNumSet(const UNum& u)
	: M_entries()
      { add(u); }

      ~UNumSet()
      {}

      bool
      contains( const UNum& unum ) const
      {
	if( unum.isWildCard() ){ // is uAll
	  // check if any unums in the UNumSet are wildcard also
	  for( container::const_iterator i = M_entries.begin();
                 i != M_entries.end(); ++i )
	    {
	      if( (*i).isWildCard() )
		return true;
	    }

	}
        else if( unum.isValid() )
          {
            for( container::const_iterator i = M_entries.begin();
                 i != M_entries.end(); ++i )
              {
                if( unum == *i )
                  return true;
              }
          }
        return false;
      }

      void
      add( const UNum& unum )
      { M_entries.push_back( unum ); }

      typedef container::iterator iterator;
      typedef container::const_iterator const_iterator;

      iterator
      erase( const iterator& iter )
      { return M_entries.erase( iter ); }

      iterator
      begin()
      { return M_entries.begin(); }

      const_iterator
      begin() const
      { return M_entries.begin(); }

      iterator
      end()
      { return M_entries.end(); }

      const_iterator
      end() const
      { return M_entries.end(); }

      void
      clear()
      { M_entries.clear(); }

      /* Added by Dongryeol Lee */
      int
      size() const
      { return M_entries.size(); }

      const UNum& 
      getIndexAt(int index) const
      { return M_entries[index]; }

      /*
       * return this - u. subtract each binding in b1 from each binding in this. 
       * ex. {1 2 3 4 5} - {1 2 3} = {4 5}
       */
      inline UNumSet operator- ( UNumSet u ){
	UNumSet newset;

	// if src set is empty or {0}, treat as all players
	if(begin() == end() || contains(UNum::uAll)){ 
	    for(UNum x = UNum::u1; x.getUNum() <= UNum::u11; x++){
	      if(!u.contains(x))
		newset.add(x);
	    }
	    return newset;
	}

	// iterate over the elements in this set
	for(iterator it = begin(); it != end(); it++){

	  // if it's not in u, we can add it to the return set
	  if(!u.contains(*it))
	    newset.add(*it);

	}

	return newset;
      }


    };
  }
}

inline
std::ostream& 
operator<<( std::ostream & os, const rcss::clang::UNumSet& u )
{
  os << "{";
  for( rcss::clang::UNumSet::const_iterator i = u.begin();
        i != u.end(); ++i )
     {
       if( i != u.begin() )
         os << " ";
       os << *i;
     }
  return os << "}";
} 


#endif
