// -*-c++-*-

/***************************************************************************
                               clangmsgbuilder.h  
                      Class for building a clang message
                             -------------------
    begin                : 25-FEB-2002
    copyright            : (C) 2002 by The RoboCup Soccer Server 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/

#ifndef _CLANGMSGBUILDER_H_
#define _CLANGMSGBUILDER_H_

#include "clangbuilder.h"
#include "visitor.h"
#include <stack>
#include <memory>
#include <list>


#ifdef DEBUG
#include <typeinfo>
#include <iostream>
#endif


namespace rcss
{
  namespace clang
  {
    class Msg;

    class MetaToken;
    class Action;
    class Dir;
    class Cond;
    class Token;
    class Def;
    class Region;
    class Point;
    class UNum;
    class UNumSet;
    class RuleIDList;
    class ActivateRules;
    class Rule;


    class MsgBuilder
      : public Builder
    {
    private:
      std::auto_ptr< Msg > M_msg;
      unsigned int M_min_ver;
      unsigned int M_max_ver;

    protected:
      typedef std::list< Cond* > CondList;
      typedef util::Tuple20< MetaToken*,
			     Action*,
			     Dir*,
			     Cond*,
			     CondList,
			     Token*,
			     Def*,
			     Region*,
			     Point*,
			     UNum,
			     BallMoveToken,
			     UNumSet,
			     std::string,
			     RuleIDList,
			     ActivateRules,
			     Rule* > ItemType;
      
      std::stack< ItemType > M_items;

      template< class X >
      X
      getItem();

      
      template< class X >
      std::auto_ptr< X >
      getPtrItem();

      template< class X >
      bool
      isItem() const;

      template< class X >
      bool
      checkIsItem() const;

      template< class X >
      X
      checkAndGetItem();

      template< class X >
      std::auto_ptr< X >
      checkAndGetPtrItem();

      template< class X >
      void
      add( const X& x );

      void
      checkItemsEmpty() const;

      void
      setMsg( Msg* msg );

      void
      emptyStack();

      void
      clear();

      void
      onNoItems() const;

      void
      onWrongItem() const;

      void
      onNotEmpty() const;
      
    public:
      MsgBuilder();

      virtual
      ~MsgBuilder();

      Msg*
      getMsg();

      const Msg*
      getMsg() const;

      std::auto_ptr< Msg >
      detatchMsg();

      virtual
      void
      setVer( const unsigned int& min, const unsigned int& max );

      virtual
      void
      setTime( const int& time );

      virtual
      void
      setSide( const int& side );
      
      virtual
      void
      setTimeRecv( const int& time );

      virtual
      void
      buildMetaMsg();

      virtual
      void
      buildMetaTokenVer( const double& ver ); 

      virtual
      void
      buildDefineMsg();

      virtual
      void
      buildDefineCond( const std::string& name );

      virtual
      void
      buildDefineDir( const std::string& name );

      virtual
      void
      buildDefineReg( const std::string& name );

      virtual
      void
      buildDefineAct( const std::string& name );

      virtual
      void
      buildFreeformMsg( const std::string& str );
      
      virtual
      void
      buildUnsuppMsg();

      virtual
      void  
      buildInfoMsg();
 
      virtual
      void
      buildAdviceMsg();

      virtual
      void
      buildTokenRule( const int& ttl );

      virtual
      void
      buildTokenClear();

      virtual
      void
      buildActPos();

      virtual
      void
      buildActHome();

      virtual
      void
      buildActBallToReg();

      virtual
      void
      buildActBallToPlayer();

      virtual
      void
      buildActMark();

      virtual
      void
      buildActMarkLinePlayer();

      virtual
      void
      buildActMarkLineReg();

      virtual
      void
      buildActOffsideLine();

      virtual
      void
      buildActHetType( const int& type );
 
      virtual
      void
      buildActNamed( const std::string& name );

      virtual
      void
      buildActPassReg();

      virtual
      void
      buildActPassUNum();

      virtual
      void
      buildActDribble();

      virtual
      void
      buildActClear();

      virtual
      void
      buildActShoot();

      virtual
      void
      buildActHold();

      virtual
      void
      buildActIntercept();

      virtual
      void
      buildActTackle();

      virtual
      void
      buildDirComm( const bool& do_dont, const bool& our_side );

      virtual
      void
      buildDirNamed( const std::string& name );

      virtual
      void
      buildCondTrue();

      virtual
      void
      buildCondFalse();

      virtual
      void
      buildCondPlayerPos( const bool& our_side, 
                          const int& min, 
                          const int& max );

      virtual
      void
      buildCondBallPos();

      virtual
      void
      buildCondBallOwner( const bool& our_side );

      virtual
      void
      buildCondPlayMode( const PlayMode& play_mode );

      virtual
      void
      buildCondAnd();

      virtual
      void
      buildCondOr();

      virtual
      void
      buildCondNot();

      virtual
      void
      buildCondNamed( const std::string& name );
      
      virtual
      void
      buildCondTime( const int& time, const util::CompOp& comp );

      virtual
      void
      buildCondOppGoal( const int& goals, const util::CompOp& comp );

      virtual
      void
      buildCondOurGoal( const int& goals, const util::CompOp& comp );

      virtual
      void
      buildCondGoalDiff( const int& goals, const util::CompOp& comp );

      virtual
      void
      buildCondUNum( const rcss::clang::UNum& unum );

      virtual
      void
      buildAddToCondList ( );

      virtual
      void
      buildCreateCondList ( );
      
      virtual
      void
      buildRegNull();

      virtual
      void
      buildRegQuad();

      virtual
      void
      buildRegArc( const double& start_rad,
                   const double& end_rad,
                   const double& start_ang,
                   const double& span_ang );

      virtual
      void
      buildRegUnion();

      virtual
      void
      buildRegNamed( const std::string& name );

      virtual
      void
      buildRegPoint();

      virtual
      void
      buildRegTri();

      virtual
      void
      buildRegRec();

      virtual
      void
      buildPointSimple( const double& x, const double& y );

      virtual
      void
      buildPointRel( const double& x, const double& y );

      virtual
      void
      buildPointBall();

      virtual
      void
      buildPointPlayer( const bool& our_side, const UNum& unum );

      virtual
      void
      buildPointArith( const rcss::util::ArithOp& arith_op );

      virtual
      void
      buildUNum( const UNum& unum );
 
      virtual
      void
      buildUNumSet();
 
      virtual
      void
      buildBallMoveToken( const BallMoveToken& bmt );     

      virtual
      void
      buildRuleMsg();

      virtual
      void
      buildActivateAllRules( const bool& on );

      virtual
      void
      buildActivateRules( const bool& on );

      virtual
      void
      buildRuleID( const std::string& id );

      virtual
      void
      buildRuleIDList();

      virtual
      void
      buildRuleIDListALL();

      virtual
      void
      buildDelMsg();

      virtual
      void
      buildDefineModelRule( const std::string& id );

      virtual
      void
      buildDefineDirectiveRule( const std::string& id );

      virtual
      void
      buildSimpleRule();

      virtual
      void
      buildNestedRule();

      virtual
      void
      buildIDRule();
    };

    template< typename X >
    inline
    void 
    traceBuild()
    {
#ifdef DEBUG
      std::cout << "Building: " << typeid( X ).name() << std::endl; 
#endif
    }

    template< typename X >
    inline
    void 
    traceRemoving()
    { 
#ifdef DEBUG
      std::cout << "Removing: " << typeid( X ).name() << std::endl; 
#endif
    }

    template< typename X >
    inline
    void 
    traceAdding()
    { 
#ifdef DEBUG
      std::cout << "Adding: " << typeid( X ).name() << std::endl; 
#endif
    }

    template< class X >
    X
    MsgBuilder::getItem()
    {
      traceRemoving< X >();
      X rval( M_items.top().template get< X >() );
      M_items.pop();
      return rval;
    }

    template< class X >
    std::auto_ptr< X >
    MsgBuilder::getPtrItem()
    {
      traceRemoving< X >();
      std::auto_ptr< X > rval( M_items.top().template get< X* >() );
      M_items.pop();
      return rval;
    }

    
    template< class X >
    bool
    MsgBuilder::isItem() const
    {
      return ( !M_items.empty()
               && M_items.top().template isTypeOf< X >() );
    }
            
    
    template< class X >
    bool
    MsgBuilder::checkIsItem() const
    {
      if( !M_items.empty() )
        {
          if( M_items.top().template isTypeOf< X >() )
            return true;
          else
            onWrongItem();
        }
      else
        onNoItems();
      return false;
    }

    
    template< class X >
    X
    MsgBuilder::checkAndGetItem()
    {
      if( checkIsItem< X >() )
        {
          return getItem< X >();
        }
      // no need for error handling as this is done by checkIsItem()
      return X();
    }

    
    template< class X >
    std::auto_ptr< X >
    MsgBuilder::checkAndGetPtrItem()
    {
      if( checkIsItem< X* >() )
        {
          return getPtrItem< X >();
        }
      // no need for error handling as this is done by checkIsItem()
      return std::auto_ptr< X >();
    }

    
    template< class X >
    void
    MsgBuilder::add( const X& x )
    {
      traceAdding< X >();
      M_items.push( x );
    }
  }
}

#endif
