// -*-c++-*-

/***************************************************************************
                               clangmsgbuilder.cc  
                      Class for building a clang message
                             -------------------
    begin                : 25-FEB-2002
    copyright            : (C) 2002 by The RoboCup Soccer Server 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/

#include "clangmsgbuilder.h"
#include "region.h"
#include "clangmsg.h"
#include "clangmetamsg.h"
#include "clangfreeformmsg.h"
#include "clangunsuppmsg.h"
#include "clangrulemsg.h"
#include "clangdelmsg.h"
#include "clanginfomsg.h"
#include "clangadvicemsg.h"
#include "clangdefmsg.h"
#include "coach_lang_comp.h"
#include "rule.h"


namespace rcss
{
  namespace clang
  {
    
    void
    MsgBuilder::emptyStack()
    {
      while( !M_items.empty() )
        {
          if( isItem< MetaToken* >() )
            getItem< MetaToken* >();
          else if( isItem< Action* >() )
            getItem< Action* >();
          else if( isItem< Dir* >() )
            getItem< Dir* >();
          else if( isItem< Cond* >() )
            getItem< Cond* >();
          else if( isItem< Token* >() )
            getItem< Token* >();
          else if( isItem< Def* >() )
            getItem< Def* >();
          else if( isItem< Region* >() )
            getItem< Region* >();
          else if( isItem< Point* >() )
            getItem< Point* >();
          else
            M_items.pop();
        }
    }

    void
    MsgBuilder::clear()
    {
      M_msg.release();
      M_min_ver = (unsigned int)-1;
      M_max_ver = 0;
    }      

    void
    MsgBuilder::onNoItems() const
    {
      throw BuilderErr( __FILE__, __LINE__,
                        "No item on stack." );        
    }
    
    void
    MsgBuilder::onWrongItem() const
    {
      throw BuilderErr( __FILE__, __LINE__,
                        "Wrong item on stack." );        
    }
    
    void
    MsgBuilder::onNotEmpty() const
    {
      throw BuilderErr( __FILE__, __LINE__,
                        "Stack is not empty when it should be." );        
    }
 
    MsgBuilder::MsgBuilder()
      : M_min_ver( (unsigned int)-1 ),
        M_max_ver( 0 )
    {}

    MsgBuilder::~MsgBuilder()
    { emptyStack(); }

    Msg*
    MsgBuilder::getMsg()
    {
      if (!M_items.empty())
	{
	  std::cerr << "MsgBuilder: Getting message with stack at" << M_items.size() << std::endl;
	}
      return M_msg.get();
    }
    
    const Msg*
    MsgBuilder::getMsg() const
    {
      if (!M_items.empty())
	{
	  std::cerr << "MsgBuilder: Const getting message with stack at" << M_items.size() << std::endl;
	}
      return M_msg.get();
    }
    
    std::auto_ptr< Msg >
    MsgBuilder::detatchMsg()
    {
      if (!M_items.empty())
	{
	  std::cerr << "MsgBuilder: Detaching message with stack at" << M_items.size() << std::endl;
	}
      return M_msg;
    }

    
    void
    MsgBuilder::checkItemsEmpty() const
    {
      if( !M_items.empty() )
        onNotEmpty();
    }

    
    void
    MsgBuilder::setMsg( Msg* msg )
    {
      std::auto_ptr< Msg > tmp( msg );
      M_msg = tmp;
      setVer( M_min_ver, M_max_ver );
      checkItemsEmpty();
    }

    
    void
    MsgBuilder::setTime( const int& time )
    {
      if( M_msg.get() != NULL )
        M_msg->setTimeSend( time );
    }
    
    
    void
    MsgBuilder::setSide( const int& side )
    {
      if( M_msg.get() != NULL )
        M_msg->setSide( side );
    }

    void
    MsgBuilder::setTimeRecv( const int& time )
    {
      if( M_msg.get() != NULL )
        M_msg->setTimeRecv( time );
    }      


    
    void
    MsgBuilder::setVer( const unsigned int& min,
                              const unsigned int& max )
    {
      if( min == max )
        {
          // Then the version is restriced to min/max
          if( min < M_max_ver )
            M_min_ver = min;
          else if( min > M_min_ver )
            M_max_ver = min;
          else if( M_max_ver < M_min_ver )
            M_min_ver = M_max_ver = min;
        }
      else
        {
          if( min < M_min_ver )
            M_min_ver = min;
          if( max > M_max_ver )
            M_max_ver = max;
          if( min > max 
              && M_max_ver < M_min_ver 
              && min > M_max_ver 
              && max < M_min_ver )
            {
              M_min_ver = min;
              M_max_ver = max;
            }
        }
      if( M_msg.get() != NULL )
        {
          M_msg->setVer( min, max );
        }
    }

      
    
    void
    MsgBuilder::buildMetaMsg()
    {
      traceBuild< MetaMsg* >();
      MetaMsg* msg( new MetaMsg() );
      if( checkIsItem< MetaToken* >() )
        {
          while( isItem< MetaToken* >() )
            msg->push_front( getPtrItem< MetaToken >() );
        }  
      setMsg( msg );
    }
      
    
    void
    MsgBuilder::buildMetaTokenVer( const double& ver )
    {
      traceBuild< MetaTokenVer* >();
      add( new MetaTokenVer( ver ) ); 
    }

      
    
    void
    MsgBuilder::buildDefineMsg()
    {
      traceBuild< DefineMsg* >();
      DefineMsg* msg = new DefineMsg();
      if( checkIsItem< Def* >() )
        {
          while( isItem< Def* >() )
            msg->push_front( getPtrItem< Def >() );
        }
      setMsg( msg );
    }

      
    
    void
    MsgBuilder::buildDefineCond( const std::string& name )
    {
      traceBuild< DefCond* >();
      add( new DefCond( name, checkAndGetPtrItem< Cond >() ) ); 
    }
      
    
    void
    MsgBuilder::buildDefineDir( const std::string& name )
    {
      traceBuild< DefDir* >();
      add( new DefDir( name, checkAndGetPtrItem< Dir >() ) ); 
    }
      
    
    void
    MsgBuilder::buildDefineReg( const std::string& name )
    {
      traceBuild< DefReg* >();
      add( new DefReg( name, checkAndGetPtrItem< Region >() ) ); 
    }
      
    
    void
    MsgBuilder::buildDefineAct( const std::string& name )
    { 
      traceBuild< DefAct* >();
      add( new DefAct( name, checkAndGetPtrItem< Action >() ) ); 
    }
      
    
    void
    MsgBuilder::buildFreeformMsg( const std::string& str )
    { 
      traceBuild< FreeformMsg* >();
      setMsg( new FreeformMsg( str ) ); 
    }
     
    
    void
    MsgBuilder::buildUnsuppMsg()
    { 
      traceBuild< UnsuppMsg* >();
      setMsg( new UnsuppMsg() ); 
    }
      
    
    void  
    MsgBuilder::buildInfoMsg()
    {
      traceBuild< InfoMsg* >();
      InfoMsg* msg = new InfoMsg();
      if( checkIsItem< Token* >() )
        {
          while( isItem< Token* >() )
            msg->push_front( getPtrItem< Token >() );
        }
      setMsg( msg );
    }
      
    
    void
    MsgBuilder::buildAdviceMsg()
    {
      traceBuild< AdviceMsg* >();
      AdviceMsg* msg = new AdviceMsg();
      if( checkIsItem< Token* >() )
        {
          while( isItem< Token* >() )
            msg->push_front( getPtrItem< Token >() );
        }
      setMsg( msg );
    }
      
    
    void
    MsgBuilder::buildTokenRule( const int& ttl )
    {
      traceBuild< TokRule* >();
      TokRule* token = new TokRule( ttl );
      if( checkIsItem< Dir* >() )
        {
          while( isItem< Dir* >() )
            token->push_front( getPtrItem< Dir >() );
        }
      token->set( checkAndGetPtrItem< Cond >() );
      add( token );
    }
      
    
    void
    MsgBuilder::buildTokenClear()
    {
      traceBuild< TokClear* >();
      add( new TokClear() );
    }
 
    
    void
    MsgBuilder::buildActPos()
    {
      traceBuild< ActPos* >();
      add( new ActPos( checkAndGetPtrItem< Region >() ) ); 
    }
      
    
    void
    MsgBuilder::buildActHome()
    {
      traceBuild< ActHome* >();
      add( new ActHome( checkAndGetPtrItem< Region >() ) );
    }
      
    
    void
    MsgBuilder::buildActBallToReg()
    { 
      traceBuild< ActBallToReg* >();
      BallMove bm;
      if( checkIsItem< BallMoveToken >() )
        {
          while( isItem< BallMoveToken >() )
            bm.addToken( getItem< BallMoveToken >() );
        }
      std::auto_ptr< Region > reg = checkAndGetPtrItem< Region >(); 
      add( new ActBallToReg( reg, bm ) );
    }
      
    
    void
    MsgBuilder::buildActBallToPlayer()
    {
      traceBuild< ActBallToPlayer* >();
      add( new ActBallToPlayer( checkAndGetItem< UNumSet >() ) );
    }
 
    
    void
    MsgBuilder::buildActMark()
    {
      traceBuild< ActMark* >();
      add( new ActMark( checkAndGetItem< UNumSet >() ) );
    }
      
    
    void
    MsgBuilder::buildActMarkLinePlayer()
    {
      traceBuild< ActMarkLinePlayer* >();
      add( new ActMarkLinePlayer( checkAndGetItem< UNumSet >() ) );
    }
      
    
    void
    MsgBuilder::buildActMarkLineReg()
    {
      traceBuild< ActMarkLineReg* >();
      add( new ActMarkLineReg( checkAndGetPtrItem< Region >() ) ); 
    }

    
    void
    MsgBuilder::buildActOffsideLine()
    {
      traceBuild< ActOffsidesLine* >();
      add( new ActOffsidesLine( checkAndGetPtrItem< Region >() ) ); 
    }
      
    
    void
    MsgBuilder::buildActHetType( const int& type )
    {
      traceBuild< ActHetType* >();
      add( new ActHetType( type ) ); 
    }
      
    
    void
    MsgBuilder::buildActNamed( const std::string& name )
    {
      traceBuild< ActNamed* >();
      add( new ActNamed( name ) ); 
    }
      
    
    void
    MsgBuilder::buildActPassReg()
    {
      traceBuild< ActPassReg* >();
      add( new ActPassReg( checkAndGetPtrItem< Region >() ) ); 
    }
      
    
    void
    MsgBuilder::buildActPassUNum()
    {
      traceBuild< ActPassUNum* >();
      add( new ActPassUNum( checkAndGetItem< UNumSet >() ) ); 
    }
 
    
    void
    MsgBuilder::buildActDribble()
    {
      traceBuild< ActDribble* >();
      add( new ActDribble( checkAndGetPtrItem< Region >() ) ); 
    }
 
    
    void
    MsgBuilder::buildActClear()
    {
      traceBuild< ActClear* >();
      add( new ActClear( checkAndGetPtrItem< Region >() ) ); 
    }
 
    
    void
    MsgBuilder::buildActShoot()
    {
      traceBuild< ActShoot* >();
      add( new ActShoot() );
    }
      
    
    void
    MsgBuilder::buildActHold()
    {
      traceBuild< ActHold* >();
      add( new ActHold() ); 
    }
 
    
    void
    MsgBuilder::buildActIntercept()
    {
      traceBuild< ActIntercept* >();
      add( new ActIntercept() ); 
    }
 
    
    void
    MsgBuilder::buildActTackle()
    {
      traceBuild< ActTackle* >();
      add( new ActTackle( checkAndGetItem< UNumSet >() ) ); 
    }
      
    
    void
    MsgBuilder::buildDirComm( const bool& do_dont, const bool& our_side )
    {
      traceBuild< DirComm* >();
      DirComm* dir = new DirComm( do_dont, our_side, UNumSet(), 
                                  std::list< Action* >() );
      if( checkIsItem< Action* >() )
        {
          while( isItem< Action* >() )
            {
              dir->add( getPtrItem< Action >() );
            }
        }
      dir->set( checkAndGetItem< UNumSet >() );
      add( dir );
    }
      
    
    void
    MsgBuilder::buildDirNamed( const std::string& name )
    { 
      traceBuild< DirNamed* >();
      add( new DirNamed( name ) ); 
    }
      
    
    void
    MsgBuilder::buildCondTrue()
    {
      traceBuild< CondBool* >();
      add( new CondBool( true ) ); 
    }
 
    
    void
    MsgBuilder::buildCondFalse()
    {
      traceBuild< CondBool* >();
      add( new CondBool( false ) ); 
    }
      
    
    void
    MsgBuilder::buildCondPlayerPos( const bool& our_side, 
                                          const int& min, 
                                          const int& max )
    {
      traceBuild< CondPlayerPos* >();
      std::auto_ptr< Region > reg = checkAndGetPtrItem< Region >(); 
      UNumSet unums = checkAndGetItem< UNumSet >();
      add( new CondPlayerPos( our_side, unums, min, max, reg ) );
    }
      
    
    void
    MsgBuilder::buildCondBallPos()
    {
      traceBuild< CondBallPos* >();
      add( new CondBallPos( checkAndGetPtrItem< Region >() ) ); 
    }
 
    
    void
    MsgBuilder::buildCondBallOwner( const bool& our_side )
    {
      traceBuild< CondBallOwner* >();
      add( new CondBallOwner( our_side, checkAndGetItem< UNumSet >() ) );
    }
      
    
    void
    MsgBuilder::buildCondPlayMode( const PlayMode& play_mode )
    {
      traceBuild< CondPlayMode* >();
      add( new CondPlayMode( play_mode ) ); 
    }
      
    
    void
    MsgBuilder::buildCondAnd()
    {
      traceBuild< CondAnd* >();
      CondList l = checkAndGetItem< CondList >();
      std::auto_ptr< CondAnd > cond( new CondAnd(l) );
      add( cond.release() );
    }
      
    
    void
    MsgBuilder::buildCondOr()
    {
      traceBuild< CondOr* >();
      CondList l = checkAndGetItem< CondList >();
      std::auto_ptr< CondOr > cond( new CondOr(l) );
      add( cond.release() );
    }
 
    
    void
    MsgBuilder::buildCondNot()
    {
      traceBuild< CondNot* >();
      add( new CondNot( checkAndGetPtrItem< Cond >() ) ); 
    }
      
    
    void
    MsgBuilder::buildCondNamed( const std::string& name )
    {
      traceBuild< CondNamed* >();
      add( new CondNamed( name ) ); 
    }
 
    
    void
    MsgBuilder::buildCondTime( const int& time,
                                     const util::CompOp& comp )
    {
      traceBuild< CondTime* >();
      add( new CondTime( time, comp ) ); 
    }
      
    
    void
    MsgBuilder::buildCondOppGoal( const int& goals,
                                        const util::CompOp& comp )
    {
      traceBuild< CondOppGoal* >();
      add( new CondOppGoal( goals, comp ) ); 
    }
 
    
    void
    MsgBuilder::buildCondOurGoal( const int& goals,
                                        const util::CompOp& comp )
    {
      traceBuild< CondOurGoal* >();
      add( new CondOurGoal( goals, comp ) ); 
    }     
 
    
    void
    MsgBuilder::buildCondGoalDiff( const int& goals, 
                                         const util::CompOp& comp )
    {
      traceBuild< CondGoalDiff* >();
      add( new CondGoalDiff( goals, comp ) ); 
    }
      
    
    void
    MsgBuilder::buildCondUNum( const rcss::clang::UNum& unum )
    { 
      traceBuild< CondUNum* >();
      add( new CondUNum( unum, checkAndGetItem< UNumSet >() ) );
    }
      
    
    void
    MsgBuilder::buildAddToCondList ( )
    {
      traceBuild< std::list<Cond*> >();
      std::auto_ptr<Cond> c = checkAndGetPtrItem< Cond >();
      CondList l = checkAndGetItem< CondList >();
      l.push_back( c.release() );
      add ( l );
    }
    
    void
    MsgBuilder::buildCreateCondList ( )
    {
      traceBuild< std::list<Cond*> >();
      std::auto_ptr<Cond> c = checkAndGetPtrItem< Cond >();
      CondList l;
      l.push_back(c.release());
      add ( l );
    }
    

    void
    MsgBuilder::buildRegNull()
    {
      traceBuild< RegNull* >();
      add( new RegNull() ); 
    }
      
    
    void
    MsgBuilder::buildRegQuad()
    {
      traceBuild< RegQuad* >();
      std::auto_ptr< Point > p4 = checkAndGetPtrItem< Point >();
      std::auto_ptr< Point > p3 = checkAndGetPtrItem< Point >();
      std::auto_ptr< Point > p2 = checkAndGetPtrItem< Point >();
      std::auto_ptr< Point > p1 = checkAndGetPtrItem< Point >();
      add( new RegQuad( p1, p2, p3, p4 ) );
    }
      
    
    void
    MsgBuilder::buildRegArc( const double& start_rad,
                                   const double& end_rad,
                                   const double& start_ang,
                                   const double& span_ang )
    {
      traceBuild< RegArc* >();
      add( new RegArc( checkAndGetPtrItem< Point >(),
                       start_rad, end_rad,
                       start_ang, span_ang ) );
    }
      
    
    void
    MsgBuilder::buildRegUnion()
    {
      traceBuild< RegUnion* >();
      std::auto_ptr< RegUnion > reg( new RegUnion() );
      if( checkIsItem< Region* >() )
        {
          while( isItem< Region* >() )
            reg->push_front( getPtrItem< Region >() );
        }
      add( reg.release() );
    }
 
    
    void
    MsgBuilder::buildRegNamed( const std::string& name )
    {
      traceBuild< RegNamed* >();
      add( new RegNamed( name ) ); 
    }
 
    
    void
    MsgBuilder::buildRegPoint()
    {
      traceBuild< RegPoint* >();
      add( new RegPoint( checkAndGetPtrItem< Point >() ) ); 
    }
 
    
    void
    MsgBuilder::buildRegTri()
    {
      traceBuild< RegTri* >();
      std::auto_ptr< Point > p3 = checkAndGetPtrItem< Point >();
      std::auto_ptr< Point > p2 = checkAndGetPtrItem< Point >();
      std::auto_ptr< Point > p1 = checkAndGetPtrItem< Point >();
      add( new RegTri( p1, p2, p3 ) ); 
    }
 
    
    void
    MsgBuilder::buildRegRec()
    {
      traceBuild< RegRec* >();
      std::auto_ptr< Point > p2 = checkAndGetPtrItem< Point >();
      std::auto_ptr< Point > p1 = checkAndGetPtrItem< Point >();
      add( new RegRec( p1, p2 ) ); 
    }
 
    
    void
    MsgBuilder::buildPointSimple( const double& x, const double& y )
    {
      traceBuild< PointSimple* >();
      add( new PointSimple( x, y ) ); 
    }
 
    
    void
    MsgBuilder::buildPointRel( const double& x, const double& y )
    {
      traceBuild< PointRel* >();
      add( new PointRel( x, y, checkAndGetPtrItem< Point >() ) ); 
    }
 
    
    void
    MsgBuilder::buildPointBall()
    {
      traceBuild< PointBall* >();
      add( new PointBall() ); 
    }
      
    
    void
    MsgBuilder::buildPointPlayer( const bool& our_side,
                                        const UNum& unum )
    {
      traceBuild< PointPlayer* >();
      add( new PointPlayer( our_side, unum ) ); 
    }
      
    
    void
    MsgBuilder::buildPointArith( const rcss::util::ArithOp& arith_op )
    {
      traceBuild< PointArith* >();
      // we need to make sure ordering is maintained, otherwise you could end
      // up with 'b / a', when 'a / b' was what was actually sent.
      std::auto_ptr< Point > second = checkAndGetPtrItem< Point >();
      std::auto_ptr< Point > first = checkAndGetPtrItem< Point >();
      add( new PointArith( first, second, arith_op ) ); 
    }
      
    
    void
    MsgBuilder::buildUNum( const UNum& unum )
    { 
      traceBuild< UNum* >();
      add( unum ); 
    }
 
    
    void
    MsgBuilder::buildUNumSet()
    { 
      traceBuild< UNumSet >();
      UNumSet unums;
      if( checkIsItem< UNum >() )
        {
          while( isItem< UNum >() )
            unums.add( getItem< UNum >() );
        }
      add( unums ); 
    }
 
    
    void
    MsgBuilder::buildBallMoveToken( const BallMoveToken& bmt )
    {
      traceBuild< BallMoveToken >();
      add( bmt ); 
    }

    
    void
    MsgBuilder::buildRuleMsg()
    {
      traceBuild< RuleMsg* >();
      RuleMsg* msg = new RuleMsg();
      if( checkIsItem< ActivateRules >() )
        {
          while( isItem< ActivateRules >() )
            msg->push_front( getItem< ActivateRules >() );
        }
      setMsg( msg ); 
    }
    
    
    void
    MsgBuilder::buildActivateAllRules( const bool& on )
    {
      traceBuild< ActivateRules >();
      add( ActivateRules( on, RuleIDList() ) );
    }

    
    void
    MsgBuilder::buildActivateRules( const bool& on )
    {
      traceBuild< ActivateRules >();
      ActivateRules act( on, RuleIDList() );
      if( isItem< RuleID >() )
        {
          RuleIDList rids;
          rids.push_front( getItem< RuleID >() );
          act.set( rids );
        }
      else
        {
          act.set( checkAndGetItem< RuleIDList >() );
        } 
      add( act );
    }

    
    void
    MsgBuilder::buildRuleID( const std::string& id )
    {
      traceBuild< RuleMsg* >();
      add( id );
    }

    
    void
    MsgBuilder::buildRuleIDList()
    {
      traceBuild< RuleIDList >();
      RuleIDList rids;
      if( checkIsItem< RuleID >() )
        {
          while( isItem< RuleID >() )
            rids.push_front( getItem< RuleID >() );
          // by adding the items to the front, the original order is
          // maintained.
        }
      add( rids ); 
    }

    
    void
    MsgBuilder::buildRuleIDListALL()
    {
      traceBuild< RuleIDList >();
      add( RuleIDList() ); // an empty list == all rules.
    }

    
    void
    MsgBuilder::buildDelMsg()
    {
      traceBuild< DelMsg* >();
      DelMsg* msg = new DelMsg;
      if( isItem< RuleID >() )
        {
          RuleIDList rids;
          rids.push_front( getItem< RuleID >() );
          msg->set( rids );
        }
      else
        {
          msg->set( checkAndGetItem< RuleIDList >() );
        }
      setMsg( msg ); 
    }

    
    void
    MsgBuilder::buildDefineModelRule( const std::string& id )
    {
      traceBuild< DefRule* >();
      add( new DefRule( id, checkAndGetPtrItem< Rule >(), true ) ); 
    }

    
    void
    MsgBuilder::buildDefineDirectiveRule( const std::string& id )
    {
      traceBuild< DefRule* >();
      add( new DefRule( id, checkAndGetPtrItem< Rule >(), false ) ); 
    }

    
    void
    MsgBuilder::buildSimpleRule()
    {
      traceBuild< SimpleRule* >();
      SimpleRule::Storage dirs;
      if( checkIsItem< Dir* >() )
        {
          do
            {
              dirs.push_front( getPtrItem< Dir >().release() );
            }
          while( isItem< Dir* >() );
        }
      add( new SimpleRule( checkAndGetPtrItem< Cond >(), dirs ) );
    }

    
    void
    MsgBuilder::buildNestedRule()
    {
      traceBuild< NestedRule* >();
      NestedRule::Storage rules;
      if( checkIsItem< Rule* >() )
        {
          do
            {
              rules.push_front( getPtrItem< Rule >().release() );
            }
          while( isItem< Rule* >() );
        }
      add( new NestedRule( checkAndGetPtrItem< Cond >(), rules ) );
    }

    
    void
    MsgBuilder::buildIDRule()
    {
      traceBuild< IDListRule* >();
      add( new IDListRule( checkAndGetItem< RuleIDList >() ) );
    }
  }
}
