/* -*- Mode: C++ -*- */

/*
 * An abstract base class for accessing a world model. 
 * A player or coachteam should define a subclass that implements these methods.
 *
 * John Davin
 */


#ifndef WORLD_MODEL_INTERFACE_H
#define WORLD_MODEL_INTERFACE_H

#include "Geometry.h"
#include "vector.h" // for Vector2D
#include "clangutil.h" // for UNum


// This is the interface that regions need
class RegionWorldModelInterface 
{
 public:
  /*
   * Returns a vector holding the global position of the agent specified by num
   * The argument ourside is true if the player number specifies a player on our team
   * (and false if it is the opponent's team)
   */
  virtual VecPosition getGlobalPosition(rcss::clang::UNum::unum_t num, 
					bool ourside) = 0;

  /*
   * Returns a vector containing the current ball position. 
   *
   */
  virtual rcss::geom::Vector2D getBallPos() = 0;

  /*
   * Returns whether the coordinates should be flipped to use right-hand coordinates
   * The player should never have to do this as the server always flips the coordinates
   * so that the player always gets left-hand coordinates.
   * The coach will want to do this to evaluate for the right hand team
   * Note that the positions returned above should ALREADY be in right hand coordinates
   * if this function returns true. 
   */
  virtual bool useRightSideCoordinates() = 0;
  
};

// This is the interface that all of clang needs
class WorldModelInterface
  : public RegionWorldModelInterface
{

 public:

  /*
   * Return true if confidence of ball location is good, false otherwise
   */
  virtual bool ballConfidenceGood() = 0;

  /*
   * Returns the current play mode. 
   *
   */
  virtual rcss::clang::PlayMode getPlayMode() = 0;

  /*
   * Returns the current time. 
   *
   */
  //virtual Time getCurrentTime() = 0;

  /*
   * Effect: Place a unum_t into player corresponding to the last player to touch the ball.
   * Return: true if the ball owner is on our team, false if opponent team.
   */
  virtual bool getBallOwner(rcss::clang::UNum::unum_t *player) = 0;

  /*
   * Return: the score for the team this agent is on. 
   */
  virtual int getOurGoal() = 0;

  /*
   * Return: score for the opponent team. 
   */
  virtual int getOppGoal() = 0;


  /*
   * This returns ourgoal - oppgoal. So positive means you're winning, negative losing.
   */
  virtual int getGoalDiff() = 0;

  /*
   * Returns the UNum of this player. 
   */
  virtual rcss::clang::UNum getMyNum() = 0;

};



#endif // WORLD_MODEL_INTERFACE_H

