/*
 * This class defines a variable bindings storage class which is used to hold the variable bindings for 
 * a condition (and attach them to a directive later)
 *
 * John Davin
 */


#ifndef VAR_BINDINGS_H
#define VAR_BINDINGS_H

#include <set>
#include "clangutil.h"


/* 
 * A binding is something of the form { X -> 1 2 3 }, 
 * ie, a mapping from a variable to its set of bound values
 */
typedef struct binding_object {
  std::string variable;
  rcss::clang::UNumSet set;

} binding;


/* comparison function for a set of bindings */
struct ltstr
{
  bool operator()(binding b1, binding b2) const
  {
    return b1.variable < b2.variable;
  }
};


/*
 * VarBindings is a set of bindings
 * eg, { X -> 1 2 3 }, { X -> 5 6 }, { Y -> 2 4 6 }
 * We'll store the list of bindings as a set so that variables can't have more than 
 * one binding list (ie, there should only be one binding list for X). 
 */
class VarBindings {

 private: 
  std::set<binding, ltstr> bindings_list;

  /*
   * helper function for the public add functions
   */
  void addHelper(const std::string& varname, 
		 const rcss::clang::UNumSet& bind_set);

  /*
   * helper function for attachBindings and mergeBindings
   */
  void attachBindingsHelper(const VarBindings& bindings, bool useunion);

 public: 

  typedef std::set<binding>::const_iterator BindingIterator;
  typedef std::set<binding>::iterator BindingIteratorNotConst;

  /* 
   * Add this binding pair to the list. 
   */
  void add(const std::string& varname, const rcss::clang::UNumSet& set);
  void add(const rcss::clang::UNum& unum, const rcss::clang::UNumSet& set);


  /* 
   * Append another list of bindings to our list, resolving conflicts with intersection
   */
  void attachBindings(VarBindings *bindings);
  void attachBindings(const VarBindings &bindings);

  /* 
   * Append another list of bindings to our list, resolving conflicts with 
   * union
   */
  void mergeBindings(const VarBindings *bindings);
  void mergeBindings(const VarBindings &bindings);

  /*
   * Given a variable name, search our bindings_list for it. 
   * Return: the variable's  rcss::clang::UNumSet if found
   *                   or an empty rcss::clang::UNumSet if the varname is not found
   */
  rcss::clang::UNumSet lookup(const std::string& varname) const ;

  bool empty() const { return bindings_list.empty(); }

  /*
   * return true if any sets have player bindings
   * return false if there are no bindings, or if all bindings are {}. 
   */
  bool hasNonemptySets() const;

  /*
   * Return the inverse of these bindings
   */
  VarBindings *inverse();

  BindingIterator Begin() const{
    return bindings_list.begin();
  }

  BindingIterator End() const {
    return bindings_list.end();
  }

  void PrintBinding(const binding &b) const;

  void PrintAll() const;

  /*
   * return this - b1. subtract each binding in b1 from each binding in this. 
   */
  inline VarBindings operator- ( const VarBindings &b1 ) const {

    VarBindings newbindings = *this;

    for(BindingIterator it = b1.Begin(); it != b1.End(); it++){

      rcss::clang::UNumSet origset = lookup((*it).variable);
      // if origset is empty (not bound in this VarBindings instance) then we should treat it as {0}
      // the UNumSet '-' operator will handle this. 

      newbindings.add((*it).variable, origset - (*it).set);

    }
    return newbindings;
  }

};


/*
 * VarBindingSet is a vector of VarBindings
 * Essentially this is used to represent the union of multiple variable binding sets. 
 */
class VarBindingSet {

 private: 
  std::vector<VarBindings> bindings_list;


 public:
  typedef std::vector<VarBindings>::const_iterator BindingSetIter;
  typedef std::vector<VarBindings>::iterator BindingSetIterNotConst;


  void add(const VarBindings& v);
  void add(const VarBindingSet& s);

  // add the varbindings to each VarBindings already in our set (ie, don't expand set)
  void merge(const VarBindings& v);
  void merge(const VarBindingSet& s);

  void join(const VarBindings& v);
  void join(const VarBindingSet& s);

  bool hasNonemptySets() const;

  rcss::clang::UNumSet lookup(const std::string& varname) const;

  // defines algorithm for selecting (randomly?) a VarBinding from the set. 
  VarBindings SelectAVarBinding() const;

  BindingSetIter Begin() const
    {
      return bindings_list.begin();
    }
  
  BindingSetIter End() const
    {
      return bindings_list.end();
    }

  BindingSetIterNotConst Begin2()
    {
      return bindings_list.begin();
    }
  
  BindingSetIterNotConst End2()
    {
      return bindings_list.end();
    }
  
  /*
   * return this - b1. subtract each binding in b1 from each binding in this. 
   */
  inline VarBindingSet operator- (const VarBindingSet& b1 ) const {

    VarBindingSet rval;
    
    for(BindingSetIter it = b1.Begin(); it != b1.End(); it++){
      rval.add(*this - *it);
    }
    
    /*PR(8, "VarBindingSet: ");
      this->PrintAll();
      PR(8, " - ");
      b1.PrintAll();
      PR(8, " = ");
      rval.PrintAll();
    */

    return rval;
  }

  /*
   * return this - b1. subtract each binding in b1 from each binding in this. 
   */
  inline VarBindingSet operator- (const VarBindings& b ) const 
    {
      
      VarBindingSet rval;
      
      // iterate over each varbinding in this set
      for(BindingSetIter it = Begin(); it != End(); it++){
	rval.add(*it - b);	
      }

      // if this set is empty, we need to treat it as {0}, 
      // represented by an empty set
      if(Begin() == End()){
	rval.add(VarBindings() - b);
      }
      
      /*PR(8, "VarBindingSet: ");
	this->PrintAll();
	PR(8, " -- ");
	b.PrintAll();
	PR(8, " = ");
	rval.PrintAll();
      */

      return rval;
    }
  
  
  
  void PrintAll() const
    {
      /* TODO: pfr: I am taking this out in order to move to clanglib
	 PR(11, "{");
	 for(BindingSetIter it = Begin(); it != End(); it++){
	 (*it).PrintAll();
	 }
	 PR(11, "}");
      */
    }
  

};


inline std::ostream& 
operator<<( std::ostream & os, VarBindings v )
{ 
  v.PrintAll();
  return os; // yeah, i know, i don't have time to do this the right way
} 



inline std::ostream& 
operator<<( std::ostream & os, VarBindingSet s )
{ 
  s.PrintAll();
  return os; 
} 



#endif // VAR_BINDINGS_H
