/*
 * This class defines a variable bindings storage class which is used to hold the variable bindings for 
 * a condition (and attach them to a directive later)
 *
 * John Davin
 */

#include <stdio.h>
#include <assert.h>
#include <algorithm>
#include "VarBindings.h"

using namespace std;


/*
 * Return the intersection of two UNumset's. 
 */
rcss::clang::UNumSet Intersection(const rcss::clang::UNumSet &u1, 
				  const rcss::clang::UNumSet &u2)   {

  rcss::clang::UNumSet newset;

  for(rcss::clang::UNum u = rcss::clang::UNum::u1; 
      u.getUNum() <= rcss::clang::UNum::u11; u++){

    if(u1.contains(u) && u2.contains(u))
      newset.add(u);

  }

  return newset;
}

/*
 * Return the union of two UNumset's. 
 */
rcss::clang::UNumSet Union(const rcss::clang::UNumSet &u1, 
			   const rcss::clang::UNumSet &u2)   {

  rcss::clang::UNumSet newset;

  for(rcss::clang::UNum u = rcss::clang::UNum::u1; 
      u.getUNum() <= rcss::clang::UNum::u11; u++){

    if(u1.contains(u) || u2.contains(u))
      newset.add(u);

  }

  return newset;
}


/* 
 * Add this binding pair to the list. 
 */
void VarBindings::add(const rcss::clang::UNum &unum, 
		      const rcss::clang::UNumSet &bind_set){

  // call helper with unum variable name
  addHelper(unum.getVar(), bind_set);
}

/* 
 * Function overloading on first param. 
 * Add this binding pair to the list. 
 */
void VarBindings::add(const string &varname, 
		      const rcss::clang::UNumSet &bind_set) {
  
  // just call the helper
  addHelper(varname, bind_set);
}


/*
 * Helper function called by the add functions
 * Does the actual work of adding a binding.
 */
void VarBindings::addHelper(const string &varname, 
			    const rcss::clang::UNumSet& bind_set){

  binding newbinding;
  newbinding.variable = varname;
  newbinding.set = bind_set;

  if(bind_set.contains(rcss::clang::UNum::uMax)){

    printf("[VarBindings:add]: tried to bind var %s to uMax!\n", 
	   varname.c_str());
    assert(0);

  }

  if(bind_set.contains(rcss::clang::UNum::uAll)){

    // TODO: pfr: I am taking this out in order to move to clanglib
    //PR(11, "[VarBindings:add]: expanding uAll to player set\n");
    rcss::clang::UNumSet all;

    for(rcss::clang::UNum u = rcss::clang::UNum::u1; 
	u.getUNum() <= rcss::clang::UNum::u11; u++)
      all.add(u);

    newbinding.set = all;
  }

  BindingIteratorNotConst it = bindings_list.find(newbinding);

  if(it != bindings_list.end()){ // variable is already bound to something

    // so take the intersection of its current binding and the added binding

    /* LOGREMOVED 
    if(Log.isInLogLevel(11)){
      *logger << "[VarBindings:add]: found dup" << endl;
      *logger << "Variable "<<(*it).variable <<" is already bound to "<<(*it).set <<endl;
      *logger << "We were trying to bind variable "<<newbinding.variable<<
	" to "<<newbinding.set <<endl;
    }
    */
    newbinding.set = Intersection((*it).set, newbinding.set);

    bindings_list.erase(it);

    /* LOGREMOVED 
    if(Log.isInLogLevel(11)){
      *logger << "Bound variable "<<newbinding.variable<<
	" to "<<newbinding.set <<endl;}
    */
  }

  bindings_list.insert( bindings_list.end(), newbinding);

  //PrintBinding(newbinding);
}

/*
 * A helper function for attachBindings and mergeBindings. 
 * If useunion is true, resolve conflicts with union. Otherwise use 
 * intersection
 */
void VarBindings::attachBindingsHelper(const VarBindings &bindings, 
				       bool useunion) {

  BindingIterator oldbinding;


  for(BindingIterator it = bindings.Begin(); it != bindings.End(); it++){

    oldbinding = bindings_list.find(*it);

    if( oldbinding == bindings_list.end()) 

      // binding not already present
      bindings_list.insert(bindings_list.end(), *it);

    else { // then binding is already present

      binding newbinding;
      newbinding.variable = (*it).variable;

      if(useunion)
	newbinding.set = Union((*oldbinding).set, (*it).set);
      else
	newbinding.set = Intersection((*oldbinding).set, (*it).set);

      // remove the old one
      bindings_list.erase(oldbinding);

      // and insert the new one
      bindings_list.insert( bindings_list.end(), newbinding);
      
    }

  }
}

/* 
 * Append another list of bindings to our list. 
 * If a variable is already bound, take the intersection of the two sets
 */
void VarBindings::attachBindings(VarBindings *bindings){
  attachBindingsHelper(*bindings, false);
}
void VarBindings::attachBindings(const VarBindings &bindings){
  attachBindingsHelper(bindings, false);
}


/* 
 * Append another list of bindings to our list. 
 * If a variable is already bound, take the Union of the two sets
 */
void VarBindings::mergeBindings(const VarBindings *bindings){
  attachBindingsHelper(*bindings, true);
}
void VarBindings::mergeBindings(const VarBindings &bindings){
  attachBindingsHelper(bindings, true);
}

/*
 * Given a variable name, search our bindings_list for it. 
 * Return: the variable's  UNumSet if found
 *                   or an empty UNumSet if the varname is not found
 */
rcss::clang::UNumSet VarBindings::lookup(const std::string &varname) const {
  
  for(BindingIterator i = bindings_list.begin(); 
      i != bindings_list.end(); i++)    {

    if( (*i).variable == varname)
      return (*i).set;

  }

  // if we get here, the varname was not found in the bindings
  // return an empty UNumSet
  return rcss::clang::UNumSet();

}

/*
 * return true if any sets have player bindings
 * return false if there are no bindings, or if all bindings are {}. 
 */
bool VarBindings::hasNonemptySets()   const  {

  if(empty()) return false; // there are no bindings. 

  for(BindingIterator i = bindings_list.begin(); 
      i != bindings_list.end(); i++)     {

    rcss::clang::UNumSet thisset = (*i).set;
    if(thisset.begin() != thisset.end()) // it's a nonempty set
      return true;

  }

  return false; // didn't find any nonempty sets. 
}


/*
 * Return the inverse set of the input binding
 */
binding invertBinding(const binding& old)   {

  binding newbinding;
  newbinding.variable = old.variable;
  rcss::clang::UNumSet newset;

  // if a player is not in the old set, then put it in the new one
  for(rcss::clang::UNum u = rcss::clang::UNum::u1; 
      u.getUNum() <= rcss::clang::UNum::u11; u++){

    if( ! old.set.contains(u))
      newset.add(u);

  }  
  newbinding.set = newset;

  return newbinding;
}


/*
 * Return the inverse of these bindings
 */
VarBindings* VarBindings::inverse()
{

  std::set<binding, ltstr> new_bindings;

  for(BindingIterator i = bindings_list.begin(); 
      i != bindings_list.end(); i++)    {

    new_bindings.insert( new_bindings.end(), invertBinding(*i));

  }

  // set this object's bindings to the new ones. 
  bindings_list = new_bindings;

  return this;
}

void VarBindings::PrintBinding(const binding &b) const
{  
  /* LOGREMOVED 
  if(Log.isInLogLevel(11)){
    *logger << "( "<<b.variable<<" --> ";
    rcss::clang::UNumSet s = b.set;

    for( rcss::clang::UNumSet::const_iterator it = s.begin(); it != s.end(); it++){
      *logger << " "<<(*it).getUNum()<<" ";
   
    }
    *logger << ") ";

  }
  */
}

void VarBindings::PrintAll() const 
{
  
  /* TODO: pfr: I am taking this out in order to move to clanglib
  if(bindings_list.begin() == bindings_list.end()){
    PR(11, "[VarBindings]: No variable bindings.\n");
    return;
  }
    
  if(Log.isInLogLevel(11)) *logger << "[VarBindings]: ";
  for(BindingIterator i = bindings_list.begin(); i != bindings_list.end(); i++){
    PrintBinding(*i);
  }
  if(Log.isInLogLevel(11)) *logger << endl;
  */
}


/******** Class VarBindingSet **********/

void VarBindingSet::add(const VarBindings& v)
{

  bindings_list.insert( bindings_list.end(), v);

}


/*
 * Add the VarBindings in s to our set. 
 *
 */
void VarBindingSet::add(const VarBindingSet& s)
{

  for(BindingSetIter it = s.Begin(); it != s.End(); it++){

    bindings_list.insert(bindings_list.end(), *it);

  }

}

/*
 * add the varbindings to each VarBindings already in our set (ie, don't expand set)
 */
void VarBindingSet::merge(const VarBindings &v)
{

  if(v.Begin() == v.End()) return; // don't add empty sets

  // if there's nothing in the binding set yet, add it as a new entry
  if(Begin() == End()){ add(v); return;}

  for(BindingSetIterNotConst it = Begin2(); it != End2(); it++){
    (*it).attachBindings(v);

  }

}

void VarBindingSet::merge(const VarBindingSet &s)
{

  for(BindingSetIter it = s.Begin(); it != s.End(); it++){
    merge(*it);
  }

}


/*
 * add the varbindings to each VarBindings already in our set (ie, don't expand set)
 */
void VarBindingSet::join(const VarBindings& v)
{

  if(v.Begin() == v.End()) return; // don't add empty sets

  // if there's nothing in the binding set yet, add it as a new entry
  if(Begin() == End()){ add(v); return;}

  for(BindingSetIterNotConst it = Begin2(); it != End2(); it++){
    (*it).mergeBindings(v);

  }

}

void VarBindingSet::join(const VarBindingSet& s)
{

  for(BindingSetIter it = s.Begin(); it != s.End(); it++){
    join(*it);
  }

}


/*
 * defines algorithm for selecting (randomly?) a VarBinding from the set. 
 */
VarBindings VarBindingSet::SelectAVarBinding() const
{
  // just use first binding for now
  
  
  // FUNCTION DEPRECATED, DO NOT USE, DELETE SOON
  assert(0);
  
  if(Begin() != End())
    return *(Begin());
  else
    return VarBindings();
}


/*
 * Look through our VarBindings and return lookup value for varname
 */
rcss::clang::UNumSet VarBindingSet::lookup(const std::string& varname) const {

  rcss::clang::UNumSet set;
  
  for(BindingSetIter it = Begin(); it != End(); it++){

    set = (*it).lookup(varname);
    
    if(set.begin() != set.end()) // found the varname

      return set;
  }
  
  return set; // didn't find the varname, returns empty set
}


/*
 * Just iterate over our set of varbindings and determine if any have empty bindings
 */
bool VarBindingSet::hasNonemptySets() const
{
  bool b = false;

  for(BindingSetIter it = Begin(); it != End(); it++){
    b |= (*it).hasNonemptySets(); 

  }

  return b;
}

