/* -*- Mode: c++ -*- */

/*
 *Copyright:

    Copyright (C) 2002 Patrick Riley
    Copyright (C) 2001 Patrick Riley and Emil Talpes

    This file is part of the SPADES simulation system.

    The SPADES simulation system is free software; you can
    redistribute it and/or modify it under the terms of the GNU Lesser
    General Public License as published by the Free Software
    Foundation; either version 2 of the License, or (at your option)
    any later version.

    The SPADES simulation system is distributed in the hope that it
    will be useful, but WITHOUT ANY WARRANTY; without even the implied
    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with the SPADES simulation system; if not, write to
    the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
    Boston, MA 02111-1307 USA

 *EndCopyright:
*/

#ifndef _LOGGING_H_
#define _LOGGING_H_

#include <iostream>
#include <fstream>
#include <string>

/* Why is this function here? There is some namespace weirdness with the << operator
   If T is inside a namespace (maybe a nested one), then in the spades namespace
   that operator can not be found. We have this function out in the global namespace
   so that it can find all of the appropriate operator<< in all other namespaces */
template <class T> void globalWriteToForLogger (std::ostream& os, const T & a)
{ os << a; }

namespace spades
{

  /**************************************************************************/

  /* The virtual base class for TagFunctions */
  class TagFunction
  {
  public:
    TagFunction () {}
    virtual ~ TagFunction () {}
    virtual std::string getTag (int level) const = 0;
    virtual std::string getErrorTag (int level) const = 0;
  };

  /* This is used if there is no other valid TagFunction available */
  class DefaultTagFunction : public TagFunction
  {
  public:
    DefaultTagFunction () {}
    ~DefaultTagFunction () {}
    std::string getTag (int level) const;
    std::string getErrorTag (int level) const;
  };
  
  /**************************************************************************/

  /* all log lines must end with ende */
  class LogManip_ende
  {
  public:
    LogManip_ende () {}
  };

  extern const LogManip_ende ende;


  class Logger
  {
  public:
    //Class definitions
    class ErrorLog
    {
    public:
      ErrorLog (): tag_next (true), next_level(-1)  {}
      ~ErrorLog () {}

      /* the levels in the error log should be used to reflect the severity of the error
	 0 is a critical error, and higher numbers are less severe. Exact details are up
	 to the user of this class. */
      void setNextLevel (int level) { next_level = level; }

      friend ErrorLog & operator << (ErrorLog & l, const LogManip_ende & ende);
      template < class T > friend ErrorLog & operator << (ErrorLog & l, const T & a);

      static const int ERROR_LOG_LEVEL;

    private:
      bool tag_next;
      int next_level;
    };

    class ActionLog
    {
    public:
      ActionLog () : tag_next(true), next_level(-1), max_level(-1) {}
      ~ActionLog () {}

      bool openFiles(std::string logfile_dir, std::string action_log_fn);
      
      int getMaxLevel() const { return max_level; }
      void setMaxLevel(int m) { max_level = m; }
      
      void setNextLevel (int level) { next_level = level; }

      friend ActionLog & operator << (ActionLog & l, const LogManip_ende & ende);
      template < class T > friend ActionLog & operator << (ActionLog & l, const T & a);

    private:
      bool tag_next;
      int next_level;

      int max_level;

      std::ofstream logfile;
    };
    
  public:

    ErrorLog & errorLogFunc (const int level)
    { elog.setNextLevel(level); return elog; }
    ActionLog & actionLogFunc (const int level)
    { alog.setNextLevel (level); return alog; }

    int getMaxActionLogLevel() const { return alog.getMaxLevel(); }
    void setMaxActionLogLevel(int l) { alog.setMaxLevel(l); }

    bool openFiles(std::string logfile_dir, std::string action_log_fn)
    { return alog.openFiles(logfile_dir, action_log_fn); }

    void setTagFunction(TagFunction* f) { if (pTag) delete pTag; pTag = f; }
    TagFunction* getTagFunction() const { return (pTag == NULL) ? &tagDefault : pTag; }

  protected:

    friend class ActionLog;
    friend class ErrorLog;

    
  private:
    TagFunction *pTag;
    static DefaultTagFunction tagDefault;

    ErrorLog elog;
    ActionLog alog;

  public:
    static Logger* instance();
    static void removeInstance();
    
  private:
    /* these are private so that the instance method is the only one that can allocate */
    Logger () : pTag(NULL), elog(), alog() { }
    ~Logger () { if (pTag) delete pTag; }

    static Logger* s_instance;
    
  };

  
  /*********** Shortcuts into this logger *************************/

#ifdef NO_ACTION_LOG
#define actionlog(x) if(1) ; else spades::Logger::instance()->actionLogFunc(x)
#else
#define actionlog(x) if(spades::Logger::instance()==NULL) std::cerr << "Called actionlog with null Logger"; else if(x>spades::Logger::instance()->getMaxActionLogLevel()) ; else spades::Logger::instance()->actionLogFunc(x)
#endif

  /* 0 is a critical error. all others should go to warning log */
#define errorlog if(spades::Logger::instance()==NULL) std::cerr << "Called errorlog with null Logger"; else spades::Logger::instance()->errorLogFunc(0)
#define warninglog(x) if(spades::Logger::instance()==NULL) std::cerr << "Called warningog with null Logger"; else spades::Logger::instance()->errorLogFunc(x)

  /*********** Implementations of templated function *************************/

  template <class T> Logger::ActionLog & operator << (Logger::ActionLog & l, const T & a)
  {
    if (l.next_level > l.max_level)
      return l;
    
    if (l.tag_next)
      {
	std::string tag (Logger::instance()->getTagFunction()->getTag (l.next_level));
	l.tag_next = false;
	if (l.logfile.is_open())
	  l.logfile << tag;
	else
	  std::cout << tag;
      }

    if (l.logfile.is_open())
      /* see note at top of file */
      globalWriteToForLogger(l.logfile, a);
    else
      /* see note at top of file */
      globalWriteToForLogger(std::cout, a);
    return l;
  }

  template <class T> Logger::ErrorLog & operator << (Logger::ErrorLog & l, const T & a)
  {
    if (l.tag_next)
      {
	std::string tag (Logger::instance()->getTagFunction()->getErrorTag (l.next_level));
	l.tag_next = false;
	std::cerr << tag;
	Logger::instance()->actionLogFunc (l.ERROR_LOG_LEVEL) << tag;
      }

    /* see note at top of file */
    globalWriteToForLogger(std::cerr, a);
    Logger::instance()->actionLogFunc (l.ERROR_LOG_LEVEL) << a;

    return l;
  }

} //spades namespace

#endif
