/*
Copyright (c) 2000-2002, Jelle Kok, University of Amsterdam
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this 
list of conditions and the following disclaimer. 

2. Redistributions in binary form must reproduce the above copyright notice, 
this list of conditions and the following disclaimer in the documentation 
and/or other materials provided with the distribution. 

3. Neither the name of the University of Amsterdam nor the names of its 
contributors may be used to endorse or promote products derived from this 
software without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/*! \file Geometry.h
<pre>
<b>File:</b>          Geometry.h
<b>Project:</b>       Robocup Soccer Simulation Team: UvA Trilearn
<b>Authors:</b>       Jelle Kok
<b>Created:</b>       12/02/2001
<b>Last Revision:</b> $ID$
<b>Contents:</b>      Header file for the classes VecPosition, Geometry, Line,
               Circle and Rectangle. All the member
               data and member method declarations for all these classes can be
               found in this file together with some auxiliary functions for
               numeric and goniometric purposes.
<hr size=2>
<h2><b>Changes</b></h2>
<b>Date</b>             <b>Author</b>          <b>Comment</b>
12/02/2001       Jelle Kok       Initial version created
09/06/2001       Remco de Boer   Version including full documentation completed
</pre>
*/

#ifndef _GEOMETRY_
#define _GEOMETRY_

#include "math.h"       // needed for M_PI constant
#include <iostream>
#include <string>       // needed for string
#include <vector>
#include "vector.h" //from clang lib

typedef double AngRad;  /*!< Type definition for angles in degrees. */
typedef double AngDeg;  /*!< Type definition for angles in radians. */

#define EPSILON 0.0001  /*!< Value used for floating point equality tests. */

// auxiliary numeric functions for determining the
// maximum and minimum of two given double values and the sign of a value
double max     ( double d1, double d2 );
double min     ( double d1, double d2 );
int    sign    ( double d1            );

// auxiliary goniometric functions which enable you to
// specify angles in degrees rather than in radians
AngDeg Rad2Deg ( AngRad x             );
AngRad Deg2Rad ( AngDeg x             );
double cosDeg  ( AngDeg x             );
double sinDeg  ( AngDeg x             );
double tanDeg  ( AngDeg x             );
AngDeg atanDeg ( double x             );
double atan2Deg( double x,  double y  );
AngDeg acosDeg ( double x             );
AngDeg asinDeg ( double x             );

// various goniometric functions
bool   isAngInInterval     ( AngDeg ang,    AngDeg angMin,    AngDeg angMax );
AngDeg getBisectorTwoAngles( AngDeg angMin, AngDeg angMax );

/*! CoordSystem is an enumeration of the different specified coordinate systems.
    The two possibilities are CARTESIAN or POLAR. These values are for instance
    used in the initializing a VecPosition. The CoordSystem indicates whether
    the supplied arguments represent the position in cartesian or in polar 
    coordinates. */
enum CoordSystemT {
  CARTESIAN,
  POLAR
};

/******************************************************************************/
/********************   CLASS VECPOSITION   ***********************************/
/******************************************************************************/

/*! This class contains an x- and y-coordinate of a position (x,y) as member
    data and methods which operate on this position. The standard arithmetic
      operators are overloaded and can thus be applied to positions (x,y). It is
      also possible to represent a position in polar coordinates (r,phi), since
      the class contains a method to convert these into cartesian coordinates
      (x,y). */
class VecPosition
{
  // private member data
private:

  double m_x;   /*!< x-coordinate of this position */
  double m_y;   /*!< y-coordinate of this position */

  // public methods
public:
  // constructor for VecPosition class
  VecPosition                               ( double            vx = 0,
                                              double            vy = 0,
                                              CoordSystemT      cs = CARTESIAN);
  VecPosition                               ( const rcss::geom::Vector2D& v);  
  
  // overloaded arithmetic operators
  VecPosition        operator -             (                                 ) const;
  VecPosition        operator +             ( const double      &d            ) const;
  VecPosition        operator +             ( const VecPosition &p            ) const;
  VecPosition        operator -             ( const double      &d            ) const;
  VecPosition        operator -             ( const VecPosition &p            ) const;
  VecPosition        operator *             ( const double      &d            ) const;
  VecPosition        operator *             ( const VecPosition &p            ) const;
  VecPosition        operator /             ( const double      &d            ) const;
  VecPosition        operator /             ( const VecPosition &p            ) const;
  void               operator =             ( const double      &d            );
  void               operator +=            ( const VecPosition &p            );
  void               operator +=            ( const double      &d            );
  void               operator -=            ( const VecPosition &p            );
  void               operator -=            ( const double      &d            );
  void               operator *=            ( const VecPosition &p            );
  void               operator *=            ( const double      &d            );
  void               operator /=            ( const VecPosition &p            );
  void               operator /=            ( const double      &d            );
  bool               operator !=            ( const VecPosition &p            );
  bool               operator !=            ( const double      &d            );
  bool               operator ==            ( const VecPosition &p            ) const;

  bool               operator ==            ( const double      &d            );

  //other tests
  bool               isnan                  () const;
  bool               isinf                  () const;
  
  // methods for producing output
  friend std::ostream&    operator <<       ( std::ostream           &os,
                                              const VecPosition&     p        );
  // method for reading back in
  friend std::istream&    operator >>       ( std::istream           &is,
                                              VecPosition&     p        );
  void               show                   ( CoordSystemT      cs = CARTESIAN);
  std::string        str                    ( CoordSystemT      cs = CARTESIAN);

  // set- and get methods for private member variables
  bool               setX                   ( double            dX            );
  double             getX                   (                           ) const;
  bool               setY                   ( double            dY            );
  double             getY                   (                           ) const;

  VecPosition flipCoords(bool flipX=true, bool flipY=true) const
  { return VecPosition(flipX ? -m_x : m_x, flipY ? -m_y : m_y); }
  VecPosition flipX() const { return flipCoords(true, false); }
  VecPosition flipY() const { return flipCoords(false, true); }

  // set- and get methods for derived position information
  void               setVecPosition         ( double            dX = 0,
                                              double            dY = 0,
                                              CoordSystemT      cs = CARTESIAN);
  double             getDistanceTo          ( const VecPosition p             ) const;
  double             getDistanceTo2         ( const VecPosition p             ) const;
  VecPosition        setMagnitude           ( double            d             );
  double             getMagnitude           (                           ) const;
  double             getMagnitude2          (                           ) const;
  AngDeg             getDirection           (                           ) const;

  //like setMagnitude, but does affect the current vector
  VecPosition        scaleTo                ( double            d             ) const
    { VecPosition v(*this); return v.setMagnitude(d); }

  // comparison methods for positions
  bool               isInFrontOf            ( const VecPosition &p            ) const;
  bool               isInFrontOf            ( const double      &d            ) const;
  bool               isBehindOf             ( const VecPosition &p            ) const;
  bool               isBehindOf             ( const double      &d            ) const;
  bool               isLeftOf               ( const VecPosition &p            ) const;
  bool               isLeftOf               ( const double      &d            ) const;
  bool               isRightOf              ( const VecPosition &p            ) const;
  bool               isRightOf              ( const double      &d            ) const;
  bool               isBetweenX             ( const VecPosition &p1,
                                              const VecPosition &p2           ) const;
  bool               isBetweenX             ( const double      &d1,
                                              const double      &d2           ) const;
  bool               isBetweenY             ( const VecPosition &p1,
                                              const VecPosition &p2           ) const;
  bool               isBetweenY             ( const double      &d1,
                                              const double      &d2           ) const;

  // conversion methods for positions
  VecPosition        normalize              (                                 ) const;
  VecPosition        rotate                 ( AngDeg            angle         ) const;
  VecPosition        rotate90               (                                 ) const
    { return VecPosition(m_y, -m_x); }
  VecPosition        rotateN90               (                                 ) const
    { return VecPosition(-m_y, m_x); }
  VecPosition        rotate180               (                                 ) const
    { return VecPosition(m_y, m_x); }

  VecPosition        globalToRelative       ( VecPosition       orig,
                                              AngDeg            ang           ) const;
  VecPosition        relativeToGlobal       ( VecPosition       orig,
                                              AngDeg            ang           ) const;
  VecPosition        getVecPositionOnLineFraction( VecPosition  &p,
                                              double            dFrac         ) const;

  //conversion methods to related types
  rcss::geom::Vector2D convertToVector2D() const
    { return rcss::geom::Vector2D(m_x, m_y); }
  
  // static class methods
  static VecPosition getVecPositionFromPolar( double            dMag,
                                              AngDeg            ang           );
  static AngDeg      normalizeAngle         ( AngDeg            angle         );
  static VecPosition getPointInBetween      ( const VecPosition& p1,
					      const VecPosition& p2,
					      double distFromP1               );
  static VecPosition getPointFracInBetween  ( const VecPosition& p1,
					      const VecPosition& p2,
					      double fracFromP1               );
  static float dotProduct(VecPosition v1, VecPosition v2)
    { return v1.m_x * v2.m_x + v1.m_y *v2.m_y; }

  static float crossProduct(VecPosition pt, VecPosition dir)
    { return pt.m_x * dir.m_y - pt.m_y * dir.m_x; }
  
};

/******************************************************************************/
/*********************   CLASS GEOMETRY   *************************************/
/******************************************************************************/

/*! This class contains several static methods dealing with geometry.*/
class Geometry
{

public:

  // geometric series
  static double getLengthGeomSeries(double dFirst,double dRatio,double dSum   );
  static double getSumGeomSeries   (double dFirst,double dRatio,double dLength);
  static double getSumInfGeomSeries(double dFirst,double dRatio               );
  static double getFirstGeomSeries (double dSum,  double dRatio,double dLength);
  static double getFirstInfGeomSeries(double dSum,double dRatio               );

  // abc formula
  static int    abcFormula(double a,double b, double c, double *s1, double *s2);
};

/******************************************************************************/
/********************** CLASS CIRCLE ******************************************/
/******************************************************************************/

/*!This class represents a circle. A circle is defined by one VecPosition
   (which denotes the center) and its radius. */
class Circle
{
    VecPosition m_posCenter;            /*!< Center of the circle  */
    double      m_dRadius;              /*!< Radius of the circle  */

public:
    Circle( );
    Circle( VecPosition pos, double dR );

    void        show                  ( std::ostream& os = std::cout ) const;
\
    // get and set methods
    bool        setCircle             ( VecPosition pos, 
                                        double      dR  );
    bool        setRadius             ( double dR       );
    double      getRadius             (                 ) const;
    bool        setCenter             ( VecPosition pos );
    VecPosition getCenter             (                 ) const;
    double      getCircumference      (                 ) const;
    double      getArea               (                 ) const;

    // calculate intersection points and area with other circle
    bool        isInside              ( VecPosition pos ) const;
    int         getIntersectionPoints ( Circle      c, 
                                        VecPosition *p1, 
                                        VecPosition *p2 ) const;
    double      getIntersectionArea   ( Circle c        ) const;

    friend      std::ostream& operator << (std::ostream & os, const Circle& c)
      { c.show(os); return os; }

}  ;

/******************************************************************************/
/*********************** CLASS LINE *******************************************/
/******************************************************************************/

/*!This class contains the representation of a line. A line is defined
   by the formula ay + bx + c = 0. The coefficients a, b and c are stored
   and used in the calculations. */
class Line
{
  // a line is defined by the formula: ay + bx + c = 0
  double m_a; /*!< This is the a coefficient in the line ay + bx + c = 0 */
  double m_b; /*!< This is the b coefficient in the line ay + bx + c = 0 */
  double m_c; /*!< This is the c coefficient in the line ay + bx + c = 0 */

public:
  Line( double a, double b, double c );
  Line();

  void setVertical( double x );
  void setHorizontal( double y );
  
  // print methods
  void        show( std::ostream& os = std::cout ) const;
  friend      std::ostream& operator << (std::ostream & os, Line l);

  // get intersection points with this line
  VecPosition getIntersection            ( Line        line                   ) const;

  int         getCircleIntersectionPoints( Circle      circle,
                                           VecPosition *posSolution1,
                                           VecPosition *posSolution2          ) const;

  Line        getTangentLine             ( VecPosition pos                    ) const;

  VecPosition getPointOnLineClosestTo    ( VecPosition pos                    ) const;

  double      getDistanceWithPoint       ( VecPosition pos                    ) const;

  bool        isInBetween                ( VecPosition pos,
                                           VecPosition point1,
                                           VecPosition point2                 );

  // calculate associated variables in the line
  double getYGivenX                      ( double      x ) const;
  double getXGivenY                      ( double      y ) const;
  double getACoefficient                 (               ) const;
  double getBCoefficient                 (               ) const;
  double getCCoefficient                 (               ) const;

  // static methods to make a line using an easier representation.
  static Line makeLineFromTwoPoints      ( VecPosition pos1,
                                           VecPosition pos2                   );
  static Line makeLineFromPositionAndAngle( VecPosition vec,
                                           AngDeg angle                       );

  /* Dongryeol Lee's method to retrieve the closest point in between two other
     points */
  static VecPosition getClosestPtInBetween(VecPosition pt,  VecPosition p0,
					   VecPosition p1);
  static double      getDistToClosestPtInBetween(VecPosition pt,  VecPosition p0,
						 VecPosition p1)
    { return pt.getDistanceTo(getClosestPtInBetween(pt, p0, p1)); }
};

/******************************************************************************/
/********************** CLASS RECTANGLE ***************************************/
/******************************************************************************/

/*!This class represents a rectangle. A rectangle is defined by two VecPositions
   the one at the upper left corner and the one at the right bottom. */
class Rectangle
{
  VecPosition m_posLeftTop;     /*!< top left position of the rectangle       */
  VecPosition m_posRightBottom; /*!< bottom right position of the rectangle   */

public:
  Rectangle                     ( );
  Rectangle                     ( VecPosition pos, VecPosition pos2 );

  void        show              ( std::ostream& os = std::cout      ) const;

  // checks whether point lies inside the rectangle
  bool        isInside          ( VecPosition pos                   ) const;

  // standard get and set methosd
  void        setRectanglePoints( VecPosition pos1,
                                  VecPosition pos2                  );
  bool        setPosLeftTop     ( VecPosition pos                   );
  VecPosition getPosLeftTop     (                                   ) const;
  bool        setPosRightBottom ( VecPosition pos                   );
  VecPosition getPosRightBottom (                                   ) const;

  VecPosition getPosRightTop() const
    { return VecPosition( m_posRightBottom.getX(), m_posLeftTop.getY() ); }
  VecPosition getPosLeftBottom() const
    { return VecPosition( m_posLeftTop.getX(), m_posRightBottom.getY() ); }

  VecPosition getCorner(int i) const;
  
  double      getWidth() const { return m_posRightBottom.getX() - m_posLeftTop.getX(); }
  double      getHeight() const { return m_posRightBottom.getY() - m_posLeftTop.getY(); }
  
  double      getArea           () const ;
  VecPosition getCenter         () const;

  Rectangle   shiftCenter       (VecPosition c                      ) const;

  double getLeft() const { return m_posLeftTop.getX(); }
  double getRight() const { return m_posRightBottom.getX(); }
  double getTop() const { return m_posLeftTop.getY(); }
  double getBottom() const { return m_posRightBottom.getY(); }
  
  void setLeft(double newval)   { setPosLeftTop(VecPosition(newval, getTop())); }
  void setRight(double newval)  { setPosRightBottom(VecPosition(newval, getBottom())); }
  void setTop(double newval)    { setPosLeftTop(VecPosition(getLeft(), newval)); }
  void setBottom(double newval) { setPosRightBottom(VecPosition(getRight(), newval)); }
  
  Line getLeftEdge() const {return Line::makeLineFromTwoPoints(getPosLeftTop(), getPosLeftBottom());}
  Line getRightEdge() const {return Line::makeLineFromTwoPoints(getPosRightTop(), getPosRightBottom());}
  Line getTopEdge() const {return Line::makeLineFromTwoPoints(getPosLeftTop(), getPosRightTop());}
  Line getBottomEdge() const {return Line::makeLineFromTwoPoints(getPosLeftBottom(), getPosRightBottom());}
  // idx is [0,3], for left, top, right, bottom
  Line getEdge(int idx) const;
  
  Line nearestHEdgeLine(const VecPosition& p) const;
  Line nearestVEdgeLine(const VecPosition& p) const;
  Line        getNearestEdgeLine(VecPosition p                      ) const;
  
  VecPosition adjustToWithin    (VecPosition p, double buffer = 0.0 ) const;
  VecPosition adjustToOutside   (VecPosition p, double buffer = 0.0 ) const;

  double      distanceToEdge    (VecPosition p                      ) const;
  Rectangle   shrink            (double amt                         ) const;
  Rectangle   expand            (double amt                         ) const
    { return shrink(-amt); }

  bool        doesIntersect     (const Rectangle& r                 ) const;

  // returns the number of solutions
  int lineIntersect(Line l, VecPosition* psol1, VecPosition* psol2) const;

  // could be up to 8 solutions
  std::vector<VecPosition> circleIntersect(const Circle& c) const;
  
  bool        isInBetween(const Rectangle& r1, const Rectangle& r2) const;

  VecPosition getRandomPtIn();

  VecPosition getClosestEdgePoint(const VecPosition& p) const 
    { return adjustToWithin(getNearestEdgeLine(p).getPointOnLineClosestTo(p)); }
  
  friend      std::ostream& operator << (std::ostream & os, const Rectangle& r);

};

/******************************************************************************/
/********************** CLASS TRIANGLE ***************************************/
/******************************************************************************/

/*!This class represents a triangle defined by three points */
class Triangle
{
  
 public:  

  /* This method takes three points on the cartesian plane and returns the
   area formed by the trianlge of three points (without taking the sign
   into account). Could return negative or negative. The user needs to
   take the absolute value if he/she wants the area */
  static double calculateSignedTriangleArea(VecPosition v0, VecPosition v1, VecPosition v2)
    {
      return (0.5 * (v0.getX() * v1.getY() + v1.getX() * v2.getY() +
		     v2.getX() * v0.getY() - v1.getX() * v0.getY() -
		     v2.getX() * v1.getY() - v0.getX() * v2.getY()));
    }
};

/******************************************************************************/
/********************** Other ************************************************/
/******************************************************************************/
double getDistToVerticalSegment(double x_val, double top_y, double bot_y, VecPosition pt);

#endif
