/* -*- Mode: C++ -*- */

#ifndef _FIELD_IMAGE_H_
#define _FIELD_IMAGE_H_

#include <vector>
#include <gd.h>
#include "Geometry.h"

class FieldImage 
{
 public:
  class Color
  {
  public:
    //Higher alpha is more transparent
    Color(int r, int g, int b, float a = 0.0) : r(r), g(g), b(b) { setAlpha(a); }
    Color(const char* rgb, float a = 0.0);
    Color() : r(0), g(0), b(0), alpha(0) {}
    ~Color() {}

    int getR() const { return r; }
    void setR(int new_r) { r = (unsigned char)new_r; }

    int getG() const { return g; }
    void setG(int new_g) { g = (unsigned char)new_g; }

    int getB() const { return b; }
    void setB(int new_b) { b = (unsigned char)new_b; }

    float getAlpha() const { return ((float)alpha) / ((float)ALPHA_MAX); }
    void setAlpha(float a) { alpha = (int)(ALPHA_MAX * a); }

  protected:
    friend class FieldImage;

    int getCIdxForImage(gdImagePtr& img) const { return gdImageColorResolveAlpha(img, r,g,b,alpha); }
    
  private:
    unsigned char r; //0 - 255	
    unsigned char g; //0 - 255
    unsigned char b; //0 - 255
    unsigned char alpha; // 0 --127

    static const unsigned char ALPHA_MAX = 127;
  };

  static const Color COLOR_BLACK;
  static const Color COLOR_WHITE;
  static const Color COLOR_RED;
  static const Color COLOR_GREEN;
  static const Color COLOR_BLUE;
  static const Color COLOR_CLEAR;
  static const Color COLOR_GREY;
  static const Color COLOR_DARKGREY;
  
 public:
  FieldImage(int legend_lines = 0, float scale_factor = 6.0);
  FieldImage(FieldImage& fi);
  ~FieldImage();
  FieldImage& operator=(const FieldImage& fi);

  // .png is appended
  bool writeTo(const char* fn, int compress_level = DEFAULT_COMPRESSION_LEVEL);
  void erase();

  void addFieldLines(const Color& c = COLOR_BLACK);

  void addGrid(double x_space, double y_space, const Color& c = COLOR_GREY);
  void addGrid(double space, const Color& c = COLOR_GREY)
  { addGrid(space, space, c); }
  
  void addPoint(VecPosition pt,
		const Color& c = COLOR_GREEN,
		float pscale = 1.0)
  { std::vector<VecPosition> v; v.push_back(pt); addPoints(v, c, pscale); }
  void addPoints(const std::vector<VecPosition>& vPts,
		 const Color& c = COLOR_GREEN,
		 float pscale = 1.0)
  { std::vector<VecPosition> vHalo; addPointsWithHalo(vPts, vHalo, c, COLOR_CLEAR, pscale); }
  //The "halo" are extra circles surrounding the points. This is useful
  //for showing variances 
  void addPointWithHalo(VecPosition pt, float haloRadX, float haloRadY,
			const Color& line_c = COLOR_GREEN,
			const Color& fill_c = COLOR_CLEAR,
			float pscale = 1.0);
  void addPointsWithHalo(const std::vector<VecPosition>& vPts,
			 const std::vector<VecPosition>& vHaloRad,
			 const Color& line_c = COLOR_GREEN,
			 const Color& fill_c = COLOR_CLEAR,
			 float pscale = 1.0);

  void addX(VecPosition pt, float xsize, float ysize, const Color& color = COLOR_GREEN, float lscale = 1.0);
  void addX(VecPosition pt, float size = 3.0, const Color& color = COLOR_GREEN, float lscale = 1.0)
  { addX(pt, size, size, color, lscale); }
  
  void addLine(VecPosition from, VecPosition to,
	       const Color& c = COLOR_RED, float lscale = 1.0)
  { addArrow(from, to, c, lscale, false); }
  void addLines(std::vector<std::pair<VecPosition,VecPosition> > vLines,
	       const Color& c = COLOR_RED, float lscale = 1.0)
  { addArrows(vLines, c, lscale, false); }
  void addArrow(VecPosition from, VecPosition to,
		const Color& c = COLOR_RED, float lscale = 1.0,
		bool with_head = true)
  { std::vector<std::pair<VecPosition,VecPosition> > v;
  v.push_back(std::pair<VecPosition,VecPosition>(from,to));
  addArrows(v, c, lscale, with_head); }
  void addArrows(std::vector<std::pair<VecPosition,VecPosition> > vLines,
		 const Color& c = COLOR_RED, float lscale = 1.0,
		 bool with_head = true);

#ifdef OLD_CODE	
#error Not converting this now
  void addPath(list<Magick::VPath> path,
	       Magick::Color c = Magick::Color("blue"), float lscale = 1.0);
#endif

  void addText(const char* str, VecPosition pt,
	       int point_size = 12,
	       const Color& textc = COLOR_BLACK);
  void addText(const char* str, VecPosition pt, const Color& textc)
  { addText(str, pt, 12, textc); }
  
  void addLegendLine(const char* str, const Color& textc = COLOR_BLACK);
  void addLegendLineWithDot(const char* str, const Color& dotc,
			    const Color& textc = COLOR_BLACK);
  int getLegendLinesLeft() { return num_legend_lines - curr_legend_line; }

  // The filling in is not quite right -- it fills inside of hte start rad
  void addArc(VecPosition origin,
	      double start_rad, double end_rad,
	      double start_ang, double span_ang,
	      const Color& border_c, const Color& fill_c);
  void addCircle(const Circle& c, const Color& border_c, const Color& fill_c)
  { return addArc(c.getCenter(), 0, c.getRadius(), 0, 360, border_c, fill_c); }
  
  void addPolygon(VecPosition aPts[], int num_pts,
		  const Color& border_c, const Color& fill_c);  

  void addRectangle(Rectangle r,
		    const Color& border_c, const Color& fill_c);  
  
  
  VecPosition pixelToSoccerSpace(int x, int y);
  gdPoint soccerSpaceToPixel(VecPosition v);
  
 private:

  //allocates the image object given the values of the other params
  // does not free any existing image!
  void initImage();

  void setThickness(float scale);
  
  void computeArrowHeadPts(VecPosition from, VecPosition to, float lscale, gdPointPtr pts);

  // puts the angle into the [0,360] degree range
  int normalizeImageAngle(double ang);
  
  float scale;
  int num_legend_lines;
  int curr_legend_line; //0 based
  gdImagePtr img;

  
  static const int font_size;
  static const gdFontPtr gd_font;
  static const int field_legend_buffer;
  static const int horiz_legend_buffer;

  static const int max_path_elements;
  static const int DEFAULT_COMPRESSION_LEVEL;
};


#endif
