/* -*- Mode: C++ -*- */

#include <strstream>
#include <gdfontmb.h>
#include "FieldImage.h"
#include "Logger.h"

using namespace spades;

/***********************************************************************************/
// some operators that gd doesn't define but should
gdPoint makeGdPoint(int x, int y)
{ gdPoint pt; pt.x = x; pt.y = y; return pt; }
gdPoint operator+(const gdPoint& pt1, const gdPoint& pt2)
{ return makeGdPoint(pt1.x + pt2.x, pt1.y + pt2.y); }
gdPoint operator-(const gdPoint& pt1, const gdPoint& pt2)
{ return makeGdPoint(pt1.x - pt2.x, pt1.y - pt2.y); }
gdPoint operator+=(gdPoint& pt1, const gdPoint& pt2)
{ pt1 = pt1 + pt2; return pt1; }
gdPoint operator-=(gdPoint& pt1, const gdPoint& pt2)
{ pt1 = pt1 - pt2; return pt1; }
std::ostream& operator<<(std::ostream& os, const gdPoint& pt)
{ os << "gdPoint(" << pt.x << ", " << pt.y << ")"; return os; } 
void myGdImageCopy(gdImagePtr dest, gdImagePtr src,
		   gdPoint dest_pos, gdPoint src_pos,
		   gdPoint size)
{ gdImageCopy(dest, src, dest_pos.x, dest_pos.y, src_pos.x, src_pos.y, size.x, size.y); }

/***********************************************************************************/
//whether to use freetype font
//#define USE_FREETYPE_FONTS

// SO we don't have to refer to the ServerParam struct
const float SP_pitch_length         = 105.0;
const float SP_pitch_width          = 68.0;
const float SP_pitch_margin         = 5.0;
const float SP_penalty_area_length  = 16.5;
const float SP_penalty_area_width   = 40.32;
const float SP_goal_area_length     = 5.5;
const float SP_goal_area_width      = 18.32;
const float SP_free_kick_buffer     = 9.15;

/*************************************** Color ********************************************/
FieldImage::Color::Color(const char* rgbstr, float a)
{
  setAlpha(a);

  if (strlen(rgbstr) != 7 || rgbstr[0] != '#')
    {
      errorlog << "Color: bad rgb string: " << rgbstr << ende;
    }

  char tmpstr[3];
  tmpstr[2] = 0;

  strncpy(tmpstr, rgbstr + 1, 2);
  r = strtol(tmpstr, NULL, 16);
  strncpy(tmpstr, rgbstr + 3, 2);
  g = strtol(tmpstr, NULL, 16);
  strncpy(tmpstr, rgbstr + 5, 2);
  b = strtol(tmpstr, NULL, 16);
}


/*************************************** FieldImage ***************************************/

const FieldImage::Color FieldImage::COLOR_BLACK(0, 0, 0);
const FieldImage::Color FieldImage::COLOR_WHITE(255, 255, 255);
const FieldImage::Color FieldImage::COLOR_RED(255, 0, 0);
const FieldImage::Color FieldImage::COLOR_GREEN(0, 255, 0);
const FieldImage::Color FieldImage::COLOR_BLUE(0, 0, 255);
const FieldImage::Color FieldImage::COLOR_CLEAR(0, 0, 0, 1.0);
const FieldImage::Color FieldImage::COLOR_GREY("#cbcbcb");
const FieldImage::Color FieldImage::COLOR_DARKGREY("#808080");

const int FieldImage::font_size = 12;
const gdFontPtr FieldImage::gd_font = gdFontMediumBold;
const int FieldImage::field_legend_buffer = 5;
const int FieldImage::horiz_legend_buffer = 4;
const int FieldImage::max_path_elements = 1000;
const int FieldImage::DEFAULT_COMPRESSION_LEVEL = 1;


FieldImage::FieldImage(int legend_lines, float scale_factor)
  : scale(scale_factor),
    num_legend_lines(legend_lines),
    curr_legend_line(0),
    img(NULL)
{
  initImage();
}

FieldImage::FieldImage(FieldImage& fi)
  : scale(fi.scale),
    num_legend_lines(fi.num_legend_lines),
    curr_legend_line(fi.curr_legend_line),
    img(NULL)
{
  initImage();
  gdImageCopy(img, fi.img, 0, 0, 0, 0, gdImageSX(fi.img), gdImageSY(fi.img));
}

FieldImage::~FieldImage()
{
  gdImageDestroy(img);
}

FieldImage&
FieldImage::operator=(const FieldImage& fi)
{
  scale = fi.scale;
  num_legend_lines = fi.num_legend_lines;
  curr_legend_line = fi.curr_legend_line;
  gdImageDestroy(img);
  initImage();
  gdImageCopy(img, fi.img, 0, 0, 0, 0, gdImageSX(fi.img), gdImageSY(fi.img));
  return *this;
}

void
FieldImage::initImage()
{
  int sx = (int)rint(SP_pitch_length * scale + 1);
  int sy = (int)rint((SP_pitch_width*scale) +
		     ((num_legend_lines > 0) ? field_legend_buffer : 0) +
		     num_legend_lines * font_size);
  img = gdImageCreateTrueColor(sx, sy);
  erase();
}


void
FieldImage::addFieldLines(const Color& c)
{
  int cidx = c.getCIdxForImage(img);
  gdPoint pt1, pt2, pt3, pt4;
  
  setThickness(2);
  gdImageSetAntiAliased(img, cidx);
  
  //main lines
  pt1 = soccerSpaceToPixel(VecPosition(-SP_pitch_length/2.0,-SP_pitch_width/2.0));
  pt2 = soccerSpaceToPixel(VecPosition( SP_pitch_length/2.0, SP_pitch_width/2.0));
  gdImageRectangle(img, pt1.x, pt1.y, pt2.x, pt2.y, cidx);
  pt1 = soccerSpaceToPixel(VecPosition(0, -SP_pitch_width/2.0));
  pt2 = soccerSpaceToPixel(VecPosition(0,  SP_pitch_width/2.0));
  gdImageLine(img, pt1.x, pt1.y, pt2.x, pt2.y, cidx);

  //penalty boxes
  pt1 = soccerSpaceToPixel(VecPosition(-SP_pitch_length/2.0,
				       -SP_penalty_area_width/2));
  pt2 = soccerSpaceToPixel(VecPosition(-SP_pitch_length/2.0 + SP_penalty_area_length,
				       SP_penalty_area_width/2));
  gdImageRectangle(img, pt1.x, pt1.y, pt2.x, pt2.y, cidx);
  pt1 = soccerSpaceToPixel(VecPosition( SP_pitch_length/2.0,
				       -SP_penalty_area_width/2));
  pt2 = soccerSpaceToPixel(VecPosition( SP_pitch_length/2.0 - SP_penalty_area_length,
				       SP_penalty_area_width/2));
  gdImageRectangle(img, pt1.x, pt1.y, pt2.x, pt2.y, cidx);

  //goalie boxes
  pt1 = soccerSpaceToPixel(VecPosition(-SP_pitch_length/2.0,
				       -SP_goal_area_width/2));
  pt2 = soccerSpaceToPixel(VecPosition(-SP_pitch_length/2.0 + SP_goal_area_length,
				       SP_goal_area_width/2));
  gdImageRectangle(img, pt1.x, pt1.y, pt2.x, pt2.y, cidx);

  pt1 = soccerSpaceToPixel(VecPosition( SP_pitch_length/2.0,
				       -SP_goal_area_width/2));
  pt2 = soccerSpaceToPixel(VecPosition( SP_pitch_length/2.0 - SP_goal_area_length,
				       SP_goal_area_width/2));
  gdImageRectangle(img, pt1.x, pt1.y, pt2.x, pt2.y, cidx);

  //center circle
  pt1 = soccerSpaceToPixel(VecPosition(0,0));
  gdImageArc(img,
	     pt1.x, pt1.y,
	     (int)rint(2*scale*SP_free_kick_buffer), (int)rint(2*scale*SP_free_kick_buffer),
	     0, 360,
	     cidx);
}

void
FieldImage::addGrid(double x_space, double y_space, const Color& c)
{
  for (double x = 0; x < SP_pitch_length/2.0; x += x_space)
    {
      addLine(VecPosition(x, -SP_pitch_width/2.0), VecPosition(x, SP_pitch_width/2.0), c);
      addLine(VecPosition(-x, -SP_pitch_width/2.0), VecPosition(-x, SP_pitch_width/2.0), c);
    }
  for (double y = 0; y < SP_pitch_width/2.0; y += y_space)
    {
      addLine(VecPosition(-SP_pitch_length/2.0, y), VecPosition(SP_pitch_length/2.0, y), c);
      addLine(VecPosition(-SP_pitch_length/2.0, -y), VecPosition(SP_pitch_length/2.0, -y), c);
    }
}


bool
FieldImage::writeTo(const char* fn, int compress_level)
{
  // Unfortunatley, the GD library uses the old C IO stuff
  std::ostrstream fullfn;
  fullfn << fn << ".png" << std::ends;
  FILE* out;
  out = fopen(fullfn.str(), "wb");
  if (!out)
    {
      errorlog << "FieldImage::writeImage: could not open out file: " << fullfn.str() << ende;
      return false;
    }
  gdImagePngEx(img, out, compress_level);
  fclose(out);
  fullfn.freeze(0);
  return true;
}

void
FieldImage::erase()
{
  int cidx_white = COLOR_WHITE.getCIdxForImage(img);
  gdImageFilledRectangle (img, 0, 0, gdImageSX (img),gdImageSY (img), cidx_white);
  curr_legend_line = 0;
}


void
FieldImage::addPointWithHalo(VecPosition pt,
			     float haloRadX, float haloRadY,
			     const Color& line_c,
			     const Color& fill_c,
			     float pscale)
{
  std::vector<VecPosition> vPts;
  vPts.push_back(pt);
  std::vector<VecPosition> vRad;
  vRad.push_back(VecPosition(haloRadX, haloRadY));
  addPointsWithHalo(vPts, vRad, line_c, fill_c, pscale);
}

void
FieldImage::addPointsWithHalo(const std::vector<VecPosition>& vPts,
			      const std::vector<VecPosition>& vHaloRad,
			      const Color& line_c,
			      const Color& fill_c,
			      float pscale)
{
  // how much to shrink the size of the point over the standard amount
  static const float POINT_SCALE_FACTOR = 0.5;
  
  int line_cidx = line_c.getCIdxForImage(img);
  int fill_cidx = fill_c.getCIdxForImage(img);
  gdPoint pt;
  
  std::vector<VecPosition>::const_iterator iPts;
  std::vector<VecPosition>::const_iterator iRad;

  setThickness(1.0);

  // draw the points
  // now draw the halos
  for(iPts = vPts.begin();
      iPts != vPts.end();
      ++iPts)
    {
      pt = soccerSpaceToPixel(*iPts);
      gdImageArc(img,
		 pt.x, pt.y,
		 (int)rint(pscale * scale * 2 * POINT_SCALE_FACTOR),
		 (int)rint(pscale * scale * 2 * POINT_SCALE_FACTOR),
		 0, 360,
		 line_cidx);
    }

  for(iRad = vHaloRad.begin(), iPts = vPts.begin();
      iRad != vHaloRad.end() && iPts != vPts.end();
      ++iRad, ++iPts)
    {
      pt = soccerSpaceToPixel(*iPts);
      gdImageArc(img,
		 pt.x, pt.y,
		 (int)rint(iRad->getX() * scale * 2), (int)rint(iRad->getY() * scale * 2),
		 0, 360,
		 line_cidx);
      gdImageFilledArc(img,
		       pt.x, pt.y,
		       (int)rint(iRad->getX() * scale * 2), (int)rint(iRad->getY() * scale * 2),
		       0, 360,
		       fill_cidx,
		       gdArc);
  }
}

void
FieldImage::addX(VecPosition pos, float xsize, float ysize, const Color& color, float lscale)
{
  int cidx = color.getCIdxForImage(img);
  setThickness(lscale);


  for (int flip = -1; flip <= 1; flip++)
    {
      VecPosition disp(flip * xsize / 2, ysize / 2);
      gdPoint pt1 = soccerSpaceToPixel(pos + disp);
      gdPoint pt2 = soccerSpaceToPixel(pos - disp);
      
      gdImageLine(img, pt1.x, pt1.y, pt2.x, pt2.y, cidx);
    }
}


void
FieldImage::computeArrowHeadPts(VecPosition from, VecPosition to, float lscale, gdPointPtr pts)
{
  float headlen = 2 * ((lscale - 1.0) / 2.0 + 1.0);
  float headwidang = 15;
  VecPosition pt;

  pts[0] = soccerSpaceToPixel(to);
  VecPosition disp = from - to;
  pt = to + disp.rotate(headwidang).scaleTo(headlen);
  pts[1] = soccerSpaceToPixel(pt);
  pt = to + disp.rotate(-headwidang).scaleTo(headlen);
  pts[2] = soccerSpaceToPixel(pt);
}

void
FieldImage::addArrows(std::vector<std::pair<VecPosition,VecPosition> > vLines,
		      const Color& c,
		      float lscale,
		      bool with_head)
{
  int cidx = c.getCIdxForImage(img);
  gdPoint pt1;
  gdPoint pt2;
  
  std::vector<std::pair<VecPosition,VecPosition> >::const_iterator iter;

  setThickness(lscale);

  for (iter = vLines.begin();
       iter != vLines.end();
       iter++)
    {
      gdPoint pt1 = soccerSpaceToPixel(iter->first);
      gdPoint pt2 = soccerSpaceToPixel(iter->second);
      gdImageLine(img, pt1.x, pt1.y, pt2.x, pt2.y, cidx);
      if (with_head)
	{
	  gdPoint head_pts[3];
	  computeArrowHeadPts(iter->first, iter->second, lscale, head_pts);
	  gdImageFilledPolygon(img, head_pts, 3, cidx);
	}
    }
}

#ifdef OLD_CODE
#error I have decided not to move over this path stuff right now. It is complicated, so if I need it, do it then

void FieldImage::addPath(list<Magick::VPath> path, Magick::Color c, float lscale)
{
  std::list<Magick::Drawable> drawList;
  drawList.push_back(DrawableStrokeColor(c)); // Outline color
  drawList.push_back(DrawableStrokeWidth(0.4*lscale)); // Stroke width
  drawList.push_back(DrawableTranslation(SP_pitch_length/2.0 * scale,
					 SP_pitch_width/2.0 * scale));
  drawList.push_back(DrawableScaling(scale, scale));
  try {
    while (!path.empty()) {
      list<Magick::VPath> l;
      list<Magick::VPath>::iterator lastpos = path.begin();
      for (int i=0; i<max_path_elements; i++) {
	if (lastpos == path.end())
	  break;
	++lastpos;
      }
      l.splice(l.begin(), path, path.begin(), lastpos);
      drawList.push_back(DrawablePath(l));
      img.draw(drawList);
      drawList.pop_back();
    }
  }
  catch( Exception &error_ ) {
    ostrstream s;
    s << error_.what() << ends;
    my_error("FieldImage::addPath error: %s", s.str());
    s.freeze(0);
  }      
}
#endif

void
FieldImage::addText(const char* str, VecPosition vorig, int point_size, const Color& textc) 
{
  gdPoint ptorig = soccerSpaceToPixel(vorig);
  int cidx = textc.getCIdxForImage(img);

#ifdef	USE_FREETYPE_FONTS
  int brect[8];
  char* err = gdImageStringFT(img, brect, cidx,
			      "/usr/share/fonts/default/TrueType/helb____.ttf", point_size, 0,
			      ptorig.x, ptorig.y, (char*)str);
  if (err != NULL)
    errorlog << "Error in drawing string with freetype: " << err << ende;
#else
  // this is a bit of an unsafe cast, but the gd library is a C thing, not a C++ one
  gdImageString(img, gd_font, ptorig.x, ptorig.y, (unsigned char*)str, cidx);
#endif	
}

  
void
FieldImage::addLegendLine(const char* str, const Color& textc)
{
  if (curr_legend_line >= num_legend_lines) {
    errorlog << "Trying to add too many legend lines! "
	     << curr_legend_line << " >= " << num_legend_lines
	     << ende;
    return;
  }

  if (str[0] == 0) {
    errorlog << "You tried to add a null string to the FieldImage (1)" << ende;
    return;
  }
  
  actionlog(100) << "FieldImage: adding legend line: '" << str << "'" << ende;

#ifdef USE_FREETYPE_FONTS	
  //we advance first because the point seems to be the baseline from where to draw for freetype fonts
  curr_legend_line++;
#endif
  
  int cidx = textc.getCIdxForImage(img);
  gdPoint ptorig;
  ptorig.x = (int)rint(horiz_legend_buffer * scale);
  ptorig.y = (int)rint(SP_pitch_width*scale) +
    field_legend_buffer + curr_legend_line * font_size;

#ifdef	USE_FREETYPE_FONTS
  int brect[8];
  char* err = gdImageStringFT(img, brect, cidx,
			      "/usr/share/fonts/default/TrueType/helb____.ttf", font_size, 0,
			      ptorig.x, ptorig.y, (char*)str);
  if (err != NULL)
    errorlog << "Error in drawing string with freetype: " << err << ende;
#else
  // this is a bit of an unsafe cast, but the gd library is a C thing, not a C++ one
  gdImageString(img, gd_font, ptorig.x, ptorig.y, (unsigned char*)str, cidx);
#endif	

#ifndef USE_FREETYPE_FONTS	
  //for the old font stuff, this is the right place to advance
  curr_legend_line++;
#endif
  

}

void FieldImage::addLegendLineWithDot(const char* str,
				      const Color& dotc,
				      const Color& textc)
{
  int dot_cidx = dotc.getCIdxForImage(img);
  float circle_y = (int)rint(SP_pitch_width*scale)+
    field_legend_buffer + curr_legend_line*font_size - font_size / 2.0;
  gdImageFilledArc(img,
		   (int)rint(horiz_legend_buffer*scale / 2), (int)rint(circle_y),
		   (int)rint(horiz_legend_buffer*scale*.75), (int)rint(font_size * .75),
		   0, 360,
		   dot_cidx,
		   gdArc);

  addLegendLine(str, textc);
}


void
FieldImage::addArc(VecPosition origin,
		   double start_rad, double end_rad,
		   double start_ang, double span_ang,
		   const Color& border_c, const Color& fill_c)
{
  setThickness(1.0);
  gdPoint pt_origin = soccerSpaceToPixel(origin);
  int i_start_rad = (int)rint(start_rad * scale);
  int i_end_rad   = (int)rint(end_rad * scale);
  int i_start_ang;
  int i_end_ang;
  int border_cidx = border_c.getCIdxForImage(img);
  int fill_cidx = fill_c.getCIdxForImage(img);

  if (span_ang >= 0)
    {
      i_start_ang = normalizeImageAngle(start_ang);
      i_end_ang   = normalizeImageAngle(start_ang + span_ang);
    }
  else
    {
      i_start_ang = normalizeImageAngle(start_ang + span_ang);
      i_end_ang   = normalizeImageAngle(start_ang);
    }

  // the normalization makes circles look like empty angles
  if (span_ang >= 360)
    i_end_ang += 360;

  // We do some wacky handling to avoid painting inside start_rad
  // first we copy the relevant section of image, then we'll use it
  // as a tile to paint
  gdImagePtr tile;
  if (i_start_rad > 0)
    {
      tile = gdImageCreate(i_start_rad * 2, i_start_rad * 2);
      // creating the tile is quit tricky!
      // the index into the tile seems to based on absolute point offsets
      // in the original image. Therefore, we build the tile in 4 segments
      // to account for that.
      //  The stars represent the section of the image we want to 
      //  grab
      //  img start
      //  v
      //  ***********************
      //  *    |                *
      //  *    |                *
      //  *----X----------------*
      //  *    |                *
      //  *    |                *
      //  *    |                *
      //  *    |                *
      //  *    |                *
      //  ***********************
      //                        ^
      //                        img_end
      //  
      //  Point X is where the tile will start drawing
      //  This is tile_point
      //  We'll grab the sections in left to right, top to bottom order
      //  1 2
      //  3 4
      gdPoint img_start = pt_origin - makeGdPoint(i_start_rad, i_start_rad);
      gdPoint img_end   = pt_origin + makeGdPoint(i_start_rad, i_start_rad);
      int tile_size = 2 * i_start_rad;
      gdPoint tile_point = makeGdPoint( (img_start.x / tile_size + 1) * tile_size,
					(img_start.y / tile_size + 1) * tile_size);
      actionlog(200) << "FieldImage::ardArc: inner correction, start="
		     << img_start << ", end=" << img_end
		     << ", tile_point=" << tile_point
		     << ende;
      // upper left (from orig) to lower right (in tile)
      myGdImageCopy(tile, img, img_end - tile_point, img_start, tile_point - img_start);
      // upper right (from orig) to lower left (in tile)
      myGdImageCopy(tile, img,
		    makeGdPoint(0, img_end.y - tile_point.y),
		    makeGdPoint(tile_point.x, img_start.y),
		    makeGdPoint(img_end.x - tile_point.x, tile_point.y - img_start.y));
      // lower left (from orig) to upper right (in tile)
      myGdImageCopy(tile, img,
		    makeGdPoint(img_end.x - tile_point.x ,0),
		    makeGdPoint(img_start.x, tile_point.y),
		    makeGdPoint(tile_point.x - img_start.x, img_end.y - tile_point.y));
      // lower right (from orig) to upper left (in tile)
      myGdImageCopy(tile, img, makeGdPoint(0,0), tile_point, img_end - tile_point);

      // Finally set this image as the tile to use
      gdImageSetTile(img, tile);
    }
  
  gdImageFilledArc(img,
		   pt_origin.x, pt_origin.y,
		   i_end_rad * 2, i_end_rad * 2,
		   i_start_ang, i_end_ang,
		   fill_cidx, gdArc);

  //Let's replace the orginal stuff
  if (i_start_rad > 0)
    {
      // we fill an ellipse even though only an arc shold have been overwritten
      // I trust the ellipse drawing more
      gdImageFilledEllipse(img,
			   pt_origin.x, pt_origin.y,
			   i_start_rad * 2, i_start_rad * 2,
			   gdTiled);
      gdImageDestroy(tile);
    }
  
  //we draw the border in 4 steps, the outer arc, the inner arc,
  // and then then two side lines (if this is not a fill circle)

  // the outer arc
  gdImageArc(img,
	     pt_origin.x, pt_origin.y,
	     i_end_rad * 2, i_end_rad * 2,
	     i_start_ang, i_end_ang,
	     border_cidx);

  // the inner arc
  if (i_start_rad > 0)
    gdImageArc(img,
	       pt_origin.x, pt_origin.y,
	       i_start_rad * 2, i_start_rad * 2,
	       i_start_ang, i_end_ang,
	       border_cidx);

  //the sides
  if (span_ang < 360)
    {
      gdPoint pt1, pt2;
      pt1 = soccerSpaceToPixel(origin + VecPosition::getVecPositionFromPolar(start_rad, start_ang));
      pt2 = soccerSpaceToPixel(origin + VecPosition::getVecPositionFromPolar(end_rad, start_ang));
      gdImageLine(img, pt1.x, pt1.y, pt2.x, pt2.y, border_cidx);
      pt1 = soccerSpaceToPixel(origin + VecPosition::getVecPositionFromPolar(start_rad, start_ang + span_ang));
      pt2 = soccerSpaceToPixel(origin + VecPosition::getVecPositionFromPolar(end_rad, start_ang + span_ang));
      gdImageLine(img, pt1.x, pt1.y, pt2.x, pt2.y, border_cidx);
    }
}

void
FieldImage::addPolygon(VecPosition aVecPts[], int num_pts,
		       const Color& border_c, const Color& fill_c)
{
  setThickness(1.0);

  gdPoint* aPts = new gdPoint[num_pts];
  for (int i=0; i<num_pts; i++)
    aPts[i] = soccerSpaceToPixel(aVecPts[i]);

  gdImageFilledPolygon(img, aPts, num_pts, fill_c.getCIdxForImage(img));
  gdImagePolygon(img, aPts, num_pts, border_c.getCIdxForImage(img));
  delete [] aPts;
}

void
FieldImage::addRectangle(Rectangle r,
			 const Color& border_c, const Color& fill_c)
{
  setThickness(1.0);

  gdPoint ptUpperLeft = soccerSpaceToPixel(r.getPosLeftTop());
  gdPoint ptLowerRight = soccerSpaceToPixel(r.getPosRightBottom());

  // We really shouldn't have to do this check, but drawing a filled rectangle
  // with an alpha of 0 seems to still shade a bit, so we'll turn it off here
  if (fill_c.getAlpha() != 1.0)
    gdImageFilledRectangle(img, ptUpperLeft.x, ptUpperLeft.y, ptLowerRight.x, ptLowerRight.y,
			   fill_c.getCIdxForImage(img));
  gdImageRectangle(img, ptUpperLeft.x, ptUpperLeft.y, ptLowerRight.x, ptLowerRight.y,
		   border_c.getCIdxForImage(img));
}


  
VecPosition
FieldImage::pixelToSoccerSpace(int x, int y)
{
  return (VecPosition(x,y) - VecPosition(SP_pitch_length/2.0 * scale, SP_pitch_width/2.0 * scale))/scale;
}

gdPoint
FieldImage::soccerSpaceToPixel(VecPosition v)
{
  gdPoint pt;

  pt.x = (int)rint((v.getX() + SP_pitch_length/2.0) * scale);
  pt.y = (int)rint((v.getY() + SP_pitch_width/2.0) * scale);

  return pt;
}

void
FieldImage::setThickness(float thick_scale)
{
  float t = thick_scale * scale / 3.0;
  if (t < 1.0)
    t = 1.0;
  gdImageSetThickness(img, (int) rint(t));
}

// puts the angle into the [0,360] degree range
int
FieldImage::normalizeImageAngle(double ang)
{
  return (int)rint(fmod(fmod(ang, 360) + 360, 360));
}

