/* Dongryeol Lee
 *
 * ClangIterator.h
 *
 * This is the iterator class for Clang objects (Region, Cond, Rule, Action,
 * Dir) that will iterate through a Clang object
 *
 */


#ifndef CLANG_ITERATOR
#define CLANG_ITERATOR


#include "region.h"
#include "cond.h"
#include "rule.h"
#include "clangaction.h"
#include "coach_lang_comp.h"
#include "ClangCallBack.h"
#include <iostream.h>
#include <stdlib.h>
#include <vector>


using namespace rcss::clang;


class ClangIterator
{
 public:

  void iterateOn(Region *reg);

  void iterateOn(Cond *cond);

  void iterateOn(Rule *rule);

  void iterateOn(Action *act);

  void iterateOn(Dir *dir);


 private:
 
  /* Region Visitor for sorting out Region types */
  class RegionVisitor: public Region::Visitor
    {

    private:
      
      ClangIterator *cIterator;
      
      ClangCallBack *func;

    public:

      RegionVisitor() {}

      RegionVisitor(ClangIterator *ci, ClangCallBack *f): 
	cIterator(ci), func(f) 
	{ }
      
      void visit(RegNull *reg)
	{
	  func->visitRegNull(reg);
	}

      void visit(RegQuad *reg)
	{
	  func->visitRegQuad(reg);
	}
      
      void visit(RegArc *reg)
	{
	  func->visitRegArc(reg);
	}
      
      void visit(RegUnion *reg)
	{
	  //func->visitRegUnion(reg);
	  
	  /* Get the list of all regions */
	  RegUnion::HasManyRegions::Storage allRegions = reg->getRegions();
	  
	  /* Iterate through the children and visit them in order */
	  for(RegUnion::HasManyRegions::Storage::iterator it = 
		allRegions.begin(); it != allRegions.end(); it++)
	    	    	    
	    cIterator->iterateOn(*it);

	  
	  func->visitRegUnion(reg);

	}
      
      void visit(RegNamed *reg)
	{
	  //func->visitRegNamed(reg);

	  Region *child = (Region *) reg->getRegion();


	  /* Visit the Region pointed by RegNamed */
	  if(child != NULL)
	    cIterator->iterateOn(child);


	  func->visitRegNamed(reg);
	}
      
      void visit(RegPoint *reg)
	{
	  func->visitRegPoint(reg);
	}
      
      void visit(RegTri *reg)
	{
	  func->visitRegTri(reg);
	}

      void visit(RegRec *reg)
	{
	  func->visitRegRec(reg);
	}

    };

  /* Cond Visitor for sorting out Cond types */
  class CondVisitor: public Cond::Visitor
    {

    private:
      
      ClangIterator *cIterator;
      
      ClangCallBack *func;
      
    public:
      
      CondVisitor() {}

      CondVisitor(ClangIterator *ci, ClangCallBack *f): 
	cIterator(ci), func(f) 
	{  }
      
      void visit(CondBool* cond )
	{
	  func->visitCondBool(cond);
	}
      
      void visit(CondAnd* cond )
	{
	  //func->visitCondAnd(cond);  
	  
	  CondAnd::HasManyConds::Storage conds = cond->getConds();
	  
	  for( CondAnd::HasManyConds::Storage::const_iterator iter = 
		 conds.begin(); iter != conds.end(); ++iter )
	    
	    cIterator->iterateOn(*iter);


	  func->visitCondAnd(cond);
	}
      
      void visit(CondOr* cond )
	{
	  //func->visitCondOr(cond);
	  
	  CondOr::HasManyConds::Storage conds = cond->getConds();
	  
	  for( CondOr::HasManyConds::Storage::const_iterator iter = 
		 conds.begin(); iter != conds.end(); ++iter )
	    
	    cIterator->iterateOn(*iter);


	  func->visitCondOr(cond);
	}
      
      void visit(CondNot* cond )
	{
	  //func->visitCondNot(cond);

	  cIterator->iterateOn( (Cond *) cond->getCond() );

	  func->visitCondNot(cond);
	}
      
      void visit(CondPlayerPos* cond )
	{
	  //func->visitCondPlayerPos(cond);
	  
	  cIterator->iterateOn((Region *) cond->getRegion() );  

	  func->visitCondPlayerPos(cond);
	}
      
      void visit(CondBallPos* cond )
	{
	  //func->visitCondBallPos(cond);
	  
	  cIterator->iterateOn((Region *) cond->getRegion() );

	  func->visitCondBallPos(cond);
	}
      
      void visit(CondBallOwner* cond )
	{
	  func->visitCondBallOwner(cond);	  
	}
      
      void visit(CondPlayMode* cond )
	{
	  func->visitCondPlayMode(cond);
	}
      
      void visit(CondNamed* cond )
	{
	  //func->visitCondNamed(cond);


	  Cond *child = (Cond *) cond->getCond();


	  /* Visit the Cond object referred by the CondNamed */
	  if(child != NULL)
	    cIterator->iterateOn(child);

	  func->visitCondNamed(cond);
	}
      
      void visit(CondTime* cond )
	{
	  func->visitCondTime(cond);
	} 
      
      void visit(CondOppGoal* cond )
	{
	  func->visitCondOppGoal(cond);
	}
      
      void visit(CondOurGoal* cond )
	{
	  func->visitCondOurGoal(cond);
	}
      
      void visit(CondGoalDiff* cond )
	{
	  func->visitCondGoalDiff(cond);
	}
      
      void visit(CondUNum* cond )
	{
	  func->visitCondUNum(cond);
	}

    };
  
  /* Rule Visitor for sorting out Rule types */
  class RuleVisitor: public Rule::Visitor
    {

    private:
      
      ClangIterator *cIterator;
      
      ClangCallBack *func;

    public:

      RuleVisitor() {}

      RuleVisitor(ClangIterator *ci, ClangCallBack *f): 
	cIterator(ci), func(f) 
	{ }
      
      void visit(CondRule *rule)
	{
	  //func->visitCondRule(rule);

	  cIterator->iterateOn( (Cond*) rule->getCond() );

	  func->visitCondRule(rule);
	}

      void visit(SimpleRule *rule)
	{ 
	  /* First, visit the rest of directives */
	  //func->visitSimpleRule(rule);

	  SimpleRule::HasManyDirs::Storage dirs = rule->getDirs();

	  for( SimpleRule::HasManyDirs::Storage::iterator i = dirs.begin(); 
	       i != dirs.end(); i++ )
	    
	    cIterator->iterateOn(*i);
	  

	  /* Then visit the CondRule */
	  cIterator->iterateOn( (Cond *) rule->getCond() );

	  
	  func->visitSimpleRule(rule);
	}
      
      void visit(NestedRule *rule)
	{	
	  /* First, visit the rest of the rules */
	  //func->visitNestedRule(rule);

	  NestedRule::HasManyRules::Storage rules = rule->getRules();
	  
	  for( NestedRule::HasManyRules::Storage::iterator i = rules.begin(); 
	       i != rules.end(); i++ )
	    
	    cIterator->iterateOn(*i);
	  
	  /* Then visit the Cond */
	  cIterator->iterateOn( (Cond *) rule->getCond() );
	  

	  func->visitNestedRule(rule);
	}
      
      void visit(IDListRule *rule)
	{
	  //func->visitIDListRule(rule);  
	  
	  /* Then, get its size */
	  int size = rule->getRuleList().size();

	  /* For each Rule pointer stored in IDListRule, iterate */
	  for( int i = 0; i < size; i++)
	    cIterator->iterateOn(rule->getRuleList()[i]);
	  
	  func->visitIDListRule(rule);

	}

    };

  /* Action Visitor for sorting out Action types */
  class ActionVisitor: public Action::Visitor
    {

    private:
      
      ClangIterator *cIterator;
      
      ClangCallBack *func;
            
    public:
      
      ActionVisitor() {}

      ActionVisitor(ClangIterator *ci, ClangCallBack *f): 
	cIterator(ci), func(f) 
	{  }
      
      void visit(ActPos *act)
	{
	  //func->visitActPos(act);

	  cIterator->iterateOn( (Region *) act->getRegion() );

	  func->visitActPos(act);
	}
      
      void visit(ActHome *act)
	{
	  //func->visitActHome(act);

	  cIterator->iterateOn( (Region *) act->getRegion() );

	  func->visitActHome(act);
	}
      
      void visit(ActBallToReg *act)
	{
	  //func->visitActBallToReg(act);

	  cIterator->iterateOn( (Region *) act->getRegion() );

	  func->visitActBallToReg(act);
	}
      
      void visit(ActBallToPlayer *act)
	{
	  func->visitActBallToPlayer(act);
	}
      
      void visit(ActMark *act)
	{
	  func->visitActMark(act);
	}
      
      void visit(ActMarkLinePlayer *act)
	{
	  func->visitActMarkLinePlayer(act);
	}
      
      void visit(ActMarkLineReg *act)
	{
	  //func->visitActMarkLineReg(act);

	  cIterator->iterateOn( (Region *) act->getRegion() );

	  func->visitActMarkLineReg(act);
	}
      
      void visit(ActOffsidesLine *act)
	{
	  //func->visitActOffsidesLine(act);

	  cIterator->iterateOn( (Region *) act->getRegion() );


	  func->visitActOffsidesLine(act);
	}
      
      void visit(ActHetType *act)
	{
	  func->visitActHetType(act);
	}
      
      void visit(ActNamed *act)
	{
	  //func->visitActNamed(act);


	  Action *child = (Action *) act->getAction();


	  /* Then, iterate on the Action object pointed by the ActNamed */
	  if(child != NULL)
	    cIterator->iterateOn(child);


	  func->visitActNamed(act);
	}
      
      void visit(ActPassReg *act)
	{
	  //func->visitActPassReg(act);

	  cIterator->iterateOn( (Region *) act->getRegion() );

	  func->visitActPassReg(act);
	}
      
      void visit(ActPassUNum *act)
	{
	  func->visitActPassUNum(act);
	}
      
      void visit(ActDribble *act)
	{
	  //func->visitActDribble(act);

	  cIterator->iterateOn( (Region *) act->getRegion() );

	  func->visitActDribble(act);
	}
      
      void visit(ActClear *act)
	{
	  //func->visitActClear(act);

	  cIterator->iterateOn( (Region *) act->getRegion() );

	  func->visitActClear(act);
	}
      
      void visit(ActShoot *act)
	{
	  func->visitActShoot(act);
	}
      
      void visit(ActHold *act)
	{
	  func->visitActHold(act);
	}
      
      void visit(ActIntercept *act)
	{
	  func->visitActIntercept(act);
	}
      
      void visit(ActTackle *act)
	{
	  func->visitActTackle(act);
	}

    };

  /* Dir Visitor for sorting our Dir types */
  class DirVisitor: public Dir::Visitor
    {

    private:
      
      ClangIterator *cIterator;
      
      ClangCallBack *func;
      
    public:
      
      DirVisitor() {}

      DirVisitor(ClangIterator *ci, ClangCallBack *f): 
	cIterator(ci), func(f) 
	{  }
      
      void visit(DirComm* dir )
	{
	  //func->visitDirComm(dir);      

	  DirComm::HasManyActions::Storage actions = dir->getActions();

	  for( DirComm::HasManyActions::Storage::iterator i = actions.begin();
	       i != actions.end(); i++)
	    
	    cIterator->iterateOn( *i );


	  func->visitDirComm(dir);
	}
      
      void visit(DirNamed* dir )
	{
	  //func->visitDirNamed(dir);


	  Dir *child = (Dir *) dir->getDir();

	  /* Then, iterate on the Dir objet pointed by DirNamed */
	  cIterator->iterateOn(child);

	  func->visitDirNamed(dir);
	}

    };
    
 public:

  /* RegionVisitor */
  RegionVisitor regvisitor;
  
  /* CondVisitor */
  CondVisitor cvisitor;
  
  /* RuleVisitor */
  RuleVisitor rulevisitor;

  /* ActionVisitor */
  ActionVisitor avisitor;

  /* DirVisitor */
  DirVisitor dvisitor;
  
  /* Constructors */
  ClangIterator(ClangCallBack *ccb): 
    regvisitor(this, ccb), cvisitor(this, ccb), rulevisitor(this, ccb),
    avisitor(this, ccb), dvisitor(this, ccb)
    { }

};


#endif
