#!/usr/local/bin/perl5 -w

# This script reads the combined matched files of the player (a specification or which
#  rules matched), the abmdp_track.log file (from the coach), and the tree2state.log file
#  (which specifies how to map rule names to abstract states (also generated by the coach)
# to print out some statistics about how these things line up

#test_count_bits(2, 4, 8, 16, 24, 11, 124, 255, 562, 2134); die;
#test_count_bits(qw(2 4 8 16 24 11 124 255 562 2134)); die;
#test_count_bits((oct("0b1110000"), oct("0b10000100"), oct("0b10"), oct("0b10000100"))); die;


$usage = "$0 summary|cycle|diff <tree2state logfiles> <abmdp_track logfile> <combine-matched logfile>";

$cmdline = join ' ', @ARGV;
$pwd = `pwd`;
chomp $pwd;
$modestr = shift @ARGV;
$tree2state_fn = shift@ARGV;
$abmdp_track_fn = shift @ARGV;
$combine_matched_fn = shift @ARGV;

die $usage unless (defined($modestr) &&
		   defined($tree2state_fn) &&
		   defined($abmdp_track_fn) &&
		   defined($combine_matched_fn));

#contants
$MODE_SUMMARY = 0;
$MODE_CYCLE = 1;
$MODE_DIFF = 2;

$modestr = lc $modestr;
if ($modestr eq "summary") {
  $mode = $MODE_SUMMARY;
} elsif ($modestr eq "cycle") {
  $mode = $MODE_CYCLE;
} elsif ($modestr eq "diff") {
  $mode = $MODE_DIFF;
} else {
  die $usage;
}

use Statistics::Descriptive;

$debug = 0;

# First, let's read the tree2state file and get a mapping of state idx to rule name
# This could be a many-to-one mapping
# Goes into %state2rule and %rule2state

open (T2SFH, "<$tree2state_fn") || die "Could not open tree2state file '$tree2state_fn': $!";
while (<T2SFH>)
  {
    chomp;
    s/\#.*//;
    next if (/^\s*$/);

    ($rule, @indices) = split;
    foreach $idx (@indices)
      {
	$state2rule{$idx} = $rule;
	$rule2state{$rule} = [@indices];
      }
  }
close (T2SFH);
#print ((join "\n", (%state2rule))."\n");

# Now read the coach abmdp_track file and store the indices
# Goes into @coachindices
open (ABMDP, "<$abmdp_track_fn") || die "Could not open abmdp file '$abmdp_track_fn': $!";
while (<ABMDP>)
  {
    chomp;
    s/\#.*//;
    next if (/^\s*$/);

    ($time, $state) = split;
    $coachindices[$time] = $state;
  }
close (ABMDP);
#print ((join "\n", (@coachindices))."\n");

SWITCH: for ($mode) {
  ($_ == $MODE_SUMMARY) && print "# This file was generated by the summary mode of $0\n";
  ($_ == $MODE_CYCLE) && print "# This file was generated by the per cycles mode of $0\n";
  ($_ == $MODE_DIFF) && print "# This file was generated by the state diff mode of $0\n";
}
print <<EOH;
# Command line: $0 $cmdline
# Working dir: $pwd
EOH

# now initilize all the values we will be tracking
$stat_nump_per_cycle = Statistics::Descriptive::Sparse->new();
$stat_nump_per_cycle_no_zero = Statistics::Descriptive::Sparse->new();
$cnt_coach_only = 0;
$stat_nump_player_only_no_advice = Statistics::Descriptive::Sparse->new();
$stat_nump_player_only_no_state = Statistics::Descriptive::Sparse->new();
$cnt_agree_none = 0;
$stat_nump_full_agree = Statistics::Descriptive::Sparse->new();
$stat_nump_no_agree = Statistics::Descriptive::Sparse->new();
$state_nump_partial_agree_yes = Statistics::Descriptive::Sparse->new();
$state_nump_partial_agree_no = Statistics::Descriptive::Sparse->new();

open (MATCH, "<$combine_matched_fn") || die "Could not open combine matched file '$combine_matched_fn': $!";
while (<MATCH>)
  {
    chomp;
    s/\#.*//;
    next if (/^\s*$/);

    /^(\d+)\s*:\s*(.*)$/ || die "Malformed line in combine matched file: '$_'";
    $time = $1;
    @player_vals = split /\s+/, $2;
    @orig_player_vals = @player_vals;
    $idx = 0;
    @orig_player_rules = grep {$idx++ % 2 == 0} @orig_player_vals;
    $idx = 0;
    @orig_player_masks = grep {$idx++ % 2 == 1} @orig_player_vals;
    print "$time: " . (join " ", @player_vals) . "\n" if ($debug);

    # We have to be a little careful here because the coach can identify a state
    # that has no matching rule. For now, we'll just consider this a no value
    $coach_has_state_val = defined($coachindices[$time]);
    $coach_has_advice_val = ($coach_has_state_val && exists($state2rule{$coachindices[$time]}));
    $coach_rule = $coach_has_advice_val ? $state2rule{$coachindices[$time]} : "";
    print "Coach: $coach_has_advice_val $coach_rule\n" if ($debug);
    $player_match_cnt = 0;
    $player_unmatch_cnt = 0;
    $player_total = 0;
    for (;;)
      {
	$rule = shift @player_vals;
	$numfield = shift @player_vals;
	last if (not defined($numfield));
	#numfield is a string represnting a number in binary
	$cnt = count_bits(oct("0b$numfield"));

	print ("\t\tProcessing ($rule, $numfield, $cnt)\n") if ($debug);
	$player_total += $cnt;
	if ($coach_has_advice_val && $coach_rule eq $rule)
	  {
	    $player_match_cnt += $cnt;
	    print ("\t\t\tMatch now $player_match_cnt\n") if ($debug);
	  }
	else
	  {
	    $player_unmatch_cnt += $cnt;
	    print ("\t\t\tUnmatch now $player_unmatch_cnt\n") if ($debug);
	  }
      }

    $stat_nump_per_cycle->add_data($player_total);
    $stat_nump_per_cycle_no_zero->add_data($player_total) if ($player_total > 0);

    if ($coach_has_advice_val)
      {
	if ($player_match_cnt == 0 && $player_unmatch_cnt == 0)
	  {
	    print "\tcoach only\n" if ($debug);
	    $cnt_coach_only++;
	    print "$time: CoachOnly $coach_rule\n" if ($mode == $MODE_CYCLE);
	    print "$time: CoachOnly $coachindices[$time]\n" if ($mode == $MODE_DIFF);
	  }
	elsif ($player_match_cnt == 0)
	  {
	    print "\tno agree\n" if ($debug);
	    $stat_nump_no_agree->add_data($player_unmatch_cnt);
	    print "$time: NoAgree $coach_rule\t" . (join ' ', @orig_player_vals)."\n" if ($mode == $MODE_CYCLE);
	    print "$time: NoAgree $coachindices[$time] " .
	      (join ' ', (map { (@{$rule2state{$_}}) } @orig_player_rules)) .
		"\n" if ($mode == $MODE_DIFF);
	  }
	elsif ($player_unmatch_cnt == 0)
	  {
	    print "\tfull agree\n" if ($debug);
	    $stat_nump_full_agree->add_data($player_match_cnt);
	    print "$time: FullAgree $coach_rule\n" if ($mode == $MODE_CYCLE);
	    print "$time: FullAgree $coachindices[$time] \n" if ($mode == $MODE_DIFF);
	  }
	else
	  {
	    print "\tpartial agree\n" if ($debug);
	    $state_nump_partial_agree_yes->add_data($player_match_cnt);
	    $state_nump_partial_agree_no->add_data($player_unmatch_cnt);
	    print "$time: PartialAgree $coach_rule\t" . (join ' ', @orig_player_vals)."\n" if ($mode == $MODE_CYCLE);
	    print "$time: PartialAgree $coachindices[$time] " .
	      (join ' ', (map { (@{$rule2state{$_}}) } @orig_player_rules)) .
		"\n" if ($mode == $MODE_DIFF);
	  }
      }
    else
      {
	if ($player_match_cnt == 0 && $player_unmatch_cnt == 0)
	  {
	    print "\tagreed none\n" if ($debug);
	    $cnt_agree_none++;
	    print "$time: AgreeNone\n" if ($mode == $MODE_CYCLE);
	    print "$time: AgreeNone\n" if ($mode == $MODE_DIFF);
	  }
	elsif ($coach_has_state_val)
	  {
	    # this is the case where the coach thinks we are in an abstract state
	    # but has not advised about it
	    print "\tplayer only (no advice)\n" if ($debug);
	    $stat_nump_player_only_no_advice->add_data($player_match_cnt + $player_unmatch_cnt);
	    print "$time: PlayerOnlyNoAdvice\t" . (join ' ', @orig_player_vals)."\n" if ($mode == $MODE_CYCLE);
	    print "$time: PlayerOnlyNoAdvice $coachindices[$time] " . (join ' ', (map { (@{$rule2state{$_}}) } @orig_player_rules))."\n" if ($mode == $MODE_DIFF);
	  }
	else
	  {
	    print "\tplayer only (no state)\n" if ($debug);
	    $stat_nump_player_only_no_state->add_data($player_match_cnt + $player_unmatch_cnt);
	    print "$time: PlayerOnlyNoState\t" . (join ' ', @orig_player_vals)."\n" if ($mode == $MODE_CYCLE);
	    print "$time: PlayerOnlyNoState\t" . (join ' ', (map { (@{$rule2state{$_}}) } @orig_player_rules))."\n" if ($mode == $MODE_DIFF);
	  }
      }
  }
close(MATCH);

if ($mode == $MODE_SUMMARY)
  {
    print "Total cycles: " . ($stat_nump_per_cycle->count()) . "\n";
    print "Players contrib per cycle: ";
    print_stat($stat_nump_per_cycle);
    print "\n";
    print "Players contrib per cycle (when >0): ";
    print_stat($stat_nump_per_cycle_no_zero);
    print "\n";
    print "Cycles where agree not in absstate: $cnt_agree_none\n";
    print "Cycles where only coach in absstate: $cnt_coach_only\n";
    print "Cycles/nump stat where only player has advice (coach has state): " . ($stat_nump_player_only_no_advice->count());
    print_stat($stat_nump_player_only_no_advice);
    print "\n";
    print "Cycles/nump stat where only player in absstate: " . ($stat_nump_player_only_no_state->count());
    print_stat($stat_nump_player_only_no_state);
    print "\n";
    print "Cycles/nump stat when no agreement: " . ($stat_nump_no_agree->count());
    print_stat($stat_nump_no_agree);
    print "\n";
    print "Cycles/nump agree/nump disagree when partial agreement: " .
      ($state_nump_partial_agree_yes->count() + $state_nump_partial_agree_no->count());
    print_stat($state_nump_partial_agree_yes);
    print_stat($state_nump_partial_agree_no);
    print "\n";
    print "Cycles/nump stat when full agreement: " . ($stat_nump_full_agree->count());
    print_stat($stat_nump_full_agree);
    print "\n";
  }

# counts the number of bits in the number which is argument 1
sub count_bits
  {
    my ($num) = @_;
    my ($cnt) = (0);

    while ($num != 0)
      {
	$cnt += $num & 1;
	$num >>= 1;
      }
    return $cnt;
  }

sub test_count_bits
  {
    my ($val);
    foreach $val (@_)
      {
	printf ("test_count_bits: %b has %d\n", $val, count_bits($val));
      }
  }
# prints info from a stat structure in a format I like
sub print_stat
  {
    my ($stat) = @_;
    if ($stat->count() > 0)
      {
	printf(" %.4f", $stat->mean());
	if ($stat->count() > 1)
	  {
	    printf(" %.4f", $stat->standard_deviation());
	  }
	else
	  {
	    print(" XXXXX")
	  }
      }
    else
      {
	print(" XXXXX XXXXX")
      }
  }
