
#ifndef _PRECOND_H_
#define _PRECOND_H_

//#include <iostream>
#include <list>
#include <string>
#include <set>
#include <map>
#include <cassert>

#include <xercesc/dom/deprecated/DOM.hpp>

typedef int HostID;
typedef unsigned int ServiceID;
typedef unsigned int AttackID;
typedef unsigned int PortID;
typedef unsigned int FirewallID;

// HostID 'GENERAL' means the attack in question is a general attack,
// with no specific target
#define GENERAL -1

using namespace std;

ostream& operator<< (ostream& target, const DOMString& s);
struct less_str : public binary_function <DOMString, DOMString, bool> 
{
  bool operator()( const DOMString &x, const DOMString &y) const { return x.compareString(y) < 0; }
};

typedef set <DOMString, less_str> DOMStringSet;
typedef map <DOMString, DOMString, less_str> DOMStringMap;

typedef map<DOMString, DOMStringMap, less_str> HostObjectMap;
typedef map<DOMString, DOMStringSet, less_str> HostObjectAttrNameMap;
typedef map< DOMString, PortID, less_str > NamePortIdMap;
typedef set<HostID> HostIdSet;

class ModelMemUsage
{
  static const long NodeImplSize;
  static const long ChildImplSize;
  static const long ParentImplSize;
  static const long ElementImplSize;
  static const long CharDataImplSize;
  static const long AttrImplSize;
  static const long EntityImplSize;
  static const long NotationImplSize;
  static const long DocImplSize;
  static const long DocTypeImplSize;

public:
  virtual ~ModelMemUsage() {}
  virtual long GetMemSize() const = 0;
    
  static long GetDOMStringSizeEx( const DOMString &str ) {
    return 2 * sizeof(int) + str.length() * sizeof(XMLCh);
  }
  static long GetDOMStringSize( const DOMString &str ) {
    return 0; //all strings are calculated in GetDOM_DocumentSize();
  }
  static long GetDOMStringSetSize( const DOMStringSet &str_set );
  static long GetDOMStringMapSize( const DOMStringMap &str_map );
  static long GetDOM_DocumentSize( DOM_Node *node );
  
private:
  static long GetDOM_NodeSize( DOM_Node *node, DOMStringSet &used_str );
};

struct FirewallRule {
  int srchost;
  int dsthost;
  int srcport;
  int dstport;
    
  bool operator == ( const FirewallRule &rule ) const {
    return (rule.srchost == srchost &&
            rule.dsthost == dsthost &&
            rule.dstport == dstport &&
            rule.srcport == srcport); 
  }
  
  bool less ( const FirewallRule &rule ) const {
    bool res = (srchost < rule.srchost);
    
    if ( !res && srchost == rule.srchost)
    {
      res = (dsthost < rule.dsthost);
      if ( !res && dsthost == rule.dsthost )
        if ( !(res = (srcport < rule.srcport)) && srcport == rule.srcport )
          res = (dstport < rule.dstport);
    }
    
    return res;
  }
  
  // returns 1 if 'this' is equal to 'rule'
  // returns -1 if 'rule' includes 'this'
  // returns 2 if 'this' include 'rule'
  // return 0 otherwise
  int MatchWildacrdRule( const FirewallRule &rule ) const {
    int res = 0;
    int srchostres = 0;
    int srcportres = 0;
    int dsthostres = 0;
    int dstportres = 0;
    
    if ( rule.srchost == srchost )
      srchostres = 1;
    else if ( rule.srchost == -1 )
      srchostres = -1;
    else if ( srchost == -1 )
      srchostres = 2;
    
    if ( rule.srcport == srcport )
      srcportres = 1;
    else if ( rule.srcport == -1 )
      srcportres = -1;
    else if ( srcport == -1 )
      srcportres = 2;
    
    if ( rule.dsthost == dsthost )
      dsthostres = 1;
    else if ( rule.dsthost == -1 )
      dsthostres = -1;
    else if ( dsthost == -1 )
      dsthostres = 2;
    
    if ( rule.dstport == dstport )
      dstportres = 1;
    else if ( rule.dstport == -1 )
      dstportres = -1;
    else if ( dstport == -1 )
      dstportres = 2;
    
    if ( srchostres == srcportres && srchostres == dstportres &&
         srchostres == dsthostres)
      res = srchostres;
    else if ( abs(srchostres) == 1 && abs(srcportres) == 1 && 
              abs(dsthostres) == 1 && abs(dstportres) == 1 )
      res = -1;
    else if ( srchostres > 0 && srcportres > 0 && dsthostres > 0 && dstportres > 0 )
      res = 2;
    
    return res;    
  }
  
  static string GetPortConstantName( const DOMString &port ) {
    return string( "PORT_" ) + string( port.transcode() );
  }
  
};

class Precondition: public ModelMemUsage {

public:
  enum PreconditionType {
    PRIVILEGE,
    SERVICE_PORT_AGREEMENT,
    VULNERABILITY,
    SERVICE,
    CONNECTIVITY,
    TRUST,
    NOTRUST,
    KNOWLEDGE,
    OBJECT,
    SENTOBJECT,
    FIREWALLRULE,
    FIREWALLREACHABLE,
    AND,
    OR,
    NOT,
    UNDEFINED
  };

  Precondition() {
    type = UNDEFINED;
  }
  Precondition(PreconditionType tp) { type = tp; }
  virtual ~Precondition() {}

  virtual void Print(ostream& out) const {};
  virtual void PrintSPIN(ostream& out) const {};

protected:
  static char *RelToString(DOMString rel);

private:

  PreconditionType type;
};

typedef list<Precondition *> ListPrecond;

class PrivilegePrecondition : public Precondition {

public:

  PrivilegePrecondition() : Precondition(PRIVILEGE) {}
  PrivilegePrecondition(DOMString& h, char *r, DOMString& v) : Precondition(PRIVILEGE) {
    host = h;
    rel = r;
    value = v;
  }
  virtual ~PrivilegePrecondition() { }

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  
  long GetMemSize() const {
    return GetDOMStringSize( host ) + strlen( rel ) + 
           GetDOMStringSize( value ) + sizeof(PrivilegePrecondition);
  }
  
private:
  DOMString host;
  char *rel;
  DOMString value;
};

class SPAPrecondition : public Precondition {

public:

  SPAPrecondition() : Precondition(SERVICE_PORT_AGREEMENT) {}
  SPAPrecondition(DOMString& s) : Precondition(SERVICE_PORT_AGREEMENT) { service = s; }
  virtual ~SPAPrecondition() {}

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( service ) + sizeof(SPAPrecondition);
  }
  
private:
  DOMString service;
};

class ServicePrecondition : public Precondition {

public:

  ServicePrecondition() : Precondition(SERVICE) {}
  ServicePrecondition(DOMString& h, DOMString& n) : Precondition(SERVICE) {
    host = h;
    name = n;
  }
  virtual ~ServicePrecondition() {}

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( host ) + GetDOMStringSize( name ) + sizeof(ServicePrecondition);
  }

private:
  DOMString host;
  DOMString name;
};

class Network;

class ConnectivityPrecondition : public Precondition {

public:

  ConnectivityPrecondition( Network *net ) : Precondition(CONNECTIVITY) { network = net; }
  ConnectivityPrecondition( Network *net, DOMString& s, ServiceID i, int connsrc ) : Precondition(CONNECTIVITY) {
    network = net;
    service = s;
    id = i;
    src = connsrc;
  }
  virtual ~ConnectivityPrecondition() {}

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( service ) + sizeof(ConnectivityPrecondition);
  }

private:
  Network *network;
  DOMString service;
  ServiceID id;
  int src;
};

class TrustPrecondition : public Precondition {

public:

  TrustPrecondition() : Precondition(TRUST) {}
  virtual ~TrustPrecondition() {}

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return sizeof(TrustPrecondition);
  }

private:
};

class NoTrustPrecondition : public Precondition {

public:

  NoTrustPrecondition( Network *net ) : Precondition(NOTRUST) { network = net; }
  virtual ~NoTrustPrecondition() {}

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return sizeof(NoTrustPrecondition);
  }

private:
  Network *network;
};

class VulnerabilityPrecondition : public Precondition {

public:

  VulnerabilityPrecondition() : Precondition(VULNERABILITY) {}
  VulnerabilityPrecondition(DOMString& h, DOMString& n) : Precondition(VULNERABILITY) {
    host = h;
    name = n;
  }
  virtual ~VulnerabilityPrecondition() {}

  virtual void Print(ostream& out) const ;
  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( host ) + GetDOMStringSize( name ) + sizeof(VulnerabilityPrecondition);
  }

private:
  DOMString host;
  DOMString name;
};

class KnowledgePrecondition : public Precondition {

public:

  KnowledgePrecondition() : Precondition(KNOWLEDGE) {}
  KnowledgePrecondition(DOMString& n, DOMString& v) : Precondition(KNOWLEDGE), name(n), value(v) {}
  virtual ~KnowledgePrecondition();

  virtual void Print(ostream& out) const ;
  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( name ) + GetDOMStringSize( value ) + sizeof(KnowledgePrecondition);
  }

private:
  DOMString name;
  DOMString value;
};

class ObjectPrecondition : public Precondition {

public:

  ObjectPrecondition() : Precondition(OBJECT) {}
  ObjectPrecondition(int host_name, const DOMString& obj_name) : Precondition(OBJECT) { 
    m_host = host_name; m_obj_name = obj_name; 
  }
  virtual ~ObjectPrecondition() {}

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  
  void   SetObjAttribute( const DOMString &attr_name, const DOMString &attr_value ) {
      m_obj_attributes[attr_name] = attr_value;
  }
  
  void SetHost( int host ) { m_host = host; }
  int GetHost( ) { return m_host; }
  
  long GetMemSize() const {
    return GetDOMStringSize( m_obj_name ) + GetDOMStringMapSize( m_obj_attributes ) + 
           sizeof(ObjectPrecondition);
  }
  
private:
  DOMString m_obj_name;
  int m_host;
  DOMStringMap m_obj_attributes;
};

class SentObjectPrecondition : public Precondition {

public:

  SentObjectPrecondition() : Precondition(SENTOBJECT) { 
    m_set_source = false;
    m_srchost = 0; 
    m_dsthost = 0;
  }
  
  SentObjectPrecondition(int dsthost, const DOMString& service, const DOMString& obj_name) : Precondition(SENTOBJECT) { 
    m_service = service; 
    m_dsthost = dsthost;
    m_obj_name = obj_name; 
    m_set_source = false;
  }
  virtual ~SentObjectPrecondition() {}

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  
  void SetSource( int host ) { m_srchost = host; m_set_source = true; }
  
  long GetMemSize() const {
    return GetDOMStringSize( m_obj_name ) + GetDOMStringSize( m_service ) + 
           sizeof(SentObjectPrecondition);
  }
  
private:
  DOMString m_obj_name;
  int m_srchost;
  int m_dsthost;
  DOMString m_service; 
  bool m_set_source;
  //DOMStringMap m_obj_attributes;
};

class FirewallRulePrecondition: public Precondition {
  
public:
  FirewallRulePrecondition(): Precondition(FIREWALLRULE) { }
  FirewallRulePrecondition( Network *net, HostID srchost, PortID srcport, 
                            HostID dsthost, PortID dstport, bool result = true ): Precondition(FIREWALLRULE) { 
    m_network = net;
    m_srchost = srchost;
    m_dsthost = dsthost;
    m_srcport = srcport;
    m_dstport = dstport;
    m_result  = result;
  }
  ~FirewallRulePrecondition(){}
    
  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  
  long GetMemSize() const {
    return sizeof( FirewallRulePrecondition );
  }
  
private:
  Network  *m_network;
  HostID    m_srchost;
  HostID    m_dsthost;
  PortID    m_srcport;
  PortID    m_dstport;
  bool      m_result;
};

class FirewallReachablePrecondition: public Precondition {
  
public:
  FirewallReachablePrecondition( int srchost, const DOMString &srcport, 
                                 int dsthost, const DOMString &dstport): Precondition(FIREWALLREACHABLE) {
    m_srchost = srchost;
    m_dsthost = dsthost;
    m_srcport = srcport;
    m_dstport = dstport;
  }
  ~FirewallReachablePrecondition(){}
    
  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  
  long GetMemSize() const {
    return sizeof( FirewallReachablePrecondition ) +
           GetDOMStringSize( m_srcport ) + GetDOMStringSize( m_dstport );
  }
    
private:
  int    m_srchost;
  int    m_dsthost;
  DOMString m_srcport;
  DOMString m_dstport;
};

class OrPrecondition : public Precondition {

public:

  OrPrecondition() : Precondition(OR) {}
  virtual ~OrPrecondition();

  virtual void AddDisjunct(Precondition *child) { children.push_back(child); }

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  
  long GetMemSize() const {
    return children.size() * sizeof(Precondition *) + sizeof(OrPrecondition);
  }

private:
  ListPrecond children;
};

class AndPrecondition : public Precondition {

public:

  AndPrecondition() : Precondition(AND) {}
  virtual ~AndPrecondition();

  virtual void AddConjunct(Precondition *child) { children.push_back(child); }

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  
  long GetMemSize() const {
    return children.size() * sizeof(Precondition *) + sizeof(AndPrecondition);
  }

private:
  ListPrecond children;
};

class NotPrecondition : public Precondition {

public:

  NotPrecondition() : Precondition(NOT), cond(NULL) {}
  NotPrecondition(Precondition *p) : Precondition(NOT), cond(p) {}
  virtual ~NotPrecondition();

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return sizeof(NotPrecondition);
  }

private:
  Precondition *cond;
};

class Effect: public ModelMemUsage {

public:

  enum EffectType {
    PRIVILEGE,
    SERVICE,
    VULNERABILITY,
    CONNECTIVITY,
    TRUST,
    KNOWLEDGE,
    SENDOBJECT,
    PASSARG,
    FIREWALLACTION,
    UNDEFINED
  };

  Effect() { type = UNDEFINED; }
  Effect( const DOMString& h, const DOMString& s, const DOMString& n, const DOMString& v, AttackID aid, EffectType t) {
    host = h;
    source = s;
    name = n;
    value = v;
    attackid = aid;
    type = t;
  }
  virtual ~Effect() {  }

  DOMString GetName() { return name; }

  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  
  virtual long GetMemSize() const {
    return GetDOMStringSize( host ) + GetDOMStringSize( source ) +
           GetDOMStringSize( name ) + GetDOMStringSize( value ) + sizeof(Effect);
  }
  
public:
  DOMString host;
  DOMString source;
  DOMString name;
  DOMString value;
  AttackID  attackid;
  EffectType type;
};

class SendObjectEffect: public Effect {
  
public:
  SendObjectEffect() { type = Effect::SENDOBJECT;}
  SendObjectEffect( const DOMString& srchost, const DOMString& dsthost, const DOMString& dstsvc, const DOMString& objname, AttackID aid ): 
                   Effect( dsthost, srchost, objname, objname, aid, Effect::SENDOBJECT) 
  {
    m_service = dstsvc;
  };
  ~SendObjectEffect() {}
  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  DOMString GetSrcHost() const { return source; }
  DOMString GetDstHost() const { return host; }
  DOMString GetDstService() const { return m_service; } 
  DOMString GetObjName() const { return value; } 
  void SetObjName( const DOMString &name ) { value = name; }
  void SetDstHost( const DOMString &name ) { host = name; }
  void SetDstService( const DOMString &name ) { m_service = name; }
  void SetSrcHost( const DOMString &name ) { source = name; }
  
  long GetMemSize() const {
    return Effect::GetMemSize() + GetDOMStringSize( m_service ) + 
           sizeof(SendObjectEffect) - sizeof(Effect); 
  }
  
private:
  DOMString m_service;
};

class PassArgEffect: public Effect {
  
public:
  PassArgEffect() { type = Effect::PASSARG;}
  PassArgEffect( const DOMString& service, const DOMString& objname, AttackID aid ): 
                   Effect( DOMString::transcode(""), service, objname, objname, aid, Effect::PASSARG) 
  {};
  ~PassArgEffect() {}
  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const;
  DOMString GetService() const { return source; }
  DOMString GetObjName() const { return value; } 
  void SetObjName( const DOMString &name ) { value = name; }
  void SetService( const DOMString &name ) { source = name; }
};

class FirewallActionEffect: public Effect {

public:
  FirewallActionEffect( Network *net, bool status, AttackID aid ):
                       Effect( DOMString::transcode(""), DOMString::transcode(""), "", DOMString::transcode(""), aid, Effect::FIREWALLACTION) { 
    m_network = net;
    m_result  = status;
    m_fstatus_mode = true;
  }
  FirewallActionEffect( Network *net, int srchost, int srcport, 
                            int dsthost, int dstport, bool result, AttackID aid ):
                        Effect( "", DOMString::transcode(""), "", "", aid, Effect::FIREWALLACTION) { 
    
    m_network = net;
    m_rule.srchost = srchost;
    m_rule.dsthost = dsthost;
    m_rule.srcport = srcport;
    m_rule.dstport = dstport;
    m_result  = result;
    m_fstatus_mode = false;
  }
  ~FirewallActionEffect() {}
  
  virtual void Print(ostream& out) const;
  virtual void PrintSPIN(ostream& out) const {}
  
  long GetMemSize() const {
    return sizeof( FirewallActionEffect )  - sizeof(Effect);
  }
  FirewallRule GetFWRule() const {return m_rule; }
  bool GetResult() const { return m_result; }
  bool IsFWStatusEffect() const { return m_fstatus_mode; }
  
private:
  Network *m_network;
  FirewallRule m_rule;
  bool   m_result;
  bool   m_fstatus_mode;
};

class LTLAtomicProposition: public ModelMemUsage {
  
public:
  virtual ~LTLAtomicProposition() {}
  virtual void PrintSPIN(ostream& out) const = 0;
};

class LTLPrivilegeProposition: public LTLAtomicProposition{
  
public:
  LTLPrivilegeProposition( HostID host, const string &rel, const DOMString& value ) {
    m_host = host;
    m_rel  = rel;
    m_value = value;
  }
    
  virtual ~LTLPrivilegeProposition() {}
  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return m_rel.size() + GetDOMStringSize( m_value ) + sizeof(LTLPrivilegeProposition);
  }
  
private:
  HostID    m_host;
  string    m_rel;
  DOMString m_value;
};

class LTLServiceProposition: public LTLAtomicProposition {
  
public:
  LTLServiceProposition(HostID host, const DOMString& service) {
    m_host    = host;
    m_service = service;
  }
  virtual ~LTLServiceProposition() {}

  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( m_service ) + sizeof(LTLServiceProposition);
  }

private:
  HostID    m_host;
  DOMString m_service;
};

class LTLVulnerabilityProposition : public LTLAtomicProposition {

public:

  LTLVulnerabilityProposition(HostID host, const DOMString& vuln) {
    m_host = host;
    m_vuln = vuln;
  }
  virtual ~LTLVulnerabilityProposition() {}

  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( m_vuln ) + sizeof(LTLVulnerabilityProposition);
  }

private:
  HostID    m_host;
  DOMString m_vuln;
};

class LTLKnowledgeProposition: public LTLAtomicProposition {

public:

  LTLKnowledgeProposition(const DOMString& know) {
    m_know_name    = know;
    m_value        = DOMString::transcode( "TRUE" );
  }
  LTLKnowledgeProposition(const DOMString& know, const DOMString& value) {
    m_know_name    = know;
    m_value        = value;
  }
  
  virtual ~LTLKnowledgeProposition() {}

  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( m_know_name) + GetDOMStringSize( m_value ) +
           sizeof(LTLKnowledgeProposition);
  }

private:
  DOMString m_know_name;
  DOMString m_value;
};

class LTLObjectProposition: public LTLAtomicProposition {

public:

  LTLObjectProposition(HostID host_name, const DOMString& obj_name) { 
    m_host = host_name; m_obj_name = obj_name; 
  }
  virtual ~LTLObjectProposition() {}

  virtual void PrintSPIN(ostream& out) const;
  
  void   SetObjAttribute( const DOMString &attr_name, const DOMString &attr_value ) {
      m_obj_attributes[attr_name] = attr_value;
  }
  
  long GetMemSize() const {
    return GetDOMStringSize( m_obj_name) + GetDOMStringMapSize( m_obj_attributes ) + 
           sizeof(LTLObjectProposition);
  }
    
private:
  DOMString m_obj_name;
  HostID m_host;
  DOMStringMap m_obj_attributes;
};

class LTLSentObjectProposition: public LTLAtomicProposition {

public:

  LTLSentObjectProposition(HostID dsthost, HostID srchost, const DOMString& obj_name) { 
    m_dsthost = dsthost;
    m_srchost = srchost;
    m_obj_name = obj_name; 
    m_set_source = true;
    m_set_service = false;
  }
  
  LTLSentObjectProposition(HostID dsthost, const DOMString& service, const DOMString& obj_name) { 
    m_service = service; 
    m_dsthost = dsthost;
    m_obj_name = obj_name; 
    m_set_source = false;
    m_set_service = true;
  }
  
  LTLSentObjectProposition(HostID dsthost, HostID srchost, const DOMString& service, const DOMString& obj_name) { 
    m_service = service; 
    m_dsthost = dsthost;
    m_srchost = srchost;
    m_obj_name = obj_name; 
    m_set_source = true;
    m_set_service = true;
  }
  virtual ~LTLSentObjectProposition() {}

  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( m_obj_name) + GetDOMStringSize( m_service ) + 
           sizeof(LTLSentObjectProposition);
  }
  
private:
  DOMString m_obj_name;
  HostID m_srchost;
  HostID m_dsthost;
  DOMString m_service; 
  bool m_set_source;
  bool m_set_service;
  //DOMStringMap m_obj_attributes;
};

class LTLConnectivityProposition: public LTLAtomicProposition {

public:

  LTLConnectivityProposition(HostID src_host, HostID dst_host) {
    m_use_service = false;
    m_service     = DOMString::transcode("");
    m_dst         = dst_host;
    m_src         = src_host;
  }
  LTLConnectivityProposition(HostID src_host, HostID dst_host, const DOMString& service) {
    m_use_service = true;
    m_service     = service;
    m_dst         = dst_host;
    m_src         = src_host;
  }
  virtual ~LTLConnectivityProposition() {}

  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return GetDOMStringSize( m_service ) + sizeof(LTLConnectivityProposition);
  }

private:
  bool      m_use_service;
  DOMString m_service;
  HostID    m_dst;
  HostID    m_src;
};

class LTLTrustProposition: public LTLAtomicProposition {

public:

  LTLTrustProposition( HostID src_host, HostID dst_host) {
    m_dst = dst_host;
    m_src = src_host;
  }
  virtual ~LTLTrustProposition() {}

  virtual void PrintSPIN(ostream& out) const;
    
  long GetMemSize() const {
    return sizeof(LTLTrustProposition);
  }

private:
  HostID m_dst;
  HostID m_src;
};

typedef map<DOMString, LTLAtomicProposition*, less_str> LTLPropositionMap;

#endif // _PRECOND_H_
