(* $Id: spec.ml,v 1.1.1.1 2003/10/28 22:15:44 rl Exp $ *)

(* Author: John D. Ramsdell *)

(* converts specs into bdd *)

(* spec is the data type returned by the parser. *)

type ip = Ip of int * int * int * int

type kind =
    Addr of ip * ip			(* pattern and mask *)
  | Range of int * (int * int) list	(* size and range list *)

type spec = 
    Permit of kind list			(* accept *)
  | Deny of kind list			(* reject *)

exception Bad_int of int		(* raised only in bits_of_int *)

let bits_of_int size n =		(* convert int to bool list *)
  let rec loop i p a =
    if p >= size then
      a
    else
      loop (2*i) (p + 1) ((n land i <> 0) :: a) in
  if n < 0 || n >= 1 lsl size then
    raise (Bad_int n)
  else
    loop 1 0 []

let octet n =
  bits_of_int 8 n
    
let ip(Ip(w,x,y,z)) =
  octet w @ octet x @ octet y @ octet z
				  
let rec range size (m, n) =		(* convert a range to a list of *)
  if m > n then				(* pattern and mask pairs *)
    []
  else
    let s = (1 lsl size) - 1 in
    let rec loop i p =
      if 0 <> p land m || m + 2*p - 1 > n then
	let pat = bits_of_int size m in
	let mask = bits_of_int size(s lxor (p - 1)) in
	(pat, mask) :: range size(m + p, n)
      else
	loop (i + 1) (2*p) in
    loop 0 1
      
let pm x =				(* converts a kind to a list of *)
  match x with				(* pattern and mask pairs *)
    Addr(pat, mask) -> [ip pat, ip mask]
  | Range(size, l) -> List.concat(List.map(range size) l)


(* takes a list of a list of items.  Call a sequence a list containing
   an item from each list taken in order.  This routine returns a list
   of all sequences. *)

let rec all_seqs l =
  match l with
    [] -> []
  | e :: [] ->
      List.map (function x -> [x]) e
  | e :: l ->
      let rec outer paths a =
	match paths with
	  [] -> a
	| e' :: l' -> 
	    let rec inner e a =
	      match e with
		[] -> outer l' a
	      |	(e'' :: l'') ->
		  inner l'' ((e'' :: e') :: a) in
	    inner e a in
      outer (all_seqs l) []  

let pms2path l =			(* converts a list of pairs *)
  let rec outer i l a =			(* of patterns and masks to *)
    let rec inner p m i l a =		(* a path *)
      match (p, m) with
	([], m) -> outer i l a
      | (p, []) -> outer i l a
      | (q :: p, n :: m) ->
	  let b = 
	    if n then
	      (not q, i) :: a
	    else
	      a in
	  inner p m (i + 1) l b in
    match l with
      [] -> List.rev a
    | (p, m) :: r -> inner p m i r a in
  outer 0 l []

let paths2bdd paths =
  let f bdd path =
    Bdd_utils.disjoin bdd (Bdd_utils.path2bdd path) in
  List.fold_left f Bdd.falsehood paths

let kinds2bdd kind =
  paths2bdd(List.map pms2path (all_seqs(List.map pm kind)))

let specs2bdd rev_specs =
  let f bdd spec =
    match spec with
      Permit kinds -> Bdd_utils.disjoin bdd (kinds2bdd kinds)
    | Deny kinds -> Bdd_utils.conjoin bdd (Bdd.complement(kinds2bdd kinds)) in
  List.fold_left f Bdd.falsehood rev_specs

let specs2actions specs =
  Actions.spines2actions(Spine.bdd2spines(specs2bdd specs))
