(* $Id: prop.ml,v 1.1.1.1 2003/10/28 22:15:44 rl Exp $ *)

(* performs a shortest path-like computation to find a fixed point
   to the frame transport operator. *)

(* Author: John D. Ramsdell *)

(* Copyright (C) 2001 The MITRE Corporation

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *)

open Bdd

open Bdd_utils

open Graph

(* A queue that adds a node only if it is not already on the queue. *)

type queue =
    Nil
  | Cons of pair
and pair = 
    {node : node; mutable rest : queue}

(* An empty queue is a pair with a dummy node. *)

let create() =
  {node = {label = ""; frames = falsehood; out = []}; rest = Nil}

let rec add q n =
  match q.rest with
    Nil ->				(* node not there so add it *)
      q.rest <- Cons {node = n; rest = Nil}
  | Cons pair ->
      if pair.node == n then		(* node already there *)
	()
      else
	add pair n

let propagate g =
  let q = create() in			(* A queue of nodes *)
  (* This function is applied to each edge associated with the src node. *)
  let propagate_node src edge = 
    let dest = edge.dest in		(* The destination node *)
    let frames = disjoin dest.frames (conjoin src.frames edge.filter) in
    if not (is_same frames dest.frames) then begin
					(* If new frames were added *)
      dest.frames <- frames;		(* update node's frame *)
      add q dest			(* and add node to queue *)
    end in (* Note: dest need not be added if it is already on the queue *)
  (* Add all nodes to the queue *)
  List.iter (add q) (graph2nodes g);
  while q.rest <> Nil do
    match q.rest with
      Nil -> ()				(* cannot happen *)
    | Cons pair ->
	let src = pair.node in
	q.rest <- pair.rest;		(* remove node from queue *)
	List.iter (propagate_node src) src.out
  done

let propagate_geometry geom =
  let graph = Graph.make() in
  load graph geom;
  propagate graph;
  unload graph
