(* $Id: graph.ml,v 1.1.1.1 2003/10/28 22:15:44 rl Exp $ *)

(* Data types used for the frame propagation computation. *)

(* Author: John D. Ramsdell *)

(* Copyright (C) 2001 The MITRE Corporation

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *)

open Bdd

open Spine

open Actions

(* A node in the graph represents a network link 
   and an edge is a path through a router. *)

type node = { 
  label : string;
  mutable frames : bdd;   (* The set of frames on this link *)
  mutable out : edge list (* The paths out of this link *)
  }
and edge = {
  filter : bdd;           (* The filter on this router path *)
  dest : node             (* The link at the output of the filter *)
  }

type graph = (string, node) Hashtbl.t

type geometry = 
    Node of string * action list	(* Network label and frame set *)
  | Edge of string * string * action list(* source, destination, and filter *)

let create : int -> graph = Hashtbl.create

let init_graph_table_size = 509

let make() = create init_graph_table_size

let clear g = Hashtbl.clear g

let find g label =
  try
    Hashtbl.find g label
  with
    Not_found ->
      let n = {label = label; frames = falsehood; out = []} in
      Hashtbl.add g label n;
      n

let construct g geom =
  match geom with
    Node (label, []) -> ()
  | Node (label, a) ->
      let n = find g label in
      let b = spines2bdd (actions2spines a) in
      n.frames <- b
  | Edge (src, dest, []) -> ()
  | Edge (src, dest, a) ->
      let s = find g src in
      let d = find g dest in
      let b = spines2bdd (actions2spines a) in
      s.out <- {filter = b; dest = d} :: s.out

let load g geoms = List.iter (construct g) geoms

let graph2nodes g =
  let stk : node list ref = ref [] in
  let collect l n = stk := n :: !stk in
  Hashtbl.iter collect g;
  !stk

let destruct lst {label = label; frames = frames; out = out} = 
  let rec loop lst out =
    match out with
      [] -> 
	let a = spines2actions (bdd2spines frames) in
	Node (label, a) :: lst 
    | {filter = filter; dest = dest} :: out ->
	let a = spines2actions (bdd2spines filter) in
	loop (Edge (label, dest.label, a) :: lst) out in
  loop lst out

let unload g = 
  List.fold_left destruct [] (graph2nodes g)

let print_geometry geom =
  match geom with
    Node (label, a) ->
      Format.print_string "node(";
      Format.open_box(0);
      Format.print_string "\"";
      Format.print_string label;
      Format.print_string "\",";
      Format.print_space();
      print_actions a;
      Format.print_string ").";
      Format.close_box()
  | Edge (src, dest, a) ->
      Format.print_string "edge(";
      Format.open_box(0);
      Format.print_string "\"";
      Format.print_string src;
      Format.print_string "\",";
      Format.print_space();
      Format.print_string "\"";
      Format.print_string dest;
      Format.print_string "\",";
      Format.print_space();
      print_actions a;
      Format.print_string ").";
      Format.close_box()

let print_geometries geoms =
  List.iter (fun geom -> print_geometry geom; Format.print_newline()) geoms
