(* $Id: bdd.mli,v 1.1.1.1 2003/10/28 22:15:44 rl Exp $ *)

(* reduced ordered binary decision diagrams *)

(* John D. Ramsdell -- Decemeber 1998 *)

(* Copyright (C) 2001 The MITRE Corporation

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *)

(* types *)

type bdd

(* Paths are lists of pairs.  The second item in the pair is the
   variable and first item is a boolean value telling if the variable
   should be negated.  A variable is a non-negative integer.  A path
   is sorted by the integer in decreasing order.  A path may not
   contain two pairs with the same variable. *)

type path = (bool * int) list

(* predicates *)

val is_same : bdd -> bdd -> bool	(* are bdds the same? *)
val is_node : bdd -> bool		(* is bdd an internal node? *)
val is_complement : bdd -> bool		(* is this a complement edge? *)
(* if so, the boolean function should be interpreted
   as the complement of what is represented by the node. *) 
val is_truth : bdd -> bool		(* is bdd the truth leaf? *)
val is_falsehood : bdd -> bool		(* is bdd the falsehood leaf? *)

(* constructors *)

val truth : bdd
val falsehood : bdd
(* construct the boolean function f(x) = x *)
val identity : int -> bdd

(* node accessors - Constant_bdd thrown when applied to a leaf *)

exception Constant_bdd

val variable : bdd -> int
val consequent : bdd -> bdd
val alternative : bdd -> bdd

(* bdd operators *)

val complement : bdd -> bdd
(* if-then-else, the bdd composition operator *)
val ite : bdd * bdd * bdd -> bdd

(* traversals *)

val each_node : ('a -> int -> 'a) -> 'a -> bdd -> 'a
val each_path : ('a -> bool -> path -> 'a) -> 'a -> bdd -> 'a

(* metrics *)

val strength : bdd -> path -> int

(* Hash table parameters *)

(* initialize the bdd hash tables with the following parameters *)
val init : int				(* number of unique tables *)
    * int				(* initial size of a table *)
    * int				(* number of computed tables *)
    * int				(* initial size of a table *)
    -> unit
