#ifndef _FSTORAGE_H_
#define _FSTORAGE_H_

#include <iostream>

using namespace std;

namespace GraphGraphics
{
  
struct GraphFileInfo
{
  enum GraphFormatType
  {
    BIN16_FORMAT = 0,
    BIN32_FORMAT = 1,
    ASCII_FORMAT = 2
  };
  
  string file_name;
  string version;
  string graph_name;
  GraphFormatType file_format;
  string comment;
  string model_file;
  bool   save_graph;
  bool   save_view;
  
  GraphFileInfo(){}
  GraphFileInfo( const GraphFileInfo& info )
  {
    file_name  = info.file_name;
    version    = info.version;
    file_format      = info.file_format;
    graph_name = info.graph_name;
    comment    = info.comment;
    model_file = info.model_file;
    save_graph = info.save_graph;
    save_view  = info.save_view;
  }
  
  GraphFileInfo& operator = ( const GraphFileInfo& info )
  {
    file_name  = info.file_name;
    version    = info.version;
    file_format = info.file_format;
    graph_name = info.graph_name;
    comment    = info.comment;
    model_file = info.model_file;
    save_graph = info.save_graph;
    save_view  = info.save_view;
    
    return *this;
  }
  
  void refresh( const GraphFileInfo& info )
  {
    if ( !info.file_name.empty() )
      file_name  = info.file_name;
    
    if ( !info.version.empty() )
      version    = info.version;
    
    file_format = info.file_format;
    
    if ( !info.graph_name.empty() )
      graph_name = info.graph_name;
    
    if ( !info.comment.empty() )
      comment    = info.comment;
    
    if ( !info.model_file.empty() )
      model_file = info.model_file;
    
    save_graph = info.save_graph;
    save_view  = info.save_view;
  }
};

class Storable
{
public:
  static const string ASCII_COMMENT_ID;
  static const string ASCII_MODEL_FILE_ID;

  static const string ASCII_FILTER_ID;
  static const string ASCII_DEF_FILTER_ID;
  static const string ASCII_NODE_RANGE_ID;
  static const string ASCII_V_DISTANCE_ID;
  static const string ASCII_H_DISTANCE_ID;
  static const string ASCII_SPLINE_ID;
  static const string ASCII_BG_COLOR_ID;
  static const string ASCII_SEL_COLOR_ID;
  static const string ASCII_EDGE_COLOR_ID;
  static const string ASCII_EDGE_WIDTH_ID;
  
  enum BinPropIds
  {
    BIN_COMMENT_ID     = 0x0D01,
    BIN_MODEL_FILE_ID  = 0x0D02,
     
    BIN_FILTER_ID      = 0x0900,
    BIN_DEF_FILTER_ID  = 0x0901,
    BIN_NODE_RANGE_ID  = 0x0A01,
    BIN_V_DISTANCE_ID  = 0x0A02,
    BIN_H_DISTANCE_ID  = 0x0A03,
    BIN_SPLINE_ID      = 0x0A04,
    BIN_BG_COLOR_ID    = 0x0B01,
    BIN_SEL_COLOR_ID   = 0x0B02,
    BIN_EDGE_COLOR_ID  = 0x0B03,
    BIN_EDGE_WIDTH_ID  = 0x0B04
  };
  
  enum FlagBits
  {
    BIN_STORAGE = 0,
    ASCII_STORAGE = 1,
    BIN32_STORAGE = 2
  };
  
  enum BinFileIds
  {
    BIN_BLOCK_SEPARATOR     = 0xFFFFFFFF,
    BIN_INTERNAL_SEPARATOR1 = 0xFFFFFFFE,
    BIN_INTERNAL_SEPARATOR2 = 0xFFFFFFFD,
    BIN_EDGE_ID             = 0xFFFFFF01,
    BIN_NAME_SET_OFFSET     = 0xFFFFFF00
  };
  
  static const string ASCII_INTERNAL_SEPARATOR1;
  static const string ASCII_INTERNAL_SEPARATOR2;
  
  static const char BIN_START_ATTR_TYPE;
  static const char BIN_START_NODE_TYPE;
  static const char BIN_END_ATTR_TYPE;
  static const string ASCII_INT_TYPE;
  static const string ASCII_SHORT_TYPE;
  static const string ASCII_BYTE_TYPE;
  static const string ASCII_BOOL_TYPE;  
  static const string ASCII_CHAN_TYPE;
  static const string ASCII_BIT_TYPE;
  static const string ASCII_TEXT_TYPE;
  static const string ASCII_STRING_TYPE;
  
  static const string ASCII_START_ID;
  static const string ASCII_END_ID;
  static const string ASCII_NAME_ID;
  static const string ASCII_SHAPE_ID;
  static const string ASCII_FILLED_ID;
  static const string ASCII_XY_RATIO_ID;
  static const string ASCII_COLOR_ID;
  static const string ASCII_FONT_ID;
  static const string ASCII_PEN_ID;
  static const string ASCII_ASSIGN_ID;
  static const string ASCII_DEFAULT_NAME;
  static const string ASCII_CIRCLE_NAME;
  static const string ASCII_DOUBLE_CIRCLE_NAME;
  static const string ASCII_RECTANGULAR_NAME;
  static const string ASCII_POINT_NAME;
   
  enum BinPropGroupIds
  {
    BIN_START_ID     = 0x0C00,
    BIN_NAME_ID      = 0x0C01,
    BIN_SHAPE_ID     = 0x0C02,
    BIN_FILLED_ID    = 0x0C03,
    BIN_XY_RATIO_ID  = 0x0C04,
    BIN_COLOR_ID     = 0x0C05,
    BIN_FONT_ID      = 0x0C06,
    BIN_PEN_ID       = 0x0C07,
    BIN_ASSIGN_ID    = 0x0C08
  };
   
  enum ErorrLevels
  {
    OK_STATUS = 0,
    NOT_GRAPH_FILE = 1,
    CRITICAL = 2
  };
  
protected:
  static unsigned long last_offset;
  static string m_last_error_description;
  static unsigned int  m_last_error_level;

public:  
  virtual ~Storable() {}
  static void write_1bytes( ostream& s_out, char value ) { s_out.put( value ); }
  static void write_2bytes( ostream& s_out, short value ) { s_out.write( (char*)&value, 2 ); }  
  static void write_4bytes( ostream& s_out, long value ) { s_out.write( (char*)&value, 4 ); } 
  static void write_bytes( ostream& s_out, char *value, unsigned int v_len ) { s_out.write( value, v_len ); }  
  
  static void write_bin_identifier( ostream& s_out, long value, unsigned long write_options = 0 )
  {
    if ( write_options & BIN32_STORAGE )
      write_4bytes( s_out, value );
    else
      write_2bytes( s_out, (short)value );
  }
  
  static bool read_1bytes( istream& s_in, char &value ) 
  { 
    s_in.get( value ); 
    return !s_in.eof();
  }
  static bool read_2bytes( istream& s_in, short &value )
  {
    unsigned char v[2];
    
    s_in.read( (char *)v, 2 );
    value = (short)(((unsigned int)(v[1]) << 8) | (unsigned int)(v[0]));
    
    return !s_in.eof();
  }
  static bool read_4bytes( istream& s_in, long &value )
  {
    unsigned char v[4];
    
    s_in.read( (char *)v, 4 );
    value = (long)(((((unsigned long)(v[3]) << 8) | (unsigned long)(v[2])) << 16) | 
                  (((unsigned long)(v[1]) << 8) | (unsigned long)(v[0])));
    
    return !s_in.eof();
  }
  static bool read_bytes( istream& s_in, char *buff, unsigned int len) 
  {
    s_in.read( buff, len );
    return !s_in.eof();    
  }
  
  static bool read_bin_identifier( istream& s_in, long &value, unsigned long read_options = 0 )
  {
    bool res = false;
    if ( read_options & BIN32_STORAGE )
      res = read_4bytes( s_in, value );
    else
    {
      short svalue;
      res = read_2bytes( s_in, svalue );
      value = (long)svalue;
    }
    return res;
  }
  
  static bool read_min_ASCII_block( istream& s_in, string &data )
  {
    bool res = true;
    char buff[65536];
    data = "";
    memset( buff, 0, sizeof(buff) );
    s_in.getline( buff, sizeof(buff) );
      
    while ( !s_in.fail() && !s_in.eof() && buff[strlen( buff ) - 1] == '\\' )
    {
      data += string( buff, strlen( buff ) - 1 );
      memset( buff, 0, sizeof(buff) );
      s_in.getline( buff, sizeof(buff) );
      res = !s_in.fail() || buff[0] == 0;
    } 
    
    data += string( buff );
       
    return res;
  }
  
  static bool read_min_ASCII_block( istream& s_in, string &data, string &data_id )
  {
    data_id = "";
    bool res = read_min_ASCII_block( s_in, data );
    
    if ( res )
    {
      unsigned int pos = data.find( ':' );
      
      if ( pos != string::npos )
      {
        if ( pos + 1 >= data.length() )
        {
          data_id = data;
          data = "";
        }
        else
        {
          data_id = data.substr( 0, pos + 1 );
          
          pos = data.find_first_not_of( ' ', pos + 1 );
          
          if ( pos != 0 )
            data = data.substr( pos );
        }
      }
    }
    
    return res;
  }
  
  static void set_error( const string& err_description ){ m_last_error_description = err_description; }
  static string get_error( ){ return m_last_error_description; }
  static void set_error_level( unsigned int level ) { m_last_error_level = level; }
  static unsigned int get_error_level( ) { return m_last_error_level; }
  
  virtual void write_to( ostream& s_out, unsigned long write_options = 0 ) = 0;
  virtual bool read_from( istream& s_in, unsigned long read_options = 0 ) = 0;
};

};
#endif /*_FSTORAGE_H_*/

