
#ifndef _SHAPES_H_
#define _SHAPES_H_

#include <cmath>
#include <string>
#include "types.h"

using namespace std;

namespace GraphGraphics
{
class GraphicArea;

//OS independent class for logical size geometry representation
class TwoDSize
{
protected:
   OneDValue m_width;
   OneDValue m_height;

public:
   OneDValue get_width() { return m_width; }
   OneDValue get_height() { return m_height; }

   void set_width( OneDValue width ) { m_width = width; }
   void set_height( OneDValue height ) { m_height = height; }

   TwoDSize() {}
   TwoDSize( OneDValue width, OneDValue height )
   {
      m_width = width;
      m_height = height;
   }

   TwoDSize( const TwoDSize &size ) 
   {
      m_width = size.m_width;
      m_height = size.m_height;
   }

   ~TwoDSize(){}

   TwoDSize& operator =( const TwoDSize &size ) 
   {
      m_width = size.m_width;
      m_height = size.m_height;
      return *this;
   }

   bool operator ==( const TwoDSize &size ) const
   {
      return ((m_width == size.m_width) && (m_height == size.m_height));
   }

   TwoDSize operator +( const TwoDSize &size )
   {
      return TwoDSize( m_width + size.m_width, m_height + size.m_height );
   }
};

//OS independent class for logical 2-D point geometry representation
class TwoDPoint
{
protected:
   OneDValue m_x_coord;
   OneDValue m_y_coord;

public:

   OneDValue get_x() const { return m_x_coord; }
   OneDValue get_y() const { return m_y_coord; }

   void set_x( OneDValue x ) { m_x_coord = x; }
   void set_y( OneDValue y ) { m_y_coord = y; }

   TwoDPoint& operator = ( const TwoDPoint &point ) 
   { 
      m_x_coord = point.m_x_coord; 
      m_y_coord = point.m_y_coord;
      return *this; 
   }

   bool operator == ( const TwoDPoint &point ) const
   { 
      return ((m_x_coord == point.m_x_coord) && (m_y_coord == point.m_y_coord));
   }

   TwoDPoint(){ m_x_coord = 0; m_y_coord = 0; };

   TwoDPoint( OneDValue x, OneDValue y )
   {
      m_x_coord = x;
      m_y_coord = y;
   }

   TwoDPoint( const TwoDPoint &point )
   {
      m_x_coord = point.m_x_coord;
      m_y_coord = point.m_y_coord;
   }

   ~TwoDPoint() { }

   OneDValue distance_to( const TwoDPoint &point ) const
   {
      double dx = (double)(point.m_x_coord - m_x_coord);
      double dy = (double)(point.m_y_coord - m_y_coord);
      double dist = sqrt( dx * dx + dy * dy );
	  
      return (OneDValue)DROUND(dist);
   }
};

class GeometryShape
{
public:
   virtual ~GeometryShape() {}

   virtual TwoDPoint get_center() const = 0;
   virtual OneDValue get_range() const = 0;

   virtual void set_center( const TwoDPoint &center ) = 0;
   virtual void set_range( OneDValue range ) = 0;

   virtual bool inside( const TwoDPoint &point ) const = 0;
   virtual TwoDPoint get_nearest_to( const TwoDPoint &point ) const = 0;
   virtual bool is_empty() const = 0;
   virtual void draw( GraphicArea* g, bool filled = false ) = 0;
   virtual TwoDSize get_label_max_size( ) const = 0;
   virtual void draw_label( GraphicArea* g, const string &label );
};

//
class HidenPoint: public GeometryShape
{
private:
  TwoDPoint m_location;

public:
  HidenPoint(){}
  HidenPoint( const TwoDPoint &point ) { m_location = point; }
  HidenPoint( const HidenPoint &h_point ) { m_location = h_point.m_location; }
  HidenPoint& operator = ( const HidenPoint &h_point ) { m_location = h_point.m_location; return (*this); }
  bool operator == ( const HidenPoint &h_point ) const { return (m_location == h_point.m_location); }
  
  TwoDPoint get_center() const { return m_location; }
  OneDValue get_range() const { return 0; }

  void set_center( const TwoDPoint &center ) { m_location = center; }
  void set_range( OneDValue range ) { }

  bool inside( const TwoDPoint &point ) const { return m_location == point; }
  TwoDPoint get_nearest_to( const TwoDPoint &point ) const { return m_location; }
  
  bool is_empty() const { return true; }
  void draw( GraphicArea* g, bool filled = false ) { }
  TwoDSize get_label_max_size( ) const { return TwoDSize( 0, 0 ); }
  void draw_label( GraphicArea* g, const string &label ) {}
};

//OS independent class for line segment geometry representation
class LineSegment: public GeometryShape
{
private:
   TwoDPoint m_start_point;
   TwoDPoint m_end_point;
   TwoDPoint m_direction;

public:
   static TwoDPoint get_scaled_line_point( TwoDPoint start_point, TwoDPoint end_point, double scale_coef )
   {
      double x = ((1 - scale_coef) * (double)start_point.get_x() + scale_coef * (double)end_point.get_x());
      double y = ((1 - scale_coef) * (double)start_point.get_y() + scale_coef * (double)end_point.get_y());

      return TwoDPoint( (OneDValue)DROUND(x), (OneDValue)DROUND(y) );
   }

private:
   void make_direction()
   {
      m_direction = get_scaled_line_point( TwoDPoint(0, 0), 
                           TwoDPoint( m_end_point.get_x() - m_start_point.get_x(), 
                                      m_end_point.get_y() - m_start_point.get_y()), 
                                      1000.0 / (double) m_start_point.distance_to(m_end_point));
   }
   
public:
   LineSegment() {}
   LineSegment( const TwoDPoint &start_point, const TwoDPoint &end_point ):
         m_start_point( start_point ), m_end_point( end_point ) 
   {
      make_direction();
   }

   LineSegment( const LineSegment &line )
   {
      m_start_point = line.m_start_point;
      m_end_point = line.m_end_point; 

      make_direction();
   }
   ~LineSegment() {}

   void set_start_point( const TwoDPoint &point )
   {
      m_start_point = point;
      make_direction();
   }

   void set_end_point( const TwoDPoint &point )
   {
      m_end_point = point;
      make_direction();
   }

   TwoDPoint get_end_point() const { return m_end_point; }
   TwoDPoint get_start_point() const { return m_start_point; }

   OneDValue get_length() const { return m_start_point.distance_to( m_end_point ); }

   LineSegment& operator = ( const LineSegment& line )
   {
      m_start_point = line.m_start_point;
      m_end_point = line.m_end_point;
      make_direction();

      return *this;
   }

   bool operator == ( const LineSegment& line )
   {
      return ( m_start_point == line.m_start_point && m_end_point == line.m_end_point );
   }

   int intersect( const LineSegment &line );
   int intersection( const LineSegment &line, TwoDPoint* result_point );
   
   int on_line( const TwoDPoint &point ) const
   {
      double dres = (double)(point.get_x() - m_start_point.get_x()) * 
                    (double)(m_end_point.get_y() - m_start_point.get_y()) -
                    (double)(point.get_y() - m_start_point.get_y()) *
                    (double)(m_end_point.get_x() - m_start_point.get_x());

      long res = (long)DROUND( dres );

      return ((res > 0)? 1: ((res < 0)? -1: 0));
   }

   bool inside( const TwoDPoint &point ) const
   {
      return (on_line( point ) == 0) && 
             ((point.get_x() <= GGMAX(m_start_point.get_x(), m_end_point.get_x())) && 
              (point.get_x() >= GGMIN(m_start_point.get_x(), m_end_point.get_x())) && 
              (point.get_y() >= GGMIN(m_start_point.get_y(), m_end_point.get_y())) &&
              (point.get_y() <= GGMAX(m_start_point.get_y(), m_end_point.get_y())) );
   }

   TwoDPoint get_center() const
   {
      return TwoDPoint( (m_start_point.get_x() + m_end_point.get_x()) / 2, 
                        (m_start_point.get_y() + m_end_point.get_y()) / 2 );
   }

   OneDValue get_range() const 
   {
      TwoDPoint center = get_center();

      return GGMAX( center.distance_to( m_start_point ), center.distance_to( m_end_point ) );
   }

   void set_center( const TwoDPoint &center )
   {
      TwoDPoint old_center = get_center();

      OneDValue x_offset = center.get_x() - old_center.get_x();
      OneDValue y_offset = center.get_y() - old_center.get_y();

      m_start_point.set_x( m_start_point.get_x() + x_offset );
      m_start_point.set_y( m_start_point.get_y() + y_offset );

      m_end_point.set_x( m_end_point.get_x() + x_offset );
      m_end_point.set_y( m_end_point.get_y() + y_offset );
   }

   void set_range( OneDValue range );

   bool is_empty() const
   {
      return (m_start_point == m_end_point);
   }
   
   TwoDPoint get_nearest_to( const TwoDPoint &point ) const;
   
   void draw( GraphicArea* g, bool filled = false );
   TwoDSize get_label_max_size( ) const { return TwoDSize( 0, 0 ); }
};

//OS independent class for rectangle geometry representation
class Rectangle: public GeometryShape
{
protected:
   TwoDPoint m_left_top_point;
   TwoDPoint m_right_bottom_point;
   double m_width_ratio;
   double m_height_ratio;

protected:
  
  void calc_ratio( )
  {
     m_width_ratio = labs( m_right_bottom_point.get_x() - m_left_top_point.get_x() );
     m_height_ratio = labs( m_right_bottom_point.get_y() - m_left_top_point.get_y() );
     
     if ( m_width_ratio == 0.0 || m_height_ratio == 0.0 )
     {
       m_width_ratio += 1.0;
       m_height_ratio += 1.0;
     }
  }
  
public:
   
   Rectangle() { };

   Rectangle( const TwoDPoint &left_top, const TwoDPoint &right_bottom ): 
                  m_left_top_point( left_top ), m_right_bottom_point( right_bottom ) 
   {
     calc_ratio( );
   }

   Rectangle( OneDValue left, OneDValue top, OneDValue right, OneDValue bottom ): 
                  m_left_top_point( left, top ), m_right_bottom_point( right, bottom ) 
   {
     calc_ratio( );
   }

   Rectangle( const Rectangle &rect )
   {
      m_left_top_point = rect.m_left_top_point; 
      m_right_bottom_point = rect.m_right_bottom_point;
      calc_ratio( );
   }
   
   Rectangle( double default_width_ratio, double default_height_ratio )
   {
      m_width_ratio = default_width_ratio;
      m_height_ratio = default_height_ratio;
   }

   ~Rectangle() { }

   OneDValue get_width() const { return (m_right_bottom_point.get_x() - m_left_top_point.get_x()); }
   OneDValue get_height() const { return (m_right_bottom_point.get_y() - m_left_top_point.get_y()); }
   TwoDSize  get_size() const { return TwoDSize( get_width(), get_height()); }

   TwoDPoint get_left_top_corner() const { return m_left_top_point; }
   TwoDPoint get_right_bottom_corner() const { return m_right_bottom_point; }
   void set_left_top_corner( const TwoDPoint &point ) { m_left_top_point = point; calc_ratio( ); }
   void set_right_bottom_corner( const TwoDPoint &point ) { m_right_bottom_point = point; calc_ratio( ); }

   Rectangle& operator = ( const Rectangle &rect )
   { 
      m_left_top_point = rect.m_left_top_point;
      m_right_bottom_point = rect.m_right_bottom_point;
      calc_ratio( );
      return *this; 
   }

   bool operator == ( const Rectangle &rect ) const
   { 
      return ( m_left_top_point == rect.m_left_top_point && 
               m_right_bottom_point == rect.m_right_bottom_point); 
   } 

   Rectangle intersection( const Rectangle &rect );
   
   Rectangle unification( const Rectangle &rect )
   {
      OneDValue min_x = GGMIN( m_left_top_point.get_x(), rect.m_left_top_point.get_x() );
      OneDValue max_x = GGMAX( m_right_bottom_point.get_x(), rect.m_right_bottom_point.get_x() );

      OneDValue min_y = GGMIN( m_left_top_point.get_y(), rect.m_left_top_point.get_y() );
      OneDValue max_y = GGMAX( m_right_bottom_point.get_y(), rect.m_right_bottom_point.get_y() );

      return Rectangle( min_x, min_y, max_x, max_y );
   }
   
   TwoDPoint get_center() const
   {
      return TwoDPoint( (m_left_top_point.get_x() + m_right_bottom_point.get_x()) / 2,
                        (m_left_top_point.get_y() + m_right_bottom_point.get_y()) / 2);
   }

   bool inside( const TwoDPoint &point ) const
   {
      return ( point.get_x() >= m_left_top_point.get_x() && point.get_x() <= m_right_bottom_point.get_x() &&
               point.get_y() >= m_left_top_point.get_y() && point.get_y() <= m_right_bottom_point.get_y() );
   }

   OneDValue get_range() const
   {
      TwoDPoint center = get_center();

      return GGMAX( center.distance_to( m_right_bottom_point), center.distance_to( m_left_top_point) );
   }

   void set_center( const TwoDPoint &center );
   
   void set_range( OneDValue range );
   

   bool is_empty() const
   {
      return ( m_right_bottom_point == m_left_top_point );
   }

   TwoDPoint get_nearest_to( const TwoDPoint &point ) const;

   void draw( GraphicArea* g, bool filled = false );
   TwoDSize get_label_max_size( ) const{ return get_size(); }
};

//OS independent class for circle geometry representation
class Circle: public GeometryShape
{
protected:
   TwoDPoint m_center;
   OneDValue m_radius;
public:
   Circle() {}
   Circle( const TwoDPoint &center, OneDValue radius ): m_center( center ), m_radius( radius ) {}
   Circle( const Circle &circle )
   {
      m_center = circle.m_center;
      m_radius = circle.m_radius;
   }

   ~Circle(){};

   TwoDPoint get_center() const { return m_center; }

   OneDValue get_range() const { return m_radius; }

   Circle& operator = ( const Circle &circle )
   {
      m_center = circle.m_center;
      m_radius = circle.m_radius;

      return *this;
   }

   bool operator == ( const Circle &rect ) const
   {
      return ( m_center == rect.get_center() && m_radius == rect.get_range() ); 
   }
   
   void set_center( const TwoDPoint &center ) { m_center = center; }
   void set_range( OneDValue range ) { m_radius = range; }

   bool inside( const TwoDPoint &point ) const 
   { 
      return (m_center.distance_to( point ) <= m_radius );
   };

   TwoDPoint get_nearest_to( const TwoDPoint &point ) const
   {
      TwoDPoint res;

      if ( inside(point) )
         res = point;
      else
         res = LineSegment::get_scaled_line_point( m_center, point, (double)m_radius / (double)m_center.distance_to( point ) );

      return res;
   }

   bool is_empty() const { return (m_radius == 0); }
   virtual void draw( GraphicArea* g, bool filled = false );
   virtual TwoDSize get_label_max_size( ) const;
};

class DoubleCircle: public Circle
{
  void draw( GraphicArea* g, bool filled = false );
  TwoDSize get_label_max_size( ) const;
};

}; /* namespace GraphGraphics */


#endif
