
#ifndef _NODEATTR_H_
#define _NODEATTR_H_

#include <vector>
#include <map>
#include <list>
#include <set>
#include <string>
#include <cctype>
#include "storable.h"

using namespace std;

namespace GraphGraphics
{
/*************************************************************

**************************************************************/

class GraphAttrNameStorage;
  
/** Definition of attribute value type. */
class AttributeValue
{
public:
  enum ActualTypes
  {
    INTEGER = 0,
    INT_RANGE = 1,
    STRING = 2
  };
  
  enum AttrOperation
  {
    NONE        = 0,
    EQUAL       = 1,
    GREATER     = 2,
    LESS        = 3,
    NOT_LESS    = 4,
    NOT_GREATER = 5,
    NOT_EQUAL   = 6
  };
  
public:
  virtual ~AttributeValue() {}
  virtual string get_string() const = 0;
  virtual bool set_string( const string &str_value ) = 0;
  virtual long get_integer() const = 0;
  virtual int get_actual_type() const = 0;
  virtual bool compare( long v, unsigned char op = AttributeValue::EQUAL ) const = 0;
  virtual bool compare( const string &v, unsigned char op = AttributeValue::EQUAL ) const = 0;
};

class AttributeIntegerValue: public AttributeValue
{
private:
  long m_value;

public:
  AttributeIntegerValue( ): m_value(0) { }
  AttributeIntegerValue( long value ): m_value(value) { }
  virtual ~AttributeIntegerValue( ){ }
  bool set_string( const string &str_value );
  string get_string() const;
  void set_integer( long v ) { m_value = v; }
  long get_integer() const { return m_value; }
  int get_actual_type() const { return AttributeValue::INTEGER; }
  bool compare( long v, unsigned char op = AttributeValue::EQUAL ) const; 
  bool compare( const string &v, unsigned char op = AttributeValue::EQUAL ) const;
};

typedef set<long> ValueSet;

class AttributeIntegerRange: public AttributeValue
{
private:
  struct SimpleIntRange
  {
    long first_value;
    long last_value;
  
    SimpleIntRange() {}
    SimpleIntRange( long first, long last ): first_value( first ), last_value( last ) {}
    bool in_range( long v ) const { return (first_value <= v && v <= last_value); }
  };
  
typedef list<SimpleIntRange> RangeList;

  RangeList m_ranges;
  ValueSet  m_values;
public:
  AttributeIntegerRange( ) { }
  virtual ~AttributeIntegerRange( ){ }
  long get_integer() const 
  {
    long res = -1;
    if ( !m_values.empty() )
      res = *(m_values.begin());
    else if ( !m_ranges.empty() )
      res = m_ranges.begin()->first_value;      
    return res; 
  }
  int get_actual_type() const { return AttributeValue::INT_RANGE; }
  bool compare( long v, unsigned char op = AttributeValue::EQUAL ) const;
  bool compare( const string &v, unsigned char op = AttributeValue::EQUAL ) const;
  
  void add_int_range( long first_value, long last_value );
  void add_int_value( long v );
  
  void get_int_values( ValueSet &values ) { values = m_values; }
  string get_string() const;
  bool set_string( const string &str_range );
};

class AttributeStringValue: public AttributeValue
{
private:
  string m_value;

public:
  AttributeStringValue(): m_value("") {}
  AttributeStringValue( const string &value ): m_value(value) {}
  virtual ~AttributeStringValue(){}
    
  bool set_string( const string &str_value ) { m_value = str_value;}
  string get_string() const { return m_value; }
  void set_integer( long v );
  long get_integer() const;
  int get_actual_type() const { return AttributeValue::STRING; }
  bool compare( long v, unsigned char op = AttributeValue::EQUAL ) const;
  bool compare( const string &v, unsigned char op = AttributeValue::EQUAL ) const;
};
 
/**
* The structure represent simple graph attribute.
* Graph attribute is triple (<code>name</code>, <code>type</code>, <code>value</code>), where <code>name</code> is a name of attribute,
* <code>type</code> is name of type of attribute (it can be empty) and <code>value</code> is value of attribute (all values have an integer type).
* There are also two types of attributes: simple atribute and complex attribute (or meta-attribute). The diference is the meta-attribute can contain sub-attributes.
* Meta-attribute is represented as sequence of attributes that started from simple attribute with predefined type <code>NodeAttribute::ATTR_START</code> 
* and finised with predifined type <code>NodeAttribute::ATTR_END</code> (name and value of these two attributes should be the same). Meta-attributes can include other meta-attibutes.
* @see NodeAttributeList 
* @see GraphAttributes
*/
struct NodeAttribute
{
  static const string ATTR_START; /**< Predefined type for indicating meta-attribute beginning.*/
  static const string ATTR_END;   /**< Predefined type for indicating meta-attribute end.*/
  static const string ATTR_PRED_OPEN_PARENTHESIS;   
  static const string ATTR_PRED_CLOSE_PARENTHESIS;   
  static const string ATTR_PRED_AND;  
  static const string ATTR_PRED_OR;
  static const string ATTR_PRED_NOT;    
  
  string         name; /**< Name of the attribute (empty value is valid in some cases).*/
  string         type; /**< Name of type of attribute (empty value is valid in some cases). */
  /** Value of attribute or index of meta-atribute. 
  *   In case of meta-attribute index -1 means empty index.
  * @see NodeAttributeList
  */
  unsigned char  value_op;
  AttributeValue *value;

  NodeAttribute( const NodeAttribute &attr );
    
  NodeAttribute() { value = 0; }
  ~NodeAttribute() { if (value != 0) delete value; }
};

typedef vector<NodeAttribute*> NAPvector;

/**
* The class represent sequence of simple attributes (usually used for representing node meta-attribute).
* There are several alternative ways of representing graph attributes for using in application.
* One of them is string representation.
* Attribute string format.
*
*  <p> <node attr.>      :: <metaattr.> </p>
*  <p> <metaattr.>       :: <id> ':' <SEP.LIST> <name> <SEP.LIST> '{' <SEP.LIST> <attr. list> <SEP.LIST> '}' |
*                       <name> <SEP.LIST> '{' <SEP.LIST> <attr. list> <SEP.LIST> '}' |
*                       <id> ':' <SEP.LIST> '{' <SEP.LIST> <attr. list> <SEP.LIST> '}' |
*                       '{' <SEP.LIST> <attr. list> <SEP.LIST> '}' </p>
*  <p> <attr. list>      :: 'empty' |
*                       <attr.> <SEP.LIST> <attr. list> </p>
*  <p> <attr.>           :: <metaattr.> |
*                       <type>':'<SEP.LIST><name><SEP.LIST>'='<SEP.LIST><value>';' |
*                       <name><SEP.LIST>'='<SEP.LIST><value> </p>
*  <p> <SEP.LIST>        :: 'empty' |
*                       <SEP.><SEP.LIST> </p>
*  <p> <SEP.>            :: '\t' | '\n' | ' ' </p>
*  <p> <id>              :: <integer number> </p>
*  <p> <value>           :: <integer number> </p>
*  <p> <name>            :: <identifier> </p>
*  <p> <type>            :: <identifier> </p>
*  <p> <identifier>      :: <first char><chars> </p>
*  <p> <first char>      :: '_' | 'a' | ... | 'z' | 'A' | ... | 'Z' </p>
*  <p> <chars>           :: 'empty' | 
*                       <char><chars> </p>
*  <p> <char>            :: 'a' | ... | 'z' | 'A' | ... | 'Z' | '0' | ... | '9' | '_' | '.' | '[' | ']' </p>
*
*  For example: "Metaattr._name1 { int: attr_name1=10; 2: Metaattr._name[2] { attr_name2 = 0;} }",
*               "{}", "1: { { attr_type1: attr_name[1] = 1;} attr_name.2=4;}"
*
*  Another way is representing node attributes as <b>NodeAttributeList</b> - it is vector of pointers to <b>NodeAttribute</b>.
*  Valid sequence of simple attributes is similar to attribute string format.
*
*  <p> <node attr.>      :: <metaattr.> </p>
*  <p> <metaattr.>       :: <NodeAttribute with type ATTR_START><attr. list><NodeAttribute with type ATTR_END> </p>
*  <p> <attr. list>      :: 'empty' |
*                       <NodeAttribute> <attr. list> |
*                       <metaattr.> <attr. list>       </p>
*  @see GraphAttributes
*  @see NodeAttribute
*/
class  NodeAttributeList: public NAPvector
{
public:
  
  /**
  * Parameterless constructor.
  */
  NodeAttributeList() {}
    
  /**
  * Copy constructor.
  */
  NodeAttributeList( const NodeAttributeList& attr_list )
  {
    NodeAttributeList::const_iterator p = attr_list.begin();
    NodeAttribute* attr;
    clear();
    
    for (; p != attr_list.end(); p++ )
    {
      attr = new NodeAttribute( *(*p) );
      push_back( attr );
    }
  }
  
  /**
  * Assigning operator. 
  */
  NodeAttributeList& operator = ( const NodeAttributeList& attr_list )
  {
    NodeAttributeList::const_iterator p = attr_list.begin();
    NodeAttribute* attr;
    clear();
    
    for (; p != attr_list.end(); p++ )
    {
      attr = new NodeAttribute( *(*p) );
      push_back( attr );
    }
    
    return *this;
  }

  void pop_back()
  {
    NAPvector::iterator p = NAPvector::end();
     
    if ( p != NAPvector::begin() )
    {
      NodeAttribute* attr = (NodeAttribute*)(*--p);

      if ( attr != 0 )
        delete attr;
    }
   
    NAPvector::pop_back();
  }
  
  void clear()
  {
    NodeAttribute* attr = 0;
    for ( NAPvector::iterator p = NAPvector::begin(); p != NAPvector::end(); p++ )
      if ( (attr = (NodeAttribute*)(*p)) != 0 )
        delete attr;

    NAPvector::clear();
  }
   
  /**
  * Destructor.
  */
  ~NodeAttributeList()
  {
    clear();
  }
  
  /**
  * The method writes attributes to a stream (in binary format).
  * @see Storable
  */  
  void write_bin_to( ostream& s_out, GraphAttrNameStorage *storage, unsigned long write_options = 0 );
  
  /**
  * The method reads attributes from a stream (in binary format).
  * @see Storable
  */
  bool read_bin_from( istream& s_in, GraphAttrNameStorage *storage, unsigned long read_options = 0 );
};

/**********************************************************************/
class Lexeme
{
public:
   //lexemes types
   enum LexTypes
   {
     TEOF   = -1,
     TERROR = 0,
   
     TUNKNOWN = 1,
     TCOL     = 2,
     TSEMI    = 3,
     TMETASTART = 4,
     TMETAEND = 5,
     TVALUEOP = 6,
     TIDENTIFIER = 7,
     TNUM     = 8,
     TNAME    = 9,
     TMETANAME = 10,
     TTYPE     = 11,
     TID       = 12,
     TVALUE    = 13,
     TRANGE    = 14,
     TPREDBINOP = 15,
     TPREDUNOP = 16,
     TOPPAREN  = 17,
     TCLPAREN  = 18,
     TEQ       = 20,
     TLS       = 21,
     TGT       = 22,
     TGTE      = 23,
     TLSE      = 24,
     TNEQ      = 25,
     TOR       = 26,
     TAND      = 27,
     TNOT      = 28,
     TSTRING   = 29
   };

   int      pos;
   int      next_pos;
   LexTypes type;
   LexTypes sub_type;
   string   name;

   Lexeme(): pos(0), next_pos(0), type(TERROR), sub_type(TERROR), name(""){}
   Lexeme( int _pos, int _next_pos, LexTypes _type, Lexeme::LexTypes _sub_type, string _name ): 
                  pos(_pos), next_pos(_next_pos), type(_type), sub_type(_sub_type), name(_name) {}

   Lexeme( const Lexeme &lex )
   {
      pos = lex.pos;
      next_pos = lex.next_pos;
      type = lex.type;
      sub_type = lex.sub_type;
      name = lex.name;
   }

   Lexeme& operator = ( const Lexeme &lex )
   {
      pos = lex.pos;
      next_pos = lex.next_pos;
      type = lex.type;
      sub_type = lex.sub_type;
      name = lex.name;

      return *this;
   }
};

/******************************************************************************/
/******************************************************************************/
typedef char   AttrTypeID;
typedef int    AttrNameID;
typedef vector<string> StringContainer;
typedef map<string, AttrTypeID> TypeStringMap;
typedef map<string, AttrNameID> NameStringMap;
typedef TypeStringMap::const_iterator TypeCI;
typedef NameStringMap::const_iterator NameCI;
typedef vector<AttrNameID> AttrNameIDVector;

typedef map<long, string> GraphAttrValueAliasMap;

class GraphAttrAliases
{
static const int     EMPTY_ALIAS_ID;
static const string  EMPTY_NAME;
  
private:
  
typedef int AliasID;
typedef map<string, AliasID> AliasNameIDMap;
typedef vector<AliasID>      AliasIDContainer;

typedef map<long, AliasID> AttrValueAliasMap;
typedef map<AliasID, long> AliasAttrValueMap;

  struct AttrAliases
  {
    AliasIDContainer  name_alias;
    AttrTypeID        type;
    AttrValueAliasMap value_aliases;
    AliasAttrValueMap value_map;
  };
  
typedef map<AttrNameID, AttrAliases> AttrAliasesMap;
struct MetaAttrAliases;
typedef map<AttrNameID, MetaAttrAliases*> MetaAliasesMap;
  
  struct MetaAttrAliases
  {
    AliasIDContainer name_alias;
    
    MetaAttrAliases *parent_meta;
    MetaAliasesMap   meta_aliases;
    AttrAliasesMap   attr_aliases;
    
    MetaAttrAliases(){ parent_meta = 0; }
      
    void clear( )
    {
      for (MetaAliasesMap::iterator it = meta_aliases.begin(); it != meta_aliases.end(); it++ )
        delete (MetaAttrAliases*)(it->second);
      
      parent_meta = 0;
      meta_aliases.clear();
      name_alias.clear();
      attr_aliases.clear();
    }
    ~MetaAttrAliases() { clear(); }
    
    void copy( const MetaAttrAliases& aliases )
    {
      name_alias = aliases.name_alias;
      attr_aliases = aliases.attr_aliases;
      
      for (MetaAliasesMap::const_iterator it = aliases.meta_aliases.begin(); it != aliases.meta_aliases.end(); it++ )
      {
        MetaAttrAliases *meta = new MetaAttrAliases();
        meta->copy( *(it->second) );
        meta->parent_meta = this;
        meta_aliases.insert( MetaAliasesMap::value_type(it->first, meta ) );
      }
    }
    
    MetaAttrAliases( const MetaAttrAliases &aliases ) { copy( aliases ); }
  };
  
  
typedef map<AliasID, AttrNameIDVector> AliasAttrNameMap;
struct AliasMetaNames;
typedef map<AliasID, AliasMetaNames*>  AleasMetaNameMap;
  
  struct AliasMetaNames
  {
    AttrNameIDVector  name;
    
    AliasMetaNames   *parent_meta;
    AleasMetaNameMap  meta_map;
    AliasAttrNameMap  attr_map;
    
    AliasMetaNames(){ parent_meta = 0;}
      
    void clear( )
    {
      for (AleasMetaNameMap::iterator it = meta_map.begin(); it != meta_map.end(); it++ )
        delete (AliasMetaNames*)(it->second);
      
      meta_map.clear();
      name.clear();
      attr_map.clear();
      parent_meta = 0;
    }
    ~AliasMetaNames() { clear(); }
    
    void copy( const AliasMetaNames& names )
    {
      name = names.name;
      attr_map = names.attr_map;
      
      for (AleasMetaNameMap::const_iterator it = names.meta_map.begin(); it != names.meta_map.end(); it++ )
      {
        AliasMetaNames *meta = new AliasMetaNames();
        meta->copy( *(it->second) );
        meta->parent_meta = this;
        meta_map.insert( AleasMetaNameMap::value_type(it->first, meta ) );
      }
    }
    
    AliasMetaNames( const AliasMetaNames &names ) { copy( names ); }
  };
  
  struct TypeAliases
  {
    AttrValueAliasMap value_aliases; 
    AliasAttrValueMap value_map;
  };
  
typedef map<AttrTypeID, TypeAliases> AttrTypeAliasMap;
  
private:  
  //aliases
  StringContainer    m_aliases;
  AliasNameIDMap     m_alias_id_map;
  
  // type value aliases
  AttrTypeAliasMap   m_type_alias_map;
  
  // attr. name/value aliases
  MetaAttrAliases    m_attr_alias_map;
  AliasMetaNames     m_alias_attr_map;

  //current level
  MetaAttrAliases*   m_curr_attr_level; 
  AliasMetaNames*    m_curr_alias_level;

private:
  AliasID get_alias_id( const string &alias )
  {
    AliasID id = EMPTY_ALIAS_ID;

    if ( alias != EMPTY_NAME )
    {
      AliasNameIDMap::iterator ait;
      if ( (ait = m_alias_id_map.find( alias )) != m_alias_id_map.end() )
        id = ait->second;
      else
      {
        id = m_aliases.size();
        m_aliases.push_back( alias );
        m_alias_id_map[ alias ] = id;
      }
    }

    return id;
  }
  
  string get_alias_name( AliasID id )
  {
    string alias = EMPTY_NAME;

    if ( id < (int)m_aliases.size() && id >= 0 )
      alias = m_aliases[ id ];

    return alias;
  }
  
  MetaAttrAliases* get_meta_alias( MetaAttrAliases *meta, AttrNameID meta_id, bool create = true )
  {
    MetaAttrAliases *res = 0;
    MetaAliasesMap::iterator p = meta->meta_aliases.find( meta_id ); 
    
    if ( p != meta->meta_aliases.end() )
      res = p->second;
    else if ( create )
    {
      res = new MetaAttrAliases(); 
      res->parent_meta = meta;
      meta->meta_aliases[meta_id] = res;
    }
    
    return res;
  }
  
  AliasMetaNames* get_alias_meta( AliasMetaNames *meta, AliasID meta_id, bool create = true )
  {
    AliasMetaNames *res = 0;
    AleasMetaNameMap::iterator p = meta->meta_map.find( meta_id ); 
    
    if ( p != meta->meta_map.end() )
      res = p->second;
    else if ( create )
    {
      res = new AliasMetaNames(); 
      res->parent_meta = meta;
      meta->meta_map[meta_id] = res;
    }
    
    return res;
  }
  
  void set_name_alias( MetaAttrAliases *meta_level, const AttrNameIDVector &attr_name, const AliasIDContainer &alias_ids, AttrTypeID type, bool is_meta = false )
  {
    MetaAttrAliases *meta = meta_level;
    unsigned int     ind = 1;
   
    for ( ; ind < attr_name.size() - 1; ind++ )
      meta = get_meta_alias( meta, attr_name[ind] );       
    
    //add alias    
    if ( is_meta )
    {
      if ( ind < attr_name.size() )
        meta = get_meta_alias( meta, attr_name[ind] );
      meta->name_alias = alias_ids;
    }
    else
    {
      AttrAliasesMap::iterator p = meta->attr_aliases.find( attr_name[ind] );
          
      if ( p == meta->attr_aliases.end() )
        p = (meta->attr_aliases.insert(AttrAliasesMap::value_type( attr_name[ind], AttrAliases() ))).first;
      
      p->second.name_alias = alias_ids;
      p->second.type = type;
    }
  }
  
  void set_alias_name( AliasMetaNames *meta_level, const AliasIDContainer &alias_ids, const AttrNameIDVector &attr_name, bool is_meta = false )
  {
    AliasMetaNames *meta = meta_level;
    unsigned int    ind = 1;
    
    for ( ; ind < alias_ids.size() - 1; ind++ )
      meta = get_alias_meta( meta, alias_ids[ind] );
      
    if ( is_meta )
    {
      if ( ind < alias_ids.size() )
        meta = get_alias_meta( meta, alias_ids[ind] );
      meta->name = attr_name;
    }
    else
      meta->attr_map[alias_ids[ind]] = attr_name;
  }
  
public:
  GraphAttrAliases()
  { 
    m_curr_alias_level = &m_alias_attr_map; 
    m_curr_attr_level = &m_attr_alias_map; 
  }
  ~GraphAttrAliases(){}
    
  void clear();
    
  GraphAttrAliases( const GraphAttrAliases &aliases );
  
  GraphAttrAliases& operator = ( const GraphAttrAliases &aliases );
   
  bool get_attr_value( const AttrNameIDVector &attr_name, AttrTypeID type_id, const string &value_alias, long &value );
   
  void open_top_alias_level() { m_curr_alias_level = &m_alias_attr_map; }
  bool open_alias_level( const string &attr_alias );
  bool get_meta_attr_name( AttrNameIDVector &meta_attr_name );
  bool get_attr_name( const string &name_alias, AttrNameIDVector &attr_name );
  bool get_attr_name( const StringContainer &name_alias, AttrNameIDVector &attr_name );
  bool get_attr_value( const string &name_alias, AttrNameID type_id, const string &value_alias, long &res_value )
  {
    AttrNameIDVector attr_name;
    bool res = get_attr_name( name_alias, attr_name );
    
    if ( res )
      res = get_attr_value( attr_name, type_id, value_alias, res_value );
    
    return res;
  }
  
  void close_alias_level()
  {
    if ( m_curr_alias_level->parent_meta != 0 )
      m_curr_alias_level = m_curr_alias_level->parent_meta;
  }
    
  void open_top_attr_level() { m_curr_attr_level = &m_attr_alias_map; }
  bool open_attr_level( AttrNameID attr_name )
  {
    MetaAttrAliases *res = get_meta_alias( m_curr_attr_level, attr_name, false );
    
    if ( res != 0 )
      m_curr_attr_level = res;
    return (res != 0);
  }
  
  bool get_meta_attr_name_alias( StringContainer &alias );
  
  bool get_attr_name_alias( AttrNameID attr_name_id, StringContainer &alias );
    
  bool get_attr_value( AttrNameID attr_name_id, AttrTypeID type_id, const string &value_alias, long &value );
  string get_attr_value_alias( AttrNameID attr_name_id, AttrTypeID type_id, long value );
  
  void close_attr_level()
  {
    if ( m_curr_attr_level->parent_meta != 0 )
      m_curr_attr_level = m_curr_attr_level->parent_meta;
  }
    
  void set_meta_attr_name_alias( const AttrNameIDVector &attr_name, const StringContainer &alias );
  void set_attr_name_alias( const AttrNameIDVector &attr_name, AttrTypeID type, const StringContainer &alias );
  void set_attr_value_alias( const AttrNameIDVector &attr_name, long value, const string &value_alias );
  void set_attr_type_value_alias( AttrTypeID type, long value, const string &value_alias );
  
  bool get_attr_value_alases( const AttrNameIDVector &attr_name, AttrTypeID type, GraphAttrValueAliasMap &alias_map );
  
};

typedef vector<GraphAttrAliases> GraphAliasesContainer;


/**
* The class represents graph attribute names storage.
* For optimization purposes all names that are used by graph attributes (attribute names and attribute type names) are stored in one storage (as far as usually nodes use the same set of the names).
* All node attributes just refer to the storage of appropriate graph and store only integer identifiers.
* @see GraphAttributes
* @see Graph::get_graph_attr_name_storage()
*/
class GraphAttrNameStorage: public Storable
{
public:
   static const int     EMPTY_ID;
   static const string  EMPTY_NAME;

private:
  
   StringContainer   m_names;
   StringContainer   m_types;

   TypeStringMap m_type_id_map;

   NameStringMap m_name_id_map;

   NodeAttributeList  m_search_format;

   GraphAliasesContainer m_aliases;
   int                   m_curr_aliases;
  
   set<AttrTypeID>       m_string_types;

public:
   GraphAttrNameStorage();
   ~GraphAttrNameStorage() {}
     
   /**
   *  The method adds a name to name storage.
   *  If the name is already exists it will not be added.
   *  @param name The name to be added.
   *  @return Identifier of the name.
   */
   inline int add_name( const string &name )
   {
      int id = EMPTY_ID;

      if ( name != EMPTY_NAME )
      {
         if ( m_name_id_map.find( name ) != m_name_id_map.end() )
            id = m_name_id_map[ name ];
         else
         {
            id = m_names.size();
            m_names.push_back( name );
            m_name_id_map[ name ] = id;
         }
      }

      return id;
   }

   /**
   *  The method returns the name identifier.
   *  @param name The name which identifier is requested.
   *  @return The name identifier or <b>EMPTY_ID</b> if there is no the name in storage. 
   */
   inline int get_name_id( const string &name )
   {
      int id = EMPTY_ID;
      NameCI p;

      if ( name != EMPTY_NAME)
        if ( (p = m_name_id_map.find( name )) != m_name_id_map.end() )
          id = p->second;

      return id;
   }

   /**
   *  The method returns name gotten by its identifier. 
   *  @param id The name identifier to get the name.
   *  @return The name or <b>EMPTY_NAME</b>. 
   */
   inline string get_id_name( int id )
   {
      string name = EMPTY_NAME;

      if ( id < (int)m_names.size() && id >= 0 )
         name = m_names[ id ];

      return name;
   }

   /**
   *  The method adds a type name to type name storage.
   *  If the type name is already exists it will not be added.
   *  @param type The type name to be added.
   *  @return Identifier of the type name.
   */
   inline int add_type( const string &type )
   {
      int id = EMPTY_ID;

      if ( type != EMPTY_NAME )
      {
         if ( m_type_id_map.find( type ) != m_type_id_map.end() )
            id = m_type_id_map[ type ];
         else
         {
            id = m_types.size();
            m_types.push_back( type );
            m_type_id_map[ type ] = id;
         }
      }

      return id;
   }

   /**
   *  The method returns the type name identifier.
   *  @param type The type name which identifier is requested.
   *  @return The type name identifier or <b>EMPTY_ID</b> if there is no the name in storage. 
   */
   inline int get_type_id( const string &type )
   {
      int id = EMPTY_ID;
      TypeCI p;

      if ( type != EMPTY_NAME )
        if ( (p = m_type_id_map.find( type )) != m_type_id_map.end() )
          id = p->second;
        

      return id;
   }

   /**
   *  The method returns type name gotten by its identifier. 
   *  @param id The type name identifier to get the type.
   *  @return The type name or <b>EMPTY_NAME</b>. 
   */
   inline string get_id_type( int id )
   {
      string type = EMPTY_NAME;

      if ( id < (int)m_types.size() && id >= 0 )
         type = m_types[ id ];

      return type;
   }
   
   inline bool is_string_attr_type( AttrTypeID type ) 
   { return m_string_types.find(type) != m_string_types.end(); }
   
   /**
   *  The method initialize name storage using attribute template string.
   *  @param search_str_format This attribute template is stored for possible using as template for creating attributes search pattern.
   *  @return <b>true</b> if initialiszing was successful.
   */
   bool init_name_set( const string &search_str_format );

   /**
   *  The method removes all names from the storage.
   */
   void clear_name_set( )
   {
      m_names.clear();
      m_names.resize(0);
      m_types.clear();
      m_types.resize(0);
      m_type_id_map.clear();
      m_name_id_map.clear();
      
      m_search_format.clear();
      m_search_format.resize(0);
      m_aliases.clear();
      m_curr_aliases = -1;
      m_string_types.clear();
   }

   /**
   *  The method returns attributes template string.
   *  @see ::init_name_set(...)
   */
   string  get_search_format_string();
   
   /**
   *  The method returns attributes template in parsed form (represented by <code>NodeAttributeList</code>).
   *  @see ::init_name_set(...)
   */
   void  get_search_format_parsed( NodeAttributeList& attr_list )
   {
      attr_list.clear();
      attr_list.resize(0);
      for ( unsigned int i = 0; i < m_search_format.size(); i++ )
          attr_list.push_back( m_search_format[i] );
   }
   
   int add_new_alias_set( )
   {
      m_aliases.push_back( GraphAttrAliases() );
     
     return m_aliases.size() - 1;
   }
   
   int get_curr_alias_set() { return m_curr_aliases; }
   void set_curr_alias_set( int id )
   {
     if ( id < 0 || id >= (int)m_aliases.size() )
       m_curr_aliases = -1;
     else
       m_curr_aliases = id;
   }
   
   bool get_attr_value( const StringContainer &attr_name, const string &type, const string &value_alias, long &value )
   {
     bool res = false;
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameIDVector attr_name_id;
       AttrNameID       id;
       AttrTypeID       type_id = get_type_id( type );
       
       for ( StringContainer::const_iterator it = attr_name.begin(); it != attr_name.end(); it++ )
       {
         if ( (id = get_name_id(*it)) != EMPTY_ID )
           attr_name_id.push_back( id ); 
       }
       if ( !attr_name_id.empty() )
         res = m_aliases[m_curr_aliases].get_attr_value( attr_name_id, type_id, value_alias, value );
     }
     return res;
   }
   
   void open_top_alias_level() 
   {
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
       m_aliases[m_curr_aliases].open_top_alias_level();
   }
   
   bool open_alias_level( const string &attr_alias )
   {
     bool res = false;
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
       res = m_aliases[m_curr_aliases].open_alias_level( attr_alias );
     return res;
   }
   
   bool get_meta_attr_name( StringContainer &meta_attr_name )
   {
     bool res = false;
     meta_attr_name.clear();
     
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameIDVector attr_name_id;
       res = m_aliases[m_curr_aliases].get_meta_attr_name( attr_name_id );
       
       if ( res )
         for ( AttrNameIDVector::const_iterator it = attr_name_id.begin(); it != attr_name_id.end(); it++ )
           meta_attr_name.push_back( get_id_name( *it ) );
     }
     
     return res;
   }
   
   bool get_attr_name( const string &name_alias, StringContainer &attr_name )
   {
     bool res = false;
     attr_name.clear();
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameIDVector attr_name_id;
       res = m_aliases[m_curr_aliases].get_attr_name( name_alias, attr_name_id );
       
       if ( res )
         for ( AttrNameIDVector::const_iterator it = attr_name_id.begin(); it != attr_name_id.end(); it++ )
           attr_name.push_back( get_id_name( *it ) );
     }
     
     return res;
   }
   
   bool get_attr_value( const string &name_alias, const string &type, const string &value_alias, long &res_value )
   {
     bool res = false;
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrTypeID type_id = get_type_id( type );
       res = m_aliases[m_curr_aliases].get_attr_value( name_alias, type_id, value_alias, res_value );
     }
     return res;
   }
  
   void close_alias_level()
   {
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
       m_aliases[m_curr_aliases].close_alias_level();
   }
  
   void open_top_attr_level() 
   {
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
       m_aliases[m_curr_aliases].open_top_attr_level();
   }
   
   bool open_attr_level( const string &attr_name )
   {
     bool res = false;
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameID attr_name_id = get_name_id( attr_name );
       res = m_aliases[m_curr_aliases].open_attr_level( attr_name_id );
     }
     return res;
   }
   
   bool get_meta_attr_name_alias( StringContainer &alias )
   {
     bool res = false;
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
       res = m_aliases[m_curr_aliases].get_meta_attr_name_alias( alias );
     return res;
   }
    
   bool get_attr_name_alias( const string &attr_name, StringContainer &alias )
   {
     bool res = false;
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameID attr_name_id = get_name_id( attr_name );
       res = m_aliases[m_curr_aliases].get_attr_name_alias( attr_name_id, alias );
     }
     return res;
   }
    
   bool get_attr_value_by_name( const string &attr_name, const string &type, const string &value_alias, long &value )
   {
     bool res = false;
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameID attr_name_id = get_name_id( attr_name );
       AttrTypeID type_id = get_type_id( type );
       res = m_aliases[m_curr_aliases].get_attr_value( attr_name_id, type_id, value_alias, value );
     }
     return res;
   }
   
   string get_attr_value_alias( const string &attr_name, const string &type, long value )
   {
     string res = "";
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameID attr_name_id = get_name_id( attr_name );
       AttrTypeID type_id = get_type_id( type );
       res = m_aliases[m_curr_aliases].get_attr_value_alias( attr_name_id, type_id, value );
     }
     return res;
   }
  
   void close_attr_level()
   {
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
       m_aliases[m_curr_aliases].close_attr_level();
   }
  
   void set_meta_attr_name_alias( const StringContainer &attr_name, const StringContainer &alias )
   {
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameIDVector attr_id_name;
       AttrNameID       id;
       
       for ( StringContainer::const_iterator it = attr_name.begin(); it != attr_name.end(); it++ )
       {
         if ( (id = get_name_id(*it)) != EMPTY_ID )
           attr_id_name.push_back( id ); 
       }
       
       if ( !attr_id_name.empty() )
         m_aliases[m_curr_aliases].set_meta_attr_name_alias( attr_id_name, alias );
     }
   }
   
   void set_attr_name_alias( const StringContainer &attr_name, const string &type, const StringContainer &alias )
   {
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameIDVector attr_id_name;
       AttrNameID       id;
       AttrTypeID       type_id;
       
       for ( StringContainer::const_iterator it = attr_name.begin(); it != attr_name.end(); it++ )
       {
         if ( (id = get_name_id(*it)) != EMPTY_ID )
           attr_id_name.push_back( id ); 
       }
              
       if ( !attr_id_name.empty() )
       {
         type_id = get_type_id( type );
         m_aliases[m_curr_aliases].set_attr_name_alias( attr_id_name, type_id, alias );
       }
     }
   }
   
   void set_attr_value_alias( const StringContainer &attr_name, long value, const string &value_alias )
   {
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameIDVector attr_id_name;
       AttrNameID       id;
              
       for ( StringContainer::const_iterator it = attr_name.begin(); it != attr_name.end(); it++ )
       {
         if ( (id = get_name_id(*it)) != EMPTY_ID )
           attr_id_name.push_back( id ); 
       }
              
       if ( !attr_id_name.empty() )
         m_aliases[m_curr_aliases].set_attr_value_alias( attr_id_name, value, value_alias );
     }
   }
   
   void set_attr_type_value_alias( const string &type, long value, const string &value_alias )
   {
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrTypeID  type_id;
       
       if ( (type_id = get_type_id( type )) != EMPTY_ID )
         m_aliases[m_curr_aliases].set_attr_type_value_alias( type_id, value, value_alias );
     }       
   }
   
   bool get_attr_value_alases( const StringContainer &attr_name, const string &type, GraphAttrValueAliasMap &alias_map )
   {
     bool res = false;
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameIDVector attr_id_name;
       AttrNameID       id;
       AttrTypeID       type_id = get_type_id( type );
              
       for ( StringContainer::const_iterator it = attr_name.begin(); it != attr_name.end(); it++ )
       {
         if ( (id = get_name_id(*it)) != EMPTY_ID )
           attr_id_name.push_back( id ); 
       }
              
       if ( !attr_id_name.empty() )
         res = m_aliases[m_curr_aliases].get_attr_value_alases( attr_id_name, type_id, alias_map );
     }
     return res;
   }
   
   bool get_attr_value_alases_by_name_alias( const StringContainer &alias, const string &type, GraphAttrValueAliasMap &alias_map )
   {
     bool res = false;
     if ( m_curr_aliases >= 0 && m_curr_aliases < (int)m_aliases.size() )
     {
       AttrNameIDVector attr_id_name;
       AttrTypeID       type_id = get_type_id( type );
       
       res = m_aliases[m_curr_aliases].get_attr_name( alias, attr_id_name );
              
       if ( res )
         m_aliases[m_curr_aliases].get_attr_value_alases( attr_id_name, type_id, alias_map );
     }
     return res;
   }
   
   /**
   *  The method writes the name storage to a stream.
   *  @see Storable
   */
   void write_to( ostream& s_out, unsigned long write_options = 0 );
   
   /**
   *  The method reads the name storage from a stream.
   *  @see Storable
   */
   bool read_from( istream& s_in, unsigned long read_options = 0 );
};

/******************************************************************************/

/**
* The class represents graph node attributes storage.
* It designed for storing graph node attributes in tree-like form.
* Any node can have one meta-attribute that contain all necessery information about the node.
* The class also provides necessary functionality for parsing/making attribute string or <b>NodeAttributeList</b>,
* comparing attributes with a patern, geting separate values etc.
* @see NodeAttributeList
*/
class GraphAttributes
{
friend class GraphAttrNameStorage;
  
private:
  //parser constants
   static string           STAB;
   static string           SEOL;
   static const char       CSEMI;
   static const char       CCOL;
   static const char       CDOT;
   static const char       CCOMMA;
   static const char       CMETASTART;
   static const char       CMETAEND;
   static const char       CEQ;
   static const char       CLS;
   static const char       CGT;
   static const char       CNOT;
   static const char       CAND;
   static const char       COR;
   static const char       COPENPAREN;
   static const char       CCLOSEPAREN;
   static const char       CQUOTATION;

   static const string     SGTE;
   static const string     SLSE;
   static const string     SNEQ;

private:

   struct Attribute
   {
      AttrTypeID     type;
      AttrNameID     name;
      long           value;

      Attribute()
      {
         value = 0;
         type  = GraphAttrNameStorage::EMPTY_ID;
         name  = GraphAttrNameStorage::EMPTY_ID;
      }
   };

   class MetaAttribute
   {
   public:
      AttrNameID      name;
      int             meta_attr_count;
      MetaAttribute **meta_attr;
      int             attr_count;
      Attribute      *attr; //sorted by name

      MetaAttribute()
      {
         meta_attr         = 0;
         meta_attr_count   = 0;
         name              = GraphAttrNameStorage::EMPTY_ID;
         attr_count        = 0;
         attr              = 0;
      }
	  
      void clear()
      {
         for ( int i = 0; i < meta_attr_count; i++ )
                delete meta_attr[i];
    
             delete[] meta_attr;
             delete[] attr;
         
         meta_attr = 0;
         attr = 0;
         attr_count = 0;
         meta_attr_count = 0;
         name = GraphAttrNameStorage::EMPTY_ID;
      }

      ~MetaAttribute()
      {
         clear();
      }

      int find_attribute( AttrNameID name_id )
      {
         int  first = 0;
         int  last  = attr_count - 1;
         bool match = false;
         int  curr = first;

         while ( first <= last && !match)
         {
            curr = (last + first) / 2;
            if ( attr[curr].name == name_id )
               match = true;
            else if ( attr[curr].name < name_id )
               first = curr + 1;
            else
               last = curr - 1;
         }
         
         return ((match)? curr: GraphAttrNameStorage::EMPTY_ID);
      }

      int find_metaatribute( AttrNameID name_id, int start_ind )
      {
         int i = start_ind;

         if ( name_id != GraphAttrNameStorage::EMPTY_ID )
            for ( ;i < meta_attr_count; i++)
               if ( meta_attr[i]->name == name_id )
                  break;

         return (i < meta_attr_count)? i: GraphAttrNameStorage::EMPTY_ID;
      }
   };
   
   typedef vector<MetaAttribute *> MetaAttrContainer;
   typedef vector<Attribute *> AttrContainer;

   
   class AttrP_less: public binary_function<Attribute*, Attribute*, bool>
	 {
	 public:
		  bool operator () ( Attribute *a1, Attribute *a2 )
		  {
			  return (a1->name < a2->name);
		  }
	 };

private:
   
   inline static int skip_separators( const string &parsed_str, int ind )
   {
      int len = parsed_str.length();
     
      if ( ind == Lexeme::TEOF )
        ind = len;
      
      for ( ; ind < len && isspace( (unsigned char)parsed_str[ind] ); ind++ );

      return ((ind < len)? ind: Lexeme::TEOF);
   }

   inline static bool is_first_id_char( int c )
   {
      return ( c == '_' || isalpha( c ) );
   }

   inline static Lexeme get_identifier( const string &parsed_str, int ind );

   inline static Lexeme get_number( const string &parsed_str, int ind );
   inline static Lexeme get_string( const string &parsed_str, int ind );
   
   static Lexeme get_range( const string &parsed_str, int ind, const Lexeme &first_num = Lexeme( ) );
   
   static Lexeme get_lex( const string &parsed_str, int ind );

   static Lexeme parse_meta( const string &parsed_str, const Lexeme &prev_lex, NodeAttributeList &attr_list, bool predicate = false );
   static Lexeme parse_attrlist( const string &parsed_str, const Lexeme &prev_lex, NodeAttributeList &attr_list, bool predicate = false );
   static Lexeme parse_pred_attrlist( const string &parsed_str, const Lexeme &prev_lex, NodeAttributeList &attr_list );

   

public:
   /**
   *  This static method parses graph attributes string and returns them as sequence of <code>NodeAttribute</code>.
   *  @param str Graph attributes in string format to be parsed.
   *  @param attr_list <code>NodeAttributeList</code> for getting result of parsing.
   *  @return <b>true</b> if str is valid graph attributes string.
   *  @see ::make_string(...)
   */
   static bool   parse_string( const string &str, NodeAttributeList &attr_list );
   
   /**
   *  This static method converts graph attributes from <code>NodeAttributeList</code> to string format.
   *  @param attr_list The constant reference to <code>NodeAttributeList</code> object that contains graph attributes for string making.
   *  @return The graph attrinutes string if successful or empty string.
   */
   static string make_string( const NodeAttributeList &attr_list );
      
private:
   StringContainer   m_strings;
   
   MetaAttribute m_attr_list;
   int           m_attr_ind;
   GraphAttrNameStorage *m_name_storage;

private:
   int find_meta_end( const NodeAttributeList &attr_list, int ind, int last_ind )
   {
      int         stack = 0;
      
      for( ;ind <= last_ind; ind++ )
      {
        if ( attr_list[ind]->type == NodeAttribute::ATTR_START )
            stack++;
        else if ( attr_list[ind]->type == NodeAttribute::ATTR_END )
        {
          stack--;
          if ( stack == 0 )
            break;
        }
      }
      
      return (ind <= last_ind)? ind: GraphAttrNameStorage::EMPTY_ID;
   }
   
   bool calc_attr_pred( MetaAttribute *meta, const NodeAttribute *attr )
   {
     bool res = false;
     int  attr_id = meta->find_attribute( m_name_storage->get_name_id( attr->name ) );
  
     if ( attr_id != GraphAttrNameStorage::EMPTY_ID )
       if ( m_name_storage->is_string_attr_type( meta->attr[attr_id].type ) )
         res = attr->value->compare( m_strings[meta->attr[attr_id].value], attr->value_op );
       else
         res = attr->value->compare( meta->attr[attr_id].value, attr->value_op );
    
     return res;
   }
   
   int calc_single_pred( MetaAttribute *meta, const NodeAttributeList &attr_list, int &ind );
   int calc_pred_and_list( MetaAttribute *meta, const NodeAttributeList &attr_list, int &ind );
   int calc_pred_or_list( MetaAttribute *meta, const NodeAttributeList &attr_list, int &ind );

   
   void make_attr_list( NodeAttributeList &attr_list, MetaAttribute *meta_attr, int meta_id );
   void make_attr_list_filtered( NodeAttributeList &attr_list, MetaAttribute *meta, const NodeAttributeList &attr_filter, int first_id, int last_id );

   bool make_meta_attr( const NodeAttributeList &attr_list, MetaAttribute *meta_attr, int first_ind, int last_ind );
   
   bool compare_metaattribute( MetaAttribute *meta, const NodeAttributeList &attr_list, int first_ind, int last_ind );
   
   int look_next_attr_name( const string &full_attr_name, int ind, string &name );

public:
   /**
   *  Constructor for node attributes object creating.
   *  @param name_storage Graph attribute names storage.
   *  @param ind The value for meta-attribute of highest level.
   */
   GraphAttributes( GraphAttrNameStorage *name_storage, int ind = -1 ) 
   { m_attr_ind = ind; m_name_storage = name_storage; } 
     
   /**
   *  Destructor.
   */   
   ~GraphAttributes( ) { }
   
   /**
   *  The method assigns a value of meta-attribute of highest level.
   *  @param ind The value to be set.
   */
   void set_highlevel_index( int ind ) { m_attr_ind = ind; }
   
   static bool is_predicate( const string &attr_string )
   {
     bool res = false;
     int ind = skip_separators( attr_string, 0 );
     
     if ( ind < (int)attr_string.length() )
       res = ( attr_string[ind] == '(' );
     
     return res;
   }
   
   static bool is_predicate( const NodeAttributeList &attr_list );

   /**
   *  The method compare the graph attributes with a pattern.
   *  The pattern should be a valid meta-attribute that can have empty names and indexes of sub meta-attributes.
   *  The matching rule is as follow
   *  <p> 1) Matching simple attributes. </p>
   *  <p> A simple attribute from the pattern matches a simple attribute in the storage if their names and values are equal (types are ignored).</p>
   *  <p> 2) Matching meta-attributes. <p>
   *  <p>A meta-attribute from the pattern matches a meta-attribute in the storage if (a) the pattern meta-attribute name is empty or it is equal to the comparing one and 
   *  (b) its index (value) is empty or equal to comaring one and (c) the pattern meta-attribute does not include any attribute or each attribute from the pattern meta-attribute matches a attribute from the comparing one.</p>
   *  <p> 3) Matching node attributes. </p>
   *  <p>The pattern matches node attributes in the storage if it matches the meta-attribute of highest level.</p>
   *  @param attr_list The attributes patern represented by <b>NodeAttributeList</b>
   *  @return <b>true</b> if the pattern is valid and it matches the graph node attributes in the storage.
   */
   bool match_attributes( const NodeAttributeList &attr_list );
   

   /**
   *  The method compare the graph attributes with a pattern.
   *  The pattern should be a valid meta-attribute that can have empty names and indexes of sub meta-attributes.
   *  The matching rule is as follow
   *  <p> 1) Matching simple attributes. </p>
   *  <p>A simple attribute from the pattern matches a simple attribute in the storage if their names and values are equal (types are ignored).</p>
   *  <p> 2) Matching meta-attributes. </p>
   *  <p>A meta-attribute from the pattern matches a meta-attribute in the storage if (a) the pattern meta-attribute name is empty or it is equal to the comparing one and 
   *  (b) its index (value) is empty or equal to the comaring one and (c) the pattern meta-attribute does not include any attribute or each attribute from it matches an attribute from the comparing one.</p>
   *  <p> 3) Matching node attributes. </p>
   *  <p>The pattern matches node attributes in the storage if it matches the meta-attribute of highest level.</p>
   *  @param attribute_str The attributes patern represented by attributes string.
   *  @return <b>true</b> if the pattern is valid and it matches the graph node attributes in the storage.
   */
   bool match_attributes( const string &attribute_str )
   {
      NodeAttributeList attr_list;
      bool res = parse_string( attribute_str, attr_list );
     
      if ( res )
        res = match_attributes( attr_list );
      
      return res;
   }
   
   /**
   *  The method returns node attributes of the storage in string format.
   */
   string get_attribute_string( )
   {
      NodeAttributeList attr_list;
      make_attr_list( attr_list, &m_attr_list, m_attr_ind );

      return make_string( attr_list );
   }

   /**
   *  The method returns node attributes of the storage in parsed format.
   *  @param attr_list The reference to <code>NodeAttributeList</code> object that is filled by attributes from the storage. 
   */
   void get_parsed_attributes( NodeAttributeList &attr_list )
   {
      attr_list.clear();
      make_attr_list( attr_list, &m_attr_list, m_attr_ind );
   }
   
   /**
   *  The method returns filtered node attributes of the storage in parsed format.
   *  Only attributes which names are included into atrributes filter will be returned. 
   *  @param attr_list The reference to <code>NodeAttributeList</code> object that is filled by attributes from the storage. 
   *  @param attr_filter An attributes pattern that represents a filter. 
   */
   bool get_parsed_attributes_filtered( NodeAttributeList &attr_list, const NodeAttributeList &attr_filter )
   {
     if ( is_predicate( attr_filter ) )
       return false;
     
     NodeAttribute *attr = attr_filter[0];
     NodeAttribute *res_attr;
     
     if ( (attr->name == GraphAttrNameStorage::EMPTY_NAME || 
            m_name_storage->get_name_id( attr->name ) == m_attr_list.name) &&
          (attr->value->get_integer() == GraphAttrNameStorage::EMPTY_ID || 
            attr->value->compare( m_attr_ind )) )
     {
       attr_list.clear();
       attr_list.resize(0);
       res_attr = new NodeAttribute();
       res_attr->name = m_name_storage->get_id_name( m_attr_list.name );
       res_attr->type = NodeAttribute::ATTR_START;
       res_attr->value_op = AttributeValue::EQUAL;
       res_attr->value = new AttributeIntegerValue( m_attr_ind );
       
       attr_list.push_back( res_attr );
       
       make_attr_list_filtered( attr_list, &m_attr_list, attr_filter, 1, attr_filter.size() - 2 );
       
       res_attr = new NodeAttribute();
       res_attr->name = m_name_storage->get_id_name( m_attr_list.name );
       res_attr->type = NodeAttribute::ATTR_END;
       res_attr->value_op = AttributeValue::NONE;
       res_attr->value = new AttributeIntegerValue( m_attr_ind );
       
       attr_list.push_back( res_attr );
     }
     
     return true;
   }

   /**
   *  The method fills the storage with attributes.
   *  If the storage already has attributes it vill be cleared.
   *  @param attribute_str Graph node attributes to be set.
   */
   bool set_attribute_string( const string &attribute_str );
   
   /**
   *  The method fills the storage with attributes.
   *  If the storage already has attributes it vill be cleared.
   *  @param attr_list Graph node attributes to be set.
   */
   bool set_attribute_list( const NodeAttributeList &attr_list )
   {
     if ( is_predicate( attr_list ) )
       return false;
     
      int  start_ind = 0;
      int  end_ind = attr_list.size() - 1;
	    m_attr_list.clear();
     
      if ( attr_list[0]->type == NodeAttribute::ATTR_START )
      {
        m_attr_list.name = m_name_storage->add_name( attr_list[0]->name );
        end_ind = find_meta_end( attr_list, 0, end_ind );
        if ( attr_list[0]->value->get_integer() != GraphAttrNameStorage::EMPTY_ID )
          m_attr_ind = attr_list[0]->value->get_integer();
        
        if ( end_ind == GraphAttrNameStorage::EMPTY_ID )
          end_ind = attr_list.size();
        
        end_ind --; 
        start_ind = 1;
      }
	   
      return  make_meta_attr( attr_list, &m_attr_list, start_ind, end_ind );
   }
   
   /**
   *  The method retrieves type and value of specified attribute.
   *  @param full_attr_name Full name of requested attribute. The name includes full path from meta-attribute of highest level to the attribute.
   *  This path consists of interleaved meta-attribute names separated by "::" (the name of meta-attribute of highest level can be omited). 
   *  For example: "::metaattr1_name::metaattr2_name::attr_name".
   *  @param type The reference to <b>string</b> to retrieve attribute type.
   *  @param value The pointer to retrieve attribute value.
   *  @return <b>true</b> if requested attribute exists.
   */
   bool query_value( const string &full_attr_name, string &type, AttributeValue **value );
   
};

};

#endif
