
#ifndef _GRAPHICS_H_
#define _GRAPHICS_H_

#include <list>
#include <deque>
#include <algorithm>
#include <map>
#include <set>

//#include "Network.hpp"
#include "drawing.h" 
#include "graph.h"
#include "ViewProperties.h"
#include "GUIWindows.h"
#include "StatusInfo.h" 

using namespace std;

namespace GraphGraphics
{
  
class GraphViewContext;

enum EdgeType{
	NORMAL,		/* an original input node */
	VIRTUAL,		/* virtual nodes in long edge chains */
	SLACKNODE,	/* encode edges in node position phase */
	REVERSED,	/* reverse of an original edge */
	FLATORDER,	/* for ordered edges */
	CLUSTER_EDGE, /* for ranking clusters */
	IGNORED,		/* concentrated multi-edges */
	SELFEDGE
};


//typedef vector<struct Node*>      NodeVector;
  
/*****************************************************************************/
	

typedef vector<struct VNode*> VNodeVector;
typedef vector<struct VEdge*> VEdgeVector;
typedef vector<struct VNode*>::iterator	NodeVectorIter;
typedef vector<struct VEdge*>::iterator	EdgeVectorIter;

typedef list<struct VNode*> VNodeList;
typedef list<struct VEdge*> VEdgeList;
typedef list<struct VNode*>::iterator	NodeListIter;
typedef list<struct VEdge*>::iterator	EdgeListIter;

typedef deque<struct VNode*> VNodeQueue;
typedef deque<struct VEdge*> VEdgeQueue;	

typedef vector<struct SplineCoords*>	SplineCoordsVector;

struct SplineCoords
{
	int			type;
	
	TwoDPoint	start;
	TwoDPoint	middle;
	TwoDPoint	middle1;
	TwoDPoint	end;
	
};

typedef list<GraphViewPropertyGroup*> ViewGroupList;

struct VNode
{
   //Node id
   NodeIndex   index;
   string      label;
   bool        has_label;
   
   bool        selected; 

   //Visual node representation
   ViewGroupList  node_view_properties;
   TwoDPoint      center;
	 
     //In-edges and out-edges for this node
   VEdgeList   in_edges;
   VEdgeList   out_edges;

     //Backup of in-edges and out-edges for this node (it is necessary for position functions)
   VEdgeList   save_in_edges;
   VEdgeList   save_out_edges;

	 VEdgeVector tree_in;
   VEdgeVector tree_out;	
	 
   //Each node will be marked by its rank number. A graph will be divided into vertical layers or ranks. 
   int         rank;
	 
   //Some fields that will be used in layout algorithms
   bool mark;
   bool reversed; //If true then the edge must be reversed before drawing
   bool count; //Number of multiply edges
   int  priority;
   int  selfedges_count;  
   bool onstack;	
   struct VEdge* par;
   int    low, lim;
   int    node_type;
   int    order;	//Order number of the vertex inside its rank list (it's used in mincross algorithm)
   int    mval; //Median value of the vetex (it's used in mincross algorithm)
   int    x, y;
		
   bool  operator < (VNode& r)
   {
      if (this->order < r.order)
         return true;
      else
         return false;
   }
   
   VNode()
   {
     selected = false;
     has_label = false;
     label = "";
     node_view_properties.resize(0);
   }
   
   VNode( GraphViewPropertyGroup* def_group )
   {
     selected = false;
     has_label = false;
     label = "";
     node_view_properties.resize(0);
     node_view_properties.push_back( def_group );
   }
   
   GraphViewPropertyGroup get_node_properties(  )
   {
     GraphViewPropertyGroup ugroup;
     
     for ( ViewGroupList::iterator p = node_view_properties.begin(); p != node_view_properties.end(); p++ )
     {
       GraphViewPropertyGroup *group = (GraphViewPropertyGroup *)(*p);
       
       if ( group->is_shape_set() )
         ugroup.set_shape_type( group->get_shape_type() );
       
       if ( group->is_font_set() )
         ugroup.set_font( group->get_font(), group->get_text_color() );
       
       if ( group->is_pen_set() )
         ugroup.set_pen( group->get_pen() );
       
       if ( group->is_color_set() )
         ugroup.set_color( group->get_color() );
       
       if ( group->is_filled_set() )
         ugroup.set_shape_filled( group->is_shape_filled() );
       
       if ( group->is_shape_ratio_set() )
         ugroup.set_shape_xy_ratio( group->get_shape_x_ratio(), group->get_shape_y_ratio() );
     }
     
     return ugroup;
   }
		
};

class VNode_less : public binary_function<VNode*, VNode*, bool>
{
	public:
	bool operator()(VNode* n, VNode* u) 
	{ 
		return n->order < u->order; 
	}
};

class VNode_more : public binary_function<VNode*, VNode*, bool>
{
	public:
	bool operator()(VNode* n, VNode* u) 
	{ 
		return n->order > u->order; 
	}
};


struct VEdge
{
   //Node id
   NodeIndex   head_indx;
   NodeIndex   tail_indx;		 
   
	  // Cut value of the edge. This value is used in network simplex algorithm
   int         mark;
   int         cutvalue;
	 
   int         multiply_edges_count;
	 
   int         minlen;		
   int         tree_index;
	 
   int         weight;
   int         edge_type;

   bool		inverse_edge;

   NodeIndex   real_head_indx;
   NodeIndex   real_tail_indx;		 

	
   SplineCoordsVector	coords;
};

class GraphLayout;
  
typedef map<StyleFlags, NodeLinksMap> MarkEdgesMap;

class AttrStringTemplateStorage
{
typedef StringMap TemplateStringMap; 
typedef TemplateStringMap::const_iterator TemplateStringMapIter;

public:  
  static const string INVALID_TEMPLATE_NAME;

private:
  TemplateStringMap  m_templates;
  string             m_curr_template;
  
public:
  AttrStringTemplateStorage() {}
  ~AttrStringTemplateStorage() { m_templates.clear(); }
  
  bool is_template_name_exist( const string& template_name ) const
  {
    return (m_templates.find( template_name ) != m_templates.end());
  }
   
  string get_current_template_name() { return m_curr_template; } const
   
  bool set_current_template_name( const string& template_name ) 
  {
    bool res = is_template_name_exist( template_name );
     
    if ( res )
      m_curr_template = template_name;
     
    return res;
  }
   
  string get_first_template_name() const
  {
    string name = INVALID_TEMPLATE_NAME;

    if ( !m_templates.empty() )
    {
      TemplateStringMapIter p = m_templates.begin();
      if ( p != m_templates.end() )
        name = p->first;
    }

    return name;
  }
   
  string get_next_template_name( const string& template_name ) const
  {
    string name = INVALID_TEMPLATE_NAME;
     
    if ( !m_templates.empty() )
    {
      TemplateStringMapIter p = m_templates.find( template_name );
      if ( p != m_templates.end() )
      {
        p++;
        if ( p != m_templates.end() )
          name = p->first;
      }
    }
     
    return name;
  }
   
  void set_template( const string& template_name, const string &attr_template ) 
  { 
    TemplateStringMapIter p = m_templates.begin();
    
    if ( p == m_templates.end() )
      m_curr_template = template_name; 
      
    m_templates[template_name] = attr_template;      
  }
   
  string get_template( const string& template_name ) const
  { 
    string res = "";
    TemplateStringMapIter p = m_templates.find( template_name );
    
    if ( p != m_templates.end() )
      res =  p->second;
    
    return res; 
  }
   
  void remove_template( const string& template_name )
  {
    if ( m_templates.find( template_name ) != m_templates.end() )
    {
      m_templates.erase( template_name );
      if ( template_name == m_curr_template )
      {
        TemplateStringMapIter p = m_templates.begin();
        if ( p != m_templates.end() )
          m_curr_template = p->first;
        else
          m_curr_template = "";
      }        
    }
  }
  
  AttrStringTemplateStorage::AttrStringTemplateStorage( const AttrStringTemplateStorage& templ ) 
  {
    for ( string name = templ.get_first_template_name(); name != INVALID_TEMPLATE_NAME; name = templ.get_next_template_name( name ) ) 
      m_templates[name] = templ.get_template( name );
    
    m_curr_template = templ.m_curr_template;
  }
};  

struct GraphNodeLabelTemplate
{
  string node_attributes;
  string node_label_template;
};

typedef vector<GraphNodeLabelTemplate> GraphNodeLabelVector;

/**
* GraphView serves for visualization of the graph. 
* It has mathods for drawing nodes, edges, edge arrows. 
* It is also responsible for setting different graph visualization properties. like node color, node shape, edge color, 
* edge line type, etc.  
*
*/  
class GraphView: public Storable
{
  friend class GraphViewPropertiesStorage;
     
public:
  
   /** @name Predefined fitting method constants. */
   //@{
   enum FitMethod
   {
      ARBITRARY_SIZE = 0,
      FIT_TO_WINDOW,
      FIT_TO_WIDTH,
      FIT_TO_HEIGHT,
      FIT_SELECTION_TO_WINDOW
   };
   //@}      
   
private:
   VNodeVector m_nodes;
   VEdgeVector m_edges;

   GraphViewContext* m_graph_view_context;
   
   Rectangle         m_bounds;

   OneDValue         m_radius;
   OneDValue         m_x_separation;
   OneDValue         m_y_separation;

	 int				m_arrow_len;
	 double			pi;
	 double			m_alpha;
	 double			m_sin_alpha;
	 double			m_cos_alpha;	

   GraphViewPropertiesStorage m_view_groups;
   
   StyleFlags        m_fit_method;
   double            m_zoom_value;
   double            m_zoom_factor;
   Rectangle         m_view_rect;    
   Rectangle         m_display_rect;
   
   GraphLayout *m_gl;
   
   NodeIndexVector m_selected_nodes;
   MarkEdgesMap    m_marked_edges;
   
   AttrStringTemplateStorage  *m_filters;
   
//Parameters for spline drawing
   double	t_step;
   int		s_iter;	
    
   
   bool	m_spline_interpltn;
   bool m_draw_arrows;
   
   GraphNodeLabelVector m_label_templates;
   
private:
   
/**************************************************************
**************************************************************/
   TwoDPoint transform_point( const TwoDPoint &point )
   {
     TwoDPoint new_point( (OneDValue)((double)(point.get_x()) * ((double)m_view_rect.get_width() / (double)m_bounds.get_width())) + m_view_rect.get_left_top_corner().get_x(), 
                          (OneDValue)((double)(point.get_y()) * ((double)m_view_rect.get_width() / (double)m_bounds.get_width())) + m_view_rect.get_left_top_corner().get_y() );
     
     
     return new_point;
   }
   
  bool transform_line( const TwoDPoint &point1, const TwoDPoint &point2, LineSegment &trans_line )
  {
    TwoDPoint pp[2];
    int       i;		
     
    if (m_display_rect.inside(point1) && m_display_rect.inside(point2))
    {
      trans_line.set_start_point(point1);
      trans_line.set_end_point(point2);

      return true;		 
    }

    LineSegment l(point1, point2);
	 
    LineSegment l_l(m_display_rect.get_left_top_corner(), TwoDPoint(m_display_rect.get_left_top_corner().get_x(), m_display_rect.get_right_bottom_corner().get_y()));	   
    LineSegment l_r(m_display_rect.get_right_bottom_corner(), TwoDPoint(m_display_rect.get_right_bottom_corner().get_x(), m_display_rect.get_left_top_corner().get_y()));	   	   
    LineSegment l_t(m_display_rect.get_left_top_corner(), TwoDPoint(m_display_rect.get_right_bottom_corner().get_x(), m_display_rect.get_left_top_corner().get_y()));	   
    LineSegment l_b(m_display_rect.get_right_bottom_corner(), TwoDPoint(m_display_rect.get_left_top_corner().get_x(), m_display_rect.get_right_bottom_corner().get_y()));	   	   
     
    i = 0;
	 
    if (l.intersect(l_l) != 0)
    {
      l.intersection(l_l, &(pp[i]));
      i++;
    }
	 
    if (l.intersect(l_r) != 0)
    {
      l.intersection(l_r, &(pp[i]));
      i++;
    }
	 
    if (l.intersect(l_t) != 0)
    {
      l.intersection(l_t, &(pp[i]));
      i++;
    }
	 
    if (l.intersect(l_b) != 0)
    {
      l.intersection(l_b, &(pp[i]));
      i++;
    }
	 
    // Line does not intersect display rectangle
    if (i == 0)
      return false;
	 
	 // Check if only one node is outside the rectangle
    if (point1.get_x() != point2.get_x())
    {
      if (m_display_rect.inside(point1))
      {
        if (((pp[0].get_x() <= point1.get_x()) && (pp[0].get_x() >= point2.get_x())) || 
            ((pp[0].get_x() <= point2.get_x()) && (pp[0].get_x() >= point1.get_x())))
          pp[1] = point1;
        else if (((pp[1].get_x() <= point1.get_x()) && (pp[1].get_x() >= point2.get_x())) || 
            ((pp[1].get_x() <= point2.get_x()) && (pp[1].get_x() >= point1.get_x())))
          pp[0] = point1;

      }
      else if (m_display_rect.inside(point2))
      {
        if (((pp[0].get_x() <= point1.get_x()) && (pp[0].get_x() >= point2.get_x())) || 
            ((pp[0].get_x() <= point2.get_x()) && (pp[0].get_x() >= point1.get_x())))
          pp[1] = point2;
        else if (((pp[1].get_x() <= point1.get_x()) && (pp[1].get_x() >= point2.get_x())) || 
            ((pp[1].get_x() <= point2.get_x()) && (pp[1].get_x() >= point1.get_x())))
          pp[0] = point2;
      }
    }
    else
    {
      if (m_display_rect.inside(point1))
      {
        if (((pp[0].get_y() <= point1.get_y()) && (pp[0].get_y() >= point2.get_y())) || 
            ((pp[0].get_y() <= point2.get_y()) && (pp[0].get_y() >= point1.get_y())))
          pp[1] = point1;
        else if (((pp[1].get_y() <= point1.get_y()) && (pp[1].get_y() >= point2.get_y())) || 
            ((pp[1].get_y() <= point2.get_y()) && (pp[1].get_y() >= point1.get_y())))
          pp[0] = point1;
      }
      else if (m_display_rect.inside(point2))
      {
        if (((pp[0].get_y() <= point1.get_y()) && (pp[0].get_y() >= point2.get_y())) || 
            ((pp[0].get_y() <= point2.get_y()) && (pp[0].get_y() >= point1.get_y())))
          pp[1] = point2;
        else if (((pp[1].get_y() <= point1.get_y()) && (pp[1].get_y() >= point2.get_y())) || 
            ((pp[1].get_y() <= point2.get_y()) && (pp[1].get_y() >= point1.get_y())))
          pp[0] = point2;
      }
    } 


/*		 cerr << "outside\n";	 */
    trans_line.set_start_point(pp[0]);
    trans_line.set_end_point(pp[1]);
    return true;
  }

   OneDValue transform_range( OneDValue range )
   {
     return (OneDValue)((double)range * (double)m_view_rect.get_width() / (double)m_bounds.get_width());
   }
   
   TwoDPoint inverse_transform_point( const TwoDPoint &point )
   {
     return TwoDPoint( (OneDValue)((double)(point.get_x() - m_view_rect.get_left_top_corner().get_x()) * ((double)m_bounds.get_width() / (double)m_view_rect.get_width())) , 
                       (OneDValue)((double)(point.get_y() - m_view_rect.get_left_top_corner().get_y()) * ((double)m_bounds.get_width() / (double)m_view_rect.get_width())));
   }
   
/**************************************************************/
   bool init_nodes_edges();
   void delete_nodes_edges();
   void delete_edges();   
   
   void draw_graph_node( GraphicArea *g, VNode *node, bool unselect = false );
   void draw_graph_edge( GraphicArea *g, VEdge *e );
   bool draw_spline_edge( GraphicArea *g, VEdge *e);   
   void draw_arrow(GraphicArea *g, VEdge *e);   
   void draw_virtual_edges( GraphicArea *g, VNode *start_virtual_node, bool is_down);  

   void redraw_node_edges( GraphicArea *g, VNode *node )
   {
     if ( node != 0 )
     {
       EdgeListIter	e_iter;
       VEdge			*e;
                    
       e_iter = node->out_edges.begin();
       while (e_iter != node->out_edges.end()) 		 
       {
         e = *e_iter;
           
         draw_spline_edge( g, e );		
         draw_arrow( g, e );
           
         if (m_nodes[e->head_indx]->node_type == VIRTUAL)
         draw_virtual_edges(g, m_nodes[e->head_indx], true);
           
         e_iter++;
       }
         
       e_iter = node->in_edges.begin();
       while (e_iter != node->in_edges.end()) 		 
       {
         e = *e_iter;
           
         draw_spline_edge( g, e );		
         draw_arrow( g, e );
           
         if (m_nodes[e->tail_indx]->node_type == VIRTUAL)
           draw_virtual_edges(g, m_nodes[e->tail_indx], false);
                 
         e_iter++;
       }
     }
   }     
   
   VEdge* find_edge( NodeIndex start_node_id, NodeIndex end_node_id ) const
   {
     if ( start_node_id >= (NodeIndex)m_nodes.size() || start_node_id < 0 )
       return 0;
     
     VNode *node = m_nodes[start_node_id];
     VEdge *res_edge = 0;
     VEdge *curr_edge = 0;
     //edges.clear();
     
     for ( EdgeListIter::const_iterator iter = node->out_edges.begin(); iter != node->out_edges.end(); iter++ )
     {
       curr_edge = *iter;
       if ( curr_edge->real_head_indx == end_node_id &&  
            curr_edge->real_tail_indx == start_node_id && 
            !curr_edge->inverse_edge )
       {
         res_edge = curr_edge;
         break;
       }
     }
     
     if ( res_edge == 0 )
     {
       node = m_nodes[end_node_id];
       for ( EdgeListIter::const_iterator iter = node->out_edges.begin(); iter != node->out_edges.end(); iter++ )
       {
         curr_edge = *iter;
         if ( curr_edge->real_head_indx == start_node_id && 
              curr_edge->real_tail_indx == end_node_id && 
              curr_edge->inverse_edge )
         {
           res_edge = curr_edge;
           break;
         }
       } 
     }
     
     return res_edge;
   }
   
   VEdge *find_next_edge( VEdge *e )
   {
     VNode *node = m_nodes[e->real_tail_indx];
     VEdge *res_edge = 0;
     VEdge *curr_edge = 0;
     EdgeListIter::const_iterator iter;
     
     for ( iter = node->out_edges.begin(); iter != node->out_edges.end(); iter++ )
       if ( *iter == e )
         break;
       
     if ( iter != node->out_edges.end() )
       iter++;
     
     for ( ; iter != node->out_edges.end(); iter++ )
     {
       curr_edge = *iter;
       if ( curr_edge->real_head_indx == e->real_head_indx && 
            curr_edge->real_tail_indx == e->real_tail_indx && 
            !curr_edge->inverse_edge)
       {
         res_edge = curr_edge;
         break;
       }
     }
     
     if ( res_edge == 0 )
     {
       node = m_nodes[e->real_head_indx];
       
       for ( iter = node->out_edges.begin(); iter != node->out_edges.end(); iter++ )
         if ( *iter == e )
           break;
       
       if ( iter != node->out_edges.end() )
         iter++;
       
       for ( iter = node->out_edges.begin(); iter != node->out_edges.end(); iter++ )
       {
         curr_edge = *iter;
         if ( curr_edge->real_head_indx == e->real_tail_indx && 
              curr_edge->real_tail_indx == e->real_head_indx &&
              curr_edge->inverse_edge )
         {
           res_edge = curr_edge;
           break;
         }
       } 
     }
     
     return res_edge;
   }
   
   void remove_all_edge_marking( GraphicArea *g );
   
   void remark_edge( VEdge *edge, StyleFlags new_mark );
   
   void mark_edge( VEdge *edge, GraphicArea *g, StyleFlags mark = GraphViewPropertiesStorage::EDGE_SELECTED, bool remark = true );
   
   NodeIndex find_rank_node( const TwoDPoint &point );
   unsigned int find_nearest_rank( OneDValue y );
   unsigned int find_nearest_node_in_rank( unsigned int nearest_rank, OneDValue x );
   bool is_edge_visible( VEdge *edge, const Rectangle &rect );
   
   string construct_node_label( GraphAttributes *attr, const string &label_template );
   void set_node_labels( Graph *graph, const GraphNodeLabelTemplate &label_template );
   
public:
  
   /**
   *  This constructor create <code>GraphView</code>.
   *  @param app_context Pointer to the <code>GraphViewContext</code> object.
   *  @see GraphViewContext
   */
   GraphView( GraphViewContext *app_context );

   ~GraphView( );

   /**
   *  Copies visualization properties from the <code>GraphView</code> object passed through <code>view</code> parameter 
   *  to the current context.
   *  @param view Pointer to the source <code>GraphView</code> object.
   */
   void copy_properties( GraphView *view );

	 void redraw_edges( bool rebuild_position = true );

   /**
   *  Sets new radius of the nodes. 
   *  @param radius New radius value.
   */
	 void set_radius(OneDValue radius) { m_radius = radius; }

   /**
   *  Gets new radius of the nodes. 
   *  @return Current radius value.
   */
	 OneDValue get_radius() const { return m_radius; }
   
   /**
   *  Sets new separation value between nodes in the horizontal direction. 
   *  @param x_separation New separation value.
   */
	 void set_x_separation(OneDValue x_separation) { m_x_separation = x_separation; }   

   /**
   *  Gets new separation value between nodes in the horizontal direction. 
   *  @return Current separation value.
   */
   OneDValue get_x_separation() const { return m_x_separation; }      
   
   /**
   *  Sets new separation value between nodes in the vertical direction. 
   *  @param y_separation New separation value.
   */
	 void set_y_separation(OneDValue y_separation) { m_y_separation = y_separation; }   

   /**
   *  Gets new separation value between nodes in the vertical direction. 
   *  @return Current separation value.
   */
	 OneDValue get_y_separation() const { return m_y_separation; } 
      
   /**
   *  Sets edge interpolation flag. If <b>true</b> then the edges will be interpolated using splines. If <b>false</b> then
   *  they will be drawn as straight lines.   
   *  @param spline_interpltn New interpolation flag.
   */
   void set_spline_interpltn(bool spline_interpltn) { m_spline_interpltn = spline_interpltn; }   
   
   /**
   *  Gets edge interpolation flag. If <b>true</b> then splines interpolation is used.
   *  @return Current interpolation flag.
   */
	 bool is_spline_interpltn() const { return m_spline_interpltn; } 
   
   void set_arrows_drawn( bool draw_arrow ) { m_draw_arrows = draw_arrow; }
   bool is_arrows_drawn( ) const { return m_draw_arrows; }


   /**
   *  Gets application context that this view object belongs to. 
   *  @return Application context.
   *  @see GraphViewContext
   */
   GraphViewContext* get_app_context() const { return m_graph_view_context; }

   /**
   *  Graph view rendering function.This function display the graph using current visualization parameters. 
   *  @param g The graphic area that the graph will be displayed on.
   *  @see GraphicArea
   */
   void display( GraphicArea *g );
   void redraw_nodes( GraphicArea *g, const NodeIndexSet &nodes, bool redraw_node_edges = false );
	 
   /**
   *  This function creates graph view from the current graph representation. It initializes internal representation 
   *  and create proper layout of the graph view. If graph view exists when the method is invoked then it will be replaced by the new one.   
   *  Graph information necessary to build graph view will be taken from <code>Graph</code> object stored in the context.   
   *  @see Graph
   *  @see GraphViewContext
   */
   void make_graph_view( );
   

   /**
   *  Returns index of the node that covers the point passed through the parameter.  
   *  @param point Point that should belong to the search node. 
   *  @return Node index or Graph::INVALID_NODE_ID if such node doesn't exist.
   *  @see TwoDPoint
   */
   NodeIndex get_node_id( const TwoDPoint &point );
     
   /**
   *  Returns indexes of the nodes that are placed inside the rectangle.  
   *  @param rect Rectangle where searched nodes should be placed. 
   *  @param res_node_ids Returned node indexes vector.
   *  @return Found nodes number.
   */
   int  get_node_id_from_rect( const Rectangle &rect, NodeIndexVector &res_node_ids );
   
   /**
   *  Returns indexes of the selected nodes.  
   *  @param res_node_ids Returned node indexes vector.
   */
   void  get_selected_nodes( NodeIndexVector &res_node_ids ) const
   {
     for( NodeIndexVector::const_iterator p = m_selected_nodes.begin(); p != m_selected_nodes.end(); p++ )
        res_node_ids.push_back( *p );
   }
   
   TwoDPoint get_node_coord( NodeIndex node_id ) const
   {
     TwoDPoint res;
     
     if ( node_id != Graph::INVALID_NODE_ID && node_id < (NodeIndex)m_nodes.size() )
       res = m_nodes[ node_id ]->center;
     
     return res;
   }
   
   /**
   *  Checks if the node is selected.  
   *  @param node_id Index of the node that should be checked.
   *  @return <b>true</b> if the node is selected.
   */
   bool is_node_selected( NodeIndex node_id ) const
   { 
     bool res = false;
     
     if ( node_id != Graph::INVALID_NODE_ID && node_id < (NodeIndex)m_nodes.size() )
       res = m_nodes[ node_id ]->selected;
     
     return res;
   }
   
   /**
   *  Returns edge with specific mark.  
   *  @see Edge
   */
   unsigned int get_marked_edges( StyleFlags mark, EdgeList &marked_edges ) const;
   
   bool remark_edges( StyleFlags mark, GraphicArea *g, StyleFlags new_mark );
   
   /**
   *  Mark graph edge. If <code>g</code> is not 0 then this function also redraws the edge.   
   *  @param edge Edge to be marked.
   *  @param g Pointer to the GraphicArea object.   
   *  @param mark Mark constant.   
   *  @return <b>true</b> if the edge is marked.
   *  @see Edge
   *  @see GraphicArea   
   */
   bool mark_edge( const Edge &edge, GraphicArea *g, StyleFlags mark = GraphViewPropertiesStorage::EDGE_SELECTED )
   {
     VEdge *e = find_edge( edge.start_node, edge.end_node );
     bool res = (e != 0);
     
     for (; e != 0; e = find_next_edge( e ) )
       mark_edge( e, g, mark );
     
     return res;
   }
   
   /**
   *  Returns mark of the edge.  
   *  @param Edge to be analized.
   *  @return Mark constant.
   *  @see Edge
   */
   StyleFlags get_edge_mark( const Edge &edge ) const
   {
     StyleFlags mark  = GraphViewPropertiesStorage::EDGE_UNMARKED;
     VEdge     *e     = find_edge( edge.start_node, edge.end_node );
     
     if ( e != 0 )
       mark = e->mark;
     
     return mark;
   }
   
   /**
   *  Selects or unselects the node. If g != 0 then the node will be redrawn  
   *  @param Edge to be selected.
   *  @param g Graphic area of the window.
   *  @see GraphicArea
   */

   void select_node( NodeIndex node_id, GraphicArea *g, bool select = true )
   {
     if ( node_id < 0 || node_id >= (NodeIndex)m_nodes.size() )
       return;
     
     VNode      *node = m_nodes[node_id];
     
     if ( node->selected == select )
       return;
	   
     node->selected = select;
     
     if ( select )
       m_selected_nodes.push_back( node_id );
     else
     {
       //EdgeListIter	iter;
       NodeIndexVector::iterator p = find( m_selected_nodes.begin(), m_selected_nodes.end(), node_id );
       if ( p != m_selected_nodes.end() )
          m_selected_nodes.erase(p);
     }
     
     if ( g != 0 )
     {
       draw_graph_node( g, node, true );
       redraw_node_edges( g, node );
	   }
   }
   
   /**
   *  Unselects all selected nodes. If g != 0 then the nodes will be redrawn  
   *  @param g Graphic area of the window.
   *  @see GraphicArea
   */
   void unselect_all_nodes( GraphicArea *g )
   {
     VNode      *node;
     NodeIndexVector::const_iterator p;
	   
     for( p = m_selected_nodes.begin(); p != m_selected_nodes.end(); p++ )
     {
       node = m_nodes[ *p ];
       node->selected = false;
     }
    
     for( p = m_selected_nodes.begin(); p != m_selected_nodes.end(); p++ )
     {       
       node = m_nodes[ *p ];

       if ( g != 0 )
       {
         draw_graph_node( g, node, true );
         
         redraw_node_edges( g, node );
       }
     }
     
     m_selected_nodes.clear();
     m_selected_nodes.resize(0);
   }

   void mark_incoming_edges( NodeIndex node_id, GraphicArea *g, StyleFlags mark )
   {
     VNode *node;
     if ( node_id >= 0 && node_id < (NodeIndex)m_nodes.size() )
     {
       node = m_nodes[node_id];
         
       for ( EdgeListIter::const_iterator iter = node->in_edges.begin(); iter != node->in_edges.end(); iter++ )
         mark_edge( *iter, g, mark );
     }
   }
   
   void mark_outgoing_edges( NodeIndex node_id, GraphicArea *g, StyleFlags mark )
   {
     VNode *node;
     if ( node_id >= 0 && node_id < (NodeIndex)m_nodes.size() )
     {
       node = m_nodes[node_id];    
       for ( EdgeListIter::const_iterator iter = node->out_edges.begin(); iter != node->out_edges.end(); iter++ )
         mark_edge( *iter, g, mark );
     }
   }
   /**
   *  Sets mark to all egdes. Usually it is used to set GraphView::EDGE_UNMARKED to all edges  
   *  @param g Graphic area of the window.
   *  @see GraphicArea
   */
   void mark_all_edges( GraphicArea *g, StyleFlags mark = GraphViewPropertiesStorage::EDGE_UNMARKED )
   {
     VNode *node;
     
     remove_all_edge_marking( g );
     
     if ( mark != GraphViewPropertiesStorage::EDGE_UNMARKED )
       for ( VNodeVector::const_iterator p = m_nodes.begin(); p != m_nodes.end(); p++ )
       {
         node = *p;
         if ( node->node_type == NORMAL )
           for ( EdgeListIter::iterator iter = node->out_edges.begin(); iter != node->out_edges.end(); iter++ )
             mark_edge( *iter, g, mark );
       }
   }

   void get_number_info(unsigned int& nodes_number, unsigned int& edges_number, unsigned int& sel_nodes_number, unsigned int& sel_edges_number);
   
   /**
   *  Sets new zoom value of the graph view. This value is used as a step in zoom-in and zoom-out operations   
   *  @param zoom New zoom value.
   */
   void set_zoom_value( double zoom ) { m_zoom_value = zoom; }
   
   /**
   *  Gets current zoom value of the graph view. This value is used as a step in zoom-in and zoom-out operations     
   *  @reurn Current zoom value.
   */
   double get_zoom_value( ) { return m_zoom_value; }
   
   /**
   *  Zoom-in of the graph view.     
   *  @see GraphView::set_zoom_value.
   *  @see GraphView::get_zoom_value.   
   */
   void zoom_in(); 
   
   /**
   *  Zoom-out of the graph view.     
   *  @see GraphView::set_zoom_value.
   *  @see GraphView::get_zoom_value.   
   */
   void zoom_out();
   
   /**
   *  Zoom graph view in such way that all node labels become visible.     
   *  @param g Graphic area of the window.   
   */
   void zoom_to_label_view( GraphicArea *g );
   
   /**
   *  Sets layout style of the graph view.     
   *  @param fit_method Fitting method to be set.   
   *  @see FitMethod
   */
   void set_layout( StyleFlags fit_method ){ m_fit_method = fit_method; }
   
   /**
   *  Gets current layout style of the graph view.     
   *  @return Current layout style.   
   *  @see FitMethod
   */
   StyleFlags get_layout() { return m_fit_method; }
   
   /**
   *  Forces the system to make layout of the graph view based on the current fitting method.     
   *  @param display_rect Display rectangle area to make fit to.   
   */
   void do_layout( const Rectangle &display_rect);

   /**
   *  The method zooms and scrolls the graph view so that selected nodes will be centered into a rectangle area.
   *  @param rect Rectangle area the selected node will be centered into.
   */   
   void centering( const Rectangle &rect ); 

   /**
   *  The method returns current rectangle area of the graph view.
   *  The rectangle area coordinates are relative to dispalaying rectangle's so they can be negative.
   */
   Rectangle get_view_rect() { return m_view_rect; }
   
   /**
   *  The method zooms graph view to a rectangle area.
   *  The rectangle area coordinates are relative to dispalaying rectangle's so they can be negative.
   *  @param rect Rectangle area to zoom graph view to.
   */
   void      set_view_rect( const Rectangle &rect ) { m_view_rect = rect; }

   /**
   *  The method scrolls the graph view horisontally.
   *  @param dx The offset the graph view rectangle area coordinates should be moved at.
   *  If the offset is positive the rectangle area coordinates will be moved left at the value otherwise it will be moved right. 
   */
   void set_hscroll_value( OneDValue dx ) 
   { 
     m_view_rect.set_left_top_corner( TwoDPoint( m_view_rect.get_left_top_corner().get_x() - dx, m_view_rect.get_left_top_corner().get_y()) );
     m_view_rect.set_right_bottom_corner( TwoDPoint( m_view_rect.get_right_bottom_corner().get_x() - dx, m_view_rect.get_right_bottom_corner().get_y()) );
   }
   
  /**
   *  The method scrolls the graph view vertically.
   *  @param dy The offset the graph view rectangle area coordinates should be moved at.
   *  If the offset is positive the rectangle area coordinates will be moved upward at the value otherwise it will be moved downward. 
   */      
   void set_vscroll_value( OneDValue dy ) 
   { 
     m_view_rect.set_left_top_corner( TwoDPoint( m_view_rect.get_left_top_corner().get_x(), m_view_rect.get_left_top_corner().get_y() - dy) );
     m_view_rect.set_right_bottom_corner( TwoDPoint( m_view_rect.get_right_bottom_corner().get_x(), m_view_rect.get_right_bottom_corner().get_y() - dy) );
   }

   /**
   *  The method gets reference to the graph view properties storage.
   */
   GraphViewPropertiesStorage& get_view_properties_storage() { return m_view_groups; }  
   
   /**
   *  The method returns currently registered attributes filters. 
   *  Each attributes filter is represented by attributes template string.
   *  @return The attributes filter storage of the graph view.
   *  @see GraphAttributes::get_parsed_attributes_filtered(...)
   */
   AttrStringTemplateStorage* get_node_attr_filters() { return m_filters; }
   
   /**
   *  The method sets new attributes filters.
   *  All old filters will be removed.
   *  @param filters Filter storage to be copied.
   */
   void set_node_attr_filters( AttrStringTemplateStorage* filters )
   {
     delete m_filters;
     m_filters = new AttrStringTemplateStorage( *filters );
   }
   
   void add_node_label_template( const string &node_attr_string, const string &label )
   {
     GraphNodeLabelTemplate label_template;
     label_template.node_attributes = node_attr_string;
     label_template.node_label_template = label;
     
     m_label_templates.push_back( label_template );
   }
   
   void assign_labels_to_nodes( );
   void remove_node_label_templates() { m_label_templates.clear(); }
      
   /**
   *  The method writes graph view properties into a stream.
   *  @see Storable::write_to(...)
   */
   void write_to( ostream& s_out, unsigned long write_options = 0 );
   
   /**
   *  The method reads graph view properties from a stream.
   *  @see Storable::write_to(...)
   */
   bool read_from( istream& s_in, unsigned long read_options = 0 );
   
};

class ScenarioApp;
class ComponentManager;
class NodeAttrCloseHandler;  
/**
* The class provides necessary functioanality for using OS-dependent code through OS-independent interface.
* Each <b>GraphGUI</b> object is associated with a <code>ScenarioApp</code> instance that creates it.
* @warning Do not create/delete a <b>GraphGUI</b> object in application. It is automatically created and deleted by the library when appropriate <code>ScenarioApp</code> instance is created/deleted.
* @see GraphView
* @see ScenarioApp::get_GUI()
*/

class GUICreateDockableWindowEventHandler
{
public:
  virtual ~GUICreateDockableWindowEventHandler(){};    
      
  virtual void operator () ( GraphGUIDockableWindow* window, bool new_created ) = 0;
};

class GraphGUI
{
friend class NodeAttrCloseHandler;
friend class InfoWindowCloseHandler;
  
public:
  
   /**
   *  Allowed types of mouse cursor. 
   *  @see ::set_cursor(...)
   */
   enum CursorTypes
   {
     DEFAULT_CURSOR = 0, /**< Default system mouse cursor (usually it is arrow).*/
     CROSS_CURSOR        /**< Thin cross mouse cursor. */
   };
   
   enum DockableWindowType
   {
     DOCKABLE_GENERAL = 0,
     DOCKABLE_NODE_ATTRIBUTES = 1,
     DOCKABLE_NODE_INFO = 2,
     DOCKABLE_EDGE_INFO = 3,
     DOCKABLE_PATH_INFO = 4,
     DOCKABLE_GRAPH_INFO = 5
   };
   
private:
typedef list<GUICreateDockableWindowEventHandler*> DockableHandlerList;

   void *m_os_data;
   void *m_wait;
   ScenarioApp *m_app;
   GraphNodeAttributesGUI *m_last_attr_window;
   set<string>             m_last_attr_expanded_rows;
   GraphInformationGUI *m_last_info_window;

   DockableHandlerList m_dockable_handler_list;
public:
  
   static void initialize_globals();

   /**
   * Constructor.
   */
   GraphGUI( ScenarioApp *app, void *os_data ) 
   { 
     m_os_data = os_data; 
     m_wait = 0; 
     m_app = app; 
     m_last_attr_window = 0; 
     m_last_info_window = 0;
   }
   
   /** 
   * destructor
   */
   ~GraphGUI() 
   {
     DockableHandlerList::iterator iter = m_dockable_handler_list.begin();
     for ( ; iter != m_dockable_handler_list.end(); iter++ )
         delete (*iter);
     m_dockable_handler_list.clear();
   }
   
   void register_create_dockable_event_handler( GUICreateDockableWindowEventHandler* handler )
   {
     if ( handler != 0 )
       m_dockable_handler_list.push_back( handler );  
   }
   
   void generate_create_dockable_event( GraphGUIDockableWindow* window, bool new_created )
   {
     DockableHandlerList::iterator iter = m_dockable_handler_list.begin();
     for ( ; iter != m_dockable_handler_list.end(); iter++ )
       (**iter)( window, new_created );
   }
     
   ScenarioApp* get_scenario_app() { return m_app; }
     
   /**
   *  The method creates OS-dependent graphic area object used for graph drawing.
   *  @param os_data A pointer to actual OS-dependent drawing area structure/object.
   *  @param screen_window It is optioanl parameter that describe <b>os_data</b>.      
   *  If <b>true</b> the <b>os_data</b> need to refer to screen object. Otherwise it need to refer on a hiden (memory) drawing object.
   *  @return The <code>GraphicArea</code> instance that refer to OS-dependent drawing area. All graph drawing operations use only <b>GraphicArea</b> objects.
   */
   GraphicArea* get_graphic_area( void *os_data, bool screen_window = true );
     
   /** @name Graph GUI dialogs. 
   *   The methods create various GUI dialogs classes that are used for displaying/changing graph information and graph visual properties.  
   */
   //@{
     
   /**
   *  The method creates GUI dialog object for manipulation with general graph view properties.
   *  @param view The pointer to <b>GraphView</b> object, which properties are used.
   *  @return A pointer to <b>GraphViewPropertiesGUI</b> instance.
   *  @see GraphViewPropertiesStorage     
   */
   GraphViewPropertiesGUI* create_view_properties_dialog( GraphView *view );
   
   /**
   *  The method creates GUI dialog object for changing visual preferences.
   *  These preferences are saved into a file and are used as default visual properties for first time graph drawing.
   *  @param view The pointer to <b>GraphView</b> object, which properties are used.
   *  @return A pointer to <b>GraphGUIModalWindow</b> instance.     
   */
   GraphGUIModalWindow* create_view_preferences_dialog( GraphView *view, bool apply_changes_after_close = false );     
   
   /**
   *  The method creates GUI dialog object for manipulation with node visual properties groups.
   *  @param view The pointer to <b>GraphView</b> object, which properties are used.
   *  @return A pointer to <b>GraphGUIModalWindow</b> instance.     
   *  @see GraphViewPropertiesStorage     
   *  @see GraphViewPropertiesGroup     
   */
   GraphViewPropertiesGUI* create_group_properties_dialog( GraphView *view );
   
   /**
   *  The method shows graph attributes GUI dialog for selected nodes.
   *  The attributes are shown filtered using current attributes filter.
   *  @param view The pointer to <b>GraphView</b> object, which properties are used.
   *  @param node_id The node index, which attributes should be shown. 
   *  If it is <code>Graph::INVALID_NODE_ID</code> attributes of all selected nodes will be shown.
   *  @see AttrStringTemplateStorage
   *  @see GraphAttributes
   */
   GraphNodeAttributesGUI *create_nodes_info_dialog( GraphView *view, NodeIndex node_id = Graph::INVALID_NODE_ID, bool ignore_modes = false );
   
   /**
   *  The method creates GUI dialog object for creating graph attributes pattern for searching nodes by attributes.
   *  @param view The pointer to <b>GraphView</b> object, which properties are used.
   */
   GraphNodeSearchGUI* create_node_search_dialog( GraphView *view );
   
   /**
   *  The method creates GUI dialog object for creating/selecting graph attributes filters.
   *  @param view The pointer to <b>GraphView</b> object, which properties are used.
   *  @return A pointer to <b>GraphAttrFilterGUI</b> instance.
   *  @see AttrStringTemplateStorage
   *  @see GraphAttributes
   */
   GraphAttrFilterGUI* create_node_attr_filter_dialog( GraphView *view );
   
   /**
   *  The method creates GUI dialog object for displaing any graph information.
   *  @return A pointer to <b>GraphInformationGUI</b> instance.
   *  @see GraphInformationGUI
   */
   GraphInformationGUI* create_information_window( StyleFlags type = GraphGUI::DOCKABLE_GENERAL, InfoStorageSubLevel *root_level_info = 0, bool ignore_modes = false );    
   
   /**
   *  The method creates GUI dialog object for saving graph information into a file.
   *  @return A pointer to <b>GraphSaveGUI</b> instance.
   */
   GraphSaveGUI* create_graph_save_dialog( );

   /**
   *  The method creates GUI dialog object for graph generation and loading from a file.
   *  @param title The title of the dialog.
   *  @return A pointer to <b>GraphGenerationGUI</b> instance.
   *  @see GraphGenerationGUI
   */
   GraphGenerationGUI* create_graph_generation_window( const string &title );
   
   /**
   *  The method creates GUI dialog object for components managing.
   *  @param comp_manager The pointer to a <b>ComponentManager</b> object.
   *  @return A pointer to <b>GraphGUIModalWindow</b> instance.
   */
   GraphGUIModalWindow * create_comp_manager_dialog( ComponentManager *comp_manager );
   
   GraphScenarioGroupRestrictionGUI* create_scenario_group_restriction_dialog( Graph *graph );
   
   GraphInformationGUI* create_graph_statistics_info_window( const GraphStatistics &statistics, StyleFlags type = GraphGUI::DOCKABLE_GRAPH_INFO, bool ignore_modes = true );
   
   //@}   
     
   /** @name Other optional GUI.*/
   //@{
   
   /**
   *  The method shows message GUI dialog.
   *  @param message The message to be shown.
   */
   void show_message_dialog( const string &message );
    
   /**
   *  The method runs standard Choose Font Dialog.
   *  @param font The font for dialog initialising.
   *  @return Selected font.
   */
   Font run_choose_font_dialog( const Font &font );
   
   /**
   *  The method runs standard Choose Color Dialog.
   *  @param color The color for dialog initialising.
   *  @return Selected color.
   */
   Color run_choose_color_dialog( const Color &color );
   
   /**
   *  The method sets mouse cursor for main window.
   *  @param cursor_type One of the <code>CursorTypes</code> to be set.
   */
   int  set_cursor( StyleFlags cursor_type = GraphGUI::DEFAULT_CURSOR );
   
   void show_about_dialog();
   
   /**
   *  The method shows wait GUI dialog.
   *  @param message The message to be shown.
   */
   void show_wait_dialog( const string &message );
   
   /**
   *  The method hides wait GUI dialog.
   */
   void hide_wait_dialog(  );
   
   /**
   *  The method runs promt dialog with OK and Cancel buttons.
   *  @param message The message to be shown.
   *  @return <b>true</b> if user select OK button.
   */
   bool run_promt_dialog( const string &message );
   
   /**
   *  The method runs standard file selection dialog.
   *  @param title The title of the dialog.
   *  @param file_name The initial file name or empty string.
   *  @param pattern The pattern for necessary file name or empty string.
   *  @return Selected file name or empty string.
   */
   string run_fileselection_dialog( const string &title, const string &file_name, const string &pattern ); 
   
   //@}   
};

/***************************************************************************/
/***************************************************************************/

/**
* This class incapsulates all base classes for working with a graph. 
* It connects internal representation of the graph, class <code>Graph</code>
* with its view stored in the class <code>GraphView</code>.  Through 
* the method of this class it is possible to reach objects of all those classes. 
* @see Graph
* @see GraphView
*/

class GraphViewContext
{
private:
  Graph               *m_graph;
  GraphView           *m_view;
  //GraphGUI             *m_gui;

  GraphFileInfo       m_load_graph_info;
  GraphFileInfo       m_save_graph_info;

  StatusManager       *m_status_mngr;
public:
   /**
   *  This constructor creates <code>GraphViewContext</code> object.
   */
   GraphViewContext()
   {
      m_graph = 0;
      m_view  = 0;
      m_status_mngr = new StatusManager();       
   }

   ~GraphViewContext()
   {
      delete m_view;
			delete m_graph;
      delete m_status_mngr; 
       
       
   }
   
   StatusManager* get_status_mngr()
   {
      return m_status_mngr;    
   }
   
   /**
   *  Sets new <code>Graph</code> object to the context.
   *  @param graph New <code>Graph</code> object to be set.
   */
   void set_graph( Graph *graph )
	 {
     if ( m_graph != 0 )
       delete m_graph;
     
		 m_graph = graph;
		 m_graph->set_app_context(this);
	 }
	 
   /**
   *  Sets new <code>GraphView</code> object to the context.
   *  @param graph_view New <code>GraphView</code> object to be set.
   */
   void set_view( GraphView* graph_view )
   {
     if ( m_view != 0 )
       delete m_view;
     
     m_view = graph_view;
	 }

   /**
   *  Gets current graph of the context.
   *  @return Current <code>Graph</code> object of the context.
   */
   Graph* get_graph() { return m_graph; }
   
   /**
   *  Gets current graph view of the context.
   *  @return Current <code>GraphView</code> object of the context.
   */
   GraphView* get_view( ) 
   { 
     return m_view;
   }

   /**
   *  The method returns information about last loaded graph file.
   */   
   GraphFileInfo get_load_info() { return m_load_graph_info; }
   
   /**
   *  The method sets/updates graph file info.
   *  @param info The graph file info to be set.
   *  @param refresh If <b>true</b> only nonzero values of <b>info</b> will be set. Otherwise all values will replaced.
   */
   void   set_load_info( const GraphFileInfo& info, bool refresh = false ) 
   { 
     if ( !refresh )
       m_load_graph_info = info; 
     else
     {
       m_load_graph_info.refresh( info );
     }
   }

   /**
   *  The method saves graph and/or graph view properties into a file.
   *  @param file_info Information about the file and saving options.
   *  @return If <b>true</b> the file was successfully saved. 
   *  @see Storable::get_error( )
   *  @see Storable::get_error_level( )
   */
   bool save_file( const GraphFileInfo &file_info );
   
   /**
   *  The method loads graph and/or graph view properties from a file.
   *  @param file_name The file name to be loaded.
   *  @return If <b>true</b> the file was successfully loaded. 
   *  @see ::get_load_info()
   *  @see Storable::get_error( )
   *  @see Storable::get_error_level( )
   */
   bool load_file( const string &file_name );
};

};

#endif /*_GRAPHICS_H_*/
