#ifndef _GRAPHVIEPROPERTYGROUP_H_
#define _GRAPHVIEPROPERTYGROUP_H_

#include <map>

#include "types.h"
#include "drawing.h" 
#include "storable.h"

namespace GraphGraphics
{
  
/**
* The class represents a graph view properties group.
* These properties are used to draw a node (node color, shape, shape filling, shape width/height ratio (for rectangles), drawing pen and node label font/color).
* The properties groups also serve to visually discriminate some graph nodes from the others using graph node attributes.
* They allow to set a part of listed above node view properties so that unset properties will not be involved for node drawing.
* Each node can have sevaral groups assigned to. 
* @see GraphViewPropertiesStorage
*/
class GraphViewPropertyGroup: public Storable
{
	friend class GraphViewPropertiesStorage;
public:
  /** Possible node shapes. */
  enum ShapeType
  {
    CIRCLE = 0, 
    DOUBLE_CIRCLE,
    SQUARE,
    HIDEN_POINT
    //....
  };

public:
  static const StyleFlags DEFAULT; /**< Identifier of default properties group. */

private:
  StyleFlags  m_type;
  string      m_name;

  Font        m_font;
  Color       m_text_color;
  bool        m_is_font_set;
  Color       m_color;
  bool        m_is_color_set;
  StyleFlags  m_shape_type;
  double      m_shape_x_ratio;
  double      m_shape_y_ratio;
  bool        m_is_shape_ratio_set;
  bool        m_is_shape_set;
  bool        m_shape_filled;
  bool        m_is_filled_set;
  DrawingPen  m_pen;
  bool        m_is_pen_set;
  string      m_attr_string;

  GraphViewPropertyGroup( StyleFlags type, string name = "" )
  {
    m_type = type;
    m_name = name;
    m_color = Color::BLACK;
    m_text_color = Color::RED;
    m_shape_type = CIRCLE;
    m_is_font_set = false;
    m_is_color_set = false;
    m_is_shape_set = false;
    m_shape_filled = false;
    m_is_filled_set = false;
    m_is_pen_set = false;
    m_attr_string = "";
    m_shape_x_ratio = 1.0;
    m_shape_y_ratio = 1.0;
    m_is_shape_ratio_set = false;
  }
	
  void set_type( StyleFlags type ) { m_type = type; }
  void assign_attributes( const string &attr_string) { m_attr_string = attr_string; }
   
public:   
  
  /**
  * The method sets a properties group name.
  * The name can be not unique.
  * @param name The name to be set.
  */
  void set_name( const string &name ) { m_name = name; }
  
  /**
  * The method sets node label font and color property.
  * @param font The label font to be set.
  * @param text_color The label color to be set.
  * @param is_set If <b>false</b> the property will not be used for node drawing.
  * @see ::is_font_set()
  * @see ::get_font()
  * @see ::get_text_color()
  */
  void set_font( const Font &font, const Color &text_color, bool is_set = true ) { m_font = font; m_text_color = text_color; m_is_font_set = is_set; }
  
  /**
  * The method sets foregraund color property.
  * @param color The color to be set.
  * @param is_set If <b>false</b> the property will not be used for node drawing.
  * @see ::is_color_set()
  * @see ::get_color()
  */
  void set_color( const Color &color, bool is_set = true ) { m_color = color; m_is_color_set = is_set; }
  
  /**
  * The method sets shape property.
  * @param shape_type The shape type to be set. It should be on of the <code>ShapeType</code> constants.
  * @param is_set If <b>false</b> the property will not be used for node drawing.
  * @see ::is_shape_set()
  * @see ::get_shape_type()
  */
  void set_shape_type( StyleFlags shape_type, bool is_set = true ) { m_shape_type = shape_type; m_is_shape_set = is_set; }
  
  /**
  * The method sets shape to be filled with foregraund color.
  * @param filled The shape filling flag.
  * @param is_set If <b>false</b> the property will not be used for node drawing.
  * @see ::is_filled_set()
  * @see ::is_shape_filled()
  */
  void set_shape_filled( bool filled, bool is_set = true ) { m_shape_filled = filled; m_is_filled_set = is_set; }
  
  /**
  * The method sets drawing pen property.
  * @param pen The drawing pen to be set.
  * @param is_set If <b>false</b> the property will not be used for node drawing.
  * @see ::is_pen_set()
  * @see ::get_pen()
  */
  void set_pen( const DrawingPen &pen, bool is_set = true ) { m_pen = pen; m_is_pen_set = is_set; }
  
  /**
  * The method sets shape width/height ratio (for rectangles only).
  * @param is_set If <b>false</b> the property will not be used for node drawing.
  * @see ::is_shape_ratio_set()
  * @see ::get_shape_x_ratio()
  * @see ::get_shape_y_ratio()
  */
  void set_shape_xy_ratio( double x_ratio, double y_ratio, bool is_set = true  ) { m_shape_x_ratio = x_ratio; m_shape_y_ratio = y_ratio; m_is_shape_ratio_set = is_set;}
      
  /**
  * Parameterless constructor.
  */
  GraphViewPropertyGroup()
  {
    m_type = DEFAULT;
    m_name = "";
    m_color = Color::BLACK;
    m_text_color = Color::RED;
    m_shape_type = CIRCLE;
    m_is_font_set = false;
    m_is_color_set = false;
    m_is_shape_set = false;
    m_shape_filled = false;
    m_is_filled_set = false;
    m_is_pen_set = false;
    m_attr_string = "";
    m_shape_x_ratio = 1.0;
    m_shape_y_ratio = 1.0;
    m_is_shape_ratio_set = false;
  }
  
  /**
  * Copy constructor.
  */
  GraphViewPropertyGroup( const GraphViewPropertyGroup& group )
  {
    m_type = group.m_type;
    m_name = group.m_name;
    m_font = group.m_font;
    m_text_color = group.m_text_color;
    m_is_font_set = group.m_is_font_set;
    m_color = group.m_color;
    m_is_color_set = group.m_is_color_set;
    m_shape_type = group.m_shape_type;
    m_is_shape_set = group.m_is_shape_set;
    m_shape_filled = group.m_shape_filled;
    m_is_filled_set = group.m_is_filled_set;
    m_pen = group.m_pen;
    m_is_pen_set = group.m_is_pen_set;
    m_attr_string = group.m_attr_string;
    m_shape_x_ratio = group.m_shape_x_ratio;
    m_shape_y_ratio = group.m_shape_y_ratio;
    m_is_shape_ratio_set = group.m_is_shape_ratio_set;
  }

  /**
  * Assignment operator.
  */
  GraphViewPropertyGroup& operator = ( const GraphViewPropertyGroup& group )
  {
    m_type = group.m_type;
    m_name = group.m_name;
    m_font = group.m_font;
    m_text_color = group.m_text_color;
    m_is_font_set = group.m_is_font_set;
    m_color = group.m_color;
    m_is_color_set = group.m_is_color_set;
    m_shape_type = group.m_shape_type;
    m_is_shape_set = group.m_is_shape_set;
    m_shape_filled = group.m_shape_filled;
    m_is_filled_set = group.m_is_filled_set;
    m_pen = group.m_pen;
    m_is_pen_set = group.m_is_pen_set;
    m_attr_string = group.m_attr_string;
    m_shape_x_ratio = group.m_shape_x_ratio;
    m_shape_y_ratio = group.m_shape_y_ratio;
    m_is_shape_ratio_set = group.m_is_shape_ratio_set;
      
    return *this;
  }
  
  /**
  * Destructor.
  */
  ~GraphViewPropertyGroup() {}

  /**
  * The method returns the properties group unique identifier.
  * @see GraphViewPropertiesStorage::add_view_group(...)
  * @see GraphViewPropertiesStorage::get_view_group(...)
  * @see GraphViewPropertiesStorage::change_view_group(...)
  * @see GraphViewPropertiesStorage::reassign_view_group(...)
  * @see GraphViewPropertiesStorage::get_first_view_group_id(...)
  * @see GraphViewPropertiesStorage::get_next_view_group_id(...)
  */
  StyleFlags get_type() const { return m_type; }
  
  /**
  * The method returns the properties group name.
  */
  string     get_name() const { return m_name; }

  /**
  * The method returns the label font property value.
  */
  Font       get_font() const { return m_font; }
  
  /**
  * The method returns the label color property value.
  */
  Color      get_text_color() const { return m_text_color; }
  
  /**
  * The method verifies whether label font and color are set.
  */
  bool       is_font_set() const { return m_is_font_set; }
  
  /**
  * The method returns the foreground color property value.
  */
  Color      get_color() const { return m_color; }
  
  /**
  * The method verifies whether foreground color property is set.
  */
  bool       is_color_set() const { return m_is_color_set; }
  
  /**
  * The method returns the shape property value.
  */
  StyleFlags get_shape_type() const { return m_shape_type; }
  
  /**
  * The method verifies whether shape property is set.
  */
  bool       is_shape_set() const { return m_is_shape_set; }
  
  /**
  * The method returns the drawing pen property value.
  */
  DrawingPen get_pen() const { return m_pen; }
  
  /**
  * The method verifies whether drawing pen property is set.
  */
  bool       is_pen_set() const { return m_is_pen_set; }
  
  /**
  * The method returns the shape filling property value.
  */
  bool       is_shape_filled() const { return m_shape_filled; }
  
  /**
  * The method verifies whether shape filling property is set.
  */
  bool       is_filled_set() const { return m_is_filled_set; }
  
  /**
  * The method verifies whether shape ratio property is set.
  */
  bool       is_shape_ratio_set() const { return m_is_shape_ratio_set; }
  
  /**
  * The method returns the shape width ratio index.
  */
  double     get_shape_x_ratio() const { return m_shape_x_ratio; }
  
  /**
  * The method returns the shape height ratio index.
  */
  double     get_shape_y_ratio() const { return m_shape_y_ratio; }

  /**
  * The method returns graph attributes pattern that specifies nodes the properies group is assigned to.
  * @see GraphViewPropertiesStorage::add_view_group(...)
  * @see GraphViewPropertiesStorage::reassign_view_group(...)
  */
  string     get_assigned_attributes() const { return m_attr_string; }
  
  /**
  * This static method creates a <code>GeometryShape</code> object specified by shape properties.
  * @param shape_type Type of the shape. It should be one of the <code>ShapeType</code> constants.
  * @param center Coordinates of geometric center of the shape.
  * @param range  Diameter of the shape.
  */
  static GeometryShape* get_node_shape( StyleFlags shape_type, const TwoDPoint &center, OneDValue range, double rect_dx_ratio = 1.0, double rect_dy_ratio = 1.0 )
  {
    GeometryShape *res = 0;
  
    switch ( shape_type )
    {
    case GraphViewPropertyGroup::HIDEN_POINT:
      res = new HidenPoint();
      break;
      
    case GraphViewPropertyGroup::SQUARE:
      res = new Rectangle(rect_dx_ratio, rect_dy_ratio);
      break;
      
    case GraphViewPropertyGroup::DOUBLE_CIRCLE:
      res  = new DoubleCircle();
      break;

    case GraphViewPropertyGroup::CIRCLE:
    default:
      res  = new Circle();
        
    }

    res->set_center( center );
    res->set_range( range );

    return res;
  }
   
  /**
  * This static method creates a <code>GeometryShape</code> object specified by the shape properties of the group.
  * @param center Coordinates of geometric center of the shape.
  * @param range  Diameter of the shape.
  */
  GeometryShape* get_node_shape( const TwoDPoint &center, OneDValue range )
  {
    GeometryShape* res = 0;
     
    if ( m_is_shape_ratio_set )
      res = get_node_shape( m_shape_type, center, range, m_shape_x_ratio, m_shape_y_ratio );
    else
      res = get_node_shape( m_shape_type, center, range );
       
    return res;
  }
  
  /**
  * The method writes the properties group to a stream.
  * @see Storable::write_to(...)
  */
  void write_to( ostream& s_out, unsigned long write_options = 0 );
  
  /**
  * The method reads the properties group from a stream.
  * @see Storable::read_from(...)
  */
  bool read_from( istream& s_in, unsigned long read_options = 0 );
};

typedef map<StyleFlags, GraphViewPropertyGroup *> ViewGroupMap;

/**
* The class represents container of all view properties for graph drawing.
* @see GraphView::get_view_properties_storage()
*/
class GraphViewPropertiesStorage
{
  friend class GraphView;
    
public:    
  /** @name Predefined edge mark constants. */
  //@{
  enum EdgeMark
  {
    EDGE_UNMARKED = 0,     /**< Unmarked edge. */
    EDGE_SELECTED = 1,     /**< Mark for selecting edges. @see GraphViewPropertiesStorage::set_selected_edge_color(...)*/
    EDGE_IN_SCENARIO = 2,  /**< Mark for selecting graph path. @see GraphViewPropertiesStorage::set_selected_path_color(...)*/
    EDGE_RESERVED1 = 3,    /**< User-defined mark. @see GraphViewPropertiesStorage::set_reserved_edge_color(...)*/
    EDGE_RESERVED2 = 4,    /**< User-defined mark. @see GraphViewPropertiesStorage::set_reserved_edge_color(...)*/
    EDGE_RESERVED3 = 5     /**< User-defined mark. @see GraphViewPropertiesStorage::set_reserved_edge_color(...)*/
  }; 
  //@}  
  
  static const StyleFlags MIN_USER_MARK_ID;  
private:
  
  struct EdgeMarkProperties
  {
    string mark_name;
    Color  mark_color;
  };
  typedef map<StyleFlags, EdgeMarkProperties*> EdgeMarkMap;
  
  static StyleFlags m_last_group_type;
  static StyleFlags m_last_edge_mark_id;

  GraphViewPropertyGroup *m_def_view_properties;
  ViewGroupMap            m_view_groups;
  GraphView              *m_view;
  EdgeMarkMap             m_edge_marking;

  Color      m_backgrnd_color;
  Color      m_selected_color;
  Color      m_edge_color;
  DrawingPen m_edge_pen;
  Color      m_selected_edge_color;
//  Color      m_scenario_edge_color;
  Color      m_outgoing_edge_color;
  Color      m_incoming_edge_color;
  //Color      m_reserved_colors[3]; 

private:
  void add_node_to_group( NodeIndex node_id, GraphViewPropertyGroup *group );  
      
  void remove_nodes_from_group( StyleFlags view_type );
  
  GraphViewPropertiesStorage( GraphView *view ); 
  
  ViewGroupMap& get_view_groups() { return m_view_groups; }
  
  GraphViewPropertiesStorage& operator = ( const GraphViewPropertiesStorage& prop_storage )
  {
    m_backgrnd_color = prop_storage.m_backgrnd_color;
    m_selected_color = prop_storage.m_selected_color;
    m_edge_color     = prop_storage.m_edge_color;
    m_edge_pen       = prop_storage.m_edge_pen;
    m_selected_edge_color = prop_storage.m_selected_edge_color;
    m_incoming_edge_color = prop_storage.m_incoming_edge_color;
 //   m_scenario_edge_color = prop_storage.m_scenario_edge_color;
    m_outgoing_edge_color = prop_storage.m_outgoing_edge_color;
 //   memcpy( m_reserved_colors, prop_storage.m_reserved_colors, sizeof(prop_storage.m_reserved_colors));
    
    remove_all_view_groups();
    
    change_view_group( GraphViewPropertyGroup::DEFAULT, prop_storage.get_view_group(GraphViewPropertyGroup::DEFAULT) );
    
    for ( ViewGroupMap::const_iterator vp = prop_storage.m_view_groups.begin(); vp != prop_storage.m_view_groups.end(); vp++ )
    {
      GraphViewPropertyGroup* group = (GraphViewPropertyGroup*)(vp->second);
      add_view_group( group->get_assigned_attributes(), *group );
    }
    
    remove_all_edge_marks();
    
    for ( EdgeMarkMap::const_iterator mp = prop_storage.m_edge_marking.begin(); mp != prop_storage.m_edge_marking.end(); mp++ )
    {
      EdgeMarkProperties* mark_properties = new EdgeMarkProperties();
      mark_properties->mark_name = ((EdgeMarkProperties*)mp->second)->mark_name;
      mark_properties->mark_color = ((EdgeMarkProperties*)mp->second)->mark_color;
      
      m_edge_marking[mp->first] = mark_properties;
    }
    
    
    return *this;
  }
  
public:
  ~GraphViewPropertiesStorage( )
  {
    for ( ViewGroupMap::iterator p = m_view_groups.begin(); p != m_view_groups.end(); p++ )
     delete (GraphViewPropertyGroup*)(p->second);
    m_view_groups.clear();
    delete m_def_view_properties;
  };
  
  /** 
  * @name General graph view properties.
  */
  
  //@{
  /**
  * The method returns background color.
  */
  Color& get_background_color() { return m_backgrnd_color; }
  
  /**
  * The method sets background color.
  * @param color The <b>Color</b> object to be set.
  */
  void  set_background_color( const Color &color ) { m_backgrnd_color = color; }
   
  /**
  * The method returns color used for drawing of unmarked edges.
  */
  Color& get_edge_color() { return m_edge_color; }
  
  /**
  * The method sets color used for drawing of unmarked edges.
  * @param color The <b>Color</b> object to be set.
  * @see GraphView::mark_edge(...)
  */
  void  set_edge_color( const Color &color ) { m_edge_color = color; }
   
  /**
  * The method returns drawing pen used for edge drawing.
  */
  DrawingPen& get_edge_pen() { return m_edge_pen; }
  
  /**
  * The method sets drawing pen used for edge drawing.
  * @param pen The <b>DrawingPen</b> object to be set.
  */
  void  set_edge_pen( const DrawingPen &pen ) { m_edge_pen = pen; }

  /**
  * The method sets color used for visually marking of selected nodes.
  * @param pen The <b>Color</b> object to be set.
  */
  void set_selected_node_color( const Color &selected_color) { m_selected_color = selected_color; }
  
  /**
  * The method returns color used for visually marking of selected nodes.
  */
  Color& get_selected_node_color() { return m_selected_color; }

  /**
  * The method sets color used for drawing of edges marked with <b>GraphView::EDGE_SELECTED</b> mark.
  * @param pen The <b>Color</b> object to be set. 
  * @see GraphView::mark_edge(...)
  */
  void set_selected_edge_color( const Color &selected_color) { m_selected_edge_color = selected_color; }
  
  /**
  * The method returns color used for drawing of edges marked with <b>GraphView::EDGE_SELECTED</b> mark.
  */
	Color& get_selected_edge_color() { return m_selected_edge_color; }
  
  /**
  * The method sets color used for drawing of edges marked with <b>GraphView::EDGE_IN_SCENARIO</b> mark.
  * It is usually used for selecting graph path.
  * @param pen The <b>Color</b> object to be set. 
  * @see GraphView::mark_edge(...)
  */
  //void set_selected_path_color( const Color &color) { m_scenario_edge_color = color; }
  
  /*
  * The method returns color used for drawing of edges marked with <b>GraphView::EDGE_IN_SCENARIO</b> mark.
  */
	//Color& get_selected_path_color() { return m_scenario_edge_color; }
  
  /**
  * The method sets color used for drawing of edges that are incoming to selected nodes.
  * @param pen The <b>Color</b> object to be set.
  */
  void set_incoming_edge_color( const Color &color) { m_incoming_edge_color = color; }
  
  /**
  * The method returns color used for drawing of edges that are incoming to selected nodes.
  */
	Color& get_incoming_edge_color() { return m_incoming_edge_color; }
  
  /**
  * The method sets color used for drawing of edges that are outgoing from selected nodes.
  * @param pen The <b>Color</b> object to be set.
  */
  void set_outgoing_edge_color( const Color &color) { m_outgoing_edge_color = color; }
  
  /**
  * The method returns color used for drawing of edges that are outgoing from selected nodes.
  */
	Color& get_outgoing_edge_color() { return m_outgoing_edge_color; }
  
  /*
  * The method sets color used for drawing of edges marked with one of <b>GraphView::EDGE_RESERVEDX</b> marks.
  * @param pen The <b>Color</b> object to be set.
  */
  //void set_reserved_edge_color( int reserved_mark_index, const Color &color); 
  
  /*
  * The method returns color used for drawing of edges marked with one of <b>GraphView::EDGE_RESERVEDX</b> marks.
  */
	//Color& get_reserved_edge_color( int reserved_mark_index );
  //@}
  
  /** @name Working with node view properties groups. */
  
  //@{
  
  /**
  * The method adds a new node view properties group.
  * @param attr_string The graph attributes string that specifies graph nodes the group to be assigned to. The string will be stored in the added group object.
  * @param group The properties group object to be copied.
  * @param assign If <b>false</b> assigning the group will not be done during the group adding but <code>attr_string</code> will be stored in the added group object in any case. 
  * @return The the unique identifier (type) for just added group. The identifier is used in group iteration operations and to get access to a group.
  * @see ::reassign_view_group(...)
  * @see GraphViewPropertyGroup::get_assigned_attributes()
  * @see NodeAttributeList
  */
  StyleFlags add_view_group( const string &attr_string, const GraphViewPropertyGroup &group, bool assign = true );
  
  /**
  * The method reassigns a node view properties group with specified identifier to another nodes.
  * The default node view properties group cannot be reassigned (it is always assigned to all nodes).
  * @param type The unique identifier of a node view properties group which assignment to nodes should be changed.
  * @param attr_string The graph attributes string that specifies graph nodes the group to be assigned to. The string will be stored in the group object.
  * @see ::change_view_group(...)
  */
  void reassign_view_group( StyleFlags type, const string &attr_string );

  /**
  * The method change node view properties of a group with specified identifier.
  * The method does not change assignment to nodes.
  * @param type The unique identifier of a node view properties group which properties values should be changed.
  * @param new_properties The node view properties group which properties should be set.
  */
  void change_view_group( StyleFlags type, const GraphViewPropertyGroup &new_properties )
  {
    GraphViewPropertyGroup *group = 0;
    bool is_ok = false;
    if ( type == GraphViewPropertyGroup::DEFAULT )
    {
      group = m_def_view_properties;
      is_ok = true;
    }
    else if ( m_view_groups.find( type ) != m_view_groups.end() )
    {
      group = m_view_groups[ type ];
      is_ok = true;
    }
      
    if ( is_ok )
    {
      group->set_name( new_properties.get_name() );
      group->set_color( new_properties.get_color(), new_properties.is_color_set() );
      group->set_font( new_properties.get_font(), new_properties.get_text_color(), new_properties.is_font_set() );
      group->set_shape_type( new_properties.get_shape_type(), new_properties.is_shape_set() );
      group->set_shape_filled( new_properties.is_shape_filled(), new_properties.is_filled_set() );
      group->set_pen( new_properties.get_pen(), new_properties.is_pen_set() );
      group->set_shape_xy_ratio( new_properties.get_shape_x_ratio(), new_properties.get_shape_y_ratio(), new_properties.is_shape_ratio_set() );
    }
  }
   
  /**
  * The method returns a node view properties group with the specific identifier.
  * If there is no such group the default one will be returned.
  */
  GraphViewPropertyGroup get_view_group( StyleFlags type ) const
  {
    GraphViewPropertyGroup *group = m_def_view_properties;
    
    ViewGroupMap::const_iterator p;

    if ( (p = m_view_groups.find( type )) != m_view_groups.end() )
    {
      group = p->second;
    }

    return *group;
  }

  /**
  * The method returns identifier of the first node view properties group in the storage.
  * If the storage does not contain groups the identifier of default group (<code>GraphViewPropertyGroup::DEFAULT</code>) will be returned.
  * @see get_next_view_group_id(...)
  */
  StyleFlags get_first_view_group_id( ) const
  {
    StyleFlags type = GraphViewPropertyGroup::DEFAULT;

    typedef ViewGroupMap::const_iterator MI;

    if ( !m_view_groups.empty() )
    {
      MI p = m_view_groups.begin();

      type = p->first;
    }

    return type;
  }

  /**
  * The method returns identifier of next node view properties group in the storage.
  * If the storage does not contain any more groups the identifier of default group (<code>GraphViewPropertyGroup::DEFAULT</code>) will be returned.
  * @param prev_type The group identifier for finding next one in the storage.
  * @see get_first_view_group_id( )
  */
  StyleFlags get_next_view_group_id( StyleFlags prev_type ) const
  {
    StyleFlags type = GraphViewPropertyGroup::DEFAULT;

    typedef ViewGroupMap::const_iterator MI;

    MI p = m_view_groups.find( prev_type );

    if ( p != m_view_groups.end() )
    {
      p++;
      if ( p != m_view_groups.end() )        
        type = p->first;
    }

    return type;
  }
  
  /**
  * The method removes a node view properties group from the storage.
  * The default group cannot be removed.
  * @param type The identifier of node view properties group to be removed.
  */
  void remove_view_group( StyleFlags type )
  {
    if ( m_view_groups.find( type ) != m_view_groups.end() )
    {
      remove_nodes_from_group( type );
      delete (GraphViewPropertyGroup*)(m_view_groups[ type ]);
      m_view_groups.erase( type );
    }
  }
  
  /**
  * The method removes all node view properties groups from the storage.
  * The default group is removed.
  */
  void remove_all_view_groups()
  {
    for ( ViewGroupMap::iterator p = m_view_groups.begin(); p != m_view_groups.end(); p++ )
    {
      remove_nodes_from_group( p->first );
      delete ((GraphViewPropertyGroup*)(p->second));
    }
    m_view_groups.clear();
  }
  //@}
  
  /** @name Working edge marking. */
  
  //@{
  StyleFlags add_edge_mark( const string& mark_name, const Color& mark_color )
  {
    StyleFlags mark_id = m_last_edge_mark_id++;
    EdgeMarkProperties *new_mark = new EdgeMarkProperties( );
  
    new_mark->mark_name = mark_name;
    new_mark->mark_color = mark_color;
  
    m_edge_marking[ mark_id ] = new_mark;
  
    return mark_id;
  }
  
  bool is_edge_mark_exist( StyleFlags mark_id ) const
  {
    return (m_edge_marking.find( mark_id ) != m_edge_marking.end());
  }
  
  StyleFlags get_first_edge_mark_id( ) const
  {
    StyleFlags mark_id = EDGE_UNMARKED;
    
    if ( !m_edge_marking.empty() )
    {
      EdgeMarkMap::const_iterator p = m_edge_marking.begin();

      mark_id = p->first;
    }

    return mark_id;
  }
  
  StyleFlags get_next_edge_mark_id( StyleFlags mark_id ) const
  {
    StyleFlags mark = EDGE_UNMARKED;

    EdgeMarkMap::const_iterator p = m_edge_marking.find( mark_id );

    if ( p != m_edge_marking.end() )
    {
      p++;
      if ( p != m_edge_marking.end() )        
         mark = p->first;
    }
    
    return  mark;
  }
  
  void remove_edge_mark( StyleFlags mark_id );
    
  void remove_all_edge_marks();
  
  void set_edge_mark_color( StyleFlags mark_id, const Color& mark_color )
  {
    EdgeMarkMap::iterator p;
    if ( mark_id == EDGE_UNMARKED )
      m_edge_color = mark_color;
    else if ( (p = m_edge_marking.find( mark_id )) != m_edge_marking.end() )
    {
      EdgeMarkProperties* mark_properties = (EdgeMarkProperties*)(p->second);
      mark_properties->mark_color = mark_color;
    }
  }

  StyleFlags get_edge_mark_id( const string& mark_name ) const
  {
    StyleFlags mark_id = EDGE_UNMARKED;
    
    for ( EdgeMarkMap::const_iterator p = m_edge_marking.begin(); (p != m_edge_marking.end()) && (mark_id == EDGE_UNMARKED); p++ )
    {
      EdgeMarkProperties* mark_properties = (EdgeMarkProperties*)(p->second);
      if ( mark_properties->mark_name == mark_name )
        mark_id = p->first;
    }
    
    return mark_id;
  }
  
  string get_edge_mark_name( StyleFlags mark_id ) const
  {
    string mark_name = "";
    EdgeMarkMap::const_iterator p;
    
    if ( (p = m_edge_marking.find( mark_id )) != m_edge_marking.end() )
    {
      EdgeMarkProperties* mark_properties = (EdgeMarkProperties*)(p->second);
      mark_name = mark_properties->mark_name;
    }

    return mark_name;
  }
  
  Color get_edge_mark_color( StyleFlags mark_id ) const
  {
    Color mark_color = m_edge_color;
    EdgeMarkMap::const_iterator p;
    
    if ( (p = m_edge_marking.find( mark_id )) != m_edge_marking.end() )
    {
      EdgeMarkProperties* mark_properties = (EdgeMarkProperties*)(p->second);
      mark_color = mark_properties->mark_color;
    }

    return mark_color;
  }
  //@}
};

  
};

#endif 

