#ifndef SCENARIOAPP_H_HEADER_INCLUDED_C103A9D5
#define SCENARIOAPP_H_HEADER_INCLUDED_C103A9D5

#include "ComponentManager.h"
#include "EventHandler.h"

namespace GraphGraphics
{
  enum PreferencesOffsets
  {
    GRAPH_LAYOUT_OFF = 0,
    BEHAVIOR_MODES = 1,
    EXTRA_BEHAVIOR_MODES = 5,
    ACTIVE_COMP_OFF = 13
  };

  /**
  * <code>ScenarioApp</code> is a main class that supply the application with access to the functionality of the scenario library.
  * It stores main scenario graph context and also it allows to register and store additional components. This class offers for the user 
  * access to the event handler object for default handling of the scenario events. It also incapsulates object of the class
  * <code>GraphGUI</code>.
  * @see GraphViewContext
  * @see GraphGUI
  * @see CompInterface  
  */  
class ScenarioApp: public ComponentManager
{
public:
  enum ScenarioModes
  {
    SINGLE_INFO_WINDOW = 1,
    SINGLE_ATTR_WINDOW = 2,
    SHOW_ATTR_ON_MOUSE_OVER = 4,
    CLOSE_OPEN_DIALOG_AFTER_GRAPH_LOAD = 8,
    EDGE_SELECTING = 16,
    SHOW_ATTR_ON_NODE_SELECT = 32
  };
  
  enum ExtraModes
  {
    DOCK_INFO_ON_START = 1,
    DOCK_ATTR_ON_START = 2,
    USE_ATTR_ALIASES   = 4,
    OUTPUT_GEN_GRAPH_METRICS = 8
  };
  
public:
  static void initialize_globals();
  /**
  *  This constructor creates <b>ScenarioApp</b> object and passes OS-depended structure.
  *  @param os_data OS-depended structure.
  */
  ScenarioApp( void *os_data );

  /**
  *  Destructor of <b>ScenarioApp</b> object.
  */
  ~ScenarioApp();

  //behavior modes
  void set_behavior_modes( unsigned int modes, bool enable )
  {
    if ( !enable )
      m_e_handler->m_modes &= (~modes);
    else
      m_e_handler->m_modes |= modes;
    
    if ( modes & EDGE_SELECTING )
      m_e_handler->enable_edge_selecting_mode( enable );
  }
  unsigned int get_behavior_modes() { return m_e_handler->m_modes; }
  
  void set_extra_behavior_modes( unsigned int modes, bool enable ) 
  {
    if ( !enable )
      m_extra_modes &= (~modes);
    else
      m_extra_modes |= modes;
  }
  
  unsigned int get_extra_behavior_modes() { return m_extra_modes; }

  /**
  *  This function returns <b>ScenarioEventHandler</b> object that is used for default handling of scenario events.
  *  @return pointer to <b>ScenarioEventHandler</b> object.
  *  @see ScenarioEventHandler
  */
  ScenarioEventHandler* get_event_handler() { return m_e_handler; }

  /**
  *  This function registers additional component.
  *  @param comp pointer to the object inherited from <b>CompInterface</b> class.
  *  @return index of the registered component  
  */
  ComponentIndex register_component( CompInterface *comp );

  /**
  *  This function returns component with index <b>comp_id</b>.
  *  @param comp_id index of the component to be returned.
  *  @return pointer to <b>CompInterface</b>  
  *  @see CompInterface
  */
  CompInterface* get_registered_component( ComponentIndex comp_id ) 
  { 
    if ( comp_id >= 0 && comp_id < (ComponentIndex)m_components.size() ) 
      return m_components[comp_id]; 
    
    return 0;
  }
  
  /**
  *  This function returns number of registered component.
  *  @return number of registered components  
  */
  int            count_registered_components() { return m_components.size(); }

  /**
  *  This function returns pointer to <b>GraphGUI</b> object.
  *  @return pointer to <b>GraphGUI</b> object.
  *  @see GraphGUI  
  */
  GraphGUI* get_GUI() { return m_gui; }

  /**
  *  This function returns pointer to <b>GraphViewContext</b> object.
  *  @return pointer to <b>GraphViewContext</b> object.
  *  @see GraphViewContext  
  */
  GraphViewContext* get_graph_view_context() 
  { 
    return m_main_context; 
  }

private:
  //##ModelId=3EF9F65900E1
  GraphViewContext *m_main_context;

  //##ModelId=3EFAA6CD03C3
  ScenarioEventHandler *m_e_handler;

  //##ModelId=3EFAAF06013E
  GraphGUI *m_gui;

  ComponentsVector m_components;
  unsigned int m_extra_modes;
};

};


#endif /* SCENARIOAPP_H_HEADER_INCLUDED_C103A9D5 */
