#include "shapes.h"

using namespace GraphGraphics;

void LineSegment::set_range( OneDValue range )
{
   TwoDPoint center = get_center();

   if ( m_direction.get_y() == 0 )
   {
      m_start_point.set_x( center.get_x() - range );
      m_end_point.set_x( center.get_x() + range );

      m_start_point.set_y( center.get_y() );
      m_end_point.set_y( center.get_y() );
   }
   else if ( m_direction.get_x() == 0 )
   {
      m_start_point.set_y( center.get_y() - range );
      m_end_point.set_y( center.get_y() + range );

      m_start_point.set_x( center.get_x() );
      m_end_point.set_x( center.get_x() );
   }
   else
   {
      TwoDPoint line_direction( m_direction.get_x() + center.get_x(),
                                m_direction.get_y() + center.get_y() );
      TwoDPoint inverse_line_direction( -m_direction.get_x() + center.get_x(),
                                -m_direction.get_y() + center.get_y() );

      m_end_point = get_scaled_line_point( center, line_direction, (double)range / (double)center.distance_to( line_direction ) );
      m_start_point = get_scaled_line_point( center, inverse_line_direction, (double)range / (double)center.distance_to( line_direction ) );
   }
}

int LineSegment::intersect( const LineSegment &line )
{
   int res    = 1;	
   int lstart = on_line( line.m_start_point );
   int lend   = on_line( line.m_end_point );
   int start  = line.on_line( m_start_point );
   int end    = line.on_line( m_end_point );

   OneDValue min_l_x = GGMIN(line.m_start_point.get_x(), line.m_end_point.get_x());
   OneDValue max_l_x = GGMAX(line.m_start_point.get_x(), line.m_end_point.get_x());
   OneDValue min_l_y = GGMIN(line.m_start_point.get_y(), line.m_end_point.get_y());
   OneDValue max_l_y = GGMAX(line.m_start_point.get_y(), line.m_end_point.get_y());

   OneDValue min_x = GGMIN(m_start_point.get_x(), m_end_point.get_x());
   OneDValue max_x = GGMAX(m_start_point.get_x(), m_end_point.get_x());
   OneDValue min_y = GGMIN(m_start_point.get_y(), m_end_point.get_y());
   OneDValue max_y = GGMAX(m_start_point.get_y(), m_end_point.get_y());

   if ( (lstart == lend && lstart != 0) || //whole line segment lies at one side from the segment
        (start == end && start != 0) ) //whole the segment lies at one side from line 
      // segments are not intersected
      res = 0;
   else if ( lend == 0 && lstart == 0 ) // both segments lies at one line
   {
      if ( (min_l_x > max_x) || 
           (max_l_x < min_x) || 
           (min_l_y > max_y) || 
           (max_l_y < min_y) )
           // segments are not covered
           res = 0;
      else if ( (min_l_x == max_x || max_l_x == min_x) && 
                (min_l_y == max_y || max_l_y == min_y) )
         // segments are covered in one of the end point
         res = 1;
      else //segments are covered
         if ( !(is_empty() || line.is_empty()) )
            res = -1;
   }
   //else segments are intersected - res = 1
   

   return res;
}

int LineSegment::intersection( const LineSegment &line, TwoDPoint* result_point )
{
   int res = 1;
/*   if ( (res = intersect( line )) != 1 || result_point == 0 )
      return res;
*/
   TwoDPoint res_point;
   //else there is one intersection point
  if ( is_empty() )
  {
    if ( line.inside( m_start_point ) )
      res_point = m_start_point;
    else
      res = 0;
  }
  else if ( line.is_empty() )
  {
    if ( inside( line.m_start_point ) )
      res_point = line.m_start_point;
    else
      res = 0;
  }    
  else
  {
      //a*x +b*y + c = 0;
      double a1 = (double)(m_end_point.get_y() - m_start_point.get_y());
      double a2 = (double)(line.m_end_point.get_y() - line.m_start_point.get_y());
      double b1 = -(double)(m_end_point.get_x() - m_start_point.get_x());
      double b2 = -(double)(line.m_end_point.get_x() - line.m_start_point.get_x());
      double c1 = (double)m_start_point.get_y() * (double)(m_end_point.get_x() - m_start_point.get_x()) -
                  (double)m_start_point.get_x() * (double)(m_end_point.get_y() - m_start_point.get_y());
      double c2 = (double)line.m_start_point.get_y() * (double)(line.m_end_point.get_x() - line.m_start_point.get_x()) -
                  (double)line.m_start_point.get_x() * (double)(line.m_end_point.get_y() - line.m_start_point.get_y());

      if ( a1== 0.0 && a2 == 0.0 ||
           b1 == 0.0 && b2 == 0.0 ||
           (a1 * b2 == a2 * b1) && a1 != 0.0 && b1 != 0.0 && a2 != 0.0 && b2 != 0.0 )
      {
        
        if ( a1 != 0.0 && c1 * a2 == c2 * a1 ||
             b1 != 0.0 && c1 * b2 == c2 * b1 )
        {
          //lines are covered          
          if ( line.inside( m_start_point ) )
            res_point = m_start_point;
          else if ( line.inside( m_end_point ))
            res_point = m_end_point;
          else if ( inside( line.m_start_point ) )
            res_point = line.m_start_point;
          else if ( inside( line.m_end_point ) )
            res_point = line.m_end_point;
          else
            res_point = m_start_point;
        }
        else
          res = 0;
          
      }
      else if ( a1 == 0.0 ) 
      {
         //a2 cannot be equal to 0

         res_point.set_y( m_end_point.get_y() );
         //x = -(b2/a2 * y + c2/a2);
         double x = - (b2 * (double)res_point.get_y() + c2) / a2;

         res_point.set_x( (OneDValue)(DROUND( x )));
      }
      else if ( a2 == 0.0 )
      {
         //a1 cannot be equal to 0

         res_point.set_y( line.m_end_point.get_y() );
         //x = -(b1/a1 * y + c1/a1);
         double x = - (b1 * (double)res_point.get_y() + c1) / a1;

         res_point.set_x( (OneDValue)(DROUND( x )));
      }
      else if ( b1 == 0.0 )
      {
         //b2 cannot be equal to 0

         res_point.set_x( m_end_point.get_x() );
         //y = -(a2/b2 * x + c2/b2);
         double y = - (a2 * (double)res_point.get_x() + c2) / b2;

         res_point.set_y( (OneDValue)(DROUND( y )));
      }
      else if ( b2 == 0.0 )
      {
         //b1 cannot be equal to 0

         res_point.set_x( line.m_end_point.get_x() );
         //y = -(a1/b1 * x + c1/b1);
         double y = - (a1 * (double)res_point.get_x() + c1) / b1;

         res_point.set_y( (OneDValue)(DROUND( y )));
      }
      else
      {
         // x = - (b1/a1 * y + c1/a1)
         // y = (c1 * a2/a1 - c2) / (b2 - a2 * b1/a1)
         double y = (c1 * a2 - a1 * c2) / (a1 * b2 - a2 * b1 );
         double x = -((b1 * y + c1) / a1);

         res_point.set_y( (OneDValue)DROUND( y ) );
         res_point.set_x( (OneDValue)DROUND( x ));
      }
   }

   *result_point = res_point;

   return res;
}

TwoDPoint LineSegment::get_nearest_to( const TwoDPoint &point ) const 
{
   TwoDPoint res;

   if ( is_empty() )
      res = m_start_point;
   else if ( inside(point) )
      res = point;
   else if ( m_start_point.get_x() == m_end_point.get_x() )
   {
      OneDValue max_y = GGMAX(m_start_point.get_y(), m_end_point.get_y());
      OneDValue min_y = GGMIN(m_start_point.get_y(), m_end_point.get_y());

      res.set_x( m_start_point.get_x() );

      if ( point.get_y() >= min_y &&
           point.get_y() <= max_y )
         res.set_y( point.get_y() );
      else if ( point.get_y() > max_y )
         res.set_y( max_y );
      else
         res.set_y( min_y );
   }
   else if ( m_start_point.get_y() == m_end_point.get_y() )
   {
      OneDValue max_x = GGMAX(m_start_point.get_x(), m_end_point.get_x());
      OneDValue min_x = GGMIN(m_start_point.get_x(), m_end_point.get_x());

      res.set_y( m_start_point.get_y() );

      if ( point.get_x() >= min_x &&
           point.get_x() <= max_x )
         res.set_x( point.get_x() );
      else if ( point.get_x() > max_x )
         res.set_x( max_x );
      else
         res.set_x( min_x );
   }
   else
   {
      // a*x + b*y + c = 0 - this line (a <> 0 and b <> 0)
      // y = -a/b * x - c/b
      // k = -a/b => k' = b/a => y' = k'* x' + c' 
      // c' = point.y - k' * point.y =>
      // y = b/a * x + (point.y - b/a * point.x) - perpendicular to this line
      // intersection of y = -a/b * x - c/b and y = b/a * x + (point.y - b/a * point.x) gives 
      // necessary point
      // intersection: x = -(point.y - b/a * point.x + c/b) / (a/b + b/a)
      double a = (double)(m_end_point.get_y() - m_start_point.get_y());
      double b = -(double)(m_end_point.get_x() - m_start_point.get_x());
      double c = (double)m_start_point.get_y() * (double)(m_end_point.get_x() - m_start_point.get_x()) -
                 (double)m_start_point.get_x() * (double)(m_end_point.get_y() - m_start_point.get_y());

		  double dist_start = (b * (double)m_start_point.get_x()) / a - (double)m_start_point.get_y() + 
                          ((double)point.get_y() - b * (double)point.get_x() / a);
      double dist_end = (b * (double)m_end_point.get_x()) / a - (double)m_end_point.get_y() + 
                          ((double)point.get_y() - b * (double)point.get_x() / a);
		 
      if ( dist_start * dist_end > 0 )
      {
         if ( fabs(dist_start) < fabs(dist_end) )
            res = m_start_point;
         else
            res = m_end_point;
      }
      else
      {
         double x = -((double)point.get_y() - b * (double)point.get_x() / a  + c / b) / ( a / b + b / a);
         double y = - a * x / b - c / b;
         res.set_x( (OneDValue)DROUND(x) );
         res.set_y( (OneDValue)DROUND(y) );
      }
   }

   return res;
}

/***********************************************************************/
/***********************************************************************/
void Rectangle::set_center( const TwoDPoint &center )
{
   TwoDPoint old_center = get_center();

   OneDValue x_offset = center.get_x() - old_center.get_x();
   OneDValue y_offset = center.get_y() - old_center.get_y();

   m_right_bottom_point.set_x( m_right_bottom_point.get_x() + x_offset );
   m_right_bottom_point.set_y( m_right_bottom_point.get_y() + y_offset );

   m_left_top_point.set_x( m_left_top_point.get_x() + x_offset );
   m_left_top_point.set_y( m_left_top_point.get_y() + y_offset );
}

void Rectangle::set_range( OneDValue range )
{
   TwoDPoint center = get_center();
   if ( is_empty() && (m_width_ratio == 0.0 || m_height_ratio == 0.0) )
   {
      //Make square
      OneDValue offset = (OneDValue)( ((double)range / 1.41421) + 0.5);
               
      m_left_top_point.set_x( center.get_x() - offset );
      m_left_top_point.set_y( center.get_y() - offset );

      m_right_bottom_point.set_x( center.get_x() + offset );
      m_right_bottom_point.set_y( center.get_y() + offset );
   }
   else
   {
      LineSegment diagonal;
      if ( is_empty() )
      {
        OneDValue dx = (OneDValue)(m_width_ratio * 100.0) / 2;
        OneDValue dy = (OneDValue)(m_height_ratio * 100.0) / 2;
        diagonal.set_start_point( TwoDPoint( center.get_x() - dx, center.get_y() - dy ) );
        diagonal.set_end_point( TwoDPoint( center.get_x() + dx, center.get_y() + dy ) );
      }
      else
      {
        diagonal.set_start_point( m_left_top_point);
        diagonal.set_end_point( m_right_bottom_point );
      }
      
      diagonal.set_range( range );
      m_left_top_point = diagonal.get_start_point();
      m_right_bottom_point = diagonal.get_end_point();
   }
}

Rectangle Rectangle::intersection( const Rectangle &rect )
{
   Rectangle res;

   if ( inside( rect.m_left_top_point ) )
   {
      res.set_left_top_corner( rect.m_left_top_point );
   }
   else if ( rect.inside( m_left_top_point ) )
   {
      res.set_left_top_corner( m_left_top_point );
   }

   if ( inside( rect.m_right_bottom_point ) )
   {
      res.set_right_bottom_corner( rect.m_right_bottom_point );
   }
   else if ( rect.inside( m_right_bottom_point ) )
   {
      res.set_right_bottom_corner( m_right_bottom_point );
   }
     
   return res;
}

TwoDPoint Rectangle::get_nearest_to( const TwoDPoint &point ) const
{
   TwoDPoint res;

   if ( is_empty() )
      res = m_left_top_point;
   else if ( inside( point) )
   {
      res = point;
   }
   else
   {
      TwoDPoint center = get_center();
      
      int left_top = LineSegment( center, m_left_top_point).on_line( point );
      int right_top = LineSegment( center, TwoDPoint( m_right_bottom_point.get_x(), m_left_top_point.get_y() ) ).on_line( point );

      if ( left_top == 0 )
      {
         if ( right_top > 0 )
            res = m_left_top_point;
         else
            res = m_right_bottom_point;
      }
      else if ( right_top == 0 )
      {
         if ( left_top < 0 )
         {
            res.set_x( m_right_bottom_point.get_x() );
            res.set_y( m_left_top_point.get_y() );
         }
         else
         {
            res.set_x( m_left_top_point.get_x() );
            res.set_y( m_right_bottom_point.get_y() );
         }
      }
      else 
      {
         double dx = ((double)point.get_x() - (double)center.get_x());
         double dy = ((double)point.get_y() - (double)center.get_y());

         if ( left_top > 0 )
         {
            if ( right_top > 0 )
            {
               //left side - dx <> 0
               res.set_x( m_left_top_point.get_x() );

               if ( dy == 0 )
                  res.set_y( point.get_y() );
               else
               {
                  double y = ( (double)res.get_x() - (double)center.get_x()) * dy / dx + (double)center.get_y();
                  res.set_y( (OneDValue)DROUND( y ) );
               }
            }
            else
            {
               // bottom side - dy <> 0
               res.set_y( m_right_bottom_point.get_y() );

               if ( dx == 0 )
                  res.set_x( point.get_x() );
               else
               {
                  double x = ( (double)res.get_y() - (double)center.get_y()) * dx / dy + (double)center.get_x();
                  res.set_x( (OneDValue)DROUND( x ) );
               }
            }
         }
         else
         {
            if ( right_top < 0 )
            {
               //right side - dx <> 0
               res.set_x( m_right_bottom_point.get_x() );

               if ( dy == 0 )
                  res.set_y( point.get_y() );
               else
               {
                  double y = ( (double)res.get_x() - (double)center.get_x()) * dy / dx + (double)center.get_y();
                  res.set_y( (OneDValue)DROUND( y ) );
               }
            }
            else
            {
               // top side - dy <> 0
               res.set_y( m_left_top_point.get_y() );

               if ( dx == 0 )
                  res.set_x( point.get_x() );
               else
               {
                  double x = ( (double)res.get_y() - (double)center.get_y()) * dx / dy + (double)center.get_x();
                  res.set_x( (OneDValue)DROUND( x ) );
               }
            }
         }

      }
      
   }

   return res;
}

TwoDSize Circle::get_label_max_size( ) const
{
  OneDValue a = (OneDValue)DROUND( sqrt( 2.0 ) * (double)m_radius );
   
  return TwoDSize( a, a );
}

TwoDSize DoubleCircle::get_label_max_size( ) const
{
  OneDValue r = m_radius - (m_radius / 5);
  OneDValue a = (OneDValue)DROUND( sqrt( 2.0 ) * (double)r );
  
  return TwoDSize( a, a );
}

/***********************************************************************/
