#include <sstream>
#include <cassert>
#include <algorithm>

#include "nodeattr.h"
#include "types.h"

using namespace GraphGraphics;

const string NodeAttribute::ATTR_START = "{";
const string NodeAttribute::ATTR_END   = "}";
const string NodeAttribute::ATTR_PRED_OPEN_PARENTHESIS  = "(";   
const string NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS = ")";   
const string NodeAttribute::ATTR_PRED_AND  = "&";  
const string NodeAttribute::ATTR_PRED_OR   = "|";
const string NodeAttribute::ATTR_PRED_NOT  = "!"; 

//const int Lexeme::TEOF = -1;

const char GraphAttributes::CSEMI = ';';
const char GraphAttributes::CCOL = ':';
const char GraphAttributes::CDOT = '.';
const char GraphAttributes::CCOMMA = ',';
const char GraphAttributes::CMETASTART = '{';
const char GraphAttributes::CMETAEND = '}';
const char GraphAttributes::CEQ = '=';

const char GraphAttributes::CLS = '<';
const char GraphAttributes::CGT = '>';
const char GraphAttributes::CNOT = '!';
const char GraphAttributes::CAND = '&';
const char GraphAttributes::COR = '|';
const char GraphAttributes::COPENPAREN = '(';
const char GraphAttributes::CCLOSEPAREN = ')';
const char GraphAttributes::CQUOTATION = '\"';

const string GraphAttributes::SGTE = ">=";
const string GraphAttributes::SLSE = "<=";
const string GraphAttributes::SNEQ = "<>";

string GraphAttributes::STAB = "";
string GraphAttributes::SEOL = "";

const int GraphAttrNameStorage::EMPTY_ID = -1;
const string GraphAttrNameStorage::EMPTY_NAME = "";

const int     GraphAttrAliases::EMPTY_ALIAS_ID = -1;
const string  GraphAttrAliases::EMPTY_NAME = "";

bool AttributeIntegerValue::set_string( const string &str_value )
{
  bool res = false;
  
  if ( !str_value.empty() )
  {
    string str;
    bool negation = str_value[0] == '-';
    if ( negation )
      str = str_value.substr( 1 );
    else
      str = str_value;
    
    if ( (res = isdigit( str[0]) ) )
    {
      m_value = atol( str.c_str() );
    
      if ( negation )
        m_value = -m_value;
    }
  }
  
  return res;
}

string AttributeIntegerValue::get_string() const
{
  stringstream str_stream;
  
  str_stream << m_value;
  
  return str_stream.str();
}

bool AttributeIntegerValue::compare( long v, unsigned char op ) const 
{
  switch( op )
  {
  case AttributeValue::EQUAL:
    return ( v == m_value );
    
  case AttributeValue::LESS:
    return ( v < m_value );
    
  case AttributeValue::GREATER:
    return ( v > m_value );
    
  case AttributeValue::NOT_EQUAL:
    return ( v != m_value );
    
  case AttributeValue::NOT_LESS:
    return ( v >= m_value );
    
  case AttributeValue::NOT_GREATER:
    return ( v <= m_value );
  
  default:;
  }
    
  return false;
}

bool AttributeIntegerValue::compare( const string &v, unsigned char op ) const
{
  bool res = false;
  
  if ( !v.empty() )
  {
    string str;
    long   int_value = 0;
    bool negation = v[0] == '-';
    if ( negation )
      str = v.substr( 1 );
    else
      str = v;
    
    if ( !str.empty() )
      if ( (res = isdigit( str[0]) ) )
      {
        int_value = atol( str.c_str() );
      
        if ( negation )
          int_value = -int_value;
        
        res = compare( int_value, op );
      }
  }
  
  return res;
}

bool AttributeIntegerRange::compare( long v, unsigned char op ) const
{
  bool res = false;
  
  if ( op == AttributeValue::EQUAL ||
       op == AttributeValue::NOT_EQUAL )
  {
    for ( RangeList::const_iterator liter = m_ranges.begin(); liter != m_ranges.end() && !res; liter++ )
      res = liter->in_range( v );
    
    if ( !res )
      res = (m_values.find( v ) != m_values.end());
    
    if ( op == AttributeValue::NOT_EQUAL )
      res = !res;
  }
    
  return res;
}

bool AttributeIntegerRange::compare( const string &v, unsigned char op ) const
{
  bool res = false;
  
  if ( !v.empty() )
  {
    string str;
    long   int_value = 0;
    bool negation = v[0] == '-';
    if ( negation )
      str = v.substr( 1 );
    else
      str = v;
    
    if ( !str.empty() )
      if ( (res = isdigit( str[0]) ) )
      {
        int_value = atol( str.c_str() );
      
        if ( negation )
          int_value = -int_value;
        
        res = compare( int_value, op );
      }
  }
  
  return res;
}
  
void AttributeIntegerRange::add_int_range( long first_value, long last_value )
{
  if ( first_value == last_value )
    add_int_value( first_value );
  else 
  {
    if ( first_value > last_value )
    {
      long tmp = last_value;
      last_value = first_value;
      first_value = tmp;
    }
    
    for ( RangeList::iterator liter = m_ranges.begin(); liter != m_ranges.end();  )
    {
      if ( !(liter->first_value > last_value || liter->last_value < first_value) )
      {
        first_value = GGMIN( liter->first_value, first_value );
        last_value  = GGMAX( liter->last_value, last_value );
        liter = m_ranges.erase( liter );
      }
      else  
        liter++;        
    }
      
    SimpleIntRange simple_range( first_value, last_value );
    
    m_ranges.push_back( simple_range );
    
    for ( ValueSet::iterator siter = m_values.begin(); siter != m_values.end();  )
    {
      if ( simple_range.in_range( *siter ) )
      {
        m_values.erase( siter++ );
      }
      else
        siter++;
    }      
  }
}

void AttributeIntegerRange::add_int_value( long v )
{
  bool ins = false;
  for ( RangeList::iterator liter = m_ranges.begin(); liter != m_ranges.end() && !ins; liter++ )
    ins = liter->in_range( v );
  
  if ( !ins )
    m_values.insert( v );
}

string AttributeIntegerRange::get_string() const
{
  stringstream str_stream;
  RangeList::const_iterator liter2;
  ValueSet::const_iterator siter2;
  
  for ( RangeList::const_iterator liter = m_ranges.begin(); liter != m_ranges.end(); liter++ )
  { 
    str_stream << liter->first_value << ".." << liter->last_value;
    
    liter2 = liter;
    if ( (++liter2) != m_ranges.end() )
      str_stream << ",";
  }
  
  if ( !(m_ranges.empty() || m_values.empty()) )
    str_stream << ",";
  
  for ( ValueSet::const_iterator siter = m_values.begin(); siter != m_values.end(); siter++ )
  {
    str_stream << *siter;
    
    siter2 = siter;
    if ( (++siter2) != m_values.end() )
      str_stream << ",";
  }
  
  return str_stream.str();
}

bool AttributeIntegerRange::set_string( const string &str_range )
{
  bool res = !str_range.empty();
  
    
  if ( res )
  {
    Lexeme lex;
    int len = (int)str_range.length();
    int ind = 0;
    long val1;
    long val2;
    string   res_num = "";
    int      c;
    
    res_num.reserve( 20 );
    
    m_values.clear();
    m_ranges.clear();
        
    while ( ind < len && ind != Lexeme::TEOF && res )
    {
      for ( ; ind < len && isspace( (unsigned char)str_range[ind] ); ind++ );
            
      if ( ind < len )
      {
        if ( !isdigit( (unsigned char)str_range[ind] ) )
          res = false;
        else
        {
          c = (unsigned char)str_range[ind];
   
          res_num = "";
          res_num += c;

          if ( ++ind < len )
          {
            c = (unsigned char)str_range[ind];
            while ( isdigit( c ) )
            {
              res_num += c;
        
              if ( ++ind >= len )
                break;
        
              c = (unsigned char)str_range[ind];
            }
          }
          
          val1 = atol( res_num.c_str() );
          
          for ( ; ind < len && isspace( (unsigned char)str_range[ind] ); ind++ );
                    
          if ( ind >= len )
            add_int_value( val1 );
          else if ( str_range[ind] == ',' )
          {
            add_int_value( val1 );
            ind++;
          }
          else if ( ind + 1 < len )
          {
            if ( str_range[ind] == '.' && str_range[ind + 1] == '.' )
            {
              ind += 2;
              
              if ( ind >= len )
                res = false;
              else 
              {
                for ( ; ind < len && isspace( (unsigned char)str_range[ind] ); ind++ );
                                
                if ( ind >= len )
                  res = false;
                else if ( !isdigit( (unsigned char)str_range[ind] ) )
                  res = false;
                else
                {
                  c = (unsigned char)str_range[ind];
   
                  res_num = "";
                  res_num += c;
        
                  if ( ++ind < len )
                  {
                    c = (unsigned char)str_range[ind];
                    while ( isdigit( c ) )
                    {
                      res_num += c;
                
                      if ( ++ind >= len )
                        break;
                
                      c = (unsigned char)str_range[ind];
                    }
                  }
                  
                  val2 = atol( res_num.c_str() );
                  
                  add_int_range( val1, val2 );
          
                  for ( ; ind < len && isspace( (unsigned char)str_range[ind] ); ind++ ); 
                                    
                  if ( ind < len )
                    if ( str_range[ind] == ',' )
                      ind++;
                }
              }
            }
          }
        }
      }
    }
  }
  
  return res;
}

void AttributeStringValue::set_integer( long v )
{
  stringstream str_stream("");
  str_stream << v;
  
  m_value = str_stream.str();
}

long AttributeStringValue::get_integer() const
{
  long res = -1;
  
  if ( !m_value.empty() )
  {
    string str;
    bool negation = m_value[0] == '-';
    if ( negation )
      str = m_value.substr( 1 );
    else
      str = m_value;
    
    bool is_digit = !str.empty();
    
    for ( unsigned int i = 0; i < str.length() && is_digit; i++ )
      is_digit = isdigit( str[i] );
    
    if ( is_digit )
    {
      res = atol( str.c_str() );
    
      if ( negation )
        res = -res;
    }
  }
  
  return res;
}
  
bool AttributeStringValue::compare( long v, unsigned char op ) const
{
  stringstream str_stream("");
  str_stream << v;
  
  return compare( str_stream.str(), op );
}

bool AttributeStringValue::compare( const string &v, unsigned char op ) const
{
  bool res = false;
  
  switch ( op )
  {
  case AttributeValue::EQUAL:
    res = (m_value == v);
    break;
  
  case AttributeValue::NOT_EQUAL:
    res = !(m_value == v);
    break;
  
  case AttributeValue::GREATER:
    res = (v.find( m_value ) != string::npos) && !(m_value == v);
    break;
  
  case AttributeValue::NOT_LESS:
    res = (v.find( m_value ) != string::npos);
    break;
  
  case AttributeValue::LESS:
    res = (m_value.find( v ) != string::npos) && !(m_value == v);
    break;
  
  case AttributeValue::NOT_GREATER:
    res = (m_value.find( v ) != string::npos);
    break;
  
  default:;
  }
  
  return res;
}

/***************************************************************************/
/***************************************************************************/
NodeAttribute::NodeAttribute( const NodeAttribute &attr )
{
  name = attr.name;
  type = attr.type;
  value_op = attr.value_op;
  value = 0;
  if ( attr.value != 0 )
  {
    switch ( attr.value->get_actual_type() )
    {
    case AttributeValue::INTEGER:
      value = new AttributeIntegerValue( attr.value->get_integer() );
      break;
    
    case AttributeValue::INT_RANGE:
      {
        AttributeIntegerRange *range = new AttributeIntegerRange( );  
        range->set_string( attr.value->get_string() );
        value = range;
      }
      break;
      
    case AttributeValue::STRING:
      value = new AttributeStringValue( attr.value->get_string() );      
    }
  }
}

/****************************************************************/
/****************************************************************/


void GraphAttrAliases::clear()
{
  m_type_alias_map.clear();
  
  m_attr_alias_map.clear();
  m_alias_attr_map.clear();
  
  m_curr_alias_level = 0; 
  m_curr_attr_level = 0;
}



GraphAttrAliases::GraphAttrAliases( const GraphAttrAliases &aliases ) 
{
  m_type_alias_map = aliases.m_type_alias_map;
  m_attr_alias_map.copy( aliases.m_attr_alias_map );
  m_alias_attr_map.copy( aliases.m_alias_attr_map );
  
  m_curr_alias_level = 0; 
  m_curr_attr_level = 0;
}
  
GraphAttrAliases& GraphAttrAliases::operator = ( const GraphAttrAliases &aliases )
{
  clear();
    
  m_type_alias_map = aliases.m_type_alias_map;
  m_attr_alias_map.copy( aliases.m_attr_alias_map );
  m_alias_attr_map.copy( aliases.m_alias_attr_map );
    
  return *this;
}

bool GraphAttrAliases::get_attr_value( const AttrNameIDVector &attr_name, AttrTypeID type_id, const string &value_alias, long &value )
{
  bool res = true;
  
  MetaAttrAliases *meta = &m_attr_alias_map;
   
  value = 0;
    
  unsigned int    ind = 1;
    
  for ( ; ind < attr_name.size() - 1 && meta != 0; ind++ )
    meta = get_meta_alias( meta, attr_name[ind], false );
    
  res = (meta != 0);
    
  if ( res )
  {
    MetaAttrAliases *curr_meta = m_curr_attr_level;
    m_curr_attr_level = meta;    
    res = get_attr_value( attr_name[ind], type_id, value_alias, value );
    m_curr_attr_level = curr_meta;
  }
     
  
    
  return res;
}    
   
bool GraphAttrAliases::open_alias_level( const string &attr_alias )
{
  bool res = false;
  AliasID alias_id = get_alias_id( attr_alias);
    
  if ( alias_id != EMPTY_ALIAS_ID )
  {
    AliasMetaNames* meta = get_alias_meta( m_curr_alias_level, alias_id, false );
      
    if ( meta != 0 )
    {
      m_curr_alias_level = meta;
      res = true;
    }
  }
   
  return res;
}

bool GraphAttrAliases::get_meta_attr_name( AttrNameIDVector &meta_attr_name )
{
  meta_attr_name = m_curr_alias_level->name;        
     
  return !meta_attr_name.empty();
}
  
bool GraphAttrAliases::get_attr_name( const string &name_alias, AttrNameIDVector &attr_name )
{
  bool res = false;
  AliasID alias_id = get_alias_id( name_alias );
    
  if ( alias_id != EMPTY_ALIAS_ID )
  {
    AliasAttrNameMap::const_iterator p = m_curr_alias_level->attr_map.find( alias_id );
      
    if ( p != m_curr_alias_level->attr_map.end() )
    {
      attr_name = p->second;
      res = true;
    }
  }
    
  return res;
}

bool GraphAttrAliases::get_attr_name( const StringContainer &name_alias, AttrNameIDVector &attr_name )
{
  bool res = false;
  AliasIDContainer alias_ids;
    
  for ( StringContainer::const_iterator it = name_alias.begin(); it != name_alias.end(); it++ )
    alias_ids.push_back( get_alias_id(*it) );
  
  if ( !alias_ids.empty() )
  {
    AliasMetaNames *meta = &m_alias_attr_map;
    unsigned int    ind = 1;
      
    for ( ; ind < alias_ids.size() - 1 && meta != 0; ind++ )
      meta = get_alias_meta( meta, alias_ids[ind], false );
    
    if ( meta != 0 )
    {
      AliasAttrNameMap::const_iterator p = meta->attr_map.find( alias_ids[ind] );
          
      if ( p != meta->attr_map.end() )
      {
        attr_name = p->second;
        res = true;
      }
    }
  }
    
  return res;
}
  
bool GraphAttrAliases::get_attr_value( AttrNameID attr_name_id, AttrTypeID type_id, const string &value_alias, long &value )
{
  bool res = false;
  AttrAliasesMap::iterator p = m_curr_attr_level->attr_aliases.find( attr_name_id );
  AttrTypeAliasMap::const_iterator t_it;
  AliasAttrValueMap::const_iterator tv_it;
  AliasID value_alias_id = get_alias_id( value_alias );
        
  value = 0;
              
  if ( p != m_curr_attr_level->attr_aliases.end() )
  {
    AliasAttrValueMap::const_iterator q;
    AttrTypeID                        attr_type_id = p->second.type;
    
            
    if ( (q = p->second.value_map.find( value_alias_id )) != p->second.value_map.end() )
    {
      value = q->second;
      res = true;
    }
    else
    {
      t_it = m_type_alias_map.find( attr_type_id );
      if ( t_it != m_type_alias_map.end() )
      {
        tv_it = t_it->second.value_map.find( value_alias_id );          
        if ( tv_it != t_it->second.value_map.end() )
        {
          value = tv_it->second;
          res = true;
        }
      }
    }
  }
    
  if ( !res )
  {
    t_it = m_type_alias_map.find( type_id );
    if ( t_it != m_type_alias_map.end() )
    {
      tv_it = t_it->second.value_map.find( value_alias_id );
      if ( tv_it != t_it->second.value_map.end() )
      {
        value = tv_it->second;
        res = true;
      }
    }
  }
    
  return res;
}

bool GraphAttrAliases::get_meta_attr_name_alias( StringContainer &alias )
{
  alias.clear();
  
  AliasIDContainer::const_iterator q;
  for ( q = m_curr_attr_level->name_alias.begin(); q != m_curr_attr_level->name_alias.end(); q++)
    alias.push_back( get_alias_name(*q) );
 
  return !alias.empty();
}

bool GraphAttrAliases::get_attr_name_alias( AttrNameID attr_name_id, StringContainer &alias )
{
  AttrAliasesMap::const_iterator p = m_curr_attr_level->attr_aliases.find( attr_name_id );
  alias.clear();
    
  if ( p != m_curr_attr_level->attr_aliases.end() )
  {
    AliasIDContainer::const_iterator q;
    for ( q = p->second.name_alias.begin(); q != p->second.name_alias.end(); q++)
      alias.push_back( get_alias_name(*q) );
  }
    
  return (!alias.empty());
}
  
string GraphAttrAliases::get_attr_value_alias( AttrNameID attr_name_id, AttrTypeID type_id, long value )
{
  string res = "";
  AttrAliasesMap::iterator p = m_curr_attr_level->attr_aliases.find( attr_name_id );
  AttrTypeAliasMap::const_iterator t_it;
  AttrValueAliasMap::const_iterator tv_it;
              
  if ( p != m_curr_attr_level->attr_aliases.end() )
  {
    AttrValueAliasMap::const_iterator q;
    AttrTypeID                        attr_type_id = p->second.type;
            
    if ( (q = p->second.value_aliases.find( value )) != p->second.value_aliases.end() )
      res = get_alias_name( q->second );
    else
    {
      t_it = m_type_alias_map.find( attr_type_id );
      if ( t_it != m_type_alias_map.end() )
      {
        tv_it = t_it->second.value_aliases.find( value );          
        if ( tv_it != t_it->second.value_aliases.end() )
          res = get_alias_name( tv_it->second );
      }
    }
  }
    
  if ( res.empty() )
  {
    t_it = m_type_alias_map.find( type_id );
    if ( t_it != m_type_alias_map.end() )
    {
      tv_it = t_it->second.value_aliases.find( value );
      if ( tv_it != t_it->second.value_aliases.end() )
        res = get_alias_name( tv_it->second );
    }
  }
      
  return res;
}
  
void GraphAttrAliases::set_meta_attr_name_alias( const AttrNameIDVector &attr_name, const StringContainer &alias )
{
  AliasIDContainer alias_ids;
    
  for ( StringContainer::const_iterator it = alias.begin(); it != alias.end(); it++ )
    alias_ids.push_back( get_alias_id(*it) );
    
  set_name_alias( &m_attr_alias_map, attr_name, alias_ids, -1, true );
  set_alias_name( &m_alias_attr_map, alias_ids, attr_name, true );
}  
  
void GraphAttrAliases::set_attr_name_alias( const AttrNameIDVector &attr_name, AttrTypeID type, const StringContainer &alias )
{
  if ( attr_name.size() < 2 )
    return;
    
  AliasIDContainer alias_ids;
    
  for ( StringContainer::const_iterator it = alias.begin(); it != alias.end(); it++ )
    alias_ids.push_back( get_alias_id(*it) );
    
  set_name_alias( &m_attr_alias_map, attr_name, alias_ids, type );
  set_alias_name( &m_alias_attr_map, alias_ids, attr_name );
}
  
void GraphAttrAliases::set_attr_value_alias( const AttrNameIDVector &attr_name, long value, const string &value_alias )
{
  if ( attr_name.size() < 2 )
    return;
    
  MetaAttrAliases *meta = &m_attr_alias_map;
  unsigned int     ind = 1;
  AliasID          alias_id = get_alias_id( value_alias );
   
  for ( ; ind < attr_name.size() - 1; ind++ )
    meta = get_meta_alias( meta, attr_name[ind] );
    
  AttrAliasesMap::iterator p = meta->attr_aliases.find( attr_name[ind] );
          
  if ( p == meta->attr_aliases.end() )
    p = (meta->attr_aliases.insert(AttrAliasesMap::value_type( attr_name[ind], AttrAliases() ))).first;
    
  p->second.value_aliases[value] = alias_id;
  p->second.value_map[alias_id] = value;
}
    
void GraphAttrAliases::set_attr_type_value_alias( AttrTypeID type, long value, const string &value_alias )
{
  AttrTypeAliasMap::iterator p = m_type_alias_map.find( type );
  AliasID id = get_alias_id( value_alias );
    
  if ( p == m_type_alias_map.end() )
    p = (m_type_alias_map.insert( AttrTypeAliasMap::value_type( type, TypeAliases()) )).first;
    
  p->second.value_aliases[ value ] = id;
  p->second.value_map[ id ] = value;
}

bool GraphAttrAliases::get_attr_value_alases( const AttrNameIDVector &attr_name, AttrTypeID type_id, GraphAttrValueAliasMap &alias_map )
{
  MetaAttrAliases *meta = &m_attr_alias_map;
  unsigned int     ind = 1;
  AttrTypeAliasMap::iterator tit;
  AttrValueAliasMap::iterator tvit;
   
  for ( ; ind < attr_name.size() - 1 && meta != 0; ind++ )
    meta = get_meta_alias( meta, attr_name[ind], false ); 
    
  if ( meta != 0 )
  {
    AttrAliasesMap::iterator p = meta->attr_aliases.find( attr_name[ind] );
      
    if ( p != meta->attr_aliases.end() )
    {
      AttrTypeID attr_type_id = p->second.type;
                
      if ( !p->second.value_aliases.empty() )
      {
        for ( AttrValueAliasMap::iterator vit = p->second.value_aliases.begin(); 
                                         vit != p->second.value_aliases.end(); vit++ )
          alias_map[vit->first] = get_alias_name( vit->second );
      }
      else
      {
        tit = m_type_alias_map.find( attr_type_id );
        if ( tit != m_type_alias_map.end() )
        {
          for ( tvit = tit->second.value_aliases.begin(); tvit != tit->second.value_aliases.end(); tvit++)
            if ( alias_map.find( tvit->first ) == alias_map.end() )
              alias_map[tvit->first] = get_alias_name( tvit->second );
        }
      }
    }
  }    
  
  tit = m_type_alias_map.find( type_id );
  if ( tit != m_type_alias_map.end() )
  {
    for ( tvit = tit->second.value_aliases.begin(); tvit != tit->second.value_aliases.end(); tvit++)
      if ( alias_map.find( tvit->first ) == alias_map.end() )
        alias_map[tvit->first] = get_alias_name( tvit->second );
  }
      
  return !alias_map.empty();
}
/****************************************************************/
/****************************************************************/

/****************************************************************/
inline Lexeme GraphAttributes::get_identifier( const string &parsed_str, int ind )
{
   string   res_id = "";
   int      len = parsed_str.length();
   int      start = ind;
   int      c = (unsigned char)parsed_str[ind];

   if ( is_first_id_char( c ) )
   {
      res_id.reserve( 20 );
      res_id += c;

      if ( ++ind < len )
      {
         c = (unsigned char)parsed_str[ind];  
         while ( ( c == '_' || c == '[' || c == ']' || c == '.' || isalnum( c )) )
         {
            res_id += c;

            if ( ++ind >= len )
               break;

            c = (unsigned char)parsed_str[ind];
         }
      }
   }

   return Lexeme( start, ((ind < len)? ind: Lexeme::TEOF), Lexeme::TIDENTIFIER, Lexeme::TIDENTIFIER, res_id );
}

inline Lexeme GraphAttributes::get_number( const string &parsed_str, int ind )
{
   int      start = ind;
   int      len = parsed_str.length();
   int      c = (unsigned char)parsed_str[ind];
   string   res_num = "";

   res_num.reserve( 20 );
  
   res_num += c;

   if ( ++ind < len )
   {
      c = (unsigned char)parsed_str[ind];
      while ( isdigit( c ) )
      {
         res_num += c;

         if ( ++ind >= len )
            break;

         c = (unsigned char)parsed_str[ind];
      }
   }

   return Lexeme( start, ((ind < len)? ind: Lexeme::TEOF), Lexeme::TNUM, Lexeme::TNUM, res_num );
}

inline Lexeme GraphAttributes::get_string( const string &parsed_str, int ind )
{
  int      start = ind;
  int      len = parsed_str.length();
  int      c;
  Lexeme::LexTypes res_type = Lexeme::TSTRING;
  string   res = "";
  
  if ( ind < len )
  {
    c = (unsigned char)parsed_str[ind];
    while ( c != CQUOTATION )
    {
      res += c;
        
      if ( ++ind >= len )
        break;
        
      c = (unsigned char)parsed_str[ind];
    }
      
    if ( ind < len )
      ind ++; //skip quotation
    else
      res_type = Lexeme::TERROR;
  }
  else
    res_type = Lexeme::TERROR;

  return Lexeme( start, ((ind < len)? ind: Lexeme::TEOF), res_type, Lexeme::TSTRING, res );
}

Lexeme GraphAttributes::get_range( const string &parsed_str, int ind, const Lexeme &first_num_lex )
{
  int      start = (first_num_lex.type != Lexeme::TEOF)? ind: first_num_lex.pos;
  int      len = parsed_str.length();
  int      res_type = Lexeme::TRANGE;
  int      c = (unsigned char)parsed_str[ind];
  Lexeme   lex;
  string   res_range = first_num_lex.name;
  
  if ( res_range.empty() )
  {
    if ( isdigit( c ) || c == '-' || c == '+' )
    {
      lex = get_number( parsed_str, ind );
      ind = skip_separators( parsed_str, ++ind );
      
      if ( ind < len )
        c = (unsigned char)parsed_str[ind];
    }
  }
  
  while ( (c == CDOT || c == CCOMMA) && ind < len - 1 && res_type != Lexeme::TERROR )
  {
    if ( c == CDOT ) //range
    {
      if ( (c = (unsigned char)parsed_str[++ind]) == CDOT )
      {
        ind = skip_separators( parsed_str, ++ind );
        
        if ( ind < len )
        {
          c = (unsigned char)parsed_str[ind];
          if ( isdigit( c ) || c == '-' || c == '+' )
          {
            lex = get_number( parsed_str, ind );
            res_range += ".." + lex.name;
            ind = lex.next_pos;
          }
          else
            res_type = Lexeme::TERROR;
        }
        else
          res_type = Lexeme::TERROR;
      }
      else
        res_type = Lexeme::TERROR;
    }
    else if ( c == CCOMMA ) //enumeration
    {
      ind = skip_separators( parsed_str, ++ind );
      if ( ind < len )
      {  
        c = (unsigned char)parsed_str[ind];
        if ( isdigit( c ) || c == '-' || c == '+' )
        {
          lex = get_number( parsed_str, ind );
          res_range += "," + lex.name;
          ind = lex.next_pos;
        }
        else
          res_type = Lexeme::TERROR;
      }
      else
        res_type = Lexeme::TERROR;
    }
    else
      res_type = Lexeme::TERROR;
    
    if ( res_type != Lexeme::TERROR )
    {
      ind = skip_separators( parsed_str, ind );
      
      if ( ind < len )
        c = (unsigned char)parsed_str[ind];
    }
  }
  
  if ( c == CDOT || c == CCOMMA )
    res_type = Lexeme::TERROR;
  
  if ( res_type == Lexeme::TERROR )
  {
    lex.type = Lexeme::TERROR;
    lex.sub_type = Lexeme::TUNKNOWN;
  }
  else  
  {
    lex.type = Lexeme::TRANGE;
    lex.sub_type = Lexeme::TRANGE;
  }
  
  lex.pos = start;
  lex.next_pos = (ind >= len)? Lexeme::TEOF: ind;
  lex.name = res_range;
  
  return lex;
}

Lexeme GraphAttributes::get_lex( const string &parsed_str, int ind )
{
   Lexeme lex;

   if ( ind == Lexeme::TEOF )
   {
      lex.name = "";
      lex.next_pos = Lexeme::TEOF;
      lex.pos = Lexeme::TEOF;
      lex.type = Lexeme::TEOF;
   }
   else
   {
      int c;

      c = (unsigned char)parsed_str[ind];

      if ( is_first_id_char( c ) ) //identifier
      {
         lex = get_identifier( parsed_str, ind );
         ind = skip_separators( parsed_str, lex.next_pos );
         
         if ( ind != Lexeme::TEOF )
         {
            c   = parsed_str[ind];

            switch( c )
            {
            case CCOL: //it is type
               lex.type = Lexeme::TTYPE;
               ind = skip_separators( parsed_str, ++ind );
               break;

            case CEQ: //it is attr. name
            case CLS:
            case CGT:
               lex.type = Lexeme::TNAME;
               break;

            case CMETASTART: //it is metaattr. name
               lex.type = Lexeme::TMETANAME;
               ind = skip_separators( parsed_str, ++ind );
               break;

            default:; // do nothing
            }
         }

         lex.next_pos = ind; // position of next Lexeme
      }
      else if ( c == CQUOTATION ) //string
      {
        lex = get_string( parsed_str, ++ind );
        ind = skip_separators( parsed_str, lex.next_pos );
        
        if ( ind != Lexeme::TEOF )
        {
          c = parsed_str[ind];
          if ( c == CSEMI ) //attr. value
          {
            lex.type = Lexeme::TVALUE;
            ind = skip_separators( parsed_str, ++ind );
          }
        }
        lex.next_pos = ind;
      }
      else if ( isdigit( c ) || c == '-' || c == '+' ) //number or range
      {
         lex = get_number( parsed_str, ind );
         ind = skip_separators( parsed_str, lex.next_pos );

         if ( ind != Lexeme::TEOF )
         {
            c = parsed_str[ind];
           
            if ( c == CDOT || c == CCOMMA )
            {
              lex = get_range( parsed_str, ind, lex );
              ind = skip_separators( parsed_str, lex.next_pos );
              
              if ( lex.type != Lexeme::TERROR )
                c = (unsigned char)parsed_str[ind];
              else
                c = '\0';
            }
             
            switch( c )
            {
            case CCOL: //metaatr. identifier
               lex.type = Lexeme::TID;
               ind = skip_separators( parsed_str, ++ind );
               break;
  
            case CSEMI: //attr. value
               lex.type = Lexeme::TVALUE;
               ind = skip_separators( parsed_str, ++ind );
               break;
             
            default:; //do nothing
            }
         }

         lex.next_pos = ind; // position of next Lexeme
      }
      else 
      {
         lex.pos = ind;
         lex.name = "";
         lex.name += c;

         switch( c )
         {
         case CCOL:
            lex.sub_type = lex.type = Lexeme::TCOL;
            break;

         case CSEMI:
            lex.sub_type = lex.type = Lexeme::TSEMI;
            break;

         case CMETASTART:
            lex.sub_type = lex.type = Lexeme::TMETASTART;
            lex.name = "";
            break;

         case CMETAEND:
            lex.sub_type = lex.type = Lexeme::TMETAEND;
            break;
         
         case COPENPAREN:
            lex.sub_type = lex.type = Lexeme::TOPPAREN;
            break;
         
         case CCLOSEPAREN:
            lex.sub_type = lex.type = Lexeme::TCLPAREN;
            break;
         
         case CAND:
            lex.type = Lexeme::TPREDBINOP;
            lex.sub_type = Lexeme::TAND;
            break;
         
         case COR:
            lex.type = Lexeme::TPREDBINOP;
            lex.sub_type = Lexeme::TOR;
            break;
         
         case CNOT:
            lex.type = Lexeme::TPREDUNOP;
            lex.sub_type = Lexeme::TNOT;
            break;         

         case CEQ:
            lex.type = Lexeme::TVALUEOP;
            lex.sub_type = Lexeme::TEQ;
            break;
         
         case CGT:
            lex.type = Lexeme::TVALUEOP;
            lex.sub_type = Lexeme::TGT;
         
            if ( ind + 1 < (int)parsed_str.length() )
            {
               if ( (c = (unsigned char)parsed_str[ind + 1]) == CEQ )
               {
                 lex.sub_type = Lexeme::TGTE;
                 lex.name += c;
                 ind++;
               }
            }  
            break;
         
         case CLS:
         {
           int len = (int)parsed_str.length();
           lex.type = Lexeme::TVALUEOP;
           lex.sub_type = Lexeme::TLS;
         
           if ( ind + 1 < len )
           {
             c = (unsigned char)parsed_str[ind + 1];
              
             switch( c )
             {
             case CEQ:
               lex.sub_type = Lexeme::TLSE;
               lex.name += c;
               ind++;
               break;
               
             case CGT:
               lex.sub_type = Lexeme::TNEQ;
               lex.name += c;
               ind++;
               break;
              
             default:;
             }
           }
         }   
         break;
            
         default:
            lex.type = Lexeme::TUNKNOWN;
         }

         lex.next_pos = skip_separators( parsed_str, ind + 1 );
      }
   }

   return lex;
}

Lexeme GraphAttributes::parse_pred_attrlist( const string &parsed_str, const Lexeme &prev_lex, NodeAttributeList &attr_list )
{
  Lexeme         lex      = prev_lex;
  NodeAttribute  *attr    = 0;
  string         type;
  
  if ( lex.type != Lexeme::TOPPAREN)
    lex.type = Lexeme::TERROR;
  else
  {
    attr = new NodeAttribute();
    attr->name = lex.name;
    attr->value = new AttributeIntegerValue( -1 );

    attr->value_op = AttributeValue::NONE;
    attr->type = NodeAttribute::ATTR_PRED_OPEN_PARENTHESIS;
    attr_list.push_back( attr );
   
    lex = get_lex( parsed_str, lex.next_pos );
   
    lex = parse_attrlist( parsed_str, lex, attr_list, true );

    if ( lex.type != Lexeme::TCLPAREN )
      lex.type = Lexeme::TERROR; // there is no closed parenthesis
    else
    {
      attr = new NodeAttribute();
      attr->name = lex.name;
      attr->value = new AttributeIntegerValue( -1 );
      attr->value_op = AttributeValue::NONE;
      attr->type = NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS;
      attr_list.push_back( attr );

      lex = get_lex( parsed_str, lex.next_pos );
    }
  }

  return lex;
}

Lexeme GraphAttributes::parse_attrlist( const string &parsed_str, const Lexeme &prev_lex, NodeAttributeList &attr_list, bool predicate )
{
   Lexeme         lex      = prev_lex;
   NodeAttribute  *attr    = 0;
   string         type;
   string         name;
   AttributeValue::AttrOperation op;
   
   while ( lex.type != Lexeme::TEOF && 
           lex.type != Lexeme::TMETAEND && 
           lex.type != Lexeme::TERROR && 
           lex.type != Lexeme::TCLPAREN)
   {
      type     = "";
      name     = "";
      
      switch( lex.type )
      {
      case Lexeme::TTYPE: 

         type = lex.name;

         lex = get_lex( parsed_str, lex.next_pos );

         if ( lex.type != Lexeme::TNAME )
         {
            //error
            lex.type = Lexeme::TERROR;
            break;
         }

      case Lexeme::TNAME: //attr.

         name = lex.name;
         lex = get_lex( parsed_str, lex.next_pos );
      
         if (lex.type != Lexeme::TVALUEOP)
         {
           //opertation shoul preceed a value
           lex.type = Lexeme::TERROR;
         }
         else
         {
           switch( lex.sub_type )
           {
           case Lexeme::TEQ:
             op = AttributeValue::EQUAL;
             break;
           
           case Lexeme::TLS:
             op = AttributeValue::LESS;
             break;
           
           case Lexeme::TGT:
             op = AttributeValue::GREATER;
             break;
           
           case Lexeme::TLSE:
             op = AttributeValue::NOT_GREATER;
             break;
           
           case Lexeme::TGTE:
             op = AttributeValue::NOT_LESS;
             break;
           
           case Lexeme::TNEQ:
             op = AttributeValue::NOT_EQUAL;
             break;
           
           default:
             lex.type = Lexeme::TERROR;
           }
         }
         
         if ( lex.type != Lexeme::TERROR )
         {
           lex = get_lex( parsed_str, lex.next_pos );
           
           switch( lex.type )
           {
           case Lexeme::TVALUE:
              //
              if ( !(lex.sub_type == Lexeme::TNUM || lex.sub_type == Lexeme::TSTRING || 
                     (op == AttributeValue::EQUAL || op == AttributeValue::NOT_EQUAL) && 
                      lex.sub_type == Lexeme::TRANGE) )
                lex.type = Lexeme::TERROR;
              else
              {
                attr = new NodeAttribute();
                attr->name  = name;
                attr->type  = type;
                attr->value_op = op;
                
                switch (lex.sub_type)
                {
                case Lexeme::TNUM:
                  attr->value = new AttributeIntegerValue( atol(lex.name.c_str()) );
                  break;
                
                case Lexeme::TSTRING:
                  attr->value = new AttributeStringValue( lex.name );
                  break;
                
                case Lexeme::TRANGE:
                  attr->value = new AttributeIntegerRange( );
                  attr->value->set_string( lex.name );
                  break;
                default:;
                }
    
                attr_list.push_back( attr );
    
                lex = get_lex( parsed_str, lex.next_pos );
              }
              break;
  
           default:
              //error
              lex.type = Lexeme::TERROR;
           };
         }
         break;

      case Lexeme::TID:
      case Lexeme::TMETANAME:
      case Lexeme::TMETASTART:

         lex = parse_meta( parsed_str, lex, attr_list, predicate );
         break;

      case Lexeme::TMETAEND:
         // do nothing
         break;
      
      case Lexeme::TPREDBINOP:
         if ( !predicate )
           lex.type = Lexeme::TERROR;
         else
         {
           if ( attr_list.empty() || //first token is operator
               !(lex.sub_type == Lexeme::TAND ||
                 lex.sub_type == Lexeme::TOR) ) 
             lex.type = Lexeme::TERROR;
           else
           {
             attr = attr_list.back();
             
             if ( attr->type == NodeAttribute::ATTR_START || //first token is operator
                  attr->type == NodeAttribute::ATTR_PRED_OPEN_PARENTHESIS )
               lex.type = Lexeme::TERROR;
             else
             {
               attr = new NodeAttribute();
               attr->name = lex.name;
               attr->value = new AttributeIntegerValue( -1 );
               attr->value_op = AttributeValue::NONE;
               
               if ( lex.sub_type == Lexeme::TAND )
                 attr->type = NodeAttribute::ATTR_PRED_AND;
               else
                 attr->type = NodeAttribute::ATTR_PRED_OR;
               attr_list.push_back( attr );
      
               lex = get_lex( parsed_str, lex.next_pos );
             }
           }
         }           
         break;
      
      case Lexeme::TPREDUNOP:
         if ( !predicate )
           lex.type = Lexeme::TERROR;
         else
         {
           if ( attr_list.empty() )
           {
             attr = attr_list.back();
             
             if ( attr->type == NodeAttribute::ATTR_PRED_NOT ) //double NOT
             {
               attr_list.pop_back();
               delete attr;
               lex = get_lex( parsed_str, lex.next_pos );
               break;
             }
           }
           
           attr = new NodeAttribute();
           attr->name = lex.name;
           attr->value = new AttributeIntegerValue( -1 );
           attr->value_op = AttributeValue::NONE;
           attr->type = NodeAttribute::ATTR_PRED_NOT;
           attr_list.push_back( attr );
      
           lex = get_lex( parsed_str, lex.next_pos );
         }           
         break;
         
      case Lexeme::TOPPAREN:
         if ( !predicate )
           lex.type = Lexeme::TERROR;
         else
           lex = parse_pred_attrlist( parsed_str, lex, attr_list );
         break;
         
      case Lexeme::TCLPAREN:
        
         if ( predicate ) //do nothing
           break;
         
      default:
         //error
         lex.type = Lexeme::TERROR;
      }
   }

   return lex;
}

Lexeme GraphAttributes::parse_meta( const string &parsed_str, const Lexeme &prev_lex, NodeAttributeList &attr_list, bool predicate )
{
   Lexeme lex           = prev_lex;
   NodeAttribute *attr  = 0;
   AttributeValue *id   = 0;
   AttributeValue::AttrOperation  op = AttributeValue::EQUAL;
        
   if ( lex.type == Lexeme::TID )
   {
      if ( lex.sub_type == Lexeme::TRANGE )
      {
        id = new AttributeIntegerRange();
        id->set_string( lex.name );
      }
      else
        id = new AttributeIntegerValue( atol( lex.name.c_str() ) );
      lex = get_lex( parsed_str, lex.next_pos );
   }
   
   if ( id == 0 )
     id = new AttributeIntegerValue( -1 );

   switch( lex.type )
   {
   case Lexeme::TMETANAME:
   case Lexeme::TMETASTART:

      // 
      attr = new NodeAttribute();
      attr->name = lex.name;
      attr->value = id;
      attr->value_op = op;
      attr->type = NodeAttribute::ATTR_START;
      attr_list.push_back( attr );
   
      lex = get_lex( parsed_str, lex.next_pos );
   
      lex = parse_attrlist( parsed_str, lex, attr_list, predicate );

      if ( lex.type == Lexeme::TMETAEND )
      {
         // 
         attr = new NodeAttribute();
         attr->name = lex.name;
         attr->value = new AttributeIntegerValue( id->get_integer() );
         attr->value_op = op;
         attr->type = NodeAttribute::ATTR_END;
         attr_list.push_back( attr );


         lex = get_lex( parsed_str, lex.next_pos );
      }
      else
      {
         //error
         lex.type = Lexeme::TERROR;
      }
      break;

   default:
      //error
      delete id;
      lex.type = Lexeme::TERROR;
   }

   return lex;
}

bool GraphAttributes::parse_string( const string &str, NodeAttributeList &attr )
{
   bool   res = true;
   int    ind = skip_separators( str, 0 );
   Lexeme lex = get_lex( str, ind );
  
   attr.clear();
   attr.resize(0);
  
   if ( lex.type == Lexeme::TOPPAREN )
     lex = parse_pred_attrlist( str, lex, attr );
   else
     lex = parse_meta( str, lex, attr );

   if ( lex.type == Lexeme::TERROR )
   {
      attr.clear();
      attr.resize(0);
      res = false;
   }

   return res;
}

string GraphAttributes::make_string( const NodeAttributeList &attr_list )
{
   string prefix = "";
   ostringstream out_sstream("");
   bool first_meta = true;

   for ( unsigned int i = 0; i < attr_list.size(); i++ )
   {
      NodeAttribute *attr = attr_list[i];

      if ( attr->type == NodeAttribute::ATTR_END )
      {
         out_sstream << prefix << "}";

         if ( prefix.length() > 2 )
            prefix = prefix.substr( 0, prefix.length() - 2 ); 
         else
            prefix = "";

      }
      else if ( attr->type == NodeAttribute::ATTR_START )
      {
         out_sstream << prefix;
        
         if ( attr->value->get_integer() != -1 )
         {
           out_sstream << (( first_meta )? "": " ") << attr->value->get_string() << ":";
           first_meta = false;
         }
         out_sstream << (( first_meta )? "": " ") << attr->name << prefix << "{" + SEOL;
         prefix += STAB;
         first_meta = false;
      }
      else if ( attr->type == NodeAttribute::ATTR_PRED_OPEN_PARENTHESIS )
      {
        out_sstream << prefix << (( first_meta )? "": " ") << "(" << SEOL;
        first_meta = false;
        prefix += STAB;
      }
      else if ( attr->type == NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS )
      {
        out_sstream << prefix << ")" << SEOL;

        if ( prefix.length() > 2 )
          prefix = prefix.substr( 0, prefix.length() - 2 ); 
        else
          prefix = "";
      }
      else if ( attr->type == NodeAttribute::ATTR_PRED_OR )
        out_sstream << prefix << " |";
      else if ( attr->type == NodeAttribute::ATTR_PRED_AND )
        out_sstream << prefix << " &";
      else if ( attr->type == NodeAttribute::ATTR_PRED_NOT )
        out_sstream << prefix << " !";
      else //attribute
      {
         out_sstream << prefix;

         if ( attr->type != "" )
            out_sstream << " " << attr->type << ":";

         out_sstream << " " << attr->name;
         
         switch ( attr->value_op )
         {
         case AttributeValue::NOT_EQUAL:
           out_sstream << " <> ";
           break;
         
         case AttributeValue::NOT_LESS:
           out_sstream << " >= ";
           break;
         
         case AttributeValue::NOT_GREATER:
           out_sstream << " <= ";
           break;
         
         case AttributeValue::LESS:
           out_sstream << " < ";
           break;
         
         case AttributeValue::GREATER:
           out_sstream << " > ";
           break;
         
         default:
          out_sstream << "=";
         }
         
         if ( attr->value->get_actual_type() == AttributeValue::STRING )
           out_sstream << CQUOTATION << attr->value->get_string() << CQUOTATION << ";" + SEOL;
         else
          out_sstream << attr->value->get_string() << ";" + SEOL;
      }
   }

   return out_sstream.str();
}

int GraphAttributes::calc_single_pred( MetaAttribute *meta, const NodeAttributeList &attr_list, int &ind )
{
  int            res = -1;
  NodeAttribute *attr = 0;
  AttrNameID     name_id;
  int            attr_id;
  int            tmp_ind = ind;
  bool           first_meta = (meta == 0);
  
  if ( first_meta )
    meta = &m_attr_list;
  
  attr = attr_list[tmp_ind];
    
  if ( attr->type == NodeAttribute::ATTR_START )
  {
    if ( tmp_ind + 1 < (int)attr_list.size() )
    {
      name_id = m_name_storage->get_name_id( attr->name );
      
      if ( attr->value->get_actual_type() == AttributeValue::STRING )
        res = -1; 
      else if ( attr->value->get_integer() != GraphAttrNameStorage::EMPTY_ID )
      {
        if ( first_meta )
        {
          res = (((name_id == GraphAttrNameStorage::EMPTY_ID) || (m_attr_list.name == name_id))? 1: 0);
          
          if ( res > 0 )
            res = attr->value->compare( m_attr_ind );
          
          if ( res > 0 )
          {
            tmp_ind = ind + 1;
            if ( attr_list[tmp_ind]->type != NodeAttribute::ATTR_END )
              res = calc_pred_or_list( meta, attr_list, tmp_ind );
          }
        }
        else
          switch ( attr->value->get_actual_type() )
          {
          case AttributeValue::INTEGER:
            if ( attr->value->get_integer() >= meta->meta_attr_count )
              res = 0;
            else if ( name_id != GraphAttrNameStorage::EMPTY_ID && 
                      meta->meta_attr[attr->value->get_integer()]->name != name_id )
              res = 0;
            else
            {
              tmp_ind = ind + 1;
              
              if ( attr_list[tmp_ind]->type == NodeAttribute::ATTR_END )
                res = 1;
              else
                res = calc_pred_or_list( meta->meta_attr[ attr->value->get_integer() ], attr_list, tmp_ind );
            }
            break;
                  
          case AttributeValue::INT_RANGE:
            res = 0;
            attr_id = meta->find_metaatribute( name_id, 0 );
    
            while ( attr_id != GraphAttrNameStorage::EMPTY_ID && (res == 0) )
            {
              if ( attr->value->compare( attr_id ) )
              {
                tmp_ind = ind + 1;
                if ( attr_list[tmp_ind]->type == NodeAttribute::ATTR_END )
                  res =  1;
                else                
                  res = calc_pred_or_list( meta->meta_attr[attr_id], attr_list, tmp_ind );
    
                if ( res == 0 )
                  attr_id = meta->find_metaatribute( name_id, attr_id + 1 );
              }
            } 
            break;
                  
          default:
            res = -1;
          }
      }
      else
      {
        res = 0;
        
        if ( first_meta )
        {
          res = (name_id == GraphAttrNameStorage::EMPTY_ID) || (m_attr_list.name == name_id);
          
          if ( res )
          {
            tmp_ind = ind + 1;
            
            if ( attr_list[tmp_ind]->type == NodeAttribute::ATTR_END )
              res = 1;
            else  
              res = calc_pred_or_list( meta, attr_list, tmp_ind );
          }
        }
        else
        {
          attr_id = meta->find_metaatribute( name_id, 0 );
    
          while ( attr_id != GraphAttrNameStorage::EMPTY_ID && (res == 0) )
          {
            tmp_ind = ind + 1;
            
            if ( attr_list[tmp_ind]->type == NodeAttribute::ATTR_END )
              res = 1;
            else  
              res = calc_pred_or_list( meta->meta_attr[attr_id], attr_list, tmp_ind );
    
            if ( res == 0 )
              attr_id = meta->find_metaatribute( name_id, attr_id + 1 );
          }
        }
      }
    }
    
    if ( res >= 0 )
    {
      if ( ind >= (int)attr_list.size() )
        res = -1;
      else if ( attr_list[tmp_ind]->type != NodeAttribute::ATTR_END )
        res = -1;
      else
        tmp_ind++;
    }    
  }
  else if ( attr->type == NodeAttribute::ATTR_PRED_OPEN_PARENTHESIS )
  {
    tmp_ind++;
    res = calc_pred_or_list( ((first_meta)? 0: meta), attr_list, tmp_ind );
    
    if ( tmp_ind >= (int)attr_list.size() )
      res = -1;
    else if ( attr_list[tmp_ind]->type != NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS )
      res = -1;
    else
      tmp_ind++;
  }
  else if ( attr->type == NodeAttribute::ATTR_PRED_NOT )
  {
    tmp_ind++;
    
    if ( tmp_ind >= (int)attr_list.size() )
      res = -1;
    else
      res = calc_single_pred( ((first_meta)? 0: meta), attr_list, tmp_ind );
    
    if ( res >= 0 )
      res = (res == 0)? 1: 0;
  }
  else if ( attr->type == NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS )
  {
    res = 1;
  }
  else if ( !(attr->type == NodeAttribute::ATTR_END ||
            attr->type == NodeAttribute::ATTR_PRED_OR ||
            attr->type == NodeAttribute::ATTR_PRED_AND) ) //attribute
  {
    res = (calc_attr_pred( meta, attr ))? 1: 0;
    tmp_ind++;
  }
  
  ind = tmp_ind;
  
  return res;
}

int GraphAttributes::calc_pred_and_list( MetaAttribute *meta, const NodeAttributeList &attr_list, int &ind )
{
  int            res = -1;
  int            val;
  int            len = attr_list.size();
  
  res = calc_single_pred( meta, attr_list, ind );
  
  while ( res >= 0 && ind < len )
  {
    if ( attr_list[ind]->type == NodeAttribute::ATTR_PRED_OR || 
         attr_list[ind]->type == NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS || 
         attr_list[ind]->type == NodeAttribute::ATTR_END ) 
      break;
    
    if ( attr_list[ind]->type == NodeAttribute::ATTR_PRED_AND )
      ind++;
    
    val = calc_single_pred( meta, attr_list, ind );
      
    if ( val < 0 )
      res = -1;
    else
      res = ((res > 0 && val > 0)? 1: 0);
  }  
  
  return res;
}

int GraphAttributes::calc_pred_or_list( MetaAttribute *meta, const NodeAttributeList &attr_list, int &ind )
{
  int            res = -1;
  int            val;
  int            len = attr_list.size();
  
  res = calc_pred_and_list( meta, attr_list, ind );
  
  while ( res >= 0 && ind < len )
  {
    if ( attr_list[ind]->type != NodeAttribute::ATTR_PRED_OR )
    {
      if ( attr_list[ind]->type == NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS ||
           attr_list[ind]->type == NodeAttribute::ATTR_END )
       break;
      
      res = -1; //error
    }
    else
    {
      ind++;
      val = calc_pred_and_list( meta, attr_list, ind );
      
      if ( val < 0 )
        res = -1;
      else
        res = ((res > 0 || val > 0)? 1: 0);
    }
  }  
  
  return res;
}

bool GraphAttributes::compare_metaattribute( MetaAttribute *meta, const NodeAttributeList &attr_list, int first_ind, int last_ind )
{
   bool           res = true;
   NodeAttribute *attr = 0;
   int            j = 0;
   AttrNameID     name_id;
   int            attr_id;

   for ( int i = first_ind; i <= last_ind && res; )
   {
      attr = attr_list[i];

      if ( attr->type == NodeAttribute::ATTR_START )
      {
         name_id = m_name_storage->get_name_id( attr->name );
         j = find_meta_end( attr_list, i, last_ind );
        
         if ( j == GraphAttrNameStorage::EMPTY_ID )
           res = false;
         else if ( attr->value->get_actual_type() == AttributeValue::STRING )
           res = false;
         else if ( attr->value->get_integer() != GraphAttrNameStorage::EMPTY_ID )
         {
            switch ( attr->value->get_actual_type() )
            {
            case AttributeValue::INTEGER:
              if ( attr->value->get_integer() >= meta->meta_attr_count )
                res = false;
              else if ( name_id != GraphAttrNameStorage::EMPTY_ID && meta->meta_attr[attr->value->get_integer()]->name != name_id )
                res = false;
              else
                res = compare_metaattribute( meta->meta_attr[ attr->value->get_integer() ], attr_list, i + 1, j - 1 );
              break;
              
            case AttributeValue::INT_RANGE:
              res = false;
              attr_id = meta->find_metaatribute( name_id, 0 );

              while ( attr_id != GraphAttrNameStorage::EMPTY_ID && !res )
              {
                if ( attr->value->compare( attr_id ) )
                {
                  res = compare_metaattribute( meta->meta_attr[attr_id], attr_list, i + 1, j - 1 );

                  if ( !res )
                    attr_id = meta->find_metaatribute( name_id, attr_id + 1 );
                }
              } 
              break;
              
            default:
              res = false;
            }
         }
         else
         {
            res = false;

            attr_id = meta->find_metaatribute( name_id, 0 );

            while ( attr_id != GraphAttrNameStorage::EMPTY_ID && !res )
            {
               res = compare_metaattribute( meta->meta_attr[attr_id], attr_list, i + 1, j - 1 );

               if ( !res )
                  attr_id = meta->find_metaatribute( name_id, attr_id + 1 );
            }
         }

         i = (j != GraphAttrNameStorage::EMPTY_ID)? j + 1: last_ind;
      }
      else if ( attr->type == NodeAttribute::ATTR_END )
         i++;
      else if ( attr->type == NodeAttribute::ATTR_PRED_OPEN_PARENTHESIS ||
                attr->type == NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS || 
                attr->type == NodeAttribute::ATTR_PRED_OR ||
                attr->type == NodeAttribute::ATTR_PRED_AND ||
                attr->type == NodeAttribute::ATTR_PRED_NOT ) // attribute set should not contain predicate operators
        res = false;
      else // attribute
      {
         res = calc_attr_pred( meta, attr );
         i++;
      }
   }

   return res;
}

bool GraphAttributes::match_attributes( const NodeAttributeList &attr_list )
{
  bool res = false;
     
  if ( attr_list.empty() )
    return res;
      
  NodeAttribute *attr;
     
  if ( is_predicate( attr_list ) )
  {
    int ind = 1;
    res = (calc_pred_or_list( 0, attr_list, ind ) > 0);
            
    if ( res )
      if ( ind >= (int)attr_list.size() )
        res = false;
      else
        res = (attr_list[ ind ]->type == NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS);
  }
  else
  {
    attr = attr_list[0];
  
    if ( attr->type == NodeAttribute::ATTR_START && attr_list[ attr_list.size() - 1 ]->type == NodeAttribute::ATTR_END )
      if ( (attr->name == GraphAttrNameStorage::EMPTY_NAME || m_name_storage->get_name_id( attr->name ) == m_attr_list.name) )
        if (attr->value->get_integer() == GraphAttrNameStorage::EMPTY_ID || attr->value->compare( m_attr_ind )) 
         res = compare_metaattribute( &m_attr_list, attr_list, 1, attr_list.size() - 2 );
  }
  return res;
}

void GraphAttributes::make_attr_list( NodeAttributeList &attr_list, MetaAttribute *meta_attr, int meta_id )
{
   NodeAttribute* attr = new NodeAttribute();
   int            i = 0;

   attr->name = m_name_storage->get_id_name( meta_attr->name );
   attr->type = NodeAttribute::ATTR_START;
   attr->value = new AttributeIntegerValue( meta_id );
   attr->value_op = AttributeValue::EQUAL;

   attr_list.push_back( attr );

   for ( i = 0; i < meta_attr->attr_count; i++ )
   {
      attr = new NodeAttribute();

      attr->name = m_name_storage->get_id_name( meta_attr->attr[i].name );
      attr->type = m_name_storage->get_id_type( meta_attr->attr[i].type );
      attr->value_op = AttributeValue::EQUAL;
      if ( m_name_storage->is_string_attr_type( meta_attr->attr[i].type ) )
        attr->value = new AttributeStringValue( m_strings[meta_attr->attr[i].value] );
      else
        attr->value = new AttributeIntegerValue( meta_attr->attr[i].value );

      attr_list.push_back( attr );
   }

   for ( i = 0; i < meta_attr->meta_attr_count; i++ )
      make_attr_list( attr_list, meta_attr->meta_attr[i], i );
   
   attr = new NodeAttribute();
   attr->name = m_name_storage->get_id_name( meta_attr->name );
   attr->type = NodeAttribute::ATTR_END;
   attr->value_op = AttributeValue::NONE;
   attr->value = new AttributeIntegerValue( meta_id );

   attr_list.push_back( attr );
}

void GraphAttributes::make_attr_list_filtered( NodeAttributeList &attr_list, MetaAttribute *meta, const NodeAttributeList &attr_filter, int first_ind, int last_ind )
{
   NodeAttribute *attr = 0;
   int            j = 0, i = 0;
   AttrNameID     name_id;
   int            attr_id;
   NodeAttribute* res_attr = 0;
  
   if ( first_ind > last_ind )
   {
     for ( i = 0; i < meta->attr_count; i++ )
     {
        res_attr = new NodeAttribute();
  
        res_attr->name = m_name_storage->get_id_name( meta->attr[i].name );
        res_attr->type = m_name_storage->get_id_type( meta->attr[i].type );
        res_attr->value_op = AttributeValue::EQUAL;
       
        if ( m_name_storage->is_string_attr_type( meta->attr[i].type ) )
          res_attr->value = new AttributeStringValue( m_strings[meta->attr[i].value] );
        else
          res_attr->value = new AttributeIntegerValue( meta->attr[i].value );
  
        attr_list.push_back( res_attr );
     }
  
     for ( i = 0; i < meta->meta_attr_count; i++ )
        make_attr_list( attr_list, meta->meta_attr[i], i );
   }
   else
     for ( i = first_ind; i <= last_ind; )
     {
        attr = attr_filter[i];
        
        if ( attr->type == NodeAttribute::ATTR_START )
        {
           name_id = m_name_storage->get_name_id( attr->name );
           j = find_meta_end( attr_filter, i, last_ind );
          
           if ( attr->value->get_integer() != GraphAttrNameStorage::EMPTY_ID )
           {
              if ( attr->value->get_integer() < meta->meta_attr_count && 
                   (name_id == GraphAttrNameStorage::EMPTY_ID || meta->meta_attr[attr->value->get_integer()]->name == name_id) )
              {
                // add metaatribute
                res_attr = new NodeAttribute();
                res_attr->name = m_name_storage->get_id_name( meta->meta_attr[attr->value->get_integer()]->name );
                res_attr->type = NodeAttribute::ATTR_START;
                res_attr->value_op = AttributeValue::EQUAL;
                res_attr->value = new AttributeIntegerValue( attr->value->get_integer() );
            
                attr_list.push_back( res_attr );
                
                make_attr_list_filtered( attr_list, meta->meta_attr[attr->value->get_integer()], attr_filter, i + 1, j - 1 );
                
                res_attr = new NodeAttribute();
                res_attr->name = m_name_storage->get_id_name( meta->meta_attr[attr->value->get_integer()]->name );
                res_attr->type = NodeAttribute::ATTR_END;
                res_attr->value_op = AttributeValue::EQUAL;
                res_attr->value = new AttributeIntegerValue( attr->value->get_integer() );
            
                attr_list.push_back( res_attr );
              }
           }
           else
           {
              attr_id = meta->find_metaatribute( name_id, 0 );
  
              while ( attr_id != GraphAttrNameStorage::EMPTY_ID )
              {
                 // add metaatribute
                 res_attr = new NodeAttribute();
                 res_attr->name = m_name_storage->get_id_name( meta->meta_attr[attr_id]->name );
                 res_attr->type = NodeAttribute::ATTR_START;
                 res_attr->value_op = AttributeValue::EQUAL;
                 res_attr->value = new AttributeIntegerValue( attr_id );
                
                 attr_list.push_back( res_attr );
                
                 make_attr_list_filtered( attr_list, meta->meta_attr[attr_id], attr_filter, i + 1, j - 1 );
                
                 res_attr = new NodeAttribute();
                 res_attr->name = m_name_storage->get_id_name( meta->meta_attr[attr_id]->name );
                 res_attr->type = NodeAttribute::ATTR_END;
                 res_attr->value_op = AttributeValue::NONE;
                 res_attr->value = new AttributeIntegerValue( attr_id );
                
                 attr_list.push_back( res_attr );
  
                 attr_id = meta->find_metaatribute( name_id, attr_id + 1 );
              }
           }
  
           i = (j != GraphAttrNameStorage::EMPTY_ID)? j + 1: last_ind;
        }
        else if ( attr->type == NodeAttribute::ATTR_END )
           i++;
        else
        {
           name_id = m_name_storage->get_name_id( attr->name );
           attr_id = meta->find_attribute( name_id );
  
           if ( attr_id != GraphAttrNameStorage::EMPTY_ID )
           {
              //add attribute
              res_attr = new NodeAttribute();
  
              res_attr->name = m_name_storage->get_id_name( meta->attr[attr_id].name );
              res_attr->type = m_name_storage->get_id_type( meta->attr[attr_id].type );
              res_attr->value_op = AttributeValue::EQUAL;
              
              if ( m_name_storage->is_string_attr_type( meta->attr[attr_id].type ) )
                res_attr->value = new AttributeStringValue( m_strings[meta->attr[attr_id].value] );
              else
                res_attr->value = new AttributeIntegerValue( meta->attr[attr_id].value );
  
        
              attr_list.push_back( res_attr );
           }
           i++;
        }
     }
}

bool GraphAttributes::make_meta_attr( const NodeAttributeList &attr_list, MetaAttribute *meta_attr, int first_ind, int last_ind )
{
  bool                res = true;
  MetaAttrContainer   meta_attributes;
  AttrContainer       attributes;
  MetaAttribute      *mattr = 0;
  Attribute          *attr = 0;
  int                 ind = first_ind;
  int                 lind = 0;
	
  meta_attributes.resize( 0 );
  attributes.resize( 0 );
	
  while ( ind <= last_ind && attr_list[ind]->type != NodeAttribute::ATTR_END && res )
  {
    if ( attr_list[ind]->type == NodeAttribute::ATTR_START )
    {
      mattr = new MetaAttribute();
      mattr->name = m_name_storage->get_name_id( attr_list[ind]->name );
			
      if ( mattr->name == GraphAttrNameStorage::EMPTY_ID && !attr_list[ind]->name.empty() )
        mattr->name = m_name_storage->add_name( attr_list[ind]->name );
		
      lind = find_meta_end( attr_list, ind, last_ind );
				
      if ( lind == GraphAttrNameStorage::EMPTY_ID )
        lind = last_ind;
				
      res = make_meta_attr( attr_list, mattr, ind + 1, lind );
      meta_attributes.push_back( mattr );
      ind = lind + 1;
    }
    else
    {
      attr = new Attribute();
			
      attr->name = m_name_storage->get_name_id( attr_list[ind]->name );
      attr->type = m_name_storage->get_type_id( attr_list[ind]->type );
			
      if ( attr->name == GraphAttrNameStorage::EMPTY_ID )
      {
        if ( attr_list[ind]->name.empty() )
        {
          res = false;
          delete attr;
        }
        else
          attr->name = m_name_storage->add_name( attr_list[ind]->name );
          
			}
			
      if ( attr->type == GraphAttrNameStorage::EMPTY_ID )
      {
        if ( attr_list[ind]->type.empty() )
        {
          if ( attr_list[ind]->value != 0 )
          {
            if ( attr_list[ind]->value->get_actual_type() == AttributeValue::STRING )
              attr->type = m_name_storage->add_type( string("text") );
            else
              attr->type = m_name_storage->add_type( string("int") );
          }
          else
            attr->type = m_name_storage->add_type( string("int") );
        }
        else
          attr->type = m_name_storage->add_type( attr_list[ind]->type );
      }
      
      if ( attr_list[ind]->value != 0 )
      {
        if ( m_name_storage->is_string_attr_type( attr->type ) )
        {
          m_strings.push_back( attr_list[ind]->value->get_string() );
          attr->value = m_strings.size() - 1;
        }
        else
          attr->value = attr_list[ind]->value->get_integer();
      }
      else
        attr->value = 0;
      
      attributes.push_back( attr );
      
      ind ++;
    }
  }
	
  if ( res )
  {
    meta_attr->meta_attr_count = meta_attributes.size();
		
    if ( meta_attr->meta_attr_count > 0)
    {
      meta_attr->meta_attr = new (MetaAttribute*)[meta_attr->meta_attr_count];
		
      for ( ind = 0; ind < meta_attr->meta_attr_count; ind++ )
        meta_attr->meta_attr[ind] = meta_attributes[ind];
    }
		
    meta_attr->attr_count = attributes.size();
		
    if ( meta_attr->attr_count > 0)
    {
      sort( attributes.begin(), attributes.end(), AttrP_less() );
      
      meta_attr->attr = new Attribute[meta_attr->attr_count];
		
      for ( ind = 0; ind < meta_attr->attr_count; ind++ )
        meta_attr->attr[ind] = *(attributes[ind]);
    } 
  }
  else
  {
    for ( ind = 0; ind < (int)meta_attributes.size(); ind++ )
      delete meta_attributes[ind];	
  }
	
  for ( ind = 0; ind < (int)attributes.size(); ind++ )
    delete attributes[ind];	
	
  meta_attributes.clear();
  attributes.clear();
	
  return res;
}

bool GraphAttributes::set_attribute_string( const string &attribute_str )
{
  NodeAttributeList attr_list;
  
  if ( is_predicate( attribute_str ) )
    return false;

  bool res = parse_string( attribute_str, attr_list );
  
  if ( res )
  {
    res = set_attribute_list( attr_list );
  }
  return res;
}

int GraphAttributes::look_next_attr_name( const string &full_attr_name, int ind, string &name )
{
  bool colon_is_first = false;
  int len = full_attr_name.size();
  name = "";
     
  if ( len <= ind )
    return -1;
     
  if ( full_attr_name[ind] == ':' )
  {
    ind++;
    if ( full_attr_name[ind] != ':' )
      return -2;
       
    ind++;
    colon_is_first = true;
  }
     
  if ( ind >= len )
    return -1;
     
  Lexeme lex = get_identifier( full_attr_name, ind );
     
  name = lex.name;
     
  if ( lex.next_pos != Lexeme::TEOF )
  {
    ind = lex.next_pos;
    
    if ( name.empty() && 
         (!colon_is_first || full_attr_name[ind] == ':') )
      ind = -2;
    
  }
  else
    ind = -1;
     
  return ind;    
}
   
bool GraphAttributes::query_value( const string &full_attr_name, string &type, AttributeValue **value )
{
  if ( value == 0 || full_attr_name.empty() )
    return false;
  
  if ( m_name_storage == 0 )
    return false;
     
  string    attr_name;
  int ind = look_next_attr_name( full_attr_name, 0, attr_name );
  AttrNameID name_id = GraphAttrNameStorage::EMPTY_ID;
  int       attr_ind = 0;
  MetaAttribute *meta_attr = &m_attr_list;
  
  bool res = !full_attr_name.empty();
     
  type = "";
  *value = 0;
  
  if ( ind == -1 && res )
  {
    if ( attr_name.empty() || attr_name == m_name_storage->get_id_name( m_attr_list.name ) )
    {
      type = NodeAttribute::ATTR_START;
      *value = new AttributeIntegerValue( m_attr_ind );
    }
    else 
    {
      name_id = m_name_storage->get_name_id( attr_name );
      
      if ( name_id != GraphAttrNameStorage::EMPTY_ID )
      {
        attr_ind = meta_attr->find_attribute( name_id );
        if ( attr_ind >= 0 )
        {
          type   = m_name_storage->get_id_type( meta_attr->attr[attr_ind].type );
          
          if ( m_name_storage->is_string_attr_type( meta_attr->attr[attr_ind].type ) )
            *value = new AttributeStringValue( m_strings[meta_attr->attr[attr_ind].value] );
          else  
            *value = new AttributeIntegerValue( meta_attr->attr[attr_ind].value );
        }
      }
    }
    
    return (*value != 0);
  }
     
  while ( ind > 0 && res )
  {
    name_id = m_name_storage->get_name_id( attr_name );
      
    if ( name_id == GraphAttrNameStorage::EMPTY_ID )
      res = false;
    else
    {
      ind = look_next_attr_name( full_attr_name, ind, attr_name );
       
      if ( ind > 0 )
      {
        attr_ind = meta_attr->find_metaatribute( name_id, 0 );
        if ( attr_ind == GraphAttrNameStorage::EMPTY_ID )
          res = false;
        else
        {
           meta_attr = meta_attr->meta_attr[attr_ind];
        }
      }
    } 
  }
  
  if ( ind < -1 )
    res = false;
  
  if ( res )  
  {
    attr_ind = meta_attr->find_metaatribute( name_id, 0 );
    if ( attr_ind == GraphAttrNameStorage::EMPTY_ID )
      res = false;
    else
    {
       meta_attr = meta_attr->meta_attr[attr_ind];
    }
    
    name_id = m_name_storage->get_name_id( attr_name );
    if ( name_id == GraphAttrNameStorage::EMPTY_ID )
      res = false;      
  }
    
  if ( res )
  {
    
    attr_ind = meta_attr->find_attribute( name_id );
     
    if ( attr_ind != GraphAttrNameStorage::EMPTY_ID )
    {
      type   = m_name_storage->get_id_type( meta_attr->attr[attr_ind].type );
      
      if ( m_name_storage->is_string_attr_type( meta_attr->attr[attr_ind].type ) )
        *value = new AttributeStringValue( m_strings[meta_attr->attr[attr_ind].value] );
      else  
        *value = new AttributeIntegerValue( meta_attr->attr[attr_ind].value );
    }
    else
    {
      attr_ind = meta_attr->find_metaatribute( name_id, 0 );
       
      if ( attr_ind == GraphAttrNameStorage::EMPTY_ID )
        res = false;
      else
      {
        type = NodeAttribute::ATTR_START;
        *value = new AttributeIntegerValue( attr_ind );
      }
    }
  }
  
  return res;
}

bool GraphAttributes::is_predicate( const NodeAttributeList &attr_list )
{
  bool res = !attr_list.empty();
  
  if ( res )
    res = attr_list[0]->type == NodeAttribute::ATTR_PRED_OPEN_PARENTHESIS;
  
  return res;
}

/**************************************************************/
/****************************************************************/
GraphAttrNameStorage::GraphAttrNameStorage() 
{ 
  m_string_types.insert( add_type("string"));
  m_string_types.insert( add_type("text")); 
}

bool GraphAttrNameStorage::init_name_set( const string &search_str_format )
{
   bool res = true;
   NodeAttribute* attr = 0;
   
   if ( GraphAttributes::is_predicate( search_str_format ) )
     return false;
   
   clear_name_set( );
   
   add_type( "int" );
   m_string_types.insert( add_type("string") );
   m_string_types.insert( add_type("text") ); 

   res = GraphAttributes::parse_string( search_str_format, m_search_format );

   if ( m_search_format.size() == 0 )
      res = false;

   if ( res )
   {
      for ( unsigned int i = 0; i < m_search_format.size(); i++ )
      {
         attr = m_search_format[i];
         if ( attr->type != NodeAttribute::ATTR_END )
         {
            if ( attr->type != NodeAttribute::ATTR_START )
            {
               add_type( attr->type );
            }

            add_name( m_search_format[i]->name );
         }
      }
   }

   return res;
}

string  GraphAttrNameStorage::get_search_format_string()
{
  string res = GraphAttributes::make_string( m_search_format );

  return res;
}
