
#include <algorithm>
#include "graph.h"
#include "graphics.h"

using namespace GraphGraphics;

const NodeIndex Graph::INVALID_NODE_ID = (NodeIndex)-1;
const int GraphPath::INVALID_PATH_INDEX = -1;

/*******************************************************************************/
/*******************************************************************************/
bool Graph::find_path( NodeIndex start_node, const NodeIndexSet &end_nodes, EdgeList &edges )
{
  bool res = false;
  if ( m_nodes[ start_node ]->get_node_mark() == Node::NODE_IN_PATH )
    res = true;
  else if ( m_nodes[ start_node ]->get_node_mark() != Node::NODE_NOT_IN_PATH ) 
  {  
    EdgeList found_edges;
    NodeLinkList *node_links = m_nodes[ start_node ]->get_links();
     
    m_nodes[ start_node ]->set_node_mark( Node::NODE_NOT_IN_PATH );
    for ( NodeLinkList::iterator iter = node_links->begin(); iter != node_links->end(); iter++ )
    { 
      found_edges.clear();
      if ( end_nodes.find( *iter ) != end_nodes.end() )
      {   
        edges.push_back( Edge( start_node, *iter ) );
        m_nodes[ start_node ]->set_node_mark( Node::NODE_IN_PATH );
        res = true;
      }
      else if ( find_path( *iter, end_nodes, found_edges ) )
      {
        edges.push_back( Edge( start_node, *iter ) );
        
        for (EdgeList::iterator p = found_edges.begin(); p != found_edges.end(); p++)
          edges.push_back( *p );
        
        m_nodes[ start_node ]->set_node_mark( Node::NODE_IN_PATH );
        res = true;
      }
    }
  }
  
  return res;
}

/********************************************************************************/
/*******************************************************************************/

Graph* Graph::get_subgraph( const NodeAttributeList &attr_list )
{
	Graph 			*sub_graph = new Graph();
	Node			*node, *new_node;
	GraphAttributes	*attr;	
	NodeIndex		i;
	NodeLinkList	*links; 	
	NodeAttributeList new_attr_list;
  
  GraphAttrNameStorage *attr_storage = sub_graph->get_graph_attr_name_storage();
  attr_storage->init_name_set( m_attr_name_storage->get_search_format_string() );
  
	for (i = 0; i < count_nodes(); i++)
		get_node(i)->clear_sub_flag();
	
  for ( NodeIndex id = find_first_node( attr_list ); id != Graph::INVALID_NODE_ID; id = find_next_node( attr_list, id ) )
  {
		node = get_node(id);
    attr = new GraphAttributes( attr_storage, node->get_sub_id() );
		(node->get_attributes( ))->get_parsed_attributes( new_attr_list );
    attr->set_attribute_list( new_attr_list );
		
		sub_graph->add_node(attr, id);//node->get_sub_id());
		node->set_sub_flag();
		node->set_sub_tmp_id(sub_graph->count_nodes() - 1);
  }
	
	for (NodeIndex id = 0; id < sub_graph->count_nodes(); id++)
	{
		new_node = sub_graph->get_node(id);
		node = get_node(new_node->get_sub_id());
		links = node->get_links();
		
		for (i = 0; i < (NodeIndex)links->size(); i++)
		{
			Node *link_node = get_node((*links)[i]);		
			if (link_node->is_sub_flag())
				new_node->add_link(link_node->get_sub_tmp_id());
		}
	}
	
	for (NodeIndex id = 0; id < sub_graph->count_nodes(); id++)
	{
		new_node = sub_graph->get_node(id);
		node = get_node(new_node->get_sub_id());
		new_node->set_sub_id(node->get_sub_id());
	}

	return sub_graph;	
}

Graph* Graph::get_selected_subgraph( )
{
  Graph 			*sub_graph = new Graph();
  Node			*node, *new_node;
  GraphAttributes	*attr;	
  int				i;
  NodeLinkList	*links; 	
  GraphView		*view = m_graph_view->get_view();
  NodeIndexVector selected_nodes;	
  NodeAttributeList attr_list;
  
  GraphAttrNameStorage *attr_storage = sub_graph->get_graph_attr_name_storage();
  attr_storage->init_name_set( m_attr_name_storage->get_search_format_string() );
	
  view->get_selected_nodes(selected_nodes);
	
  for (i = 0; i < count_nodes(); i++)
    get_node(i)->clear_sub_flag();
	
  for ( NodeIndex id = 0; id < (NodeIndex)selected_nodes.size(); id++ )
  {
    node = get_node(selected_nodes[id]);
    attr = new GraphAttributes( attr_storage, node->get_sub_id() );
    (node->get_attributes( ))->get_parsed_attributes( attr_list );

    attr->set_attribute_list( attr_list );
    
		
    sub_graph->add_node(attr, selected_nodes[id]);//node->get_sub_id());
    node->set_sub_flag();
    node->set_sub_tmp_id(sub_graph->count_nodes() - 1);
  }
	
  for (NodeIndex id = 0; id < sub_graph->count_nodes(); id++)
  {
    new_node = sub_graph->get_node(id);
    node = get_node(new_node->get_sub_id());
    links = node->get_links();
		
    for (i = 0; i < (NodeIndex)links->size(); i++)
    {
      Node *link_node = get_node((*links)[i]);
      if (link_node->is_sub_flag())
        new_node->add_link(link_node->get_sub_tmp_id());
    }
		
  }
	
  for (NodeIndex id = 0; id < sub_graph->count_nodes(); id++)
  {
    new_node = sub_graph->get_node(id);
    node = get_node(new_node->get_sub_id());
    new_node->set_sub_id(node->get_sub_id());
  }

  return sub_graph;	
}

bool Graph::mark_scenario( NodeIndex node_id )
{
  bool res = true;
  Node *node = m_nodes[ node_id ];
  NodeLinkList	*links;
  bool attr_exist;
  string type;
  AttributeValue *value = 0;
  GraphAttributes	*attr;
  
  if ( node->get_node_mark() == Node::NODE_NOT_IN_PATH )
    return false;
  
  if ( node->get_node_mark() != Node::NODE_IN_PATH )
  {
    res = false;
    node->set_node_mark( Node::NODE_NOT_IN_PATH );
    links = node->get_links();
    
    for ( NodeLinkList::iterator it = links->begin(); it != links->end(); it++ )
      if ( mark_scenario( *it ) )
        res = true;
    
    if ( !res )
    {
      attr = node->get_attributes();
      attr_exist = attr->query_value( "::accept", type, &value );
      if ( attr_exist && value != 0 )
        res = (value->get_integer() == 1);
      
      if ( value != 0 )
        delete value;
      
      if ( !res )
      {
        attr_exist = attr->query_value( "::final", type, &value );
        if ( attr_exist && value != 0 )
          res = (value->get_integer() == 1);
        
        if ( value != 0 )
        delete value;
      }
    }
    
    if ( res )
      node->set_node_mark( Node::NODE_IN_PATH );
  }
  
  return res;
}

Graph* Graph::get_reduced_graph( const vector<NodeAttributeList> &attr_lists, bool copy_attr )
{
  Graph* res_graph = 0;
  Graph* work_graph = new Graph();
  GraphAttributes	*attr;
  Node* node = 0;
  Node* new_node = 0;
  NodeLinkList	*links; 
  bool attr_exist = true;
//  long  src_host_id = 0, dest_host_id = 0;
//  long  attack_id = 0;
  string type;
  NodeIndex node_id;
  bool is_not_one_of_attacks = true;
  int i;
  NodeAttributeList attr_list;
  
  //copy nodes from the graph to work_graph (without attributes)
  for ( PNodesVector::iterator p = m_nodes.begin(); p != m_nodes.end() && attr_exist; p++ )
  {
    node = *p;
    attr = node->get_attributes();
    new_node = new Node( node->get_id(), attr );
    new_node->set_sub_id( node->get_sub_id() ); 

    is_not_one_of_attacks = true;
    for ( i = 0; i < (int)attr_lists.size() && is_not_one_of_attacks; i++ )
      is_not_one_of_attacks = !(attr->match_attributes( attr_lists[i] ));
    
    if ( !is_not_one_of_attacks )
      new_node->set_node_mark( Node::NODE_NOT_IN_PATH );
    
    work_graph->add_node( new_node );
  }
  
  if ( attr_exist)
  {
    //copy necessary edges
    for ( i = 0; i < work_graph->count_nodes(); i++ )
    {
      node  = get_node( i );
      new_node = work_graph->get_node( i );
      links = node->get_links();  
      
      for (NodeLinkList::iterator lit = links->begin(); lit != links->end(); lit++)
      {
        node_id = *lit;
        
        if ( work_graph->get_node( node_id )->get_node_mark() != Node::NODE_NOT_IN_PATH )
          new_node->add_link( node_id );
      }
    }
    work_graph->mark_all_nodes();
    
    //mark existed scenarios
    attr_exist = work_graph->mark_scenario( 0 );
    
    //get result graph
    if ( attr_exist && work_graph->count_nodes() > 0 )
    {
      vector<NodeIndex> real_indexes;
      res_graph = new Graph();
      GraphAttrNameStorage *attr_storage = res_graph->get_graph_attr_name_storage();
      attr_storage->init_name_set( m_attr_name_storage->get_search_format_string() );
      
      real_indexes.resize(work_graph->count_nodes(), -1 );
      attr = 0;
            
      //add nodes
      for ( i = 0; i < work_graph->count_nodes(); i++ )
      {
        node = work_graph->get_node( i );
        
        if ( node->get_node_mark() == Node::NODE_IN_PATH )
        {
          attr = new GraphAttributes( attr_storage, node->get_sub_id() );
          
          if ( copy_attr )
          {
            (node->get_attributes( ))->get_parsed_attributes( attr_list );
            attr->set_attribute_list( attr_list );
          }
          
          new_node = new Node( node->get_id(), attr );
          new_node->set_sub_id( node->get_sub_id() );
          real_indexes[ i ] = res_graph->add_node( new_node );
        }
      }
      
      //add edges
      for ( i = 0; i < work_graph->count_nodes(); i++ )
      {
        node = work_graph->get_node( i );
        
        if ( real_indexes[ i ] >= 0 )
        {
          new_node = res_graph->get_node( real_indexes[ i ] );
          links = node->get_links();
          
          for (NodeLinkList::iterator lit = links->begin(); lit != links->end(); lit++)
          {
            node_id = *lit;
            
            if ( real_indexes[ node_id ] > 0 )
              new_node->add_link( real_indexes[ node_id ] );
          }   
        }
      }
    }
  }
  
  for ( i = 0; i < work_graph->count_nodes(); i++ )
  {
    node = work_graph->get_node( i );
    node->set_attributes( 0, false );
  }
  
  delete work_graph;
  
  return res_graph;
}

bool Graph::get_graph_path( NodeIndex start_node, const NodeIndexSet &end_nodes, 
                     const NodeAttributeList &include_nodes, 
                     const NodeAttributeList &exclude_nodes, GraphPath &path )
{
  if ( m_nodes.empty() )
    return false;
  
  GraphPath all_pathes;
  bool res;
  EdgeList edges;
     
  if ( start_node >= count_nodes() || 
       start_node < 0 )
    res = false;
  else
  {
    res = find_path( start_node, end_nodes, edges );
    mark_all_nodes();
       
    if ( res )
    {
      all_pathes.init( start_node, end_nodes, edges );
      
      res = all_pathes.find_path_sub_group( this, include_nodes, exclude_nodes );
      
      if ( res )
      {
        EdgeList path_edges;
        
        edges.clear();
        
        for ( int path_ind = 0; path_ind < all_pathes.get_pathes_count(); path_ind++ )
        {
          all_pathes.get_path_edges( path_ind, path_edges );
          edges.insert(  edges.end(), path_edges.begin(), path_edges.end() );
        }
        
        path.init( start_node, end_nodes, edges );
      }
    }
  }
     
  return res;
}

/*******************************************************************************/
/*******************************************************************************/
GraphPath::GraphPath()
{
  m_init_node = Graph::INVALID_NODE_ID;
}
  
bool GraphPath::init( NodeIndex start_node, const NodeIndexSet &end_nodes, const EdgeList &edges )
{
  bool res = false;
    
  if ( start_node != Graph::INVALID_NODE_ID && 
       !end_nodes.empty() )
  {
    NodeLinksMap::iterator mi; 
    
    m_init_node = start_node;
    m_end_nodes  = end_nodes;
    
    //store edges in map for fast searching
    for ( EdgeList::const_iterator p = edges.begin(); p != edges.end(); p++ )
    {
      if ( (mi = m_edges_map.find( p->start_node )) == m_edges_map.end() )
      {
        NodeLinksMap::value_type vpair( p->start_node, NodeIndexSet() );
        mi = (m_edges_map.insert( vpair )).first;
      }
      
      mi->second.insert( p->end_node );
    }
    
    separate_pathes( 0 );
    res = true;
  }
    
  return res;
}

void GraphPath::uninit()
{
  m_init_node = Graph::INVALID_NODE_ID;
  m_end_nodes.clear();
        
  m_edges_map.clear();
  m_pathes.clear();
  m_path_groups.clear();
}

bool GraphPath::empty() const
{
  return m_init_node == Graph::INVALID_NODE_ID ||
         m_end_nodes.empty() ||
         m_pathes.empty();
}

void GraphPath::find_pathes( NodeIndex start_node, NodeLinkList &curr_path, NodeIndexSet &curr_node_set, const NodeLinkListMap &edges_map )
{
  NodeLinkListMap::const_iterator mi;
    
  if ( (mi = edges_map.find( start_node )) != edges_map.end() )
  {
    for ( NodeLinkList::const_iterator p = mi->second.begin(); p != mi->second.end(); p++ )
    {
      if ( m_end_nodes.find( *p ) != m_end_nodes.end() )
      {
        curr_path.push_back( *p );
        m_pathes.push_back( curr_path );
        curr_path.pop_back();
      }
      else //ignore cycles 
        if ( curr_node_set.find( *p ) == curr_node_set.end() ) 
      {
        curr_node_set.insert( *p );
        curr_path.push_back( *p );
        find_pathes( *p, curr_path, curr_node_set, edges_map );
        curr_path.pop_back();
        curr_node_set.erase( *p );
      }
    }
  }
}

class NodeCoordLess
{
private:
  GraphView *m_view;

public:
  NodeCoordLess( GraphView *view ) { m_view = view; }
  
  int operator () ( const NodeIndex &x, const NodeIndex &y ) const
  {
    return (m_view->get_node_coord( x ).get_x() < m_view->get_node_coord( y ).get_x());
  }
};

void GraphPath::sort_pathes_by_coords( GraphView *view )
{
  separate_pathes( view );
}

void GraphPath::separate_pathes( GraphView *view )
{
  NodeLinkListMap edge_vector_map;
  NodeLinkList path;
  NodeIndexSet path_node_set;
  
  m_pathes.clear();
  
  //store edges in map for fast searching
  for ( NodeLinksMap::const_iterator p = m_edges_map.begin(); p != m_edges_map.end(); p++ )
  {
    for ( NodeIndexSet::const_iterator q = p->second.begin(); q != p->second.end(); q++ )
      path.push_back( *q );
    
    if ( view != 0 )
      stable_sort( path.begin(), path.end(), NodeCoordLess( view ) );
    
    edge_vector_map[ p->first ] = path;
    
    path.clear();
  }
  
  
  path.push_back( m_init_node );
  path_node_set.insert( m_init_node );
  
  find_pathes( m_init_node, path, path_node_set, edge_vector_map );
}

bool GraphPath::find_path_sub_group( Graph *graph, const NodeAttributeList &include, const NodeAttributeList &exclude )
{
  bool res = false;
    
  if ( !m_pathes.empty() && 
       !(exclude.empty() && include.empty()) && 
       graph != 0 )
  {
    PathGroup group;
    bool      find_exclude;
    bool      find_include;
    Node     *node;
    unsigned int ind;
    
    if ( m_path_groups.empty() )
    {
      for ( ind = 0; ind < m_pathes.size(); ind++)
      {
        find_exclude = false;
        find_include = false;
        NodeLinkList::const_iterator p = m_pathes[ind].begin(); // skip first node
        p++;
        
        for ( ; p != m_pathes[ind].end() && !find_exclude; p++ )
        {
          node = graph->get_node( *p );
          
          if ( !exclude.empty() )
            find_exclude = node->compare_attributes( exclude );
            
          if ( !find_exclude && !find_include && !include.empty() )
            find_include = node->compare_attributes( include );
        }          
        
        if ( !find_exclude && (find_include || include.empty()) )
          group.push_back( ind );
      }        
    }
    else
    {
      unsigned int curr_group = m_path_groups.size() - 1;
      unsigned int curr_group_len = m_path_groups[curr_group].size();
      
      for ( ind = 0; ind < curr_group_len; ind++)
      {
        find_exclude = false;
        find_include = false;
        NodeLinkList::const_iterator p = m_pathes[m_path_groups[curr_group][ind]].begin(); // skip first node
        NodeLinkList::const_iterator end_it = m_pathes[m_path_groups[curr_group][ind]].end();
        p++;
        
        for ( ; p != end_it && !find_exclude; p++ )
        {
          node = graph->get_node( *p );
          
          if ( !exclude.empty() )
            find_exclude = node->compare_attributes( exclude );
            
          if ( !find_exclude && !find_include && !include.empty() )
            find_include = node->compare_attributes( include );
        }          
        
        if ( !find_exclude && (find_include || include.empty()) )
          group.push_back( m_path_groups[curr_group][ind] );
      }      
    }
    
    if ( !group.empty() )
    {
      m_path_groups.push_back( group );
      res = true;
    }
  }
  
  return res;
}

NodeIndexSet GraphPath::get_end_nodes() 
{
  NodeIndexSet res;
  
  if ( m_path_groups.empty() )
    res = m_end_nodes;
  else
  {
    NodeIndex end_node;
    unsigned int curr_group = m_path_groups.size() - 1;
    unsigned int curr_group_len = m_path_groups[curr_group].size();
      
    for ( unsigned int ind = 0; ind < curr_group_len; ind++ )
    {
      end_node = m_pathes[m_path_groups[curr_group][ind]][ m_pathes[m_path_groups[curr_group][ind]].size() - 1];
      res.insert( end_node );
    }
  }    
  
  return res; 
}

int GraphPath::get_path_edges( int path_ind, EdgeList &path ) const
{
  path.clear();
  if ( m_path_groups.empty() )
  {
    if ( path_ind >= 0 && path_ind < (int)m_pathes.size() )
    { 
      NodeLinkList::const_iterator p = m_pathes[path_ind].begin();
      NodeLinkList::const_iterator q = p++;
      for ( ; p != m_pathes[path_ind].end(); p++, q++ )
        path.push_back( Edge( *q, *p ) );
    }
  }
  else
  {
    unsigned int curr_group = m_path_groups.size() - 1;
    if ( path_ind >= 0 && path_ind < (int)m_path_groups[curr_group].size() )
    {
      NodeLinkList::const_iterator p = m_pathes[m_path_groups[curr_group][path_ind]].begin();
      NodeLinkList::const_iterator endp = m_pathes[m_path_groups[curr_group][path_ind]].end();
      NodeLinkList::const_iterator q = p++;
      for ( ; p != endp; p++, q++ )
        path.push_back( Edge( *q, *p ) );
    }
  }      
    
  return path.size();
}

int GraphPath::get_path_nodes( int path_ind, NodeLinkList &path ) const
{
  path.clear();
    
  if ( m_path_groups.empty() )
  {
    if ( path_ind >= 0 && path_ind < (int)m_pathes.size() )
      for ( NodeLinkList::const_iterator p = m_pathes[path_ind].begin(); p != m_pathes[path_ind].end(); p++ )
        path.push_back( *p );
  }
  else 
  {
    unsigned int curr_group = m_path_groups.size() - 1;
    if ( path_ind >= 0 && path_ind < (int)m_path_groups[curr_group].size() )
    {
      NodeLinkList::const_iterator p = m_pathes[m_path_groups[curr_group][path_ind]].begin();
      NodeLinkList::const_iterator endp = m_pathes[m_path_groups[curr_group][path_ind]].end();
      for ( ; p != endp; p++ )
        path.push_back( *p );
    }
  }
    
  return path.size();
}

void Graph::calc_pathes( NodeIndex start_node_id, const NodeAttributeList &end_attr_list, StringLongTriple &res )
{
  Node *node;
  Node *child;
  unsigned int  counter = 0;
  unsigned int  length = 0;
  unsigned int  max_length = 0;
  unsigned int  min_length = 0;
  NodeIndex     node_id;
  NodeIndex     child_id;
  NodeIndex     link_id;
  
  NodeIndexList dfs_stack;
  NodeIndexList dfs_child_stack;
    
  max_length = 0;
  min_length = m_nodes.size();
    
  dfs_stack.push_back( start_node_id );
  dfs_child_stack.push_back( -1 );
  node = m_nodes[start_node_id];
    
  node->set_node_mark( Node::NODE_IN_PATH );
    
  while ( !dfs_stack.empty() )
  {
    node_id = dfs_stack.back();
    node = m_nodes[node_id];
    link_id = dfs_child_stack.back();
      
    child = 0;
    //get next child
    if ( ++link_id >= node->get_links()->size() )
    {
      //all childs are passed
      if ( end_attr_list.empty() && link_id == 0)
      {
        length = dfs_stack.size();
        max_length = GGMAX( length, max_length );
        min_length = GGMIN( length, min_length );   
        counter++;
      }
      
      dfs_stack.pop_back();
      dfs_child_stack.pop_back();
        
      node->set_node_mark( Node::NODE_UNMARKED );
    }
    else
    {
      child_id = *(node->get_links()->begin() + link_id);
      dfs_child_stack.back() = link_id;
      child = m_nodes[child_id];
    }
      
    if ( child != 0 )
    {
      if ( child->get_node_mark() != 0 ) //cycle
      {
        if ( end_attr_list.empty() )
        {
          if ( node->get_links()->size() == 1 )
          {
            length = dfs_stack.size();
            max_length = GGMAX( length, max_length );
            min_length = GGMIN( length, min_length );               
            counter++;
          }
        }
      }
      else if ( !end_attr_list.empty() && child->compare_attributes( end_attr_list ) )
      {
        length = dfs_stack.size() + 1;
        max_length = GGMAX( length, max_length );
        min_length = GGMIN( length, min_length );   
        counter++;
      }
      else
      {
        dfs_stack.push_back( child_id );
        dfs_child_stack.push_back( -1 );
        child->set_node_mark( Node::NODE_IN_PATH );
      }
    }
  }
    
  min_length = GGMIN( min_length, max_length );
  res.value2 = max_length;
  res.value1 = counter;
  res.value3 = min_length;
}

void Graph::get_graph_statistics( GraphStatistics &statistics )
{
  if ( m_nodes.empty() )
  {
    statistics.edge_count = 0;
    statistics.node_count = 0;
    statistics.max_nodes_out_degree = 0;
    statistics.max_nodes_in_degree = 0;
    statistics.graph_depth = 0;
    statistics.min_path_length = 0;
    statistics.path_count = 0;
    return;
  }
  
  Node *node;
  Node *child;
  unsigned int  counter = 0;
  unsigned int  length = 0;
  unsigned int  max_length = 0;
  unsigned int  min_length = 0;
  NodeIndex     node_id;
  NodeIndex     child_id;
  NodeIndex     link_id;
  vector<long>  in_degree;
  
  statistics.edge_count = 0;
  in_degree.resize( m_nodes.size(), 0 );
  
  for ( node_id = 0; node_id < (NodeIndex)m_nodes.size(); node_id++ )
  {
    node = m_nodes[node_id];
    node->set_node_mark( 0 );
    length = node->get_links()->size();
    statistics.edge_count += length;
    
    max_length = GGMAX(length, max_length);
    
    NodeLinkList::iterator it = node->get_links()->begin();
    for ( ; it != node->get_links()->end(); it++)
      in_degree[*it]++;
  }
  
  if ( statistics.calc_nodes_edges )
  {
    statistics.node_count = m_nodes.size();
      
    statistics.max_nodes_out_degree = max_length;
    
    max_length = 0;
    for ( unsigned int i = 0; i < in_degree.size(); i++ )
       max_length = GGMAX( max_length, in_degree[i] );
    
    statistics.max_nodes_in_degree = max_length;
  }
  
  if ( !statistics.calc_path_depth )
  {
    statistics.graph_depth = 0;
    statistics.min_path_length = 0;
    statistics.path_count = 0;
  }
  else
  {
    StringLongTriple res;
    NodeAttributeList  attr_list;
    
    calc_pathes( get_init_node(), attr_list, res );
    
    statistics.graph_depth = res.value2;
    statistics.min_path_length = res.value3;
    statistics.path_count = res.value1;
  }
  
  // calc nodes
  if ( !statistics.nodes_to_count.empty() )
  {
    NodeValuesMap::iterator n_it = statistics.nodes_to_count.begin();
  
    for ( ; n_it != statistics.nodes_to_count.end(); n_it++ )
    {
      StringLongTriple  &res = n_it->second;
      string             attr_string  = n_it->first;
      NodeAttributeList  attr_list; 
      
      bool parse_res = GraphAttributes::parse_string( attr_string, attr_list );
      
      res.value1 = 0;
      res.value2 = 0;
      
      if ( parse_res )
      {
        counter = 0;
        max_length = 0;
        min_length = 0;
        
        node_id = find_first_node( attr_list );
        for ( ; node_id != INVALID_NODE_ID; node_id = find_next_node( attr_list, node_id ) )
        {
          node = m_nodes[node_id];
          
          length = node->get_links()->size();
          max_length = GGMAX(length, max_length);
          min_length = GGMAX(in_degree[ node_id ], min_length);
          counter++;
        }
        res.value1 = counter;
        res.value2 = max_length;
        res.value3 = min_length;
      }
    }
  }
  
  PathValuesMap::iterator p_it = statistics.pathes_to_count.begin();
  
  for ( ; p_it != statistics.pathes_to_count.end(); p_it++ )
  {
    StringLongTriple  &res = p_it->second;
    string             start_attr_string  = p_it->first.first;
    string             end_attr_string  = p_it->first.second;
    NodeAttributeList  start_attr_list;
    NodeAttributeList  end_attr_list;
    bool parse_res;
    
    res.value1 = 0;
    res.value2 = 0;
    
    if ( start_attr_string.empty() )
    {
      bool parse_res = true;
      if ( !end_attr_string.empty() )
        parse_res = GraphAttributes::parse_string( end_attr_string, end_attr_list );
      
      if ( parse_res )
        calc_pathes( get_init_node(), end_attr_list, res );
    }
    else
    {
      bool parse_res = GraphAttributes::parse_string( start_attr_string, start_attr_list );
    
      if ( parse_res )
      {
        end_attr_list.clear();
        
        if ( !end_attr_string.empty() )
          parse_res = GraphAttributes::parse_string( end_attr_string, end_attr_list );
        
        if ( parse_res )
        {
          node_id = find_first_node( start_attr_list );
          for ( ; node_id != INVALID_NODE_ID; node_id = find_next_node( start_attr_list, node_id ) )
            calc_pathes( node_id, end_attr_list, res );
        }
      }
    }
  }
}

