
#include "graphics.h"
#include <iostream>
#include "Layout.h"
#include "limits.h"

using namespace GraphGraphics;

void GraphLayout::acyclic()
{
	unsigned int		i;

	for (i = 0; i < m_nodes.size(); i++) 
	{
		m_nodes[i]->mark = false;
		m_nodes[i]->onstack = false;
	}
	
	for (i = 0; i < m_nodes.size(); i++) 
		dfs(m_nodes[i]);
}
 
void GraphLayout::dfs(VNode *v_node)
{
	EdgeListIter	e_iter, e_nxt_iter;
	VNode	*w;
	
	if (v_node->mark) 
		return;
		
	v_node->mark = true;
	v_node->onstack = true;
	
	e_iter = e_nxt_iter = v_node->out_edges.begin();
	while (e_iter != v_node->out_edges.end()) 
	{
		w = m_nodes[(*e_iter)->head_indx];
		VEdge	*e = *e_iter;
		e_nxt_iter = e_iter;
		e_nxt_iter++;		
		
		if (w->onstack) 
		{ 
			NodeIndex	node_tmp;
			
			m_nodes[e->tail_indx]->out_edges.remove(e);
			m_nodes[e->head_indx]->in_edges.remove(e);
			
			m_nodes[e->tail_indx]->in_edges.push_back(e);
			m_nodes[e->head_indx]->out_edges.push_back(e);

			node_tmp = e->head_indx;
			e->head_indx = e->tail_indx;
			e->tail_indx = node_tmp;
			
			e->inverse_edge = true;
		}
		else
			if (w->mark == false) 
				dfs(w); 
			
		e_iter = e_nxt_iter;
	}
	v_node->onstack = false;
}

void GraphLayout::cut_rank_info(char *info)
{
	NodeIndex	i;
	FILE		*f = fopen("/home/RL/cut_rank_info", "a");
	
	fprintf(f, "%s\n", info);
	
	fprintf(f, "Ranks: ");		
	for (i = 0; i < (long)N_nodes; i++) 
	{
		fprintf(f, "(%i):%i ", (int)m_nodes[i]->index, m_nodes[i]->rank);
	}

	fprintf(f, "\n\nCutvalues: ");			
	for (i = 0; i < (long)m_edges.size(); i++) 
	{
		fprintf(f, "(%i, %i):%i ", (int)m_nodes[m_edges[i]->tail_indx]->index, (int)m_nodes[m_edges[i]->head_indx]->index, m_edges[i]->cutvalue);
	}
	
	fclose(f);
}

void GraphLayout::debug_info(char *info)
{
	VNode		*n;
	NodeIndex	i, k;
	FILE		*f = fopen("/home/RL/rank_dbg", "a");
	
	fprintf(f, "%s\n", info);
	
	for (i = 0; i < N_nodes; i++) 
	{
		n = m_nodes[i];
		fprintf(f, "Node %i (rank %i):\n\t", (int)n->index, n->rank);

		fprintf(f, "Tree in: ");		
		for (k = 0; k < (long)n->tree_in.size(); k++)
			fprintf(f, "(%i, %i, cutvalue %i) ", (int)m_nodes[n->tree_in[k]->tail_indx]->index, (int)m_nodes[n->tree_in[k]->head_indx]->index, n->tree_in[k]->cutvalue);
		
		fprintf(f, "\n\tTree out: ");		
		for (k = 0; k < (long)n->tree_out.size(); k++)
			fprintf(f, "(%i, %i, cutvalue %i) ", (int)m_nodes[n->tree_out[k]->tail_indx]->index, (int)m_nodes[n->tree_out[k]->head_indx]->index, n->tree_out[k]->cutvalue);
		
		fprintf(f, "\n");
	}
	 
	fprintf(f, "\n_________________________________________\n");	
	
	fclose(f);
}

bool GraphLayout::init_graph()
{
	int		i, l, feasible;
	VNode	*n;
	VEdge	*e;
	EdgeListIter	e_iter;	
	
	N_nodes = m_nodes_comp.size(), 
	N_edges = 0;
	S_i = 0;
	
	for (i = 0; i < N_nodes; i++) {
		m_nodes_comp[i]->mark = false;
		N_edges += m_nodes_comp[i]->out_edges.size();
	}

	Tree_node.clear();
	Tree_edge.clear();
	Tree_node.reserve(N_nodes);
	Tree_edge.reserve(N_nodes);

	feasible = true;
	for (l = 0; l < N_nodes; l++)
	{
		n = m_nodes_comp[l];
		n->priority = 0;
		
		e_iter = n->in_edges.begin();		
		i = 0;
		while (e_iter != n->in_edges.end()) 
		{
			e = *e_iter;
			
			n->priority++;
			e->cutvalue = 0;
			e->tree_index = -1;
			
/*			if (feasible && (m_nodes[e->head_indx]->rank - m_nodes[e->tail_indx]->rank < e->minlen))
				feasible = false;*/
			
			e_iter++;
			i++;
		}
		
		n->tree_in.clear();
		n->tree_out.clear();
		n->tree_in.reserve(i+1);
		n->tree_out.reserve(n->out_edges.size() + 2);
	}
	
/*	return feasible;*/
	return false;	
}

void GraphLayout::init_rank(void)
{
	int						i,ctr = 0;
	int						node_size = m_nodes_comp.size();
	VNodeQueue		Q;
	VNode					*v;
	VEdge					*e;
	EdgeListIter	e_iter;	
	
	for (i = 0; i < node_size; i++) 
	{
		v = m_nodes_comp[i];
		if (v->priority == 0) 
			Q.push_back(v);
	}

	while (!(Q.empty())) 
	{
		v = Q.front();
		Q.pop_front();
		
		v->rank = 0;
		ctr++;
		
		e_iter = v->in_edges.begin();
		while (e_iter != v->in_edges.end())
		{
			e = *e_iter;
			v->rank = GGMAX(v->rank, m_nodes[e->tail_indx]->rank + e->minlen);
			e_iter++;
		}
		
		e_iter = v->out_edges.begin();		
		while (e_iter != v->out_edges.end()) 
		{
			e = *e_iter;			
			if (--(m_nodes[e->head_indx]->priority) <= 0) 
				Q.push_back(m_nodes[e->head_indx]);
			e_iter++;			
		}
	}
	
/*	if (ctr != node_size) {
		fprintf(stderr,"trouble in init_rank\n");
		for (v = G->u.nlist; v; v = v->u.next)
			if (v->u.priority)
				fprintf(stderr,"\t%s %d\n",v->name,v->u.priority);
	}*/
}

int GraphLayout::tight_tree(void)
{
	NodeIndex	i;
	VNode		*n;

	for (i = 0; i < N_nodes; i++) 
	{
		n = m_nodes_comp[i];
		n->mark = false;
		n->tree_in.clear();	n->tree_out.clear();
	}
	for (i = 0; i < (long)Tree_edge.size(); i++) 
		Tree_edge[i]->tree_index = -1;

	Tree_node.clear(); Tree_edge.clear();
	
	for (i = 0; i < N_nodes && Tree_edge.empty(); i++)
	{
		n = m_nodes_comp[i];
		treesearch(n);
	}
		
/*	debug_info("tight");*/
	return Tree_node.size();
}

bool GraphLayout::treesearch(VNode* v)
{
	VEdge			*e;
	EdgeListIter	e_iter;	
	
	e_iter = v->out_edges.begin();
	
	while (e_iter != v->out_edges.end()) 
	{
		e = *e_iter;
		
		if ((m_nodes[e->head_indx]->mark == false) && (SLACK(e) == 0)) {
			add_tree_edge(e);
			if (((long)Tree_edge.size() == N_nodes-1) || treesearch(m_nodes[e->head_indx])) 
				return true;
		}
		
		e_iter++;
	}
	
	e_iter = v->in_edges.begin();
	
	while (e_iter != v->in_edges.end()) 	
	{
		e = *e_iter;
		
		if ((m_nodes[e->tail_indx]->mark == false) && (SLACK(e) == 0)) {
			add_tree_edge(e);
			if (((long)Tree_edge.size() == N_nodes-1) || treesearch(m_nodes[e->tail_indx])) 
				return true;
		}
		
		e_iter++;		
	}
	
	return false;
}

void GraphLayout::add_tree_edge(VEdge* e)
{
	VNode	*n;
	
	if (TREE_EDGE(e)) 
		abort();
		
	e->tree_index = Tree_edge.size();
	Tree_edge.push_back(e);
	if (!m_nodes[e->tail_indx]->mark) 
		Tree_node.push_back(m_nodes[e->tail_indx]);
	
	if (!m_nodes[e->head_indx]->mark) 
		Tree_node.push_back(m_nodes[e->head_indx]);
	
	n = m_nodes[e->tail_indx];
	n->mark = true;
	n->tree_out.push_back(e);

	if (n->out_edges.size() < n->tree_out.size()) abort();
		
	n = m_nodes[e->head_indx];
	n->mark = true;
	n->tree_in.push_back(e);

	if (n->in_edges.size() < n->tree_in.size()) abort();
}

NodeIndex GraphLayout::incident(VEdge* e)
{
	if (m_nodes[e->tail_indx]->mark) 
	{
		if (m_nodes[e->head_indx]->mark == false)
			return e->tail_indx;
	}
	else 
	{
		if (m_nodes[e->head_indx]->mark)
			return e->head_indx;
	}
	return -1;
}

void GraphLayout::feasible_tree(void)
{
	int						i,delta;
	VNode					*n;
	VEdge					*e,*f;
	EdgeListIter	f_iter;		

	if (N_nodes <= 1) return;
		
	while (tight_tree() < N_nodes) {
		e = NULL;
		
		for (i = 0; i < N_nodes; i++) 
		{
			n = m_nodes_comp[i];
			f_iter = n->out_edges.begin();		
		
			while (f_iter != n->out_edges.end()) 				
			{
				f = *f_iter;

				if ((TREE_EDGE(f) == false) && incident(f) >= 0 && ((e == NULL)
					|| (SLACK(f) < SLACK(e)))) 
					e = f;
				
				f_iter++;
			}
		}
		
		if (e) {
			delta = SLACK(e);
			if (delta)
			{
				if (incident(e) == e->head_indx) 
					delta = -delta;
				
				for (i = 0; i < (long)Tree_node.size(); i++)
					Tree_node[i]->rank += delta;
			}
		}
		else {
/*#ifdef DEBUG
			fprintf(stderr,"not in tight tree:\n");
			for (n = G->u.nlist; n; n = n->u.next) {
				for (i = 0; i < Tree_node.size; i++)
					if (Tree_node.list[i] == n) break;
				if (i >= Tree_node.size) fprintf(stderr,"\t%s\n",n->name);
			}
#endif*/
			abort();
		}
	}
	init_cutvalues();
	
/*	debug_info("feasable");	*/
}

void GraphLayout::x_cutval(VEdge* f)
{
	VNode	*v;
	int		sum,dir;
	EdgeListIter	e_iter;		
	
	/* set v to the node on the side of the edge already searched */
	if (m_nodes[f->tail_indx]->par == f) 
	{ 
		v = m_nodes[f->tail_indx]; 
		dir = 1; 
	}
	else 
	{ 
		v = m_nodes[f->head_indx]; 
		dir = -1; 
	}

	sum = 0;
	e_iter = v->out_edges.begin();
	while (e_iter != v->out_edges.end()) 
	{
		sum += x_val(*e_iter,v,dir);
		e_iter++;
	}
	
	e_iter = v->in_edges.begin();
	while (e_iter != v->in_edges.end()) 
	{
		sum += x_val(*e_iter,v,dir);
		e_iter++;
	}

	f->cutvalue = sum;
}

int GraphLayout::x_val(VEdge* e, VNode* v, int dir)
{
	VNode	*other;
	int		d,rv,f;

	if (m_nodes[e->tail_indx] == v) 
		other = m_nodes[e->head_indx]; 
	else 
		other = m_nodes[e->tail_indx];
			
	if (!(SEQ(v->low, other->lim, v->lim))) 
	{
		f = 1; 
		rv = e->weight;
	}
	else 
	{
		f = 0;
		if (TREE_EDGE(e)) 
			rv = e->cutvalue;
		else 
			rv = 0;
		
		rv -= e->weight;
	}
	
	if (dir > 0) 
	{
		if (m_nodes[e->head_indx] == v) 
			d = 1; 
		else 
			d = -1;
	}
	else 
	{
		if (m_nodes[e->tail_indx] == v) 
			d = 1; 
		else 
			d = -1; 
	}
	
	if (f) 
		d = -d;
	if (d < 0) 
		rv = -rv;
	
	return rv;
}

void GraphLayout::dfs_cutval(VNode* v, VEdge* par)
{
	NodeIndex	i;
	VEdge		*e;

	for (i = 0; (i < (long)v->tree_out.size()); i++)
	{
		e = v->tree_out[i];
		if (e != par) 
			dfs_cutval(m_nodes[e->head_indx], e);
	}
			
	for (i = 0; (i < (long)v->tree_in.size()); i++)
	{
		e = v->tree_in[i];
		if (e != par) 
			dfs_cutval(m_nodes[e->tail_indx], e);
	}
	
	if (par) x_cutval(par);
}

int GraphLayout::dfs_range(VNode* v, VEdge* par, int low)
{
	VEdge		*e;
	NodeIndex	i,lim;

	lim = low;
	v->par = par;
	v->low = low;
	
	for (i = 0; (i < (long)v->tree_out.size()); i++)
	{
		e = v->tree_out[i];
		if (e != par) 
			lim = dfs_range(m_nodes[e->head_indx], e, lim);
	}
			
	for (i = 0; (i < (long)v->tree_in.size()); i++)
	{
		e = v->tree_in[i];
		if (e != par) 
			lim = dfs_range(m_nodes[e->tail_indx], e, lim);
	}
			
	v->lim = lim;
	return lim + 1;
}

void GraphLayout::init_cutvalues(void)
{
	dfs_range(m_nodes_comp[0], NULL, 1);
	dfs_cutval(m_nodes_comp[0], NULL);
}

VEdge *GraphLayout::leave_edge (void)
{
	VEdge			*f,*rv = NULL;
	int				j,cnt = 0;

	j = S_i;
	while (S_i < (long)Tree_edge.size()) 
	{
		if ((f = Tree_edge[S_i])->cutvalue < 0) {
			if (rv) 
			{
				if (rv->cutvalue > f->cutvalue) 
					rv = f;
			}
			else 
				rv = Tree_edge[S_i];
			
			if (++cnt >= Search_size) 
				return rv;
		}
		S_i++;
	}
	
	if (j > 0) 
	{
		S_i = 0;
		while (S_i < j) 
		{
			if ((f = Tree_edge[S_i])->cutvalue < 0) 
			{
				if (rv) 
				{
					if (rv->cutvalue > f->cutvalue) 
						rv = f;
				}
				else 
					rv = Tree_edge[S_i];
				
				if (++cnt >= Search_size) 
					return rv;
			}
			S_i++;
		}
	}
	
	return rv;
}

VEdge *GraphLayout::enter_edge(VEdge* e)
{
	VNode	*v;
	bool	outsearch;

	/* v is the down node */
	if (m_nodes[e->tail_indx]->lim < m_nodes[e->head_indx]->lim) 
	{
		v = m_nodes[e->tail_indx]; 
		outsearch = false;
	}
	else 
	{
		v = m_nodes[e->head_indx];
		outsearch = true;
	}
	
	Enter = NULL;
	Slack = INT_MAX;
	Low = v->low;
	Lim = v->lim;
	
	if (outsearch) 
		dfs_enter_outedge(v);
	else 
		dfs_enter_inedge(v);
	
	return Enter;
}

void GraphLayout::dfs_enter_outedge(VNode* v)
{
	NodeIndex		i;
	int				slack;
	VEdge			*e;
	EdgeListIter	e_iter;		

	e_iter = v->out_edges.begin();
	while (e_iter != v->out_edges.end()) 
	{
		e = *e_iter;
		if (TREE_EDGE(e) == false) 
		{
			if (!SEQ(Low, m_nodes[e->head_indx]->lim, Lim)) 
			{
				slack = SLACK(e);
				if ((slack < Slack) || (Enter == NULL)) 
				{
					Enter = e;
					Slack = slack;
				}
			}
		}
		else if (m_nodes[e->head_indx]->lim < v->lim) 
			dfs_enter_outedge(m_nodes[e->head_indx]);
		
		e_iter++;
	}
	for (i = 0; (i < (long)v->tree_in.size()) && (Slack > 0); i++)
	{
		e = v->tree_in[i];
		if (m_nodes[e->tail_indx]->lim < v->lim) 
			dfs_enter_outedge(m_nodes[e->tail_indx]);
	}
}

void GraphLayout::dfs_enter_inedge(VNode* v)
{
	int				slack;
	NodeIndex		i;
	VEdge			*e;
	EdgeListIter	e_iter;		
	
	e_iter = v->in_edges.begin();
	while (e_iter != v->in_edges.end()) 
	{
		e = *e_iter;
		if (TREE_EDGE(e) == false) 
		{
			if (!SEQ(Low, m_nodes[e->tail_indx]->lim, Lim)) 
			{
				slack = SLACK(e);
				if ((slack < Slack) || (Enter == NULL)) 
				{
					Enter = e;
					Slack = slack;
				}
			}
		}
		else if (m_nodes[e->tail_indx]->lim < v->lim) 
			dfs_enter_inedge(m_nodes[e->tail_indx]);
		
		e_iter++;
	}
	for (i = 0; (i < (long)v->tree_out.size()) && (Slack > 0); i++)
	{
		e = v->tree_out[i];
		if (m_nodes[e->head_indx]->lim < v->lim) 
			dfs_enter_inedge(m_nodes[e->head_indx]);
	}
}

void GraphLayout::rerank(VNode* v, int delta)
{
	NodeIndex	i;
	VEdge		*e;

	v->rank -= delta;
	for (i = 0; (i < (long)v->tree_out.size()); i++)
	{
		e = v->tree_out[i];
		if (e != v->par) 
			rerank(m_nodes[e->head_indx], delta);
	}
		
	for (i = 0; (i < (long)v->tree_in.size()); i++) 
	{
		e = v->tree_in[i];
		if (e != v->par) 
			rerank(m_nodes[e->tail_indx], delta);
	}
}

/* walk up from v to LCA(v,w), setting new cutvalues. */
VNode* GraphLayout::treeupdate(VNode *v, VNode *w, int cutvalue, bool dir)
{
	VEdge	*e;
	bool		d;

	while (!SEQ(v->low,w->lim,v->lim)) {
		e = v->par;
		if (v == m_nodes[e->tail_indx]) 
			d = dir; 
		else 
			d = !dir;
		
		if (d) 
			e->cutvalue += cutvalue; 
		else 
			e->cutvalue -= cutvalue;
		
		if (m_nodes[e->tail_indx]->lim > m_nodes[e->head_indx]->lim) 
			v = m_nodes[e->tail_indx]; 
		else 
			v = m_nodes[e->head_indx];
	}
	
	return v;
}

void GraphLayout::exchange_tree_edges(VEdge *e, VEdge *f)
{
	int		i,j;
	VNode	*n;

	f->tree_index = e->tree_index;
	Tree_edge[e->tree_index] = f;
	e->tree_index = -1;

	n = m_nodes[e->tail_indx];
	i = (n->tree_out.size()) - 1;
	for (j = 0; j <= i; j++) 
		if (n->tree_out[j] == e) 
			break;
		
	n->tree_out[j] = n->tree_out[i];
/*	n->tree_out[i] = NULL;*/
	n->tree_out.pop_back();
	
	n = m_nodes[e->head_indx];
	i = (n->tree_in.size()) - 1;
	for (j = 0; j <= i; j++) 
		if (n->tree_in[j] == e) 
			break;
		
	n->tree_in[j] = n->tree_in[i];
/*	n->tree_in[i] = NULL;*/
	n->tree_in.pop_back();
		
	n = m_nodes[f->tail_indx];
/*	n->tree_out[n->tree_out.size() - 1] = f;*/
	n->tree_out.push_back(f);

	n = m_nodes[f->head_indx];
/*	n->tree_in[n->tree_in.size() - 1] = f;*/
	n->tree_in.push_back(f);		
}

void GraphLayout::update(VEdge *e, VEdge *f)
{
	int		cutvalue,delta;
	VNode	*lca;

	delta = SLACK(f);
	/* "for (v = in nodes in tail side of e) do v->u.rank -= delta;" */
	if (delta > 0) 
	{
		int s;
		s = m_nodes[e->tail_indx]->tree_in.size() + m_nodes[e->tail_indx]->tree_out.size();
		
		if (s == 1) 
			rerank(m_nodes[e->tail_indx], delta);
		else 
		{
			s = m_nodes[e->head_indx]->tree_in.size() + m_nodes[e->head_indx]->tree_out.size();
			if (s == 1) 
				rerank(m_nodes[e->head_indx], -delta);
			else 
			{
				if (m_nodes[e->tail_indx]->lim < m_nodes[e->head_indx]->lim) 
					rerank(m_nodes[e->tail_indx], delta);
				else 
					rerank(m_nodes[e->head_indx], -delta);
			}
		}
	}

	cutvalue = e->cutvalue;
	lca = treeupdate(m_nodes[f->tail_indx], m_nodes[f->head_indx], cutvalue, true);
	
	if (treeupdate(m_nodes[f->head_indx], m_nodes[f->tail_indx], cutvalue, false) != lca) 
		abort();
	f->cutvalue = -cutvalue;
	e->cutvalue = 0;
	exchange_tree_edges(e,f);
	dfs_range(lca,lca->par,lca->low);
}
 
void GraphLayout::scan_and_normalize(void)
{
	VNode		*n;
	NodeIndex	i;
	int			Minrank,Maxrank;
	
	Minrank = INT_MAX;
	Maxrank = -INT_MAX;
	for (i = 0; i < (long)m_nodes_comp.size(); i++)
	{
		n = m_nodes_comp[i];
/*		if (n->node_type == NORMAL) {*/
			Minrank = GGMIN(Minrank, n->rank);
			Maxrank = GGMAX(Maxrank, n->rank);
/*		}*/
	}
	if (Minrank != 0) 
	{
		for (i = 0; i < (long)m_nodes_comp.size(); i++)
		{
			n = m_nodes_comp[i];
			n->rank -= Minrank;
		}
		
		Maxrank -= Minrank;
		Minrank = 0;
	}
}

void GraphLayout::rank(int balance, int maxiter)
{
	int			iter = 0;
	bool		feasible;
	VEdge		*e,*f;

	feasible = init_graph();
	
	if (!feasible) 
		init_rank();

	if (maxiter <= 0) 
		return;

	Search_size = SEARCHSIZE;

	feasible_tree();
	
	while ((e = leave_edge())) 
	{
		f = enter_edge(e);
		update(e,f);
		iter++;
		
		if (iter >= maxiter) break;
	}
	
/*	switch(balance){
	case 1: TB_balance(); break;
	case 2: LR_balance(); break;
	default: 
		scan_and_normalize(); 
		break;
	}*/
	 
	if (balance == 2)
	{
		LR_balance();
		scan_and_normalize(); 		
	}
	else
		scan_and_normalize(); 
	
/*	for (int i = 0; i < m_nodes_comp.size(); i++)
		m_nodes_comp[i]->center.set_y(m_nodes_comp[i]->rank);*/
	
/*	if (Verbose) {
		if (iter >= 100) fputc('\n',stderr);
		fprintf(stderr,"%s%d nodes %d edges %d iter %.2f sec\n",
			ns,N_nodes,N_edges,iter,elapsed_sec());
	}*/
}
 
void GraphLayout::LR_balance(void)
{
	int			delta;
	NodeIndex	i;
	VEdge		*e,*f;

	for (i = 0; i < (long)Tree_edge.size(); i++) 
	{
		e = Tree_edge[i];
		if (e->cutvalue == 0) 
		{
			f = enter_edge(e);
			if (f == NULL) continue;
			delta = SLACK(f);
			if (delta <= 1) continue;
			if (m_nodes[e->tail_indx]->lim < m_nodes[e->head_indx]->lim) 
				rerank(m_nodes[e->tail_indx], delta / 2);
			else 
				rerank(m_nodes[e->head_indx], -delta/2);
		}
	}
}

void GraphLayout::rank_connected_comps(int balance, int maxiter)
{
	VNodeList		node_list;
	NodeIndex		i;
	VNode			*v;
	NodeListIter	v_iter;
	
	
	for (i = 0; i < (NodeIndex)m_nodes.size(); i++)
		node_list.push_back(m_nodes[i]);
	
	while (!node_list.empty())
	{

		for (i = 0; i < (NodeIndex)m_nodes.size(); i++) 
			m_nodes[i]->onstack = false;

		m_nodes_comp.clear();
		m_nodes_comp.resize(0);

		v_iter = node_list.begin();
		v = *v_iter;

		node_list.remove(v);
		m_nodes_comp.push_back(v);

		v->onstack = true;
		comp_dfs(v, node_list);
/*		comp_dfs_down(v, node_list);		*/
		
		rank(balance, maxiter);
	}
}
 
void GraphLayout::comp_dfs(VNode *v, VNodeList &node_list)
{
	EdgeListIter	e_iter;
	VEdge	*e;	
	VNode	*v_new;
	
	e_iter = v->in_edges.begin();
	while (e_iter != v->in_edges.end())
	{
		e = *e_iter;
		
		v_new = m_nodes[e->tail_indx];
		
		if (!v_new->onstack)
		{
			node_list.remove(v_new);
			m_nodes_comp.push_back(v_new);

			v_new->onstack = true;
			comp_dfs(v_new, node_list);
		}
		
		e_iter++;
	}
	
	e_iter = v->out_edges.begin();
	while (e_iter != v->out_edges.end())
	{
		e = *e_iter;
		
		v_new = m_nodes[e->head_indx];
		
		if (!v_new->onstack)
		{
			node_list.remove(v_new);
			m_nodes_comp.push_back(v_new);

			v_new->onstack = true;
			comp_dfs(v_new, node_list);
		}
		
		e_iter++;		
	}
}
/*
void GraphLayout::comp_dfs_down(VNode *v, VNodeList &node_list)
{
	EdgeListIter	e_iter;
	VEdge	*e;	
	VNode	*v_new;	
	
	e_iter = v->out_edges.begin();
	while (e_iter != v->out_edges.end())
	{
		e = *e_iter;
		
		v_new = m_nodes[e->head_indx];
		
		if (!v_new->onstack)
		{
			node_list.remove(v_new);
			m_nodes_comp.push_back(v_new);

			comp_dfs_down(v_new, node_list);
			v_new->onstack = true;
		}
		
		e_iter++;		
	}
}*/
