#include <sstream>
#include <iostream>
#include <fstream>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
 
#include "graphics.h"
#include "Layout.h"
 
using namespace GraphGraphics;

const string AttrStringTemplateStorage::INVALID_TEMPLATE_NAME = "";


void GraphView::make_graph_view( ) 
{
 // <AR> 
  
  delete_nodes_edges();
  
/*************************************************/
  if (m_gl != 0)
  {
    delete m_gl;
    m_gl = 0;  
  }
  
  m_gl = new GraphLayout(m_nodes, m_edges, m_view_groups.get_view_groups() );
	  
  init_nodes_edges();
/************************************************************/
// <AR> set groups
  
  for ( StyleFlags type = m_view_groups.get_first_view_group_id(); type != GraphViewPropertyGroup::DEFAULT; type = m_view_groups.get_next_view_group_id( type) )
  {
    GraphViewPropertyGroup group = m_view_groups.get_view_group( type );
    m_view_groups.reassign_view_group( type, group.get_assigned_attributes() );  
  }
 
/************************************************************/	

  if (m_nodes.size() > 0)
  {
    m_gl->acyclic();
    m_gl->rank_connected_comps(0, INT_MAX);	  
    m_gl->SetRadius(m_radius);
		m_gl->SetSeparation(m_x_separation, m_y_separation);		
		m_gl->init_mincross();
		m_gl->mincross(0, 2);				
		m_gl->set_selfedges( );	  	  
    m_gl->position();	
/*		m_gl->set_edges_coords();	*/
		m_gl->set_edges_coords_new();		
	}

  OneDValue x_max = 0;
  OneDValue y_max = 0;
  for ( unsigned int i = 0; i < m_nodes.size(); i++ )
  {
    VNode *vnode = m_nodes[i];
/*RL    vnode->center = TwoDPoint( vnode->center.get_x() * 3 * m_radius + m_radius, vnode->center.get_y() * 3 * m_radius + m_radius);*/
    
    if ( x_max < vnode->center.get_x() + m_radius )
      x_max = vnode->center.get_x() + m_radius;
    
    if ( y_max < vnode->center.get_y() + m_radius )
      y_max = vnode->center.get_y() + m_radius;
  }

  m_bounds.set_left_top_corner( TwoDPoint(0, 0));
  m_bounds.set_right_bottom_corner( TwoDPoint( x_max, y_max ) );
  m_view_rect = m_bounds;
  
  assign_labels_to_nodes( );
}

void GraphView::zoom_in() 
{ 
  if ( m_bounds.get_width() != 0 && m_bounds.get_height() != 0 )
  {
    if ( m_view_rect.get_width() <= 1 || m_view_rect.get_height() <= 1 )
    {
      TwoDPoint right_bottom;
      right_bottom.set_x( m_view_rect.get_left_top_corner().get_x() + m_view_rect.get_width() + (OneDValue)(m_zoom_value + 1) );
      right_bottom.set_y( m_view_rect.get_left_top_corner().get_y() + 
                         (OneDValue)DROUND((double)m_view_rect.get_width() * (double)m_bounds.get_height() / (double)m_bounds.get_width() ) );
      m_view_rect.set_right_bottom_corner( right_bottom );
    }
    else if ( (double)m_view_rect.get_width() * m_zoom_value < 1000000000.0 && (double)m_view_rect.get_height() * m_zoom_value < 1000000000.0 )
    {
      TwoDPoint left_top = m_view_rect.get_left_top_corner();
      TwoDPoint right_bottom = m_view_rect.get_right_bottom_corner();
         
      left_top.set_x( (OneDValue)DROUND((double)left_top.get_x() * m_zoom_value) );
      left_top.set_y( (OneDValue)DROUND((double)left_top.get_y() * m_zoom_value) );
         
      right_bottom.set_x( (OneDValue)DROUND((double)right_bottom.get_x() * m_zoom_value) );
      right_bottom.set_y( left_top.get_y() + 
                         (OneDValue)DROUND((double)(right_bottom.get_x() - left_top.get_x()) * (double)m_bounds.get_height() / (double)m_bounds.get_width() ) );
         
      m_view_rect.set_left_top_corner( left_top ); 
      m_view_rect.set_right_bottom_corner( right_bottom ); 
    }
    m_zoom_factor = (double)m_view_rect.get_width() / (double)m_bounds.get_width();
  }
  else
    m_zoom_factor = 1.0;
  
  m_fit_method = ARBITRARY_SIZE;
}
   
void GraphView::zoom_out()
{ 
  if ( m_bounds.get_width() != 0 && m_bounds.get_height() != 0 )
  {
    TwoDPoint left_top = m_view_rect.get_left_top_corner();
    TwoDPoint right_bottom = m_view_rect.get_right_bottom_corner();
         
    left_top.set_x( (OneDValue)((double)left_top.get_x() / m_zoom_value) );
    left_top.set_y( (OneDValue)((double)left_top.get_y() / m_zoom_value) );
         
    right_bottom.set_x( (OneDValue)((double)right_bottom.get_x() / m_zoom_value) );
    right_bottom.set_y( left_top.get_y() + 
                           (OneDValue)DROUND((double)(right_bottom.get_x() - left_top.get_x()) * (double)m_bounds.get_height() / (double)m_bounds.get_width() ) );
         
    m_view_rect.set_left_top_corner( left_top ); 
    m_view_rect.set_right_bottom_corner( right_bottom ); 
       
    m_zoom_factor = (double)m_view_rect.get_width() / (double)m_bounds.get_width();
  }
  else
    m_zoom_factor = 1.0;
  
  m_fit_method = ARBITRARY_SIZE;
}

void GraphView::zoom_to_label_view( GraphicArea *g )
{
  FontGeometry     *font_geometry;
  OneDValue         width, max_width = 0;
  OneDValue         height, max_height = 0;
  GraphViewPropertyGroup group;
  VNode            *node = 0;
  Font              orig_font = g->get_font();
  Font              font;
  OneDValue         new_radius = m_radius;
  OneDValue         view_radius = transform_range( m_radius );
  Graph	           *graph = m_graph_view_context->get_graph(); 
  char              c_str[255];
  string            label;
    
  //find max label width and height
  for ( VNodeVector::iterator iter = m_nodes.begin(); iter != m_nodes.end(); iter++ )
  {
    node  = *iter;
    
    if ( node->node_type == NORMAL )
    {
      group = node->get_node_properties();
      font  = group.get_font();
      
      g->set_font( font );
      font_geometry = g->get_font_geometry();
      
      label = node->label; 
            
      width = font_geometry->get_string_width( label );
      height = font_geometry->get_string_height( label );
      
      if ( width > max_width )
        max_width = width;
      
      if ( height > max_height )
        max_height = height;
    }
  }
  
  g->set_font( orig_font );
  
  new_radius = GGMAX( max_width, max_height );
  
  if ( new_radius > view_radius )
  {
    double scale_factor = (double)new_radius / (double)m_radius;
    TwoDPoint scroll_point = inverse_transform_point( m_view_rect.get_left_top_corner() );
    
    m_view_rect.set_left_top_corner( TwoDPoint( 0, 0 ) );
    m_view_rect.set_right_bottom_corner( TwoDPoint( (OneDValue)((double)(m_bounds.get_width()) * scale_factor),
                                                    (OneDValue)((double)(m_bounds.get_height()) * scale_factor) ) );
    
    scroll_point = transform_point( scroll_point );
    
    set_hscroll_value( -scroll_point.get_x() );
    set_vscroll_value( -scroll_point.get_y() );
    
    if ( m_view_rect.get_left_top_corner().get_y() > 0 )
      set_vscroll_value( - m_view_rect.get_left_top_corner().get_y() );
    
    if ( m_view_rect.get_left_top_corner().get_x() > 0 )
      set_hscroll_value( - m_view_rect.get_left_top_corner().get_x() );
  }
}

void GraphView::centering( const Rectangle &rect ) 
{
  if ( m_fit_method == FIT_SELECTION_TO_WINDOW )
    m_fit_method = ARBITRARY_SIZE;
  
  if ( m_selected_nodes.size() > 0 && rect.get_width() > 0 && rect.get_height() > 0 )
  {
    OneDValue min_x = m_bounds.get_right_bottom_corner().get_x();
    OneDValue min_y = m_bounds.get_right_bottom_corner().get_y();
    OneDValue max_y = 0;
    OneDValue max_x = 0;
    TwoDPoint node_center;
       
    for ( unsigned int i = 0; i < m_selected_nodes.size(); i++ )
    {
      node_center = m_nodes[ m_selected_nodes[i] ]->center;
         
      if ( node_center.get_x() > max_x )
        max_x = node_center.get_x();
         
      if ( node_center.get_x() < min_x )
        min_x = node_center.get_x();
         
      if ( node_center.get_y() > max_y )
        max_y = node_center.get_y();
         
      if ( node_center.get_y() < min_y )
        min_y = node_center.get_y();
    }
    
    Rectangle centering_rect = m_bounds.intersection( Rectangle(TwoDPoint( min_x - m_radius, min_y - m_radius), 
                                                                TwoDPoint( max_x + m_radius, max_y + m_radius)) );
        
    Rectangle centering_view_rect( transform_point( centering_rect.get_left_top_corner() ), 
                                   transform_point( centering_rect.get_right_bottom_corner() ) );
    
    TwoDPoint center = centering_view_rect.get_center();
    TwoDPoint rect_center = rect.get_center();
    
    if ( centering_view_rect.get_width() > rect.get_width() || 
         centering_view_rect.get_height() > rect.get_height() )
    {
      double scale_factor =  GGMAX( (double)centering_view_rect.get_width() / (double)rect.get_width(), (double)centering_view_rect.get_height() / (double)rect.get_height() );
      
      TwoDPoint left_top = m_view_rect.get_left_top_corner();
      TwoDPoint right_bottom = m_view_rect.get_right_bottom_corner();
      
      m_view_rect.set_left_top_corner( TwoDPoint( (OneDValue)((double)left_top.get_x() / scale_factor), (OneDValue)((double)left_top.get_y() / scale_factor)) ); 
      m_view_rect.set_right_bottom_corner( TwoDPoint( (OneDValue)((double)right_bottom.get_x() / scale_factor), (OneDValue)((double)right_bottom.get_y() / scale_factor)) );
      
      center = transform_point( centering_rect.get_center() );
    }
       
    set_hscroll_value( center.get_x() - rect_center.get_x() );
    set_vscroll_value( center.get_y() - rect_center.get_y() );
    
    if ( m_view_rect.get_right_bottom_corner().get_y() < rect.get_height() )
      set_vscroll_value( m_view_rect.get_right_bottom_corner().get_y() - rect.get_height() );
    else if ( m_view_rect.get_left_top_corner().get_y() > 0 )
      set_vscroll_value( - m_view_rect.get_left_top_corner().get_y() );
    
    if ( m_view_rect.get_right_bottom_corner().get_x() < rect.get_width() )
      set_hscroll_value( m_view_rect.get_right_bottom_corner().get_x() - rect.get_width() );
    else if ( m_view_rect.get_left_top_corner().get_x() > 0 )
      set_hscroll_value( - m_view_rect.get_left_top_corner().get_x() );
  }
}

void GraphView::do_layout( const Rectangle &display_rect)
{
  double   scale_factor = 1.0;
  Rectangle rect = display_rect;
  TwoDPoint left_top;
  TwoDPoint right_bottom;
  OneDValue height;
  
  switch( m_fit_method )
  {
  case FIT_SELECTION_TO_WINDOW:
    
    if ( m_selected_nodes.size() > 0 && rect.get_width() > 0 && rect.get_height() > 0 )
    {
      OneDValue min_x = m_bounds.get_right_bottom_corner().get_x();
      OneDValue min_y = m_bounds.get_right_bottom_corner().get_y();
      OneDValue max_y = 0;
      OneDValue max_x = 0;
      TwoDPoint node_center;
         
      for ( unsigned int i = 0; i < m_selected_nodes.size(); i++ )
      {
        node_center = m_nodes[ m_selected_nodes[i] ]->center;
           
        if ( node_center.get_x() > max_x )
          max_x = node_center.get_x();
           
        if ( node_center.get_x() < min_x )
          min_x = node_center.get_x();
           
        if ( node_center.get_y() > max_y )
          max_y = node_center.get_y();
           
        if ( node_center.get_y() < min_y )
          min_y = node_center.get_y();
      }
      
      Rectangle centering_rect = m_bounds.intersection( Rectangle(TwoDPoint( min_x - m_radius, min_y - m_radius), 
                                                                  TwoDPoint( max_x + m_radius, max_y + m_radius)) );
          
      Rectangle centering_view_rect( transform_point( centering_rect.get_left_top_corner() ), 
                                     transform_point( centering_rect.get_right_bottom_corner() ) );
      
      scale_factor = (double)rect.get_width() / (double)centering_rect.get_width();
      height = (OneDValue)((double)centering_rect.get_height() * scale_factor);
      
      if ( height > rect.get_height() )
        scale_factor = (double)rect.get_height() / (double)centering_rect.get_height();
      
      left_top = TwoDPoint( (OneDValue)0, (OneDValue)0 );
      right_bottom = TwoDPoint( (OneDValue)((double)m_bounds.get_width() * scale_factor), (OneDValue)((double)m_bounds.get_height() * scale_factor) );
    
      m_view_rect.set_left_top_corner( left_top );
      m_view_rect.set_right_bottom_corner( right_bottom );
      
      left_top = transform_point( centering_rect.get_left_top_corner() );
      
      set_hscroll_value( left_top.get_x() );
      set_vscroll_value( left_top.get_y() );
    }
    
    break;
  
  case FIT_TO_HEIGHT:
    if ( rect.get_height() != m_view_rect.get_height() )
    {
      scale_factor = (double)rect.get_height() / (double)m_bounds.get_height();
      
      right_bottom.set_y( rect.get_right_bottom_corner().get_y() );
      right_bottom.set_x( (OneDValue)DROUND( (double)m_bounds.get_right_bottom_corner().get_x() * scale_factor ) );
      left_top.set_y( rect.get_left_top_corner().get_y() );
      left_top.set_x( (OneDValue)DROUND( (double)m_bounds.get_left_top_corner().get_x() * scale_factor ) );
      m_view_rect.set_right_bottom_corner( right_bottom );
      m_view_rect.set_left_top_corner( left_top );
    }
    break;
  
  case FIT_TO_WIDTH:
    if ( rect.get_width() != m_view_rect.get_width() )
    {
      scale_factor = (double)rect.get_width() / (double)m_bounds.get_width();
      
      right_bottom.set_x( rect.get_right_bottom_corner().get_x() );
      right_bottom.set_y( (OneDValue)DROUND( (double)m_bounds.get_right_bottom_corner().get_y() * scale_factor ) );
      left_top.set_x( rect.get_left_top_corner().get_x() );
      left_top.set_y( (OneDValue)DROUND( (double)m_bounds.get_left_top_corner().get_y() * scale_factor ) );
          
      m_view_rect.set_right_bottom_corner( right_bottom );
      m_view_rect.set_left_top_corner( left_top );
        
    }
    break;
    
  case FIT_TO_WINDOW:
    scale_factor = (double)rect.get_width() / (double)m_bounds.get_width();
    height = (OneDValue)((double)m_bounds.get_height() * scale_factor);
  
    if ( height <= rect.get_height() )
    {
      //fit to width
      right_bottom.set_x( rect.get_right_bottom_corner().get_x() );
      right_bottom.set_y( (OneDValue)DROUND( (double)m_bounds.get_right_bottom_corner().get_y() * scale_factor ) );
      left_top.set_x( rect.get_left_top_corner().get_x() );
      left_top.set_y( (OneDValue)DROUND( (double)m_bounds.get_left_top_corner().get_y() * scale_factor ) );
      
      m_view_rect.set_right_bottom_corner( right_bottom );
      m_view_rect.set_left_top_corner( left_top );
    }      
    else
    {
      //fit to height
      scale_factor = (double)rect.get_height() / (double)m_bounds.get_height();
      
      right_bottom.set_y( rect.get_right_bottom_corner().get_y() );
      right_bottom.set_x( (OneDValue)DROUND( (double)m_bounds.get_right_bottom_corner().get_x() * scale_factor ) );
      left_top.set_y( rect.get_left_top_corner().get_y() );
      left_top.set_x( (OneDValue)DROUND( (double)m_bounds.get_left_top_corner().get_x() * scale_factor ) );
      m_view_rect.set_right_bottom_corner( right_bottom );
      m_view_rect.set_left_top_corner( left_top );
    }
    
    break;
  }
  
  if ( m_view_rect.get_left_top_corner().get_y() > 0 )
    set_vscroll_value( - m_view_rect.get_left_top_corner().get_y() );
    
  if ( m_view_rect.get_left_top_corner().get_x() > 0 )
    set_hscroll_value( - m_view_rect.get_left_top_corner().get_x() );
}

unsigned int GraphView::find_nearest_rank( OneDValue y )
{
  int first_indx = 0;
  int last_indx = m_gl->maxrank;
  int curr_indx = 0;
  OneDValue    curr_dist;
  unsigned int nearest_rank;
  
  while ( first_indx < last_indx )
  {
    curr_indx = (last_indx + first_indx) / 2;
    curr_dist = m_gl->m_ranks[(unsigned int)curr_indx]->v[0]->center.get_y() - y;
      
    if ( labs( curr_dist ) < m_radius ) 
    {
      last_indx = curr_indx;
      first_indx = curr_indx;
    }
    else if ( curr_dist > 0 )
      last_indx = curr_indx - 1;
    else
      first_indx = curr_indx + 1;
  }
    
  if ( last_indx < 0 )
    last_indx = 0;
    
  if ( first_indx > m_gl->maxrank )
    first_indx = m_gl->maxrank;
    
  nearest_rank = (unsigned int)(labs( m_gl->m_ranks[first_indx]->v[0]->center.get_y() - y) >= 
                  labs( m_gl->m_ranks[last_indx]->v[0]->center.get_y() - y))? last_indx: first_indx;
  
  return nearest_rank;
}

unsigned int GraphView::find_nearest_node_in_rank( unsigned int nearest_rank, OneDValue x )
{
  int first_indx = 0;
  int last_indx = m_gl->m_ranks[nearest_rank]->v.size() - 1;
  int curr_indx = 0;
  OneDValue    curr_dist;
  unsigned int nearest_node;
    
  while ( first_indx < last_indx )
  {
    curr_indx = (last_indx + first_indx) / 2;
    curr_dist =  m_gl->m_ranks[nearest_rank]->v[(unsigned int)curr_indx]->center.get_x() - x;
      
    if ( labs( curr_dist ) < m_radius ) 
    {
      last_indx = curr_indx;
      first_indx = curr_indx;
    }
    else if ( curr_dist > 0 )
      last_indx = curr_indx - 1;
    else
      first_indx = curr_indx + 1;
  }
    
  if ( last_indx < 0 )
    last_indx = 0;
    
  if ( (unsigned int)first_indx >= m_gl->m_ranks[nearest_rank]->v.size() )
    first_indx = m_gl->m_ranks[nearest_rank]->v.size() - 1;
    
  nearest_node = (unsigned int)(labs( m_gl->m_ranks[nearest_rank]->v[first_indx]->center.get_x() - x) >= 
                  labs( m_gl->m_ranks[nearest_rank]->v[last_indx]->center.get_x() - x))? last_indx: first_indx;
  
  return nearest_node;
}

NodeIndex GraphView::find_rank_node( const TwoDPoint &point )
{
  NodeIndex    nearest_node_id = Graph::INVALID_NODE_ID;
  if ( m_bounds.inside( point ) )
  {  
    int first_indx = 0;
    int last_indx = m_gl->maxrank;
    int curr_indx = 0;
    OneDValue    curr_dist;
    unsigned int nearest_rank;
    unsigned int nearest_node;
    GraphViewPropertyGroup group;
    
    while ( first_indx < last_indx )
    {
      curr_indx = (last_indx + first_indx) / 2;
      curr_dist = m_gl->m_ranks[(unsigned int)curr_indx]->v[0]->center.get_y() - point.get_y();
      
      if ( labs( curr_dist ) < m_radius ) 
      {
        last_indx = curr_indx;
        first_indx = curr_indx;
      }
      else if ( curr_dist > 0 )
        last_indx = curr_indx - 1;
      else
        first_indx = curr_indx + 1;
    }
    
    if ( last_indx < 0 )
      last_indx = 0;
    
    if ( first_indx > m_gl->maxrank )
      first_indx = m_gl->maxrank;
    
    nearest_rank = (unsigned int)(labs( m_gl->m_ranks[first_indx]->v[0]->center.get_y() - point.get_y()) >= 
                    labs( m_gl->m_ranks[last_indx]->v[0]->center.get_y() - point.get_y()))? last_indx: first_indx;
    
    first_indx  = 0;
    last_indx   = m_gl->m_ranks[nearest_rank]->v.size() - 1;
    
    while ( first_indx < last_indx )
    {
      curr_indx = (last_indx + first_indx) / 2;
      curr_dist =  m_gl->m_ranks[nearest_rank]->v[(unsigned int)curr_indx]->center.get_x() - point.get_x();
      
      if ( labs( curr_dist ) < m_radius ) 
      {
        last_indx = curr_indx;
        first_indx = curr_indx;
      }
      else if ( curr_dist > 0 )
        last_indx = curr_indx - 1;
      else
        first_indx = curr_indx + 1;
    }
    
    if ( last_indx < 0 )
      last_indx = 0;
    
    if ( (unsigned int)first_indx >= m_gl->m_ranks[nearest_rank]->v.size() )
      first_indx = m_gl->m_ranks[nearest_rank]->v.size() - 1;
    
    nearest_node = (unsigned int)(labs( m_gl->m_ranks[nearest_rank]->v[first_indx]->center.get_x() - point.get_x()) >= 
                    labs( m_gl->m_ranks[nearest_rank]->v[last_indx]->center.get_x() - point.get_x()))? last_indx: first_indx;
    
    VNode *node = m_gl->m_ranks[nearest_rank]->v[nearest_node];
    if ( node->node_type == NORMAL)
    {
      group = node->get_node_properties( );
    
      GeometryShape *shape = group.get_node_shape( node->center, m_radius );
    
      if ( shape->inside( point ) )
        nearest_node_id = node->index;
      
      delete shape;
    }
  }
  
  return nearest_node_id;
}

NodeIndex GraphView::get_node_id( const TwoDPoint &point )
{
  if ( m_gl == 0 || m_nodes.size() <= 0 )
    return Graph::INVALID_NODE_ID;
  
  TwoDPoint v_point = inverse_transform_point( point );
  NodeIndex id = find_rank_node(v_point);
  
  return id;
}

int GraphView::get_node_id_from_rect( const Rectangle &rect, NodeIndexVector &res_node_ids )
{
  if ( m_gl == 0 || m_nodes.size() <= 0 )
    return Graph::INVALID_NODE_ID;
  
  Rectangle v_rect;
  v_rect.set_left_top_corner( inverse_transform_point( rect.get_left_top_corner() ) );
  v_rect.set_right_bottom_corner( inverse_transform_point( rect.get_right_bottom_corner() ) );
  
  res_node_ids.clear();
  res_node_ids.resize(0);
  
  for ( NodeIndex i = 0; i < (NodeIndex)m_nodes.size(); i++ )
  {
    if ( m_nodes[i]->node_type == NORMAL && v_rect.inside( m_nodes[i]->center ) )
       res_node_ids.push_back( i );      
  }
  
  return res_node_ids.size();
}

bool GraphView::init_nodes_edges()
{
	VNode *v_node;
	VEdge	*v_edge;
	Graph	*graph = m_graph_view_context->get_graph();
	int	count_nodes = graph->count_nodes();
	NodeIndex i, k;
	bool	mult_edge;

   s_iter = 20;
   t_step = 1.0 / s_iter; 

	
	m_nodes.resize(count_nodes);
	m_edges.reserve(count_nodes);
	
	for (i = 0; i < count_nodes; i++)
	{
		Node*	graph_node = graph->get_node(i);
		int 	out_count = graph_node->get_links_count();		
		
		try
    {
			v_node = new VNode( m_view_groups.m_def_view_properties );
			v_node->selfedges_count = 0;
/*			memset((void*)v_node, 0, sizeof(VNode));*/
			
			for (int j = 0; j < out_count; j++)
			{
				mult_edge = false;
				for (int l = 0; l < j; l++)
					if (graph_node->get_head_node(l) == graph_node->get_head_node(j))
					{
						mult_edge = true;
						break;
					}
				
				if (mult_edge)
				{
					for (k = m_edges.size()-1; k >= 0; k--)
					{
						v_edge = m_edges[k];
						if ((v_edge->tail_indx == j) && (v_edge->head_indx == graph_node->get_head_node(j)))
						{
							v_edge->multiply_edges_count++;
							break;
						}
					}
					
					continue;
				}
				
				if (i == graph_node->get_head_node(j))
				{
					v_node->selfedges_count++;
					continue;
				}
					
				v_edge = new VEdge();
				
				memset((void*)v_edge, 0, sizeof(VEdge));
				
				v_edge->real_head_indx = v_edge->head_indx = graph_node->get_head_node(j);
				v_edge->real_tail_indx = v_edge->tail_indx = i;

        LinkAttr attr = graph_node->get_link_attr(j);
        v_edge->minlen = attr.minlen;
				v_edge->multiply_edges_count = 1;
        v_edge->weight = attr.weight;
        v_edge->inverse_edge = false;
				v_edge->edge_type = NORMAL;
				
				m_edges.push_back(v_edge);
				
				v_node->out_edges.push_back(v_edge);
			}
		} 
		catch (bad_alloc &e)
		{
			cout << "Out of memory!";
			return false;
		}
	
		v_node->index = i;
		
		m_nodes[i] = v_node;
	}
	
	for (i = 0; i < (NodeIndex)m_edges.size(); i++)
	{
		VNode *v_node = m_nodes[m_edges[i]->head_indx];
		
		v_node->in_edges.push_back(m_edges[i]);
	}
	
	for (i = 0; i < (NodeIndex)m_nodes.size(); i++)
		m_nodes[i]->node_type = NORMAL;
/*	
//Debug	
	FILE	*f = fopen("/home/Alex/gr_dbg0", "w");	
	VEdge	*e;
	VNode *n;	
	EdgeListIter	e_iter;		
	
	for (i = 0; i < m_edges.size(); i++)
	{
		VEdge *e = m_edges[i];
		fprintf(f, "%i(%i, %i); ", i, e->tail_indx, e->head_indx);
		
		if ((i % 25) == 0)
			fprintf(f, "\n");
	}
	
	fprintf(f, "\n\n\n");
	
	for (i = 0; i < m_nodes.size(); i++)
	{
		n = m_nodes[i];
		
		fprintf(f, "%i in = ", n->index);
		e_iter = n->in_edges.begin();
		while (e_iter != n->in_edges.end())
		{
			e = *e_iter;
			fprintf(f, "(%i, %i); ", e->tail_indx, e->head_indx);			
			e_iter++;
		}
		
		fprintf(f, "\n\tout = ", n->index);		
		e_iter = n->out_edges.begin();
		while (e_iter != n->out_edges.end())
		{
			e = *e_iter;
			fprintf(f, "(%i, %i); ", e->tail_indx, e->head_indx);			
			e_iter++;			
		}

		fprintf(f, "\n");		
	}
	
	fclose(f);
//End debug		
*/	
	
	return true;
}

void GraphView::delete_nodes_edges()
{
	NodeIndex i, j;
	
	for ( i = 0; i < (NodeIndex)m_nodes.size(); i++ )
    	delete m_nodes[i];
	
	for ( i = 0; i < (NodeIndex)m_edges.size(); i++ )
	{
		for (j = 0; j < (NodeIndex)m_edges[i]->coords.size(); j++)
			delete m_edges[i]->coords[j];
		
    	delete m_edges[i];
	}

   m_edges.clear();
   m_nodes.clear();
   m_edges.resize(0);
   m_nodes.resize(0);
   m_selected_nodes.clear();
   m_selected_nodes.resize(0);
}

void GraphView::delete_edges()
{
	NodeIndex 		i, j;
	
	for ( i = 0; i < (NodeIndex)m_edges.size(); i++ )
	{
		for (j = 0; j < (NodeIndex)m_edges[i]->coords.size(); j++)
			delete m_edges[i]->coords[j];
		
	   	m_edges[i]->coords.clear();
   		m_edges[i]->coords.resize(0);
	}
}

void GraphView::redraw_edges( bool rebuild_position )
{
  if ( m_nodes.empty() || m_edges.empty() )
    return;
  
  delete_edges();
  m_gl->SetRadius(m_radius);
  m_gl->SetSeparation(m_x_separation, m_y_separation);		
  if ( rebuild_position )
    m_gl->position();		
  m_gl->set_edges_coords_new();		
	
  
  OneDValue x_max = 0;
  OneDValue y_max = 0;
    
  for ( unsigned int i = 0; i < m_nodes.size(); i++ )
  {
    VNode *vnode = m_nodes[i];
      
    if ( x_max < vnode->center.get_x() + m_radius )
        x_max = vnode->center.get_x() + m_radius;
      
    if ( y_max < vnode->center.get_y() + m_radius )
        y_max = vnode->center.get_y() + m_radius;
  }
  
  m_bounds.set_left_top_corner( TwoDPoint(0, 0));
  m_bounds.set_right_bottom_corner( TwoDPoint( x_max, y_max ) );
    
  if ( m_bounds.get_width() <= 1 || m_bounds.get_height() <= 1 )
  {
    m_view_rect = m_bounds;
  }
  else
  {
    TwoDPoint left_top = m_view_rect.get_left_top_corner();
    TwoDPoint right_bottom = m_view_rect.get_right_bottom_corner();
             
    right_bottom.set_y( left_top.get_y() + 
                        (OneDValue)DROUND((double)(m_view_rect.get_width()) * (double)m_bounds.get_height() / (double)m_bounds.get_width() ) );
             
    m_view_rect.set_right_bottom_corner( right_bottom ); 
  }
}

GraphView::GraphView( GraphViewContext *app_context ): m_view_groups( this )
{
//   char  *home_path = getenv("HOME");
//   char  pref_path[255];
//   int   fd;  

   m_graph_view_context = app_context;
   m_zoom_value = 1.3;
   m_zoom_factor = 1.0;
/*  
   sprintf(pref_path, "%s/.sgraph/preferences", home_path);
   if ((fd = open(pref_path, O_RDONLY, 0)) != -1)
   {
     unsigned char layout;
     lseek(fd, GRAPH_LAYOUT_OFF, SEEK_SET);
     read(fd, &layout, 1);
     m_fit_method = layout;
   }
   else
*/
   m_fit_method = ARBITRARY_SIZE;

   m_nodes.resize(0);
   m_edges.resize(0);
   m_gl = 0;
  
  // m_selected_edge = 0;
	   
   m_arrow_len = 10;
   pi = 3.141592;
   m_alpha = pi / 9.0;
   m_sin_alpha = sin(m_alpha);
   m_cos_alpha = cos(m_alpha);	 
     
   m_filters = new AttrStringTemplateStorage(  );
   m_filters->set_template( "All attributes", "{ }");
      
   m_radius = 20;
   m_x_separation = 40;
   m_y_separation = 60;	 
   
   m_spline_interpltn = true;
   m_draw_arrows      = true;
}

GraphView::~GraphView( ) 
{
   if (m_gl != 0)
	 {
     delete m_gl;
     m_gl = 0;  
   }
   delete_nodes_edges();
     
   delete m_filters;
}

void GraphView::copy_properties( GraphView *view )
{
  if ( view != 0 )
  {
    if ( m_filters != 0 )
      delete m_filters;
    
    m_filters = new AttrStringTemplateStorage( *(view->m_filters) );
    m_radius = view->m_radius;
    m_zoom_value = view->m_zoom_value;
    m_fit_method = view->m_fit_method;
    m_x_separation = view->m_x_separation;
    m_y_separation = view->m_y_separation;
    m_spline_interpltn = view->m_spline_interpltn;
    
    m_view_groups = view->get_view_properties_storage();
  }  
}

unsigned int GraphView::get_marked_edges( StyleFlags mark, EdgeList &marked_edges ) const
{
  Edge edge;
  marked_edges.clear();
     
  if ( mark == GraphViewPropertiesStorage::EDGE_UNMARKED )
  {
    VNode *node;
    VEdge *e;
    for ( VNodeVector::const_iterator p = m_nodes.begin(); p != m_nodes.end(); p++ )
    {
      node = (VNode*)(*p);
          
      if ( node->node_type == NORMAL )
        for ( VEdgeList::const_iterator iter = node->out_edges.begin(); iter != node->out_edges.end(); iter++ )
        {
          e = *iter;
          if ( e->mark == GraphViewPropertiesStorage::EDGE_UNMARKED )
            marked_edges.push_back( Edge(e->real_tail_indx, e->real_head_indx ) );
        }
    }
  }
  else
  {
    MarkEdgesMap::const_iterator p;
    NodeLinksMap::const_iterator q;
    NodeIndexSet::const_iterator s;
       
    if ( (p = m_marked_edges.find( mark )) != m_marked_edges.end() )
    {
      for ( q = p->second.begin(); q != p->second.end(); q++ )
      {
        edge.start_node = q->first;
        for ( s = q->second.begin(); s != q->second.end(); s++ )
        {
          edge.end_node =  *s;

          if ( edge.is_valid() ) 
            marked_edges.push_back( edge );               
         }
       }
     }
   }
     
   return marked_edges.size();
 }
   
 bool GraphView::remark_edges( StyleFlags mark, GraphicArea *g, StyleFlags new_mark )
 {
   bool       res = false;
   VEdge      *e;
     
   if ( mark != new_mark )
   {
     if ( mark == GraphViewPropertiesStorage::EDGE_UNMARKED )
     {
       VNode *node;
       for ( VNodeVector::iterator p = m_nodes.begin(); p != m_nodes.end(); p++ )
       {
         node = (VNode*)(*p);
           
         if ( node->node_type == NORMAL )
           for ( EdgeListIter iter = node->out_edges.begin(); iter != node->out_edges.end(); iter++ )
           {
             e = *iter;
             if ( e->mark == GraphViewPropertiesStorage::EDGE_UNMARKED )
             {
               mark_edge( e, g, new_mark );
               res = true;
             }
           }
       }
     }
     else 
     {
       MarkEdgesMap::iterator p, new_p;
       NodeLinksMap::iterator q, new_q;
       NodeIndexSet::iterator s;
                  
       if ( (p = m_marked_edges.find( mark )) != m_marked_edges.end() )
       {
         res = true;
         for ( q = p->second.begin(); q != p->second.end(); q++ )
           for ( s = q->second.begin(); s != q->second.end(); s++ )
           {
             for ( e = find_edge( q->first, *s ); e != 0; e = find_next_edge( e ) )
               mark_edge( e, g, new_mark, false );
           }
             
         if ( new_mark != GraphViewPropertiesStorage::EDGE_UNMARKED )
           if ( (new_p = m_marked_edges.find( new_mark )) == m_marked_edges.end() )
             m_marked_edges[new_mark] = p->second;
           else
           {
             for ( q = p->second.begin(); q != p->second.end(); q++ )
             {
               if ( (new_q = new_p->second.find( q->first )) == new_p->second.end() )
                 new_p->second[ q->first ] = q->second;
               else  
                 for ( s = q->second.begin(); s != q->second.end(); s++ )
                   new_q->second.insert( *s ); 
             }
           }
             
         m_marked_edges.erase( p );
       }
     }
   }
     
   return res;
 }
 
 
 void GraphView::remove_all_edge_marking( GraphicArea *g )
 {
   MarkEdgesMap::const_iterator p;
   NodeLinksMap::const_iterator q;
   NodeIndexSet::const_iterator s;
   VEdge *edge;
     
   for ( p = m_marked_edges.begin(); p != m_marked_edges.end(); p++ )
     for ( q = p->second.begin(); q != p->second.end(); q++ )
       for ( s = q->second.begin(); s != q->second.end(); s++ )
       {
         for ( edge = find_edge( q->first, *s ); edge != 0; edge = find_next_edge( edge ) )
           mark_edge( edge, g, GraphViewPropertiesStorage::EDGE_UNMARKED, false );
       }
         
    m_marked_edges.clear();
 }
   
 void GraphView::remark_edge( VEdge *edge, StyleFlags new_mark )
 {
   StyleFlags edge_mark = edge->mark;
   MarkEdgesMap::iterator p;    
   NodeLinksMap::iterator q;
     
   if ( edge_mark != new_mark )
   {
     if ( edge_mark != GraphViewPropertiesStorage::EDGE_UNMARKED )
     {
       if ( (p = m_marked_edges.find( edge_mark )) != m_marked_edges.end() )
       {
         if ( (q = p->second.find(edge->real_tail_indx)) != p->second.end() )
         {
           NodeIndexSet &links = q->second;
           if ( links.size() <= 1 )
             p->second.erase( q );
           else
             links.erase( edge->real_head_indx );
         }
         if ( p->second.empty() )
           m_marked_edges.erase( p );
       }
     }
       
     if ( new_mark != GraphViewPropertiesStorage::EDGE_UNMARKED )
     {
       NodeIndexSet  links;
       NodeLinksMap  links_map;
         
       if ( (p = m_marked_edges.find( new_mark )) == m_marked_edges.end() )
       {
         links.insert( edge->real_head_indx );
         links_map[edge->real_tail_indx] = links;
         m_marked_edges[new_mark] = links_map;
       }
       else
       {
         if ( (q = p->second.find( edge->real_tail_indx )) == p->second.end() )
         {
           links.insert( edge->real_head_indx );
           p->second[edge->real_tail_indx] = links;
         }
         else
           q->second.insert( edge->real_head_indx );
       }
     }
   }
 }
   
void GraphView::mark_edge( VEdge *edge, GraphicArea *g, StyleFlags mark, bool remark )
{
   VEdge *e = edge;
         
   if ( remark )
     remark_edge( edge, mark );
     
   edge->mark = mark;
     
   while ( m_nodes[edge->head_indx]->node_type == VIRTUAL )
   {
     edge = *(m_nodes[edge->head_indx]->out_edges.begin());
     edge->mark = mark;
   }
     
   edge = e;
    
   while ( m_nodes[edge->tail_indx]->node_type == VIRTUAL )
   {
     edge = *(m_nodes[edge->tail_indx]->in_edges.begin());
     edge->mark = mark;
   }
     
   e = edge;
     
   if ( g != 0 )
   {
     draw_spline_edge( g, e );		
     draw_arrow( g, e );
       
     if (m_nodes[e->head_indx]->node_type == VIRTUAL)
       draw_virtual_edges(g, m_nodes[e->head_indx], true);
   }
}
 
void GraphView::get_number_info(unsigned int& nodes_number, unsigned int& edges_number, unsigned int& sel_nodes_number, unsigned int& sel_edges_number)
{
   Graph    *graph = m_graph_view_context->get_graph();
     
   sel_edges_number = 0;
     
   edges_number = graph->count_edges();
   nodes_number = graph->count_nodes();
   sel_nodes_number = m_selected_nodes.size();
     
   for ( MarkEdgesMap::iterator p = m_marked_edges.begin(); p != m_marked_edges.end(); p++ )
     for ( NodeLinksMap::iterator q = p->second.begin(); q != p->second.end(); q++ )
       sel_edges_number += q->second.size();
/*
   VNode    *node;
   EdgeList e_list;
     
   sel_edges_number = edges_number = nodes_number = sel_nodes_number = 0;
   for ( unsigned int i = 0; i < m_nodes.size(); ++i )
   {
     node = m_nodes[i];
     if ( node->node_type == NORMAL )
     {
       ++nodes_number;
       if (node->selected) ++sel_nodes_number;
          
       for ( EdgeListIter::iterator iter = node->out_edges.begin(); iter != node->out_edges.end(); ++iter, ++edges_number );
         
       sel_edges_number = edges_number - get_marked_edges(GraphViewPropertiesStorage::EDGE_UNMARKED, e_list);
     }
   }
*/
}

string GraphView::construct_node_label( GraphAttributes *attr, const string &label_template )
{
  string res = "";
  unsigned int len = label_template.length();
  int    c;
  unsigned int pos;
  
  for ( unsigned int i = 0; i < len; i++ )
  {
    c = label_template[i];
    if ( c != '{' )
      res += c;
    else
    {
      pos = label_template.find( '}', i );
        
      if ( pos == string::npos )
        res += c;
      else
      {
        string attr_name;
        string type;
        AttributeValue *value = 0;
        attr_name = label_template.substr( i + 1, pos - i - 1 );
          
        if ( !(attr->query_value( attr_name, type, &value )) )
          res += c;
        else
        {
          i = pos;
          res += value->get_string();
          delete value;          
        }
      }
    }
  }
  
  return res;
}

void GraphView::set_node_labels( Graph *graph, const GraphNodeLabelTemplate &label_template )
{
  NodeAttributeList  attr_list;
  
  bool parse_res = GraphAttributes::parse_string( label_template.node_attributes, attr_list );
  
  if ( parse_res )
  {
    string label;
    GraphAttributes *attr;
      
    for ( NodeIndex id = graph->find_first_node( attr_list ); id != Graph::INVALID_NODE_ID && id < m_nodes.size(); id = graph->find_next_node( attr_list, id ) )
    {
      attr = graph->get_node(id)->get_attributes();
      if ( attr != 0 )
      {
        label = construct_node_label( attr, label_template.node_label_template );
          
        m_nodes[id]->label = label;
        m_nodes[id]->has_label = true;
      }
    }
  }    
}

void GraphView::assign_labels_to_nodes( )
{
  Graph *graph = m_graph_view_context->get_graph();
     
  if ( m_nodes.size() > 0 && graph != 0 )
  {
    VNode *node;
       
    GraphNodeLabelVector::iterator it = m_label_templates.begin();
    for( ; it != m_label_templates.end(); it++ )
      set_node_labels( graph, *it );
       
    for ( NodeIndex ind = 0; ind < (NodeIndex)m_nodes.size(); ind ++ )
    {
      node = m_nodes[ind];
      if ( !node->has_label && node->node_type == NORMAL )
      {
        stringstream str_stream("");
        str_stream << graph->get_node(ind)->get_sub_id();
        node->label = str_stream.str();
      }
    }
  }
}
