
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <gdk/gdkkeysyms.h>
#include "UnixGroupGUI.h"

#ifdef ENABLE_NLS
#  include <libintl.h>
#  undef _
#  define _(String) dgettext (PACKAGE, String)
#  ifdef gettext_noop
#    define N_(String) gettext_noop (String)
#  else
#    define N_(String) (String)
#  endif
#else
#  define textdomain(String) (String)
#  define dcgettext(Domain,Message,Type) (Message)
#  define bindtextdomain(Domain,Directory) (Domain)*/
#  define _(String) (String)
#  define N_(String) (String)
#endif

#include "UnixTemplateCreationGUI.h"

using namespace GraphGraphics;

#define NAME_CIRCLE  "CIRCLE"
#define NAME_SQUARE  "SQUARE"
#define NAME_POINT   "POINT"
#define NAME_DOUBLE_CIRCLE   "DOUBLE CIRCLE"

const int    GTKPropertiesGroupGUI::UNKNOWN = -2;
const string GTKPropertiesGroupGUI::font_text = "0123456789_abcd ABCD";

GroupStoredData* GTKPropertiesGroupGUI::view_properties_to_group_data( GraphViewPropertyGroup *group )
{
  GroupStoredData *data = new GroupStoredData();
  
  data->pen_set = group->is_pen_set();
  data->pen = group->get_pen();
  data->color_set = group->is_color_set();
  data->color = group->get_color();
  data->font_set = group->is_font_set();
  data->font = group->get_font();
  data->font_color = group->get_text_color();
  data->name = group->get_name();
  data->shape_set = group->is_shape_set();
  data->shape = group->get_shape_type();
  data->filled_set = group->is_filled_set();
  data->filled = group->is_shape_filled();
  data->change = UNCHANGED;
  data->type = group->get_type();
  data->attr_condition = group->get_assigned_attributes();
  data->ratio_set = group->is_shape_ratio_set();
  data->xratio = group->get_shape_x_ratio();
  data->yratio = group->get_shape_y_ratio();
  
  return data;
}

GraphViewPropertyGroup GTKPropertiesGroupGUI::make_view_properties_group( GroupStoredData *data )
{
  GraphViewPropertyGroup group;
  
  group.set_name( data->name );
  group.set_shape_type( data->shape, data->shape_set );
  group.set_shape_filled( data->filled, data->filled_set );
  group.set_font( data->font, data->font_color, data->font_set );
  group.set_color( data->color, data->color_set );
  group.set_pen( data->pen, data->pen_set );
  group.set_shape_xy_ratio( data->xratio, data->yratio, data->ratio_set );
  
  return group;
}

void GTKPropertiesGroupGUI::view_apply_group( GroupStoredData *data )
{
  GraphViewPropertyGroup group;
  switch( data->change )
  {
  case NEW:
    group = make_view_properties_group( data );
    m_view->get_view_properties_storage().add_view_group( data->attr_condition, group );
    break;
  
  case CHANGED:
    group = make_view_properties_group( data );
    m_view->get_view_properties_storage().change_view_group( data->type, group );
    break;
  
  case REASSIGNED:
    group = make_view_properties_group( data );
    m_view->get_view_properties_storage().change_view_group( data->type, group );
    m_view->get_view_properties_storage().reassign_view_group( data->type, data->attr_condition );
    break;
  
  case DELETED:
    m_view->get_view_properties_storage().remove_view_group( data->type );
    break;
  
  default:;  
  }
}

GtkTreeIter GTKPropertiesGroupGUI::add_property_group_to_list( GroupStoredData *data )
{
  GtkTreeModel *model = gtk_tree_view_get_model( GTK_TREE_VIEW( group_treeview ) );
  GtkListStore *list_store = GTK_LIST_STORE( model );
  GtkTreeIter   iter;
    
  gtk_list_store_append( list_store, &iter );
  
  gtk_list_store_set( list_store, &iter, 0, data->name.c_str(), 1, m_last_data_ind, -1 );
  m_group_data_storage[m_last_data_ind] = data;
  
  m_last_data_ind++;

  return iter;
}

void GTKPropertiesGroupGUI::update_pen_state( bool enabled )
{
  if ( enabled )
  {
    GTK_WIDGET_SET_FLAGS (pen_spinbutton, GTK_CAN_FOCUS);
  }
  else
  {
    GTK_WIDGET_UNSET_FLAGS (pen_spinbutton, GTK_CAN_FOCUS);
  }
  
  gtk_entry_set_visibility (GTK_ENTRY (pen_spinbutton), enabled );
    
  gtk_entry_set_visibility (GTK_ENTRY (pen_combo_entry), enabled );
  gtk_widget_set_sensitive (pen_spinbutton, enabled);
  gtk_widget_set_sensitive (pen_style_combo, enabled);
}

void GTKPropertiesGroupGUI::update_filled_state( bool enabled )
{
  gtk_toggle_button_set_inconsistent( GTK_TOGGLE_BUTTON( filled_checkbutton ), !enabled );
  gtk_widget_set_sensitive (filled_checkbutton, enabled);
}

void GTKPropertiesGroupGUI::update_shape_state( bool enabled )
{
  gtk_entry_set_visibility (GTK_ENTRY (shape_combo_entry), enabled );
  gtk_widget_set_sensitive (shape_combo, enabled);
}

void GTKPropertiesGroupGUI::update_shape_ratio_state( bool enabled )
{
  if ( enabled )
  {
    GTK_WIDGET_SET_FLAGS (xratio_spinbutton, GTK_CAN_FOCUS);
    GTK_WIDGET_SET_FLAGS (yratio_spinbutton, GTK_CAN_FOCUS);
  }
  else
  {
    GTK_WIDGET_UNSET_FLAGS (xratio_spinbutton, GTK_CAN_FOCUS);
    GTK_WIDGET_UNSET_FLAGS (yratio_spinbutton, GTK_CAN_FOCUS);
  }
  
  gtk_entry_set_visibility (GTK_ENTRY (xratio_spinbutton), enabled );
  gtk_entry_set_visibility (GTK_ENTRY (yratio_spinbutton), enabled );
    
  gtk_widget_set_sensitive (xratio_spinbutton, enabled);
  gtk_widget_set_sensitive (yratio_spinbutton, enabled);
}

void GTKPropertiesGroupGUI::update_color_state( bool enabled, bool init )
{
  GdkRectangle      invalidate_rect;
  
  gtk_widget_set_sensitive (color_button, enabled);
  
  if ( !init && color_drawingarea->window != NULL )
  {
    invalidate_rect.x = 0;
    invalidate_rect.y = 0;
    gdk_window_get_size( color_drawingarea->window, &(invalidate_rect.width), &(invalidate_rect.height) );
  
    gdk_window_invalidate_rect( color_drawingarea->window, &invalidate_rect, TRUE );
    gdk_window_process_updates( color_drawingarea->window, TRUE );
  }
}

void GTKPropertiesGroupGUI::update_font_state( bool enabled, bool init )
{
  GdkRectangle invalidate_rect;
  
  gtk_widget_set_sensitive (font_button, enabled);
  gtk_widget_set_sensitive (font_color_button, enabled);
  
  if ( !init && font_drawingarea->window != NULL)
  {
    invalidate_rect.x = 0;
    invalidate_rect.y = 0;
    gdk_window_get_size( font_drawingarea->window, &(invalidate_rect.width), &(invalidate_rect.height) );
    
    gdk_window_invalidate_rect( font_drawingarea->window, &invalidate_rect, TRUE );
    gdk_window_process_updates( font_drawingarea->window, TRUE );
  }
}

void GTKPropertiesGroupGUI::update_is_default( GroupStoredData *data )
{
  bool is_default = data->type == GraphViewPropertyGroup::DEFAULT;
      
  gtk_toggle_button_set_inconsistent( GTK_TOGGLE_BUTTON( set_filled_checkbutton ), is_default );
  gtk_widget_set_sensitive (set_filled_checkbutton, !is_default);
  
  gtk_toggle_button_set_inconsistent( GTK_TOGGLE_BUTTON( set_ratio_checkbutton ), is_default );
  gtk_widget_set_sensitive (set_ratio_checkbutton, !is_default);
  
  gtk_toggle_button_set_inconsistent( GTK_TOGGLE_BUTTON( set_font_checkbutton ), is_default );
  gtk_widget_set_sensitive (set_font_checkbutton, !is_default);
  
  gtk_toggle_button_set_inconsistent( GTK_TOGGLE_BUTTON( set_color_checkbutton ), is_default );
  gtk_widget_set_sensitive (set_color_checkbutton, !is_default);
  
  gtk_toggle_button_set_inconsistent( GTK_TOGGLE_BUTTON( set_shape_checkbutton ), is_default );
  gtk_widget_set_sensitive (set_shape_checkbutton, !is_default);
  
  gtk_toggle_button_set_inconsistent( GTK_TOGGLE_BUTTON( set_pen_checkbutton ), is_default );
  gtk_widget_set_sensitive (set_pen_checkbutton, !is_default);
  
  gtk_entry_set_editable( GTK_ENTRY( name_entry), !is_default );
  gtk_widget_set_sensitive (attr_button, !is_default);
  gtk_widget_set_sensitive (del_button, !is_default);
}

void GTKPropertiesGroupGUI::set_pen_properties( GroupStoredData *data )
{
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( set_pen_checkbutton ), data->pen_set );
  
  if ( data->pen_set )
  {
    switch( data->pen.get_line_style() )
    {
    case DrawingPen::DOUBLE_DASH:
    case DrawingPen::ON_OFF_DASH:
      gtk_entry_set_text( GTK_ENTRY( pen_combo_entry ), "DASHED" );
      break;
    
    case DrawingPen::SOLID:
    default:
      gtk_entry_set_text( GTK_ENTRY( pen_combo_entry ), "SOLID" );
      
    }
    
    gtk_spin_button_set_value( GTK_SPIN_BUTTON( pen_spinbutton ), data->pen.get_line_width() );
  }
  else
  {
    gtk_entry_set_text( GTK_ENTRY( pen_combo_entry ), "SOLID" );
    gtk_spin_button_set_value( GTK_SPIN_BUTTON( pen_spinbutton ), 1 );
  }
  update_pen_state( data->pen_set );  
}

void GTKPropertiesGroupGUI::get_pen_properties( GroupStoredData *data )
{
  char      *text;
  StyleFlags pen_style = DrawingPen::SOLID;
  int        width = 1;
  
  data->pen_set = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_pen_checkbutton ) );
  
  if ( data->pen_set )
  {
    text = (char *)gtk_entry_get_text( GTK_ENTRY( pen_combo_entry ) ); 
    string str( text );
    
    if ( str == "SOLID" )
      pen_style = DrawingPen::SOLID;
    else if ( str == "DASHED" )
      pen_style = DrawingPen::ON_OFF_DASH;
    else if ( str == "DOUBLE DASHED" )
      pen_style = DrawingPen::DOUBLE_DASH;
    
    width = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( pen_spinbutton ) );
  }
  
  data->pen = DrawingPen( pen_style, width );
}

void GTKPropertiesGroupGUI::set_shape_properties( GroupStoredData *data )
{
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( set_shape_checkbutton ), data->shape_set );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( set_filled_checkbutton ), data->filled_set );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( set_ratio_checkbutton ), data->ratio_set );
  
  if ( data->shape_set )
  {
    switch( data->shape )
    {
    case GraphViewPropertyGroup::HIDEN_POINT:
      gtk_entry_set_text( GTK_ENTRY( shape_combo_entry ), NAME_POINT );
      break;
    
    case GraphViewPropertyGroup::DOUBLE_CIRCLE:
      gtk_entry_set_text( GTK_ENTRY( shape_combo_entry ), NAME_DOUBLE_CIRCLE );
      break;
    
    case GraphViewPropertyGroup::SQUARE:
      gtk_entry_set_text( GTK_ENTRY( shape_combo_entry ), NAME_SQUARE );
      break;
    
    case GraphViewPropertyGroup::CIRCLE:
    default:
      gtk_entry_set_text( GTK_ENTRY( shape_combo_entry ), NAME_CIRCLE );
    }
    m_curr_shape = data->shape;
  }
  else
  {
    gtk_entry_set_text( GTK_ENTRY( shape_combo_entry ), NAME_CIRCLE );
    m_curr_shape = GraphViewPropertyGroup::CIRCLE;
  }
  
  update_shape_state( data->shape_set );
  
  if ( data->filled_set )
  {
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( filled_checkbutton ), data->filled );
  }
  else
  {
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( filled_checkbutton ), false );
  }
  
  update_filled_state( data->filled_set );
  
  if ( data->ratio_set )
  {
    gtk_spin_button_set_value( GTK_SPIN_BUTTON( xratio_spinbutton ), data->xratio );
    gtk_spin_button_set_value( GTK_SPIN_BUTTON( yratio_spinbutton ), data->yratio );
    if ( data->yratio != 0.0 )
      m_curr_xy_ratio = data->xratio / data->yratio;
    else
      m_curr_xy_ratio = 1.0;
  }
  else
  {
    gtk_spin_button_set_value( GTK_SPIN_BUTTON( xratio_spinbutton ), 1.0 );
    gtk_spin_button_set_value( GTK_SPIN_BUTTON( yratio_spinbutton ), 1.0 );
    m_curr_xy_ratio = 1.0;
  }
  
  update_shape_ratio_state( data->ratio_set );
}

void GTKPropertiesGroupGUI::get_shape_properties( GroupStoredData *data )
{
  char      *text;
  StyleFlags shape = GraphViewPropertyGroup::CIRCLE;
  double     xy_ratio = 1.0;
  
  data->shape_set = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_shape_checkbutton ) );
  
  if ( data->shape_set )
  {
    text = (char *)gtk_entry_get_text( GTK_ENTRY( shape_combo_entry ) ); 
    string str( text );
    
    if ( str == NAME_POINT )
      shape = GraphViewPropertyGroup::HIDEN_POINT;
    else if ( str == NAME_SQUARE )
      shape = GraphViewPropertyGroup::SQUARE;
    else if ( str == NAME_DOUBLE_CIRCLE )
      shape = GraphViewPropertyGroup::DOUBLE_CIRCLE;   
  }
  
  data->shape = shape;
  
  if ((m_curr_shape == GraphViewPropertyGroup::SQUARE && shape != GraphViewPropertyGroup::SQUARE) ||
      (m_curr_shape != GraphViewPropertyGroup::SQUARE && shape == GraphViewPropertyGroup::SQUARE))
    m_critical = true;
  
  m_curr_shape = shape;
  
  data->filled_set = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_filled_checkbutton ) );
  
  if ( data->filled_set )
    data->filled = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( filled_checkbutton ) );
  else
    data->filled = false;
  
  data->ratio_set = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_ratio_checkbutton ) );
  
  if ( data->ratio_set )
  {
    data->xratio = (double)gtk_spin_button_get_value_as_float( GTK_SPIN_BUTTON( xratio_spinbutton ) );
    data->yratio = (double)gtk_spin_button_get_value_as_float( GTK_SPIN_BUTTON( yratio_spinbutton ) );
  }
  else
  {
    data->xratio = 1.0;
    data->yratio = 1.0;
  }
  
  if ( data->yratio != 0 )
    xy_ratio = data->xratio / data->yratio;
  
  if ( m_curr_xy_ratio != xy_ratio )
    m_critical = true;
  
  m_curr_xy_ratio = xy_ratio;
}

void GTKPropertiesGroupGUI::set_color_properties( GroupStoredData *data, bool init )
{
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( set_color_checkbutton ), data->color_set );
  
  if ( data->color_set )
  {
    m_curr_color = data->color;
  }
  else
  {
    m_curr_color = Color::BLACK;
  }
  
  update_color_state( data->color_set, init );
}

void GTKPropertiesGroupGUI::get_color_properties( GroupStoredData *data )
{
  data->color_set = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_color_checkbutton ) );
  
  if ( data->color_set )
    data->color = m_curr_color;
  else
    data->color = Color::BLACK;
}

void GTKPropertiesGroupGUI::set_font_properties( GroupStoredData *data, bool init )
{
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( set_font_checkbutton ), data->font_set );
  
  if ( data->font_set )
  {
    m_curr_font = data->font;
    m_curr_font_color = data->font_color;
  }
  else
  {
    m_curr_font = Font();
    m_curr_font_color = Color::RED;
  }
  update_font_state( data->font_set, init );
}

void GTKPropertiesGroupGUI::get_font_properties( GroupStoredData *data )
{
  data->font_set = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_font_checkbutton ) );
  
  if ( data->font_set )
  {
    data->font = m_curr_font;
    data->font_color = m_curr_font_color;
  }
  else
  {
    data->font = Font();
    data->font_color = Color::RED;
  }
}

void GTKPropertiesGroupGUI::set_all_properties( bool init )
{
  GtkTreeSelection *selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( group_treeview ) );
  GtkTreeIter       sel_iter;
  GtkTreeModel     *model;
  
  if ( gtk_tree_selection_get_selected( selection, &model, &sel_iter) )
  {
    int ind = 0;
    GroupStoredData *data;
    gtk_tree_model_get( model, &sel_iter, 1, &ind, -1 );
    data = m_group_data_storage[ind];
    
    update_is_default( data );
    
    gtk_entry_set_text( GTK_ENTRY( name_entry ), data->name.c_str() );
    gtk_entry_set_text( GTK_ENTRY( attr_entry ), data->attr_condition.c_str() );
    set_color_properties( data, init );
    set_pen_properties( data );
    set_shape_properties( data );
    set_font_properties( data, init );
  }
}

void GTKPropertiesGroupGUI::get_all_properties( GroupStoredData *data )
{
  char      *text;
  
  text = (char *)gtk_entry_get_text( GTK_ENTRY( name_entry ) );
  data->name = string( text );
  
  text = (char *)gtk_entry_get_text( GTK_ENTRY( attr_entry ) );
  data->attr_condition = string( text );
  
  get_pen_properties( data );
  get_font_properties( data );
  get_shape_properties( data );
  get_color_properties( data );
}

/******************************************************************************/
/******************************************************************************/

void GTKPropertiesGroupGUI::handle_color_button_clicked_event( )
{
  GdkRectangle invalidate_rect;
  
  m_curr_color = m_gui->run_choose_color_dialog( m_curr_color );
  
  invalidate_rect.x = 0;
  invalidate_rect.y = 0;
  gdk_window_get_size( color_drawingarea->window, &(invalidate_rect.width), &(invalidate_rect.height) );
  
  gdk_window_invalidate_rect( color_drawingarea->window, &invalidate_rect, TRUE );
  gdk_window_process_updates( color_drawingarea->window, TRUE );
}

void GTKPropertiesGroupGUI::handle_attr_button_clicked_event( )
{
  Graph                           *graph = m_view->get_app_context()->get_graph();
  GTKGraphAttrTemplateCreationView templ_creation( graph->get_graph_attr_name_storage(), 
                                                   UnixGraphAttrView::ATTRIBUTE_TEMPLATE, "", 
                                                   UnixGraphAttrView::ATTRIBUTE_EXPRESSION, "" );
  GraphAttrNameStorage *attr_name_storage = graph->get_graph_attr_name_storage();
  
  templ_creation.set_base_template( attr_name_storage->get_search_format_string() );
  int res = templ_creation.run_dialog( "Create graph attribute template" );
    
  if ( res == GraphGUIModalWindow::APPLIED )
  {
    string    attr = templ_creation.get_result_attr_string();
  
    gtk_entry_set_text( GTK_ENTRY(attr_entry), attr.c_str() );
    m_attr_changed = true;
  }
}

void GTKPropertiesGroupGUI::handle_new_button_clicked_event( )
{
  GtkTreeSelection *selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( group_treeview ) );
  GroupStoredData *data = new GroupStoredData();
  GtkTreeIter      iter;
  
  data->name = "UNNAMED";
  data->pen_set = false;
  data->pen = DrawingPen( DrawingPen::SOLID, 1 );
  data->color_set = false;
  data->color = Color::BLACK;
  data->font_set = false;
  data->font = Font();
  data->font_color = Color::RED;
  data->attr_condition = "{ }";
  data->shape_set = false;
  data->shape = GraphViewPropertyGroup::CIRCLE;
  data->filled_set = false;
  data->filled = false;
  data->change = NEW;
  data->type = UNKNOWN;
  
  iter = add_property_group_to_list( data );
  
  gtk_tree_selection_select_iter( selection, &iter );
  set_all_properties( false );
  m_attr_changed = false;
}

void GTKPropertiesGroupGUI::handle_del_button_activate_event( )
{
  GtkTreeSelection *selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( group_treeview ) );
  
  GtkTreeIter       sel_iter;
  GtkTreeModel     *model;
  
  if ( gtk_tree_selection_get_selected( selection, &model, &sel_iter) )
  {
    int ind = 0;
    GroupStoredData *data;
    gtk_tree_model_get( model, &sel_iter, 1, &ind, -1 );
    data = m_group_data_storage[ind];
    
    if ( data->change != NEW )
      data->change = DELETED;
    else
    {
      delete (GroupStoredData *)m_group_data_storage[ind];
      m_group_data_storage.erase( ind );
    }
      
    GtkListStore *list_store = GTK_LIST_STORE( model );
    
    gtk_list_store_remove( list_store, &sel_iter );
    
    if ( gtk_tree_model_get_iter_first( model, &sel_iter ) )
    {
      gtk_tree_selection_select_iter( selection, &sel_iter );
      set_all_properties( false );
    }
  }
}

void GTKPropertiesGroupGUI::handle_apply_button_clicked_event( )
{
  GtkTreeSelection *selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( group_treeview ) );
  
  GtkTreeIter       sel_iter;
  GtkTreeModel     *model;
  
  if ( gtk_tree_selection_get_selected( selection, &model, &sel_iter) )
  {
    int ind = 0;
    GroupStoredData *data;
    gtk_tree_model_get( model, &sel_iter, 1, &ind, -1 );
    data = m_group_data_storage[ind];
    
    get_all_properties( data );
    
    if ( data->change != NEW )
    {
      if ( m_attr_changed )
        data->change = REASSIGNED;
      else
        data->change = CHANGED;
    }
    
    gtk_list_store_set( GTK_LIST_STORE( model ), &sel_iter, 0, data->name.c_str(), -1 );
  }
}

void GTKPropertiesGroupGUI::handle_set_filled_checkbutton_clicked_event( )
{
  bool enabled = (bool)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_filled_checkbutton ) );
  update_filled_state( enabled );
}

void GTKPropertiesGroupGUI::handle_set_ratio_checkbutton_clicked_event( )
{
  bool enabled = (bool)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_ratio_checkbutton ) );
  update_shape_ratio_state( enabled );
}

void GTKPropertiesGroupGUI::handle_set_shape_checkbutton_clicked_event( )
{
  bool enabled = (bool)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_shape_checkbutton ) );
  update_shape_state( enabled );
}

void GTKPropertiesGroupGUI::handle_set_color_checkbutton_clicked_event( )
{
  bool enabled = (bool)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_color_checkbutton ) );
  update_color_state( enabled, false );
}

gboolean GTKPropertiesGroupGUI::handle_drawingarea1_expose_event( )
{
  GraphicArea *g = m_gui->get_graphic_area( color_drawingarea );
  Rectangle  view_rect;
  gint width, height;
  Color     background = m_view->get_view_properties_storage().get_background_color();
  gboolean enabled = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_color_checkbutton ) );
  
  gdk_window_get_size( color_drawingarea->window, &width, &height );
  view_rect.set_left_top_corner( TwoDPoint( 0, 0 ) );
  view_rect.set_right_bottom_corner( TwoDPoint( width - 1, height - 1 ) );
  
  if ( enabled )
  {
    g->set_background_color( background );
    g->set_foreground_color( background );
    view_rect.draw( g, true );
  }
  
  g->set_foreground_color( Color::BLACK );
  view_rect.draw( g, false );
  
  if ( enabled )
  {
    g->set_foreground_color( m_curr_color );
    view_rect.set_left_top_corner( TwoDPoint( 3, 3 ) );
    view_rect.set_right_bottom_corner( TwoDPoint( width - 3, height - 3 ) );
    
    view_rect.draw( g, true );
  }

  if (g != 0)
    delete g;
  
  return FALSE;
}

void GTKPropertiesGroupGUI::handle_set_font_checkbutton_clicked_event( )
{
  bool enabled = (bool)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_font_checkbutton ) );
  update_font_state( enabled, false );
}

gboolean GTKPropertiesGroupGUI::handle_drawingarea2_expose_event( )
{
  GraphicArea *g = m_gui->get_graphic_area( font_drawingarea );
  Rectangle  view_rect;
  
  gint width, height;
  Color     background = m_view->get_view_properties_storage().get_background_color();
  gboolean enabled = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_font_checkbutton ) );
  
  gdk_window_get_size( font_drawingarea->window, &width, &height );
  view_rect.set_left_top_corner( TwoDPoint( 0, 1 ) );
  view_rect.set_right_bottom_corner( TwoDPoint( width - 1, height - 2 ) );
  
  if ( enabled )
  {
    g->set_background_color( background );
    g->set_foreground_color( background );
    view_rect.draw( g, true );
  }
  
  g->set_foreground_color( Color::BLACK );
  view_rect.draw( g, false );
  
  if (enabled)
  {
    TwoDPoint  center = view_rect.get_center();
    g->set_font( m_curr_font );
    FontGeometry *font_geometry = g->get_font_geometry();
     
    g->set_foreground_color( m_curr_font_color );
    g->draw_text( TwoDPoint( center.get_x() - (font_geometry->get_string_width( font_text ) / 2),
                             center.get_y() - (font_geometry->get_string_height( font_text ) / 2) - 1), font_text );
  }    
  
  if (g != 0)
    delete g;
  
  return FALSE;
}

void GTKPropertiesGroupGUI::handle_font_button_clicked_event( )
{
  GdkRectangle invalidate_rect;
  
  m_curr_font = m_gui->run_choose_font_dialog( m_curr_font );
  
  invalidate_rect.x = 0;
  invalidate_rect.y = 0;
  gdk_window_get_size( font_drawingarea->window, &(invalidate_rect.width), &(invalidate_rect.height) );
  
  gdk_window_invalidate_rect( font_drawingarea->window, &invalidate_rect, TRUE );
  gdk_window_process_updates( font_drawingarea->window, TRUE );
}

void GTKPropertiesGroupGUI::handle_font_color_button_clicked_event( )
{
  GdkRectangle invalidate_rect;
  
  m_curr_font_color = m_gui->run_choose_color_dialog( m_curr_font_color );
  
  invalidate_rect.x = 0;
  invalidate_rect.y = 0;
  gdk_window_get_size( font_drawingarea->window, &(invalidate_rect.width), &(invalidate_rect.height) );
  
  gdk_window_invalidate_rect( font_drawingarea->window, &invalidate_rect, TRUE );
  gdk_window_process_updates( font_drawingarea->window, TRUE );
}

void GTKPropertiesGroupGUI::handle_set_pen_checkbutton_clicked_event( )
{
  bool enabled = (bool)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( set_pen_checkbutton ) );
  update_pen_state( enabled );
}

gboolean GTKPropertiesGroupGUI::handle_group_treeview_button_release_event( )
{
  set_all_properties( false );
  m_attr_changed = false;
  return FALSE;
}

/******************************************************************************/
/******************************************************************************/
static void on_color_button_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_color_button_clicked_event();
}

static void on_attr_button_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_attr_button_clicked_event();
}

static void on_new_button_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_new_button_clicked_event();
}

static void on_del_button_activate( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_del_button_activate_event();
}

static void on_apply_button_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_apply_button_clicked_event();
}

static void on_set_filled_checkbutton_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_set_filled_checkbutton_clicked_event();
}

static void on_set_ratio_checkbutton_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_set_ratio_checkbutton_clicked_event();
}

static void on_filled_checkbutton_clicked( GtkButton *button, gpointer user_data)
{

}


static void on_set_shape_checkbutton_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_set_shape_checkbutton_clicked_event();
}

static void on_set_color_checkbutton_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_set_color_checkbutton_clicked_event();
}

static gboolean on_drawingarea1_expose_event( GtkWidget *widget, GdkEventExpose *event, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  return group_dialog->handle_drawingarea1_expose_event();
}

static void on_set_font_checkbutton_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_set_font_checkbutton_clicked_event();
}

static gboolean on_drawingarea2_expose_event( GtkWidget *widget, GdkEventExpose *event, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  return group_dialog->handle_drawingarea2_expose_event();
}

static void on_font_button_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_font_button_clicked_event();
}

static void on_font_color_button_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_font_color_button_clicked_event();
}

static void on_set_pen_checkbutton_clicked( GtkButton *button, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  group_dialog->handle_set_pen_checkbutton_clicked_event();
}

static gboolean on_group_treeview_button_release_event(GtkWidget *widget, GdkEventButton  *event, gpointer user_data )
{
  GTKPropertiesGroupGUI *group_dialog = (GTKPropertiesGroupGUI*)user_data;
  return group_dialog->handle_group_treeview_button_release_event();
}

/******************************************************************************/
/******************************************************************************/

void GTKPropertiesGroupGUI::create()
{
  GtkTreeModel *list_model;
  GtkTreeViewColumn *column;
  GtkCellRenderer   *render;
  GraphViewPropertyGroup curr_group;
  GroupStoredData *data;
  Font       curr_font;
  Color      curr_color;
  
  m_last_data_ind = 0;
  m_critical = false;
  
  group_dialog = gtk_dialog_new_with_buttons( "View Properties Groups...", NULL, GTK_DIALOG_MODAL, 
                                                                GTK_STOCK_OK, GTK_RESPONSE_OK, 
                                                                GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL );

  gtk_window_set_position (GTK_WINDOW (group_dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_resizable (GTK_WINDOW (group_dialog), FALSE);

  dialog_vbox1 = GTK_DIALOG (group_dialog)->vbox;
  gtk_widget_show (dialog_vbox1);

  table1 = gtk_table_new (7, 3, FALSE);
  gtk_widget_show (table1);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), table1, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (table1), 1);

  label6 = gtk_label_new (_("Name:"));
  gtk_widget_show (label6);
  gtk_table_attach (GTK_TABLE (table1), label6, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 3, 0);
  gtk_label_set_justify (GTK_LABEL (label6), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label6), 0, 0.5);

  frame1 = gtk_frame_new (NULL);
  gtk_widget_show (frame1);
  gtk_table_attach (GTK_TABLE (table1), frame1, 1, 3, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame1), GTK_SHADOW_OUT);

  table2 = gtk_table_new (8, 2, FALSE);
  gtk_widget_show (table2);
  gtk_container_add (GTK_CONTAINER (frame1), table2);

  set_filled_checkbutton = gtk_check_button_new_with_mnemonic ("");
  gtk_widget_show (set_filled_checkbutton);
  gtk_table_attach (GTK_TABLE (table2), set_filled_checkbutton, 1, 2, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  GTK_WIDGET_UNSET_FLAGS (set_filled_checkbutton, GTK_CAN_FOCUS);

  filled_checkbutton = gtk_check_button_new_with_mnemonic ("");
  gtk_widget_show (filled_checkbutton);
  gtk_table_attach (GTK_TABLE (table2), filled_checkbutton, 1, 2, 4, 5,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  GTK_WIDGET_UNSET_FLAGS (filled_checkbutton, GTK_CAN_FOCUS);

  set_shape_checkbutton = gtk_check_button_new_with_mnemonic ("");
  gtk_widget_show (set_shape_checkbutton);
  gtk_table_attach (GTK_TABLE (table2), set_shape_checkbutton, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  //gtk_button_set_relief (GTK_BUTTON (set_shape_checkbutton), GTK_RELIEF_HALF);
  GTK_WIDGET_UNSET_FLAGS (set_shape_checkbutton, GTK_CAN_FOCUS);

  hseparator_ratio = gtk_hseparator_new ();
  gtk_widget_show (hseparator_ratio);
  gtk_table_attach (GTK_TABLE (table2), hseparator_ratio, 0, 2, 5, 6,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 1);
  gtk_widget_set_sensitive (hseparator_ratio, FALSE);
  
  label_ratio = gtk_label_new (_("Set ratio property:"));
  gtk_widget_show (label_ratio);
  gtk_table_attach (GTK_TABLE (table2), label_ratio, 0, 1, 6, 7,
                    (GtkAttachOptions) (GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
  gtk_label_set_justify (GTK_LABEL (label_ratio), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label_ratio), 0, 0.5);
  
  set_ratio_checkbutton = gtk_check_button_new_with_mnemonic ("");
  gtk_widget_show (set_ratio_checkbutton);
  gtk_table_attach (GTK_TABLE (table2), set_ratio_checkbutton, 1, 2, 6, 7,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  GTK_WIDGET_UNSET_FLAGS (set_ratio_checkbutton, GTK_CAN_FOCUS);
  
  label_ratio1 = gtk_label_new (_("Width/height ratio:"));
  gtk_widget_show (label_ratio1);
  gtk_table_attach (GTK_TABLE (table2), label_ratio1, 0, 1, 7, 8,
                    (GtkAttachOptions) (GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
  gtk_label_set_justify (GTK_LABEL (label_ratio1), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label_ratio1), 0, 0.5);
  
  ratio_hbox = gtk_hbox_new( FALSE, 0 );
  gtk_widget_show(ratio_hbox);
  
  xratio_spinbutton_adj = gtk_adjustment_new (0.01, 0.01, 100.00, 0.01, 1.0, 1.0);
  xratio_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (xratio_spinbutton_adj), 0.1, 2);
  gtk_widget_show (xratio_spinbutton);
  gtk_box_pack_start (GTK_BOX (ratio_hbox), xratio_spinbutton, TRUE, TRUE, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (xratio_spinbutton), GTK_UPDATE_IF_VALID);
  gtk_entry_set_invisible_char (GTK_ENTRY (xratio_spinbutton), 32);
  
  label_ratio2 = gtk_label_new (_(" / "));
  gtk_widget_show (label_ratio2);
  gtk_box_pack_start (GTK_BOX (ratio_hbox), label_ratio2, FALSE, FALSE, 0);
  gtk_label_set_justify (GTK_LABEL (label_ratio2), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label_ratio2), 0, 0.5);
  
  yratio_spinbutton_adj = gtk_adjustment_new (0.01, 0.01, 100.00, 0.01, 1.0, 1.0);
  yratio_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (yratio_spinbutton_adj), 0.1, 2);
  gtk_widget_show (yratio_spinbutton);
  gtk_box_pack_start (GTK_BOX (ratio_hbox), yratio_spinbutton, TRUE, TRUE, 0);
  
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (yratio_spinbutton), GTK_UPDATE_IF_VALID);
  gtk_entry_set_invisible_char (GTK_ENTRY (yratio_spinbutton), 32);

  
  gtk_table_attach (GTK_TABLE (table2), ratio_hbox, 1, 2, 7, 8,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
                    
  hseparator2 = gtk_hseparator_new ();
  gtk_widget_show (hseparator2);
  gtk_table_attach (GTK_TABLE (table2), hseparator2, 0, 2, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 1);
  gtk_widget_set_sensitive (hseparator2, FALSE);

  label10 = gtk_label_new (_("Set shape property:"));
  gtk_widget_show (label10);
  gtk_table_attach (GTK_TABLE (table2), label10, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label10, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label10), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label10), 0, 0.5);

  label1 = gtk_label_new (_("Shape:"));
  gtk_widget_show (label1);
  gtk_table_attach (GTK_TABLE (table2), label1, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label1, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label1), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label1), 0, 0.5);

  label12 = gtk_label_new (_("Set 'filled' property:"));
  gtk_widget_show (label12);
  gtk_table_attach (GTK_TABLE (table2), label12, 0, 1, 3, 4,
                    (GtkAttachOptions) (GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label12, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label12), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label12), 0, 0.5);

  label13 = gtk_label_new (_("Filled shape:"));
  gtk_widget_show (label13);
  gtk_table_attach (GTK_TABLE (table2), label13, 0, 1, 4, 5,
                    (GtkAttachOptions) (GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label13, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label13), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label13), 0, 0.5);

  shape_combo = gtk_combo_new ();
  gtk_object_set_data (GTK_OBJECT (GTK_COMBO (shape_combo)->popwin),
                       "GladeParentKey", shape_combo);
  gtk_widget_show (shape_combo);
  gtk_widget_set_size_request (shape_combo, 100, 23);
  gtk_table_attach (GTK_TABLE (table2), shape_combo, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
  GTK_WIDGET_SET_FLAGS (shape_combo, GTK_CAN_FOCUS);
  gtk_combo_set_value_in_list (GTK_COMBO (shape_combo), TRUE, FALSE);
  gtk_combo_set_use_arrows_always (GTK_COMBO (shape_combo), TRUE);
  shape_combo_items = g_list_append (shape_combo_items, (gpointer) NAME_CIRCLE);
  shape_combo_items = g_list_append (shape_combo_items, (gpointer) NAME_DOUBLE_CIRCLE);
  shape_combo_items = g_list_append (shape_combo_items, (gpointer) NAME_SQUARE);
  shape_combo_items = g_list_append (shape_combo_items, (gpointer) NAME_POINT);
  gtk_combo_set_popdown_strings (GTK_COMBO (shape_combo), shape_combo_items);
  g_list_free (shape_combo_items);

  shape_combo_entry = GTK_COMBO (shape_combo)->entry;
  gtk_widget_show (shape_combo_entry);
  GTK_WIDGET_UNSET_FLAGS (shape_combo_entry, GTK_CAN_FOCUS);
  gtk_entry_set_editable (GTK_ENTRY (shape_combo_entry), FALSE);
  gtk_entry_set_invisible_char (GTK_ENTRY (shape_combo_entry), 32);

  label11 = gtk_label_new (_("Node shape"));
  gtk_widget_show (label11);
  gtk_frame_set_label_widget (GTK_FRAME (frame1), label11);
  gtk_label_set_justify (GTK_LABEL (label11), GTK_JUSTIFY_LEFT);

  name_entry = gtk_entry_new ();
  gtk_widget_show (name_entry);
  gtk_table_attach (GTK_TABLE (table1), name_entry, 2, 3, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_SHRINK), 2, 0);

  frame2 = gtk_frame_new (NULL);
  gtk_widget_show (frame2);
  gtk_table_attach (GTK_TABLE (table1), frame2, 1, 3, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame2), GTK_SHADOW_OUT);

  table3 = gtk_table_new (2, 2, FALSE);
  gtk_widget_show (table3);
  gtk_container_add (GTK_CONTAINER (frame2), table3);

  set_color_checkbutton = gtk_check_button_new_with_mnemonic ("");
  gtk_widget_show (set_color_checkbutton);
  gtk_table_attach (GTK_TABLE (table3), set_color_checkbutton, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  GTK_WIDGET_UNSET_FLAGS (set_color_checkbutton, GTK_CAN_FOCUS);

  label15 = gtk_label_new (_("Set color property:  "));
  gtk_widget_show (label15);
  gtk_table_attach (GTK_TABLE (table3), label15, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_SHRINK |GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label15, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label15), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label15), 0, 0.5);

  label2 = gtk_label_new (_("Color:"));
  gtk_widget_show (label2);
  gtk_table_attach (GTK_TABLE (table3), label2, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label2, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label2), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label2), 0, 0.5);

  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox2);
  gtk_table_attach (GTK_TABLE (table3), hbox2, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 1);

  color_drawingarea = gtk_drawing_area_new ();
  gtk_widget_show (color_drawingarea);
  gtk_box_pack_start (GTK_BOX (hbox2), color_drawingarea, FALSE, TRUE, 0);
  gtk_widget_set_size_request (color_drawingarea, 23, 23);

  color_button = gtk_button_new_with_mnemonic (_("Select foreground color"));
  gtk_widget_show (color_button);
  gtk_box_pack_start (GTK_BOX (hbox2), color_button, TRUE, TRUE, 0);

  label14 = gtk_label_new (_("Foreground color"));
  gtk_widget_show (label14);
  gtk_frame_set_label_widget (GTK_FRAME (frame2), label14);
  gtk_label_set_justify (GTK_LABEL (label14), GTK_JUSTIFY_LEFT);

  frame3 = gtk_frame_new (NULL);
  gtk_widget_show (frame3);
  gtk_table_attach (GTK_TABLE (table1), frame3, 1, 3, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame3), GTK_SHADOW_OUT);

  table4 = gtk_table_new (3, 2, FALSE);
  gtk_widget_show (table4);
  gtk_container_add (GTK_CONTAINER (frame3), table4);

  set_font_checkbutton = gtk_check_button_new_with_mnemonic ("");
  gtk_widget_show (set_font_checkbutton);
  gtk_table_attach (GTK_TABLE (table4), set_font_checkbutton, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  GTK_WIDGET_UNSET_FLAGS (set_font_checkbutton, GTK_CAN_FOCUS);

  hbox4 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox4);
  gtk_table_attach (GTK_TABLE (table4), hbox4, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 0);

  font_button = gtk_button_new_with_mnemonic (_("Select font face"));
  gtk_widget_show (font_button);
  gtk_box_pack_start (GTK_BOX (hbox4), font_button, TRUE, TRUE, 0);

  font_color_button = gtk_button_new_with_mnemonic (_("Select font color"));
  gtk_widget_show (font_color_button);
  gtk_box_pack_start (GTK_BOX (hbox4), font_color_button, TRUE, TRUE, 0);

  label17 = gtk_label_new (_("Set font property:    "));
  gtk_widget_show (label17);
  gtk_table_attach (GTK_TABLE (table4), label17, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL | GTK_SHRINK),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label17, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label17), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label17), 0, 0.5);

  label3 = gtk_label_new (_("Font:"));
  gtk_widget_show (label3);
  gtk_table_attach (GTK_TABLE (table4), label3, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label3, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label3), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label3), 0, 0.5);

  label21 = gtk_label_new (_("Font preview:"));
  gtk_widget_show (label21);
  gtk_table_attach (GTK_TABLE (table4), label21, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label21, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label21), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label21), 0, 0.5);

  font_drawingarea = gtk_drawing_area_new ();
  gtk_widget_show (font_drawingarea);
  gtk_table_attach (GTK_TABLE (table4), font_drawingarea, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 2, 1);
  gtk_widget_set_size_request (font_drawingarea, 80, 46);

  label16 = gtk_label_new (_("Label font"));
  gtk_widget_show (label16);
  gtk_frame_set_label_widget (GTK_FRAME (frame3), label16);
  gtk_label_set_justify (GTK_LABEL (label16), GTK_JUSTIFY_LEFT);

  frame4 = gtk_frame_new (NULL);
  gtk_widget_show (frame4);
  gtk_table_attach (GTK_TABLE (table1), frame4, 1, 3, 4, 5,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame4), GTK_SHADOW_OUT);

  table5 = gtk_table_new (3, 2, FALSE);
  gtk_widget_show (table5);
  gtk_container_add (GTK_CONTAINER (frame4), table5);

  set_pen_checkbutton = gtk_check_button_new_with_mnemonic ("");
  gtk_widget_show (set_pen_checkbutton);
  gtk_table_attach (GTK_TABLE (table5), set_pen_checkbutton, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  GTK_WIDGET_UNSET_FLAGS (set_pen_checkbutton, GTK_CAN_FOCUS);           

  pen_spinbutton_adj = gtk_adjustment_new (1, 1, 10, 1, 1, 1);
  pen_spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (pen_spinbutton_adj), 1, 0);
  gtk_widget_show (pen_spinbutton);
  gtk_table_attach (GTK_TABLE (table5), pen_spinbutton, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (pen_spinbutton), GTK_UPDATE_IF_VALID);
  gtk_entry_set_invisible_char (GTK_ENTRY (pen_spinbutton), 32);

  label19 = gtk_label_new (_("Set pen property:    "));
  gtk_widget_show (label19);
  gtk_table_attach (GTK_TABLE (table5), label19, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label19, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label19), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label19), 0, 0.5);

  label4 = gtk_label_new (_("Pen width:"));
  gtk_widget_show (label4);
  gtk_table_attach (GTK_TABLE (table5), label4, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label4, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label4), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label4), 0, 0.5);

  label5 = gtk_label_new (_("Pen style:"));
  gtk_widget_show (label5);
  gtk_table_attach (GTK_TABLE (table5), label5, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label5, 96, 13);
  gtk_label_set_justify (GTK_LABEL (label5), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label5), 0, 0.5);

  pen_style_combo = gtk_combo_new ();
  gtk_object_set_data (GTK_OBJECT (GTK_COMBO (pen_style_combo)->popwin),
                       "GladeParentKey", pen_style_combo);
  gtk_widget_show (pen_style_combo);
  gtk_widget_set_size_request (pen_style_combo, 100, 23);
  gtk_table_attach (GTK_TABLE (table5), pen_style_combo, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 1);
  GTK_WIDGET_SET_FLAGS (pen_style_combo, GTK_CAN_FOCUS);
  gtk_combo_set_value_in_list (GTK_COMBO (pen_style_combo), TRUE, FALSE);
  gtk_combo_set_use_arrows_always (GTK_COMBO (pen_style_combo), TRUE);
  pen_style_combo_items = g_list_append (pen_style_combo_items, (gpointer) _("SOLID"));
  pen_style_combo_items = g_list_append (pen_style_combo_items, (gpointer) _("DASHED"));
//  pen_style_combo_items = g_list_append (pen_style_combo_items, (gpointer) _("DOUBLE DASHED"));
  gtk_combo_set_popdown_strings (GTK_COMBO (pen_style_combo), pen_style_combo_items);
  g_list_free (pen_style_combo_items);

  pen_combo_entry = GTK_COMBO (pen_style_combo)->entry;
  gtk_widget_show (pen_combo_entry);
  GTK_WIDGET_UNSET_FLAGS (pen_combo_entry, GTK_CAN_FOCUS);
  gtk_entry_set_editable (GTK_ENTRY (pen_combo_entry), FALSE);
  gtk_entry_set_invisible_char (GTK_ENTRY (pen_combo_entry), 32);

  label18 = gtk_label_new (_("Drawing Pen"));
  gtk_widget_show (label18);
  gtk_frame_set_label_widget (GTK_FRAME (frame4), label18);
  gtk_label_set_justify (GTK_LABEL (label18), GTK_JUSTIFY_LEFT);

  frame5 = gtk_frame_new (NULL);
  gtk_widget_show (frame5);
  gtk_table_attach (GTK_TABLE (table1), frame5, 1, 3, 5, 6,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame5), GTK_SHADOW_OUT);

  table6 = gtk_table_new (2, 2, FALSE);
  gtk_widget_show (table6);
  gtk_container_add (GTK_CONTAINER (frame5), table6);

  attr_button = gtk_button_new_with_mnemonic (_("Select node attributes"));
  gtk_widget_show (attr_button);
  gtk_table_attach (GTK_TABLE (table6), attr_button, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);

  attr_entry = gtk_entry_new ();
  gtk_widget_show (attr_entry);
  gtk_table_attach (GTK_TABLE (table6), attr_entry, 0, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 2, 1);
  gtk_entry_set_editable (GTK_ENTRY (attr_entry), FALSE);

  label8 = gtk_label_new (_("Assigned to:"));
  gtk_widget_show (label8);
  gtk_table_attach (GTK_TABLE (table6), label8, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 2, 0);
//  gtk_widget_set_size_request (label8, 44, 13);
  gtk_label_set_justify (GTK_LABEL (label8), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (label8), 0, 0.5);

  label20 = gtk_label_new (_("Nodes assigned to group"));
  gtk_widget_show (label20);
  gtk_frame_set_label_widget (GTK_FRAME (frame5), label20);
  gtk_label_set_justify (GTK_LABEL (label20), GTK_JUSTIFY_LEFT);

  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox1);
  gtk_table_attach (GTK_TABLE (table1), hbox1, 1, 3, 6, 7,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 1);

  apply_alignment = gtk_alignment_new (1, 0.5, 0.3, 1);
  gtk_widget_show (apply_alignment);
  gtk_box_pack_start (GTK_BOX (hbox1), apply_alignment, TRUE, TRUE, 0);

  apply_button = gtk_button_new_with_mnemonic (_("Apply changes"));
  gtk_widget_show (apply_button);
  gtk_container_add (GTK_CONTAINER (apply_alignment), apply_button);

  frame6 = gtk_frame_new (NULL);
  gtk_widget_show (frame6);
  gtk_table_attach (GTK_TABLE (table1), frame6, 0, 1, 0, 7,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame6), GTK_SHADOW_OUT);

  table7 = gtk_table_new (2, 2, FALSE);
  gtk_widget_show (table7);
  gtk_container_add (GTK_CONTAINER (frame6), table7);

  list_scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (list_scrolledwindow);
  gtk_table_attach (GTK_TABLE (table7), list_scrolledwindow, 0, 2, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 2, 2);
  gtk_widget_set_size_request (list_scrolledwindow, 160, 80);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (list_scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (list_scrolledwindow), GTK_SHADOW_OUT);

  list_model = (GtkTreeModel *)gtk_list_store_new( 2, G_TYPE_STRING, G_TYPE_INT );
  
  group_treeview = gtk_tree_view_new_with_model( list_model );
  render = gtk_cell_renderer_text_new( );
  column = gtk_tree_view_column_new_with_attributes( "Attribute", render, "text", 0, NULL );
  
  gtk_tree_view_append_column( GTK_TREE_VIEW( group_treeview ), column );
  
  curr_group = m_view->get_view_properties_storage().get_view_group(  GraphViewPropertyGroup::DEFAULT );
  data = view_properties_to_group_data( &curr_group );
  
  add_property_group_to_list( data );
  
  for ( StyleFlags gr_id = m_view->get_view_properties_storage().get_first_view_group_id(); gr_id != GraphViewPropertyGroup::DEFAULT; gr_id = m_view->get_view_properties_storage().get_next_view_group_id( gr_id ) )
  {
    curr_group = m_view->get_view_properties_storage().get_view_group( gr_id );
    data = view_properties_to_group_data( &curr_group );
    add_property_group_to_list( data );
  }
  
 // group_treeview = gtk_tree_view_new_with_model( list_model );
//  render = gtk_cell_renderer_text_new( );
//  column = gtk_tree_view_column_new_with_attributes( "Attribute", render, "text", 0, NULL );
  
//  gtk_tree_view_append_column( GTK_TREE_VIEW( group_treeview ), column );
  
  gtk_widget_show (group_treeview);
  gtk_container_add (GTK_CONTAINER (list_scrolledwindow), group_treeview);
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (group_treeview), FALSE);

  new_button = gtk_button_new_with_mnemonic (_("New group"));
  gtk_widget_show (new_button);
  gtk_table_attach (GTK_TABLE (table7), new_button, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 2);

  del_button = gtk_button_new_with_mnemonic (_("Delete group"));
  gtk_widget_show (del_button);
  gtk_table_attach (GTK_TABLE (table7), del_button, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 2);

  label22 = gtk_label_new (_("Group list"));
  gtk_widget_show (label22);
  gtk_frame_set_label_widget (GTK_FRAME (frame6), label22);
  gtk_label_set_justify (GTK_LABEL (label22), GTK_JUSTIFY_LEFT);
  
  
  gtk_signal_connect (GTK_OBJECT (set_filled_checkbutton), "clicked",
                      GTK_SIGNAL_FUNC (on_set_filled_checkbutton_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (filled_checkbutton), "clicked",
                      GTK_SIGNAL_FUNC (on_filled_checkbutton_clicked),
                      (gpointer)this);
                      
  gtk_signal_connect (GTK_OBJECT (set_ratio_checkbutton), "clicked",
                      GTK_SIGNAL_FUNC (on_set_ratio_checkbutton_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (set_shape_checkbutton), "clicked",
                      GTK_SIGNAL_FUNC (on_set_shape_checkbutton_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (set_color_checkbutton), "clicked",
                      GTK_SIGNAL_FUNC (on_set_color_checkbutton_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (color_drawingarea), "expose_event",
                      GTK_SIGNAL_FUNC (on_drawingarea1_expose_event),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (color_button), "clicked",
                      GTK_SIGNAL_FUNC (on_color_button_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (set_font_checkbutton), "clicked",
                      GTK_SIGNAL_FUNC (on_set_font_checkbutton_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (font_button), "clicked",
                      GTK_SIGNAL_FUNC (on_font_button_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (font_color_button), "clicked",
                      GTK_SIGNAL_FUNC (on_font_color_button_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (font_drawingarea), "expose_event",
                      GTK_SIGNAL_FUNC (on_drawingarea2_expose_event),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (set_pen_checkbutton), "clicked",
                      GTK_SIGNAL_FUNC (on_set_pen_checkbutton_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (attr_button), "clicked",
                      GTK_SIGNAL_FUNC (on_attr_button_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (apply_button), "clicked",
                      GTK_SIGNAL_FUNC (on_apply_button_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (group_treeview), "button_release_event",
                      GTK_SIGNAL_FUNC (on_group_treeview_button_release_event),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (new_button), "clicked",
                      GTK_SIGNAL_FUNC (on_new_button_clicked),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (del_button), "clicked",
                      GTK_SIGNAL_FUNC (on_del_button_activate),
                      (gpointer)this);
  
}

int GTKPropertiesGroupGUI::run( )
{
  create();

  GtkTreeModel *list_model = gtk_tree_view_get_model( GTK_TREE_VIEW( group_treeview ) );  
  GtkTreeSelection *selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( group_treeview ) );
  GtkTreeIter       iter;
  
  if ( gtk_tree_model_get_iter_first( list_model, &iter ) )
  {
    gtk_tree_selection_select_iter( selection, &iter );
    set_all_properties( true );
  }
  
  gint result = gtk_dialog_run( GTK_DIALOG(group_dialog) );
  
  switch( result )
  {
  case GTK_RESPONSE_OK:
    for (GroupStorage::iterator p = m_group_data_storage.begin(); p != m_group_data_storage.end(); p++)
      view_apply_group( p->second );

    break;
  
  default:;
  }
  
  for (GroupStorage::iterator p = m_group_data_storage.begin(); p != m_group_data_storage.end(); p++ )
    delete (GroupStoredData*)p->second;
  
  m_group_data_storage.clear();
  
  gtk_widget_destroy( group_dialog );
  
  m_changes_level = (result == GTK_RESPONSE_OK)? ((m_critical)? NEED_EDGES_RECONSTRUCTION: NEED_REDRAW_ONLY): PROPERTIES_UNCHANGED;
    
  return ((result == GTK_RESPONSE_OK)? GraphGUIModalWindow::APPLIED: GraphGUIModalWindow::CANCELED);
}
