// UnixGraphicArea.cpp: implementation of the UnixGraphicArea class.
//
//////////////////////////////////////////////////////////////////////

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include "../OS_indep/drawing.h"
#include "UnixGraphicArea.h"

using namespace GraphGraphics;

static const gchar stipple_bits[] = { (gchar)0x11, (gchar)(0x11 << 1), (gchar)(0x11 << 2), (gchar)(0x11 << 3), 
                                      (gchar)(0x11), (gchar)(0x11 << 1), (gchar)(0x11 << 2), (gchar)(0x11 << 3) };

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

UnixGraphicArea::UnixGraphicArea( GdkDrawable *drawable, bool pixmap ) : m_os_indep_font(Font::NONE, 12, "")
{
  GdkPixmap *stipple;
	m_draw = drawable;
  m_colormap = gdk_colormap_get_system();
  m_gc = gdk_gc_new(m_draw);

	m_context = gdk_pango_context_get();
	gdk_pango_context_set_colormap(m_context, m_colormap);
	m_p_layout = pango_layout_new(m_context);	
	pango_layout_set_alignment(m_p_layout, PANGO_ALIGN_LEFT);
	
	m_is_pixmap = pixmap;
	
	m_font_geometry = new UnixFontGeometry();
  
  stipple = gdk_bitmap_create_from_data( GDK_WINDOW(drawable), stipple_bits, 8, 8 );
  gdk_gc_set_stipple( m_gc, stipple );
  
  set_paint_mode( GraphicArea::SOLID_COLOR );
  
}

UnixGraphicArea::~UnixGraphicArea()
{
	if ( m_is_pixmap )
		gdk_drawable_unref( m_draw );
	
	gdk_gc_unref( m_gc );
			
  delete m_font_geometry;

  g_object_unref (G_OBJECT(m_p_layout));
  g_object_unref (G_OBJECT(m_context));
}

void UnixGraphicArea::set_foreground_color( const Color &c )
{
  GdkColor    color;
  color.red   = c.get_RGB().red << 8;
  color.green = c.get_RGB().green << 8;
  color.blue  = c.get_RGB().blue << 8;

  gdk_color_alloc(m_colormap, &color);
  gdk_gc_set_foreground(m_gc, &color);
}

Color UnixGraphicArea::get_foreground_color()
{
  Color       c;
  GdkGCValues	values;

  gdk_gc_get_values(m_gc, &values);
  c.set_RGB(RGB(values.foreground.red, values.foreground.green, values.foreground.blue));

  return c;
}

void UnixGraphicArea::set_background_color( const Color &c )
{
  GdkColor    color;
  color.red   = c.get_RGB().red << 8;
  color.green = c.get_RGB().green << 8;
  color.blue  = c.get_RGB().blue << 8;
  gdk_color_alloc(m_colormap, &color);

  gdk_gc_set_background(m_gc, &color);
}

Color UnixGraphicArea::get_background_color()
{
  Color       c;
  GdkGCValues	values;


  gdk_gc_get_values(m_gc, &values);
  c.set_RGB(RGB(values.foreground.red, values.foreground.green, values.foreground.blue));

  return c;
}

void UnixGraphicArea::set_drawing_pen( const DrawingPen &pen )
{
  GdkLineStyle  line_style = GDK_LINE_SOLID;
  GdkCapStyle   cap_style;
  GdkJoinStyle  join_style;

  GdkGCValues	  values;
    
  gdk_gc_get_values(m_gc, &values);

  cap_style = values.cap_style;
  join_style = values.join_style;

  switch (pen.get_line_style())
  {
  case  DrawingPen::SOLID:
    line_style = GDK_LINE_SOLID;
    break;
  case  DrawingPen::ON_OFF_DASH:
    line_style = GDK_LINE_ON_OFF_DASH;
    break;
  case  DrawingPen::DOUBLE_DASH:
    line_style = GDK_LINE_DOUBLE_DASH;
    break;
  }

  gdk_gc_set_line_attributes(m_gc, pen.get_line_width(), line_style, cap_style, join_style);
}

DrawingPen UnixGraphicArea::get_drawing_pen(  )
{
  GdkGCValues	  values;
  DrawingPen    pen;
  StyleFlags    line_style = DrawingPen::SOLID;

  gdk_gc_get_values(m_gc, &values);

  switch (values.line_style)
  {
  case  GDK_LINE_SOLID:
    line_style = DrawingPen::SOLID;
    break;
  case  GDK_LINE_ON_OFF_DASH:
    line_style = DrawingPen::ON_OFF_DASH;
    break;
  case  GDK_LINE_DOUBLE_DASH:
    line_style = DrawingPen::DOUBLE_DASH;
    break;
  }

  pen.set_line_style(line_style);
  pen.set_line_width(values.line_width);

  return pen;
}
 
void UnixGraphicArea::set_paint_mode( StyleFlags paint_mode )
{
  GdkFunction paint_func;

  switch (paint_mode)
  {
  case XOR_MODE:
    paint_func = GDK_XOR;
    break;
  case AND_MODE:
    paint_func = GDK_AND;
    break;
  case OR_MODE:
    paint_func = GDK_OR;
    break;
  case INVERT_MODE:
    paint_func = GDK_INVERT;
    break;
  
  case SOLID_COLOR:
  default:
    paint_func = GDK_COPY;
    
  }

  gdk_gc_set_function(m_gc, paint_func);
}

void UnixGraphicArea::set_fill_mode( StyleFlags mode )
{
  GdkFill fill_func;
  
  switch( mode )
  {
  case FILL_GRID:
    fill_func = GDK_STIPPLED;
    break;
  
  default:
    fill_func = GDK_SOLID;
  }    
  gdk_gc_set_fill( m_gc, fill_func );
  
}

StyleFlags UnixGraphicArea::get_fill_mode()
{
  StyleFlags    fill_mode;
  GdkGCValues	  values;

  gdk_gc_get_values(m_gc, &values);
  switch( values.fill )
  {
  case GDK_STIPPLED:
    fill_mode = FILL_GRID;
    break;
  
  default:
    fill_mode = FILL_SOLID;
  }    
  
  return fill_mode;
}

StyleFlags UnixGraphicArea::get_paint_mode()
{
  StyleFlags    paint_mode;
  GdkGCValues	  values;

  gdk_gc_get_values(m_gc, &values);

  switch (values.function)
  {
  case GDK_XOR:
    paint_mode = XOR_MODE;
    break;
  
  case GDK_AND:
    paint_mode = AND_MODE;
    break;
  
  case GDK_OR:
    paint_mode = OR_MODE;
    break;
  
  case GDK_COPY:
    paint_mode = SOLID_COLOR;
    break;
  
  case GDK_INVERT:
    paint_mode = INVERT_MODE;
    break;
  
	default:
	  paint_mode = SOLID_COLOR;
		break;
  }

  return paint_mode;
}

bool UnixGraphicArea::set_font( const Font &f )
{
  PangoFontDescription *font_desc;

  
  if (f.get_face_name().size() > 0)	
  {
	  font_desc = pango_font_description_from_string (f.get_face_name().c_str());
	  pango_layout_set_font_description(m_p_layout,font_desc);	  
  }
  else
  {
	  font_desc = pango_context_get_font_description(m_context);
	  pango_font_description_set_size(font_desc, f.get_size()*PANGO_SCALE);	  
	  pango_layout_set_font_description(m_p_layout,font_desc);	  	  
  }
  
  m_os_indep_font = f;
  
  m_font_geometry->set_font(f);
  
  return true;
}

void UnixFontGeometry::set_font( const Font &f )
{
  PangoFontDescription *font_desc;

  if (f.get_face_name().size() > 0)	
  {
  	font_desc = pango_font_description_from_string (f.get_face_name().c_str());
	  pango_layout_set_font_description(m_p_layout,font_desc);	  
  }
  else
  {
	  font_desc = pango_context_get_font_description(m_context);
	  pango_font_description_set_size(font_desc, f.get_size()*PANGO_SCALE);	  
	  pango_layout_set_font_description(m_p_layout,font_desc);	  	  
  }
}

Font UnixGraphicArea::get_font()
{
  return m_os_indep_font;
}

Rectangle UnixGraphicArea::get_drawing_bounds()
{
  int	width, height;
		
	gdk_drawable_get_size( GDK_WINDOW(m_draw), &width, &height );
  
	return Rectangle( 0, 0, width - 1, height - 1 );
}

void UnixGraphicArea::set_drawing_bounds( const Rectangle &bounds )
{
	if ( m_is_pixmap )
	{
		gint          width = (gint)bounds.get_width();
		gint					height = (gint)bounds.get_height();
		GdkPixmap			*pixmap = 0;
		
		pixmap = gdk_pixmap_new( GDK_WINDOW(m_draw), width, height, -1 );
		gdk_drawable_unref( m_draw );
		m_draw = pixmap;
	}
}

Rectangle UnixGraphicArea::get_clipping_bounds()
{
  Rectangle rect;
  TwoDPoint p;
	int	x, y, width, height, depth;
	
  gdk_window_get_geometry(m_draw, &x, &y, &width, &height, &depth);
  
  p.set_x(x); p.set_y(y);
  rect.set_left_top_corner( p );
  
  p.set_x(x + width - 1); p.set_y(y + height - 1);
  rect.set_right_bottom_corner( p );

  return rect;
}

void UnixGraphicArea::draw_point( const TwoDPoint &point )
{
  gdk_draw_point(m_draw, m_gc, point.get_x(), point.get_y());
}

void UnixGraphicArea::draw_line( const TwoDPoint &start_point, const TwoDPoint &end_point )
{
	Color c = get_foreground_color();
  gdk_draw_line(m_draw, m_gc, start_point.get_x(), start_point.get_y(), end_point.get_x(), end_point.get_y());
}

void UnixGraphicArea::draw_hollow_rectangle( OneDValue left, OneDValue top, OneDValue right, OneDValue bottom )
{
  gdk_draw_rectangle(m_draw, m_gc, FALSE, left, top, right - left, bottom - top);
}

void UnixGraphicArea::draw_filled_rectangle( OneDValue left, OneDValue top, OneDValue right, OneDValue bottom )
{
  gdk_draw_rectangle(m_draw, m_gc, TRUE, left, top, right - left, bottom - top);
}

void UnixGraphicArea::draw_hollow_circle( const TwoDPoint &center, OneDValue radius )
{
  gdk_draw_arc(m_draw, m_gc, FALSE, center.get_x() - radius, center.get_y() - radius, 2 * radius, 2 * radius, 0, 360 * 64);
}

void UnixGraphicArea::draw_filled_circle( const TwoDPoint &center, OneDValue radius )
{
  gdk_draw_arc(m_draw, m_gc, TRUE, center.get_x() - radius, center.get_y() - radius, 2 * radius, 2 * radius, 0, 360 * 64);
}



UnixFontGeometry::UnixFontGeometry() 
{ 
	m_context = gdk_pango_context_get();
	m_p_layout = pango_layout_new(m_context);	
	pango_layout_set_alignment(m_p_layout, PANGO_ALIGN_LEFT);
}

UnixFontGeometry::~UnixFontGeometry( ) 
{ 
  g_object_unref (G_OBJECT(m_p_layout));
  g_object_unref (G_OBJECT(m_context));
}

OneDValue UnixFontGeometry::get_string_height( const string &str )
{
  int height = 0;

  pango_layout_set_text(m_p_layout, str.c_str(), strlen(str.c_str()));	
  pango_layout_get_pixel_size(m_p_layout, 0, &height);
	

  return height;
}

OneDValue UnixFontGeometry::get_char_height( char ch )
{
  int height = 0;
  char	sym[1];
  sym[0]=ch;	
  pango_layout_set_text(m_p_layout, sym, 1);	
  pango_layout_get_pixel_size(m_p_layout, 0, &height);


  return height;
}

OneDValue UnixFontGeometry::get_char_width( char ch )
{
  int width = 0;

  char	sym[1];
  sym[0]=ch;	
  pango_layout_set_text(m_p_layout, sym, 1);	
  pango_layout_get_pixel_size(m_p_layout, &width, 0);

  return width;
}

OneDValue UnixFontGeometry::get_string_width( const string &str )
{
  int width = 0;

  pango_layout_set_text(m_p_layout, str.c_str(), strlen(str.c_str()));	
  pango_layout_get_pixel_size(m_p_layout, &width, 0);

  return width;
}

FontGeometry* UnixGraphicArea::get_font_geometry()
{
	return m_font_geometry;
}

void UnixGraphicArea::draw_text( const TwoDPoint &start_point, const string &text )
{
	pango_layout_set_text(m_p_layout, text.c_str(), strlen(text.c_str()));
	PangoRectangle r;
	pango_layout_get_pixel_extents(m_p_layout, &r, 0);	
	
	gdk_draw_layout(m_draw, m_gc, start_point.get_x(), start_point.get_y(), m_p_layout);
	
}
 
void UnixGraphicArea::map_into( const Rectangle &rect, GraphicArea *g_into, const Rectangle &rect_into, bool stretch)
{
	gdk_draw_drawable(((UnixGraphicArea*)g_into)->get_drawable(), m_gc, m_draw, rect.get_left_top_corner().get_x(), rect.get_left_top_corner().get_y(), 
										rect_into.get_left_top_corner().get_x(), rect_into.get_left_top_corner().get_y(), rect.get_width(), rect.get_height());  				
}
