(* $Id: reachable.ml,v 1.1.1.1 2003/10/28 22:15:44 rl Exp $ *)

open Spine

open Bdd

open Term

open Actions

open Action_term

open Graph

open Ippkts

open Spec

exception Bad_reachability
exception Bad_spec

(* The main structure for the reachability analysis *)

let term2geom_action t =
  match t with
    Complex("ippkts", _) -> specs2actions (term2ippkts t)
  | _ -> term2list term2action t

let term2geom t = 
  try
    match t with
    | Complex("reachable", [Name src; Name dest; ls]) ->
	Edge(src, dest, term2geom_action ls)
    | _ -> raise Bad_reachability
  with
    Bad_list -> raise Bad_reachability
  | Bad_action ->  raise Bad_reachability
  | Bad_ippkt ->  raise Bad_reachability

let print_node output start_node node =
  if node != start_node && node.frames != falsehood then begin
    let a = spines2actions (bdd2spines node.frames) in
    let action_term = list2term action2term a in
    let term =
      Complex("reachable", [Name start_node.label; 
			     Name node.label; 
			     action_term]) in
    display_term output term;
    output_string output ".\n"
  end

let propagate_node g nodes node =
  List.iter (fun node -> node.frames <- falsehood) nodes;
  node.frames <- truth;
  Prop.propagate g

let propagate_node_print g output nodes node =
	propagate_node g nodes node;
  List.iter (print_node output node) nodes

let load_file input_file =
  let input =
    if input_file = "-" then
      stdin
    else
      open_in input_file in

  (* read input *)

  let geom = List.map term2geom (Read_term.read_all_terms input) in
  let g = make() in
  load g geom;

	(* Return the graph *)
	g

let propagate_file input_file output_file =
  let input =
    if input_file = "-" then
      stdin
    else
      open_in input_file in

  (* read input *)

  let geom = List.map term2geom (Read_term.read_all_terms input) in
  let g = make() in
  load g geom;

  (* Open output *)

  let output =
    if output_file <> "-" && output_file <> input_file then
      open_out output_file
    else
      stdout in

  (* do the main calculation *)

  let nodes = graph2nodes g in

  List.iter (propagate_node_print g output nodes) nodes;

	(* Return the graph *)
	g

let read_spec spec_file =
  let input =
    if spec_file = "-" then
      stdin
    else
      open_in spec_file in

  (* read input *)

	match (Read_term.read_all_terms input) with
	    [] -> raise Bad_spec
	  | head :: tail -> head

let get_spec reach_graph spec savant_proc =
  match spec with
    Complex("all_reachable", [Name src; Name dest; pktlst]) ->
		  let source_node = find reach_graph src and
					nodes = graph2nodes reach_graph
			in
				(propagate_node reach_graph nodes source_node;
				 let target_node = find reach_graph dest in
				 let actions = spines2actions (bdd2spines target_node.frames) in
				 let action_term = list2term action2term actions in
				 let savant_input_term = Complex("subset", [savant_proc pktlst; action_term]) in
						 savant_input_term)
  | Complex("some_reachable", [Name src; Name dest; pktlst]) ->
		  let source_node = find reach_graph src and
					nodes = graph2nodes reach_graph
			in
				(propagate_node reach_graph nodes source_node;
				 let target_node = find reach_graph dest in
				 let actions = spines2actions (bdd2spines target_node.frames) in
				 let action_term = list2term action2term actions in
				 let savant_input_term = Complex("not_empty_intersection", [savant_proc pktlst; action_term]) in
						 savant_input_term)
  | _ -> raise Bad_spec
