
// $Log: genspin.cpp,v $
// Revision 1.8  2004/04/23 23:07:31  alex
// *** empty log message ***
//
// Revision 1.7  2004/04/15 19:06:35  alex
// Support of firewalls
//
// Revision 1.6  2004/03/04 23:41:48  alex
// *** empty log message ***
//
// Revision 1.4  2004/01/31 22:23:32  alex
// *** empty log message ***
//
// Revision 1.2  2003/11/21 01:47:19  alex
// Implementation of Host Object tags support
//
// Revision 1.1.1.1  2003/10/28 22:15:44  rl
// Initial import
//
// Revision 1.1  2002/11/01 21:40:52  oleg
// Initial revision
//

#include <stdlib.h>

#include "Network.hpp"

extern inline const char *BoolStr(bool val);

inline const char *BoolStrLC(bool val)
{
  return (val ? "true" : "false");
}

void PrivilegePrecondition::PrintSPIN(ostream& out) const
{
  out << "net.intruder.privilege[" << host << "] " << ((strcmp(rel, "=") == 0) ? "==" : rel)
      << " " << value;
}

void KnowledgePrecondition::PrintSPIN(ostream& out) const
{
  out << "net.intruder." << name << " == " << value;
}

void SPAPrecondition::PrintSPIN(ostream& out) const
{
  out << "net." << service << "_port = net.intruder." << service << "_port";
}

void VulnerabilityPrecondition::PrintSPIN(ostream& out) const
{
  out << "net.hosts[" << host << "]." << name;
}

void ServicePrecondition::PrintSPIN(ostream& out) const
{
  out << "net.hosts[" << host << "].running_" << name;
}

void ConnectivityPrecondition::PrintSPIN(ostream& out) const
{
  const char *source = (src == SOURCE) ? "source" : "target";
  const char *target = (src == SOURCE) ? "target" : "source";
  out << "net.hosts[" << source << "].conn[" << target << "].";

  if (id == 0)
    out << "path";
  else
    out << network->GetServiceName(id);
}

void TrustPrecondition::PrintSPIN(ostream& out) const
{
  out << "net.hosts[target].trust[source]";
}

void NoTrustPrecondition::PrintSPIN(ostream& out) const
{
  out << "(!(";

  for (HostID id = 0; id < network->GetNumHosts(); id++) {
    if (id != 0)
      out << " && ";
    out << "net.hosts[target].trust[" << id << "]";
  }
  
  out << "))";
}

void ObjectPrecondition::PrintSPIN(ostream& out) const
{
  string host = (m_host == SOURCE) ? "source" : "target";
  out << "(net.hosts[" << host << "].objects." << m_obj_name << ".exists)";
  for ( DOMStringMap::const_iterator p = m_obj_attributes.begin(); p != m_obj_attributes.end(); p++ ) {
    out << " && \\" << endl;
    out << "    (net.hosts[" << host << "].objects." << m_obj_name << "." << p->first << " == " << p->second << ")";
  }
  //out << ")";
}

void SentObjectPrecondition::PrintSPIN(ostream& out) const
{
  string dsthost = (m_dsthost == SOURCE) ? "source" : "target";
  
  out << "(net.hosts[" << dsthost << "].objects." << m_obj_name << ".exists) && \\" << endl;
  
  if ( m_set_source ) {
    string srchost = (m_srchost == SOURCE) ? "source" : "target";
    out << "    (net.hosts[" << dsthost << "].objects." << m_obj_name << ".source_host == " << srchost << ") && \\" << endl;
  }
  
  out << "    (net.hosts[" << dsthost << "].objects." << m_obj_name << ".service == SERVICE_" << m_service << ")";
}

void FirewallReachablePrecondition::PrintSPIN(ostream& out) const
{
  string dsthost = (m_dsthost == SOURCE) ? "source" : "target";
  string srchost = (m_srchost == SOURCE) ? "source" : "target";
  
  out << "net.hosts[" << srchost << "].fw_reachable[" << dsthost << "].srcport[" 
      << FirewallRule::GetPortConstantName( m_srcport ) << "].dstport["
      << FirewallRule::GetPortConstantName( m_dstport ) << "]";
}

void FirewallRulePrecondition::PrintSPIN( ostream& out ) const
{
  if ( !m_result )
    out << "!";
  out << "net.firewalls[target_fw].rule.src[" << m_srchost  
      << "].srcport[" << FirewallRule::GetPortConstantName( m_network->GetFirewallPortName(m_srcport) ) 
      << "].dst[" << m_dsthost
      << "].dstport[" << FirewallRule::GetPortConstantName( m_network->GetFirewallPortName(m_dstport) ) << "]";
}

void OrPrecondition::PrintSPIN(ostream& out) const
{
  out << "(";

  bool first = true;
  for (ListPrecond::const_iterator disjunct = children.begin(); disjunct != children.end(); disjunct++) {
    if (!first) {
      out << " || ";
    }
    else
      first = false;
    (*disjunct)->PrintSPIN(out);
  }
  out << ")";
}

void AndPrecondition::PrintSPIN(ostream& out) const
{
  out << "(";

  bool first = true;
  for (ListPrecond::const_iterator conjunct = children.begin(); conjunct != children.end(); conjunct++) {
    if (!first) {
      out << " && ";
    }
    else
      first = false;
    (*conjunct)->PrintSPIN(out);
  }
  out << ")";
}

void NotPrecondition::PrintSPIN(ostream& out) const
{
  assert(cond != NULL);
  
  out << "(!(";
  cond->PrintSPIN(out);
  out << ")";
}

void Effect::PrintSPIN(ostream& out) const
{
  switch (type) {
  case PRIVILEGE:
    out << "privilege";
    break;
  case SERVICE:
    out << "service";
    break;
  case VULNERABILITY:
    out << "vulnerability";
    break;
  case CONNECTIVITY:
    out << "connectivity";
    break;
  case TRUST:
    out << "trust";
    break;
  }

  if (!host.equals(""))
    out << " host=" << host;
  if (!name.equals(""))
    out << " name=" << name;
  if (!value.equals(""))
    out << " value=" << value;

  out << " attack id=" << attackid;
}

void SendObjectEffect::PrintSPIN(ostream& out) const
{
  out << "sendobject";
  out << " srchost=" << GetSrcHost();
  out << " dsthost=" << GetDstHost();
  out << " object=" << GetObjName();
  out << " attack id=" << attackid;
}

void PassArgEffect::PrintSPIN(ostream& out) const
{
  out << "passarg";
  out << " service=" << GetService();
  out << " object=" << GetObjName();
  out << " attack id=" << attackid;
}

//
void LTLPrivilegeProposition::PrintSPIN(ostream& out) const
{
  out << "(net.intruder.privilege[" << m_host << "] " << ((m_rel == "=") ? "==" : m_rel)
      << " " << m_value << ")";
}

void LTLKnowledgeProposition::PrintSPIN(ostream& out) const
{
  out << "(net.intruder." << m_know_name << " == " << m_value << ")";
}

void LTLVulnerabilityProposition::PrintSPIN(ostream& out) const
{
  out << "(net.hosts[" << m_host << "]." << m_vuln << ")";
}

void LTLServiceProposition::PrintSPIN(ostream& out) const
{
  out << "(net.hosts[" << m_host << "].running_" << m_service << ")";
}

void LTLConnectivityProposition::PrintSPIN(ostream& out) const
{
  out << "(net.hosts[" << m_src << "].conn[" << m_dst << "].";

  if ( !m_use_service )
    out << "path)";
  else
    out << m_service << ")";
}

void LTLTrustProposition::PrintSPIN(ostream& out) const
{
  out << "(net.hosts[" << m_src << "].trust[" << m_dst << "])";
}

void LTLObjectProposition::PrintSPIN(ostream& out) const
{
  out << "((net.hosts[" << m_host << "].objects." << m_obj_name << ".exists)";
  for ( DOMStringMap::const_iterator p = m_obj_attributes.begin(); p != m_obj_attributes.end(); p++ ) {
    out << " && \\" << endl;
    out << "       (net.hosts[" << m_host << "].objects." << m_obj_name << "." << p->first << " == " << p->second << ")";
  }
  out << ")";
}

void LTLSentObjectProposition::PrintSPIN(ostream& out) const
{
  out << "((net.hosts[" << m_dsthost << "].objects." << m_obj_name << ".exists)";
  
  if ( m_set_source ) {
    out << " && \\" << endl;
    out << "       (net.hosts[" << m_dsthost << "].objects." << m_obj_name << ".source_host == " << m_srchost << ")";
  }
  
  if ( m_set_service ) {
    out << " && \\" << endl;
    out << "       (net.hosts[" << m_dsthost << "].objects." << m_obj_name << ".service == SERVICE_" << m_service << ")";
  }
  
  out << ")";
}

bool LTLFormula::GenerateSPIN( ostream& out )
{
  if ( m_exist ) {
    LTLPropositionMap::const_iterator pit = m_atomic_precond.begin();
    for ( ; pit != m_atomic_precond.end(); pit++ ) {
      out << "#define " << pit->first << " ";
      pit->second->PrintSPIN( out );
      out << endl;
    }
    out << endl;
    out << "/*LTL_START" << endl;
    out << m_formula << endl;
    out << "LTL_END*/" << endl << endl;
  }
  
  return m_exist;
}
//

// Generators to SPIN
void Network::GenerateSPIN(ostream& out, string& target)
{
  GenerateDefinesSPIN(out);
  GenerateAttackPreconditionsSPIN(out);
  GenerateAttackEffectsSPIN(out);
  GenerateTypeDefsSPIN(out);
  GenerateMainProcessSPIN(out);
  GenerateNeverClaimSPIN(out, target);
}

//Verifies whether value can be digital or named constant 
DOMString Network::GetAttrConstant( const DOMString &value )
{
  string new_value = value.transcode();
  bool is_digit = new_value.length() < 10;
  
  for ( unsigned int i = 0; i < new_value.length() && is_digit; i++ ) 
    is_digit = isdigit( new_value[i] );
  
  if ( !is_digit ) {
    new_value = "VALUE_" + new_value;
  }
  
  return DOMString::transcode( new_value.c_str() );
}

void Network::GenerateObjAttrValueDefinesSPIN( ostream& out )
{
typedef set<string> AttrValueStringSet;
typedef set<int> AttrValueIntSet;
  
  if ( !obj_attr_values.empty() ) {
    AttrValueIntSet digital_values;
    AttrValueStringSet named_values;
    string      value;
    
    int ivalue = 0;
    
    out << "//Named constants for host object attribute values" << endl; 
    
    for ( DOMStringSet::const_iterator iter = obj_attr_values.begin(); iter != obj_attr_values.end(); iter++ ) {
      value = iter->transcode();
      
      if ( isdigit( value[0] ) ) {
        sscanf( value.c_str(), "%d", &ivalue );
        digital_values.insert( ivalue );
      }
      else      
        named_values.insert( value );
    }
    
    // the unique value
    ivalue = 512;
    for ( AttrValueStringSet::const_iterator p = named_values.begin(); p != named_values.end(); p++, ivalue++ ) {
      
      while ( digital_values.find(ivalue) != digital_values.end() ) 
        ivalue++;
      
      out << "#define " << *p << " " << ivalue << endl;
    }
    
    out << endl;
  }
}

void Network::GenerateDefinesSPIN(ostream& out)
{
  out
    << "" << endl
    << "#define NUMHOSTS " << nHosts << endl;
  if ( DoesModelHaveFirewalls() ) {
    out << "#define NUMPORTS " << fport2name.size() << endl;
    out << "#define NUMFIREWALLS " << fid2name.size() << endl;
    out << "#define MAXPATHS " << max_path_count << endl;
  }
  out << "" << endl
    << "#define none 0" << endl
    << "#define user 1" << endl
    << "#define root 2" << endl
    << "" << endl;
  
  if ( DoesModelHaveFirewalls() ) {
    for ( PortID pid = 0; pid < fport2name.size(); pid++)
      out << "#define " << FirewallRule::GetPortConstantName( fport2name[pid] ) << " "
          << pid << endl;
    cout << "" << endl;
  }
  
  //generate object attribute values named constants
  GenerateObjAttrValueDefinesSPIN( out );
 
  //generate object service names
  if ( !obj_attr_values.empty() ) {  
    
    int serv_id = 1;
    for ( ServiceID sid = 1; sid <= GetNumServices(); sid++, serv_id++) 
      out << "#define SERVICE_" << GetServiceName(sid) << " " << serv_id << endl;
    out << "#define SERVICE_none 0" << endl;
    out << endl;
  }
  
  for (AttackID id = 0; id < nAttacks; id++) {
    char *aname = AttackNameDefineStr(attacks[id]->GetName());
    out << "#define " << aname << " " << id << endl;
    delete [] aname;
  }

  out
    << "" << endl
    << "#define DETECTABLE 1" << endl
    << "#define STEALTHY 0" << endl
    << "" << endl
    << "#define FALSE false" << endl
    << "#define TRUE true" << endl
    << "" << endl;
}

void Network::GenerateAttackPreconditionsSPIN(ostream& out)
{
  out << "// Attack preconditions" << endl << endl;
  
  for (AttackID id = 0; id < nAttacks; id++) {
    out << "#define " << attacks[id]->GetName() << "_enabled("
	      << ((attacks[id]->IsLocal()) ? "target" : 
            (attacks[id]->IsAttackOnFirewall())? "source, target_fw": "source, target") << ") \\" << endl;
    
    bool first = true;
    bool is_empty = attacks[id]->locals.empty() && attacks[id]->globals.empty();
    for (set <Precondition *>::iterator cond = attacks[id]->locals.begin();
	    cond != attacks[id]->locals.end(); cond++) {
      if (!first) {
	      out << " && \\" << endl << "   (";
      }
      else {
        out << "  ((";
        first = false;
      }
      (*cond)->PrintSPIN(out);
      out << ")";
    }

    for (set <Precondition *>::iterator cond = attacks[id]->globals.begin();
	    cond != attacks[id]->globals.end(); cond++) {
      if (!first) {
	      out << " && \\" << endl << "   (";
      }
      else {
        out << "  ((";
        first = false;
      }
      (*cond)->PrintSPIN(out);
      out << ")";
    }
    if ( is_empty ) {
      cerr << "Warning: Attack '" << attacks[id]->GetName() << "' does not have preconditions, it is assumed to be always disabled" << endl;
      out << "  (false";
    }
    out << ")" << endl << endl;
  }

  out << "// No attacks enabled" << endl << endl;
  /*
  out << "#define no_attacks_enabled \\" << endl;
  out << "  (!(";
  bool first = true;
  for (AttackID id = 0; id < nAttacks; id++) {
    DOMString aname = attacks[id]->GetName();
    if (attacks[id]->IsLocal()) {
      for (HostID tgt = 0; tgt < nHosts; tgt++) {
	if (!first) {
	  out << "  || ";
	}
	else
	  first = false;
	out << aname << "_enabled(" << tgt << ")";
	if ((id == nAttacks-1) && (tgt == nHosts-1))
	  out << "))" << endl;
	else
	  out << " \\" << endl;
      }
    }
    else {
      for (HostID src = 0; src < nHosts; src++) {
	for (HostID tgt = 0; tgt < nHosts; tgt++) {
	  if (!first) {
	    out << "  || ";
	  }
	  else
	    first = false;
	  out << aname << "_enabled(" << src << "," << tgt << ")";
	  if ((id == nAttacks-1) && (src == nHosts-1) && (tgt == nHosts-1))
	    out << "))" << endl;
	  else
	    out << " \\" << endl;
	}
      }
    }
  }
  out << endl;
  */
}

void Network::GenerateAttackEffectsSPIN(ostream& out)
{
  bool recompute_fw_reachable = false;
  
  out << "// Attack effects" << endl << endl;
  
  for (AttackID id = 0; id < nAttacks; id++) {
    out << "#define execute_" << attacks[id]->GetName() << "("
	      << ((attacks[id]->IsLocal()) ? "target" : 
            (attacks[id]->IsAttackOnFirewall())? "source, target_fw": "source, target") << ") \\" << endl;
    out << "  atomic { \\" << endl;

    char *aname = AttackNameDefineStr(attacks[id]->GetName());
    out << "    net.attack_id = " << aname << "; \\" << endl;
    delete [] aname;

    // Attack source and target
    out << "    net.source_ip = " << ((attacks[id]->IsLocal()) ? "target" : "source") << "; \\" << endl;
    out << "    net.target_ip = " << ((attacks[id]->IsAttackOnFirewall()) ? "-1": "target") << "; \\" << endl;
    
    if ( DoesModelHaveFirewalls() )
      out << "    net.target_fwi = " << (attacks[id]->IsAttackOnFirewall()? "target_fw": "-1" ) << "; \\" << endl;

    // Detectability
    switch (attacks[id]->IsDetectable()) {
      case Attack::Yes:
        out << "    net.attack_flavor = DETECTABLE; \\" << endl;
        out << "    net.alarm = net.alarm || net.ids[source].detected[target]; \\" << endl;
        break;
      case Attack::No:
        out << "    net.attack_flavor = STEALTHY; \\" << endl;
        break;
      case Attack::Varies:
        out << "    if \\" << endl;
        out << "    :: net.attack_flavor = STEALTHY; \\" << endl;
        out << "    :: net.attack_flavor = DETECTABLE; \\" << endl;
        out << "    fi; \\" << endl;
        out << "    net.alarm = net.alarm || (net.ids[target].detected[target] && net.attack_flavor == DETECTABLE); \\" << endl;
        break;
    }

    // Attack Effects
    for (vector <Effect *>::iterator effp = attacks[id]->effects.begin(); effp != attacks[id]->effects.end();
	        effp++) {
      Effect *eff = *effp;
      switch (eff->type) {
        case Effect::PRIVILEGE:
          out << "    net.intruder.privilege[" << eff->host << "] = " << eff->value << "; \\" << endl;
          break;
        case Effect::SERVICE:
	        out << "    net.hosts[" << eff->host << "].running_" << eff->name
              << " = " << eff->value << "; \\" << endl;
          break;
        case Effect::VULNERABILITY:
          out << "    net.hosts[" << eff->host << "]." << eff->name
              << " = " << eff->value << "; \\" << endl;
          break;
        case Effect::CONNECTIVITY:
	        break;
        case Effect::TRUST:
          for (HostID hid = 0; hid < GetNumHosts(); hid++) {
            out << "    net.hosts[target].trust[" << hid << "] = " << eff->value << "; \\" << endl;
          }
          break;
        case Effect::KNOWLEDGE:
          out << "    net.intruder." << eff->name << " = " << eff->value << "; \\" << endl;
          break;
        case Effect::SENDOBJECT: {
          SendObjectEffect *so_effect = dynamic_cast<SendObjectEffect *>(eff);
          out << "    net.hosts[" << so_effect->GetDstHost() << "].objects." << so_effect->GetObjName()
              << ".exists = TRUE; \\" << endl;
             
          out << "    net.hosts[" << so_effect->GetDstHost() << "].objects." << so_effect->GetObjName()
              << ".source_host = " << so_effect->GetSrcHost() << "; \\" << endl;
          
          out << "    net.hosts[" << so_effect->GetDstHost() << "].objects." << so_effect->GetObjName()
              << ".service = SERVICE_" << so_effect->GetDstService() << "; \\" << endl;
          
          HostObjectAttrNameMap::const_iterator a_iter = host_objects.find(so_effect->GetObjName());
          if ( a_iter != host_objects.end() )
            for ( DOMStringSet::const_iterator p = a_iter->second.begin(); p != a_iter->second.end(); p++ ) {
              out << "    net.hosts[" << so_effect->GetDstHost() << "].objects." << so_effect->GetObjName()
                  << "." << *p << " = " << "net.hosts[" << so_effect->GetSrcHost() << "].objects." 
                  << so_effect->GetObjName() << "." << *p << "; \\" << endl;
            }
        }
          break;
        
        case Effect::PASSARG: {
          PassArgEffect *pa_effect = dynamic_cast<PassArgEffect *>(eff);
          
          out << "    net.hosts[target].objects." << pa_effect->GetObjName() << ".service = SERVICE_" 
              << pa_effect->GetService() << "; \\" << endl;
        }
          break;
        
        case Effect::FIREWALLACTION: {
          FirewallActionEffect *fa_effect = dynamic_cast<FirewallActionEffect *>(eff);
          if ( fa_effect->IsFWStatusEffect() ) {
            out << "    net.firewalls[";
            out << "target_fw";
            out << "].on = " << BoolStrLC( fa_effect->GetResult() ) << "; \\" << endl;
          }
          else
            GenerateFWRuleSPIN( out, fa_effect->GetFWRule(), fa_effect->GetResult() ); 

          recompute_fw_reachable = true;          
        }
          break;
        
        case Effect::UNDEFINED:
          cerr << "**Warning**: Undefined effect!" << endl;
          break;
      }
    }
    
    if ( recompute_fw_reachable )
      out << "    goto COMPUTE_FW_REACHABILITY; \\" << endl; 
    
    out << "  }" << endl << endl;
  }
}

void Network::GenerateTypeDefsSPIN(ostream& out)
{
  if ( !DoesModelHaveFirewalls() ) {
    // Connectivity
    out << "typedef Conn {" << endl;
    out << "  bool path;" << endl;
    for (ServiceID sid = 1; sid <= GetNumServices(); sid++) {
      out << "  bool " << GetServiceName(sid) << ";" << endl;
    }
    out << "}" << endl << endl;
  }
  else {
    // Firewalls
    out << "typedef FWRuleDst {" << endl;
    out << "  bool dstport[NUMPORTS];" << endl;
    out << "}" << endl << endl;
    
    out << "typedef FWRuleSrcPort {" << endl;
    out << "  FWRuleDst dst[NUMHOSTS];" << endl;
    out << "}" << endl << endl;
    
    out << "typedef FWRuleSrc {" << endl;
    out << "  FWRuleSrcPort srcport[NUMPORTS];" << endl;
    out << "}" << endl << endl;
    
    out << "typedef FWRule {" << endl;
    out << "  FWRuleSrc src[NUMHOSTS];" << endl;
    out << "}" << endl << endl;
    
    out << "typedef Firewall {" << endl;
    out << "  bool on;" << endl;
    out << "  FWRule rule;" << endl;
    out << "}" << endl << endl;
    
    out << "typedef FWReachable {" << endl;
    out << "  FWRuleDst srcport[NUMPORTS];" << endl;
    out << "}" << endl << endl;
    
    out << "typedef FWPath {" << endl;
    out << "  bool firewalls[NUMFIREWALLS];" << endl;
    out << "}" << endl << endl;
    
    out << "typedef FWDestPath {" << endl;
    out << "  FWPath _[MAXPATHS];" << endl;
    out << "}" << endl << endl;
  }
  
  //Host Objects
  if ( !host_objects.empty() ) {
    
    HostObjectAttrNameMap::const_iterator obj;
    for (obj = host_objects.begin(); obj != host_objects.end(); obj++) {
      out << "typedef HostObj_" << obj->first << " {" << endl;
      out << "  bool exists;" << endl;
      out << "  int source_host;" << endl;
      out << "  int service;" << endl;
      for (DOMStringSet::const_iterator attr = obj->second.begin(); attr != obj->second.end(); attr++ ) {
        out << "  int " << *attr << ";" << endl;
      }
      out << "}" << endl << endl;
    }
  
    out << "typedef HostObjects {" << endl;
    for (obj = host_objects.begin(); obj != host_objects.end(); obj++) {
      out << "  HostObj_" << obj->first << " " << obj->first << ";" << endl;
    }
    out << "}" << endl << endl;
  }
  
  // Host
  out << "typedef Host {" << endl;
  for (ServiceID sid = 1; sid <= GetNumServices(); sid++) {
    out << "  bool running_" << GetServiceName(sid) << ";" << endl;
  }
  out << endl;
  for (set <DOMString, less_str>::const_iterator vul = vulnerabilities.begin();
       vul != vulnerabilities.end(); vul++) {
    out << "  bool " << *vul << ";" << endl;
  }
  out << endl;
  
  if ( !host_objects.empty() )
    out << "  HostObjects objects;" << endl;
  
  if ( DoesModelHaveFirewalls() ) {
    out << "  FWReachable fw_reachable[NUMHOSTS];" << endl;
    out << "  FWDestPath path[NUMHOSTS];" << endl;
  }
  else
    out << "  Conn conn[NUMHOSTS];" << endl;
  
  out << "  bool trust[NUMHOSTS];" << endl;
  out << "}" << endl << endl;

  // Intruder
  out << "typedef Intruder {" << endl;
  out << "  byte privilege[NUMHOSTS];" << endl;
  for (map<DOMString, bool, less_str>::const_iterator kitem = adversary.knowledge.begin();
       kitem != adversary.knowledge.end(); kitem++) {
    out << "  bool " << kitem->first << ";" << endl;
  }
  out << "}" << endl << endl;

  // IDS
  out << "typedef HostIDS {" << endl;
  out << "  bool detected[NUMHOSTS];" << endl;
  out << "}" << endl << endl;

  // Network
  out << "typedef Network {" << endl;
  if ( DoesModelHaveFirewalls() )
    out << "  Firewall firewalls[NUMFIREWALLS];    /* firewall-specific data*/" << endl;   
  out << "  Host hosts[NUMHOSTS];                /* host-specific data */" << endl;
  out << "  Intruder intruder;" << endl;
  out << "  HostIDS ids[NUMHOSTS];" << endl;
  out << "" << endl;
  out << "  byte attack_id;" << endl;
  out << "  byte source_ip;" << endl;
  
  if ( DoesModelHaveFirewalls() )
  {
    out << "  int target_ip;" << endl;
    out << "  int target_fwi;" << endl;
  }
  else
    out << "  byte target_ip;" << endl;  
  out << "  bool attack_flavor;" << endl;
  out << "" << endl;
  out << "  bool alarm;" << endl;
  out << "}" << endl;
  out << endl;
}

void Network::GenerateMainProcessSPIN(ostream& out)
{
  BoolVectorContainer fconnectivity;
  
  out << "Network net;" << endl;
  out << "" << endl;
  out << "init" << endl;
  out << "{" << endl;
  if ( DoesModelHaveFirewalls() ) {
    if ( nHosts <= 256 )
      out << "  byte ";
    else
      out << "  int ";
    out << "hi = 0, hj = 0;" << endl;
    if ( fport2name.size() <= 256 )
      out << "  byte ";
    else
      out << "  int ";
    out << "pi = 0, pj = 0;" << endl;
    if ( max_path_count <= 256 )
      out << "  byte ";
    else
      out << "  int ";
    out << "pathi = 0, pathcount = 0;" << endl<< endl;
  }
  
  out << "  /* Initialization */" << endl;
  if ( DoesModelHaveFirewalls() ) {
    out << "  atomic {" << endl;
    out << "    net.target_ip = -1;" << endl;
    out << "    net.target_fwi = -1;" << endl;
  }
  else {
    out << "  d_step {" << endl;
  }
  
  out << "    net.alarm = false;" << endl;
  out << "" << endl;
  
  if ( DoesModelHaveFirewalls() ) {
    
    out << "    // Initial firewalls configuration" << endl;
    
    NameFirewallMap::iterator f_it = firewalls.begin();
    
    for ( ; f_it != firewalls.end(); f_it++ ) {
      
      out << "    net.firewalls[" << f_it->second->GetId() << "].on = " 
          << BoolStrLC(f_it->second->GetStatus()) << ";" << endl;
      
      FirewallRuleActionMap *rules = f_it->second->GetRules();
      FirewallRuleActionMap::iterator fr_it = rules->begin();
      for ( ;fr_it != rules->end(); fr_it++ )
        if ( fr_it->second )        
          GenerateFWRuleSPIN( out, fr_it->first, true, f_it->second->GetId() ); 
      
    }
    
    out << endl;
    HostID h_id1;
    HostID h_id2;
    unsigned int i = 0;
    FirewallID   f_id;
    unsigned int path_count;
    
    for ( h_id1 = 0; h_id1 < nHosts; h_id1++ )
      for ( h_id2 = 0; h_id2 < nHosts; h_id2++ ) {
          
        path_count = pathes_firewalls[h_id1][h_id2].size();
        
        out << "    // " << path_count << " paths between " 
            << GetHost( h_id1 )->GetName() << " and " << GetHost( h_id2 )->GetName() << endl;
        
        for ( i = 0; i < path_count; i++ )
          for ( f_id = 0; f_id < fid2name.size(); f_id++ ) {
            
            if ( pathes_firewalls[h_id1][h_id2][i][f_id] ) 
              out << "    net.hosts[" << h_id1 << "].path[" << h_id2 << "]._[" 
                  << i << "].firewalls[" << f_id << "] = true;" << endl;
          }
        if ( path_count > 0 )
          out << endl;
      }
      
    out << endl;
      
  }
  
  
  out << "    // Services and Vulnerabilities" << endl;

  for (HostID id = 0; id < nHosts; id++) {
    out << endl;
    out << "    // Initialize host " << hosts[id]->GetName() << endl;

    // Host services
    for (map<DOMString, ServiceID, less_str>::iterator service = sid.begin();
          service != sid.end(); service++) {
/*      if ( DoesModelHaveFirewalls() ) {
        if ( hosts[id]->ServiceRunning(service->first) )
          out << "    net.hosts[" << id << "].running_"
            << service->first << " = true;" << endl;
      }
      else
*/        out << "    net.hosts[" << id << "].running_"
            << service->first << " = "
            << BoolStrLC(hosts[id]->ServiceRunning(service->first)) << ";" << endl;
    }
    // Host vulnerabilities
    for (DOMStringSet::iterator vul = vulnerabilities.begin();
          vul != vulnerabilities.end(); vul++) {
/*      if ( DoesModelHaveFirewalls() ) {
        if ( hosts[id]->VulExists(*vul) ) 
          out << "    net.hosts[" << id << "]."
            << *vul << " = true;" << endl;
      }
      else 
*/        out << "    net.hosts[" << id << "]."
            << *vul << " = "
            << BoolStrLC(hosts[id]->VulExists(*vul)) << ";" << endl;
    }
  }

  if ( !DoesModelHaveFirewalls() ) {
    // Connectivity
    out << endl << "    // Connectivity" << endl;
    for (HostID src = 0; src < nHosts; src++) {
      for (HostID tgt = 0; tgt < nHosts; tgt++) {
        for (ServiceID svc = 0; svc < nServices+1; svc++) {
          out << "    net.hosts[" << src << "].conn["
              << tgt << "]." << GetServiceName(svc) << " = "
              << BoolStrLC(hosts[src]->Connected(hosts[tgt]->GetName(), sid2name[svc])) << ";" << endl;
              }
            }
            out << endl;
    }
  }
  
  // Trust
  out << "    // Trust" << endl;
  for (HostID src = 0; src < nHosts; src++) {
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      out << "    net.hosts[" << src << "].trust["
            << tgt << "] = "
            << BoolStrLC(hosts[tgt]->Trusts(hosts[src]->GetName())) << ";" << endl;
    }
    out << endl;
  }
  
  //Host Objects
  if ( !host_objects.empty() ) {
    out << "    // Objects" << endl;
    
    for (HostID hId = 0; hId < nHosts; hId++) {
      HostObjectMap &objs = hosts[hId]->GetHostObjects( );
      HostObjectMap::const_iterator p;
      DOMStringMap::const_iterator q;
      
      for ( HostObjectAttrNameMap::const_iterator oi = host_objects.begin(); oi != host_objects.end(); oi++) {
      
        if ( (p = objs.find( oi->first )) == objs.end() )      
          out << "    net.hosts[" << hId << "].objects." << oi->first << ".exists = FALSE;" << endl;
        else
          out << "    net.hosts[" << hId << "].objects." << oi->first << ".exists = TRUE;" << endl;
        
        out << "    net.hosts[" << hId << "].objects." << oi->first << ".source_host = " << hId << ";" << endl;
        out << "    net.hosts[" << hId << "].objects." << oi->first << ".service = SERVICE_none;" << endl;
      
        for ( DOMStringSet::const_iterator oai = oi->second.begin(); oai != oi->second.end(); oai++ ) {
          
          if ( p != objs.end() && (q = p->second.find( *oai )) != p->second.end() )
            out << "    net.hosts[" << hId << "].objects." << p->first << "." << q->first << " = " 
                << q->second  << ";"<< endl;
          else
            out << "    net.hosts[" << hId << "].objects." << oi->first << "." << *oai << " = " 
                << "0" << ";"<< endl;
        }
      }
    }
    
    out << endl;
  }
  // Intruder privilege levels
  out << "    // Intruder Privilege Levels" << endl;
  for (HostID hid = 0; hid < nHosts; hid++) {
    DOMString hname = hosts[hid]->GetName();
    out << "    net.intruder.privilege[" << hid << "] = "
	<< adversary.GetPrivilege(hname) << ";" << endl;
  }

  // Knowledge initialization
  out << endl << "    // Intruder Knowledge" << endl;
  for (map<DOMString, bool, less_str>::const_iterator kitem = adversary.knowledge.begin();
       kitem != adversary.knowledge.end(); kitem++) {
/*    if ( DoesModelHaveFirewalls() ) {
      if ( kitem->second )
        out << "    net.intruder." << kitem->first << " = true;" << endl;
    }
    else
*/      out << "    net.intruder." << kitem->first << " = " << BoolStrLC(kitem->second) << ";" << endl;
  }
  
  // Intrusion Detection System
  out << endl << "    // Intrusion Detection System" << endl;
  for (HostID src = 0; src < nHosts; src++) {
    DOMString sname = hosts[src]->GetName();
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      DOMString tname = hosts[tgt]->GetName();
/*      if ( DoesModelHaveFirewalls() ) {
        if ( ids.Monitors(sname, tname) )
          out << "    net.ids[" << src << "].detected["
            << tgt << "] = true;" << endl;
      }
      else 
*/        out << "    net.ids[" << src << "].detected["
            << tgt << "] = "
            << BoolStrLC(ids.Monitors(sname, tname)) << ";" << endl;
      
    }
  }
  
  if ( DoesModelHaveFirewalls() ) {
    out << endl << "    // Compute initial hosts' reachability" << endl;
    out << "    goto COMPUTE_FW_REACHABILITY;" << endl;
  }

  out << "  }" << endl << endl;

  // Transitions
  if ( DoesModelHaveFirewalls() )
    out << " MAIN_LOOP:" << endl;
  
  out << "  do" << endl;
  for (AttackID id = 0; id < nAttacks; id++) {
    DOMString aname = attacks[id]->GetName();
    if (attacks[id]->IsLocal()) {
      for (HostID tgt = 0; tgt < nHosts; tgt++) {
        out << "  :: " << aname << "_enabled("
            << tgt << ") -> execute_" << aname << "(" << tgt << ");" << endl;
      }
    }
    else if ( attacks[id]->IsAttackOnFirewall() ) {
      NameFirewallMap::iterator f_it = firewalls.begin();
      for ( ; f_it != firewalls.end(); f_it++ ) {
        
        out << "  :: " << aname << "_enabled(" << f_it->second->GetFirstLocationHost() << ","
            << f_it->second->GetId() << ") -> execute_" << aname << "(" 
            << f_it->second->GetFirstLocationHost() << ","
            << f_it->second->GetId() << ");" << endl;
        
        out << "  :: " << aname << "_enabled(" << f_it->second->GetSecondLocationHost() << ","
            << f_it->second->GetId() << ") -> execute_" << aname << "(" 
            << f_it->second->GetSecondLocationHost() << ","
            << f_it->second->GetId() << ");" << endl;
      }
    }
    else {
      for (HostID src = 0; src < nHosts; src++) {
        for (HostID tgt = 0; tgt < nHosts; tgt++) {
          out << "  :: " << aname << "_enabled(" << src << ","
              << tgt << ") -> execute_" << aname << "(" << src << "," << tgt << ")";
               
          out << ";" << endl;
        }
      }
    }
    if (id != nAttacks-1)
      out << endl;
  }
  // out << "  :: no_attacks_enabled -> skip;" << endl;
  out << "  :: else -> skip;" << endl;
  out << "  od;" << endl;
  
  GenerateFWReachabilityComputationSPIN( out );
  
  out << "}" << endl << endl;
}

void Network::GenerateEffectsSPIN(ostream& out, vector <Effect *>& effects,
			     const char *varname, HostID tgt, const char *entityname)
{
}

void Network::GenerateNeverClaimSPIN(ostream& out, string& target)
{
  if ( !ltl_formula.GenerateSPIN( out ) ) {
    DOMString tgtname = DOMString::transcode(target.c_str());
    int tgtid = 0;
    if (hid.find(tgtname) != hid.end()) {
      tgtid = hid[tgtname];
    }
  
    // Define the security invariant
    out << "#define security_inv (net.intruder.privilege[" << tgtid << "] < root)" << endl << endl;
  
    out << "never {    /* ![] security_inv  */" << endl;
    out << "T0_init:" << endl;
    out << "	if" << endl;
    out << "	:: (! ((security_inv))) -> stop" << endl;
    out << "	:: (((security_inv))) -> goto T0_init" << endl;
    out << "	fi;" << endl;
//    out << "accept_all:" << endl;
//    out << "	do" << endl;
//    out << "	:: (1)" << endl;
//    out << "	od;" << endl;
    out << "}" << endl;
  }
}

void Network::GenerateFWReachabilityComputationSPIN( ostream& out ) 
{
  if ( DoesModelHaveFirewalls() ) {
    out << endl << "  // Skip the computation if the atomic is reached in the usual way" << endl;
    out << "  goto END_FW_REACHABILITY;" << endl << endl;
    out << "  // Re-computation of host reachability array" << endl;
    out << "  atomic { " << endl;
    out << "    skip; " << endl;
    out << " COMPUTE_FW_REACHABILITY:" << endl;
        
    HostID h_id1;
    HostID h_id2;
    unsigned int path_count;
    unsigned int i = 0;
    FirewallID   f_id;
/*    
    for ( h_id1 = 0; h_id1 < nHosts; h_id1++ )
      for ( h_id2 = 0; h_id2 < nHosts; h_id2++ ) {
        path_count = pathes_firewalls[h_id1][h_id2].size();
        
        if ( path_count == 0 ) {
          out << endl << "    //Host " << GetHost( h_id2 )->GetName() << " is unreached from "
              <<  GetHost( h_id1 )->GetName() << endl << endl;
          for ( p_id1 = 0; p_id1 < fport2name.size(); p_id1++ )
            for ( p_id2 = 0; p_id2 < fport2name.size(); p_id2++ ){
              out << "    net.hosts[" << h_id1 << "].fw_reachable[" << h_id2 
                << "].srcport[" << FirewallRule::GetPortConstantName(fport2name[p_id1]) 
                << "].dstport[" << FirewallRule::GetPortConstantName(fport2name[p_id2]) 
                << "] = " << "false;" << endl;
            }
        }
        else {
          out << "    // " << GetHost( h_id2 )->GetName() << " -> " <<  GetHost( h_id1 )->GetName() << endl;
          for ( p_id1 = 0; p_id1 < fport2name.size(); p_id1++ )
            for ( p_id2 = 0; p_id2 < fport2name.size(); p_id2++ ){
              out << "    net.hosts[" << h_id1 << "].fw_reachable[" << h_id2 
                << "].srcport[" << FirewallRule::GetPortConstantName(fport2name[p_id1]) 
                << "].dstport[" << FirewallRule::GetPortConstantName(fport2name[p_id2]) 
                << "] = " << endl;
            
              for ( i = 0; i < path_count; i++ ) { 
                out << "        ";
                
                out << (( i == 0 )? "(": " ");
                                  
                out << "(";
                for ( f_id = 0; f_id < fid2name.size(); f_id++ ) {
                  
                  out << "(!net.hosts[" << h_id1 << "].path[" << h_id2 << "]._[" << i 
                      << "].firewalls["  << f_id << "] || !net.firewalls["  << f_id << "].on || "
                      << "net.firewalls[" << f_id << "].rule.src[" << h_id1 << "]."
                      << "srcport[" << FirewallRule::GetPortConstantName( fport2name[p_id1] ) << "]."
                      << "dst[" << h_id2 << "].dstport[" << FirewallRule::GetPortConstantName( fport2name[p_id2] ) << "])";
                  
                  if ( f_id < fid2name.size() - 1 )
                    out << " && " << endl;
                                   
                }
                
                out << ")";
                
                if ( i < path_count - 1 )
                  out << " || " << endl;
              }
              out << ");" << endl;
            }
        }
      }

*/
    //out << endl << "    // " << GetHost( h_id2 )->GetName() << " -> " <<  GetHost( h_id1 )->GetName() << endl << endl;
    out << "    hi = 0; " << endl;
    out << "    do" << endl;
    out << "    :: (hi < NUMHOSTS);" << endl;
    out << "       hj = 0;" << endl;
    out << "       do" << endl;
    out << "       :: (hj < NUMHOSTS);" << endl;
    out << "          pi = 0;" << endl;
    out << "          pathcount = " << endl;    
    for ( h_id1 = 0; h_id1 < nHosts; h_id1++ )
      for ( h_id2 = 0; h_id2 < nHosts; h_id2++ ) {
        path_count = pathes_firewalls[h_id1][h_id2].size();
        if ( path_count > 0 ) {
          i++;
          out << "                ((hi == " << h_id1 << " && hj == " << h_id2 << ") -> "
              << path_count << ": ";
              
          if ( !(h_id1 == (nHosts - 1) && h_id2 == (nHosts - 1)) )
            out << "" << endl;
        }
    }
    
    out << "0";
                  
    for ( ; i > 0; i-- )
      out << ")";
    
    out << ";" << endl;
        
    out << "          do" << endl;
    out << "          :: (pi < NUMPORTS);" << endl;
    out << "             pj = 0; " << endl;                        
    out << "             do" << endl;
    out << "             :: (pj < NUMPORTS);" << endl;
    out << "                pathi = 0;" << endl;
    out << "                net.hosts[hi].fw_reachable[hj].srcport[pi].dstport[pj] = false;" << endl;
    out << "                do" << endl;
    out << "                :: (pathi < pathcount && !net.hosts[hi].fw_reachable[hj].srcport[pi].dstport[pj]);" << endl;
    out << "                   net.hosts[hi].fw_reachable[hj].srcport[pi].dstport[pj] = " << endl;
    
    for ( f_id = 0; f_id < fid2name.size(); f_id++ ) {
      out << "                       ";
      if ( f_id == 0 )            
        out << "(";
      else
        out << " ";
      out << "(!net.hosts[hi].path[hj]._[pathi].firewalls["  << f_id << "] || !net.firewalls["  << f_id << "].on || "
          << "net.firewalls[" << f_id << "].rule.src[hi].srcport[pi].dst[hj].dstport[pj])";
                 
      if ( f_id < fid2name.size() - 1 )
        out << " && " << endl;
    }
                
    out << ");" << endl;
    out << "                   pathi++;" << endl;
    out << "                :: else; break;" << endl;
    out << "                od;" << endl;
    out << "                pj++;" << endl;
    out << "             :: else; break;" << endl;
    out << "             od;" << endl;
    out << "             pi++;" << endl;
    out << "          :: else; break;" << endl;
    out << "          od;" << endl;
    out << "          hj++;" << endl;
    out << "       :: else; break;" << endl;
    out << "       od;" << endl;
    out << "       hi++;" << endl;
    out << "    :: else; break;" << endl;
    out << "    od;" << endl;
        
    out << endl << "    hi = 0; hj = 0; pi = 0; pj = 0; pathi = 0; pathcount = 0;" << endl;

    out << endl << "    goto MAIN_LOOP;" << endl << endl;
    out << "  }" << endl << endl;
    out << " END_FW_REACHABILITY:" << endl;
    out << "  skip;" << endl;
  }
}

void Network::GenerateFWExplicitRuleSPIN( ostream& out, HostID srchost, PortID srcport, HostID dsthost, PortID dstport, 
                                           bool action, int firewall )
{
  out << "    net.firewalls[";
  if ( firewall >= 0 )
    out << firewall;
  else
    out << "target_fw";      
  out << "].rule.src[" << srchost  
      << "].srcport[" << FirewallRule::GetPortConstantName( fport2name[srcport] ) 
      << "].dst[" << dsthost
      << "].dstport[" << FirewallRule::GetPortConstantName( fport2name[dstport] ) << "] = " 
      << BoolStrLC(action) << ";" << ((firewall < 0)? " \\": "" ) << endl;
}

void Network::GenerateFWRuleSPIN( ostream& out, const FirewallRule &rule, bool action, int firewall )
{
  HostID h_id1;
  HostID h_id2;
  PortID p_id1;
  PortID p_id2;
  bool wsrchost = rule.srchost < 0;
  bool wsrcport = rule.srcport < 0;
  bool wdsthost = rule.dsthost < 0;
  bool wdstport = rule.dstport < 0;
  
  if ( !wsrchost && !wsrcport && !wdsthost && !wdstport ) {
    GenerateFWExplicitRuleSPIN(out, rule.srchost, rule.srcport, rule.dsthost, rule.dstport, action, firewall);
  }
  else if ( wsrchost && wsrcport && wdsthost && wdstport ) {
    if ( (nHosts * nHosts * fport2name.size() * fport2name.size()) <= 40 ) {
      for ( h_id1 = 0; h_id1 < nHosts; h_id1++ )
        for ( h_id2 = 0; h_id2 < nHosts; h_id2++ )
          for ( p_id1 = 0; p_id1 < fport2name.size(); p_id1++ )
            for ( p_id2 = 0; p_id2 < fport2name.size(); p_id2++ ) {
              GenerateFWExplicitRuleSPIN(out, h_id1, p_id1, h_id2, p_id2, action, firewall);
            }
    }
    else {
      out << "    do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: (hi < NUMHOSTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       hj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: (hj < NUMHOSTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          pi = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          :: (pi < NUMPORTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "             pj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "             do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "             :: (pj < NUMPORTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "                net.firewalls[";
      if ( firewall >= 0 )
        out << firewall;
      else
        out << "target_fw";
      out << "].rule.src[hi].srcport[pi].dst[hj].dstport[pj] = " << BoolStrLC(action) << ";"
          << ((firewall < 0)? " \\": "" ) << endl;
      out << "                pj++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "             :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "             od; pi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          od; hj++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       od; hi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    od;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    hi = 0; hj = 0; pi = 0; pj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
    }
  }
  else if (  wsrcport && wdstport && (wdsthost || wsrchost) ) {
    if ( (nHosts * fport2name.size() * fport2name.size()) <= 25 ) {
      for (h_id1 = 0; h_id1 < nHosts; h_id1++ )
        for (p_id1 = 0; p_id1 < fport2name.size(); p_id1++ )
          for (p_id2 = 0; p_id2 < fport2name.size(); p_id2++ ) {
            if ( wdsthost )
              GenerateFWExplicitRuleSPIN(out, rule.srchost, p_id1, h_id1, p_id2, action, firewall);
            else
              GenerateFWExplicitRuleSPIN(out, h_id1, p_id1, rule.dsthost, p_id2, action, firewall);
          }
    }
    else {
      out << "    do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: (hi < NUMHOSTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       pi = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: (pi < NUMPORTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          pj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          :: (pj < NUMPORTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "             net.firewalls[";
      
      if ( firewall >= 0 )
        out << firewall;
      else
        out << "target_fw";
      out << "].rule.src[";
      
      if (wsrchost) 
        out << "hi";
      else
        out << rule.srchost;
      
      out << "].srcport[pi].dst[";
      
      if (wdsthost)
        out << "hi";
      else
        out << rule.dsthost;
      
      out << "].dstport[pj] = " << BoolStrLC(action) << ";" << ((firewall < 0)? " \\": "" ) << endl;
      
      out << "             pj++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          od; pi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       od; hi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    od;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    hi = 0; pi = 0; pj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
    }
  }
  else if (  (wsrcport || wdstport) && wdsthost && wsrchost ) {
    if ( (nHosts * nHosts * fport2name.size()) <= 25 ) {
      for (h_id1 = 0; h_id1 < nHosts; h_id1++ )
        for (h_id2 = 0; h_id2 < nHosts; h_id2++ )
          for (p_id1 = 0; p_id1 < fport2name.size(); p_id1++ ) {
            if ( wdstport )
              GenerateFWExplicitRuleSPIN(out, h_id1, rule.srcport, h_id1, p_id2, action, firewall);
            else
              GenerateFWExplicitRuleSPIN(out, h_id1, p_id1, h_id1, rule.dstport, action, firewall);
          }
    }
    else {
      out << "    do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: (hi < NUMHOSTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       hj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: (hj < NUMHOSTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          pi = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          :: (pi < NUMPORTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "             net.firewalls[";
      if ( firewall >= 0 )
        out << firewall;
      else
        out << "target_fw";
      
      out << "].rule.src[hi].srcport[";
      if (wsrcport)
        out << "pi";
      else
        out << FirewallRule::GetPortConstantName( fport2name[rule.srcport] );
      out <<"].dst[hj].dstport[";
      
      if (wdstport)
        out << "pi";
      else
        out << FirewallRule::GetPortConstantName( fport2name[rule.dstport] );
      out << "] = " << BoolStrLC(action) << ";" << ((firewall < 0)? " \\": "" ) << endl;
      
      out << "             pi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          od; hj++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       od; hi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    od;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    hi = 0; hj = 0; pi = 0;" << ((firewall < 0)? " \\": "" ) << endl;
    }
  }
  else if ( wdsthost && wsrchost ) {
    if ( nHosts <= 4 ) {
      for (h_id1 = 0; h_id1 < nHosts; h_id1++ )
        for (h_id2 = 0; h_id2 < nHosts; h_id2++ )
           GenerateFWExplicitRuleSPIN(out, h_id1, rule.srcport, h_id1, rule.dstport, action, firewall);
    }
    else {
      out << "    do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: (hi < NUMHOSTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       hj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: (hj < NUMHOSTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "             net.firewalls[";
      if ( firewall >= 0 )
        out << firewall;
      else
        out << "target_fw";
      
      out << "].rule.src[hi].srcport[" << FirewallRule::GetPortConstantName( fport2name[rule.srcport] ) 
          << "].dst[hj].dstport[" << FirewallRule::GetPortConstantName( fport2name[rule.dstport] )
          << "] = " << BoolStrLC(action) << ";" << ((firewall < 0)? " \\": "" ) << endl;
      
      out << "          hj++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       od; hi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    od;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    hi = 0; hj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
    }
  }
  else if ( wsrcport && wdstport ) {
    if ( fport2name.size() <= 4 ) {
      for (p_id1 = 0; p_id1 < fport2name.size(); p_id1++ )
        for (p_id2 = 0; p_id2 < fport2name.size(); p_id2++ )        
          GenerateFWExplicitRuleSPIN(out, rule.srchost, p_id1, rule.dsthost, p_id2, action, firewall);
    }
    else {
      
      out << "    do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: (pi < NUMPORTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       pj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: (pj < NUMPORTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          net.firewalls[";
      if ( firewall >= 0 )
        out << firewall;
      else
        out << "target_fw";
      out << "].rule.src[" << rule.srchost << "].srcport[pi].dst[" << rule.dsthost 
          << "].dstport[pj] = " << BoolStrLC(action) << ";" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          pj++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       od; pi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    od; " << ((firewall < 0)? " \\": "" ) << endl;
      out << "    pi = 0; pj = 0;" << ((firewall < 0)? " \\": "" ) << endl;
    }
  }
  else if ( (wsrcport || wdstport) && (wdsthost || wsrchost) ) {
    if ( (fport2name.size() * nHosts) <= 20 ) {
      for (h_id1 = 0; h_id1 < nHosts; h_id1++ )
        for (p_id1 = 0; p_id1 < fport2name.size(); p_id1++ ) {
          if ( wsrcport ) {
            if ( wsrchost )
              GenerateFWExplicitRuleSPIN(out, h_id1, p_id1, rule.dsthost, rule.dstport, action, firewall);
            else
              GenerateFWExplicitRuleSPIN(out, rule.srchost, p_id1, h_id1, rule.dstport, action, firewall);
          }
          else {
            if ( wsrchost )
              GenerateFWExplicitRuleSPIN(out, h_id1, rule.srcport, rule.dsthost, p_id1, action, firewall);
            else
              GenerateFWExplicitRuleSPIN(out, rule.srchost, rule.srcport, h_id1, p_id1, action, firewall);
          }
        }
    }
    else {
      out << "    do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: (hi < NUMHOSTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       pi = 0;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: (pi < NUMPORTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          net.firewalls[";
      if ( firewall >= 0 )
        out << firewall;
      else
        out << "target_fw";
      out << "].rule.src[";
      if (wsrchost)
        out << "hi";
      else
        out << rule.srchost;
      out << "].srcport[";
      if (wsrcport)
        out << "pi";
      else
        out << FirewallRule::GetPortConstantName( fport2name[rule.srcport] );
      out <<"].dst[";
      if (wdsthost)
        out << "hi";
      else
        out << rule.dsthost;
      out << "].dstport[";
      
      if (wdstport)
        out << "pi";
      else
        out << FirewallRule::GetPortConstantName( fport2name[rule.dstport] );
      
      out << "] = " << BoolStrLC(action) << ";" << ((firewall < 0)? " \\": "" ) << endl;
      out << "          pi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       od; hi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    od;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    hi = 0; pi = 0;" << ((firewall < 0)? " \\": "" ) << endl;
    }
  }
  else if ( wdsthost || wsrchost ) {
    if ( nHosts <= 9 ) {
      for (h_id1 = 0; h_id1 < nHosts; h_id1++ ) {
        if ( wsrchost )
          GenerateFWExplicitRuleSPIN(out, h_id1, rule.srcport, rule.dsthost, rule.dstport, action, firewall);
        else
          GenerateFWExplicitRuleSPIN(out, h_id1, rule.srcport, rule.dsthost, rule.dstport, action, firewall);
      }
    }
    else {
      out << "    do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: (hi < NUMHOSTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       net.firewalls[";
      
      if ( firewall >= 0 )
        out << firewall;
      else
        out << "target_fw";
      out << "].rule.src[";
      
      if (wsrchost) 
        out << "hi";
      else
        out << rule.srchost;
      
      out << "].srcport[" << FirewallRule::GetPortConstantName( fport2name[rule.srcport] ) << "].dst[";
      
      if (wdsthost)
        out << "hi";
      else
        out << rule.dsthost;
      
      out << "].dstport[" << FirewallRule::GetPortConstantName( fport2name[rule.dstport] ) 
          << "] = " << BoolStrLC(action) << ";" << ((firewall < 0)? " \\": "" ) << endl;
      
      out << "       hi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    od;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    hi = 0;" << ((firewall < 0)? " \\": "" ) << endl;
    }
  }
  else {
    if ( fport2name.size() <= 7 ) {
      for (p_id1 = 0; p_id1 < fport2name.size(); p_id1++ ) {
        if ( wsrcport )
          GenerateFWExplicitRuleSPIN(out, rule.srchost, p_id1, rule.dsthost, rule.dstport, action, firewall);
        else
          GenerateFWExplicitRuleSPIN(out, rule.srchost, rule.srcport, rule.dsthost, p_id1, action, firewall);
      }
    }
    else {
      out << "    do" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: (pi < NUMPORTS);" << ((firewall < 0)? " \\": "" ) << endl;
      out << "       net.firewalls[";
      if ( firewall >= 0 )
        out << firewall;
      else
        out << "target_fw";
      
      out << "].rule.src[" << rule.srchost << "].srcport[";
      if (wsrcport)
        out << "pi";
      else
        out << FirewallRule::GetPortConstantName( fport2name[rule.srcport] );
      out <<"].dst[" << rule.dsthost << "].dstport[";
      
      if (wdstport)
        out << "pi";
      else
        out << FirewallRule::GetPortConstantName( fport2name[rule.dstport] );
      out << "] = " << BoolStrLC(action) << ";" << ((firewall < 0)? " \\": "" ) << endl;
      
      out << "       pi++;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    :: else; break;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    od;" << ((firewall < 0)? " \\": "" ) << endl;
      out << "    pi = 0;" << ((firewall < 0)? " \\": "" ) << endl;
    }
  }
}

