
// $Log: generate.cpp,v $
// Revision 1.6  2004/04/15 19:06:54  alex
// Support of firewalls
//
// Revision 1.5  2004/03/04 23:42:04  alex
// *** empty log message ***
//
// Revision 1.4  2003/11/27 21:58:50  alex
// Host objects: implementation of smv generation
//
// Revision 1.1.1.1  2003/10/28 22:15:44  rl
// Initial import
//
// Revision 1.3  2003/04/03 12:52:01  RL
// full xml_comp version
//
// Revision 1.2  2001/05/21 02:59:25  oleg
// Finished preconditions processing
//
// Revision 1.1  2001/05/21 01:51:34  oleg
// Initial revision
//

#include <stdlib.h>

#include "Network.hpp"

inline const char *BoolStr(bool val)
{
  return (val ? "TRUE" : "FALSE");
}

void PrivilegePrecondition::Print(ostream& out) const
{
  out << "adversary.privilege[" << host << "] " << rel
      << " priv." << value;
}

void KnowledgePrecondition::Print(ostream& out) const
{
  out << "adversary." << name << " = " << value;
}

void SPAPrecondition::Print(ostream& out) const
{
  out << "net." << service << "_port = adversary." << service << "_port";
}

void VulnerabilityPrecondition::Print(ostream& out) const
{
  out << "hosts[" << host << "]." << name;
}

void ServicePrecondition::Print(ostream& out) const
{
  out << "hosts[" << host << "].running_" << name;
}

void ConnectivityPrecondition::Print(ostream& out) const
{
  const char *source = (src == SOURCE) ? "source" : "target";
  const char *target = (src == SOURCE) ? "target" : "source";
  out << "conn[" << source << "][" << target << "][" << id << "]";
}

void TrustPrecondition::Print(ostream& out) const
{
  out << "trust[source][target]";
}

void NoTrustPrecondition::Print(ostream& out) const
{
  out << "(!(";

  for (HostID id = 0; id < network->GetNumHosts(); id++) {
    if (id != 0)
      out << " & ";
    out << "trust[" << id << "][target]";
  }
  
  out << "))";
}

void ObjectPrecondition::Print(ostream& out) const
{
  string host = (m_host == SOURCE) ? "source" : "target";
  out << "((host[" << host << "].objects." << m_obj_name << ".exists = TRUE)";
  
  for ( DOMStringMap::const_iterator p = m_obj_attributes.begin(); p != m_obj_attributes.end(); p++ ) {
    out << " & ";
    out << "(hosts[" << host << "].objects." << m_obj_name << "." << p->first << " = objdef." << p->second << " )";
  }
  out << ")";
}

void SentObjectPrecondition::Print(ostream& out) const
{
  string dsthost = (m_dsthost == SOURCE) ? "source" : "target";
  
  out << "((hosts[" << dsthost << "].objects." << m_obj_name << ".exists = TRUE) & ";
  
  if ( m_set_source ) {
    string srchost = (m_srchost == SOURCE) ? "source" : "target";
    out << "(hosts[" << dsthost << "].objects." << m_obj_name << ".source_host = " << srchost << ") & ";
  }
  
  out << "(hosts[" << dsthost << "].objects." << m_obj_name << ".service = objdef.SERVICE_" << m_service << "))";
}

void FirewallReachablePrecondition::Print(ostream& out) const
{
  string dsthost = (m_dsthost == SOURCE) ? "source" : "target";
  string srchost = (m_srchost == SOURCE) ? "source" : "target";
  
  out << "net.hosts[" << srchost << "].fw_reachable[" << dsthost << "].srcport[" 
      << FirewallRule::GetPortConstantName( m_srcport ) << "].dstport["
      << FirewallRule::GetPortConstantName( m_dstport ) << "]";
}

void FirewallRulePrecondition::Print( ostream& out ) const
{
  if ( !m_result )
    out << "!";
  out << "net.firewalls[target_fw].rule.src[" << m_srchost  
      << "].srcport[" << FirewallRule::GetPortConstantName( m_network->GetFirewallPortName(m_srcport) ) 
      << "].dst[" << m_dsthost
      << "].dstport[" << FirewallRule::GetPortConstantName( m_network->GetFirewallPortName(m_dstport) ) << "]";
}

void OrPrecondition::Print(ostream& out) const
{
  out << "(";

  bool first = true;
  for (ListPrecond::const_iterator disjunct = children.begin(); disjunct != children.end(); disjunct++) {
    if (!first) {
      out << " | ";
    }
    else
      first = false;
    (*disjunct)->Print(out);
  }
  out << ")";
}

void AndPrecondition::Print(ostream& out) const
{
  out << "(";

  bool first = true;
  for (ListPrecond::const_iterator conjunct = children.begin(); conjunct != children.end(); conjunct++) {
    if (!first) {
      out << " & ";
    }
    else
      first = false;
    (*conjunct)->Print(out);
  }
  out << ")";
}

void NotPrecondition::Print(ostream& out) const
{
  assert(cond != NULL);
  
  out << "(!(";
  cond->Print(out);
  out << ")";
}

KnowledgePrecondition::~KnowledgePrecondition()
{
}

OrPrecondition::~OrPrecondition()
{
}

AndPrecondition::~AndPrecondition()
{
}

NotPrecondition::~NotPrecondition()
{
}

void Effect::Print(ostream& out) const
{
  switch (type) {
  case PRIVILEGE:
    out << "privilege";
    break;
  case SERVICE:
    out << "service";
    break;
  case VULNERABILITY:
    out << "vulnerability";
    break;
  case CONNECTIVITY:
    out << "connectivity";
    break;
  case TRUST:
    out << "trust";
    break;
  }

  if (!host.equals(""))
    out << " host=" << host;
  if (!name.equals(""))
    out << " name=" << name;
  if (!value.equals(""))
    out << " value=" << value;

  out << " attack id=" << attackid;
}

void SendObjectEffect::Print(ostream& out) const
{
  out << "sendobject";
  out << " srchost=" << GetSrcHost();
  out << " dsthost=" << GetDstHost();
  out << " dstsvc=" << GetDstService();
  out << " object=" << GetObjName();
  out << " attack id=" << attackid;
}

void PassArgEffect::Print(ostream& out) const
{
  out << "passarg";
  out << " service=" << GetService();
  out << " object=" << GetObjName();
  out << " attack id=" << attackid;
}

void FirewallActionEffect::Print(ostream& out) const
{
  out << "firewall rule";
  
  if ( m_fstatus_mode )
    out << " status=" << ((m_result)? "On": "Off");
  else {
    out << " firewall=target_fw";
    out << " srchost=";
    if ( m_rule.srchost >= 0)
      out << m_rule.srchost;
    else
       out << "*";
    out << " srcport=" << ((m_rule.srcport < 0)? "*": FirewallRule::GetPortConstantName( m_network->GetFirewallPortName( m_rule.srcport ) ));    
    out << " dsthost=";
    if ( m_rule.dsthost >= 0)
      out << m_rule.dsthost;
    else
       out << "*";
    out << " dstport=" << ((m_rule.dstport < 0)? "*": FirewallRule::GetPortConstantName( m_network->GetFirewallPortName( m_rule.dstport ) ));    
    out << " result=" << ((m_result)? "allow": "deny");
  }
  out << " attack_id=" << attackid;
}

void Network::GenerateObjAttrValueDefinesSMV( ostream& out )
{
typedef set<string> AttrValueStringSet;
typedef set<int> AttrValueIntSet;
  
  if ( !obj_attr_values.empty() ) {
    AttrValueIntSet digital_values;
    AttrValueStringSet named_values;
    string      value;
    
    int ivalue = 0;
    
    for ( DOMStringSet::const_iterator iter = obj_attr_values.begin(); iter != obj_attr_values.end(); iter++ ) {
      value = iter->transcode();
      
      if ( isdigit( value[0] ) ) {
        sscanf( value.c_str(), "%d", &ivalue );
        digital_values.insert( ivalue );
      }
      else      
        named_values.insert( value );
    }
    
    // the unique value
    ivalue = 512;
    for ( AttrValueStringSet::const_iterator p = named_values.begin(); p != named_values.end(); p++, ivalue++ ) {
      
      while ( digital_values.find(ivalue) != digital_values.end() ) 
        ivalue++;
      
      out << "  " << *p << " := " << ivalue << ";" << endl;
    }
    
    out << endl;
  }
}

// Generators to SMV
void Network::GenerateSMV(ostream& out, string& target)
{
  out
    << "MODULE main" << endl
    << "" << endl
    << "VAR" << endl
    << "  network: network;" << endl
    << "" << endl
    << "ASSIGN" << endl
    << "" << endl
    << "SPEC" << endl
    ;

  DOMString tgtname = DOMString::transcode(target.c_str());
  int tgtid = 0;
  if (hid.find(tgtname) == hid.end()) {
    cerr << endl << "--Warning--: target " << tgtname << " not found, using default target '"
	 << hosts[0]->GetName() << "'" << endl;
  }
  else
    tgtid = hid[tgtname];
  
  out
    << "  AG (network.adversary.privilege[" << tgtid << "] < network.priv.root)"
    << endl
    << "" << endl
    << "" << endl
    << "-----------------------" << endl
    << "-- Network and Hosts --" << endl
    << "-----------------------" << endl
    << "" << endl
    << "-- Description of the network" << endl
    << "" << endl
    << "MODULE network" << endl
    << "" << endl
    << "VAR" << endl
    << "  hosts: array 0.." << nHosts-1 << " of host;	-- Hosts" << endl
    << "" << endl
    << "  rel: host_relations;		-- Connectivity and Rsh trust" << endl
    << "  priv: privilege;		-- Privilege level definitions" << endl
    << "  adversary: adversary(priv);	-- The adversary state" << endl
    << "  ids: ids;			-- The intrusion detection system" << endl
    << "" << endl
    //    << "
    ;
    
  if (!host_objects.empty())
    out << "  objdef: hostobjectdef;    -- Host object definitions" << endl << endl;

  GenerateAttackDeclarationsSMV(out);

  out
    << "" << endl
    << "  attack_id: 0.." << nAttacks-1 << ";		-- Which attack is being tried" << endl
    << "  source_ip: 0.." << nHosts-1 << ";		-- Attack source ip" << endl
    << "  target_ip: 0.." << nHosts-1 << ";		-- Attack target ip" << endl
    << "  attack_flavor: boolean;	-- Describes whether attack flavor is detectable" << endl
    << "" << endl
    << "  detected: boolean;		-- Whether the attack(s) have been detected by the IDS" << endl
    << "" << endl
    << "ASSIGN" << endl
    //    << "
    ;

  GenerateHostInitializationsSMV(out);
  GenerateRelationInitializationsSMV(out);
  GenerateAttackIdsSMV(out);
  GenerateTransitionsSMV(out);
  GenerateDefinesSMV(out);
  GenerateInvarSMV(out);
  GenerateHostSMV(out);
  GenerateHostRelationsSMV(out);
  GenerateAdversarySMV(out);
  GenerateIDSSMV(out);
  GenerateAttacks(out);
  
  if (!host_objects.empty()) {
         
    HostObjectAttrNameMap::const_iterator obj;
    for (obj = host_objects.begin(); obj != host_objects.end(); obj++) {
      
      out
      << "" << endl
      << "----------------" << endl
      << "-- Host object " << obj->first << " --" << endl
      << "----------------" << endl
      << "" << endl
      << "MODULE hostobj_" << obj->first << endl
      << "" << endl
      << "VAR" << endl;
      
      out << "  exists: boolean;" << endl;
      out << "  source_host: 0.." << (GetNumHosts() - 1) << ";" << endl;
      out << "  service: 0.." << GetNumServices() << ";" << endl;
      for (DOMStringSet::const_iterator attr = obj->second.begin(); attr != obj->second.end(); attr++ ) {
        out << "  " << *attr << ": 0..65535;" << endl;
      }
    }
  
    out
      << "" << endl
      << "----------------" << endl
      << "-- Host objects --" << endl
      << "----------------" << endl
      << "" << endl
      << "MODULE hostobjects" << obj->first << endl
      << "" << endl
      << "VAR" << endl;
    
      for (obj = host_objects.begin(); obj != host_objects.end(); obj++) {
        out << "  " << obj->first << ": hostobj_" << obj->first << ";" << endl;
      }
    
  }
  
  out
    << "" << endl
    << "----------------" << endl
    << "-- Priviliges --" << endl
    << "----------------" << endl
    << "" << endl
    << "MODULE privilege" << endl
    << "" << endl
    << "DEFINE" << endl
    << "  none := 1;" << endl
    << "  user := 2;" << endl
    << "  root := 3;" << endl
    // << "  
    ;
    
  if (!host_objects.empty()) {
    out
      << "" << endl
      << "----------------" << endl
      << "-- Host object definitions --" << endl
      << "----------------" << endl
      << "" << endl
      << "MODULE hostobjectdef" << endl
      << "" << endl
      << "DEFINE" << endl;
      
    //generate object attribute values named constants
    GenerateObjAttrValueDefinesSMV( out );
   
    //generate object service names
    if ( !obj_attr_values.empty() ) {  
      
      int serv_id = 1;
      for ( ServiceID sid = 1; sid <= GetNumServices(); sid++, serv_id++) 
        out << "  SERVICE_" << GetServiceName(sid) << " := " << serv_id << ";" << endl;
      out << "  SERVICE_none := 0;" << endl;
      out << endl;
    }
  }
  
  // Add the privilege constants to the NetInfo object.
  auxilary->theNet.ignore.insert("network.priv.none");
  auxilary->theNet.ignore.insert("network.priv.user");
  auxilary->theNet.ignore.insert("network.priv.root");
  auxilary->theNet.numIVars += 3;
}

void Network::GenerateAttackDeclarationsSMV(ostream& out)
{
  char num[ 11 ];
  memset( num, 0, sizeof( num ) );
  
  for (AttackID id = 0; id < nAttacks; id++) {
    DOMString aname = attacks[id]->GetName();

    out << endl;
    out << "  -- " << attacks[id]->GetDescription() << endl;
    if (attacks[id]->IsLocal()) {
      for (HostID tgt = 0; tgt < nHosts; tgt++) {
        out << "  " << aname << "_" << tgt << ": " << aname << "(" << tgt
            << ", self, hosts, rel.rsh_trust, rel.conn, adversary, ids, priv" << (host_objects.empty()? "": ", objdef") << ");" << endl;
        
        // Add this local attack to the NetInfo object.
        string aEnabled( "network." );
        aEnabled = aEnabled + aname.transcode() + "_" + itoa(tgt,num) + ".";
          
        string aDetected = aEnabled;
        aEnabled += "enabled";
        aDetected += "detected";
        
        auxilary->theNet.ignore.insert( aEnabled );
        auxilary->theNet.ignore.insert( aDetected );
        auxilary->theNet.numIVars += 2;
      }
    }
    else {
      for (HostID src = 0; src < nHosts; src++) {
        for (HostID tgt = 0; tgt < nHosts; tgt++) {
          out << "  " << aname << "_" << src << "_" << tgt
              << ": " << aname << "(" << src << ", " << tgt
              << ", self, hosts, rel.rsh_trust, rel.conn, adversary, ids, priv" << (host_objects.empty()? "": ", objdef") << ");" << endl;
        
          // Add each host's enabled and detected fields to the NetInfo object.
          string aEnabled( aname.transcode() );
          aEnabled = "network." + aEnabled;
          aEnabled = aEnabled + "_" + itoa(src,num) + "_" + itoa(tgt,num)+ ".";
            
          string aDetected = aEnabled;
          aEnabled += "enabled";
          aDetected += "detected";
        
          auxilary->theNet.ignore.insert( aEnabled );
          auxilary->theNet.ignore.insert( aDetected );
          auxilary->theNet.numIVars += 2;
        }
      }
    }
  }
}

void Network::GenerateHostInitializationsSMV(ostream& out)
{
  char num[ 11 ];
  memset( num, 0, sizeof( num ) );
  
  out << "  -- Detection" << endl
      << "  init(detected) := FALSE;" << endl
      << endl;

  auxilary->theNet.vars.insert( mType(string("network.detected"),0) );
  auxilary->theNet.numVars++;

  for (HostID id = 0; id < nHosts; id++) {
    out << endl;
    out << "  -- Initialize host " << id << endl;

    // Host services
    for (map<DOMString, ServiceID, less_str>::iterator service = sid.begin();
          service != sid.end(); service++) {
      out << "  init(hosts[" << id << "].running_"
      << service->first << ") := "
      << BoolStr(hosts[id]->ServiceRunning(service->first)) << ";" << endl;
  
      // Add this service indicator to the NetInfo object.
      string running( "network." );
      running += "hosts[";
      running += itoa(id,num);
      running = running + "].running_" + service->first.transcode();
      auxilary->theNet.vars.insert( mType(running,
                (int)(hosts[id]->ServiceRunning(service->first) ) ) );
      auxilary->theNet.numVars++;
    }
    // Host vulnerabilities
    for (set <DOMString, less_str>::iterator vul = vulnerabilities.begin();
	        vul != vulnerabilities.end(); vul++) {
      out << "  init(hosts[" << id << "]."
	  << *vul << ") := "
	  << BoolStr(hosts[id]->VulExists(*vul)) << ";" << endl;

      // Add this vulnerability to the NetInfo object.
      string vulner( "network." );
      vulner = vulner + "hosts[" + itoa(id,num) + "]." + (*vul).transcode();
      auxilary->theNet.vars.insert( mType(vulner,(int)(hosts[id]->VulExists(*vul))) );
      auxilary->theNet.numVars++;
    }
    // Host id
    out << "  init(hosts[" << id << "].ip) := " << id << ";" << endl;
    
    //Host Objects
    if ( !host_objects.empty() ) {
            
        HostObjectMap &objs = hosts[id]->GetHostObjects( );
        HostObjectMap::const_iterator p;
        DOMStringMap::const_iterator q;
        
        for ( HostObjectAttrNameMap::const_iterator oi = host_objects.begin(); oi != host_objects.end(); oi++) {
        
          if ( (p = objs.find( oi->first )) == objs.end() )      
            out << "  init(hosts[" << id << "].objects." << oi->first << ".exists) := FALSE;" << endl;
          else
            out << "  init(hosts[" << id << "].objects." << oi->first << ".exists) := TRUE;" << endl;
          
          out << "  init(hosts[" << id << "].objects." << oi->first << ".source_host) := " << id << ";" << endl;
          out << "  init(hosts[" << id << "].objects." << oi->first << ".service) := objdef.SERVICE_none;" << endl;
        
          for ( DOMStringSet::const_iterator oai = oi->second.begin(); oai != oi->second.end(); oai++ ) {
            
            if ( p != objs.end() && (q = p->second.find( *oai )) != p->second.end() )
              out << "  init(hosts[" << id << "].objects." << p->first << "." << q->first << ") := objdef." 
                  << q->second  << ";"<< endl;
            else
              out << "  init(hosts[" << id << "].objects." << oi->first << "." << *oai << ") := " 
                  << "0" << ";"<< endl;
          }
        }
            
      out << endl;
    }

    // Add this to the list as well.
    string ip( "network.hosts[" );
    ip = ip + itoa(id,num) + "].ip";
    auxilary->theNet.vars.insert( mType(ip,id) );
    auxilary->theNet.numVars++;
  }
}

void Network::GenerateRelationInitializationsSMV(ostream& out)
{
  char num[ 11 ];
  memset( num, 0, sizeof( num ) );
  
  out << endl << "  -- Initialize connectivity relation" << endl;

  //  cout << sizeof(auxilary->theNet) << endl;
  for (HostID src = 0; src < nHosts; src++) {
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      for (ServiceID svc = 0; svc < nServices+1; svc++) {
	out << "  init(rel.conn[" << src << "]["
	    << tgt << "][" << svc << "]) := "
	    << BoolStr(hosts[src]->Connected(hosts[tgt]->GetName(), sid2name[svc])) << ";" << endl;

	// Add this particular relation to the list of state vars.
	string relconn( "network.rel.conn[" );
	relconn = relconn + itoa(src,num) + "][" + itoa(tgt,num) + "][" +
	  itoa(svc,num) + "]";
	auxilary->theNet.vars.insert( mType(relconn,
	 (int)(hosts[src]->Connected(hosts[tgt]->GetName(), sid2name[svc]))) );
	auxilary->theNet.numVars++;
      }
    }
  }

  out << endl << "  -- Initialize trust relation" << endl;
  for (HostID src = 0; src < nHosts; src++) {
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      out << "  init(rel.rsh_trust[" << src << "]["
	  << tgt << "]) := "
	  << BoolStr(hosts[tgt]->Trusts(hosts[src]->GetName())) << ";" << endl;

      // Add the trust relation to the list of variables.
      string trust( "network.rel.rsh_trust[" );
      trust = trust + itoa(src,num) + "][" + itoa(tgt,num) + "]";
      auxilary->theNet.vars.insert( mType(trust,
			   (int)(hosts[tgt]->Trusts(hosts[src]->GetName()))) );
      auxilary->theNet.numVars++;
    }
  }
}

void Network::GenerateAttackIdsSMV(ostream& out)
{
  out
    << "" << endl
    << "  -- Selection of attack, source, target, and flavor" << endl
    << "  init(attack_id) := 0.." << nAttacks-1 << ";" << endl
    << "  init(source_ip) := " << endl
    << "    case" << endl
    << "      -- When the attack is local, source doesn't matter, so" << endl
    << "      -- make it deterministic to reduce graph size" << endl
    ;

  // These are non-deterministically chosen.
  for( int i = 0; i < nAttacks; i++ )
    auxilary->theNet.vars.insert( mType(string("network.attack_id"),i) );
  auxilary->theNet.numVars += nAttacks;

  for( int i = 0; i < nHosts; i++ )
    auxilary->theNet.vars.insert( mType(string("network.source_ip"),i) );
  auxilary->theNet.numVars += nHosts;

  for (AttackID id = 0; id < nAttacks; id++) {
    if (attacks[id]->IsLocal()) {
      out << "      attack_id = " << id << ": target_ip;" << endl;
    }
  }

  out
    << "      -- Otherwise, nondeterministic choice" << endl
    << "      1: 0.." << nHosts-1 << ";" << endl
    << "    esac;" << endl
    << "  init(target_ip) := 0.." << nHosts-1 << ";" << endl
    << "  init(attack_flavor) :=" << endl
    << "    case" << endl
    ;

  for( int i = 0; i < nHosts; i++ )
    auxilary->theNet.vars.insert( mType(string("network.target_ip"),i) );
  auxilary->theNet.numVars += nHosts;

  // This can potentially be true or false.
  auxilary->theNet.vars.insert( mType(string("network.attack_flavor"),0) );
  auxilary->theNet.vars.insert( mType(string("network.attack_flavor"),1) );
  auxilary->theNet.numVars += 2;

  char detect_set[20];
  for (AttackID id = 0; id < nAttacks; id++) {
    switch (attacks[id]->IsDetectable()) {
    case Attack::Yes:
      sprintf(detect_set, "%s", BoolStr(true));
      break;
    case Attack::No:
      sprintf(detect_set, "%s", BoolStr(false));
      break;
    default:
      sprintf(detect_set, "{ %s, %s }", BoolStr(true), BoolStr(false));
      break;
    }
    out << "      attack_id = " << id << ": " << detect_set << ";" << endl;
  }

  out
    << "    esac;" << endl
    << "" << endl
    << "  next(attack_id) :=" << endl
    << "    case" << endl
    << "      -- When no attacks are enabled," << endl
    << "      -- make it deterministic to reduce graph size" << endl
    << "      !next(enabled_attack_exists): 0;" << endl
    << "      1: 0.." << nAttacks-1 << ";" << endl
    << "    esac;" << endl
    << "  next(source_ip) :=" << endl
    << "    case" << endl
    << "      -- When no attacks are enabled," << endl
    << "      -- make it deterministic to reduce graph size" << endl
    << "      !next(enabled_attack_exists): 1;" << endl
    << "      -- When the attack is local, source doesn't matter, so" << endl
    << "      -- make it deterministic to reduce graph size" << endl
    ;

  for (AttackID id = 0; id < nAttacks; id++) {
    if (attacks[id]->IsLocal()) {
      out << "      next(attack_id) = " << id << ": next(target_ip);" << endl;
    }
  }

  out
    << "      -- Otherwise, nondeterministic choice" << endl
    << "      1: 0.." << nHosts-1 << ";" << endl
    << "    esac;" << endl
    << "  next(target_ip) :=" << endl
    << "    case" << endl
    << "      -- When no attacks are enabled," << endl
    << "      -- make it deterministic to reduce graph size" << endl
    << "      !next(enabled_attack_exists): 1;" << endl
    << "      1: 0.." << nHosts-1 << ";" << endl
    << "    esac;" << endl
    << "  next(attack_flavor) :=" << endl
    << "    case" << endl
    << "      -- When no attacks are enabled," << endl
    << "      -- make it deterministic to reduce graph size" << endl
    << "      !next(enabled_attack_exists): FALSE;" << endl
    ;
  
  for (AttackID id = 0; id < nAttacks; id++) {
    switch (attacks[id]->IsDetectable()) {
    case Attack::Yes:
      sprintf(detect_set, "%s", BoolStr(true));
      break;
    case Attack::No:
      sprintf(detect_set, "%s", BoolStr(false));
      break;
    default:
      sprintf(detect_set, "{ %s, %s }", BoolStr(true), BoolStr(false));
      break;
    }
    out << "      next(attack_id) = " << id << ": " << detect_set << ";" << endl;
  }

  out
    << "    esac;" << endl
    << "" << endl
    //    << "
    ;
}

void Network::GenerateTransitionsSMV(ostream& out)
{
  out
    << "  -------------------------------------" << endl
    << "  -- Transitions: results of attacks --" << endl
    //    << "
    ;

  char varname[50];
  // Connectivity transitions
  out << endl << "  -- Connectivity transitions" << endl;
  for (HostID src = 0; src < nHosts; src++) {
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      for (ServiceID svc = 0; svc < nServices+1; svc++) {
	char *svcname = sid2name[svc].transcode();
	sprintf(varname, "rel.conn[%d][%d][%d]", src, tgt, svc);
	GenerateTransitionSMV(out, conneff, varname, tgt, svcname);
	delete [] svcname;
      }
    }
  }

  out << endl << "  -- Trust transitions" << endl;
  for (HostID src = 0; src < nHosts; src++) {
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      sprintf(varname, "rel.rsh_trust[%d][%d]", src, tgt);
      GenerateTransitionSMV(out, trusteff, varname, tgt);
    }
  }  

  out << endl << "  -- Services" << endl;
  for (ServiceID svc = 1; svc < nServices+1; svc++) {
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      char *svcname = sid2name[svc].transcode();
      sprintf(varname, "hosts[%d].running_%s", tgt, svcname);
      out << endl;
      GenerateTransitionSMV(out, svceff, varname, tgt, svcname);
      delete [] svcname;
    }
  }

  out << endl << "  -- Vulnerabilities" << endl;
  for (set <DOMString, less_str>::iterator vul = vulnerabilities.begin();
       vul != vulnerabilities.end(); vul++) {
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      char *vulname = vul->transcode();
      sprintf(varname, "hosts[%d].%s", tgt, vulname);
      out << endl;
      GenerateTransitionSMV(out, vuleff, varname, tgt, vulname);
      delete [] vulname;
    }
  }

  out << endl << "  -- Adversary privileges" << endl;
  for (HostID tgt = 0; tgt < nHosts; tgt++) {
    sprintf(varname, "adversary.privilege[%d]", tgt);
    out << endl;
    GenerateTransitionSMV(out, priveff, varname, tgt);
  }  

  out << endl << "  -- Adversary knowledge" << endl;
  for (map<DOMString, bool, less_str>::const_iterator kitem = adversary.knowledge.begin();
       kitem != adversary.knowledge.end(); kitem++) {
    char *kname = kitem->first.transcode();
    sprintf(varname, "adversary.%s", kname);
    out << endl;
    GenerateTransitionSMV(out, knoweff, varname, GENERAL, kname);
    delete [] kname;
  }
  
  out << endl << "  -- Detection" << endl
      << "  next(detected) :=" << endl
      << "    detected" << endl;

  for (AttackID id = 0; id < nAttacks; id++) {
    out << "    -- Attack " << id << ": " << attacks[id]->GetName() << endl;
    if (attacks[id]->IsLocal()) {
      for (HostID tgt = 0; tgt < nHosts; tgt++) {
	out
	  << "    | (attack_id = " << id
	  << " & target_ip = " << tgt
	  << " & " << attacks[id]->GetName() << "_" << tgt
	  << ".detected)" << endl;
      }
    }
    else {
      for (HostID src = 0; src < nHosts; src++) {
	for (HostID tgt = 0; tgt < nHosts; tgt++) {
	  out
	    << "    | (attack_id = " << id
	    << " & source_ip = " << src
	    << " & target_ip = " << tgt
	    << " & " << attacks[id]->GetName() << "_" << src << "_" << tgt
	    << ".detected)" << endl;
	}
      }
    }
  }
  
  out << "    ;" << endl << endl;
  
  if ( !host_objects.empty() ) {
    out << endl << "  -- Host objects" << endl;
    vector <Effect *> all_obj_eff;
    vector <Effect *>::const_iterator iter;
    
    for ( iter = objeff.begin(); iter != objeff.end(); iter++ )
      all_obj_eff.push_back( *iter );
    
    for ( iter = passargeff.begin(); iter != passargeff.end(); iter++ ) 
      all_obj_eff.push_back( *iter );
    
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      HostObjectAttrNameMap::const_iterator obj;
      for (obj = host_objects.begin(); obj != host_objects.end(); obj++) {
        sprintf(varname, "hosts[%d].objects.%s.exists", tgt, obj->first.transcode() );
        out << endl;
        GenerateTransitionSMV(out, objeff, varname, tgt, obj->first.transcode(), "exists");
        
        sprintf(varname, "hosts[%d].objects.%s.source_host", tgt, obj->first.transcode());
        out << endl;
        GenerateTransitionSMV(out, objeff, varname, tgt, obj->first.transcode(), "source_host");
        
        sprintf(varname, "hosts[%d].objects.%s.service", tgt, obj->first.transcode());
        out << endl;
        GenerateTransitionSMV(out, all_obj_eff, varname, tgt, obj->first.transcode(), "service");
        
        for (DOMStringSet::const_iterator attr = obj->second.begin(); attr != obj->second.end(); attr++ ) {
          sprintf(varname, "hosts[%d].objects.%s.%s", tgt, obj->first.transcode(), attr->transcode() );
          out << endl;
          GenerateTransitionSMV(out, objeff, varname, tgt, obj->first.transcode(), attr->transcode() );
        }
      }
    }  
    out << endl;
  }
}

void Network::GenerateDefinesSMV(ostream& out)
{
  out
    << "DEFINE" << endl;

  for (map<DOMString, PortID, less_str>::iterator svc = ports.begin(); svc != ports.end(); svc++) {
    out << "  " << svc->first << "_port := " << svc->second << ";" << endl;

    // Add each of the service port definitions to the ignore list.
    auxilary->theNet.ignore.insert( string("network.")+svc->first.transcode()+"_port" );
    auxilary->theNet.numIVars++;
  }
  
  for (AttackID id = 0; id < nAttacks; id++) {
    out << endl << "  -- Attack " << id << ": " << attacks[id]->GetDescription() << endl;

    out << "  " << attacks[id]->GetName() << "_enabled :=" << endl;
    out << "    ( ";

    if (attacks[id]->IsLocal()) {
      for (HostID tgt = 0; tgt < nHosts; tgt++) {
	out << attacks[id]->GetName() << "_" << tgt << ".enabled" << endl;
	if (tgt != nHosts-1)
	  out << "    | ";
      }
    }
    else {
      for (HostID src = 0; src < nHosts; src++) {
	for (HostID tgt = 0; tgt < nHosts; tgt++) {
	  out << attacks[id]->GetName() << "_" << src << "_" << tgt << ".enabled" << endl;
	  if (!(src == nHosts-1 && tgt == nHosts-1))
	    out << "    | ";
	}
      }
    }
    out << "    );" << endl;
  }

  out << endl << "  enabled_attack_exists :=" << endl;

  for (AttackID id = 0; id < nAttacks; id++) {
    out << "    -- Attack " << id << ": " << attacks[id]->GetDescription() << endl;
    out << ((id == 0) ? "    ( " : "    | ");

    if (attacks[id]->IsLocal()) {
      for (HostID tgt = 0; tgt < nHosts; tgt++) {
	out << attacks[id]->GetName() << "_" << tgt << ".enabled" << endl;
	if (tgt != nHosts-1)
	  out << "    | ";
      }
    }
    else {
      for (HostID src = 0; src < nHosts; src++) {
	for (HostID tgt = 0; tgt < nHosts; tgt++) {
	  out << attacks[id]->GetName() << "_" << src << "_" << tgt << ".enabled" << endl;
	  if (!(src == nHosts-1 && tgt == nHosts-1))
	    out << "    | ";
	}
      }
    }
  }

  out << "    );" << endl;

  auxilary->theNet.ignore.insert( "network.enabled_attack_exists" );
  auxilary->theNet.numIVars++;
}

void Network::GenerateInvarSMV(ostream& out)
{
  out
    << "" << endl
    << "INVAR" << endl
    << "  -- The adversary should pick only enabled attacks.  This shrinks the state" << endl
    << "  -- space and scenario graph size." << endl
    ;

  for (AttackID id = 0; id < nAttacks; id++) {
    out << "  -- Attack " << id << ": " << attacks[id]->GetDescription() << endl;
    out << ((id == 0) ? "  ( " : "  | ");

    if (attacks[id]->IsLocal()) {
      for (HostID tgt = 0; tgt < nHosts; tgt++) {
	out
	  << "(attack_id = " << id
	  << " & target_ip = " << tgt << " & "
	  << attacks[id]->GetName() << "_" << tgt << ".enabled)" << endl;
	if (tgt != nHosts-1)
	  out << "  | ";
      }
    }
    else {
      for (HostID src = 0; src < nHosts; src++) {
	for (HostID tgt = 0; tgt < nHosts; tgt++) {
	  out
	    << "(attack_id = " << id << " & source_ip = " << src
	    << " & target_ip = " << tgt << " & "
	    << attacks[id]->GetName() << "_" << src << "_" << tgt << ".enabled)" << endl;
	  if (!(src == nHosts-1 && tgt == nHosts-1))
	    out << "  | ";
	}
      }
    }
  }

  out << "  )" << endl << "  | (!enabled_attack_exists)" << endl;
}

void Network::GenerateHostSMV(ostream& out)
{
  out
    << endl
    << "-- Host description" << endl
    << "" << endl
    << "MODULE host" << endl
    << "" << endl
    << "VAR" << endl
    ;

  for (ServiceID id = 1; id <= nServices; id++) {
    out << "  running_" << sid2name[id] << ": boolean;  -- Host running "
	<< sid2name[id] << "?" << endl;
  }

  out << endl;

  for (set <DOMString, less_str>::iterator vul = vulnerabilities.begin();
       vul != vulnerabilities.end(); vul++) {
    out << "  " << *vul << ": boolean;" << endl;
  }
  
  if (!host_objects.empty()) {
    out << endl;
    out << "  objects: hostobjects;";
  }
  
  out << endl << "  ip: 0.." << nHosts-1 << ";" << endl << endl;
  out << "ASSIGN" << endl << "  next(ip) := ip;" << endl;
}

void Network::GenerateHostRelationsSMV(ostream& out)
{
  out
    << "" << endl
    << "-- Binary host relations, described as NxN matrices." << endl
    << "-- These describe connectivity and trust; in principle, any inter-host" << endl
    << "-- properties go here." << endl
    << "" << endl
    << "MODULE host_relations" << endl
    << "" << endl
    << "VAR" << endl
    << "  conn:" << endl
    << "    array 0.." << nHosts-1
    << " of               -- NxN matrix with dimensions representing hosts" << endl
    << "      array 0.." << nHosts-1 << " of" << endl
    << "        array 0.." << nServices
    << " of boolean;  -- " << nServices+1
    << " connectivity flags for each pair (S,T) of hosts:" << endl
    << "                                -- 0. There is a path from S to T" << endl
    ;

  for (ServiceID id = 1; id <= nServices; id++) {
    out << "                                -- " << id
	<<". S can connect to T on " << sid2name[id] << " port" << endl;
  }

  out
    << "  rsh_trust:" << endl
    << "    array 0.." << nHosts-1
    << " of               -- NxN matrix with dimensions representing hosts" << endl
    << "      array 0.." << nHosts-1
    << " of boolean;    -- each pair (S,T) is true if T trusts S to rsh" << endl
    << "                                -- w/o authentication" << endl
    << "" << endl
    ;
}

void Network::GenerateAdversarySMV(ostream& out)
{
  char num[ 11 ];
  memset( num, 0, sizeof( num ) );
  
  out
    << "---------------" << endl
    << "-- Adversary --" << endl
    << "---------------" << endl
    << "" << endl
    << "MODULE adversary (priv)" << endl
    << "" << endl
    << "VAR" << endl
    << "  privilege: array 0.." << nHosts-1
    << " of 1..3; -- Adversary privilege level on each host" << endl;

  // Knowledge declaration
  for (map<DOMString, bool, less_str>::const_iterator kitem = adversary.knowledge.begin();
       kitem != adversary.knowledge.end(); kitem++) {
    out << "  " << kitem->first << ": boolean;" << endl;
  }
  
  out
    << "" << endl
    << "ASSIGN" << endl;

  for (HostID id = 0; id < nHosts; id++) {
    DOMString hname = hosts[id]->GetName();
    out << "  init(privilege[" << id << "]) := priv."
	<< adversary.GetPrivilege(hname) << ";" << endl;

    // Add the adversary's privileges to the list of state variables.
    string priv( "network.adversary.privilege[" );
    priv = priv + itoa(id,num) + "]";
    auxilary->theNet.vars.insert( mType(priv,priv2int(adversary.GetPrivilege(hname))) );
    auxilary->theNet.numVars++;
  }

  // Knowledge initialization
  for (map<DOMString, bool, less_str>::const_iterator kitem = adversary.knowledge.begin();
       kitem != adversary.knowledge.end(); kitem++) {
    out << "  init(" << kitem->first << ") := " << BoolStr(kitem->second) << ";" << endl;
  }
  
  out
    << "" << endl
    << "DEFINE" << endl
    ;

  for (ServiceID id = 1; id <= nServices; id++) {
    DOMString sname = sid2name[id];
    out << "  " << sname << "_port := " << adversary.GetPort(sname) << ";" << endl;

    // The service ports can be ignored as well.
    string port( "network.adversary." );
    port = port + sname.transcode() + "_port";
    auxilary->theNet.ignore.insert( port );
    auxilary->theNet.numIVars++;
  }
}

void Network::GenerateIDSSMV(ostream& out)
{
  char num[ 11 ];
  memset( num, 0, sizeof( num ) );
  
  out
    << "  " << endl
    << "---------" << endl
    << "-- IDS --" << endl
    << "---------" << endl
    << "" << endl
    << "MODULE ids" << endl
    << "" << endl
    << "VAR" << endl
    << "  detect:" << endl
    << "    array 0.." << nHosts-1
    << " of              -- NxN matrix with dimensions representing hosts" << endl
    << "      array 0.." << nHosts-1
    << " of boolean;   -- each pair (S,T) is true if there is an IDS" << endl
    << "                               -- detecting activity on the path from S to T." << endl
    << "                               -- Entries (T,T) designate host-based IDS components."
    << endl
    << "" << endl
    << "ASSIGN" << endl
    ;

  for (HostID src = 0; src < nHosts; src++) {
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      DOMString host1 = hosts[src]->GetName();
      DOMString host2 = hosts[tgt]->GetName();
      out << "  init(detect[" << src << "][" << tgt << "]) := "
	  << ((ids.Monitors(host1, host2)) ? "TRUE" : "FALSE") << ";" << endl;

      // Add the IDS paths to the NetInfo object.
      string idsPath( "network.ids.detect[" );
      idsPath = idsPath + itoa(src,num) + "][" + itoa(tgt,num) + "]";
      auxilary->theNet.vars.insert( mType(idsPath,ids.Monitors(host1,host2)) );
      auxilary->theNet.numVars++;
    }
  }
  out << endl;
  for (HostID src = 0; src < nHosts; src++) {
    for (HostID tgt = 0; tgt < nHosts; tgt++) {
      out << "  next(detect[" << src << "][" << tgt << "]) := "
	  << "detect[" << src << "][" << tgt << "];" << endl;
    }
  }
  out << endl;
}

void Network::GenerateAttacks(ostream& out)
{
  out
    << "" << endl
    << "-------------" << endl
    << "-- Attacks --" << endl
    << "-------------" << endl
    << "" << endl;

  for (AttackID id = 0; id < nAttacks; id++) {
    out << "-- " << attacks[id]->GetDescription() << endl << endl;
    out << "MODULE " << attacks[id]->GetName();
    if (attacks[id]->IsLocal())
	out << " (target, net, hosts, trust, conn, adversary, ids, priv" << (host_objects.empty()? "": ", objdef") << ")" << endl;
    else
	out << " (source, target, net, hosts, trust, conn, adversary, ids, priv" << (host_objects.empty()? "": ", objdef") << ")" << endl;
    if (!attacks[id]->IsLocal())
      out << "-- source = source host ip" << endl;
    out
      << "-- target = target host ip" << endl
      << "-- net = network description" << endl
      << "-- conn = connectivity matrix" << endl
      << "-- priv = privilege level definitions" << endl
      << "" << endl
      << "DEFINE" << endl
      << "  enabled :=" << endl
      << "    -- Local preconditions" << endl;

    bool first = true;
    for (set <Precondition *>::iterator cond = attacks[id]->locals.begin();
	 cond != attacks[id]->locals.end(); cond++) {
      out << "    ";
      if (!first) {
	out << "& ";
      }
      else
	first = false;
      out << *cond << endl;
    }

    out
      << "    -- Global preconditions" << endl;

    for (set <Precondition *>::iterator cond = attacks[id]->globals.begin();
	 cond != attacks[id]->globals.end(); cond++) {
      out << "    ";
      if (!first) {
	out << "& ";
      }
      else
	first = false;
      out << *cond << endl;
    }
    out << "    ;" << endl;

    out << endl << "  detected := ";
    switch (attacks[id]->IsDetectable()) {
    case Attack::Yes:
      out << "ids.detect[" << ((attacks[id]->IsLocal()) ? "target" : "source")
	  << "][target];";
      break;
    case Attack::No:
      out << "FALSE;";
      break;
    case Attack::Varies:
      out << "ids.detect[" << ((attacks[id]->IsLocal()) ? "target" : "source")
	  << "][target] & net.attack_flavor = TRUE;";
      break;
    }
    out << endl << endl;
  }
}


void Network::GenerateTransitionSMV(ostream& out, vector <Effect *>& effects,
				    const char *varname, HostID tgt, const char *entityname, const char *subentityname)
{
  out
    << "  next(" << varname << ") :=";

  if (effects.empty()) {
    out << " " << varname << ";" << endl;
    return;
  }

  out
    << endl
    << "    case" << endl;

  for (vector <Effect *>::iterator effp = effects.begin(); effp != effects.end(); effp++) {
    Effect *eff = *effp;

    
    // If entityname is not empty, we must check if this effect applies to the
    // entity in question
    if (entityname != NULL)
      if (strcmp(entityname, eff->name.transcode()))
        continue;
    
    if (eff->type == Effect::PASSARG) 
      if (strcmp(subentityname, "service"))
        continue;        
    
    out << "      -- Attack " << eff->attackid << ", "
	<< attacks[eff->attackid]->GetDescription() << endl;
    
    // If this is a general attack, the target doesn't matter.  All that
    // matters is that the attack is enabled at some one place.
    if (tgt == GENERAL) {
      out << "      ((attack_id = " << eff->attackid << ") & "
	  << attacks[eff->attackid]->GetName() << "_enabled)";
    }
    else if (attacks[eff->attackid]->IsLocal()) {
      out << "      ((attack_id = " << eff->attackid << ") & (target_ip = " << tgt << ") &" << endl;
      out << "        " << attacks[eff->attackid]->GetName() << "_" << tgt << ".enabled)";
    }
    else {
      out << "      ((attack_id = " << eff->attackid << ") & (target_ip = " << tgt << ") &" << endl;
      out << "        (";
      for (HostID src = 0; src < nHosts; src++) {
	if (src != 0) { out << "         "; }
	out << "(source_ip = " << src << " & "
	    << attacks[eff->attackid]->GetName()
	    << "_" << src << "_" << tgt << ".enabled)";
	if (src != nHosts-1) { out << " |" << endl; }
      }
      out << "))";
    }
    
    out << endl << "       : ";
    
    switch (eff->type) {
    case Effect::PASSARG: {
        PassArgEffect *pa_eff = dynamic_cast<PassArgEffect *>(eff);
        out << "objdef.SERVICE_" << pa_eff->GetService() << ";" << endl;
      }
      break;
      
    case Effect::SENDOBJECT:
      if ( !strcmp( subentityname, "exists") )
        out << "TRUE;";
      else if ( !strcmp( subentityname, "source_host") )
        out << "source_ip;";
      else if ( !strcmp( subentityname, "service") ) {
        SendObjectEffect *so_eff = dynamic_cast<SendObjectEffect *>(eff);
        out << "objdef.SERVICE_" << so_eff->GetDstService() << ";";
      }
      else
        out << "hosts[source_ip].objects." << string( entityname ) << "." << string( subentityname ) << ";";
      out << endl;
      break;
      
    default:
      out <<  ((eff->type == Effect::PRIVILEGE) ? "priv." : "")
          << eff->value << ";" << endl;
    }
  }

  out
    << "      1: " << varname << ";" << endl
    << "    esac;" << endl;
}
