(* $Id: spine.ml,v 1.1.1.1 2003/10/28 22:15:44 rl Exp $ *)

(* Convertions between spine lists and bdds *)

(* Author: John D. Ramsdell *)

(* Copyright (C) 2001 The MITRE Corporation

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *)

open Bdd

open Bdd_utils

type spine = bool * path

(* Generate a bdd from a spine list. *)

let rec spine2bdd (negate, path) =
  let bdd = path2bdd path in
  if negate then
    complement bdd
  else
    bdd

let rec filter spines output input =
  match spines with
    [] -> output
  | (false, path) :: rest -> 
      let accept = conjoin input (path2bdd path) in
      let input = difference input accept in
      filter rest (disjoin output accept) input
  | (true, path) :: rest -> 
      let input = difference input (path2bdd path) in
      filter rest output input

(* let spines2bdd spines = filter spines falsehood truth *)

let spines2bdd spines =
  let rec loop spines a =
    match spines with
      [] -> a
   | (false, path) :: rest ->
       loop rest (disjoin a (path2bdd path))
   | (true, path) :: rest -> 
       loop rest (difference a (path2bdd path)) in
  loop (List.rev spines) falsehood

(* Generate a spine list from a bdd. *)

let debug = false

let debug_print(h, n_h, p_f, n_f, p_h) =
  print_string "h = ";
  print_bdd h;
  Format.print_newline();
  print_string "n_h = ";
  print_int n_h;
  print_string " n_f = ";
  print_int n_f;
  print_newline();
  print_newline()

(* This could be rewritten so as to produce the pair
   (strongest f, strongest (complement f)) in one pass. *)
let strongest f =
  let leaf (n, p) negate path =
    if negate then			(* Ignore antispines *)
      (n, p)				(* but see comment above *)
    else
      let n' = strength f path in
      if n' > n then
	(n', path)
      else
	(n, p) in
  each_path leaf (0, []) f

let bdd2spines g =
  let rec loop h a =
    if is_falsehood h then
      List.rev a
    else
      let (n_h, p_h) = strongest h in 
      let (n_f, p_f) = strongest (complement h) in
      if debug then 
	debug_print(h, n_h, p_f, n_f, p_h);
      if n_h >= n_f then
	loop (difference h (path2bdd p_h)) ((false, p_h) :: a)
      else
	loop (disjoin h (path2bdd p_f)) ((true, p_f) :: a) in
  loop g []
