(* $Id: main.ml,v 1.1.1.1 2003/10/28 22:15:44 rl Exp $ *)

(* The main structure for the frame propagation program *)

(* Author: John D. Ramsdell *)

(* Copyright (C) 2001 The MITRE Corporation

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *)

(* This program should report a version number somehow *)

(* This section is all for argument processing *)

(* Bdd defaults *)

let utn = ref 1
let uts = ref 509
let ctn = ref 1
let cts = ref 509

exception Bad_arg of int

let check_arg loc int =
  if int > 0 then
    loc := int
  else
    raise (Bad_arg int)

let default loc =
  " (default " ^ string_of_int !loc ^ ")"

(* file names *)

let input_file = ref "-"

let output_file = ref "-"

let anon_var = ref 0

let anon_fun str =
  match !anon_var with
    0 -> 
      input_file := str;
      anon_var := 1
  | 1 ->
      output_file := str;
      anon_var := 2
  | _ ->
      ()

(* set up Arg.parse *)

let arg_spec =
  [("-utn", Arg.Int(check_arg utn), 
    "int - unique table number" ^ default utn);
    ("-uts", Arg.Int(check_arg uts), 
     "int - unique table size" ^ default uts);
    ("-ctn", Arg.Int(check_arg ctn), 
     "int - computed table number" ^ default ctn);
    ("-cts", Arg.Int(check_arg cts), 
     "int - computed table size" ^ default cts);
    ("--", Arg.Rest(anon_fun), 
     "      - treat remaining args as file names")]

let usage_msg = "Usage: prop [options] [input [output]]";;

Arg.parse arg_spec anon_fun usage_msg;;

(* init bdd *)

Bdd.init(!utn, !uts, !ctn, !cts);;

let input =
  if !input_file = "-" then
    stdin
  else
    open_in !input_file

(* read input *)

let lexbuf = Lexing.from_channel input

let geom = Parser.main Lexer.token lexbuf

(* do the main calculation *)

let prop_geom = Prop.propagate_geometry geom;;

if !output_file <> "-" && !output_file <> !input_file then
  Format.set_formatter_out_channel (open_out !output_file);;

(* print the results *)

Graph.print_geometries prop_geom;;

exit 0
