(* $Id: bdd.ml,v 1.1.1.1 2003/10/28 22:15:44 rl Exp $ *)

(*  ****************************************
    reduced ordered binary decision diagrams
    as recommended in 
    Karl S. Brace/ Richard L. Rudell/ Randal E. Bryant
    "Efficient implementation of a BDD Package", 
    27th ACM/IEEE Design Automation Conference, 
    1990, pp. 40--45
    John D. Ramsdell -- December 1998
    *****************************************)

(* Copyright (C) 2001 The MITRE Corporation

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *)

type path = (bool * int) list (* See interface for description *)


type internal = Truth        (* Only sink *)
              | Node of
                  int        (* variable (a non-negative integer) *)
                * internal   (* consequent *)
                * internal   (* alternative *)
                * bool       (* complement alternative *)
		* int        (* hash of node *)
type bdd      = bool         (* complement internal bdd *)
              * internal

exception Constant_bdd

(* predicates *)

let is_same (c, i) (c', i') =
  c == c' && i == i'

let is_node =
  function
      (_, Node _) -> true
    | _           -> false

(* if c is true, the boolean function should be interpreted
   as the complement of what is represented by the internal node. *) 
let is_complement (c, _) = c

let is_truth =
  function
      (false, Truth) -> true
    | _              -> false

let is_falsehood =
  function
      (true, Truth) -> true
    | _             -> false

(* atomic bdd's *)

let truth = (false, Truth)

let falsehood = (true, Truth)

(* accessors of parts of a node *)

let variable = 
  function
      (_, Truth)                -> raise Constant_bdd
    | (_, Node (v, _, _, _, _)) -> v

let consequent = 
  function
      (_, Truth)                -> raise Constant_bdd
    | (_, Node (_, i, _, _, _)) -> (false, i)

let alternative = 
  function
      (_, Truth)                -> raise Constant_bdd
    | (_, Node (_, _, i, b, _)) -> (b, i)

(* unique table *)

let alpha = 65599
let truth_hash = 8123
let variable_hash = 8117
let complement_hash = 8111

let node_hash : internal -> int = 
  function
      Truth                -> truth_hash
    | Node (_, _, _, _, h) -> h

let unique_hash (v, t, e, c) =
  let h = if c then complement_hash else 0 in
  let h' = variable_hash * v + h in
  let h'' = truth_hash * h' + node_hash t in
  let h''' = alpha * h'' + node_hash e in
  abs h'''

let unique_equal (v, t, e, c) (v', t', e', c') =
  v == v' && t == t' && e == e' && c == c'

type unique_table_key = int * internal * internal * bool

module Unique_table =
  Hashtbl.Make(struct
    type t = unique_table_key
    let equal = unique_equal
    let hash = unique_hash
  end)

let var2index n v =
  (abs v) mod n  

let init_unique_table_size = 509

let init_unique_table(number, size) =
  Array.init number 
    (function i ->
      (Unique_table.create size : internal Unique_table.t))

let unique_table =
  ref (init_unique_table(1, init_unique_table_size))

let complement (b, i) = (not b, i)

(* make sure no duplicate nodes appear in a bdd using the unique table *)

let rec find_or_make_node : int * bdd * bdd -> bdd =
  function 
      (v, (false, t), (false, e)) when t == e -> (false, t)
    | (v, (false, t), (c, e)) -> 
	begin
	  let key = (v, t, e, c) in
	  let size = Array.length !unique_table in
	  let ht = !unique_table.(var2index size v) in
	  try
	    (false, Unique_table.find ht key)
	  with
	    Not_found ->
	      let h = unique_hash(v, t, e, c) in
	      let node = Node(v, t, e, c, h) in
	      begin
	      	Unique_table.add ht key node;
	      	(false, node)
	      end
	end
    | (v, (true, t), e) -> 
	complement (find_or_make_node(v, (false, t), complement e))

let identity v = find_or_make_node(v, truth, falsehood)

(* computed table *)

let computed_equal (i, t, e, c) (i', t', e', c') = 
  i == i' && t == t' && e == e' && c == c'

let computed_hash (i, t, e, c) =
  let h = if c then complement_hash else 0 in
  let h' = variable_hash * node_hash i + h in
  let h'' = truth_hash * h' + node_hash t in
  let h''' = alpha * h'' + node_hash e in
  abs h'''

type computed_table_key = internal * internal * internal * bool

module Computed_table =
  Hashtbl.Make(struct
    type t = computed_table_key
    let equal = computed_equal
    let hash = computed_hash
  end)

let init_computed_table_size = 509

let init_computed_table (number, size) =
  Array.init number 
    (function i ->
      (Computed_table.create size : bdd Computed_table.t))

let computed_table =
  ref (init_computed_table(1, init_computed_table_size))

(* the variable ordering is implemented here *)

let var : internal -> int = 
  function
      Truth                -> max_int (* a trick to make top_var simple *)
    | Node (v, _, _, _, _) -> v

let top_var (i, t, e) =
  min (var i) (min (var t) (var e))

(* used to find a canonical representation for the computed table *)

let ordered (i, i') =
  let v = var i in
  let v' = var i' in
  v < v' || (v == v' && (node_hash i < node_hash i'))

(* substitute a value for a variable *)

let restrict value var exp =
  match exp with
    (c', Node (v, t, e, c, _)) when v == var ->
      if value then
	(c', t)
      else
	(c <> c', e)
  | _ -> exp

(* if-then-else, the bdd composition operator *)

let rec ite : bdd * bdd * bdd -> bdd =
  function
      ((true, i), t, e) ->
	ite((false, i), e, t)
    | ((false, i), (false, t), (c, e)) ->
	ite'(i, t, e, c)
    | ((false, i), (true, t), (c, e)) ->
	complement (ite'(i, t, e, not c))
and ite' : internal * internal * internal * bool -> bdd =
  function
      (Truth, t, _, _) -> (false, t)
    | (i, Truth, Truth, true) -> (false, i)
    | (_, t, e, false) when t == e -> (false, t)
    | (i, t, e, c) when i == t ->
	ite'(i, Truth, e, c)
    | (i, t, e, c) when i == e ->
      	ite'(i, t, Truth, not c)
    | (i, t, Truth, true) when ordered(t, i) ->
	ite'(t, i, Truth, true)
    | (i, Truth, e, false) when ordered(e, i) ->
	ite'(e, Truth, i, false)
    | (i, t, e, c) ->
	let var = top_var(i, t, e) in
	let size = Array.length !computed_table in
	let ht = !computed_table.(var2index size var) in
	let key = (i, t, e, c) in
	try
	  Computed_table.find ht key
	with
	  Not_found ->
	    let con = ite(restrict true var (false, i),
			  restrict true var (false, t),
			  restrict true var (c, e)) in
	    let alt = ite(restrict false var (false, i),
			  restrict false var (false, t),
			  restrict false var (c, e)) in
	    let result = find_or_make_node(var, con, alt) in
	    begin			(* remember result *)
	      Computed_table.add ht key result;
	      result
	    end

(* traversals *)

let each_node_internal node value internal =
  let rec follow value internal =
    match internal with
      Truth -> value
    | Node (v, c, a, _, _) ->
	let value' = follow (node value v) c in
	follow value' a in
  follow value internal

let each_node node value (_, internal) =
  each_node_internal node value internal

let each_path leaf value (negate, internal) =
  let rec follow value negate path internal =
    match internal with
      Truth -> leaf value negate path
    | Node (v, c, a, n, _) ->
	let value' = follow value negate ((false, v) :: path) c in
	follow value' (negate <> n) ((true, v) :: path) a in
  follow value negate [] internal

(* metrics *)

let debug = false

let inc_node value neg negate internal =
  match internal with
    Truth -> 0
  | Node (_, Truth, Truth, true, _) ->
      if neg = negate then value + 1 else 0
  | Node (_, Truth, _, _, _) ->
      if neg && negate then value + 1 else 0
  | Node (_, _, Truth, n, _) ->
      if not neg && n <> negate then value + 1 else 0
  | _ -> 0

let debug_print(neg, var, negate, v, internal, value) =
  let print_bool = function 
      true -> print_string "t"
    | false -> print_string "f" in
  print_string "path = (";
  print_bool neg;
  print_string ", ";
  print_int var;
  print_string ") bdd = (";
  print_bool negate;
  print_string ", ";
  print_int v;
  print_string ") = inc_node = ";
  print_int (inc_node value neg negate internal);
  print_newline()
  
(* The expression (strength bdd path) computes the number of nodes
   that would be eliminated in a bdd that is represented as a tree by
   performing (difference bdd (path2bdd path)).  In other words, the
   calculation does not take shared nodes into account.
   *)

let strength (negate, internal) path =
  let rec follow value path negate internal =
    match path with
      [] -> each_node_internal (fun n _ -> n + 1) value internal
    | (neg, var) :: p ->
	match internal with
	  Truth -> value
	| Node (v, c, a, n, _) ->
	    if debug then 
	      debug_print(neg, var, negate, v, internal, value);
	    if v = var then
	      let value' = inc_node value neg negate internal in
	      let negate' = if neg then n <> negate else negate in
	      let internal' = if neg then a else c in
	      follow value' p negate' internal'
	    else if v < var then
	      let value' = follow (value + 1) path negate c in
	      follow value' path (n <> negate) a
	    else
	      follow 0 p negate internal in
  follow 0 (List.rev path) negate internal

(* initialize the bdd hash tables with the following parameters *)

let init (unique_table_number,
	  unique_table_size,
	  computed_table_number,
	  computed_table_size) =
  begin
    unique_table := init_unique_table(unique_table_number,
				      unique_table_size);
    computed_table := init_computed_table(computed_table_number,
					  computed_table_size)
  end
