(* $Id: actions.ml,v 1.1.1.1 2003/10/28 22:15:44 rl Exp $ *)

(* Convertions between action lists and spine lists *)

(* Author: John D. Ramsdell *)

(* Copyright (C) 2001 The MITRE Corporation

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *)

open Spine

let base_var = 0

type action =
    Accept of string * string
  | Reject of string * string

exception Bad_action

let string2bits s =
  let n = String.length s in
  let rec loop i l =
    if i >= n then
      l
    else
      let l' =
	match String.get s i with
	  '0' -> false :: false :: false :: false :: l
	| '1' ->  true :: false :: false :: false :: l
	| '2' -> false ::  true :: false :: false :: l
	| '3' ->  true ::  true :: false :: false :: l
	| '4' -> false :: false ::  true :: false :: l
	| '5' ->  true :: false ::  true :: false :: l
	| '6' -> false ::  true ::  true :: false :: l
	| '7' ->  true ::  true ::  true :: false :: l
	| '8' -> false :: false :: false ::  true :: l
	| '9' ->  true :: false :: false ::  true :: l
	| 'a' -> false ::  true :: false ::  true :: l
	| 'b' ->  true ::  true :: false ::  true :: l
	| 'c' -> false :: false ::  true ::  true :: l
	| 'd' ->  true :: false ::  true ::  true :: l
	| 'e' -> false ::  true ::  true ::  true :: l
	| 'f' ->  true ::  true ::  true ::  true :: l
	| _ -> raise Bad_action in
      loop (i + 1) l' in
  let rec lead i =
    if i >= n then
      []
    else
      match String.get s i with
	'0' -> lead (i + 1)
      | '1' -> loop (i + 1) [ true]
      | '2' -> loop (i + 1) [false;  true]
      | '3' -> loop (i + 1) [ true;  true]
      | '4' -> loop (i + 1) [false; false;  true]
      | '5' -> loop (i + 1) [ true; false;  true]
      | '6' -> loop (i + 1) [false;  true;  true]
      | '7' -> loop (i + 1) [ true;  true;  true]
      | '8' -> loop (i + 1) [false; false; false;  true]
      | '9' -> loop (i + 1) [ true; false; false;  true]
      | 'a' -> loop (i + 1) [false;  true; false;  true]
      | 'b' -> loop (i + 1) [ true;  true; false;  true]
      | 'c' -> loop (i + 1) [false; false;  true;  true]
      | 'd' -> loop (i + 1) [ true; false;  true;  true]
      | 'e' -> loop (i + 1) [false;  true;  true;  true]
      | 'f' -> loop (i + 1) [ true;  true;  true;  true]
      | _ -> raise Bad_action in
  lead 0

let bitmask2path bits mask =
  let rec loop path i = function
      (_, []) -> 
	path
    | ([], ms) -> 
	loop path i ([false], ms)
    | (b::bs, m::ms) ->
	let path' =
	  if m then
	    (not b, i) :: path
	  else
	    path in
	loop path' (i + 1) (bs, ms) in
  loop [] base_var (bits, mask)

let action2spine action =
  match action with
    Accept (bits, mask) -> 
      let b = string2bits bits in
      let m = string2bits mask in
      (false, bitmask2path b m)
  | Reject (bits, mask) -> 
      let b = string2bits bits in
      let m = string2bits mask in
      (true, bitmask2path b m)

let actions2spines actions =
  List.map action2spine actions

let path2bitmask path =
  let rec lead v bits mask =
    if v < base_var then
      (bits, mask)
    else
      lead (v - 1) (false :: bits) (false :: mask) in
  let rec loop path v bits mask =
    match path with
      [] -> lead v bits mask
    | (b, i) :: p ->
	if i = v then
	  loop p (v - 1) (not b :: bits) (true :: mask)
	else
	  loop path (v - 1) (false :: bits) (false :: mask) in
  match path with
    [] -> ([], [])
  | (b, i) :: p ->
      loop path i [] []

let get_nibble bits =
  let rec loop n a bits =
    match bits with
      [] -> (a, bits)
    | b :: bs ->
	let a' = 
	  if b then
	    a + n
	  else
	    a in
	if n >= 8 then
	  (a', bs)
	else
	  loop (2 * n) a' bs in
  loop 1 0 bits

let nibble2char n =
  match n with
    0 -> '0'
  | 1 -> '1'
  | 2 -> '2'
  | 3 -> '3'
  | 4 -> '4'
  | 5 -> '5'
  | 6 -> '6'
  | 7 -> '7'
  | 8 -> '8'
  | 9 -> '9'
  | 10 -> 'a'
  | 11 -> 'b'
  | 12 -> 'c'
  | 13 -> 'd'
  | 14 -> 'e'
  | 15 -> 'f'
  | _ -> raise Bad_action

let chars2string chars =
  let n = List.length chars in
  let s = String.create n in
  let rec loop i chars =
    match chars with
      [] -> s
    | c :: cs ->
	String.set s i c;
	loop (i + 1) cs in
  loop 0 chars    

let bits2string bits =
  let rec loop bits chars =
    if bits = [] then
      chars2string chars
    else
      let (n, bs) = get_nibble bits in
      loop bs (nibble2char n :: chars) in
  loop bits []

let spine2action spine =
  match spine with
    (false, path) -> 
      let (bits, mask) = path2bitmask path in
      Accept (bits2string bits, bits2string mask)
  | (true, path) -> 
      let (bits, mask) = path2bitmask path in
      Reject (bits2string bits, bits2string mask)

let spines2actions spines =
  List.map spine2action spines

(* A printer *)

let print_bitmask b m =
  Format.print_string "(\"";
  Format.print_string b;
  Format.print_string "\", \"";
  Format.print_string m;
  Format.print_string "\")"

let print_action = function
    Accept(b, m) ->
      Format.print_string "Accept";
      print_bitmask b m
  | Reject(b, m) ->
      Format.print_string "Reject";
      print_bitmask b m

let print_actions actions =
  let rec print_rest actions =
    match actions with
      [] -> ()
    | action :: rest -> 
	Format.print_string ",";
	Format.print_space();
	print_action action;
	print_rest rest in
  match actions with
    [] -> 
      Format.print_string "[]"
  | action :: rest ->
      Format.print_string "[";
      Format.open_box 0;
      print_action action;
      print_rest rest;
      Format.close_box();
      Format.print_string "]"
