
/*
 * $Log: Network.hpp,v $
 * Revision 1.10  2004/04/15 19:07:15  alex
 * Support of firewalls
 *
 * Revision 1.9  2004/03/04 23:42:15  alex
 * *** empty log message ***
 *
 * Revision 1.8  2004/02/17 00:39:53  alex
 * Calculating performance numbers
 *
 * Revision 1.6  2004/01/31 22:23:42  alex
 * *** empty log message ***
 *
 * Revision 1.2  2003/10/31 09:45:26  roman
 * *** empty log message ***
 *
 * Revision 1.1.1.1  2003/10/28 22:15:44  rl
 * Initial import
 *
 * Revision 1.3  2003/04/03 12:52:01  RL
 * full xml_comp version
 *
 * Revision 1.4  2001/05/21 02:59:44  oleg
 * Finished preconditions processing
 *
 * Revision 1.3  2001/05/14 22:02:46  oleg
 * Writing attack processing
 *
 * Revision 1.2  2001/05/12 23:49:57  oleg
 * Development checkpoint
 *
 * Revision 1.1  2001/05/12 03:34:35  oleg
 * Initial revision
 *
 */

#ifndef __NETWORK__HPP__
#define __NETWORK__HPP__

#include <iostream>
#include <set>
#include <map>
#include <vector>
#include <stdlib.h>
#include <xercesc/dom/deprecated/DOM.hpp>
#include <cstring>
#include <string>
#include <stdio.h>
#include <outpost.h>
#ifndef NO_ODBC
#include "odbc.hpp"
#endif

#include "StateInfo.hpp"
#include "precond.hpp"
//RL
#include "scanners.hpp"
#include "nessus_scanner.hpp"

// Element types
enum ElementTypes {
  NETWORK,
  SERVICES,
  SERVICE,
  FTP,
  SSHD,
  DATABASE,
  VULNERABILITIES,
  VULNERABILITY,
  HOST,
  CONNECTIVITY,
  CVE,
  CVE1,  
  CVEID,
  REMOTE,
  TRUST,
  NO_TRUST,
  ADVERSARY,
  PRIVILEGES,
  PRIVILEGE,
  ATTACK,
  LOCAL_PRECONDITIONS,
  GLOBAL_PRECONDITIONS,
  TRACE,
  LOCAL_EFFECTS,
  GLOBAl_EFFECTS,
  IDS,
  PATH,
  PORT,
  HOSTS,
  ATTACKS,
  ID,
  IP,
  EMPTY,
  LEVEL,
  HOST1,
  HOST2,
  DETECTABLE,
  MODE,
  YES,
  NO,
  XML_TRUE,
  XML_FALSE,
  BOTH,
  NAME,
  LOCAL,
  TARGET,
  REL,
  GTE,
  EQ,
  LTE,
  LS,
  GT,
  VALUE,
  NONE,
  USER,
  ROOT,
  SOURCE,
  FROM,
  SERVICE_PORT_AGREEMENT,
  KNOWLEDGE,
  DESCRIPTION,
  TEXT,
  OBJECTS,
  OBJECT,
  OBJECT_ATTRIBUTE,
  SENDOBJECT,
  SENTOBJECT,
  SRCHOST,
  DSTHOST,
  DSTSVC,
  PASSARG,
  LTL_FORMULA,
  FORMULA,
  LTL_ATOMIC_PROPOSITION,
  
  //Firewalls
  FIREWALL,
  STATUS,
  RESULT,
  RULE,
  SRCPORT,
  DSTPORT,
  FIREWALL_REACHABLE,
  ON,
  OFF,
  ALLOW,
  DENY,
  TARGET_FW,
  WILDCARD_ALL,
  
  // Data sources
  OUTPOST_DS,
  ODBC_DS,
  END_MARKER
};

struct AuxilaryParserData
{
  // Element names in DOMString format
  DOMString enames[END_MARKER];
  
  // Data sources
#ifdef OUTPOST
  DBApi Outpost;
#endif
#ifndef NO_ODBC
  ODBC Odbc;
#endif
  
  ScannerManager scan_mang; //Scanner manager 
  NetInfo theNet; // state info for the network
  
  void Initialize();
};

class Network;

class Host: public ModelMemUsage
{
public:
  enum DFS_FLAGS {
    NONE_FLAG = 0,
    IN_STACK_FLAG = 1,
    PASSED_FLAG = 2
  };

  // Constructors and Destructor
  Host( Network *net ) { 
    network = net; 
    name.transcode(""); 
    ip_addrs = NULL; 
    netname = ""; 
    defined = false; 
    dfs_flags = NONE_FLAG;
  }
  Host( Network *net_obj, DOMString hostname, HostID hostid, const char *net = "") {
    network = net_obj;
    name = hostname;
    id = hostid;
    ip_addrs = NULL;
    netname = net;
    defined = false;
    dfs_flags = NONE_FLAG;
  }  
  ~Host() { Clear(); }

  // Accessors
  DOMString GetName() { return name; }
  HostID GetId() { return id; }

  bool ServiceRunning(const DOMString& svc) {
    return (services.find(svc) != services.end());
  }
  bool VulExists(const DOMString& vul) {
    return (vulnerabilities.find(vul) != vulnerabilities.end());
  }
  bool Connected(const DOMString& host, const DOMString& svc) {
    if (connectivity.count(host) == 0) return false;
    return (connectivity[host]->find(svc) != connectivity[host]->end());
  }
  bool Trusts(const DOMString& host) {
    return (trust.find(host) != trust.end());
  }
  StringSet *IPs() { return ip_addrs; }
  DOMStringSet *GetVulnerabilities() { return &vulnerabilities; }
  const char *NetworkName() { return netname; }
  bool IsDefined() { return defined; }
  void SetDefined(bool def) { defined = def; }
  
  unsigned char GetFlags() { return dfs_flags; }
  void SetFlags( unsigned char flags, bool enable ) {
    if ( enable )
      dfs_flags |= flags;
    else
      dfs_flags &= ~(flags);
  }

  // Generators (from XML)
  void GenerateIPs(DOM_Attr &attr);
  void GenerateServices(DOM_Element *element);
  void GenerateConnectivity(DOM_Element *element);
  void GenerateTrust(DOM_Element *element);
  void GenerateVulnerabilities(DOM_Element *element);
  void GenerateCVE(DOM_Element *element);
  void GenerateObjects(DOM_Element *element);

  void CountingServices(DOM_Element *element);
  void CountingConnectivity(DOM_Element *element);
  void CountingVulnerabilities(DOM_Element *element);
  void CountingCVE(DOM_Element *element);
    
  void SetDocument(DOM_Document *doc) { curr_doc = doc; }

  void SetIP(const string &ip) { 
    if (ip_addrs == 0)
      ip_addrs = new StringSet();
    
    ip_addrs->insert(ip);
  }
  
  HostObjectMap& GetHostObjects( ) { return objects; }
  
  long GetMemSize() const;
  
private:
  // Private functionality
  void GenerateConnectivityServices(DOM_Element *element, DOMStringSet *svcs);
  void CountingConnectivityServices(DOM_Element *element);
  void GenerateObject(DOM_Element *element);

  void Clear() {
    delete ip_addrs;
    map <DOMString, DOMStringSet *, less_str>::iterator conn_it;
    for ( conn_it = connectivity.begin(); conn_it != connectivity.end(); conn_it++ )
      if ( conn_it->second != 0 )
        delete (conn_it->second);
      
    connectivity.clear();
  }
    
private:
  Network *network;
  DOMString name;
  HostID id;
  StringSet *ip_addrs;
  const char *netname;
  DOMStringSet services;
  map <DOMString, DOMStringSet *, less_str> connectivity;
  DOMStringSet trust;
  DOMStringSet vulnerabilities;
  HostObjectMap objects;
  bool defined;
  unsigned char dfs_flags;

  // Document related to the network 
  DOM_Document  *curr_doc;
};


class Attack: public ModelMemUsage {

  enum AttackType {
    A_GLOBAL = 1,
    A_LOCAL  = 2,
    A_FIREWALL = 3
  };
public:
  Attack( Network *net ) {
    network = net;
    name.transcode("");
    detectable = No;
    type = A_GLOBAL;
    defined = false;
  }
  Attack( Network *net, DOMString& attackname, AttackID attackid ) {
    network = net;
    name = attackname;
    id = attackid;
    detectable = No;
    type = A_GLOBAL;
    defined = false;
  }
  
  ~Attack() {
    ClearPreconditionSet(locals);
    ClearPreconditionSet(globals);
    ClearPreconditionSet(derived);
  }

  enum Detectable {
    Yes,
    No,
    Varies
  };

  // Accessors
  bool IsLocal() { return (type == A_LOCAL); }
  bool IsGlobal() { return (type == A_GLOBAL); }
  bool IsAttackOnFirewall() { return (type == A_FIREWALL); }
  Detectable IsDetectable() { return detectable; }
  void SetLocal() { type = A_LOCAL; }
  void SetGlobal() { type = A_GLOBAL; }
  void SetAttackOnFirewall() { type = A_FIREWALL; }
  DOMString GetName() { return name; }
  void SetDescription(DOMString& desc) { description = desc; }
  DOMString GetDescription() { return description; }
  void AddCVE(DOMString &cveid) { cveids.insert(cveid); }
  
  DOMStringSet* GetCveids() { return &cveids; }
  
  // Generators (from XML)
  void GenerateLocalPreconditions(DOM_Element *element);
  void GenerateGlobalPreconditions(DOM_Element *element);
  void GenerateTrace(DOM_Element *element);
  void GenerateLocalEffects(DOM_Element *element);
  void GenerateGlobalEffects(DOM_Element *element);
  void GenerateDetectable(DOM_Element *element);

  void CountingPreconditions(DOM_Element *element);
  void CountingEffects(DOM_Element *element);
  
  bool IsDefined() { return defined; }
  void SetDefined(bool def) { defined = def; }
  
  long GetMemSize() const;

private:
  void GeneratePreconditions(DOM_Element *element, set <Precondition *>& preconditions);
  void GenerateEffects(DOM_Element *element);
  long GetPreconditionsMemSize( const set <Precondition *> &precond) const;

  void ClearPreconditionSet( set <Precondition *> &precond ) {
    set <Precondition *>::iterator it = precond.begin();
    for ( ; it != precond.end(); it++ )
      if ( *it != 0 )
        delete *it;
      
    precond.clear();
  }
  
private:
  Network   *network;
  DOMString name;
  HostID id;
  DOMString description;

  Detectable detectable;
  unsigned char type;
  bool defined;

  set <Precondition *> locals;
  set <Precondition *> globals;
  set <Precondition *> derived;

  vector <Effect *> effects;

  DOMStringSet cveids;

  friend class Network;
};

class LTLFormula: public ModelMemUsage
{
public:
  LTLFormula( ){ m_exist = false; }
    
  void Clear() {
    LTLPropositionMap::iterator pit = m_atomic_precond.begin();
    for ( ; pit != m_atomic_precond.end(); pit++ )
      delete pit->second;
    m_atomic_precond.clear();
  }
  
  ~LTLFormula(){ Clear(); }
  
  void GenerateLTL( Network *network, DOM_Element *element );
  bool GenerateSPIN( ostream& out );
  
  long GetMemSize() const;
  
private:
  void GenerateAtomicProposition( Network *network, DOM_Element *element );

private:
  LTLPropositionMap m_atomic_precond;
  DOMString         m_formula;
  bool              m_exist;
};

struct FirewallRule_less: public binary_function <FirewallRule, FirewallRule, bool> {
  bool operator () ( const FirewallRule &rule1, const FirewallRule &rule2 ) {
    return rule1.less( rule2 );
  }
};
  
typedef map< FirewallRule, bool, FirewallRule_less > FirewallRuleActionMap;
    
class Firewall: public ModelMemUsage {
  
public:
  Firewall( Network *network, const DOMString &name, FirewallID id, bool init_status ) {
    m_network = network;
    m_status = init_status;
    m_name = name;
    m_id = id;
  }
  
  ~Firewall(){ m_rules.clear(); }
  void SetLocation ( HostID host1, HostID host2 ) {
    m_host1 = host1;
    m_host2 = host2;
  }
  
  bool GetStatus() { return m_status; }
  DOMString GetName() { return m_name; }
  FirewallID GetId() { return m_id; }
    
  HostID GetFirstLocationHost() { return m_host1; }
  HostID GetSecondLocationHost() { return m_host2; }
  
  bool IsBetweenHosts( HostID host1, HostID host2 ) {
    return (m_host1 == host1 && m_host2 == host2 ||
            m_host1 == host2 && m_host2 == host1);
  }
  
  bool IsFirewallRuleDefined( const FirewallRule &rule ) {
    return (m_rules.find( rule ) != m_rules.end());
  }
  
  bool GetRuleAction( const FirewallRule &rule ) {
    if ( !IsFirewallRuleDefined(rule) )
      return false;
    return m_rules[ rule ];
  }
  
  bool AddRuleAction( const FirewallRule &rule, bool action = true ) {
    bool res = !IsFirewallRuleDefined( rule );
    
    if ( res ) {
      int add_flag = 0;
      FirewallRuleActionMap::iterator it = m_rules.begin();
      FirewallRuleActionMap::iterator tmp_it;
        
      for ( ; it != m_rules.end() && add_flag <= 0; ) {
        add_flag = it->first.MatchWildacrdRule( rule );
          
        if (add_flag == -1) {
          // rule includes it->first
          tmp_it = it++;
          m_rules.erase( tmp_it );
        }
        else  
          it++;
      }
        
      if ( add_flag <= 0 )
        m_rules[rule] = action;
    }
    
    return res;
  }
  
  FirewallRuleActionMap *GetRules() { return &m_rules; }
  
  long GetMemSize() const;
  
  void GenerateRules( DOM_Element *element );
  
private:
  Network   *m_network;
  
  DOMString  m_name;
  FirewallID m_id;
  //location
  HostID     m_host1;
  HostID     m_host2;
  
  bool      m_status;
  
  FirewallRuleActionMap m_rules;
};


typedef map<DOMString, Firewall*, less_str> NameFirewallMap;

class Adversary: public ModelMemUsage {

  friend class Network;
  friend class Attack;
  friend class LTLFormula;

public:
  Adversary() { network = 0; }
  ~Adversary() {}

  // Generators (from XML)
  void Generate(DOM_Element *element);

  DOMString GetPrivilege(DOMString& host) { return privileges[host]; }
  bool GetKnowledge(DOMString& ktype) { return knowledge[ktype]; }
  int GetPort(DOMString& svc) { return ports[svc]; }
  
  DOMStringSet GetKnowledgeNames() 
  {
    DOMStringSet res;
    for ( map<DOMString, bool, less_str>::const_iterator it = knowledge.begin(); it != knowledge.end(); it++ )
      res.insert( it->first );
    return res;    
  }
  
  long GetMemSize() const;
  
  void SetNetwork( Network *net ) { network = net;}
  
private:

  void GeneratePrivileges( DOM_Element *element);
  void GenerateServices(DOM_Element *element);
  void GenerateKnowledge(DOM_Element *element);

private:
  Network *network;
  // Service ports
  map<DOMString, int, less_str> ports;

  // Privilege levels
  map<DOMString, DOMString, less_str> privileges;

  // Knowledge
  map<DOMString, bool, less_str> knowledge;
};

class IntrusionDetectionSystem: public ModelMemUsage {

public:
  IntrusionDetectionSystem() { network = 0; }
  ~IntrusionDetectionSystem() {
    map <DOMString, DOMStringSet *, less_str>::iterator it;
    for ( it = paths.begin(); it != paths.end(); it++ )
      if ( it->second != 0 )
        delete (it->second);
      
    paths.clear();
  }

  bool Monitors(DOMString& host1, DOMString& host2) {
    bool monitors;
    map <DOMString, DOMStringSet *, less_str>::iterator src = paths.find(host1);
    monitors =
      ((src != paths.end()) ?
       (src->second->find(host2) != src->second->end()) :
       false);
    map <DOMString, DOMStringSet *, less_str>::iterator tgt = paths.find(host2);
    monitors = monitors ||
      ((tgt != paths.end()) ?
       (tgt->second->find(host1) != tgt->second->end()) :
       false);
    return monitors;
  }

  // Generators (from XML)
  void Generate(DOM_Element *element);
  
  long GetMemSize() const;
  
  void SetNetwork( Network *net ) { network = net;}

private:
  Network *network;
  // Paths monitored by the IDS
  map <DOMString, DOMStringSet *, less_str> paths;
};

typedef vector< vector<bool> > BoolVectorContainer;

class Network: public ModelMemUsage
{
public:

  // Constructors and Destructor
  Network();  
  ~Network() { Clear(); }

  // Accessors
  int GetNumHosts() { return nHosts; }
  int GetNumServices() { return nServices; }
  int GetNumAttacks() { return nAttacks; }

  void SetNumHosts(int numHosts) {
    if (numHosts < 0) return;
    nHosts = numHosts;
    hosts.resize(numHosts);
    for (HostID id = 0; id < numHosts; id++)
      hosts[id] = NULL;
    
    hid.clear();
  }
  void SetNumServices(int numServices) {
    if (numServices < 0) return;
    nServices = numServices;
    DOMString path = DOMString::transcode("path");
    sid2name.resize(nServices+1);
    sid2name[0] = path;
    
    sid.clear();
  }
  void SetNumAttacks(int numAttacks) {
    if (numAttacks < 0) return;
    nAttacks = numAttacks;
    attacks.resize(numAttacks);
    for (AttackID id = 0; id < (AttackID)numAttacks; id++)
      attacks[id] = NULL;
    
    aid.clear();
  }

  bool AddHost(DOMString& hostname, const string &ip, const char *net = "") {
    // Returns false if host by that name already exists, otherwise
    // adds the host
    if (hid.find(hostname) != hid.end()) return false;

    HostID newid = nHosts;
    hid[hostname] = newid;
    hosts.push_back(new Host(this, hostname, newid, net));
    hosts[nHosts]->SetIP(ip);
    nHosts++;
    return true;
  }

  bool AddAttack(DOMString& attackname) {
    // Returns false if attack by that name already exists, otherwise
    // adds the attack
    if (aid.find(attackname) != aid.end()) return false;

    AttackID newid = nAttacks;
    aid[attackname] = newid;
    attacks.push_back( new Attack( this, attackname, newid) );
    nAttacks++;
    return true;
  }

  bool AddService(DOMString& servicename, PortID port) {
    // Returns false if service by that name already exists, otherwise
    // adds the service
    if (sid.find(servicename) != sid.end()) return false;

    nServices++;
    ServiceID newid = nServices;
    sid[servicename] = newid;
    sid2name.resize(nServices+1);
    sid2name[newid] = servicename;
    ports[servicename] = port;

    return true;
  }
  
  bool DoesModelHaveFirewalls() { return !firewall_names.empty(); }
  
  bool AddFirewallName( const DOMString &name ) {
    if (firewall_names.find( name ) != firewall_names.end() )
      return false;
    
    firewall_names[name] = fid2name.size();
    fid2name.push_back( name );
    return true;
  }
  
  FirewallID GetFirewallNumber() { return fid2name.size(); }
  
  FirewallID GetFirewallID( const DOMString &name ) {
    if (firewall_names.find( name ) == firewall_names.end() )
      return 0;
    return firewall_names[name];
  }
  
  Firewall* GetFirewall( const DOMString &name ) {
    Firewall *f = 0;
    NameFirewallMap::iterator f_it;
    if ( (f_it = firewalls.find( name )) != firewalls.end() )
      f = f_it->second;
    
    return f;
  }
  
  Firewall* GetFirewall( FirewallID id ) {
    Firewall* f = 0;
    
    if ( id < fid2name.size() )
      f = GetFirewall( fid2name[id] );
    
    return f;
  }
  
  DOMString GetFirewallName( FirewallID id ) {
    if ( id >= fid2name.size() )
      return "";
    
    return fid2name[id];
  }
  
  bool IsFirewallDefined( const DOMString &name ) {
    return firewall_names.find( name ) != firewall_names.end();
  }
  
  bool AddFirewallPort( const DOMString &name ) {
    if (firewall_ports.find( name ) != firewall_ports.end() )
      return false;
    
    firewall_ports[name] = fport2name.size();
    fport2name.push_back( name );
    return true;
  }
  
  PortID GetFirewallPortID( const DOMString &name ) {
    if ( firewall_ports.find( name ) == firewall_ports.end() )
      return 0;
    return firewall_ports[name];
  }
  
  PortID GetFirewallPortNumber() { return fport2name.size(); }
  
  DOMString GetFirewallPortName( PortID id ) {
    if ( id >= fport2name.size() )
      return "";
    return fport2name[id];
  }
  
  bool IsFirewallPortDefined( const DOMString &name ) {
    return firewall_ports.find( name ) != firewall_ports.end();
  }
  
  int GetMaxFWHostPaths() { return max_path_count; }
  
  BoolVectorContainer* GetFWPathsBetweenHosts( HostID host1, HostID host2 ) {
    return &(pathes_firewalls[host1][host2]);
  }
  
  bool AreFWHostsConnected( HostID host1, HostID host2 ) {
    return (hostlinking[host1].find( host2 ) != hostlinking[host1].end());
  }    
  

  bool AddVul(DOMString& vulname) {
    // Returns false if vul by that name already exists, otherwise
    // adds the vul
    if (vulnerabilities.find(vulname) != vulnerabilities.end()) return false;

    vulnerabilities.insert(vulname);
    return true;
  }

  bool IsInitialized() {
    return (nHosts > 0) && (nServices > 0) && (nAttacks > 0);
  }

  DOMString& GetServiceName(ServiceID id) {
    assert(id <= (ServiceID)nServices);
    return sid2name[id];
  }
  
  bool IsHostAdded ( const DOMString& hostname ) {
    return hid.find(hostname) != hid.end();
  }
  
  HostID GetHostID(DOMString& hostname) {
    assert(hid.find(hostname) != hid.end());
    return hid[hostname];
  }
  
  AttackID GetAttackID(DOMString& attackname) {
    assert(aid.find(attackname) != aid.end());
    return aid[attackname];
  }
  
  void SetDocument(DOM_Document *doc) { curr_doc = doc; }
  // Generators (from XML)
  void GenerateHost(DOM_Element *element);
  void GenerateHosts(DOM_Element *element);
  void GenerateAttack(DOM_Element *element);
  void GenerateAttacks(DOM_Element *element);
  void GenerateServices(DOM_Element *element);
  void GenerateVulnerabilities(DOM_Element *element);
  void GenerateLTLFormula( DOM_Element *element ) { ltl_formula.GenerateLTL( this, element ); }
  void GenerateFirewall( DOM_Element *element );

  // Generators to SMV
  void GenerateSMV(ostream& out, string& target);
  
  long GetMemSize() const;

private:
  void GenerateAttackDeclarationsSMV(ostream& out);
  void GenerateHostInitializationsSMV(ostream& out);
  void GenerateRelationInitializationsSMV(ostream& out);
  void GenerateAttackIdsSMV(ostream& out);
  void GenerateTransitionsSMV(ostream& out);
  void GenerateDefinesSMV(ostream& out);
  void GenerateInvarSMV(ostream& out);
  void GenerateHostSMV(ostream& out);
  void GenerateHostRelationsSMV(ostream& out);
  void GenerateAdversarySMV(ostream& out);
  void GenerateIDSSMV(ostream& out);
  void GenerateAttacks(ostream& out);
  void GenerateTransitionSMV(ostream& out, vector <Effect *>& effects,
                             const char *varname, HostID tgt, const char *entityname = NULL, 
                             const char *subentityname = NULL);
  
  char* GenerateWebAttack(DOM_Element *element);
  bool AttackURLParse(const string& url, string& host_name, int& port, string& path);
  void GenerateObjAttrValueDefinesSMV( ostream& out );
  
public:
  // Generators to SPIN
  void GenerateSPIN(ostream& out, string& target);

    
private:
  void CountingPass(DOM_Element *element);
  void CountingHost(DOM_Element *element);
  void CountingAttack(DOM_Element *element);
  void CountingFirewall(DOM_Element *element);
  void CountingFirewallRulePorts( DOM_Element *element );

  void GenerateDefinesSPIN(ostream& out);
  void GenerateAttackPreconditionsSPIN(ostream& out);
  void GenerateAttackEffectsSPIN(ostream& out);
  void GenerateTypeDefsSPIN(ostream& out);
  void GenerateMainProcessSPIN(ostream& out);
  void GenerateNeverClaimSPIN(ostream& out, string& target);
  void GenerateEffectsSPIN(ostream& out, vector <Effect *>& effects,
			     const char *varname, HostID tgt, const char *entityname = NULL);

  void GenerateFWReachabilityComputationSPIN( ostream& out );
  void GenerateFWRuleSPIN( ostream& out, const FirewallRule &rule, bool action, int firewall = -1 );
  void GenerateFWExplicitRuleSPIN( ostream& out, HostID srchost, PortID srcport, HostID dsthost, PortID dstport, 
                                           bool action, int firewall = -1 );

  
  // Output a PAA-style packet specification that host hid1 can reach host hid2
  void OutputIPPacketsSpec(ostream& out, const char *predicate, HostID hid1, HostID hid2);
  // Test if host hid1 can reach host hid2 on port
  bool Reachable(const char *netspec, HostID hid1, HostID hid2);
  
  // Output a PAA-style packet specification that host hid1 can reach host hid2
  void OutputIPPacketsSpec(ostream& out, const char *predicate, HostID hid1, HostID hid2, PortID port);
  // Test if host hid1 can reach host hid2 on port
  bool Reachable(const char *netspec, HostID hid1, HostID hid2, PortID port);
  
  //Generate named constants for object attribute values
  DOMString GetAttrConstant( const DOMString &value );
  void GenerateObjAttrValueDefinesSPIN( ostream& out );
  long GetEffectsMemSize( const vector<Effect*> &eff ) const;
  
  void ClearEffectVector( vector< Effect* > &eff ) {
    for ( unsigned int i = 0; i < eff.size(); i++ ) {
      delete eff[i];
      eff[i] = 0;
    }
    eff.clear();
  }
  
  void Clear() {
    unsigned int i;
    for ( i = 0; i < hosts.size(); i++ ) {
      delete (Host*)hosts[i];
      hosts[i] = 0;
    }
    hosts.clear();
    for ( i = 0; i < attacks.size(); i++ ) {
      delete (Attack*)attacks[i];
      attacks[i] = 0;
    }    
    attacks.clear();
    
    NameFirewallMap::iterator f_it = firewalls.begin();
    for ( ; f_it != firewalls.end(); f_it++ )
      delete (Firewall*)(f_it->second);
    firewalls.clear();
    
    ClearEffectVector( priveff );
    ClearEffectVector( svceff );
    ClearEffectVector( vuleff );
    ClearEffectVector( conneff );
    ClearEffectVector( trusteff );
    ClearEffectVector( knoweff );
    ClearEffectVector( objeff );
    ClearEffectVector( passargeff );
    ClearEffectVector( firewalleff );
  }
  
  void ComputeFConnectivity();
  
private:
  AuxilaryParserData *auxilary;

public:
  // Global variables
  Adversary adversary; // Hacker description
  IntrusionDetectionSystem ids; // IDS description

private:
  int nHosts;
  int nServices;
  int nAttacks;

  // Document related to the network 
  DOM_Document  *curr_doc;
  // Map from host names to host ids
  map<DOMString, HostID, less_str> hid;  
  // Map from host ids to hosts
  vector<Host *> hosts;

  // Map from attack names to attack ids
  map<DOMString, AttackID, less_str> aid;  
  // Map from attack ids to attacks
  vector<Attack *> attacks;

  // Service ports
  map<DOMString, PortID, less_str> ports;
  // Service ids
  map<DOMString, ServiceID, less_str> sid;
  vector<DOMString> sid2name;

  // Vulnerabilities
  DOMStringSet vulnerabilities;
  
  //Names of host objects/attributes
  HostObjectAttrNameMap host_objects;
  //All object attribute values
  DOMStringSet          obj_attr_values;
  
  //LTL formula
  LTLFormula            ltl_formula;
  
  /////////////////////////////////
  unsigned int          max_path_count;
  vector< vector<BoolVectorContainer> > pathes_firewalls;
  
  vector<HostIdSet>     hostlinking;
  vector<DOMString>     fport2name;
  NamePortIdMap         firewall_ports;
  
  vector<DOMString>     fid2name;
  map< DOMString, FirewallID, less_str>  firewall_names;
  NameFirewallMap       firewalls;
  /////////////////////////////////

  // Attack effects on various state parts
  vector <Effect *> priveff;
  vector <Effect *> svceff;
  vector <Effect *> vuleff;
  vector <Effect *> conneff;
  vector <Effect *> trusteff;
  vector <Effect *> knoweff;
  vector <Effect *> objeff;
  vector <Effect *> passargeff;
  vector <Effect *> firewalleff;

  friend class Attack;
  friend class Host;
  friend class Firewall;
  friend void GenerateNetwork( AuxilaryParserData *auxilary, Network *network, DOM_Element *element );
  friend class LTLFormula;
  friend class Adversary;
  friend class IntrusionDetectionSystem;
    
public:
  Host*   GetHost(HostID id)
  {
    if (id < (HostID)nHosts)
      return hosts[id];
    else
      return 0;
  }
  
  Attack* GetAttack(AttackID id)
  {
    if (id < (AttackID)nAttacks)
      return attacks[id];
    else
      return 0;
  }
  
  DOMStringSet* GetVulnerabilities() { return &vulnerabilities; }
  HostObjectAttrNameMap* GetHostObjectNames() { return &host_objects; }
  DOMStringSet* GetHostObjectValues() { return &obj_attr_values; }
  NameFirewallMap* GetFirewalls() { return &firewalls; }
  
};

extern char *AttackNameDefineStr(const DOMString &aname);
extern char *FixUpCVEname(char *cvename);

int atoi(DOMString dstr);
ostream& operator<< (ostream& target, const Precondition *pre);
ostream& operator<< (ostream& target, const Effect *eff);


#endif
