
#ifndef _GRAPH_H_
#define _GRAPH_H_

#include <vector>
#include <list>
#include <set>
#include <map>
#include "storable.h"
#include "types.h"
#include "nodeattr.h"

using namespace std;

namespace GraphGraphics
{
  
class GraphViewContext;
  
class Graph;

/**
* The structure represents node link attributes (weight and minimal length).
* Any link between two nodes (edge) has link attributes.
* @see Node::get_link_attr(...)
* @see Node::add_link(...)
*/
struct LinkAttr
{
  int weight;
  int minlen;
  LinkAttr() { weight = 1; minlen = 1; }
  LinkAttr(int _weight, int _minlen) : weight(_weight), minlen(_minlen) {}
};
  
/** Vector of node link attributes. */
typedef vector<LinkAttr> LinkAttrVector;

typedef vector<LinkAttr>::iterator LinkAttrVectorIter;

/** Vector of node identifiers used for representing list of node links.*/
typedef vector<NodeIndex> NodeIndexVector;
typedef list<NodeIndex> NodeIndexList;

typedef NodeIndexVector NodeLinkList;

typedef vector<NodeIndex>::iterator NodeLinkListIter;

typedef set<NodeIndex> NodeIndexSet;
typedef map<NodeIndex, NodeIndexSet> NodeLinksMap;
typedef vector<NodeLinkList> NodeLinksVector;

typedef map<string, string> StringMap;

/**
* The class represent a graph node.
* It includes links to another nodes, the graph node attributes and provides optional functionality for node marking.
* @see Graph
* @see GraphAttributes
*/
class Node
{
  friend class Graph;
public:
  
  /** Predefined node marks. */
  enum NodeMarking
  {
    NODE_UNMARKED = 0,
    NODE_IN_PATH,
    NODE_NOT_IN_PATH,
    NODE_FIXED
  };
  
private:
   NodeIndex m_index;
   NodeIndex m_sub_index;	
   unsigned char  m_mark;
   //bool      has_label;
   //string    m_label; 
   //Node attributes
   GraphAttributes *m_attributes;

   //links
   NodeLinkList m_links;
   LinkAttrVector m_link_attrs;
//this flag is used for fast finding of subgraph edges	
   bool 		m_sub_flag;	
   NodeIndex 	m_sub_tmp_id;	

private:
  void set_sub_tmp_id( NodeIndex id ) { m_sub_tmp_id = id; }
  NodeIndex get_sub_tmp_id() { return m_sub_tmp_id; }
  inline void clear_sub_flag() { m_sub_flag = false; }
  inline void set_sub_flag() { m_sub_flag = true; }   
  inline bool is_sub_flag() { return m_sub_flag; } 
  
  void set_id( NodeIndex id ) { m_index = id; }
  
  Node( NodeIndex id, GraphAttributes *attributes )
  {
      m_index = id;
      m_links.resize(0);
      m_link_attrs.resize(0);
      m_attributes = attributes;
      m_mark = Node::NODE_UNMARKED;
      //has_label = false;
      m_sub_index = id;
  }

public:
   /**
   *  Parameterless constructor.
   */
   Node( ) { m_mark = Node::NODE_UNMARKED; m_attributes = 0; }
   
   /**
   *  Destructor. 
   */
   ~Node() { remove_all_links(); if ( m_attributes != 0 ) delete m_attributes; }
   
   /**
   *  The method marks a node.
   *  @param mark The node mark. It can be one of the <code>NodeMarking</code> constants or user defined one.
   *  @see Graph::mark_all_nodes(...)
   */
   inline void set_node_mark( unsigned char mark ){ m_mark = mark; }
   
   /**
   *  The method gets a node mark.
   *  @see ::NodeMarking
   */
   inline unsigned char get_node_mark(  ){ return m_mark; }
   
   /** 
   *  @name Graph node indexation. 
   *  Node index (identifier) points to relationship between the node and a graph. This identifier is used to get acces to the node and to do node iteration. 
   *  There are two kinds of node indexation.
   *  Primary node index always refers to graph the node is part of. 
   *  Secondary node index (sub-index) can refer to any graph. For example if a node is a part of sub-graph the sub-index can refer to original graph.
   *  By default secondary node index is shown as node label.
   */
   
   //@{
      
   /**
   *  The method returns primary node index.
   *  The primary node index is set when node is added to file.
   *  @see Graph::add_node(...)
   */
   NodeIndex get_id() { return m_index; }

   /**
   *  The method sets secondary node index.
   *  @param id The identifier to be set.
   */
   void set_sub_id( NodeIndex id ) { m_sub_index = id; }
   
   /**
   *  The method returns secondary node index.
   */
   NodeIndex get_sub_id() { return m_sub_index; }
   
   //@}
   
   /**
   *  The method sets a new graph node attributes.
   *  @param attributes The pointer to <code>GraphAttributes</b> object to be set.
   */
   void set_attributes( GraphAttributes *attributes, bool delete_old = true )
   {
      if ( m_attributes != 0 && delete_old )
        delete m_attributes;
      
      m_attributes = attributes;
   }

   /**
   *  The method returns the graph node attributes.
   */
   GraphAttributes* get_attributes( )
   {
      return m_attributes;
   }

   /**
   *  The method verifies whether the node attributes match a graph attributes patern represented by <b>string</b>.
   *  @param str_attr The graph attributes in string form that should be compared with node's.
   *  @return <b>true</b> if the node attributes match the graph attributes patern.
   */
   bool compare_attributes( const string &str_attr )
   {
      return m_attributes->match_attributes( str_attr );
   }

   /**
   *  The method verifies whether the node attributes match a graph attributes patern represented by <code>NodeAttributeList</code>.
   *  @param attr_list The graph attributes that should be compared with the node's.
   *  @return <b>true</b> if the node attributes match the graph attributes patern.
   */
   bool compare_attributes( const NodeAttributeList &attr_list )
   {
      return m_attributes->match_attributes( attr_list );
   }

   /**
   *  The method sets link (oriented edge) from this node to another one of the graph.
   *  It is possible to set several links between the same nodes. In this case each link can have diferent weights and/or length attributes.  
   *  @param node_id Primary index of a node the link to it should be set.
   *  @param attr The link attributes (weight and length) to assign to the link.
   */
   void add_link( NodeIndex node_id, LinkAttr attr = LinkAttr())
   {
      m_links.push_back( node_id );
      m_link_attrs.push_back( attr );
   }

   /**
   *  Returns all links from the node to another nodes from graph.
   *  The links are primary node indexes that represent ends of oriented edges from the node. These links are sorted in order of adding.
   *  @returns The pointer to <b>vector</b> of node indexes.
   *  @warning Do not delete the returned object.
   */
   NodeLinkList* get_links() { return &m_links; }
   
   /**
   *  Returns node link.
   *  @param i The number of the node link to return. Valid values of parameter are from 0 to <code>::get_links_count()</code> - 1.
   */
   NodeIndex get_head_node( int i )
   {
     if (i < (int)m_links.size() && i >= 0 )
       return m_links[i];
     else
       return -1;
   }
   
   /**
   *  Returns node link atributes.
   *  @param i The number of the node link to return its attributes. Valid values of parameter are from 0 to <code>::get_links_count()</code> - 1.
   */
   LinkAttr get_link_attr( int i ) 
   { 
     if (i < (int)m_links.size() && i >= 0 )
       return m_link_attrs[i];
     else
     {
       cerr << "Non-correct link index \n";
       return LinkAttr(1, 1);
     }
   }
   
   /**
   *  The method returns the number of links from the node. 
   */
   int get_links_count() { return m_links.size(); }
   
   /**
   *  The method removes all links from the node.
   */
   void remove_all_links( ) 
   { 
     m_links.clear(); m_links.resize(0); 
     m_link_attrs.clear(); m_link_attrs.resize(0);
   }
};

typedef vector<Node*> NodeVector;
typedef vector<Node*>::iterator	GraphNodeIter;

/******************************************************************/

/**
*  The structure represents graph edge.
*/
struct Edge
{
  NodeIndex start_node; /**< Node index of edge beginning. */
  NodeIndex end_node;   /**< Node index of edge end. */
  Edge() { start_node = -1; end_node = -1; }
  ~Edge() {}
  Edge( NodeIndex start, NodeIndex end ): start_node(start), end_node(end) { }
  
  /**
  * Copying constructor.
  */
  Edge( const Edge &edge ): start_node( edge.start_node ), end_node( edge.end_node ) { }
  
  /**
  * Comparison operator.
  */
  bool operator == ( const Edge &edge ) const { return (start_node == edge.start_node && end_node == edge.end_node); }
  
  /**
  * The method verifies whether the edge has valid begining and end node indexes.
  */
  bool is_valid() const { return (start_node >= 0 && end_node >= 0); }
  bool less( const Edge &edge ) const 
  {
    return ((start_node == edge.start_node)? (end_node < edge.end_node): (start_node < edge.start_node));
  }
};

class EdgeLessPred: public binary_function< Edge, Edge, bool>
{
public:
  bool operator () ( const Edge &edge1, const Edge &edge2 ) const
  {
    return edge1.less( edge2 );
  }
};

typedef list<Edge> EdgeList;
typedef vector<EdgeList> EdgeListVector;
typedef set< Edge, EdgeLessPred> EdgeSet;

class GraphView;

class GraphPath
{
public:
  static const int INVALID_PATH_INDEX;

private:

typedef map<NodeIndex, NodeLinkList> NodeLinkListMap;
typedef vector<unsigned int> PathGroup;
typedef vector<PathGroup> PathLevelGroups;

  NodeIndex m_init_node;
  NodeIndexSet m_end_nodes;
  NodeLinksMap m_edges_map;
  
  NodeLinksVector m_pathes;

  PathLevelGroups m_path_groups;

private:
  void find_pathes( NodeIndex start_node, NodeLinkList &curr_path, NodeIndexSet &curr_node_set, const NodeLinkListMap &edges_map );
  void separate_pathes( GraphView *view = 0 );

public:
    
  GraphPath();
  
  bool init( NodeIndex start_node, const NodeIndexSet &end_nodes, const EdgeList &edges );
  
  GraphPath( const GraphPath& path )
  {
    m_init_node = path.m_init_node;
    m_end_nodes = path.m_end_nodes;
    m_edges_map = path.m_edges_map;
    m_pathes = path.m_pathes;
    m_path_groups = path.m_path_groups;
  }
  
  void uninit();
  void sort_pathes_by_coords( GraphView *view );
  
  ~GraphPath() {}
    
  GraphPath& operator = ( const GraphPath& path )
  {
    m_init_node = path.m_init_node;
    m_end_nodes = path.m_end_nodes;
    m_edges_map = path.m_edges_map;
    m_pathes = path.m_pathes;
    m_path_groups = path.m_path_groups;
    
    return *this;
  }
  
  bool empty() const;
  
  NodeIndex get_init_node() const { return m_init_node; }
  NodeIndexSet get_end_nodes();
  
  int get_nodes( NodeIndexSet &nodes ) const
  {
    nodes.clear();
    
    for ( NodeLinksMap::const_iterator p = m_edges_map.begin(); p != m_edges_map.end(); p++ )
      nodes.insert( p->first );
    
    return nodes.size();
  }
  
  int get_edges( EdgeList &edges ) const
  { 
    edges.clear(); 
    
    for ( NodeLinksMap::const_iterator p = m_edges_map.begin(); p != m_edges_map.end(); p++ )
      for ( NodeIndexSet::const_iterator q = p->second.begin(); q != p->second.end(); q++ )
        edges.push_back( Edge(p->first, *q) );
      
    return edges.size(); 
  }
  
  void get_contiguity_map( NodeLinksMap &path_graph ) const { path_graph = m_edges_map; }
  
  int get_pathes_count() const 
  { 
    int res = 0;
    if ( m_path_groups.empty() )
      res = m_pathes.size(); 
    else
      res = m_path_groups[ m_path_groups.size() - 1].size();
    
    return res;
  }
  
  int get_path_edges( int path_ind, EdgeList &path ) const;
 
  int get_path_nodes( int path_ind, NodeLinkList &path ) const;
  
  bool is_node_in_path( NodeIndex node_id ) const
  {
    return (m_edges_map.find( node_id ) != m_edges_map.end());
  }
  
  bool is_edge_in_path( const Edge &edge ) const
  {
    bool res = false;
    NodeLinksMap::const_iterator p = m_edges_map.find( edge.start_node );
    
    if ( p != m_edges_map.end() )
      res = (p->second.find( edge.end_node ) != p->second.end());
      
    return res;
  }
  
  bool find_path_sub_group( Graph *graph, const NodeAttributeList &include, const NodeAttributeList &exclude );
  
  int  sub_group_finding_count(){ return m_path_groups.size(); }
  bool undo_find_sub_group()
  {
    bool res = false;
    if ( res = !m_path_groups.empty() )
      m_path_groups.pop_back();
    
    return res;
  }
};

struct StringPair
{
  string first;
  string second;
  
  StringPair(): first(""), second(""){}
  StringPair( const string &str1, const string &str2): first(str1), second(str2){}
  StringPair( const StringPair &str_pair ): first(str_pair.first), second(str_pair.second){}
    
  ~StringPair(){}
    
  StringPair& operator = ( const StringPair &str_pair )
  {
    first = str_pair.first;
    second = str_pair.second;    
    return *this;
  }
  
  bool operator == ( const StringPair &str_pair ) const
  {
    return (first == str_pair.first &&
            second == str_pair.second);
  }    
    
  bool less ( const StringPair &str_pair ) const
  {
    bool res = (first < str_pair.first);
    if ( res )
      res = second < str_pair.second;
    return res;
  }
};

struct StringPair_Less
{
  bool operator () ( const StringPair &p1, const StringPair &p2 ) const
  {
    return p1.less( p2 );
  }
};

struct StringLongTriple
{
  string name;
  long   value1;
  long   value2;
  long   value3; //reserved
  
  StringLongTriple(): name(""), value1(0), value2(0), value3(0){}
  StringLongTriple( const string &n ): name(n), value1(0), value2(0), value3(0) {}  
  StringLongTriple( const StringLongTriple& sl_triple): name(sl_triple.name), 
                                                  value1(sl_triple.value1),
                                                  value2(sl_triple.value2),
                                                  value3(sl_triple.value3) {}
  ~StringLongTriple(){}                                                    
};

typedef map<string, StringLongTriple> NodeValuesMap;
typedef map<StringPair, StringLongTriple, StringPair_Less> PathValuesMap;

struct GraphStatistics
{
  //general
  long node_count;
  long edge_count;
  long max_nodes_out_degree;
  long max_nodes_in_degree;
  bool calc_nodes_edges;
  
  long graph_depth;
  long min_path_length;
  long path_count;
  bool calc_path_depth;
  
  //specified nodes to count
  // map of pairs <attribute string, (dispayed name, result1, result2, result3)>, where 
  // 'attribute string' defines nodes to count
  // result1 - the number of the nodes
  // result2 - max. out-degree of the nodes
  // result3 - max. in-degree of the nodes
  NodeValuesMap nodes_to_count;
  
  // specified pathes to count
  // map of pair <attribute string, (dispayed name, result1, result2, result3)>, where
  // 'attribute string' defines end nodes, result1 - number of pathes, 
  // result2 - max. depth  
  // result3 - min. depth
  PathValuesMap pathes_to_count;
};

/**
* The class represent directed graph.
* A <b>Graph</b>
* @see Node
* @see GraphAttributes
* @see GraphAttrNameStorage
*/
class Graph: public Storable
{
   friend class GraphViewContext;

public:
   /** Invalid node index. */
   static const NodeIndex INVALID_NODE_ID;
 
private:
  
typedef vector< Node* > PNodesVector;

   NodeIndex    m_init_node; 
     
   PNodesVector m_nodes;
   GraphViewContext* m_graph_view;

   GraphAttrNameStorage *m_attr_name_storage;

private:
   void set_app_context( GraphViewContext* app_context )
   {
      m_graph_view = app_context;
   }
   
   bool find_path( NodeIndex start_node, const NodeIndexSet &end_nodes, EdgeList &edges );
   
   bool mark_scenario( NodeIndex node_id );
   void calc_pathes( NodeIndex start_node_id, const NodeAttributeList &end_attr_list, StringLongTriple &res );
public:
  
   /**
   *  Parameterless constructor.
   */
   Graph( )
   {
      m_nodes.resize(0);
      m_nodes.reserve(0);
      m_attr_name_storage = new GraphAttrNameStorage();
      m_init_node = 0;
   }
   
   /**
   *  The method removes node of the graph.
   *  @param i Index of the node to be removed
   */
   void remove_node( NodeIndex i )
   {
      Node *node = get_node(i);
      PNodesVector::iterator node_iter;
     
      if (i >= (NodeIndex)m_nodes.size())
        return;

      delete node;
     
      node_iter = m_nodes.begin() + i;     
      
      m_nodes.erase(node_iter); 
      
      GraphAttributes *attr;
      
      for (NodeIndex k = 0; k < (NodeIndex)m_nodes.size(); k++)
      {
        if (k >= i)
        {
          node = m_nodes[k];
          node->set_id(node->get_id() - 1);
          node->set_sub_id(node->get_sub_id() - 1);
          attr = node->get_attributes();
          attr->set_highlevel_index( node->get_sub_id() );
        }
        
        NodeLinkList *list = m_nodes[k]->get_links();
        
        for (NodeIndex j = 0; j < (NodeIndex)list->size(); j++)
        {
          if ((*list)[j] == i)
          {
            list->erase(list->begin() + j);
            j--;
          }
          else if ((*list)[j] > i)
            (*list)[j]--;
        }
      }
   }

   /**
   *  The method removes all nodes from the graph.
   */
   void remove_all_nodes(  )
   {
      for ( NodeIndex i = 0; i < (NodeIndex)m_nodes.size(); i++ )
         if ( m_nodes[i] != 0 )
            delete m_nodes[i];

      m_nodes.clear();
      m_nodes.resize(0);
      m_nodes.reserve(0);
   }
   
   /**
   *  Destructor.
   */
   ~Graph() { remove_all_nodes(  ); delete m_attr_name_storage; }
   
   /**
   *  The method returns graph attributes name storage object.
   *  Any graph can have attributes assigned to its nodes. Graph attribute is triple (<code>name</code>, <code>type</code>, <code>value</code>), 
   *  where set of attribute <code>names</code> are fastened to the graph. Any graph node attribute refers to the name in apropriate graph name storage.
   *  This method returns pointer to <code>GraphAttrNameStorage</code> object that contains all defined graph attribute names. 
   *  This object is used for linking the graph node attributes with set of the graph attribute names during node attributes creating.
   *  @warning Do not delete returned object.
   */
   GraphAttrNameStorage *get_graph_attr_name_storage() { return m_attr_name_storage; }

   /**
   *  The method returns graph view context object that link the graph with its view.
   */
   GraphViewContext* get_app_context()
   {
      return m_graph_view;
   }

   /**
   *  The method adds a new node to the graph.
   *  @param node The pointer to <code>Node</code> object to be added.
   *  @return Node identifier of just added node. The identifier is assigned to the node as primary node index.Secondary node index is not changed.
   *  @warning To prevent crashing do not add the same <code>Node</code> object to several graphs (or twice to the same graph) and do not delete the object after adding.
   */
   NodeIndex add_node( Node* node )
   {
      NodeIndex id = m_nodes.size();
      node->set_id( id );
      m_nodes.push_back( node );

      return id;
   }

   /**
   *  The method creates and adds a new node to the graph.
   *  @param node_attr The pointer to graph attributes object that will be assigned to the added node.
   *  It is strongly recomended the graph attributes object is linked with the graph attribute name storage.
   *  @return Node identifier of just added node. The identifier is assigned to the node as primary and secondary node index.
   *  @warning To prevent crashing do not use the same <code>GraphAttributes</code> object creating several nodes and do not delete the object after calling the method.
   */
   NodeIndex add_node( GraphAttributes *node_attr )
   {
      NodeIndex id = m_nodes.size();
      Node* node = new Node( id, node_attr );
      node->set_sub_id(id); 	   	   
      m_nodes.push_back( node );

      return id;
   }

   /**
   *  The method creates and adds a new node to the graph.
   *  @param node_attr The pointer to graph attributes object that will be assigned to the added node.
   *  It is strongly recomended the graph attributes object is linked with the graph attribute name storage.
   *  @param sub_id The secondary node index to be assigned to the added node.
   *  @return Node identifier of just added node. The identifier is assigned to the node as primary node index.
   *  @warning To prevent crashing do not use the same <code>GraphAttributes</code> object creating several nodes and do not delete the object after calling the method.
   */
   NodeIndex add_node( GraphAttributes *node_attr, NodeIndex sub_id )
   {
      NodeIndex id = m_nodes.size();
      Node* node = new Node( id, node_attr );
      node->set_sub_id(sub_id); 
      m_nodes.push_back( node );

      return id;
   }
   
   /**
   *  The method mark all nodes.
   *  @param mark The mark to be set. If ommited the method is unmark all graph nodes. 
   *  @see Node::NodeMarking
   *  @see Node::set_node_mark(...)
   *  @see Node::get_node_mark()
   */
   void mark_all_nodes( unsigned char mark = Node::NODE_UNMARKED )
   {
     for ( PNodesVector::iterator iter = m_nodes.begin(); iter != m_nodes.end(); iter++ )
       (*iter)->set_node_mark( mark );
   }

   /**
   *  The method returns node by node index.
   *  @param The node index of requested node. Valid values of parameter are from 0 to <code>::count_nodes()</code> - 1
   *  @return The pointer to a <code>Node</code> object, which primary node index correspond to the parameter.
   */
   Node* get_node( NodeIndex node_id )
   {
      if ( node_id >= (NodeIndex)m_nodes.size() )
         return 0;

      return m_nodes[ node_id ];
   }

   /**
   *  The method returns the number of nodes in the graph.
   */
   int count_nodes() { return m_nodes.size(); }
   
   int count_edges() 
   {
     int res = 0;
     for ( PNodesVector::iterator p = m_nodes.begin(); p != m_nodes.end(); p++ )
       res += (*p)->get_links_count();
     
     return res;
   }
   
   /**
   *  The method returns primary node index of the first node which attributes match a graph attributes patern represented by <code>NodeAttributeList</code>.
   *  @param attr_list The graph attributes that should be compared with the node's.
   *  @return Node index or <code>INVALID_NODE_ID</code> if appropriate node is not found.
   */
   NodeIndex find_first_node( const NodeAttributeList &attr_list )
   {
      NodeIndex id = INVALID_NODE_ID;
      for ( NodeIndex i = 0; i < (NodeIndex)m_nodes.size(); i++ )
         if ( m_nodes[i]->compare_attributes( attr_list ) )
         {
            id = m_nodes[i]->get_id();
            break;
         }

      return id;
   }

   /**
   *  The method returns primary node index of the first node which attributes match a graph attributes patern represented by <b>string<b>.
   *  @param node_search_string The graph attributes in string form that should be compared with node's.
   *  @return Node index or <code>INVALID_NODE_ID</code> if appropriate node is not found.
   */
   NodeIndex find_first_node( const string &node_search_string )
   {
      NodeIndex id = INVALID_NODE_ID;
      
      for ( NodeIndex i = 0; i < (NodeIndex)m_nodes.size(); i++ )
         if ( m_nodes[i]->compare_attributes( node_search_string ) )
         {
            id = m_nodes[i]->get_id();
            break;
         }

      return id;
   }

   /**
   *  The method returns primary node index of next node which attributes match a graph attributes patern represented by <code>NodeAttributeList</code>.
   *  @param attr_list The graph attributes that should be compared with the node's.
   *  @return Node index or <code>INVALID_NODE_ID</code> if appropriate node is not found.
   */
   NodeIndex find_next_node( const NodeAttributeList &attr_list, NodeIndex prev_node )
   {
      NodeIndex id = INVALID_NODE_ID;
      for ( NodeIndex i = prev_node + 1; i < (NodeIndex)m_nodes.size(); i++ )
         if ( m_nodes[i]->compare_attributes( attr_list ) )
         {
            id = m_nodes[i]->get_id();
            break;
         }

      return id;
   }

   /**
   *  The method returns primary node index of next node which attributes match a graph attributes patern represented by <b>string<b>.
   *  @param node_search_string The graph attributes in string form that should be compared with node's.
   *  @return Node index or <code>INVALID_NODE_ID</code> if appropriate node is not found.
   */
   NodeIndex find_next_node( const string &node_search_string, NodeIndex prev_node )
   {
      NodeIndex id = INVALID_NODE_ID;
      for ( NodeIndex i = prev_node + 1; i < (NodeIndex)m_nodes.size(); i++ )
         if ( m_nodes[i]->compare_attributes( node_search_string ) )
         {
            id = m_nodes[i]->get_id();
            break;
         }

      return id;
   }

   /**
   *  The method mark a node node as initial.
   *  Marking a node as initial is optional. By default the node with index 0 is initial.
   *  @param node_id The index of node that should be treated as initial.
   */
   void set_init_node( NodeIndex node_id ) 
   {
      if ( node_id < (NodeIndex)m_nodes.size() )
         m_init_node = node_id; 
   }
   
   /**
   *  The method returns index of node that marked as initial.
   *  @see ::set_init_node(...)
   */
   NodeIndex get_init_node() { return m_init_node; }
   
   /**
   *  The method adds directed edge between already existed graph nodes. 
   *  @param node_from The index of egde beginning node.
   *  @param node_to The index of egde end node.
   *  @param attr The link attributes of edge (weight and/or length).
   */
   void add_edge( NodeIndex node_from, NodeIndex node_to, LinkAttr attr = LinkAttr() )
   {
      if ( node_from >= (NodeIndex)m_nodes.size() )
         return;

      Node* node = m_nodes[ node_from ];
      node->add_link( node_to, attr );
   }
   
   /**
   *  The method verifies whether an edge exists in the graph.
   *  @param edge Verified edge.
   *  @return <b>true</b> if the edge exists in the graph.
   */
   bool does_edge_exist( const Edge &edge ) const
   {
     bool res = edge.is_valid() && 
                edge.start_node < (NodeIndex)m_nodes.size() && 
                edge.end_node < (NodeIndex)m_nodes.size();
     
     if ( res )
     {
       NodeLinkList *llist = m_nodes[ edge.start_node ]->get_links();
       
       res = false;
       for ( NodeLinkList::const_iterator p = llist->begin(); p != llist->end() && !res; p++ ) 
         res = (*p == edge.end_node);
     }
     
     return res;
   }
	 
   /**
   *  The method returns subgraph of the graph which node attributes match a pattern.
   *  The resulting graph can consist of several unconnected components.
   *  @param attr_list The graph attributes patern represented by <code>NodeAttributeList</code>.
   *  @return Graph object that represent the subgraph.
   *  @see ::get_reduced_graph(...)
   */
   Graph* get_subgraph( const NodeAttributeList &attr_list );
   
   /**
   *  The method returns a graph that consists of currently selected nodes.
   */
   Graph* get_selected_subgraph( );   
   
   /**
   *  The method verifies whether a path between to nodes exists.
   *  It returns all found pathes. 
   *  @param start_node The index of begining node.
   *  @param end_nodes The set of indexes of end nodes.
   *  @param edges The list is filled by edges of all existing pathes between the nodes.
   *  @return <b>true</b> if at least one path is found.
   */
   bool get_graph_path( NodeIndex start_node, const NodeIndexSet &end_nodes, GraphPath &path )
   {
     if ( m_nodes.empty() )
       return false;
     
     bool res;
     EdgeList edges;
     
     if ( start_node >= count_nodes() || 
         start_node < 0 || end_nodes.empty() )
       res = false;
     else
     {
       res = find_path( start_node, end_nodes, edges );
       mark_all_nodes();
       
       if ( res )
         path.init( start_node, end_nodes, edges );
     }
     
     return res;
   }
   
   bool get_graph_path( NodeIndex start_node, const NodeIndexSet &end_nodes, 
                        const NodeAttributeList &include_nodes, 
                        const NodeAttributeList &exclude_nodes, GraphPath &path );
   
   bool get_graph_path( NodeIndex start_node, const NodeAttributeList &end_nodes, 
                        const NodeAttributeList &include_nodes, 
                        const NodeAttributeList &exclude_nodes, GraphPath &path )
   {
     NodeIndexSet nodes;
     
     for ( NodeIndex i = 0; i < (NodeIndex)m_nodes.size(); i++ )
       if ( m_nodes[i]->compare_attributes( end_nodes ) )
         nodes.insert( m_nodes[i]->get_id() );
       
     return get_graph_path( start_node, nodes, include_nodes, exclude_nodes, path );
   }
      
   bool get_graph_path( NodeIndex start_node, const NodeAttributeList &end_nodes_attr, GraphPath &path )
   {
     NodeIndexSet nodes;
       
     for ( NodeIndex i = 0; i < (NodeIndex)m_nodes.size(); i++ )
       if ( m_nodes[i]->compare_attributes( end_nodes_attr ) )
         nodes.insert( m_nodes[i]->get_id() );
         
     return get_graph_path( start_node, nodes, path );
   }
   
   /**
   *  The method returns connected subgraph that consists of all nodes of the graph that are reached from initial node after nodes, which attributes match paterns, have been removed.
   *  @param attr_lists Graph attributes paterns represented by <code>NodeAttributeList</code> to remove nodes that match them.
   *  @param copy_attr If <b>true</b> the resulting graph attributes will be copy of the original ones.Otherwise they will be just referenced to original ones to minimize memory usage. 
   *  @warning If <code>copy_attr</code> parameter is false the attributes of the resulting graph will refer to unexisting objects in case the original graph is deleted. In this case any operation with the attributes produces application failure.    
   *  @return A subgraph or 0 if it does not exist.
   *  @see ::get_subgraph(...)
   */
   Graph* get_reduced_graph( const vector<NodeAttributeList> &attr_lists, bool copy_attr = true );
   
   void get_graph_statistics( GraphStatistics &statistics );
   
   /**
   *  The method writes graph to a stream.
   *  @see Storable::write_to(...)
   */
   void write_to( ostream& s_out, unsigned long write_options = 0 );
   
   /**
   *  The method reads graph from a stream.
   *  @see Storable::read_from(...)
   */
   bool read_from( istream& s_in, unsigned long read_options = 0 );
};

  
};
  
#endif
