
#ifndef _DRAWING_H_
#define _DRAWING_H_

#include <string>
#include "shapes.h"
#include "storable.h"

using namespace std;

namespace GraphGraphics
{
/**
* The <b>RGB</b> structure represents a RGB color triple.
* 
* @see Color
*/  
struct RGB
{
   ColorComponent red; /**< A <b>red</b> color component of the RGB triple.*/
   ColorComponent green; /**< A <b>green</b> color component of the RGB triple.*/
   ColorComponent blue; /**< A <b>blue</b> color component of the RGB triple.*/

   /**
   * The constructor to build cpecified <b>RGB</b> triple.
   * @param r The <b>red</b> color component of the RGB triple
   * @param g The <b>green</b> color component of the RGB triple
   * @param r The <b>blue</b> color component of the RGB triple
   */
   RGB( ColorComponent r, ColorComponent g, ColorComponent b )
   {
      red = r;
      green = g;
      blue = b;
   }
   
   /**
   * The parameterless constructor.
   * It sets all color components to 0 (<b>black</b> color).
   */
   RGB() { RGB(0, 0, 0);}
};

/**
* The <b>CMYK</b> structure represents a CMYK color quadruple.
*
* @see Color
*/  
struct CMYK
{
   ColorComponent cyan; /**< A <b>cyan</b> color component of the CMYK quadruple.*/
   ColorComponent magenta; /**< A <b>magenta</b> color component of the CMYK quadruple.*/
   ColorComponent yellow; /**< A <b>yellow</b> color component of the CMYK quadruple.*/
   ColorComponent black; /**< A <b>black</b> color component of the CMYK quadruple.*/

   /**
   * The constructor to build cpecified <b>CMYK</b> quadruple.
   * @param c The <b>cyan</b> color component of the CMYK quadruple
   * @param m The <b>magenta</b> color component of the CMYK quadruple
   * @param y The <b>yellow</b> color component of the CMYK quadruple
   * @param k The <b>black</b> color component of the CMYK quadruple
   */
   CMYK( ColorComponent c, ColorComponent m, ColorComponent y, ColorComponent k )
   {
      cyan = c;
      magenta = m;
      yellow = y;
      black = k;
   }
   
   /**
   * The parameterless constructor.
   * It sets color components to be <b>black</b> color.
   */
   CMYK() { CMYK(0, 0, 0, 255);}
};

/**
* The OS-independent <b>Color</b> class implements device-dependent color representation.
*
*/

class Color: public Storable
{
public:
  
   /** @name Predefined color constants. */
   //@{
   static const Color WHITE;
   static const Color RED;
   static const Color GREEN;
   static const Color BLUE;
   static const Color MAGENTA;
   static const Color CYAN;
   static const Color YELLOW;
   static const Color BLACK;
   //@}

private:
   RGB m_rgb;
   ColorComponent m_alpha;

private:
   CMYK RGB_to_CMYK( const RGB &rgb ) const
   {
      CMYK     cmyk;
      double	c, m, y, k;

	   c = ((double)(255 - rgb.red) / 255.0);
	   m = ((double)(255 - rgb.green) / 255.0);
	   y = ((double)(255 - rgb.blue) / 255.0);
	   
	   k = (GGMIN( GGMIN( c, m), y ));

	   if( k > 0.996 )
	   {
		   cmyk.cyan = cmyk.magenta = cmyk.yellow = 0;
		   cmyk.black = 255;
	   }
      else
	   {
	      cmyk.cyan = (ColorComponent)( ((c - k) / (1 - k)) * 255.0 + 0.5 );
	      cmyk.magenta = (ColorComponent)( ((m - k) / (1 - k)) * 255.0 + 0.5);
	      cmyk.yellow = (ColorComponent)( ((y - k) / (1 - k)) * 255.0 + 0.5);
	      cmyk.black = (ColorComponent)(k * 255.0 + 0.5);
	   }

      return cmyk;
   }

   RGB CMYK_to_RGB( const CMYK &cmyk ) const
   {
      RGB      rgb;
      double	c, m, y, k;
	
	   c = ((double)cmyk.cyan) / 255.0;
	   m = ((double)cmyk.magenta) / 255.0;
	   y = ((double)cmyk.yellow) / 255.0;
	   k = ((double)cmyk.black) / 255.0;
	   
	   c = (c * (1 - k) + k) * 255.0 + 0.5;
	   rgb.red = (ColorComponent)GGMIN( 255,	c );

	   m = (m * (1 - k) + k) * 255.0 + 0.5;
	   rgb.green = (ColorComponent)GGMIN( 255,	m );

	   y = (y * (1 - k) + k) * 255.0 + 0.5;
	   rgb.blue = (ColorComponent)GGMIN( 255, y );

	   rgb.red = (ColorComponent)(255 - rgb.red);
	   rgb.green = (ColorComponent)(255 - rgb.green);
	   rgb.blue = (ColorComponent)(255 - rgb.blue);

      return rgb;
   }

public:
   /** 
   * This method gets <b>RGB</b> color triple. 
   * @return <b>RGB</b> color structure.
   * @see RGB
   * @see #set_RGB()
   * @see #get_CMYK()
   * @see #get_alpha()
   */
   RGB get_RGB() const { return m_rgb; }
   
   /** 
   * This method gets <b>CMYK</b> color quadruple. 
   * @return <b>CMYK</b> color structure.
   * @see CMYK
   * @see #set_CMYK()
   * @see #get_RGB()
   * @see #get_alpha()
   */
   CMYK get_CMYK() const { return RGB_to_CMYK( m_rgb ); }
   
   /**
   * This method sets a <b>RGB</b> triple of a color.
   * @param c The <b>RGB</b> color structure to be set.
   * @see RGB   
   * @see #get_RGB()
   * @see #set_CMYK()
   * @see #set_alpha()
   */   
   void set_RGB( const RGB &c ) { m_rgb = c; }
   
   /**
   * This method sets a <b>CMYK</b> color quadruple.
   * @param c The <b>CMYK</b> color structure to be set.
   * @see CMYK   
   * @see #get_CMYK()
   * @see #set_RGB()
   * @see #set_alpha()
   */
   void set_CMYK( const CMYK &c ) { }
   
   /**
   * This method sets an alpha (transparency) color component.
   * @param a The transparency value to be set.
   * @see #get_alpha()
   * @see #set_RGB()
   * @see #set_CMYK()
   */
   void set_alpha( ColorComponent a ) { m_alpha = a; }
   
   /**
   * This method returns an alpha (transparency) color component value.
   * @see #set_alpha()
   * @see #get_RGB()
   * @see #get_CMYK()
   */
   ColorComponent get_alpha( ) const { return m_alpha; }

   /** @name Constructors */
   //@{
   
   /** Parameterless constructor. */
   Color() {}
     
   /** Copy constructor. 
   * @param c The <code>Color</code> object to be copied.
   */
   Color( const Color &c ) { m_rgb = c.m_rgb; }

   /** 
   * Constructor to build color from <b>RGB</b> triple. 
   * @param c The <b>RGB</b> color structure to be set.
   * @see RGB
   */
   Color( const RGB &c ) { m_rgb = c; }

   /** 
   * Constructor to build color from <code>CMYK</code> quadruple. 
   * @param c The <b>CMYK</b> color structure to be set.
   * @see CMYK
   */
   Color( const CMYK &c ) { m_rgb = CMYK_to_RGB( c ); }

   /** 
   * Constructor to build color from <b>RGB</b> triple and alpha (transparency) component. 
   * @param c The <b>RGB</b> color structure to be set.
   * @param a The transparency component to be set.
   * @see RGB
   */
   Color( const RGB &c, ColorComponent a ) { m_rgb = c; m_alpha = a; }

   /** 
   * Constructor to build color from <b>CMYK</b> quadruple and alpha (transparency) component. 
   * @param c The <code>CMYK</code> color structure to be set.
   * @param a The transparency component to be set.
   * @see RGB
   */
   Color( const CMYK &c, ColorComponent a )
   {
      m_rgb = CMYK_to_RGB( c );
      m_alpha = a;
   }
   //@}

   /** @name Destructors */
   //@{
   /**
   * Destructor.
   */
   ~Color() {}
   //@}
     
   /**
   *  Assignment operator.
   *  It copies all color object components.
   *  @param c The <code>Color</code> object to be copied.
   */
   Color& operator = ( const Color &c )
   {
      m_rgb = c.m_rgb;
      m_alpha = c.m_alpha;

      return *this;
   }

   /**
   *  Equality operator.
   *  @param c The <code>Color</code> object to be compared with.
   *  @return True if colors are equal (including transparence components).
   */
   bool  operator == ( const Color &c ) const
   {
      RGB rgb = c.m_rgb;

      return (m_rgb.red == rgb.red && 
              m_rgb.green == rgb.green && 
              m_rgb.blue == rgb.blue && 
              m_alpha == c.get_alpha());
   }

   /** @name Operators for colors manipulation. */
   //@{
   
   /**
   * The colors <b>AND</b> mixing.
   */
   Color operator & ( const Color &c ) const
   {
      RGB rgb = c.m_rgb;
      
      rgb.red &= m_rgb.red;
      rgb.green &= m_rgb.green;
      rgb.blue &= m_rgb.blue;
      
      return Color( rgb );
   }

   /**
   * The colors <b>OR</b> mixing.
   */
   Color operator | ( const Color &c ) const
   {
      RGB rgb = c.m_rgb;
      
      rgb.red |= m_rgb.red;
      rgb.green |= m_rgb.green;
      rgb.blue |= m_rgb.blue;
      
      return Color( rgb );
   }

   /**
   * The colors <b>XOR</b> mixing.
   */
   Color operator ^ ( const Color &c ) const
   {
      RGB rgb = c.m_rgb;
      
      rgb.red ^= m_rgb.red;
      rgb.green ^= m_rgb.green;
      rgb.blue ^= m_rgb.blue;
      
      return Color( rgb );
   }

   /** 
   * Color inversion.
   */
   Color operator ! () const
   {
	    return Color( RGB( 255 - m_rgb.red, 255 - m_rgb.green, 255 - m_rgb.blue) );
   }
   //@}
   
   /**
   * The method writes the color data to a stream.
   * @see Storable
   */
   void write_to( ostream& s_out, unsigned long write_options = 0 );
   
   /**
   * The method reads the color data from a stream.
   * @see Storable
   */
   bool read_from( istream& s_in, unsigned long read_options = 0 );
};

/********************************************************************/

/**
* The OS-independent <b>DrawingPen</b> class represents pen attributes for line drawing.
*/
class DrawingPen: public Storable
{
public:
   /** Predefined pen styles. */
   enum PenStyle
   {
     SOLID = 0,   /**< Solid line. */
	   ON_OFF_DASH, /**< Dashed line. */
	   DOUBLE_DASH  /**< Dashed line. */
   };
	
   // 
protected:
   StyleFlags m_line_style;
   OneDValue  m_line_width;
public:
   /** @name Constructors. */
   //@{

   /** Parameterless constructor. */
   DrawingPen()
   {
      m_line_style = SOLID;
      m_line_width = 1;
   }
   
   /**
   *  The constructor biuld <b>pen</b> from attributes.
   *  @param style The line style value to be set.
   *  @param width The line width value to be set.
   *  @see PenStyle
   */
   DrawingPen( StyleFlags style, OneDValue width ): m_line_style(style), m_line_width( width ) {}
     
   /**
   *  Copy constructor.
   *  @param pen The object to be copied.
   */
   DrawingPen( const DrawingPen &pen )
   {
      m_line_style = pen.m_line_style;
      m_line_width = pen.m_line_width;
   }
   //@}
   
   /** @name Destructors. */
   //@{
   ~DrawingPen() { }
   //@}
   
   /**
   *  Assignment operator.
   *  It copies all pen object attributes.
   *  @param pen The <b>DrawingPen</b> object to be copied.
   *  &return The referance to the object.
   */
   DrawingPen& operator = ( const DrawingPen &pen )
   {
      m_line_style = pen.m_line_style;
      m_line_width = pen.m_line_width;
	  
	  return *this;
   }

   /**
   *  Equality operator.
   *  @param pen The <code>DrawingPen</code> object to be compared with.
   *  @return True if pens are equal.
   */
   bool operator == ( const DrawingPen &pen ) const
   {
      return ( m_line_style == pen.m_line_style &&
               m_line_width == pen.m_line_width );
   }

   /**
   *  This method sets line style.
   *  @param style The line style to be set. It should be one of the value of predefined line styles <b>PenStyle</b>.
   *  @see #PenStyle
   *  @see #get_line_style()
   */
   void set_line_style( StyleFlags style ) { m_line_style = style; }
   
   /**
   *  This method returns line style.
   *  @see #PenStyle
   *  @see #set_line_style(...)
   */
   unsigned int get_line_style() const { return m_line_style; }
   
   /**
   *  This method sets line width.
   *  @param width The line width to be set.
   *  @see #get_line_width()
   */
   void set_line_width( OneDValue width ) { m_line_width = width; }
   
   /**
   *  This method returns line width attribute value.
   *  @param width The line width to be set.
   *  @see #set_line_width(...)
   */
   unsigned short get_line_width() const { return m_line_width; }
   
   /**
   * The method writes the pen data to a stream.
   * @see Storable
   */
   void write_to( ostream& s_out, unsigned long write_options = 0 );
   
   /**
   * The method reads the pen data from a stream.
   * @see Storable
   */
   bool read_from( istream& s_in, unsigned long read_options = 0 );
};

/********************************************************************/

/**
*  This OS-independent class represents font.
*/
class Font: public Storable
{
public:
   enum FontFlags
   {
      NONE = 0,    
      BOLD = 1,    
      ITALIC = 2, 
      UNDERLINED = 4,
      STRIKEDOUT = 8
   };

protected:
   StyleFlags   m_style;
   unsigned int m_size;
   string       m_face_name;

public:
   Font()
   {
      m_style = NONE;
      m_size  = 8;
      m_face_name = "";
   }

   Font( const Font &font )
   {
      m_style = font.m_style;
      m_size  = font.m_size;
      m_face_name = font.m_face_name;
   }

   Font( StyleFlags style, unsigned int size, const string &face_name )
   {
      m_style = style;
      m_size = size;
      m_face_name = face_name;
   }

   Font& operator = ( const Font &font )
   {
      m_style = font.m_style;
      m_size = font.m_size;
      m_face_name = font.m_face_name;

      return *this;
   }
   
   bool operator == ( const Font &font ) const 
   {
     return (m_style == font.m_style &&  
             m_size == font.m_size &&  
             m_face_name == font.m_face_name);
   }

   StyleFlags get_style() const { return m_style; }
   void set_style( StyleFlags style ) { m_style = style; }
   string get_face_name() const { return m_face_name; }
   void set_face_name( string face_name ) { m_face_name = face_name; }
   unsigned int get_size() const { return m_size; }
   void set_size( unsigned int size ) { m_size = size; }
   
   void write_to( ostream& s_out, unsigned long write_options = 0 );
   bool read_from( istream& s_in, unsigned long read_options = 0 );

};

/********************************************************************/
/********************************************************************/

//Interface for OS dependent font geometry representation
class FontGeometry
{
public:
   virtual ~FontGeometry() {}

   virtual OneDValue get_string_height( /*const Font &f,*/ const string &str ) = 0;
   virtual OneDValue get_char_height( /*const Font &f,*/ char ch ) = 0;
   virtual OneDValue get_char_width( /*const Font &f, */char ch ) = 0;
   virtual OneDValue get_string_width( /*const Font &f, */const string &str ) = 0;
//   virtual TwoDSize  get_string_size( const Font &f, const string &str ) = 0;

   // possible other methods

};

/********************************************************************/

// Interface for OS dependent graphic area representation.
// 
class GraphicArea
{
public:
   enum DrawMode
   {
      SOLID_COLOR = 0,
      XOR_MODE,
      AND_MODE,
      OR_MODE,
      INVERT_MODE
   };
   
   enum FillMode
   {
     FILL_SOLID = 0,
     FILL_GRID
   };

public:
   virtual ~GraphicArea() {}
       
   virtual GraphicArea* clone() = 0;    

   virtual void set_foreground_color( const Color &c ) = 0;
   virtual Color get_foreground_color() = 0;

   virtual void set_background_color( const Color &c ) = 0;
   virtual Color get_background_color() = 0;

   virtual void set_drawing_pen( const DrawingPen &pen ) = 0;
   virtual DrawingPen get_drawing_pen( ) = 0;

   virtual void set_paint_mode( StyleFlags mode ) = 0;
   virtual StyleFlags get_paint_mode() = 0;
   virtual void set_fill_mode( StyleFlags mode ) = 0;
   virtual StyleFlags get_fill_mode() = 0;

   virtual bool set_font( const Font &f ) = 0;
   virtual Font get_font() = 0;

   virtual FontGeometry* get_font_geometry() = 0;

   virtual Rectangle get_drawing_bounds() = 0;
   virtual void set_drawing_bounds( const Rectangle &bounds ) = 0;
   virtual Rectangle get_clipping_bounds() = 0;
   
   // drowing geometric primitives 
   virtual void draw_point( const TwoDPoint &point ) = 0;
   virtual void draw_line( const TwoDPoint &start_point, const TwoDPoint &end_point ) = 0;
   virtual void draw_hollow_rectangle( OneDValue left, OneDValue top, OneDValue right, OneDValue bottom ) = 0;
   virtual void draw_filled_rectangle( OneDValue left, OneDValue top, OneDValue right, OneDValue bottom ) = 0;
   virtual void draw_hollow_circle( const TwoDPoint &center, OneDValue radius ) = 0;
   virtual void draw_filled_circle( const TwoDPoint &center, OneDValue radius ) = 0;
   //virtual void draw_arc( TwoDPoint center, OneDValue radius, TwoDPoint start_point, double angle ) = 0;

   virtual void draw_text( const TwoDPoint &start_point, const string &text ) = 0;

   virtual void map_into( const Rectangle &rect, GraphicArea *g_into, const Rectangle &rect_into, bool stretch = false ) = 0;
};

};

#endif
